//  DateLineAppDelegate.m
//  DateLine
//
//  Created by Alex Clarke on 6/05/05.
//  Copyright CocoaLab 2005 . All rights reserved.

#import "DateLineAppDelegate.h"
#import "CLDateElement.h"
#import "NSCalendarDate+CLExtensions.h"
#import "CLBoundView.h"
#import "RadiansToDegreesTransformer.h"
#import "KFAppleScriptHandlerAdditionsCore.h"
#import "KFASHandlerAdditions-TypeTranslation.h"
#import "NSColor+CLExtensions.h"
#import "CLManagedGraphic.h"
#import <Carbon/Carbon.h>
#import "XMArgumentKeys.h"

@class NDAppleScriptObject;

int updateTime = 60; //seconds between auto updates = 1 min

@interface DateLineAppDelegate (Private)

- (void)registerForNotifications;
- (void)registerAsObserver;

@end

@implementation DateLineAppDelegate

+(void) initialize 
{	
	[super initialize]; //NSLog(@"initialize");
	[self initialiseValueTransformers]; //NSLog(@"initialiseValueTransformers");
	[self registerDefaults]; //NSLog(@"registerDefaults");
	[self initializeColorPanel]; //NSLog(@"initializeColorPanel");
}

- (id) init
{
    if (self = [super init])
	{		
		[self registerForNotifications];
		[self registerAsObserver];
		[self launchCheck];
	}
	return self;
}

/* Neat dockless hack using Carbon from http://codesorcery.net/2008/02/06/feature-requests-versus-the-right-way-to-do-it */

- (void) setDockStatus {
	
	// this should be called from the application delegate's applicationDidFinishLaunching
	// method or from some controller object's awakeFromNib method
	
	if ([[NSUserDefaults standardUserDefaults] boolForKey:kXMDoShowInDock]) {
		ProcessSerialNumber psn = { 0, kCurrentProcess };
		// display dock icon
		TransformProcessType(&psn, kProcessTransformToForegroundApplication);
		// enable menu bar
		SetSystemUIMode(kUIModeNormal, 0);
		// switch to Dock.app
		[[NSWorkspace sharedWorkspace] launchAppWithBundleIdentifier:@"com.apple.dock" options:NSWorkspaceLaunchDefault additionalEventParamDescriptor:nil launchIdentifier:nil];
		// switch back
		[[NSApplication sharedApplication] activateIgnoringOtherApps:TRUE];
	}
}

- (void)awakeFromNib 
{	
	[self initialiseSortDescriptors];
	
	[self setUpBindings];
	[self refreshDateElements:nil];
	[closeButton setFocusRingType:NSFocusRingTypeNone];
	[graphicsController setSortDescriptors:dateDescriptorsArray];
	
	updateTimer = [[NSTimer scheduledTimerWithTimeInterval:updateTime target:self selector:@selector(refreshDateElements:) userInfo:nil repeats:YES] retain];	
	
	NSUserDefaults * defaults = [NSUserDefaults standardUserDefaults];

	[window setFloatWidth:[defaults floatForKey:kXMWidth]];
	[daysView setLineColor: [NSUnarchiver unarchiveObjectWithData:[[NSUserDefaults standardUserDefaults] valueForKey:kXMLineColor]]];
	[daysView setDayMarkColor: [NSUnarchiver unarchiveObjectWithData:[[NSUserDefaults standardUserDefaults] valueForKey:kXMDayMarkColor]]];
	[daysView setTodayTextColor: [NSUnarchiver unarchiveObjectWithData:[[NSUserDefaults standardUserDefaults] valueForKey:kXMTodayTextColor]]];
	[daysView setTodayMarkColor: [NSUnarchiver unarchiveObjectWithData:[[NSUserDefaults standardUserDefaults] valueForKey:kXMTodayMarkColor]]];
	[daysView setElementColor:[NSUnarchiver unarchiveObjectWithData:[[NSUserDefaults standardUserDefaults] valueForKey:kXMElementBackColor]]];
	
	[self setDockStatus];
	
	int level = [[defaults valueForKey:kXMDateLineWindowLevel] intValue];
	[window setLevel:level];
	int tag = [self tagForLevel:level];	
	[windowLevelPopup selectItemWithTag:tag];
}

-(void) dealloc {
	
	[updateTimer invalidate];
	[updateTimer release];

	[super dealloc];
}


-(void) resetDays:(id)sender
{
	NSArray * oldDays = [self days];
	if (oldDays != nil)
	{
		NSEnumerator * oldDaysEnumerator = [oldDays objectEnumerator];
		id anOldMonth;
		while (anOldMonth = [oldDaysEnumerator nextObject])
		{
			[managedObjectContext deleteObject:anOldMonth];
		}
		[managedObjectContext processPendingChanges];
		[self saveAction:self];
	}
}

-(void) refreshDateElements:(id)sender 
{	
	NSUserDefaults * defaults = [NSUserDefaults standardUserDefaults];
	NSColor * elementColor = [NSUnarchiver unarchiveObjectWithData: [defaults valueForKey:kXMElementBackColor]];
	NSColor * textColor = [NSUnarchiver unarchiveObjectWithData: [defaults valueForKey:kXMTextColor]];
	NSColor * todayTextColor = [defaults valueForKey:kXMTodayTextColor];
	NSColor * todayMarkColor = [NSUnarchiver unarchiveObjectWithData: [defaults valueForKey:kXMTodayMarkColor]];
	NSColor * dayMarkColor = [NSUnarchiver unarchiveObjectWithData: [defaults valueForKey:kXMDayMarkColor]];
	
	[self resetDays:nil];

	//  One month...
	CLManagedGraphic * aMonth = [NSEntityDescription insertNewObjectForEntityForName: @"CLMonth" inManagedObjectContext: managedObjectContext];
	[self setMonth:aMonth];
	[aMonth setColor:elementColor];
	[aMonth setTextColor:textColor];
	[aMonth setWidth:300.0];	
		
	//  holds a set of dates...
	NSCalendarDate * referenceDate = [NSCalendarDate date]; // today
	NSString * monthYearString = [referenceDate descriptionWithCalendarFormat:@"%B %Y" timeZone:[NSTimeZone systemTimeZone] locale:[[NSUserDefaults standardUserDefaults] dictionaryRepresentation]];
	NSString * refDateString = [referenceDate descriptionWithCalendarFormat:@"%d %B %Y" timeZone:[NSTimeZone systemTimeZone] locale:[[NSUserDefaults standardUserDefaults] dictionaryRepresentation]];
	[aMonth setValue:monthYearString forKey:@"monthName"];
	[aMonth setValue:referenceDate forKey:@"today"];
	
	// starting at the beginning of this month...
	NSArray * daysOfMonth = [NSCalendarDate daysOfMonth:referenceDate];
	NSEnumerator * daysOfMonthEnumerator = [daysOfMonth objectEnumerator];
	NSCalendarDate * aDay;
	while (aDay = [daysOfMonthEnumerator nextObject])
	{
		CLDateElement * anElement = [NSEntityDescription insertNewObjectForEntityForName: @"CLDateElement" inManagedObjectContext: managedObjectContext];
		[anElement setDate:aDay];
		[anElement setValue:aMonth forKey:@"month"];
		[anElement setColor:elementColor];
		[anElement setTextColor:textColor];
		[anElement setTodayTextColor:todayTextColor];
		[anElement setDayMarkColor:dayMarkColor];
		[anElement setTodayMarkColor:todayMarkColor];
			
		NSString * aDayString = [aDay descriptionWithCalendarFormat:@"%d %B %Y" timeZone:[NSTimeZone systemTimeZone] locale:[[NSUserDefaults standardUserDefaults] dictionaryRepresentation]];
		
		if ([refDateString isEqualToString:aDayString]) //If it's today
		{
			[anElement setValue:[NSNumber numberWithBool:TRUE] forKey:@"isToday"];
		}
		else //It's not
			[anElement setValue:[NSNumber numberWithBool:FALSE] forKey:@"isToday"];
	}
	[self saveAction:self];
	//NSLog(@"reset"); //thanks barry!
}

-(void)openInDayliteInMode:(NSString *)mode dateValue:(NSCalendarDate *)aDate;
{
	NSString *dlScriptPath = [[NSBundle mainBundle] pathForResource:@"Daylite3GotoCal" ofType:@"applescript"];
	
	NSError *error = nil;
	NSMutableString *dlSource = [NSMutableString stringWithContentsOfFile:dlScriptPath encoding:NSUTF8StringEncoding error:&error];
	
	if (error){
		NSAlert *alr = [[NSAlert alertWithError:error] retain];
		[alr runModal];
		[alr release];
	}
	
	
	NSString *date_str = [[NSNumber numberWithDouble:[aDate timeIntervalSinceReferenceDate]] stringValue];
	
	[dlSource replaceOccurrencesOfString:@"#TIME_INTERVAL" withString:date_str options:NSLiteralSearch range:NSMakeRange(0, [dlSource length])];
	[dlSource replaceOccurrencesOfString:@"#VIEW_MODE" withString:mode options:NSLiteralSearch range:NSMakeRange(0, [dlSource length])];
	
	NSAppleScript *scpt = [[[NSAppleScript alloc] initWithSource:dlSource] autorelease];
	[scpt executeAndReturnError:nil];
	
}

- (NSCalendarDate *)workaroundDateForApplescriptBugFromDate:(NSCalendarDate *)workingDate {
	
	NSArray * lastCity = [[[NSUserDefaults standardUserDefaults] dictionaryRepresentation] valueForKey:@"com.apple.TimeZonePref.Last_Selected_City"];
	
	//	NSLog(@"lastCity: %@", lastCity);
	
	float degreesEast = [[lastCity objectAtIndex:1] floatValue];
	if (degreesEast > 0.0) {
		
		return [workingDate dateByAddingYears:0 months:0 days:1 hours:0 minutes:0 seconds:0];
	}
	
	return workingDate;
}

-(void) selectDay:(id)sender //The sender in this case is the notification centre
{
	if (sender != nil) 
	{
		NSInteger appToUse = [[NSUserDefaults standardUserDefaults] integerForKey:kXMApplicationToUse];
		NSCalendarDate * aDate = [sender object];
		
		if ([[NSApp currentEvent] modifierFlags] & NSAlternateKeyMask)
			appToUse = XMUseiCal;
		
		if (appToUse == XMUseiCal){
			
			NSCalendarDate * workingDate = [NSCalendarDate dateWithString:[aDate descriptionWithCalendarFormat:@"%d %m %Y %z" timeZone:[NSTimeZone localTimeZone] locale:[[NSUserDefaults standardUserDefaults] dictionaryRepresentation]] calendarFormat:@"%d %m %Y %z"];

			workingDate = [self workaroundDateForApplescriptBugFromDate:workingDate];
			NSLog(@"Working Date: %@", workingDate);
			
			NSAppleEventDescriptor* dateDescriptor = [workingDate aeDescriptorValue];
			
			NSURL * iCalScriptPath = [NSURL fileURLWithPath:[[NSBundle mainBundle] pathForResource:@"iCalGoToDate" ofType:@"scpt"  inDirectory:nil]];
			iCalDateScript = [[[NSAppleScript alloc]  initWithContentsOfURL:iCalScriptPath error:nil] autorelease];
			
			[iCalDateScript executeHandler:@"goToDate" withParameter:dateDescriptor];
			
		}
		else if (appToUse == XMUseDaylite) {
			
			[self openInDayliteInMode:@"2" dateValue:aDate];
		}
		
	}
}

-(void) selectMonth:(id)sender //The sender in this case is the notification centre
{
	if (sender != nil) 
	{
		NSInteger app_code = [[NSUserDefaults standardUserDefaults] integerForKey:@"kXMApplicationToUse"];
		NSCalendarDate * aDate = [sender object];
		
		if ([[NSApp currentEvent] modifierFlags] & NSAlternateKeyMask)
			app_code = 0;
		
		
		if (app_code == 0){
			NSCalendarDate * workingDate = [NSCalendarDate dateWithString:[aDate descriptionWithCalendarFormat:@"%d %B %Y" timeZone:[NSTimeZone localTimeZone] locale:[[NSUserDefaults standardUserDefaults] dictionaryRepresentation]] calendarFormat:@"%d %B %Y"];
			NSAppleEventDescriptor* dateDescriptor = [workingDate aeDescriptorValue];
			
			NSURL * iCalScriptPath = [NSURL fileURLWithPath:[[NSBundle mainBundle] pathForResource:@"iCalGoToMonth" ofType:@"scpt"  inDirectory:nil]];
			iCalMonthScript = [[[NSAppleScript alloc]  initWithContentsOfURL:iCalScriptPath error:nil] autorelease];
			
			[iCalMonthScript executeHandler:@"goToMonth" withParameter:dateDescriptor];
		}
		else if (app_code == 1){
			[self openInDayliteInMode:@"3" dateValue:aDate];
		}
		
	}
}

-(IBAction) setElementColor:(id)sender
{
	NSUserDefaults * defaults = [NSUserDefaults standardUserDefaults];
	NSColor * color = [NSUnarchiver unarchiveObjectWithData:
		[defaults valueForKey:kXMElementBackColor]];
	[daysView setElementColor:color];
	[monthNameView setElementColor:color];
}

-(IBAction) setTextColor:(id)sender
{
	NSUserDefaults * defaults = [NSUserDefaults standardUserDefaults];
	NSColor * color = [NSUnarchiver unarchiveObjectWithData:
		[defaults valueForKey:kXMTextColor]];
	[daysView setTextColor:color];
	[monthNameView setTextColor:color];
}

-(IBAction) setLineColor:(id)sender
{
	NSUserDefaults * defaults = [NSUserDefaults standardUserDefaults];
	NSColor * color = [NSUnarchiver unarchiveObjectWithData:
					   [defaults valueForKey:kXMLineColor]];
	[daysView setLineColor:color];
}

-(IBAction) setTodayTextColor:(id)sender
{
	NSUserDefaults * defaults = [NSUserDefaults standardUserDefaults];
	NSColor * color = [NSUnarchiver unarchiveObjectWithData:
							  [defaults valueForKey:kXMTodayTextColor]];
	NSLog(@"color", color);
	[daysView setTodayTextColor:color];
}

-(IBAction) setTodayMarkColor:(id)sender
{
	NSUserDefaults * defaults = [NSUserDefaults standardUserDefaults];
	NSColor * color = [NSUnarchiver unarchiveObjectWithData:
					   [defaults valueForKey:kXMTodayMarkColor]];
	[daysView setTodayMarkColor:color];
}

-(IBAction) setDayMarkColor:(id)sender
{
	NSUserDefaults * defaults = [NSUserDefaults standardUserDefaults];
	NSColor * color = [NSUnarchiver unarchiveObjectWithData:
					   [defaults valueForKey:kXMDayMarkColor]];
	[daysView setTodayMarkColor:color];
}

#pragma mark Accessors

- (NSArray *) fonts {
	
	NSFontManager * fontManager = [NSFontManager sharedFontManager];
	return [fontManager availableFonts];
}

- (id)month
{
    return [[month retain] autorelease]; 
}
- (void)setMonth:(id)aMonth
{
    if (month != aMonth) {
        [month release];
        month = aMonth;
    }
}

-(NSArray *) days {
	
	NSFetchRequest *request = [[[NSFetchRequest alloc] init] autorelease];
	NSEntityDescription *entity = [NSEntityDescription entityForName:@"CLMonth" inManagedObjectContext:managedObjectContext];
	[request setEntity:entity];
	NSPredicate *predicate = [NSPredicate predicateWithValue:TRUE];
	[request setPredicate:predicate];
	NSError *error = nil;
	NSArray *array = [managedObjectContext executeFetchRequest:request error:&error];
	if (array != nil)
	{
		int count = [array count]; // may be 0
		if (count != 0)
		{
			return array;
		}
	}
		return nil;
}

- (IBAction) setDateLineWindowLevel:(id)sender
{
	int level, tag = [[sender selectedItem] tag];
//	NSLog(@"tag %i", tag);
	
	switch (tag) {
		case 0:
			level = kCGDesktopWindowLevel;
			break;
		case 1:
			level = kCGDesktopIconWindowLevel;
			break;
		case 2:
			level = NSNormalWindowLevel;
			break;
		case 3:
			level = NSFloatingWindowLevel;
			break;
		case 4:
			level = NSFloatingWindowLevel;
			break;
			
		default:
			level = NSNormalWindowLevel;
			break;
	}
	
	[window setLevel:level];
	
	NSUserDefaults *defaults = [NSUserDefaults standardUserDefaults];
	NSNumber * windowLevel = [NSNumber numberWithInt:level];
	
	[defaults setObject:windowLevel forKey:kXMDateLineWindowLevel];
}

- (int)tagForLevel:(int) level {
	
	int tag = 2; 
	
	if(level == kCGDesktopWindowLevel)
		tag = 0;
	if(level == kCGDesktopIconWindowLevel)
		tag = 1;
	if(level ==  NSNormalWindowLevel)
		tag = 2;
	if(level == NSFloatingWindowLevel)
		tag = 3;
	if(level == NSFloatingWindowLevel)
		tag = 4;
	
	return tag;
}

#pragma mark Initialise

+(void) initialiseValueTransformers
{
	RadiansToDegreesTransformer *vt = [[RadiansToDegreesTransformer alloc] init];
	[NSValueTransformer setValueTransformer:vt forName: @"RadiansToDegreesTransformer"];
}

+(void)initializeColorPanel
{
	NSColorPanel * colorPanel = [NSColorPanel sharedColorPanel];
	[colorPanel setShowsAlpha:YES];
}

+(void) registerDefaults
{
	NSFont * systemFont = [NSFont systemFontOfSize:10];
	NSFont * boldSystemFont = [NSFont boldSystemFontOfSize:10];
	NSUserDefaults * defaults = [NSUserDefaults standardUserDefaults];
	
	NSNumber * dayStringFormat = [NSNumber numberWithInt:0];
	
	NSString * regCode = @"";
	NSString * regName = @"";
	NSString * markShape = @"Square";
	NSString * fontName = [systemFont fontName];
	NSString * todayFontName = [boldSystemFont fontName];
	NSString * fontSize = @"10";

	NSNumber * dateLineWindowLevel = [NSNumber numberWithInt:2];
	NSNumber * doDrawMonth = [NSNumber numberWithBool:YES];
	NSNumber * doShowInDock = [NSNumber numberWithBool:YES];
	NSNumber * width = [NSNumber numberWithFloat:1024.00];
	
	NSData *defaultElementBackColor = [NSArchiver archivedDataWithRootObject:[NSColor elementBackgroundColor]];
	NSData *defaultTextColor = [NSArchiver archivedDataWithRootObject:[NSColor whiteColor]];
	NSData *defaultTodayTextColor = [NSArchiver archivedDataWithRootObject:[NSColor whiteColor]];
	NSData *defaultTodayMarkColor = [NSArchiver archivedDataWithRootObject:[NSColor whiteColor]];
	NSData *defaultDayMarkColor = [NSArchiver archivedDataWithRootObject:[NSColor cl_whiteColor]];
	NSData *defaultLineColor =  [NSArchiver archivedDataWithRootObject:[NSColor cl_whiteColor]];
	
	NSDictionary * appDefaults = [NSDictionary dictionaryWithObjectsAndKeys:
								  
								  regCode, kXMRegCode,
								  regName, kXMRegName,
								  dayStringFormat, kXMDayStringFormat, 
								  fontName, kXMFontName,
								  fontSize, kXMFontSize,
								  todayFontName, kXMTodayFontName,
								  markShape, kXMMarkShape,
								  dateLineWindowLevel, kXMDateLineWindowLevel,
								  width, kXMWidth,
								  doDrawMonth, kXMDoDrawMonth,
								  doShowInDock, kXMDoShowInDock,
								  defaultElementBackColor, kXMElementBackColor,
								  defaultTextColor, kXMTextColor,
								  defaultTodayTextColor, kXMTodayTextColor,
								  defaultTodayMarkColor, kXMTodayMarkColor,
								  defaultDayMarkColor, kXMDayMarkColor,
								  defaultLineColor, kXMLineColor,
								  
								  nil ];
	
	[defaults registerDefaults:appDefaults];
}

-(void) setUpBindings
{
	// we can't do these in IB at the moment, as
	// we don't have palette items for them
	[daysView bind: @"graphics" toObject: graphicsController
		withKeyPath:@"arrangedObjects" options:nil];
	[daysView bind: @"selectionIndexes" toObject: graphicsController
		withKeyPath:@"selectionIndexes" options:nil];
	
	[monthNameView bind: @"graphics" toObject: monthController
			withKeyPath:@"arrangedObjects" options:nil];
	[monthNameView bind: @"selectionIndexes" toObject: monthController
			withKeyPath:@"selectionIndexes" options:nil];
}

-(void) initialiseSortDescriptors
{	
	NSSortDescriptor * dateDescriptor = [[[NSSortDescriptor alloc]initWithKey:@"date" ascending:YES]autorelease];
	dateDescriptorsArray = [NSArray arrayWithObject:dateDescriptor];
}

//==============// Core Data Methods //==============//


- (NSManagedObjectModel *)managedObjectModel {
    if (managedObjectModel) return managedObjectModel;
	
	NSMutableSet *allBundles = [[NSMutableSet alloc] init];
	[allBundles addObject: [NSBundle mainBundle]];
	[allBundles addObjectsFromArray: [NSBundle allFrameworks]];
    
    managedObjectModel = [[NSManagedObjectModel mergedModelFromBundles: [allBundles allObjects]] retain];
    [allBundles release];
    
    return managedObjectModel;
}

/* Change this path/code to point to your App's data store. */
- (NSString *)applicationSupportFolder {
    NSString *applicationSupportFolder = nil;
    FSRef foundRef;
    OSErr err = FSFindFolder(kUserDomain, kApplicationSupportFolderType, kDontCreateFolder, &foundRef);
    if (err != noErr) {
        NSRunAlertPanel(@"Alert", @"Can't find application support folder", @"Quit", nil, nil);
        [[NSApplication sharedApplication] terminate:self];
    } else {
        unsigned char path[1024];
        FSRefMakePath(&foundRef, path, sizeof(path));
        applicationSupportFolder = [NSString stringWithUTF8String:(char *)path];
        applicationSupportFolder = [applicationSupportFolder stringByAppendingPathComponent:@"DateLine"];
    }
    return applicationSupportFolder;
}

- (NSManagedObjectContext *) managedObjectContext {
    NSError *error;
    NSString *applicationSupportFolder = nil;
    NSURL *url;
    NSFileManager *fileManager;
    NSPersistentStoreCoordinator *coordinator;
    
    if (managedObjectContext) {
        return managedObjectContext;
    }
    
    fileManager = [NSFileManager defaultManager];
    applicationSupportFolder = [self applicationSupportFolder];
    if ( ![fileManager fileExistsAtPath:applicationSupportFolder isDirectory:NULL] ) {
        [fileManager createDirectoryAtPath:applicationSupportFolder attributes:nil];
    }
    
    url = [NSURL fileURLWithPath: [applicationSupportFolder stringByAppendingPathComponent: @"DateLine.xml"]];
    coordinator = [[NSPersistentStoreCoordinator alloc] initWithManagedObjectModel: [self managedObjectModel]];
    if ([coordinator addPersistentStoreWithType:NSInMemoryStoreType configuration:nil URL:url options:nil error:&error]){
        managedObjectContext = [[NSManagedObjectContext alloc] init];
        [managedObjectContext setPersistentStoreCoordinator: coordinator];
    } else {
        [[NSApplication sharedApplication] presentError:error];
    }    
    [coordinator release];
    
    return managedObjectContext;
}

- (NSUndoManager *)windowWillReturnUndoManager:(NSWindow *)window {
    return [[self managedObjectContext] undoManager];
}

- (IBAction) saveAction:(id)sender {
    NSError *error = nil;
    if (![[self managedObjectContext] save:&error]) {
        [[NSApplication sharedApplication] presentError:error];
    }
}

- (NSApplicationTerminateReply)applicationShouldTerminate:(NSApplication *)sender {
    NSError *error;
    NSManagedObjectContext *context;
    int reply = NSTerminateNow;
    
    context = [self managedObjectContext];
    if (context != nil) {
        if ([context commitEditing]) {
            if (![context save:&error]) {
				
				// This default error handling implementation should be changed to make sure the error presented includes application specific error recovery. For now, simply display 2 panels.
                BOOL errorResult = [[NSApplication sharedApplication] presentError:error];
				
				if (errorResult == YES) { // Then the error was handled
					reply = NSTerminateCancel;
				} else {
					
					// Error handling wasn't implemented. Fall back to displaying a "quit anyway" panel.
					int alertReturn = NSRunAlertPanel(nil, @"Could not save changes while quitting. Quit anyway?" , @"Quit anyway", @"Cancel", nil);
					if (alertReturn == NSAlertAlternateReturn) {
						reply = NSTerminateCancel;	
					}
				}
            }
        } else {
            reply = NSTerminateCancel;
        }
    }
    return reply;
}

#pragma mark -
#pragma mark KVO

- (void)registerAsObserver {
	
	NSUserDefaults * defaults = [NSUserDefaults standardUserDefaults];
	
	[defaults addObserver:self forKeyPath:kXMMarkShape options:(NSKeyValueObservingOptionNew | NSKeyValueObservingOptionOld) context:nil];
	[defaults addObserver:self forKeyPath:kXMFontName options:(NSKeyValueObservingOptionNew | NSKeyValueObservingOptionOld) context:nil];
	[defaults addObserver:self forKeyPath:kXMTodayFontName options:(NSKeyValueObservingOptionNew | NSKeyValueObservingOptionOld) context:nil];
	[defaults addObserver:self forKeyPath:kXMFontSize options:(NSKeyValueObservingOptionNew | NSKeyValueObservingOptionOld) context:nil];
	[defaults addObserver:self forKeyPath:kXMDayStringFormat options:(NSKeyValueObservingOptionNew | NSKeyValueObservingOptionOld) context:nil];
}

- (void)observeValueForKeyPath:(NSString *)keyPath  ofObject:(id)object change:(NSDictionary *)change context:(void *)context {
	
	if ([keyPath isEqual:kXMMarkShape] || [keyPath isEqual:kXMFontName] || [keyPath isEqual:kXMTodayFontName] || [keyPath isEqual:kXMFontSize] || [keyPath isEqual:kXMDayStringFormat]) {
		
		[daysView setNeedsDisplay:YES];
		[monthNameView setNeedsDisplay:YES];
	}
}

#pragma mark -
#pragma mark Notification

- (void)registerForNotifications {
	
	NSNotificationCenter * centre = [NSNotificationCenter defaultCenter];
	[centre addObserver:self selector:@selector(selectDay:) name:@"selectDayNotification" object:nil];
	[centre addObserver:self selector:@selector(selectMonth:) name:@"selectMonthNotification" object:nil];
	
	[centre addObserver:self selector:@selector(registrationChanged:) name:XMDidChangeRegistrationNotification object:nil];	
}

@end
