// (c) Microsoft Corporation 2005-2007.
#light

#if CLI_AT_MOST_1_1
#else

namespace Microsoft.FSharp.Control

    #nowarn "40"

    open System
    open System.Threading
    open System.IO
    open Microsoft.FSharp.Core
    open Microsoft.FSharp.Core.LanguagePrimitives.IntrinsicOperators
    open Microsoft.FSharp.Core.Operators
    open Microsoft.FSharp.Control
    open Microsoft.FSharp.Collections

    type IChannel<'a> =
        abstract Post : 'a -> unit

    [<Sealed>]
    type AsyncReplyChannel<'reply>(replyf : 'reply -> unit) =
        member x.Reply(reply) = replyf(reply)
        member x.Post(reply) = replyf(reply)

    type Fifo<'a>() =
        let mutable xs  = [] : 'a list
        let mutable rxs = []
        member q.IsEmpty = (xs = []) && (rxs = [])
        member q.Enqueue(x) = rxs <- x :: rxs
        member q.ToList() = List.append xs (List.rev rxs)
        member q.Take()  = if xs = [] then
                               xs <- List.rev rxs;
                               rxs <- []
                           match xs with
                           | []    -> failwith "fifo.Take: empty queue"
                           | y::ys -> xs <- ys; y
        member x.UnsafeContents = x.ToList()


    type AsyncGroup() =
        let mutable cancellationFlag = false
        let triggerCancel,cancelEvent = IEvent.create_HandlerEvent<_,_>()
        member x.TriggerCancel(?message:string) =
            let message = defaultArg message "an exception happened on another parallel process"
            //Console.WriteLine(" ** TriggerCancel on group {0}", x.GetHashCode())
            cancellationFlag <- true; triggerCancel(null,message)
        member x.Cancel = cancelEvent
        member x.Test() = cancellationFlag

        member x.CreateSubGroup() =
            let innerAsyncGroup = new AsyncGroup()
            let handler = Handler(fun _ msg -> innerAsyncGroup.TriggerCancel(msg))
            //Console.WriteLine(" ** AddHandler for subgroup {0} to group {1}", innerAsyncGroup.GetHashCode(), x.GetHashCode); 
            x.Cancel.AddHandler(handler);
            innerAsyncGroup, (fun () -> x.Cancel.RemoveHandler(handler))


    // Some versions of F# don't always take tailcalls to functions returning 'unit' because this
    // is represented as type 'void' in the underlying IL.
    // Hence we don't use the 'unit' return type here, and instead invent our own type.
    type FakeUnitValue =
        | FakeUnit

    type Async<'a> =
        P of (AsyncGroup * ('a -> FakeUnitValue) * (exn -> FakeUnitValue) * (OperationCanceledException -> FakeUnitValue) -> FakeUnitValue)

    type IAsyncFuture<'a> =
        interface IDisposable
        abstract Value : 'a
        abstract AsIAsyncResult : System.IAsyncResult

    type Future<'a> = IAsyncFuture<'a>
    type AsyncFuture<'a> = IAsyncFuture<'a>


    [<CompilationRepresentation(CompilationRepresentationFlags.ModuleSuffix)>]
    module Async =
        // To consider: augment with more exception traceability information
        // To consider: add the ability to suspend running ps in debug mode
        // To consider: add the ability to trace running ps in debug mode
        open System
        open System.Threading
        open System.IO

        let fake () = FakeUnit
        let unfake FakeUnit = ()
        let ignoreFake _ = FakeUnit


        let defaultAsyncGroup = ref (AsyncGroup())

        type Result<'a> =
            | Error of exn
            | Ok of 'a

        // Apply f to x and return a result
        let trylet f x =
            try Ok (f x)
            with
                // Note: using a :? catch keeps FxCop happy
                | :? System.Exception as exn -> Error(exn)

        // JIT 64 and Mono don't always take tailcalls correctly. Hence hijack the continuation onto a new work item
        // every so often by counting the number of tailcalls we make from each particular O/S thread.
        // Note this also gives more interleaving of I/O.
        [<System.ThreadStatic>]
        let mutable tailcallCount = 0
        let tailcallLim = 500

        // Apply f to x and call either the continuation or exception continuation depending what happens
        let protect econt f x (cont : 'a -> FakeUnitValue) : FakeUnitValue =
            let res = trylet f x

            // Hijack every so often
            tailcallCount <- tailcallCount + 1
            if tailcallCount >= tailcallLim then
                tailcallCount <- 0
                match res with
                | Ok v ->
                    if ThreadPool.QueueUserWorkItem(fun _ -> cont v |> unfake) then
                        FakeUnit
                    else
                        failwith "unexpected QueueUserWorkItem failure"
                | Error exn ->
                    if ThreadPool.QueueUserWorkItem(fun _ -> econt exn |> unfake)  then
                        FakeUnit
                    else
                        failwith "unexpected QueueUserWorkItem failure"
            else
                match res with
                | Ok v ->
                    // NOTE: this must be a tailcall
                    cont v
                | Error exn ->
                    // NOTE: this must be a tailcall
                    econt exn

        // Reify exceptional results as exceptions
        let commit res =
            match res with
            | Ok res -> res
            | Error exn -> raise exn

        // Reify exceptional results as exceptions
        let commitWithPossibleTimeout res =
            match res with
            | None -> raise (System.TimeoutException())
            | Some res -> commit res

        // Generate async computation which calls its continuation with the given result
        let resultPrimA x = P (fun (_,cont,_,_) -> cont x)

        // Apply the underlying implementation of an async computation to its inputs
        let invokeA args (P pf) = pf args

        // The primitive bind operation. Generate a process that runs the first process, takes
        // its result, applies f and then runs the new process produced.
        let bindPrimA p1 f  =
            P (fun ((asyncGroup,cont,econt,ccont) as info) ->
                invokeA (asyncGroup,(fun a ->
                    protect econt f a (invokeA info)),econt,ccont) p1)


        // callPrim == "bindPrimA (return x) f"
        let callPrim f x =
            P (fun ((_,_,econt,_) as info) ->
                protect econt f x (invokeA info))

        // delayPrim == "bindPrimA (return ()) f"
        let delayPrim f = callPrim f ()

        // Call p but augment the normal, exception and cancel continuations with a call to finallyFunction.
        // If the finallyFunction raises an exception then call the original exception continuation
        // with the new exception. If exception is raised after a cancellation, exception is ignored
        // and cancel continuation is called.
        let tryFinallyPrim finallyFunction p  =
            P (fun (asyncGroup,cont,econt,ccont) ->
                invokeA (asyncGroup,
                          (fun b -> protect econt finallyFunction () (fun () -> cont b)),
                          (fun exn -> protect econt finallyFunction () (fun () -> econt exn)),
                          (fun cexn -> protect (fun _ -> ccont cexn) finallyFunction () (fun () -> ccont cexn)))
                         p)

        let protectedPrimitive f =
            P (fun ((_,_,econt,_) as args) ->
                try f args
                with exn -> econt exn)

        // Cancellation
        let cancelCheckA p =
            P (fun ((asyncGroup,_,_,ccont) as args) ->
                if asyncGroup.Test() then ccont (new OperationCanceledException())
                else invokeA args p)

        // Re-rout the exception continuation to call to catchFunction. This generates
        // a new process. If catchFunction or the new process fail
        // then call the original exception continuation with the failure.
        let tryWithPrimA catchFunction p =
            P (fun ((asyncGroup,cont,_,ccont) as args) ->
                invokeA (asyncGroup,cont,(fun exn -> invokeA args (callPrim catchFunction exn)), ccont) p)

        let tryFinallyA f p = cancelCheckA (tryFinallyPrim f p)
        let tryWithA    f p = cancelCheckA (tryWithPrimA f p)
        let bindA       p f = cancelCheckA (bindPrimA p f)
        let resultA       x = cancelCheckA (resultPrimA x)
        let delayA        f = cancelCheckA (delayPrim f)
        let doneA           = resultA()
        let fakeUnitA       = resultA FakeUnit
        let usingA (r:#IDisposable) f =  tryFinallyA (fun () -> r.Dispose()) (callPrim f r)
        let raiseA exn = P (fun (asyncGroup,cont,econt,_) -> econt (exn :> Exception))
        let ignoreA p = bindA p (fun _ -> resultA ())

        let catchA p =
            P (fun (asyncGroup,cont,econt,ccont) ->
                invokeA (asyncGroup,(Ok >> cont),(Error >> cont),ccont) p)

        let whenCancelledA finallyFunction p =
            P (fun (asyncGroup,cont,econt,ccont) ->
                invokeA (asyncGroup,cont,econt,(fun exn -> finallyFunction(exn); ccont(exn))) p)

        let primitiveA f = protectedPrimitive (fun (_,cont,econt,ccont) -> f (cont,econt, ccont))

        // Start on the current thread. Only Async.Run uses this.
        let startA asyncGroup cont econt ccont p =
            invokeA (asyncGroup,cont,econt,ccont) p
            |> unfake

        // Start as a work item in the thread pool
        let queueWorkItem asyncGroup cont econt ccont p =
            if not (ThreadPool.QueueUserWorkItem(fun _ -> invokeA (asyncGroup,cont,econt,ccont) p |> unfake)) then
                failwith "failed to queue user work item"

        let getAsyncGroup()  =
            P (fun ((asyncGroup,cont,_,_)) -> cont asyncGroup)

        let rec whileA gd prog =
            if gd() then bindA prog (fun () -> whileA gd prog) else resultA()

        [<Sealed>]
        type AsyncBuilder() =
            member b.Zero() = resultA()
            member b.Delay(f) = delayA(f)
            member b.Return(x) = resultA(x)
            member b.Bind(p1,p2) = bindA p1 p2
            member b.BindUsing(p1,p2) = bindA p1 (fun x -> usingA x p2)
            member b.Let(v,p2) : Async<_> = p2 v
            member b.Using(g,p) = usingA g p
            member b.While(gd,prog) = whileA gd prog
            member b.For(e:#seq<_>,prog) =
                usingA (e.GetEnumerator()) (fun ie ->
                    whileA
                        (fun () -> ie.MoveNext())
                        (delayA(fun () -> prog ie.Current)))

            member b.Combine(p1,p2) = bindA p1 (fun () -> p2)
            member b.TryFinally(p,cf) = tryFinallyA cf p
            member b.TryWith(p,cf) = tryWithA cf p

        let async = AsyncBuilder()

        let post (syncContext : SynchronizationContext) trigger args =
                match syncContext with
                | null -> ()
                | ctxt -> ctxt.Post(SendOrPostCallback(fun _ -> trigger args),state=null)


        let boxed(a:Async<'b>) : Async<obj> = async{ let! res = a in return box res }

        // A utility type to provide a synchronization point between an asynchronous computation and callers waiting
        // on the result of that computation.
        type ResultCell<'a>() =
            // Use the reference cell as a lock
            let mutable result = None
            // The WaitHandle event for the result. Only created if needed, and set to None when disposed.
            let mutable resEvent = null
            let mutable disposed = false
            // All writers of result are protected by lock on syncRoot.
            // Note: don't use 'resEvent' as a lock - it is option-valued. Option values use 'null' as a representation.
            let syncRoot = new Object()

            member x.GetWaitHandle() =
                lock syncRoot (fun () -> 
                    if disposed then 
                        raise (System.ObjectDisposedException("ResultCell"));
                    match resEvent with 
                    | null ->
                        // Start in signalled state if a result is already present.
                        let ev = new ManualResetEvent(result.IsSome)
                        resEvent <- ev
                        (ev :> WaitHandle)
                    | ev -> 
                        (ev :> WaitHandle))

            member x.Close() =
                lock syncRoot (fun () ->
                    if not disposed then 
                        disposed <- true;
                        match resEvent with
                        | null -> ()
                        | ev -> 
                            ev.Close(); 
                            resEvent <- null)

            interface IDisposable with
                member x.Dispose() = x.Close()


            member x.GrabResult() =
                match result with
                | Some res -> res
                | None -> failwith "unexpected no result"


            // Record the result in the ResultCell.
            member x.RegisterResult (res:'a) =
                lock syncRoot (fun () ->
                    // In this case the ResultCell has already been disposed, e.g. due to a timeout.
                    // The result is dropped on the floor.
                    if disposed then 
                        ()
                    else
                        result <- Some res;
                        // If the resEvent exists then set it. If not we can skip setting it altogether and it won't be
                        // created
                        match resEvent with
                        | null -> 
                            ()
                        | ev ->
                            // If the resEvent exists then set it. If not we can skip setting it altogether and it won't be
                            // created.
                            ev.Set()  |> ignore)

            /// Wait until a result is available in the ResultCell. Should be called at most once.

            member x.ResultAvailable = result.IsSome

            member x.TryWaitForResult (?timeout,?exitContext) : 'a option =
                // Check if a result is available.
                match result with
                | Some res as r ->
                    r
                | None ->
                    // Force the creation of the WaitHandle
                    let resHandle = x.GetWaitHandle()
                    // Check again. While we were in GetWaitHandle, a call to RegisterResult may have set result then skipped the
                    // Set because the resHandle wasn't forced.
                    match result with
                    | Some res as r ->
                        r
                    | None ->
                        // OK, let's really wait for the Set signal. This may block.
                        let timeout = defaultArg timeout (-1) in
                        let exitContext = defaultArg exitContext true in
                        let ok = resHandle.WaitOne(millisecondsTimeout= timeout,exitContext= exitContext) in
                        if ok then
                            // Now the result really must be available
                            result
                        else
                            // timed out
                            None

    open Async

    type Async<'res> with

        static member CancelCheck () = resultA()

        static member TryFinally (p,f) =  tryFinallyA f p
        static member Bind(p,f) = bindA p f
        static member Return x = resultA x
        static member Delay f = delayA f
        static member Done = doneA
        static member Using (r,pf) =  usingA r pf
        static member Raise exn = raiseA exn

        [<OverloadID("Primitive")>]
        static member Primitive f =
            primitiveA (fun (cont,econt,_) ->
                f (cont >> unfake, econt >> unfake) |> fake)

        [<OverloadID("Primitive_with_ccont")>]
        static member Primitive f =
            primitiveA (fun (cont,econt,ccont) ->
                f (cont >> unfake, econt >> unfake,ccont >> unfake) |> fake)

        static member CancelDefaultGroup(?message) =
            let asyncGroup = !defaultAsyncGroup
            asyncGroup.TriggerCancel(?message=message)
            defaultAsyncGroup := new AsyncGroup()

        static member Catch p =
            P (fun (asyncGroup,cont,econt,ccont) ->
                invokeA (asyncGroup,(Choice2_1 >> cont),(Choice2_2 >> cont),ccont) p)

        /// Run the asynchronous workflow and wait for its result.
        static member Run (p,?asyncGroup,?timeout,?exitContext) =
            let asyncGroup = defaultArg asyncGroup !defaultAsyncGroup
            let innerAsyncGroup, disposeFct = asyncGroup.CreateSubGroup()
            use resultCell = new ResultCell<Result<_>>()
            startA
                innerAsyncGroup
                (fun res ->
                    // Note the ResultCell may have been disposed if the operation
                    // timed out. In this case RegisterResult drops the result on the floor.
                    resultCell.RegisterResult(Ok(res));
                    FakeUnit)
                (fun exn -> resultCell.RegisterResult(Error(exn)); FakeUnit)
                (fun exn -> resultCell.RegisterResult(Error(exn :> Exception)); FakeUnit)
                p
            let res = resultCell.TryWaitForResult(?timeout = timeout, ?exitContext = exitContext) in
            match res with
            | None ->
                innerAsyncGroup.TriggerCancel();
                disposeFct()
                raise (System.TimeoutException())
            | Some res ->
                disposeFct()
                commit res


    type Async<'res> with
        static member Spawn (computation,?asyncGroup) =
            let asyncGroup = defaultArg asyncGroup !defaultAsyncGroup
            queueWorkItem
                  asyncGroup
                  (fun () -> FakeUnit)   // nothing to do on success
                  (fun _ -> FakeUnit)   // ignore exception in child
                  (fun _ -> FakeUnit)    // ignore cancellation in child
                  computation

        static member AsyncFuture (p,?asyncGroup)  =
            let asyncGroup = defaultArg asyncGroup !defaultAsyncGroup
            let resultCell = new ResultCell<_>()

            // OK, queueWorkItem the async computation and when it's done register completion
            queueWorkItem
                asyncGroup
                // on success, record the result
                (fun res -> resultCell.RegisterResult (Ok res); FakeUnit)
                // on exception...
                (fun exn -> resultCell.RegisterResult (Error(exn)); FakeUnit)
                // on cancellation...
                (fun exn -> resultCell.RegisterResult (Error(exn :> Exception)); FakeUnit)
                p

            { new IAsyncFuture<_> with
                  member x.Value = commitWithPossibleTimeout (resultCell.TryWaitForResult())
                  member x.AsIAsyncResult =
                      { new IAsyncResult with
                           member iar.IsCompleted= resultCell.ResultAvailable
                           member iar.AsyncState=null
                           member iar.CompletedSynchronously=false
                           member iar.AsyncWaitHandle = resultCell.GetWaitHandle() }

              // Dispose the result cell if needed
              interface System.IDisposable with
                  member x.Dispose() = resultCell.Close() }

        static member Future (p,?asyncGroup)  =
            Async<_>.AsyncFuture(p,?asyncGroup=asyncGroup)


    type Async<'a> with
        static member Parallel l =
            protectedPrimitive (fun (asyncGroup,cont,econt,ccont) ->
                let tasks = Seq.to_array l
                if tasks.Length = 0 then cont [| |] else
                let count = ref tasks.Length
                let results = Array.zero_create tasks.Length
                // Attept to cancel the individual operations if an exception happens on any the other threads
                let innerAsyncGroup, disposeFct = asyncGroup.CreateSubGroup()
                tasks |> Array.iteri (fun i p ->
                    queueWorkItem
                        innerAsyncGroup
                        // on success, record the result
                        (fun res ->
                            results.[i] <- res;
                            let last = lock count (fun () -> count := !count - 1; (!count = 0))
                            if last then disposeFct(); cont results else FakeUnit)
                        // on exception...
                        (fun exn ->
                            let first = lock count (fun () -> let first = (!count > -1) in count := -1; first) in
                            //Console.WriteLine("exn - {0}", first)
                            if first then innerAsyncGroup.TriggerCancel()
                            if first then disposeFct(); econt exn else FakeUnit)
                        // on cancellation...
                        (fun cexn ->
                            let first = lock count (fun () -> let first = (!count > -1) in count := -1; first) in
                            if first then disposeFct(); ccont cexn else FakeUnit)
                        p);
                FakeUnit)

    type Async<'a> with

        static member Parallel2 (a:Async<'b>,b:Async<'c>) =
            async { let! res = Async.Parallel [Async.boxed a; Async.boxed b]
                    return (unbox<'b>(res.[0]), unbox<'c>(res.[1])) }

        static member Parallel3 (a:Async<'b>,b:Async<'c>,c:Async<'d>) =
            async { let! res = Async.Parallel [Async.boxed a; Async.boxed b; Async.boxed c]
                    return (unbox<'b>(res.[0]), unbox<'c>(res.[1]), unbox<'d>(res.[2])) }

        [<OverloadID("BuildPrimitve_zero_arg")>]
        static member BuildPrimitive(beginFunc,endFunc) =
            primitiveA(fun (cont,econt,_) ->
                (beginFunc(System.AsyncCallback(fun iar -> protect econt endFunc iar cont |> unfake),(null:obj)) : System.IAsyncResult) |> ignoreFake)

        [<OverloadID("BuildPrimitve_one_arg")>]
        static member BuildPrimitive(arg1,beginFunc,endFunc) =
            primitiveA(fun (cont,econt,_) ->
                (beginFunc(arg1,System.AsyncCallback(fun iar -> protect econt endFunc iar cont |> ignore),(null:obj)) : System.IAsyncResult) |> ignore; FakeUnit)

        [<OverloadID("BuildPrimitve_two_arg")>]
        static member BuildPrimitive(arg1,arg2,beginFunc,endFunc) =
            primitiveA(fun (cont,econt,_) ->
                (beginFunc(arg1,arg2,System.AsyncCallback(fun iar -> protect econt endFunc iar cont |> ignore),(null:obj)) : System.IAsyncResult) |> ignore; FakeUnit)


        [<OverloadID("BuildPrimitve_three_arg")>]
        static member BuildPrimitive(arg1,arg2,arg3,beginFunc,endFunc) =
            primitiveA(fun (cont,econt,_) ->
                (beginFunc(arg1,arg2,arg3,System.AsyncCallback(fun iar -> protect econt endFunc iar cont |> ignore),(null:obj)) : System.IAsyncResult) |> ignore; FakeUnit)

(*
        static member ParallelNoCatch l =
            protectedPrimitive (fun (asyncGroup,cont,econt) ->
                let tasks = Seq.to_array l
                let count = ref tasks.Length
                let results = Array.zero_create tasks.Length
                tasks |> Array.iteri (fun i p ->
                    runInThreadPoolWithCancellation asyncGroup (bindA p (fun res ->
                        results.[i] <- res;
                        let last = lock count (fun () -> count := !count - 1; (!count = 0))
                        primitiveA (fun _ -> if last then cont results)))))
*)

    type Async<'a> with
        static member Generate (n,f, ?numChunks) =
            let procs = defaultArg numChunks System.Environment.ProcessorCount
            if procs <= 0 then invalid_arg "numChunks";
            if n < 0 then invalid_arg "n";
            let resArray = Array.zero_create n
            async { let! res = Async<_>.Parallel
                                  [ for pid in 0 .. procs-1 ->
                                        async { for i in 0 .. (n/procs+(if n%procs > pid then 1 else 0)) - 1 do
                                                    let elem = (i + pid * (n/procs) + min (n%procs) pid)
                                                    let! res = f elem
                                                    do resArray.[elem] <- res;  } ]
                    return resArray }

        static member Ignore p = bindPrimA p (fun _ -> resultA ())
        static member SwitchToNewThread() =
            protectedPrimitive(fun (_,cont,_,_) ->
                  (new Thread(ThreadStart(fun () -> cont () |> ignore),IsBackground=true)).Start(); FakeUnit)

        static member SwitchToThreadPool() =
            protectedPrimitive(fun (_,cont,econt,_) ->
                    if not (ThreadPool.QueueUserWorkItem(fun _ -> cont () |> ignore)) then
                        econt (Failure "SwitchToThreadPool: failed to queue user work item")
                    else
                        FakeUnit)

    type Async<'a> with

        static member SpawnChild (p:Async<unit>) =
            let ignoreExceptions = true
            async { let! asyncGroup = getAsyncGroup ()
                    do queueWorkItem asyncGroup
                                      (fun () -> FakeUnit)   // nothing to do on success
                                      (fun _ -> FakeUnit)   // ignore exception in child
                                      (fun _ -> FakeUnit)    // ignore cancellation in child
                                      p }

        static member SpawnThenPostBack (computation,postBack,?asyncGroup) =
            let syncContext = SynchronizationContext.Current
            match syncContext with
            | null -> failwith "The System.Threading.SynchronizationContext.Current of the calling thread is null"
            | _ ->
                Async<_>.Spawn (async { let! res = catchA(computation) in
                                        do post syncContext (fun () -> postBack (commit res)) () },
                                ?asyncGroup=asyncGroup)

        static member UnblockViaNewThread p =
             async { do! Async<_>.SwitchToNewThread ()
                     let! res = p
                     do! Async<_>.SwitchToThreadPool ()
                     return res }

        static member UnblockedPrimitive f =
             Async<_>.UnblockViaNewThread (primitiveA (fun (cont,econt,_) -> protect econt f () cont))

        static member OnCancel action =
            async { let h = new Handler<_>(fun sender x -> try action x with _ -> ())
                    let! asyncGroup = getAsyncGroup ()
                    do asyncGroup.Cancel.AddHandler(h);
                    return { new IDisposable with
                                 member x.Dispose() = asyncGroup.Cancel.RemoveHandler(h) } }

        static member WhenCancelled (p,f) = 
            whenCancelledA f p

    type AsyncWorker<'a>(p,?asyncGroup) =

        let syncContext = SynchronizationContext.Current
        do match syncContext with
            | null -> failwith "The System.Threading.SynchronizationContext.Current of the calling thread is null"
            | _ -> ()

        // Trigger one of the following events when the iteration completes.
        let triggerCompleted,completed = IEvent.create<'a>()
        let triggerError    ,error     = IEvent.create()
        let triggerCanceled,canceled   = IEvent.create()
        let triggerProgress ,progress  = IEvent.create<int>()

        let asyncGroup = defaultArg asyncGroup !defaultAsyncGroup

        member x.ReportProgress(progressPercentage) =
            Async.post syncContext triggerProgress progressPercentage

        member x.RunAsync()    =
            queueWorkItem
                asyncGroup
                (fun res -> Async.post syncContext triggerCompleted res; FakeUnit)
                (fun exn -> Async.post syncContext triggerError exn; FakeUnit)
                (fun cexn -> Async.post syncContext triggerCanceled (cexn :> Exception); FakeUnit)
                p;
            true

        member x.CancelAsync(?message) =
            asyncGroup.TriggerCancel(?message=message);

        member x.ProgressChanged     = progress
        member x.Completed  = completed
        member x.Canceled  = canceled
        member x.Error      = error


namespace Microsoft.FSharp.Control

    #nowarn "40"

    open System
    open System.Threading
    open System.IO
    open Microsoft.FSharp.Core
    open Microsoft.FSharp.Core.LanguagePrimitives.IntrinsicOperators
    open Microsoft.FSharp.Core.Operators
    open Microsoft.FSharp.Control
    open Microsoft.FSharp.Control.Async
    open Microsoft.FSharp.Collections
    open System.Collections.Generic

    module CommonExtensions =
        open Async

        type System.IO.File with
            static member OpenTextAsync(path)   = Async<_>.UnblockedPrimitive (fun () -> System.IO.File.OpenText(path))
            static member AppendTextAsync(path) = Async<_>.UnblockedPrimitive (fun () -> System.IO.File.AppendText(path))
            static member OpenReadAsync(path)   = Async<_>.UnblockedPrimitive (fun () -> System.IO.File.OpenRead(path))
            static member OpenWriteAsync(path)  = Async<_>.UnblockedPrimitive (fun () -> System.IO.File.OpenWrite(path))
            static member OpenAsync(path,mode,?access,?share) =
                let access = match access with Some v -> v | None -> System.IO.FileAccess.ReadWrite
                let share = match share with Some v -> v | None -> System.IO.FileShare.None
                Async<_>.UnblockedPrimitive (fun () -> System.IO.File.Open(path,mode,access,share))

        type System.IO.Stream with


            [<OverloadID("ReadAsync1")>]
            member stream.ReadAsync(buffer: byte[],?offset,?count) =
                let offset = defaultArg offset 0
                let count  = defaultArg count buffer.Length
                Async<_>.BuildPrimitive (buffer,offset,count,stream.BeginRead,stream.EndRead)

            [<OverloadID("ReadAsync2")>]
            member stream.ReadAsync(count) =
                async { let buffer = Array.zero_create count
                        let! n = stream.ReadAsync(buffer,0,count)
                        do! (if n < count then Async<_>.Raise(Failure("ReadAsync: failed to read enough bytes"))  else resultA())
                        return buffer }

            member stream.WriteAsync(buffer:byte[], ?offset:int, ?count:int) =
                let offset = defaultArg offset 0
                let count  = defaultArg count buffer.Length
                Async<_>.BuildPrimitive (buffer,offset,count,stream.BeginWrite,stream.EndWrite)

        type System.Threading.WaitHandle with
            member wh.WaitOneAsync(?millisecondsTimeout:int) =
                let millisecondsTimeout = (match millisecondsTimeout with Some v -> v | None -> -1)
                protectedPrimitive(fun (asyncGroup,cont,econt,ccont) ->

                    let rwh = ref (None : RegisteredWaitHandle option)

                    let cancelHandler =
                        Handler(fun obj (msg:string) ->
                            match !rwh with
                            | None -> ()
                            | Some rwh ->
                                // If we successfully Unregister the operation then
                                // the callBack below will never get called. So we call the cancel
                                // continuation directly in a new work item.
                                let succeeded = rwh.Unregister(wh)
                                if succeeded then
                                    Async.Spawn (async { do (ccont (new OperationCanceledException(msg)) |> unfake) }))

                    rwh := Some(ThreadPool.RegisterWaitForSingleObject
                                  (waitObject=wh,
                                   callBack=WaitOrTimerCallback(fun _ timeOut ->
                                                rwh := None;
                                                asyncGroup.Cancel.RemoveHandler(cancelHandler);
                                                cont (not timeOut) |> ignore),
                                   state=null,
                                   millisecondsTimeOutInterval=millisecondsTimeout,
                                   executeOnlyOnce=true));

                    asyncGroup.Cancel.AddHandler(cancelHandler);
                    FakeUnit)


        type System.IO.StreamReader with

            // TODO: this is not actually asynchronous as yet
            member s.ReadToEndAsync () =
                async { return s.ReadToEnd() }

        type System.Threading.Timer with

            static member SleepAsync(dueTime) =
               Async<_>.Primitive (fun (cont,econt) ->
                   let timer = ref None
                   timer := Some (new Timer((fun _ ->
                                                 // Try to Dispose of the TImer.
                                                 // Note: there is a very slight race here: the System.Threading.Timer time very occasionally
                                                 // calls the callback _before_ the timer object has been recorded anywhere. This makes it difficult to dispose the
                                                 // timer in this situation. In this case we just let the timer be collected by finalization.
                                                (match !timer with
                                                 | None -> ()
                                                 | Some t -> t.Dispose());
                                                // Now we're done, so call the continuation
                                                cont() |> ignore),
                                            null, dueTime=dueTime, period = -1)))

#if NOT_SILVERLIGHT
        type System.Net.WebRequest with
            member x.GetResponseAsync() =
                Async.BuildPrimitive(x.BeginGetResponse, x.EndGetResponse)
#endif


        type Microsoft.FSharp.Control.IChannel<'msg> with
            member x.TryPostAndReply(msgf : (_ -> 'msg), ?timeout, ?exitContext) = 
                use resultCell = new ResultCell<_>()
                let msg = msgf { new IChannel<_> with
                                     member x.Post(reply) =
                                        // Note the ResultCell may have been disposed if the operation
                                        // timed out. In this case RegisterResult drops the result on the floor.
                                        resultCell.RegisterResult(reply) }
                x.Post(msg)
                resultCell.TryWaitForResult(?timeout=timeout, ?exitContext=exitContext) 
                    
            member x.PostAndReply(msgf, ?timeout, ?exitContext) = 
                match x.TryPostSync(msgf,?timeout=timeout,?exitContext=exitContext) with
                | None ->  raise (TimeoutException("IChannel.PostSync timed out"))
                | Some res -> res

            member x.TryPostSync(msgf, ?timeout, ?exitContext) =  x.TryPostAndReply(msgf,?timeout=timeout, ?exitContext=exitContext) 
            member x.PostSync(msgf, ?timeout, ?exitContext) =  x.PostAndReply(msgf,?timeout=timeout, ?exitContext=exitContext) 
                    

namespace Microsoft.FSharp.Control.Mailboxes

    #nowarn "40"

    open System
    open System.Threading
    open System.IO
    open Microsoft.FSharp.Core
    open Microsoft.FSharp.Core.LanguagePrimitives.IntrinsicOperators
    open Microsoft.FSharp.Core.Operators
    open Microsoft.FSharp.Control
    open Microsoft.FSharp.Control.Async
    open Microsoft.FSharp.Collections
    open System.Collections.Generic
    open Microsoft.FSharp.Control.CommonExtensions

    type Mailbox<'msg>() =
        let mutable inbox_store  = null
        let mutable arrivals = new Fifo<'msg>()

        // REVIEW: This event is not being deterministically disposed
        // Mailbox should implement IDisposable.
        let (* use *) pulse = new AutoResetEvent(false)

        member x.inbox = (match inbox_store with null -> inbox_store <- new ResizeArray<'msg>(1) | _ -> ()); inbox_store

        member x.scanArrivalsUnsafe(f) =
            if arrivals.IsEmpty then None
            else let msg = arrivals.Take()
                 match f msg with
                 | None -> x.inbox.Add(msg); x.scanArrivalsUnsafe(f)
                 | res -> res
        // Lock the arrivals queue while we scan that
        member x.scanArrivals(f) = lock arrivals (fun () -> x.scanArrivalsUnsafe(f))

        member x.scanInbox(f,n) =
            match inbox_store with
            | null -> None
            | inbox ->
                if n >= inbox.Count
                then None
                else
                    let msg = inbox.[n]
                    match f msg with
                    | None -> x.scanInbox (f,n+1)
                    | res -> inbox.RemoveAt(n); res

        member x.receiveFromArrivalsUnsafe() =
            if arrivals.IsEmpty then None
            else Some(arrivals.Take())
        member x.receiveFromArrivals() = lock arrivals (fun () -> x.receiveFromArrivalsUnsafe())

        member x.receiveFromInbox() =
            match inbox_store with
            | null -> None
            | inbox ->
                if inbox.Count = 0
                then None
                else
                    let x = inbox.[0]
                    inbox.RemoveAt(0);
                    Some(x)

        member x.Post(msg) =
            lock arrivals (fun () ->
                arrivals.Enqueue(msg);
                pulse.Set() |> ignore)

        member x.TryScan ((f: 'msg -> (Async<'res>) option), timeout) : Async<'res option> =
            let rec scan() =
                async { match x.scanArrivals(f) with
                        | None -> // deschedule and wait for a message! When it comes, rescan the arrivals
                                  let! ok = pulse.WaitOneAsync(timeout)
                                  if ok then return! scan() else return None
                        | Some resP -> let! res = resP
                                       return Some(res) }
            // Look in the inbox first
            async { match x.scanInbox(f,0) with
                    | None  -> return! scan()
                    | Some resP -> let! res = resP
                                   return Some(res) }

        member x.Scan((f: 'msg -> (Async<'res>) option), timeout) =
            async { let! resOpt = x.TryScan(f,timeout)
                    match resOpt with
                    | None -> return raise(TimeoutException("Mailbox.Scan timed out"))
                    | Some res -> return res }


        member x.TryReceive(timeout) =
            let rec processFirstArrival() =
                async { match x.receiveFromArrivals() with
                        | None -> // Have we been notified about a message?
                                  // If so, rescan the arrivals
                                  let! ok = pulse.WaitOneAsync(timeout)
                                  if ok then return! processFirstArrival()
                                  else return None
                        | res -> return res }
            // look in the inbox first
            async { match x.receiveFromInbox() with
                    | None -> return! processFirstArrival()
                    | res -> return res }

        member x.Receive(timeout) =
            async { let! resOpt = x.TryReceive(timeout)
                    match resOpt with
                    | None -> return raise(TimeoutException("Mailbox.Receive timed out"))
                    | Some res -> return res }

        member x.UnsafeContents =
            lock arrivals (fun () -> (Seq.append x.inbox arrivals.UnsafeContents) |> Seq.to_array |> Seq.of_array)


    type MailboxProcessor<'msg>(initialState,?asyncGroup) =
        let mailbox = new Mailbox<'msg>()
        let mutable defaultTimeout = -1

        member x.DefaultTimeout with get() = defaultTimeout and set(v) = defaultTimeout <- v
        member x.UnsafeMessageQueueContents = mailbox.UnsafeContents
        member x.Start() = Async.Spawn(?asyncGroup=asyncGroup,computation=initialState(x))
        member x.Post(msg) = mailbox.Post(msg)
        member x.TryPostAndReply(msgf : (_ -> 'msg), ?timeout, ?exitContext) = 
            let timeout = defaultArg timeout defaultTimeout
            use resultCell = new ResultCell<_>()
            let msg = msgf (AsyncReplyChannel<_>(fun reply ->
                                    // Note the ResultCell may have been disposed if the operation
                                    // timed out. In this case RegisterResult drops the result on the floor.
                                    resultCell.RegisterResult(reply)))
            mailbox.Post(msg)
            resultCell.TryWaitForResult(timeout=timeout, ?exitContext=exitContext) 

        member x.PostAndReply(msgf, ?timeout, ?exitContext) = 
            match x.TryPostAndReply(msgf,?timeout=timeout,?exitContext=exitContext) with
            | None ->  raise (TimeoutException("PostAndReply timed out"))
            | Some res -> res

        member x.PostSync(msgf, ?timeout, ?exitContext) =  x.PostAndReply(msgf,?timeout=timeout, ?exitContext=exitContext) 

        member x.TryPostAndReplyAsync(msgf, ?timeout, ?exitContext:bool) : Async<_> = 
            let timeout = defaultArg timeout defaultTimeout
            let resultCell = new ResultCell<_>()
            let msg = msgf (AsyncReplyChannel<_>(fun reply ->
                                    // Note the ResultCell may have been disposed if the operation
                                    // timed out. In this case RegisterResult drops the result on the floor.
                                    resultCell.RegisterResult(reply)))
            mailbox.Post(msg)
            async { let! ok = resultCell.GetWaitHandle().WaitOneAsync(millisecondsTimeout=timeout)
                    let res = (if ok then Some(resultCell.GrabResult()) else None)
                    do resultCell.Close()
                    return res }

        member x.PostAndReplyAsync(msgf, ?timeout, ?exitContext) =
            let timeout = defaultArg timeout defaultTimeout
            async { let! res = x.TryPostAndReplyAsync(msgf,timeout=timeout,?exitContext=exitContext) 
                    match res with 
                    | None ->  return! raise (TimeoutException("PostAndReplyAsync timed out"))
                    | Some res -> return res }



        member x.Receive(?timeout)    = mailbox.Receive(timeout=defaultArg timeout defaultTimeout)
        member x.TryReceive(?timeout) = mailbox.TryReceive(timeout=defaultArg timeout defaultTimeout)
        member x.Scan(f,?timeout)     = mailbox.Scan(f,timeout=defaultArg timeout defaultTimeout)
        member x.TryScan(f,?timeout)  = mailbox.TryScan(f,timeout=defaultArg timeout defaultTimeout)
        interface IChannel<'msg> with
            member x.Post(msg) = (x : MailboxProcessor<_>).Post(msg)

        static member Create (initialState,?asyncGroup) : MailboxProcessor<'msg> =
            new MailboxProcessor<_>(initialState,?asyncGroup=asyncGroup)

        static member Start(initialState,?asyncGroup) : MailboxProcessor<'msg> =
            let mb = MailboxProcessor.Create(initialState,?asyncGroup=asyncGroup)
            mb.Start();
            mb



#endif

namespace Microsoft.FSharp.Control.SharedMemory

    open System.Threading
    open Microsoft.FSharp.Core

    module Helpers =
        let inline readLock (rwlock : ReaderWriterLock) f  =
          rwlock.AcquireReaderLock(Timeout.Infinite)
          try
              f()
          finally
              rwlock.ReleaseReaderLock()

        let inline writeLock (rwlock : ReaderWriterLock) f  =
          rwlock.AcquireWriterLock(Timeout.Infinite)
          try
              let res = f() in
#if CLI_AT_MOST_1_1
#else
              Thread.MemoryBarrier();
#endif
              res
          finally
              rwlock.ReleaseWriterLock()

