// (c) Microsoft Corporation 2005-2007.

#light

namespace Microsoft.FSharp.Text.StructuredFormat

open System.IO
open System.Reflection
open Microsoft.FSharp.Core
open Microsoft.FSharp.Core.LanguagePrimitives.IntrinsicOperators
open Microsoft.FSharp.Core.Operators
open Microsoft.FSharp.Collections
open Microsoft.FSharp.Compatibility
open Microsoft.FSharp.Reflection
open Microsoft.FSharp.Text
open Microsoft.FSharp.Math
open Microsoft.FSharp.Text.StructuredFormat
open Microsoft.FSharp.Text.StructuredFormat.LayoutOps
open Microsoft.FSharp.Primitives.Basics

/// These are a typical set of options used to control structured formatting.
type FormatOptions = 
    { FloatingPointFormat: string;
      AttributeProcessor: (string -> (string * string) list -> bool -> unit);
      PrintIntercepts: (StructuredFormat.IEnvironment -> obj -> StructuredFormat.Layout option) list;
      FormatProvider: System.IFormatProvider;
      PrintWidth : int; 
      PrintDepth : int; 
      PrintLength : int;
      ShowProperties : bool;
      ShowIEnumerable: bool; }
    static member Default =
        { FormatProvider = (System.Globalization.CultureInfo.InvariantCulture :> System.IFormatProvider);
          PrintIntercepts = [];
          AttributeProcessor= (fun _ _ _ -> ());
          FloatingPointFormat = "g10";
          PrintWidth = 80 ; 
          PrintDepth = 100 ; 
          PrintLength = 100;
          ShowProperties = false;
          ShowIEnumerable = true; }


module Display = 

    let string_of_int (i:int) = i.ToString()
    
    // An implementation of break stack.
    // Uses mutable state, relying on linear threading of the state.
    

    type breaks = 
        Breaks of
            int *     // pos of next free slot 
            int *     // pos of next possible "outer" break - OR - outer=next if none possible 
            int array // stack of savings, -ve means it has been broken   

    // next  is next slot to push into - aka size of current occupied stack.  
    // outer counts up from 0, and is next slot to break if break forced.
    // - if all breaks forced, then outer=next.
    // - popping under these conditions needs to reduce outer and next.
    

    //let dumpBreaks prefix (Breaks(next,outer,stack)) = ()
    //   printf "%s: next=%d outer=%d stack.Length=%d\n" prefix next outer stack.Length;
    //   flush stdout 
         
    let chunkN = 400      
    let breaks0 () = Breaks(0,0,Array.create chunkN 0)

    let pushBreak saving (Breaks(next,outer,stack)) =
        //dumpBreaks "pushBreak" (next,outer,stack);
        let stack = 
            if next = Array.length stack then
              Array.init (next + chunkN) (fun i -> if i < next then stack.(i) else 0) // expand if full 
            else
              stack
       
        stack.(next) <- saving;
        Breaks(next+1,outer,stack)

    let popBreak (Breaks(next,outer,stack)) =
        //dumpBreaks "popBreak" (next,outer,stack);
        if next=0 then raise (Failure "popBreak: underflow");
        let topBroke = stack.(next-1)<0
        let outer = if outer=next then outer-1 else outer  // if all broken, unwind 
        let next  = next - 1
        Breaks(next,outer,stack),topBroke

    let forceBreak (Breaks(next,outer,stack)) =
        //dumpBreaks "forceBreak" (next,outer,stack);
        if outer=next then
          // all broken 
            None
        else
            let saving = stack.(outer)
            stack.(outer) <- -stack.(outer);    
            let outer = outer+1
            Some (Breaks(next,outer,stack),saving)

    // -------------------------------------------------------------------------
    // fitting
    // ------------------------------------------------------------------------
      
    let squashTo (maxWidth,leafFormatter) layout =
        if maxWidth <= 0 then layout else 
        let rec fit breaks (pos,layout) =
            // breaks = break context, can force to get indentation savings.
            // pos    = current position in line
            // layout = to fit
            //------
            // returns:
            // breaks
            // layout - with breaks put in to fit it.
            // pos    - current pos in line = rightmost position of last line of block.
            // offset - width of last line of block
            // NOTE: offset <= pos -- depending on tabbing of last block
           
            let breaks,layout,pos,offset =
                match layout with
                | Attr (tag,attrs,l) ->
                    let breaks,layout,pos,offset = fit breaks (pos,l) 
                    let layout = Attr (tag,attrs,layout) 
                    breaks,layout,pos,offset
                | Leaf (jl,obj,jr) ->
                    let text:string = leafFormatter obj 
                    // save the formatted text from the squash
                    let layout = Leaf(jl,(text :> obj),jr) 
                    let textWidth = text.Length
                    let rec fitLeaf breaks pos =
                      if pos + textWidth <= maxWidth then
                          breaks,layout,pos + textWidth,textWidth // great, it fits 
                      else
                          match forceBreak breaks with
                          | None                 -> 
                              breaks,layout,pos + textWidth,textWidth // tough, no more breaks 
                          | Some (breaks,saving) -> 
                              let pos = pos - saving 
                              fitLeaf breaks pos
                   
                    fitLeaf breaks pos
                | Node (jl,l,jm,r,jr,joint) ->
                    let mid = if jm then 0 else 1
                    match joint with
                    | Unbreakable    ->
                        let breaks,l,pos,offsetl = fit breaks (pos,l)    // fit left 
                        let pos = pos + mid                              // fit space if juxt says so 
                        let breaks,r,pos,offsetr = fit breaks (pos,r)    // fit right 
                        breaks,Node (jl,l,jm,r,jr,Unbreakable),pos,offsetl + mid + offsetr
                    | Broken indent ->
                        let breaks,l,pos,offsetl = fit breaks (pos,l)    // fit left 
                        let pos = pos - offsetl + indent                 // broken so - offset left + ident 
                        let breaks,r,pos,offsetr = fit breaks (pos,r)    // fit right 
                        breaks,Node (jl,l,jm,r,jr,Broken indent),pos,indent + offsetr
                    | Breakable indent ->
                        let breaks,l,pos,offsetl = fit breaks (pos,l)    // fit left 
                        // have a break possibility, with saving 
                        let saving = offsetl + mid - indent
                        let pos = pos + mid
                        if saving>0 then
                            let breaks = pushBreak saving breaks
                            let breaks,r,pos,offsetr = fit breaks (pos,r)
                            let breaks,broken = popBreak breaks
                            if broken then
                                breaks,Node (jl,l,jm,r,jr,Broken indent)   ,pos,indent + offsetr
                            else
                                breaks,Node (jl,l,jm,r,jr,Breakable indent),pos,offsetl + mid + offsetr
                        else
                            // actually no saving so no break 
                            let breaks,r,pos,offsetr = fit breaks (pos,r)
                            breaks,Node (jl,l,jm,r,jr,Breakable indent)  ,pos,offsetl + mid + offsetr
           
           //Printf.printf "\nDone:     pos=%d offset=%d" pos offset;
            breaks,layout,pos,offset
       
        let breaks = breaks0 ()
        let pos = 0
        let breaks,layout,pos,offset = fit breaks (pos,layout)
        layout

    // -------------------------------------------------------------------------
    // showL
    // ------------------------------------------------------------------------

    let showL leafFormatter layout =
        let push x rstrs = x::rstrs
        let z0 = [],0
        let addText (rstrs,i) (text:string) = push text rstrs,i + text.Length
        let index   (rstrs,i)               = i
        let extract rstrs = String.concat "" (List.rev rstrs)
        let newLine (rstrs,i) n     = // \n then spaces... 
            let indent = new System.String(' ', n)
            let rstrs = push "\n"   rstrs
            let rstrs = push indent rstrs
            rstrs,n

        // addL: pos is tab level 
        let rec addL z pos layout = 
            match layout with 
            | Leaf (jl,obj,jr)                 -> 
                let text = leafFormatter obj 
                addText z text
            | Node (jl,l,jm,r,jr,Broken indent) -> 
                let z = addL z pos l
                let z = newLine z (pos+indent)
                let z = addL z (pos+indent) r
                z
            | Node (jl,l,jm,r,jr,_)             -> 
                let z = addL z pos l
                let z = if jm then z else addText z " "
                let pos = index z
                let z = addL z pos r
                z
            | Attr (tag,attrs,l) ->
                addL z pos l
       
        let rstrs,i = addL z0 0 layout
        extract rstrs


    // -------------------------------------------------------------------------
    // outL
    // ------------------------------------------------------------------------

    let outL outAttribute leafFormatter (chan : #TextWriter) layout =
        // write layout to output chan directly 
        let write (s:string) = chan.Write(s)
        // z is just current indent 
        let z0 = 0
        let index i = i
        let addText z text  = write text;  (z + text.Length)
        let newLine z n     = // \n then spaces... 
            let indent = new System.String(' ',n)
            write "\n";
            write indent;
            n
            
        // addL: pos is tab level 
        let rec addL z pos layout = 
            match layout with 
            | Leaf (jl,obj,jr)                 -> 
                let text = leafFormatter obj 
                addText z text
            | Node (jl,l,jm,r,jr,Broken indent) -> 
                let z = addL z pos l
                let z = newLine z (pos+indent)
                let z = addL z (pos+indent) r
                z
            | Node (jl,l,jm,r,jr,_)             -> 
                let z = addL z pos l
                let z = if jm then z else addText z " "
                let pos = index z
                let z = addL z pos r
                z 
            | Attr (tag,attrs,l) ->
            let _ = outAttribute tag attrs true
            let z = addL z pos l
            let _ = outAttribute tag attrs false
            z
       
        let i = addL z0 0 layout
        ()

    // --------------------------------------------------------------------
    // pprinter: using general-purpose reflection...
    // -------------------------------------------------------------------- 
      
    let getValueInfo x = Value.GetInfo x

    let unpackCons recd =
        match recd with 
        | [(_,h);(_,t)] -> (h,t)
        | _             -> failwith "unpackCons"

    let getListValueInfo (x:obj) =
        match x with 
        | null -> None 
        | _ -> 
            match getValueInfo x with
            | ConstructorValue (ty,"Cons",recd) -> Some (unpackCons recd)
            | ConstructorValue (ty,"Nil",[]) -> None
            | _ -> failwith "List value had unexpected ValueInfo"

    let compactCommaListL xs = sepListL (sepL ",") xs // compact, no spaces around "," 
    let nullL = wordL "null"
    let unitL = wordL "()"

    // --------------------------------------------------------------------
    // pprinter: attributes
    // -------------------------------------------------------------------- 

    let consA   x = tagL "Constructor" x
    let stringA x = tagL "String"      x
    let numberA x = tagL "Number"      x
    let listA   x = tagL "List"        x
    let arrayA  x = tagL "Array"       x
    let recordA x = tagL "Record"      x
    let tupleA  x = tagL "Tuple"       x

    let makeTupleL xs =
        (leftL "(" |> tupleA) $$
        sepListL (rightL "," |> tupleA) xs $$
        (rightL ")" |> tupleA)

    let makeRecordL nameXs =
        let itemL (name,xL) = let labelL = wordL name in ((labelL $$ wordL "=") |> recordA) -- (xL  $$ (rightL ";" |> recordA))
        let braceL xs = (leftL "{" |> recordA) $$ xs $$ (rightL "}" |> recordA)
        braceL (aboveListL (List.map itemL nameXs))

    let makePropertiesL nameXs =
        let itemL (name,v) = 
           let labelL = wordL name 
           ((labelL $$ wordL "=") |> recordA)
           $$ (match v with 
               | None -> wordL "?" 
               | Some xL -> xL)
           $$ (rightL ";" |> recordA)
        let braceL xs = (leftL "{" |> recordA) $$ xs $$ (rightL "}" |> recordA)
        braceL (aboveListL (List.map itemL nameXs))

    let makeListL itemLs =
        (leftL "[" |> listA) $$
        sepListL (rightL ";" |> listA) itemLs $$
        (rightL "]" |> listA)

    let makeArrayL xs =
        (leftL "[|" |> arrayA) $$
        sepListL (rightL ";" |> arrayA) xs $$
        (rightL "|]" |> arrayA)

    let makeArray2L xs = leftL "[|" $$ semiListL xs $$ rightL "|]"  

    // --------------------------------------------------------------------
    // pprinter: anyL
    // -------------------------------------------------------------------- 

    let getProperty (obj: obj) name =
        let ty = obj.GetType()
        ty.InvokeMember(name, (BindingFlags.GetProperty ||| BindingFlags.Instance ||| BindingFlags.Public ||| BindingFlags.NonPublic), null, obj, CompatArray.of_list [ ]) 

    let formatChar isChar c = 
        match c with 
        | '\'' when isChar -> "\\\'"
        | '\"' when not isChar -> "\\\""
        | '\n' -> "\\n"
        | '\r' -> "\\r"
        | '\t' -> "\\t"
        | '\\' -> "\\\\"
        | '\b' -> "\\b"
        | _ when System.Char.IsControl(c) -> 
             let c = (Char.code c)
             let d1 = (c / 100) % 10 
             let d2 = (c / 10) % 10 
             let d3 = c % 10 
             "\\" + d1.ToString() + d2.ToString() + d3.ToString()
        | _ -> c.ToString()
        
    let formatString (s:string) = 
        let rec check i = i < s.Length && not (System.Char.IsControl(s,i)) && s.[i] <> '\"' && check (i+1) 
        let rec conv i acc = if i = s.Length then String.concat "" (List.rev acc) else conv (i+1) (formatChar false s.[i] :: acc)  
        "\"" + (if check 0 then s else conv 0 []) + "\""
                                          
    // polymorphic and inner recursion limitations prevent us defining polyL in the recursive loop 
    let polyL (objL: int -> ValueInfo -> obj -> StructuredFormat.Layout) i (x:'a) = 
        objL i (getValueInfo x)  (box x) 

    let anyL (opts:FormatOptions) (x:'a) =
        let path = HashSet.Create(HashIdentity.Reference,10) 
        let rec objL i (x:obj) = polyL objWithReprL i x
        and objWithReprL i (info:ValueInfo) (x:obj) = 
            try
              if i<=0 then wordL "..." else
              match x with 
              | null -> reprL (i-1) info x
              | _    ->
                if (path.Contains(x)) then 
                   wordL "..."
                else 
                    path.Add(x);
                    let env = { new StructuredFormat.IEnvironment 
                                with GetLayout(y) = objL (i-1) y 
                                and  get_MaxColumns() = opts.PrintLength
                                and  get_MaxRows() = opts.PrintLength }
                    let res = List.first (fun intercept -> intercept env x) opts.PrintIntercepts
                    let res = 
                        match res with 
                        | Some res -> res
                        | None     -> 
                        match x with 
                        | :? StructuredFormat.IFormattable as f -> f.GetLayout(env)
                        | _ -> reprL (i-1) info x
                    path .Remove(x);
                    res
            with
              e -> wordL ("Error: " + e.Message)

        and recdAtomicTupleL i recd = 
            match recd with 
            | [t,x] -> objL i x
            | txs   -> bracketL (compactCommaListL (List.map (snd >> objL i) txs))

        and reprL i repr x =
            match repr with 
            | TupleValue vals -> 
                makeTupleL (List.map (objL i) vals)
            | RecordValue items -> 
                let itemL (name,x) = (name,objL i x)
                makeRecordL (List.map itemL items)
            | ConstructorValue (ty,constr,recd) when Type.IsListType (Value.GetType x) -> 
                match constr with 
                | "Cons" -> 
                    let (x,xs) = unpackCons recd
                    let project xs = getListValueInfo xs
                    let itemLs = objL i x :: unfoldL (objL i) project xs (opts.PrintLength - 1)
                    makeListL itemLs
                | _ -> 
                    wordL "[]" |> consA
            | ConstructorValue(ty,nm,[])   -> (wordL nm |> consA)
            | ConstructorValue(ty,nm,recd) -> (wordL nm |> consA) -- recdAtomicTupleL i recd
            | ExceptionValue(ty,recd)      -> wordL ty.Name -- recdAtomicTupleL i recd
            | FunctionClosureValue(ty,obj) -> wordL ("<fun:"+ty.Name+">")
            | ObjectValue(obj)  ->
                match obj with 
                | null           -> nullL
                | _ -> 
                let ty = obj.GetType()
                match obj with 
                | :? string as s -> wordL (formatString s) |> stringA
                | :? System.Array as arr -> 
                    begin match arr.Rank with
                    | 1 -> 
                         let n = arr.Length
                         let b1 = arr.GetLowerBound(0) 
                         let project i = if i=(b1+n) then None else Some (box (arr.GetValue(i)),i+1)
                         let itemLs = unfoldL (objL i) project b1 opts.PrintLength
                         makeArrayL (if b1 = 0 then itemLs else wordL("bound1="+string_of_int b1)::itemLs)
                    | 2 -> 
                         let n1 = arr.GetLength(0)
                         let n2 = arr.GetLength(1)
                         let b1 = arr.GetLowerBound(0) 
                         let b2 = arr.GetLowerBound(1) 
                         let project2 x y =
                           if x>=(b1+n1) || y>=(b2+n2) then None
                           else Some (box (arr.GetValue(x,y)),y+1)
                         let rowL x = unfoldL (objL i) (project2 x) b2 opts.PrintLength |> makeArrayL
                         let project1 x = if x>=(b1+n1) then None else Some (x,x+1)
                         let rowsL  = unfoldL rowL project1 b1 opts.PrintLength
                         makeArray2L (if b1=0 && b2 = 0 then rowsL else wordL("bound1=" + string_of_int b1)::wordL("bound2=" + string_of_int b1)::rowsL)
                      | n -> 
                         makeArrayL [wordL("rank=" + string_of_int n)]
                    end
                | :? System.Collections.IEnumerable as ie when opts.ShowIEnumerable -> 
                     let it = ie.GetEnumerator() 
                     let itemLs = unfoldL (objL i) (fun () -> if it.MoveNext() then Some(it.Current,()) else None) () (1+opts.PrintLength/30)
                     wordL "seq" --- makeListL itemLs
                | _                 -> 
                     let basicL = Microsoft.FSharp.Text.StructuredFormat.LayoutOps.objL obj
                     match obj with 
                     | :? BigNum | :? BigInt -> basicL
                     | _ when opts.ShowProperties ->
                        let props = ty.GetProperties(BindingFlags.GetField ||| BindingFlags.Instance ||| BindingFlags.Public)
                        // massively reign in deep printing of properties 
                        let nDepth = i/10
                        System.Array.Sort((props:>System.Array),{ new System.Collections.IComparer with Compare(p1,p2) = compare ((p1 :?> PropertyInfo).Name) ((p2 :?> PropertyInfo).Name) } );
                        if props.Length = 0 || (nDepth <= 0) then basicL 
                        else basicL --- 
                               (props 
                                |> CompatArray.to_list 
                                |> List.map (fun p -> (p.Name,(try Some (objL nDepth (getProperty obj p.Name)) 
                                                               with _ -> None)))
                                |> makePropertiesL)
                     | _ -> basicL 
            | UnitValue -> unitL
       
        polyL objWithReprL opts.PrintDepth x

    // --------------------------------------------------------------------
    // pprinter: leafFormatter
    // -------------------------------------------------------------------- 
        
    let leafFormatter (opts:FormatOptions) (obj :obj) =
        match obj with 
        | null -> "null"
        | :? double as d -> 
            let s = d.ToString(opts.FloatingPointFormat,opts.FormatProvider)
            if opts.FloatingPointFormat.[0] = 'g'  && String.for_all (fun c -> System.Char.IsDigit(c) || c = '-')  s
            then s + ".0" 
            else s
        | :? single as d -> 
            (if opts.FloatingPointFormat.Length >= 1 && opts.FloatingPointFormat.[0] = 'g' 
              && float32(System.Int32.MinValue) < d && d < float32(System.Int32.MaxValue) 
              && float32(int32(d)) = d 
             then (System.Convert.ToInt32 d).ToString(opts.FormatProvider) + ".0"
             else d.ToString(opts.FloatingPointFormat,opts.FormatProvider)) 
            + "f"
        | :? System.Decimal as d -> d.ToString("g",opts.FormatProvider) + "M"
        | :? uint64 as d -> d.ToString(opts.FormatProvider) + "UL"
        | :? int64  as d -> d.ToString(opts.FormatProvider) + "L"
        | :? int32  as d -> d.ToString(opts.FormatProvider)
        | :? uint32 as d -> d.ToString(opts.FormatProvider) + "u"
        | :? int16  as d -> d.ToString(opts.FormatProvider) + "s"
        | :? uint16 as d -> d.ToString(opts.FormatProvider) + "us"
        | :? sbyte  as d -> d.ToString(opts.FormatProvider) + "y"
        | :? byte   as d -> d.ToString(opts.FormatProvider) + "uy"
        | :? BigNum as d -> d.ToString() + "N"
        | :? BigInt as d -> d.ToString() + "I"
        | :? bool   as b -> (if b then "true" else "false")
        | :? char   as c -> "\'" + formatChar true c + "\'"
        | _ -> obj.ToString()

    let any_to_layout opts x = anyL opts x
    let squash_layout opts l = l |> squashTo (opts.PrintWidth,leafFormatter opts)

    let output_layout opts oc l = 
        l |> squash_layout opts 
          |> outL opts.AttributeProcessor (leafFormatter opts) oc

    let layout_to_string opts l = 
        l |> squash_layout opts 
          |> showL (leafFormatter opts) 

    let output_any_ex opts oc x = x |> any_to_layout opts |> output_layout opts oc
    let output_any oc x = output_any_ex FormatOptions.Default oc x
    let layout_as_string opts x = x |> any_to_layout opts |> layout_to_string opts
    let any_to_string x = layout_as_string FormatOptions.Default x

