(* (c) Microsoft Corporation. All rights reserved *)

(*F# 
module Microsoft.Research.AbstractIL.IL 
open Microsoft.Research.AbstractIL 
open Microsoft.Research.AbstractIL.Internal 
module Illib = Microsoft.Research.AbstractIL.Internal.Library 
module Ildiag = Microsoft.Research.AbstractIL.Diagnostics 
F#*) 

open Illib
open Ildiag
open Nums

type idx = int

let notlazy v = Lazy.lazy_from_val v
type bytes = Bytes.bytes
type ('a,'b) choice = Choice1of2 of 'a | Choice2of2 of 'b
let logging = false let _ = if logging then dprint_endline "*** warning: Il.logging is on"

let deep_eq a b = Pervasives.(=) a b
type doNotUseEqual = UsesOfStructuralEqualityMustBeExplicitInThisFile
let (=) = UsesOfStructuralEqualityMustBeExplicitInThisFile
let (compare) = UsesOfStructuralEqualityMustBeExplicitInThisFile
let (<>) = UsesOfStructuralEqualityMustBeExplicitInThisFile

let isnull x = match x with [] -> true | _ -> false
let notnull x = match x with [] -> false | _ -> true
let int_eq (a:int) b = Pervasives.(=) a b
let int_compare (a:int) b = Pervasives.compare a b
let int32_eq (a:int32) b = Pervasives.(=) a b
let string_eq (a:string) b = Pervasives.(=) a b
(* Yes, this is gross, but the idea is that if a data structure does not *)
(* contain lazy values then we don't add laziness.  So if the thing to map  *)
(* is already evaluated then immediately apply the function.  *)
let lazy_map f x = 
      if Lazy.lazy_is_val x then notlazy (f (Lazy.force x)) else lazy (f (Lazy.force x))

let computed_map f x = (fun () -> f (x()))
let not_computed x = (fun () -> x)
let force_computed f = f()

let popRangeM lo hi (m:('a,'b) Zmap.map) =
  let collect k v (rvs,m) = (v :: rvs) , Zmap.remove k m in
  let rvs,m = Zmap.fold_section lo hi collect m ([],m) in
  List.rev rvs,m
  
(* -------------------------------------------------------------------- 
 * Ordered lists with a lookup table
 * -------------------------------------------------------------------- *)


module LazyOrderedMultiMap = struct
    type ('key,'data) t =  ('key,'data) orderedMultiMap Lazy.t
    (* Ownership conditions mean that this is a defensively-copied functional data structure *)
    and ('key,'data) orderedMultiMap = 
       { ommObjs: 'data array;
         ommMap: ('key,'data) Hashtbl.t }

    let dest_objs ltab = (Lazy.force ltab).ommObjs |> Array.to_list
    let add_obj_to_tab f tab y  =
      let key = f y in 
      Hashtbl.add tab key y 

    let add_obj keyf y ltab = lazy_map (fun omm -> {omm with ommObjs=Array.of_list (y:: Array.to_list omm.ommObjs); 
                                                             ommMap= let t = Hashtbl.copy omm.ommMap in add_obj_to_tab keyf t y; t}) ltab
    let mk_objs' keyf l =  let arr = Array.of_list l in  {ommObjs=arr;ommMap= let t = Hashtbl.create (Array.length arr) in Array.iter (add_obj_to_tab keyf t) arr; t }
    let mk_objs keyf l =  notlazy (mk_objs' keyf l)
    let mk_lazy_objs keyf l =  lazy_map (mk_objs' keyf) l
    let filter_objs keyf f lomm =
       lazy_map (fun omm -> 
          let l = List.filter f (Array.to_list omm.ommObjs) in 
          mk_objs' keyf  l) lomm
    let find_obj x m = let tab = (Lazy.force m).ommMap in Hashtbl.find_all tab x 
end


(* --------------------------------------------------------------------
 * 
 * -------------------------------------------------------------------- *)

type version_info = Nums.u16 * Nums.u16 * Nums.u16 * Nums.u16

type assembly_name = string (* uses exact comparisons.  REVIEW: ECMA Partition 2 is inconsistent about this. *)
type module_name = string (* uses exact comparisons. REVIEW: ECMA Partition 2 is inconsistent about this. *)
type locale = string
type public_key_info = (*F# PublicKey and PublicKey = F#*)
  | PublicKey of bytes
  | PublicKeyToken of bytes
 (*F# with 
        member x.IsKey=match x with PublicKey _ -> true | _ -> false
        member x.IsKeyToken=match x with PublicKeyToken _ -> true | _ -> false
        member x.Key=match x with PublicKey b -> b | _ -> invalid_arg "not a key"
        member x.KeyToken=match x with PublicKeyToken b -> b | _ -> invalid_arg "not a key token"
      end F#*)

type assembly_ref =  (*F# AssemblyRef and AssemblyRef = F#*) 
    { assemRefName: assembly_name;
      assemRefHash: bytes option;
      assemRefPublicKeyInfo: public_key_info option;
      assemRefRetargetable: bool;
      assemRefVersion: version_info option;
      assemRefLocale: locale option; } 
 (*F# with 
        member x.Name=x.assemRefName
        member x.Hash=x.assemRefHash
        member x.PublicKey=x.assemRefPublicKeyInfo
        member x.Retargetable=x.assemRefRetargetable  
        member x.Version=x.assemRefVersion
        member x.Locale=x.assemRefLocale
      end F#*)


type modul_ref = (*F# ModuleRef and ModuleRef = F#*) 
    { modulRefName: module_name;
      modulRefNoMetadata: bool; (* only for file references *)
      modulRefHash: bytes option; (* only for file references *)
    }
 (*F# with 
        member x.Name=x.modulRefName
        member x.HasMetadata=not x.modulRefNoMetadata
        member x.Hash=x.modulRefHash 
      end F#*)

type scope_ref = (*F# ScopeRef and ScopeRef = F#*) 
  | ScopeRef_local
  | ScopeRef_module of modul_ref 
  | ScopeRef_assembly of assembly_ref
 (*F# with 
        static member Local = ScopeRef_local
        static member Module(mref) = ScopeRef_module(mref)
        static member Assembly(aref) = ScopeRef_assembly(aref)
        member x.IsLocalRef=match x with ScopeRef_local -> true | _ -> false
        member x.IsModuleRef=match x with ScopeRef_module _ -> true | _ -> false
        member x.IsAssemblyRef=match x with ScopeRef_assembly _ -> true | _ -> false
        member x.ModuleRef=match x with ScopeRef_module x -> x | _ -> failwith "not a module reference"
        member x.AssemblyRef=match x with ScopeRef_assembly x -> x | _ -> failwith "not an assembly reference"
      end F#*)

(* REVIEW: check the array types that are relevant for binding etc. *)
type array_bound = i32 option 
type array_bounds = array_bound * array_bound
type array_shape = (*F# ArrayShape and ArrayShape = F#*) 
  | ArrayShape of array_bounds list (* lobound/size pairs *)


(* --------------------------------------------------------------------
 * Calling conventions.  These are used in method pointer types.
 * -------------------------------------------------------------------- *)

type basic_callconv = (*F# ArgumentConvention and ArgumentConvention = F#*) 
  | CC_default
  | CC_cdecl 
  | CC_stdcall 
  | CC_thiscall 
  | CC_fastcall 
  | CC_vararg
      
type hasthis =  (*F# ThisConvention and ThisConvention = F#*) 
  | CC_instance
  | CC_instance_explicit
  | CC_static

type callconv = (*F# CallingConvention and CallingConvention = F#*) 
  | Callconv of hasthis * basic_callconv
 (*F# with 
        member x.ThisConv           = let (Callconv(a,b)) = x in a
        member x.BasicConv          = let (Callconv(a,b)) = x in b
        member x.IsInstance         = match x.ThisConv with CC_instance -> true | _ -> false
        member x.IsInstanceExplicit = match x.ThisConv with CC_instance_explicit -> true | _ -> false
        member x.IsStatic           = match x.ThisConv with CC_static -> true | _ -> false
      end F#*)

let callconv_eq (a:callconv) b = Pervasives.(=) a b

type type_ref = (*F# TypeRef and TypeRef = F#*) 
   { trefScope: scope_ref;     
     trefNested: string list;  
     trefName: string }        
 (*F# with 
        /// Where is the type, i.e. is it in this module, in another module in this assembly or in another assembly? 
        member x.Scope=x.trefScope
        ///Names of any enclosing types
        member x.Nesting=x.trefNested
        /// The name of the type.  This also contains the namespace is trefNested is empty
        member x.Name=x.trefName
      end F#*)

type type_spec = (*F# TypeSpec and TypeSpec = F#*)  
   { tspecTypeRef: type_ref;    (* Which type is being referred to? *)
     tspecInst: genactuals }    (* The type instantiation if the type is generic *)
 (*F# with 
        member x.TypeRef=x.tspecTypeRef
        member x.GenericArguments=x.tspecInst
      end F#*)

and typ = (*F# TypeSig and TypeSig = F#*)  
  | Type_void                   
  | Type_array of array_shape * typ 
  | Type_value of type_spec      
  | Type_boxed of type_spec      
  | Type_ptr of typ             
  | Type_byref of typ           
  | Type_fptr of callsig 
  | Type_tyvar of u16              
  | Type_modified of bool * type_ref * typ
  | Type_other of ext_typ
and ext_typ = Ext_typ of Obj.t

and callsig =  (*F# CallingSignature and CallingSignature = F#*)  
  { callsigCallconv: callconv;
    callsigArgs: typ list;
    callsigReturn: typ }

(* ----------------------------------------------------------
 * Generic parameters, i.e. parameters reified statically. 
 * Currently only two kinds of parameters are in  
 * the term structure: types and type representations. 
 * Type representations are only used internally.
 * --------------------------------------------------------- *)

and genparams = genparam list
and genactuals = genactual list
and genactual = typ
and genvariance = (*F# GenericVarianceSpec and GenericVarianceSpec = F#*)  
  | NonVariant            
  | CoVariant             
  | ContraVariant         

and genparam =  (*F# GenericParameterDef and GenericParameterDef = F#*)  
   { gpName: string;
     gpConstraints: typ list;
     gpVariance: genvariance; (* Variance of type parameters, only applicable to generic parameters for generic interfaces and delegates *)

     gpReferenceTypeConstraint: bool;     (* type argument must be a reference type *)
     gpNotNullableValueTypeConstraint: bool;  (* type argument must be a value type but not Nullable *)
     gpDefaultConstructorConstraint: bool; (* type argument must have a public nullary constructor *)
  }



type method_ref =  (*F# MethodRef and MethodRef = F#*)  
   { mrefParent: type_ref;
     mrefCallconv: callconv;
     mrefArity: int; 
     mrefName: string;
     mrefArgs: typ list;
     mrefReturn: typ }
(*F# with 
       member x.EnclosingTypeRef = x.mrefParent
       member x.CallingConvention = x.mrefCallconv
       member x.Name = x.mrefName
       member x.GenericArity = x.mrefArity
       member x.ArgumentTypes = x.mrefArgs
       member x.ReturnType = x.mrefReturn
     end F#*)

type field_ref = (*F# FieldRef and FieldRef = F#*)  
   { frefParent: type_ref;
     frefName: string;
     frefType: typ }
(*F# with 
       member x.EnclosingTypeRef = x.frefParent
       member x.Name = x.frefName
       member x.Type = x.frefType
     end F#*)

type method_spec = (*F# MethodSpec and MethodSpec = F#*)  
    { mspecMethodRefF: method_ref;
      mspecEnclosingTypeF: typ;          (* MS-GENERICS *) 
      mspecMethodInstF: genactuals;
      mspecOptionalID: idx option }     (* MS-GENERICS *) 
  (*F# with 
         static member Create(a,b,c) = { mspecOptionalID=None;mspecEnclosingTypeF=a; mspecMethodRefF =b; mspecMethodInstF=c }
         member x.MethodRef = x.mspecMethodRefF
         member x.EnclosingType=x.mspecEnclosingTypeF
         member x.GenericArguments=x.mspecMethodInstF
       end F#*)
let dest_mspec x = (x.mspecMethodRefF, x.mspecEnclosingTypeF, x.mspecMethodInstF)

type field_spec =    (*F# FieldSpec and FieldSpec = F#*)  
    { fspecFieldRef: field_ref;
      fspecEnclosingType: typ }         (* MS-GENERICS *) 
  (*F# with 
         member x.FieldRef = x.fspecFieldRef
         member x.EnclosingType=x.fspecEnclosingType
       end F#*)


(* --------------------------------------------------------------------
 * Debug info.                                                     
 * -------------------------------------------------------------------- *)

type guid =  bytes
type source_document =  (*F# SourceDocument and SourceDocument = F#*) 
    { sourceLanguage: guid option; 
      sourceVendor: guid option;
      sourceDocType: guid option;
      sourceFile: string; }
 (*F# with 
        member x.Language=x.sourceLanguage
        member x.Vendor=x.sourceVendor
        member x.DocumentType=x.sourceDocType
        member x.File=x.sourceFile
      end F#*)

type source =  (*F# SourceMarker and SourceMarker = F#*) 
    { sourceDocument: source_document;
      sourceLine: int;
      sourceColumn: int;
      sourceEndLine: int;
      sourceEndColumn: int }
 (*F# with 
        member x.Document=x.sourceDocument
        member x.Line=x.sourceLine
        member x.Column=x.sourceColumn
        member x.EndLine=x.sourceEndLine
        member x.EndColumn=x.sourceEndColumn
      end F#*)

(* --------------------------------------------------------------------
 * Custom attributes                                                     
 * -------------------------------------------------------------------- *)

type custom_attr_elem =  (*F# AttributeElement and AttributeElement = F#*)   
  | CustomElem_string of string  option
  | CustomElem_bool of bool
  | CustomElem_char of unichar
  | CustomElem_int8 of i8
  | CustomElem_int16 of i16
  | CustomElem_int32 of i32
  | CustomElem_int64 of i64
  | CustomElem_uint8 of u8
  | CustomElem_uint16 of u16
  | CustomElem_uint32 of u32
  | CustomElem_uint64 of u64
  | CustomElem_float32 of ieee32
  | CustomElem_float64 of ieee64
  | CustomElem_type of type_ref  
  | CustomElem_array of custom_attr_elem list  

type custom_attr_named_arg =  (string * typ * bool * custom_attr_elem)
type custom_attr = (*F# Attribute and Attribute = F#*)   
    { customMethod: method_spec;
      customData: bytes }

type custom_attrs = (*F# Attributes and Attributes = F#*)   
   CustomAttrs of (unit -> custom_attr list) 

type code_label = int
type data_label = string * int32

(* --------------------------------------------------------------------
 * Instruction set.                                                     
 * -------------------------------------------------------------------- *)

type basic_type = (*F# BasicType and BasicType = F#*)  
  | DT_R
  | DT_I1
  | DT_U1
  | DT_I2
  | DT_U2
  | DT_I4
  | DT_U4
  | DT_I8
  | DT_U8
  | DT_R4
  | DT_R8
  | DT_I
  | DT_U
  | DT_REF

type ldtoken_info = (*F# TokenSpec and TokenSpec = F#*)  
  | Token_type of typ 
  | Token_method of method_spec 
  | Token_field of field_spec

type ldc_info = (*F# ConstSpec and ConstSpec = F#*)  
  | NUM_I4 of i32
  | NUM_I8 of i64
  | NUM_R4 of ieee32
  | NUM_R8 of ieee64

type tailness = (*F# TailcallSpec and TailcallSpec = F#*)  
  | Tailcall
  | Normalcall
type alignment =  (*F# AlignmentSpec and AlignmentSpec = F#*)  
  | Aligned
  | Unaligned_1
  | Unaligned_2
  | Unaligned_4
type volatility =  (*F# VolatilitySpec and VolatilitySpec = F#*)   
  | Volatile
  | Nonvolatile
type readonlyByref =  (*F# ReadonlySpec and ReadonlySpec = F#*)   
  | ReadonlyAddress
  | NormalAddress

type varargs = typ list option

type comparison = (*F# CompareOp and CompareOp = F#*)   
  | BI_beq        
  | BI_bge        
  | BI_bge_un     
  | BI_bgt        
  | BI_bgt_un        
  | BI_ble        
  | BI_ble_un        
  | BI_blt        
  | BI_blt_un 
  | BI_bne_un 
  | BI_brfalse 
  | BI_brtrue 

type arithmetic = (*F# ArithOp and ArithOp = F#*)   
  | AI_add    
  | AI_add_ovf
  | AI_add_ovf_un
  | AI_and    
  | AI_div   
  | AI_div_un
  | AI_ceq      
  | AI_cgt      
  | AI_cgt_un   
  | AI_clt     
  | AI_clt_un  
  | AI_conv      of basic_type
  | AI_conv_ovf  of basic_type
  | AI_conv_ovf_un  of basic_type
  | AI_mul       
  | AI_mul_ovf   
  | AI_mul_ovf_un
  | AI_rem       
  | AI_rem_un       
  | AI_shl       
  | AI_shr       
  | AI_shr_un
  | AI_sub       
  | AI_sub_ovf   
  | AI_sub_ovf_un   
  | AI_xor       
  | AI_or        
  | AI_neg       
  | AI_not       
  | AI_ldnull    
  | AI_dup       
  | AI_pop
  | AI_ckfinite 
  | AI_nop
  | AI_ldc of basic_type * ldc_info 


type instr =  (*F# Instr and Instr = F#*)   
  | I_arith of arithmetic
  | I_ldarg     of u16
  | I_ldarga    of u16
  | I_ldind     of alignment * volatility * basic_type
  | I_ldloc     of u16
  | I_ldloca    of u16
  | I_starg     of u16
  | I_stind     of  alignment * volatility * basic_type
  | I_stloc     of u16

  | I_br    of  code_label
  | I_jmp   of method_spec
  | I_brcmp of comparison * code_label * code_label (* second label is fall-through *)
  | I_switch    of (code_label list * code_label) (* last label is fallthrough *)
  | I_ret 

  | I_call     of tailness * method_spec * varargs
  | I_callvirt of tailness * method_spec * varargs
  | I_callconstraint of tailness * typ * method_spec * varargs
  | I_calli    of tailness * callsig * varargs
  | I_ldftn    of method_spec
  | I_newobj  of method_spec  * varargs
  
  | I_throw
  | I_endfinally
  | I_endfilter
  | I_leave     of  code_label

  | I_ldsfld      of volatility * field_spec
  | I_ldfld       of alignment * volatility * field_spec
  | I_ldsflda     of field_spec
  | I_ldflda      of field_spec 
  | I_stsfld      of volatility  *  field_spec
  | I_stfld       of alignment * volatility * field_spec
  | I_ldstr       of bytes (* unicode encoding of the string! *)
  | I_isinst      of typ
  | I_castclass   of typ
  | I_ldtoken     of ldtoken_info
  | I_ldvirtftn   of method_spec

  | I_cpobj       of typ
  | I_initobj     of typ
  | I_ldobj       of alignment * volatility * typ
  | I_stobj       of alignment * volatility * typ
  | I_box         of typ
  | I_unbox       of typ
  | I_unbox_any   of typ
  | I_sizeof      of typ

  | I_ldelem      of basic_type
  | I_stelem      of basic_type
  | I_ldelema     of readonlyByref * array_shape * typ
  | I_ldelem_any  of array_shape * typ
  | I_stelem_any  of array_shape * typ
  | I_newarr      of array_shape * typ 
  | I_ldlen

  | I_mkrefany    of typ
  | I_refanytype  
  | I_refanyval   of typ
  | I_rethrow

  | I_break 
  | I_seqpoint of source

  | I_arglist  

  | I_localloc
  | I_cpblk of alignment * volatility
  | I_initblk of alignment  * volatility

  (* FOR EXTENSIONS, e.g. MS-ILX *)  
  | I_other    of ext_instr

and ext_instr = Ext_instr of Obj.t

type local_debug_info = (*F# DebugMapping and DebugMapping = F#*)   
    { localNum: int;
      localName: string; }

type basic_block = (*F# BasicBlock and BasicBlock = F#*)   
    { bblockLabel: code_label;
      bblockInstrs: instr array }

type code = (*F# Code and Code = F#*)   
  | BasicBlock of basic_block
  | GroupBlock of local_debug_info list * code list
  | RestrictBlock of code_label list * code
  | TryBlock of code * seh
and seh = (*F# ExceptionBlock and ExceptionBlock = F#*)   
  | FaultBlock of code 
  | FinallyBlock of code
  | FilterCatchBlock of (filter * code) list
and filter = (*F# FilterBlock and FilterBlock = F#*)   
  | TypeFilter of typ
  | CodeFilter of code

type local (*F# = Local and Local F#*)   
  = { localType: typ;
      localPinned: bool }
      
type il_method_body (*F# = ILMethodBody and ILMethodBody F#*)   
  = { ilZeroInit: bool;
      ilMaxStack: i32;
      ilNoInlining: bool;
      ilLocals: local list;
      ilCode:  code;
      ilSource: source option }

type member_access = (*F# MemberAccess and MemberAccess = F#*)   
  | MemAccess_assembly
  | MemAccess_compilercontrolled
  | MemAccess_famandassem
  | MemAccess_famorassem
  | MemAccess_family
  | MemAccess_private 
  | MemAccess_public 

and field_init = (*F# FieldInit and FieldInit = F#*)   
  | FieldInit_bytes of bytes
  | FieldInit_bool of bool
  | FieldInit_char of u16
  | FieldInit_int8 of i8
  | FieldInit_int16 of i16
  | FieldInit_int32 of i32
  | FieldInit_int64 of i64
  | FieldInit_uint8 of u8
  | FieldInit_uint16 of u16
  | FieldInit_uint32 of u32
  | FieldInit_uint64 of u64
  | FieldInit_float32 of ieee32
  | FieldInit_float64 of ieee64
  | FieldInit_ref
  
let locals_of_ilmbody m = m.ilLocals

let code_of_ilmbody m = m.ilCode
  

(* -------------------------------------------------------------------- 
 * Native Types, for marshalling to the native C interface.
 * These are taken directly from the ILASM syntax, and don't really
 * correspond yet to the ECMA Spec (Partition II, 7.4).  
 * -------------------------------------------------------------------- *)

type native_type = (*F# NativeType and NativeType = F#*)   
  | NativeType_empty
  | NativeType_custom of bytes * string * string * bytes (* guid,nativeTypeName,custMarshallerName,cookieString *)
  | NativeType_fixed_sysstring of i32
  | NativeType_fixed_array of i32
  | NativeType_currency
  | NativeType_lpstr
  | NativeType_lpwstr
  | NativeType_lptstr
  | NativeType_byvalstr
  | NativeType_tbstr
  | NativeType_lpstruct
  | NativeType_struct
  | NativeType_void
  | NativeType_bool
  | NativeType_int8
  | NativeType_int16
  | NativeType_int32
  | NativeType_int64
  | NativeType_float32
  | NativeType_float64
  | NativeType_unsigned_int8
  | NativeType_unsigned_int16
  | NativeType_unsigned_int32
  | NativeType_unsigned_int64
  | NativeType_array of native_type option * (i32 * i32 option) option (* optional idx of parameter giving size plus optional additive i.e. num elems *)
  | NativeType_int
  | NativeType_unsigned_int
  | NativeType_method
  | NativeType_as_any
  | (* COM interop *) NativeType_bstr
  | (* COM interop *) NativeType_iunknown
  | (* COM interop *) NativeType_idsipatch
  | (* COM interop *) NativeType_interface
  | (* COM interop *) NativeType_error               
  | (* COM interop *) NativeType_safe_array of variant_type * string option 
  | (* COM interop *) NativeType_ansi_bstr
  | (* COM interop *) NativeType_variant_bool


and variant_type = (*F# VariantType and VariantType = F#*)   
  | VariantType_empty
  | VariantType_null
  | VariantType_variant
  | VariantType_currency
  | VariantType_decimal               
  | VariantType_date               
  | VariantType_bstr               
  | VariantType_lpstr               
  | VariantType_lpwstr               
  | VariantType_iunknown               
  | VariantType_idispatch               
  | VariantType_safearray               
  | VariantType_error               
  | VariantType_hresult               
  | VariantType_carray               
  | VariantType_userdefined               
  | VariantType_record               
  | VariantType_filetime
  | VariantType_blob               
  | VariantType_stream               
  | VariantType_storage               
  | VariantType_streamed_object               
  | VariantType_stored_object               
  | VariantType_blob_object               
  | VariantType_cf                
  | VariantType_clsid
  | VariantType_void 
  | VariantType_bool
  | VariantType_int8
  | VariantType_int16                
  | VariantType_int32                
  | VariantType_int64                
  | VariantType_float32                
  | VariantType_float64                
  | VariantType_unsigned_int8                
  | VariantType_unsigned_int16                
  | VariantType_unsigned_int32                
  | VariantType_unsigned_int64                
  | VariantType_ptr                
  | VariantType_array of variant_type                
  | VariantType_vector of variant_type                
  | VariantType_byref of variant_type                
  | VariantType_int                
  | VariantType_unsigned_int                

and security_action = (*F# SecurityAction and SecurityAction = F#*)  
  | SecAction_request 
  | SecAction_demand
  | SecAction_assert
  | SecAction_deny
  | SecAction_permitonly
  | SecAction_linkcheck 
  | SecAction_inheritcheck
  | SecAction_reqmin
  | SecAction_reqopt
  | SecAction_reqrefuse
  | SecAction_prejitgrant
  | SecAction_prejitdeny
  | SecAction_noncasdemand
  | SecAction_noncaslinkdemand
  | SecAction_noncasinheritance
  | SecAction_linkdemandchoice
  | SecAction_inheritancedemandchoice
  | SecAction_demandchoice

and permission_value = (*F# PermissionValue and PermissionValue = F#*)  
    PermissionValue_bool of bool
  | PermissionValue_int32 of i32
  | PermissionValue_string of string
  | PermissionValue_enum_int8 of type_ref * i8
  | PermissionValue_enum_int16 of type_ref * i16
  | PermissionValue_enum_int32 of type_ref * i32

(* REVIEW: factor this differently - ILDASM currently doesn't expose *)
(* the contents of a security permission set in detail *)
and permission = (*F# Permission and Permission = F#*)  
    Permission of security_action * typ * (string * permission_value) list
  | PermissionSet of security_action * bytes

and security_decls (*F# = Permissions and Permissions F#*)  
  = SecurityDecls of (permission list) Lazy.t

and pinvoke_method (*F# = PInvokeMethod and PInvokeMethod F#*)  
  = { pinvokeWhere: modul_ref;
      pinvokeName: string;
      pinvokeCallconv: pinvokeCallConv;
      pinvokeEncoding: pinvokeEncoding;
      pinvokeNoMangle: bool;
      pinvokeLastErr: bool;
      pinvokeThrowOnUnmappableChar: pinvokeThrowOnUnmappableChar;
      pinvokeBestFit: pinvokeBestFit }
and pinvokeBestFit  (*F# = PInvokeCharBestFit and PInvokeCharBestFit F#*) =
  | PInvokeBestFitUseAssem
  | PInvokeBestFitEnabled
  | PInvokeBestFitDisabled

and pinvokeThrowOnUnmappableChar (*F# = PInvokeThrowOnUnmappableChar and PInvokeThrowOnUnmappableChar F#*) =
  | PInvokeThrowOnUnmappableCharUseAssem
  | PInvokeThrowOnUnmappableCharEnabled
  | PInvokeThrowOnUnmappableCharDisabled

and pinvokeCallConv  (*F# = PInvokeCallingConvention and PInvokeCallingConvention F#*)  = 
  | PInvokeCallConvNone
  | PInvokeCallConvCdecl
  | PInvokeCallConvStdcall
  | PInvokeCallConvThiscall
  | PInvokeCallConvFastcall
  | PInvokeCallConvWinapi

and pinvokeEncoding (*F# = PInvokeCharEncoding and PInvokeCharEncoding F#*)  = 
  | PInvokeEncodingNone
  | PInvokeEncodingAnsi
  | PInvokeEncodingUnicode
  | PInvokeEncodingAuto

(* Default values for parameters and ret. val.  Metadata Only. *)
and param (*F# = Parameter and Parameter F#*)  
  = { paramName: string option;
      paramType: typ;
      paramDefault: field_init option;  
      paramMarshal: native_type option; (* Marshalling map for parameters. COM Interop only. *)
      paramIn: bool;
      paramOut: bool;
      paramOptional: bool;
      paramCustomAttrs: custom_attrs }


type returnv = (*F# ReturnValue and ReturnValue = F#*)  
    { returnMarshal: native_type option;
      returnType: typ;
      returnCustomAttrs: custom_attrs ; }

type overrides_spec  (*F# = OverridesSpec and OverridesSpec F#*) = OverridesSpec of method_ref * typ

type virtual_info (*F# = MethodVirtualInfo and MethodVirtualInfo F#*) =   
    { virtFinal: bool; 
      virtNewslot: bool; 
      virtStrict: bool; (* mdCheckAccessOnOverride *)
      virtAbstract: bool;
       virtOverrides: overrides_spec option;
      (* virtOverrides is the method this method_def overrides *)
      (* None means "unknown" and can be used when the signature *)
      (* of the method being overriden is identical to the *)
      (* signature of the method that does the overriding. *)
    }

type method_kind (*F# = MethodKind and MethodKind F#*) =  
  | MethodKind_static 
  | MethodKind_cctor 
  | MethodKind_ctor 
  | MethodKind_nonvirtual 
  | MethodKind_virtual of virtual_info

type method_body_details (*F# = MethodBody and MethodBody F#*)   = 
  | MethodBody_il of il_method_body
  | MethodBody_pinvoke of pinvoke_method       (* platform invoke to native  *)
  | MethodBody_abstract
  | MethodBody_native

type method_body (*F# = LazyMethodBody and LazyMethodBody F#*) = LazyMethodBody of method_body_details Lazy.t

type method_code_kind (*F# = MethodCodeKind and MethodCodeKind F#*) =
  | MethodCodeKind_il
  | MethodCodeKind_native
  | MethodCodeKind_runtime

let mk_mbody mb = LazyMethodBody (notlazy mb)
let dest_mbody (LazyMethodBody mb) = Lazy.force mb
let mk_lazy_mbody mb = LazyMethodBody mb

type method_def = (*F# MethodDef and MethodDef = F#*)   
    { mdName: string;
      mdKind: method_kind;
      mdCallconv: callconv;
      mdParams: param list;
      mdReturn: returnv;
      mdAccess: member_access;
      mdBody: method_body;   
      mdCodeKind: method_code_kind;   
      mdInternalCall: bool;
      mdManaged: bool;
      mdForwardRef: bool;
      mdSecurityDecls: security_decls;
      mdHasSecurity: bool;
      mdEntrypoint:bool;
      (* Flags and other stuff of little interest *)
      mdReqSecObj: bool;
      mdHideBySig: bool;
      mdSpecialName: bool;
      mdUnmanagedExport: bool; (* -- The method is exported to unmanaged code *)
                               (*    using COM interop. *)
      mdSynchronized: bool;
      mdPreserveSig: bool;
      mdMustRun: bool; (* Whidbey feature: SafeHandle finalizer must be run *)
      mdExport: (i32 * string option) option;
      mdVtableEntry: (i32 * i32) option;
     
      mdGenericParams: genparams;
      mdCustomAttrs: custom_attrs; 
  }

(* Index table by name and arity. *)
type methods (*F# = MethodDefs and MethodDefs F#*)   
  = Methods of (method_def list * methods_map) Lazy.t
and methods_map = (string, method_def list) Pmap.t

type event_def (*F# = EventDef and EventDef F#*) = 
    { eventType: typ option; 
      eventName: string;
      eventRTSpecialName: bool;
      eventSpecialName: bool;
      eventAddOn: method_ref; 
      eventRemoveOn: method_ref;
      eventFire: method_ref option;
      eventOther: method_ref list;
      eventCustomAttrs: custom_attrs; }

(* Index table by name. *)
type events (*F# = EventDefs and EventDefs F#*) = Events of (string, event_def) LazyOrderedMultiMap.t

type property_def (*F# = PropertyDef and PropertyDef F#*) 
  = { propName: string;
      propRTSpecialName: bool;
      propSpecialName: bool;
      propSet: method_ref option;
      propGet: method_ref option;
      propCallconv: hasthis;
      propType: typ;
      propInit: field_init option;
      propArgs: typ list;
      propCustomAttrs: custom_attrs; }

(* Index table by name. *)
type properties (*F# = PropertyDefs and PropertyDefs F#*)= Properties of (string, property_def) LazyOrderedMultiMap.t

type field_def (*F# = FieldDef and FieldDef F#*)   
  = { fdName: string;
      fdType: typ;
      fdStatic: bool;
      fdAccess: member_access;
      fdData:  bytes option;
      fdInit:  field_init option;
      fdOffset:  i32 option; (* -- the explicit offset in bytes *)
      fdSpecialName: bool;
      fdMarshal: native_type option; 
      fdNotSerialized: bool;
      fdLiteral: bool ;
      fdInitOnly: bool;
      fdCustomAttrs: custom_attrs; }
(*F# with 
      member x.Name = x.fdName
      member x.Type = x.fdType
      member x.IsStatic = x.fdStatic
      member x.Access = x.fdAccess
      member x.Data = x.fdData
      member x.LiteralValue = x.fdInit
      /// The explicit offset in bytes when explicit layout is used.
      member x.Offset = x.fdOffset
      member x.Marshal = x.fdMarshal
      member x.NotSerialized = x.fdNotSerialized
      member x.IsLiteral = x.fdLiteral
      member x.IsInitOnly = x.fdInitOnly
      member x.CustomAttrs = x.fdCustomAttrs
     end F#*)


(* Index table by name.  Keep a canonical list to make sure field order is not disturbed for binary manipulation. *)
type fields (*F# = FieldDefs and FieldDefs F#*) 
   = Fields of (string, field_def) LazyOrderedMultiMap.t

type method_impl (*F# = MethodImplDef and MethodImplDef F#*) 
  = { mimplOverrides: overrides_spec;
      mimplOverrideBy: method_spec }

(* Index table by name and arity. *)
type method_impls (*F# = MethodImplDefs and MethodImplDefs F#*) 
  = MethodImpls of method_impls_map Lazy.t
and method_impls_map = (string * int, method_impl list) Pmap.t

type type_layout (*F# = TypeDefLayout and TypeDefLayout F#*)=
  | TypeLayout_auto
  | TypeLayout_sequential of type_layout_info
  | TypeLayout_explicit of type_layout_info (* REVIEW: add field info here *)

and type_layout_info  (*F# = TypeDefLayoutInfo and TypeDefLayoutInfo F#*)=
    { typeSize: i32 option;
      typePack: u16 option } 

type type_init (*F# = TypeDefInitSemantics and TypeDefInitSemantics F#*)=
  | TypeInit_beforefield
  | TypeInit_beforeany

type type_encoding  (*F# = TypeDefDefaultPInvokeEncoding and TypeDefDefaultPInvokeEncoding F#*)= 
  | TypeEncoding_ansi
  | TypeEncoding_autochar
  | TypeEncoding_unicode

type type_access (*F# = TypeDefAccess and TypeDefAccess F#*)=  
  | TypeAccess_public 
  | TypeAccess_private
  | TypeAccess_nested of member_access 

type type_def_kind (*F# = TypeDefKind and TypeDefKind F#*)=  
  | TypeDef_class
  | TypeDef_valuetype
  | TypeDef_interface
  | TypeDef_enum 
  | TypeDef_delegate
(* EXTENSIONS, e.g.  *)  | TypeDef_other of ext_type_def_kind
and ext_type_def_kind = Ext_type_def_kind of Obj.t


and type_def =  (*F# TypeDef and TypeDef = F#*)   
    { tdKind: type_def_kind;
      tdName: string;  
      tdGenericParams: genparams;   (* class is generic *)
      tdAccess: type_access;  
      tdAbstract: bool;
      tdSealed: bool; 
      tdSerializable: bool; 
      tdComInterop: bool; (* Class or interface generated for COM interop *) 
      tdLayout: type_layout;
      tdSpecialName: bool;
      tdEncoding: type_encoding;
      tdNested: types;
      tdImplements: typ list;  
      tdExtends: typ option; 
      tdMethodDefs: methods;
      tdSecurityDecls: security_decls;
      tdHasSecurity: bool;
      tdFieldDefs: fields;
      tdMethodImpls: method_impls;
      tdInitSemantics: type_init;
      tdEvents: events;
      tdProperties: properties;
      tdCustomAttrs: custom_attrs; }
 (*F#    with
      member x.IsClass=     (match x.tdKind with TypeDef_class -> true | _ -> false)
      member x.IsValueType= (match x.tdKind with TypeDef_valuetype -> true | _ -> false)
      member x.IsInterface= (match x.tdKind with TypeDef_interface -> true | _ -> false)
      member x.IsEnum=      (match x.tdKind with TypeDef_enum -> true | _ -> false)
      member x.IsDelegate=  (match x.tdKind with TypeDef_delegate -> true | _ -> false)
      member x.Name = x.tdName
      member x.GenericParams = x.tdGenericParams
      member x.Access = x.tdAccess
      member x.IsAbstract = x.tdAbstract
      member x.Sealed = x.tdSealed
      member x.IsSerializable = x.tdSerializable
      member x.IsComInterop = x.tdComInterop
      member x.Layout = x.tdLayout
      member x.IsSpecialName = x.tdSpecialName
      member x.Encoding = x.tdEncoding
      member x.NestedTypes = x.tdNested
      member x.Implements = x.tdImplements
      member x.Extends = x.tdExtends
      member x.Methods = x.tdMethodDefs
      member x.SecurityDecls = x.tdSecurityDecls
      member x.HasSecurity = x.tdHasSecurity
      member x.Fields = x.tdFieldDefs
      member x.MethodImpls = x.tdMethodImpls
      member x.InitSemantics = x.tdInitSemantics
      member x.Events = x.tdEvents
      member x.Properties = x.tdProperties
      member x.CustomAttrs = x.tdCustomAttrs
    end F#*)

and types (*F# = TypeDefs and TypeDefs F#*) 
  = TypeDefTable of (string list * string * custom_attrs * type_def Lazy.t) array Lazy.t * types_map Lazy.t
and types_map = 
   (* keyed first on namespace then on type name.  The namespace is often a unique key for a given type map. *)
   (string list,(string,type_def Lazy.t)Hashtbl.t) Pmap.t

and namespace_and_typename = string list * string



type nested_exported_type (*F# = NestedExportedType and NestedExportedType F#*) = 
    { nestedExportedTypeName: string;
      nestedExportedTypeAccess: member_access;
      nestedExportedTypeNested: nested_exported_types;
      nestedExportedTypeCustomAttrs: custom_attrs } 

and nested_exported_types (*F# = NestedExportedTypes and NestedExportedTypes F#*)  = NestedExportedTypes of nested_exported_types_map Lazy.t
and nested_exported_types_map = (string,nested_exported_type) Pmap.t

and exported_type (*F# = ExportedType and ExportedType F#*) = 
    { exportedTypeScope: scope_ref;
      exportedTypeName: string;
      exportedTypeForwarder: bool;
      exportedTypeAccess: type_access;
      exportedTypeNested: nested_exported_types;
      exportedTypeCustomAttrs: custom_attrs } 

and exported_types (*F# = ExportedTypes and ExportedTypes F#*) = ExportedTypes of exported_types_map Lazy.t
and exported_types_map = (string,exported_type) Pmap.t


type resource_access (*F# = ResourceAccess and ResourceAccess F#*) = 
  | Resource_public 
  | Resource_private 
type resource_where (*F# = ResourceLocation and ResourceLocation F#*)  = 
  | Resource_local of (unit -> bytes)
  | Resource_file of modul_ref * int32
  | Resource_assembly of assembly_ref

type resource (*F# = Resource and Resource F#*)   = 
    { resourceName: string;
      resourceWhere: resource_where;
      resourceAccess: resource_access;
      resourceCustomAttrs: custom_attrs }

type resources (*F# = Resources and Resources F#*) 
  = Resources of resource list Lazy.t

type fixup = Fixup of (i32 * string list * data_label)
type fixups = fixup list

(* -------------------------------------------------------------------- 
 * One module in the "current" assembly
 * -------------------------------------------------------------------- *)

type longevity  (*F# = Longevity and Longevity F#*) = 
  | LongevityUnspecified
  | LongevityLibrary
  | LongevityPlatformAppDomain
  | LongevityPlatformProcess
  | LongevityPlatformSystem


type manifest = (*F# Manifest and Manifest = F#*)   
    { manifestName: string;
      manifestAuxModuleHashAlgorithm: i32;
      manifestSecurityDecls: security_decls;
      manifestPublicKey: bytes option;
      manifestVersion: version_info option;
      manifestLocale: locale option;
      manifestCustomAttrs: custom_attrs;

      manifestLongevity: longevity; 
      manifestDisableJitOptimizations: bool;
      manifestJitTracking: bool;

      manifestExportedTypes: exported_types;
               (* -- Records the types impemented by other modules. *)
      manifestEntrypointElsewhere: modul_ref option; 
               (* -- Records whether the entrypoint resides in another module. *)
    } 

type modul = (*F# ModuleDef and ModuleDef = F#*)   
  { modulManifest: manifest option;
    modulCustomAttrs: custom_attrs;
    modulName: string;
    modulTypeDefs: types;
    (* Random bits of relatively uninteresting data *)
    modulSubSystem: i32;
    modulDLL: bool;
    modulILonly: bool;
    modul32bit: bool;
    modulVirtAlignment: i32;
    modulPhysAlignment: i32;
    modulImageBase: i32;
    modulResources: resources;
    modulNativeResources: bytes Lazy.t option; (* e.g. win86 resources *)
(*    modulFixups: fixups; *)
  }
  (*F# 
     with
      member x.Manifest = x.modulManifest
      member x.CustomAttrs = x.modulCustomAttrs
      member x.Name = x.modulName
      member x.TypeDefs = x.modulTypeDefs
      member x.SubSystemFlags = x.modulSubSystem
      member x.IsDLL = x.modulDLL
      member x.IsILOnly = x.modulILonly
      member x.Is32Bit = x.modul32bit
      member x.VirtualAlignment = x.modulVirtAlignment
      member x.PhysicalAlignment = x.modulPhysAlignment
      member x.ImageBase = x.modulImageBase
      member x.Resources = x.modulResources
      member x.NativeResources = x.modulNativeResources
     end
  F#*)



(* -------------------------------------------------------------------- 
 * Extensions
 * -------------------------------------------------------------------- *)

type internal_instr_extension = 
    { internalInstrExtIs: ext_instr -> bool; 
      internalInstrExtDests: ext_instr -> code_label list;
      internalInstrExtFallthrough: ext_instr -> code_label option;
      internalInstrExtIsTailcall: ext_instr -> bool;
      internalInstrExtRelabel: (code_label -> code_label) -> ext_instr -> ext_instr; }

type internal_typ_extension = 
    { internalTypeExtIs: ext_typ -> bool; 
      internalTypeExtRescope: scope_ref -> ext_typ -> ext_typ;
      internalTypeExtInstAux: int -> genactuals -> ext_typ -> ext_typ }

type internal_type_def_kind_extension = 
    { internalTypeDefKindExtIs: ext_type_def_kind -> bool; }

type 'a instr_extension = 
    { instrExtDests: 'a -> code_label list;
      instrExtFallthrough: 'a -> code_label option;
      instrExtIsTailcall: 'a -> bool;
      instrExtRelabel: (code_label -> code_label) -> 'a -> 'a; }

type 'a typ_extension = 
    { typeExtRescope: scope_ref -> 'a -> 'a;
      typeExtInstAux: int -> genactuals -> 'a -> 'a }

let instr_extensions = ref []
let typ_extensions = ref []
let type_def_kind_extensions = ref []

let define_instr_extension  (ext: 'a instr_extension) = 
    if notnull !instr_extensions then failwith "define_instr_extension: only one extension currently allowed";
    let mk (x: 'a) = Ext_instr (Obj.repr x) in
    let test (Ext_instr x) = true in 
    let dest (Ext_instr x) = (Obj.obj x : 'a) in 
    instr_extensions := 
       { internalInstrExtIs=test;
         internalInstrExtDests=(fun x -> ext.instrExtDests (dest x));
         internalInstrExtFallthrough=(fun x -> ext.instrExtFallthrough (dest x));
         internalInstrExtIsTailcall=(fun x -> ext.instrExtIsTailcall (dest x));
         internalInstrExtRelabel=(fun f x -> mk (ext.instrExtRelabel f (dest x))); }
         :: !instr_extensions;
    mk,test,dest

let define_typ_extension (ext: 'a typ_extension) = 
    if notnull !typ_extensions then failwith "define_typ_extension: only one extension currently allowed";
    let mk (x:'a) = Ext_typ (Obj.repr x) in
    let test (Ext_typ x) = true in 
    let dest (Ext_typ x) = (Obj.obj x: 'a) in 
    typ_extensions := 
       { internalTypeExtIs=test;
         internalTypeExtInstAux=(fun n m x -> mk (ext.typeExtInstAux n m (dest x)));
         internalTypeExtRescope=(fun n x -> mk (ext.typeExtRescope n (dest x))); }
         :: !typ_extensions;
    mk,test,dest

type 'a type_def_kind_extension = Type_def_kind_extension

let define_type_def_kind_extension (Type_def_kind_extension : 'a type_def_kind_extension) = 
    if notnull !type_def_kind_extensions then failwith "define_type_extension: only one extension currently allowed";
    let mk (x:'a) = Ext_type_def_kind (Obj.repr x) in
    let test (Ext_type_def_kind x) = true in 
    let dest (Ext_type_def_kind x) = (Obj.obj x: 'a) in 
    type_def_kind_extensions := 
       { internalTypeDefKindExtIs=test;}
         :: !type_def_kind_extensions;
    mk,test,dest

(* -------------------------------------------------------------------- 
 * Utilities: type names
 * -------------------------------------------------------------------- *)

let split_name_at nm idx = 
    if idx < 0 then failwith "split_name_at: idx < 0";
    let last = String.length nm - 1 in 
    if idx > last then failwith "split_name_at: idx > last";
    (String.sub nm 0 idx),
    (if idx < last then String.sub nm (idx+1) (last - idx) else "")

let rec split_namespace_aux nm = 
    if String.contains nm '.' then 
        let idx = String.index nm '.' in
        let s1,s2 = split_name_at nm idx in 
        s1::split_namespace_aux s2 
    else [nm]
let memoize_namespace_tab = Hashtbl.create 10
let split_namespace nm =
  if Hashtbl.mem memoize_namespace_tab nm then
    Hashtbl.find memoize_namespace_tab nm
  else
    let x = split_namespace_aux nm in
    (Hashtbl.add  memoize_namespace_tab nm x; x)
let split_namespace_memoized nm = split_namespace nm


let split_type_name nm = 
    let slen = String.length nm in 
    let slenm = slen - 1 in 
    if String.rcontains_from nm slenm '.' then 
        let s1,s2 = split_name_at nm (String.rindex nm '.') in 
        split_namespace s1,s2
    else [],nm

let unsplit_type_name (ns,n) = if notnull ns then n else String.concat "." ns ^"."^n



(* -------------------------------------------------------------------- 
 * Assembly, Module and File references (2)
 * -------------------------------------------------------------------- *)

let mk_simple_assref n = 
    { assemRefName=n;
      assemRefHash=None;
      assemRefPublicKeyInfo=None;
      assemRefRetargetable=false;
      assemRefVersion=None;
      assemRefLocale=None; } 

let mk_simple_modref n = 
    { modulRefName=n;
      modulRefNoMetadata=false;
      modulRefHash=None }

let scoref_for_modname modul = ScopeRef_module(mk_simple_modref modul)

let module_name_of_scoref = function 
    | ScopeRef_module(mref) -> mref.modulRefName
    | _ -> failwith "module_name_of_scoref"

let module_is_mainmod m =
    match m.modulManifest with None -> false | _ -> true

let manifest_of_mainmod mainmod = 
    match mainmod.modulManifest with 
    | Some m -> m
    | None -> failwith "manifest_of_mainmod: no manifest.  It is possible you are using an auxiliary module of an assembly in a context where the main module of an assembly is expected.  Typically the main module of an assembly must be specified first within a list of the modules in an assembly."

let assname_of_mainmod mainmod = (manifest_of_mainmod mainmod).manifestName

(* -------------------------------------------------------------------- 
 * Calling conventions (2)
 * -------------------------------------------------------------------- *)

let mk_callconv hasthis = Callconv (hasthis, CC_default)
let instance_callconv = mk_callconv CC_instance
let static_callconv = mk_callconv CC_static

let is_vararg_callconv (Callconv (x, y)) = (match y with CC_vararg -> true | _ -> false)
let is_static_callconv (Callconv (x, y)) = (match x with CC_static -> true | _ -> false)

(* -------------------------------------------------------------------- 
 * Types
 * -------------------------------------------------------------------- *)

let tname_of_tref tr = tr.trefName
let enclosing_tnames_of_tref tr = tr.trefNested
let nested_tname_of_tref tr = String.concat "." (tr.trefNested @ [tr.trefName])
let scoref_of_tref x = x.trefScope

let tref_of_tspec ts = ts.tspecTypeRef
let inst_of_tspec ts = ts.tspecInst

let tname_of_tspec x = (tref_of_tspec >> tname_of_tref) x
let scoref_of_tspec x = (tref_of_tspec >> scoref_of_tref) x
let enclosing_tnames_of_tspec x = (tref_of_tspec >> enclosing_tnames_of_tref) x

let tspec_of_typ = function 
  | Type_boxed tr | Type_value tr -> tr
  | _ -> failwith "tspec_of_typ"

type boxity = (*F# Boxity and Boxity = F#*)
  | AsObject 
  | AsValue


let boxity_of_typ = function 
  | Type_boxed _ -> AsObject
  | Type_value _ -> AsValue
  | _ -> failwith "boxity_of_typ"

(* nb. throws away information in the typ *)
let tref_of_typ = function 
  | Type_boxed tr | Type_value tr -> tref_of_tspec tr
  | _ -> failwith "tref_of_typ"

let is_tref_typ = function 
  | Type_boxed tr | Type_value tr -> true
  | _ -> false


(* --------------------------------------------------------------------
 *  operations
 * -------------------------------------------------------------------- *)

let mk_empty_gparams = ([]: genparams)
let mk_empty_gactuals = ([]: genactuals)

let inst_of_typ = function 
  | Type_boxed tr | Type_value tr -> inst_of_tspec tr
  | _ -> mk_empty_gactuals

let mk_typ boxed tspec = 
  match boxed with AsObject -> Type_boxed tspec | _ -> Type_value tspec

let mk_named_typ vc tref tinst = mk_typ vc {tspecTypeRef=tref; tspecInst=tinst}

let mk_value_typ tref tinst = mk_named_typ AsValue tref tinst
let mk_boxed_typ tref tinst = mk_named_typ AsObject tref tinst

let mk_nongeneric_value_typ tref = mk_named_typ AsValue tref []
let mk_nongeneric_boxed_typ tref = mk_named_typ AsObject tref []

(* --------------------------------------------------------------------
 * Operations on method identifiers and signatures.
 * -------------------------------------------------------------------- *)

let callconv_of_callsig s = s.callsigCallconv
let args_of_callsig s = s.callsigArgs
let ret_of_callsig s = s.callsigReturn

let name_of_mref x = x.mrefName
let callconv_of_mref x = x.mrefCallconv
let ret_of_mref x = x.mrefReturn
let args_of_mref x = x.mrefArgs
let genarity_of_mref x = x.mrefArity

let parent_of_mref x = x.mrefParent
let tref_of_mref mref = parent_of_mref mref

let mk_callsig (cc,args,ret) = { callsigArgs=args; callsigCallconv=cc; callsigReturn=ret}

let callsig_of_mref mref = 
  mk_callsig (callconv_of_mref mref,args_of_mref mref,ret_of_mref mref)

let rename_mref n mref = {mref with mrefName=n}

(* Produce a new method reference which refer to the same method *)
(* in a parent class. *)
let relocate_mref typ mref = {mref with mrefParent=tref_of_typ typ }

(* --------------------------------------------------------------------
 * Operations on method references.
 * -------------------------------------------------------------------- *)

let formal_mref_of_mspec mspec = mspec.mspecMethodRefF
let enclosing_typ_of_mspec mspec = mspec.mspecEnclosingTypeF
let minst_of_mspec mspec = mspec.mspecMethodInstF
let callconv_of_mspec x = (formal_mref_of_mspec >> callconv_of_mref) x
let formal_ret_of_mspec x = (formal_mref_of_mspec >> ret_of_mref) x
let formal_args_of_mspec x = (formal_mref_of_mspec >> args_of_mref) x
let formal_parent_of_mspec x = (formal_mref_of_mspec >> parent_of_mref) x
let name_of_mspec x = (formal_mref_of_mspec >> name_of_mref) x
let formal_callsig_of_mspec x = (formal_mref_of_mspec >> callsig_of_mref) x
let genarity_of_mspec x =  (formal_mref_of_mspec >> genarity_of_mref) x

let typ_of_fref fref = fref.frefType
let name_of_fref fref = fref.frefName
let tref_of_fref fref = fref.frefParent

let fref_of_fspec x = x.fspecFieldRef
let enclosing_typ_of_fspec x = x.fspecEnclosingType
let formal_typ_of_fspec x = (fref_of_fspec >> typ_of_fref) x
let name_of_fspec x = (fref_of_fspec >> name_of_fref) x
let tref_of_fspec x = (fref_of_fspec >> tref_of_fref) x

(* --------------------------------------------------------------------
 * Make references to methods
 * -------------------------------------------------------------------- *)

let mk_nested_tref (scope,l,nm) =  {trefScope=scope; trefNested=l; trefName=nm}
let mk_tref (scope,nm) =  mk_nested_tref (scope,[],nm)
let mk_tspec (tref,inst) =  {tspecTypeRef=tref; tspecInst=inst}
let mk_nongeneric_tspec tref =  mk_tspec (tref,[])

let mk_tref_in_tref (tref,nm) = 
  mk_nested_tref (scoref_of_tref tref,enclosing_tnames_of_tref tref@[tname_of_tref tref],nm)

(* --------------------------------------------------------------------
 * The toplevel class of a module is called "<Module>"
 *
 * REVIEW: the  following comments from the ECMA Spec (Parition II, Section 9.8)
 *
 * "For an ordinary type, if the metadata merges two definitions 
 * of the same type, it simply discards one definition on the 
 * assumption they are equivalent and that any anomaly will be 
 * discovered when the type is used.  For the special class that 
 * holds global members, however, members are unioned across all 
 * modules at merge time. If the same name appears to be defined 
 * for cross-module use in multiple modules then there is an 
 * error.  In detail:
 *  - If no member of the same kind (field or method), name, and 
 *    signature exists, then add this member to the output class.
 *  - If there are duplicates and no more than one has an 
 *    accessibility other than compilercontrolled, then add them 
 *    all in the output class.
 *  - If there are duplicates and two or more have an accessibility 
 *    other than compilercontrolled an error has occurred."
 * -------------------------------------------------------------------- *)

let tname_for_toplevel = "<Module>"

let tref_for_toplevel scoref = {trefScope=scoref;trefNested=[];trefName=tname_for_toplevel}

let tspec_for_toplevel scoref = mk_nongeneric_tspec (tref_for_toplevel scoref)

let typ_for_toplevel scorefs = Type_boxed (tspec_for_toplevel scorefs)

let tname_eq (a:string) b = Pervasives.(=) a b

let is_toplevel_tname d = (tname_eq d tname_for_toplevel)

let mk_mref (tref,callconv,nm,gparams,args,rty) =
  { mrefParent=tref; 
    mrefCallconv=callconv;
    mrefArity=gparams;
    mrefName=nm;
    mrefArgs=args;
    mrefReturn=rty}

let mk_mref_mspec_in_typ (mref,typ,minst) = 
  { mspecOptionalID=None;
    mspecMethodRefF=mref;
    mspecEnclosingTypeF=typ;
    mspecMethodInstF=minst }

let mk_mspec (mref, vc, tinst, minst) =mk_mref_mspec_in_typ (mref,mk_named_typ vc (tref_of_mref mref) tinst,minst)

let mk_mspec_in_tref (tref,vc,cc,nm,args,rty,tinst,minst) =
  mk_mspec (mk_mref ( tref,cc,nm,List.length minst,args,rty),vc,tinst,minst)

let mk_mspec_in_tspec (tspec,vc,cc,nm,args,rty,minst) =
  mk_mspec_in_tref (tref_of_tspec tspec,vc,cc,nm,args,rty,inst_of_tspec tspec,minst)

let mk_nongeneric_mspec_in_tspec (tspec,vc,cc,nm,args,rty) =
  mk_mspec_in_tspec (tspec,vc,cc,nm,args,rty,mk_empty_gactuals)

let mk_mspec_in_typ (typ,cc,nm,args,rty,minst) =
  mk_mref_mspec_in_typ (mk_mref (tref_of_typ typ,cc,nm,List.length minst,args,rty),typ,minst)

let mk_nongeneric_mspec_in_typ (typ,cc,nm,args,rty) = 
  mk_mspec_in_typ (typ,cc,nm,args,rty,mk_empty_gactuals)

let mk_instance_mspec_in_tref (tref,vc,nm,args,rty,cinst,minst) =
  mk_mspec_in_tref (tref,vc,instance_callconv,nm,args,rty,cinst,minst)

let mk_instance_mspec_in_tspec (tspec,vc,nm,args,rty,minst) =
  mk_instance_mspec_in_tref (tref_of_tspec tspec, vc,nm,args,rty,inst_of_tspec tspec,minst)

let mk_instance_mspec_in_typ (typ,nm,args,rty,minst) =
  mk_instance_mspec_in_tspec (tspec_of_typ typ, boxity_of_typ typ,nm,args,rty,minst)

let mk_instance_mspec_in_boxed_tspec (tspec,nm,args,rty,minst) =
  mk_instance_mspec_in_tspec (tspec,AsObject,nm,args,rty,minst)

let mk_instance_mspec_in_nongeneric_boxed_tref(tref,nm,args,rty,minst) =
  mk_instance_mspec_in_boxed_tspec (mk_nongeneric_tspec tref,nm,args,rty,minst)

let mk_nongeneric_instance_mspec_in_tref (tref,vc,nm,args,rty,cinst) =
  mk_instance_mspec_in_tref (tref,vc,nm,args,rty,cinst,mk_empty_gactuals)

let mk_nongeneric_instance_mspec_in_tspec (tspec,vc,nm,args,rty) =
  mk_nongeneric_instance_mspec_in_tref (tref_of_tspec tspec,vc,nm,args,rty,inst_of_tspec tspec)
let mk_nongeneric_instance_mspec_in_typ (typ,nm,args,rty) =
  mk_nongeneric_instance_mspec_in_tspec (tspec_of_typ typ,boxity_of_typ typ,nm,args,rty)

let mk_nongeneric_instance_mspec_in_boxed_tspec (tspec,nm,args,rty) =
  mk_nongeneric_instance_mspec_in_tspec(tspec,AsObject,nm,args,rty)

let mk_nongeneric_instance_mspec_in_nongeneric_boxed_tref(tref,nm,args,rty) =
  mk_nongeneric_instance_mspec_in_boxed_tspec (mk_nongeneric_tspec tref,nm,args,rty)

let mk_nongeneric_mspec_in_tref (tref,vc,cc,nm,args,rty,cinst) =
  mk_mspec (mk_mref (tref,cc,nm,0,args,rty),vc,cinst,mk_empty_gactuals)

let mk_nongeneric_mspec_in_nongeneric_tref (tref,vc,callconv,nm,args,rty) =
  mk_nongeneric_mspec_in_tref (tref,vc,callconv,nm,args,rty,mk_empty_gactuals)

let mk_static_mref_in_tref (tref,nm,gparams,args,rty) =
  mk_mref(tref,static_callconv,nm,gparams,args,rty)

let mk_static_mspec_in_nongeneric_boxed_tref (tref,nm,args,rty,minst) =
  mk_mspec_in_tref (tref,AsObject,static_callconv,nm,args,rty,mk_empty_gactuals,minst)

let mk_static_mspec_in_boxed_tspec (tspec,nm,args,rty,minst) =
  mk_mspec_in_tspec (tspec,AsObject,static_callconv,nm,args,rty,minst)

let mk_static_mspec_in_typ (typ,nm,args,rty,minst) =
  mk_mspec_in_typ (typ,static_callconv,nm,args,rty,minst)

let mk_static_nongeneric_mspec_in_nongeneric_boxed_tref (tref,nm,args,rty) =
  mk_static_mspec_in_nongeneric_boxed_tref (tref,nm,args,rty,mk_empty_gactuals)

let mk_static_nongeneric_mspec_in_boxed_tspec (tspec,nm,args,rty) =
  mk_static_mspec_in_boxed_tspec (tspec,nm,args,rty,mk_empty_gactuals)

let mk_static_nongeneric_mspec_in_typ (typ,nm,args,rty) =
  mk_static_mspec_in_typ (typ,nm,args,rty,mk_empty_gactuals)

let mk_toplevel_static_mref scoref (nm,args,rty,gparams) =
  mk_static_mref_in_tref ( (tref_for_toplevel scoref),nm,args,rty,gparams)

let mk_toplevel_static_mspec scoref (nm,args,rty,minst) =
  mk_static_mspec_in_nongeneric_boxed_tref (tref_for_toplevel scoref,nm,args,rty,minst)

let mk_toplevel_static_nongeneric_mspec scoref (nm,args,rty) =
 mk_toplevel_static_mspec scoref (nm,args,rty,mk_empty_gactuals)

let mk_ctor_mspec (tref,vc,args,cinst) = 
  mk_mspec_in_tref(tref,vc,instance_callconv,".ctor",args,Type_void,cinst, mk_empty_gactuals)

let mk_ctor_mspec_for_typ (ty,args) = 
  mk_mspec_in_typ(ty,instance_callconv,".ctor",args,Type_void, mk_empty_gactuals)

let mk_nongeneric_ctor_mspec (tref,vc,args) = 
  mk_ctor_mspec (tref,vc,args,mk_empty_gactuals)

let mk_ctor_mspec_for_boxed_tspec (tr,argtys) =
  mk_ctor_mspec(tref_of_tspec tr,AsObject,argtys, inst_of_tspec tr)

let mk_ctor_mspec_for_nongeneric_boxed_tref (tr,argtys) =
  mk_ctor_mspec(tr,AsObject,argtys, mk_empty_gactuals)

(* --------------------------------------------------------------------
 * Make references to fields
 * -------------------------------------------------------------------- *)

let mk_fref_in_tref(tref,nm,ty) = 
  { frefParent=tref;
    frefName=nm; 
    frefType=ty}

let mk_fspec (tref,ty) = 
  { fspecFieldRef= tref;
    fspecEnclosingType=ty }

let mk_fspec_in_tspec (tspec,boxity,nm,ty) =
  mk_fspec (mk_fref_in_tref (tref_of_tspec tspec,nm,ty), mk_typ boxity tspec)
    
let mk_fspec_in_typ (typ,nm,fty) = 
  mk_fspec (mk_fref_in_tref (tref_of_tspec (tspec_of_typ typ),nm,fty), typ)
    
let mk_fspec_in_boxed_tspec (tspec,nm,ty) = 
  mk_fspec_in_tspec (tspec,AsObject,nm,ty) 
    
let mk_fspec_in_nongeneric_boxed_tref (tref,nm,ty) =
  mk_fspec_in_tspec (mk_nongeneric_tspec tref, AsObject,nm,ty)
    
let add_custom_attr_to_tab ca tab =  ca::tab
let add_custom_attr ca (CustomAttrs ltab) =  CustomAttrs (computed_map (add_custom_attr_to_tab ca) ltab)
let mk_custom_attrs l = CustomAttrs (not_computed (List.fold_right add_custom_attr_to_tab l []))
let mk_computed_custom_attrs l = CustomAttrs l
let dest_custom_attrs (CustomAttrs m) = force_computed m

let and_tailness x y = 
  match x with Tailcall when y -> Tailcall | _ -> Normalcall

(* -------------------------------------------------------------------- 
 * Attributes on code blocks (esp. debug info)
 * -------------------------------------------------------------------- *)

let rec last l = match l with [] -> failwith "last" | [h] -> h | h::t -> last t
let code_label_eq (x:code_label) y = int_eq x y 
let string_of_code_label x = "L"^string_of_int x

module CodeLabels = struct
    let insert (e:code_label) l = Zset.add e l
    let remove e l = Zset.remove e l
    let fold f s acc = Zset.fold f s acc
    let add s x = Zset.add s x
    let addL s xs = Zset.addL s xs
    let diff l1 l2 = Zset.diff l1 l2
    let union l1 l2 = Zset.union l1 l2
    let inter (l1:code_label Zset.t) l2 = Zset.inter l1 l2
    let subset (l1:code_label Zset.t) l2 = Zset.subset l1 l2
    let empty = Zset.empty int_compare
    let is_non_empty s = not (Zset.is_empty s)
    let of_list l = Zset.addL l empty
    let to_list l = Zset.elements l
end

(* -------------------------------------------------------------------- 
 * Basic operations on code.
 * -------------------------------------------------------------------- *)
let label_of_bblock b = b.bblockLabel

let instrs_of_bblock bk = bk.bblockInstrs

let last_of_bblock bb = 
  let n = Array.length bb.bblockInstrs in 
  if int_eq n 0 then failwith "last_of_bblock: empty bblock";
  bb.bblockInstrs.(n - 1)

let rec find_extension s f l = 
  let rec look l1 = 
    match l1 with
    | [] -> failwith ("extension for "^s^" not found")
    | (h::t) -> match f h with None -> look t | Some res -> res  in 
  look l
          
let destinations_of_instr i = 
  match i with 
  | I_leave l | I_br l -> [l]
  | I_brcmp (_,l1,l2) -> [l1; l2]
  | I_switch (ls,l) -> CodeLabels.to_list (CodeLabels.of_list (l::ls))
  | I_endfinally | I_endfilter | I_ret | I_throw | I_rethrow 
  | I_call (Tailcall,_,_)| I_callvirt (Tailcall,_,_)| I_callconstraint (Tailcall,_,_,_)
  | I_calli (Tailcall,_,_) -> []
  | I_other e -> find_extension "instr" (fun ext -> if ext.internalInstrExtIs e then Some (ext.internalInstrExtDests e) else None) !instr_extensions
  | _ -> []

let destinations_of_bblock (bblock:basic_block) = destinations_of_instr (last_of_bblock bblock)

let fallthrough_of_bblock (bblock:basic_block) = 
  begin match last_of_bblock bblock with 
  | I_br l | I_brcmp (_,_,l) | I_switch (_,l) -> Some l
  | I_other e -> find_extension "instr" (fun ext -> if ext.internalInstrExtIs e then Some (ext.internalInstrExtFallthrough e) else None) !instr_extensions
  | _ -> None
  end

let instr_is_tailcall i = 
  match i with 
  | I_call (Tailcall,_,_)| I_callvirt (Tailcall,_,_) | I_callconstraint (Tailcall,_,_,_) | I_calli (Tailcall,_,_) -> true
  | I_other e -> find_extension "instr" (fun ext -> if ext.internalInstrExtIs e then Some (ext.internalInstrExtIsTailcall e) else None) !instr_extensions
  | _ -> false

let instr_is_bblock_end i = 
  instr_is_tailcall i or
  match i with 
  | I_leave _ | I_br _ | I_brcmp _ | I_switch _ | I_endfinally
  | I_endfilter | I_ret | I_throw | I_rethrow  ->  true
  | I_other e -> find_extension "instr" (fun ext -> if ext.internalInstrExtIs e then Some (notnull (ext.internalInstrExtDests e)) else None) !instr_extensions
  | _ -> false

let checks = false let _ = if checks then dprint_endline "Warning - Il.checks is on"

let rec acc_entries_of_code c acc =
  match c with
  | BasicBlock bb -> CodeLabels.add bb.bblockLabel acc
  | GroupBlock (_,l) -> List.fold_right acc_entries_of_code l acc
  | RestrictBlock (ls,c) -> CodeLabels.union acc (CodeLabels.diff (entries_of_code' c) (CodeLabels.of_list ls))
  | TryBlock (l,r) -> acc_entries_of_code l acc
and entries_of_code' c = acc_entries_of_code c CodeLabels.empty 

let rec acc_exits_of_code c acc =
  let basic_outside_labels = 
    match c with
    | BasicBlock bblock -> CodeLabels.addL (destinations_of_bblock bblock) acc
    | GroupBlock (_,l) -> List.fold_right acc_exits_of_code l acc
    | RestrictBlock (ls,c) ->  CodeLabels.union acc (CodeLabels.diff (exits_of_code' c) (CodeLabels.of_list ls))
    | TryBlock (l,r) -> acc_exits_of_code l acc in
  CodeLabels.diff basic_outside_labels (entries_of_code' c)
and exits_of_code' c = acc_exits_of_code c CodeLabels.empty

let entries_of_code c = CodeLabels.to_list (entries_of_code' c)
let exits_of_code c = CodeLabels.to_list (exits_of_code' c)

let rec acc_labels_of_code acc c =
  (* Finds all labels defined within this code block, seeing through restrictions.
   * This assumes that labels are unique within the code blocks, even if hidden behind restrictions.
   *------
   * Repeats in the list indicate this invariant is broken.
   *)
  match c with
  | BasicBlock bb        -> bb.bblockLabel::acc
  | GroupBlock (_,l)     -> List.fold_left acc_labels_of_code acc l 
  | RestrictBlock (ls,c) -> acc_labels_of_code acc c
  | TryBlock (l,r)       -> let acc = acc_labels_of_code acc l in
                            let acc = acc_labels_of_seh  acc r in
                            acc
and acc_labels_of_seh acc = function
  | FaultBlock       code   -> acc_labels_of_code acc code
  | FinallyBlock     code   -> acc_labels_of_code acc code
  | FilterCatchBlock fcodes -> List.fold_left acc_labels_of_fcode acc fcodes
      
and acc_labels_of_fcode acc = function
  | TypeFilter typ,code  -> acc_labels_of_code acc code
  | CodeFilter test,code -> let accA = acc_labels_of_code acc code in
                            let accB = acc_labels_of_code accA test in
                            accB

let labels_of_code code = acc_labels_of_code [] code

(*

From the ECMA spec:

There are only two ways to enter a try block from outside its lexical body:
 - Branching to or falling into the try blocks first instruction. The branch may be made using a 37
conditional branch, an unconditional branch, or a leave instruction. 38
 - Using a leave instruction from that trys catch block. In this case, correct CIL code may 39
branch to any instruction within the try block, not just its first instruction, so long as that 40
branch target is not protected by yet another try, nested withing the first 
*)

    (* if n <> 1 then dprint_endline "*** warning: zero or more than one entry to a try block"; *)


let check_code code = 
    if checks then begin
        match code with
        | RestrictBlock (ls,c') -> 
            (*
              if not (CodeLabels.subset ls (entries_of_code c')) then begin
                dprint_endline ("*** warning: Restricting labels that are not declared in block, e.g. "^ (List.hd (CodeLabels.diff ls (entries_of_code c'))));
                dprint_endline ("*** warning: Labels in block are: "^ (String.concat "," (entries_of_code c')));
                dprint_endline ("*** warning: Labels being restricted are: "^ (String.concat "," ls));
              end;
            *)
            let cls = (CodeLabels.inter (CodeLabels.of_list ls) (exits_of_code' c')) in 
            if (CodeLabels.is_non_empty cls) then 
              dprint_endline ("*** warning: restricting unsatisfied exits from a block, e.g. "^ string_of_code_label (List.hd (CodeLabels.to_list cls)));
        | TryBlock (l,r) -> 
            begin match r with 
            | FaultBlock b | FinallyBlock b -> 
                if (CodeLabels.is_non_empty (CodeLabels.inter (exits_of_code' b) (entries_of_code' b))) then 
                  dprint_endline "*** warning: exits from fault or finally blocks must leave the block";
                let n = List.length (entries_of_code b) in 
                if not (int_eq n 1) then dprint_endline "*** warning: zero or more than one entry to a fault or finally block";
            | FilterCatchBlock r -> 
                List.iter 
                  (fun (flt,z) -> 
                    let m = List.length (entries_of_code z) in 
                    if not (int_eq m 1) then dprint_endline "*** warning: zero or more than one entry to a catch block";
                    match flt with 
                    | CodeFilter y -> 
                        if (CodeLabels.is_non_empty (exits_of_code' y)) then dprint_endline "*** warning: exits exist from filter block - you must always exit using endfinally";
                        let n = List.length (entries_of_code y) in 
                        if not (int_eq n 1) then dprint_endline "*** warning: zero or more than one entry to a filter block";
                    | TypeFilter ty -> ())
                  r;
            end;
        | BasicBlock bb ->
            if int_eq (Array.length bb.bblockInstrs) 0 then dprint_endline ("*** warning: basic block "^string_of_code_label bb.bblockLabel^" is empty")
            else if not (instr_is_bblock_end (bb.bblockInstrs.(Array.length bb.bblockInstrs - 1))) then failwith "*** warning: bblock does not end in an appropriate instruction";
            
        | _ -> ()
    end;
    match code with 
    | RestrictBlock (labs,c) when (isnull labs) -> c 
    | GroupBlock ([],[c]) -> c 
    | _ -> code


let mk_bblock bb = BasicBlock bb
let mk_scope_block (a,b) = GroupBlock (a,[check_code b])
let mk_group_block_from_code (internals,codes) = RestrictBlock (internals,check_code (GroupBlock ([],codes)))
let mk_group_block (internals,blocks) = mk_group_block_from_code (internals,List.map check_code blocks)

let mk_restrict_block lab c = RestrictBlock (CodeLabels.to_list (CodeLabels.remove lab (entries_of_code' c)),c)
let mk_try_finally_block (tryblock, enter_finally_lab, finallyblock) = 
  TryBlock(check_code tryblock, FinallyBlock (check_code (mk_restrict_block enter_finally_lab (check_code finallyblock))))

let mk_try_fault_block (tryblock, enter_fault_lab, faultblock) = 
  TryBlock(check_code tryblock, FaultBlock (check_code (mk_restrict_block enter_fault_lab (check_code faultblock))))

let mk_try_multi_filter_catch_block (tryblock, clauses) = 
    TryBlock
      (check_code tryblock, 
       FilterCatchBlock 
         (clauses |> List.map (fun (flt, (enter_catch_lab, catchblock)) -> 
                let fltcode = 
                  match flt with 
                  | Choice1of2 (enter_filter_lab, filterblock) ->
                      CodeFilter (check_code (mk_restrict_block enter_filter_lab (check_code filterblock)))
                  | Choice2of2 ty -> 
                      TypeFilter ty in 
                fltcode,
                check_code (mk_restrict_block enter_catch_lab (check_code catchblock)))))


let new_generator () = 
    let i = ref 0 in 
    fun n -> 
      incr i; !i

let code_label_generator = (new_generator () : unit -> code_label) 
let generate_code_label x  = code_label_generator x

let unique_entry_of_code c = 
    match entries_of_code c with 
    | [] -> failwith ("unique_entry_of_code: no entries to code")
    | [inlab] -> inlab
    | labs -> failwith ("unique_entry_of_code: need one entry to code, found: "^String.concat "," (List.map string_of_code_label labs))

let unique_exit_of_code c = 
    match exits_of_code c with 
    | [] -> failwith ("unique_exit_of_code: no exits from code")
    | [outlab] -> outlab
    | labs -> failwith ("unique_exit_of_code: need one exit from code, found: "^String.concat "," (List.map string_of_code_label labs))

let nonbranching_instrs inplab instrs = 
    check_code (mk_bblock {bblockLabel=inplab; bblockInstrs= Array.of_list instrs})

let nonbranching_instrs_then inplab instrs instr = 
    if notnull instrs && instr_is_bblock_end (last instrs) then failwith "nonbranching_instrs_then: bblock already terminates with a control flow instruction";
    nonbranching_instrs inplab (instrs @ [ instr ]) 

let nonbranching_instrs_then_ret inplab instrs = 
    nonbranching_instrs_then inplab instrs I_ret

let nonbranching_instrs_then_br inplab instrs lab = 
    nonbranching_instrs_then inplab instrs (I_br lab)

let nonbranching_instrs_to_code instrs = 
    let inplab = (generate_code_label ()) in 
    if notnull instrs && instr_is_bblock_end (last instrs) then 
      nonbranching_instrs inplab instrs
    else
      nonbranching_instrs_then_ret inplab  instrs

let join_code code1 code2 = 
    if not (code_label_eq (unique_exit_of_code code1) (unique_entry_of_code code2))  then 
      dprint_endline "*** warning: join_code: exit of code1 is not entry of code 2";
    check_code 
      (RestrictBlock ([unique_exit_of_code code1],
          (check_code (mk_group_block ([],[ code1; code2 ])))))

(* -------------------------------------------------------------------- 
 * Security declarations (2)
 * -------------------------------------------------------------------- *)

let add_security_decl_to_tab sd tab =  sd::tab
let add_security_decl ca (SecurityDecls sofar) =  SecurityDecls (lazy_map (add_security_decl_to_tab ca) sofar)
let mk_security_decls l = SecurityDecls (notlazy (List.fold_right add_security_decl_to_tab l []))
let mk_lazy_security_decls l = SecurityDecls (lazy (List.fold_right add_security_decl_to_tab (Lazy.force l) []))
let dest_security_decls (SecurityDecls m) = Lazy.force m

(* --------------------------------------------------------------------
 * ILX stuff
 * -------------------------------------------------------------------- *)

let mk_tyvar_ty tv = Type_tyvar tv

let is_tyvar_ty = function Type_tyvar _ -> true | _ -> false

let list_read l n = try List.nth l n with _ -> failwith "uninterp: read"
  
let inst_read (inst:genactuals) v =
  try list_read inst  (Nums.u16_to_int v)
  with _ -> failwith ("type variable no. "^Nums.u16_to_string v^" needs a value")

let inst_add (x1:genactuals) (x2:genactuals) = (x1@x2 : genactuals)

let gparams_add (x1:genparams) (x2:genparams) = (x1@x2:genparams)

let mk_simple_gparam nm =
   { gpName=nm;
     gpConstraints=[];
     gpVariance=NonVariant;
     gpReferenceTypeConstraint=false;
     gpNotNullableValueTypeConstraint=false;
     gpDefaultConstructorConstraint=false; }

let gparam_of_gactual (ga:genactual) = mk_simple_gparam "T"

let gparams_of_inst (x: genactuals) = List.map gparam_of_gactual x

let generalize_gparams (gparams:genparams)  =
    list_mapi (fun n gf -> mk_tyvar_ty (int_to_u16 n)) gparams
 
let generalize_tref tref gparams = mk_tspec (tref,generalize_gparams gparams)

(* -------------------------------------------------------------------- 
 * Operations on class etc. defs.
 * -------------------------------------------------------------------- *)

let is_value_tdef tdef = 
    match tdef.tdKind with
    | TypeDef_valuetype | TypeDef_enum -> true
    | _ -> false

let is_enum_tdef tdef = 
    match tdef.tdKind with
    | TypeDef_enum -> true
    | _ -> false

let kind_of_tdef td = td.tdKind
let name_of_tdef td = td.tdName
let name_of_nested_tdef (enc,td) = String.concat "." (List.map name_of_tdef enc @ [name_of_tdef td])
let tref_for_nested_tdef scope (enc,td)  = mk_nested_tref (scope, List.map name_of_tdef enc, name_of_tdef td)
let tref_for_tdef scope td = tref_for_nested_tdef scope ([],td)

let gparams_of_tdef td  = td.tdGenericParams
let gparams_of_nested_tdef ((enc : type_def list),td) = gparams_of_tdef td

let tspec_for_nested_tdef scope (enc,td) = generalize_tref (mk_nested_tref(scope, List.map name_of_tdef enc, td.tdName)) td.tdGenericParams

let generalize_tdef scope tdef =
  generalize_tref (tref_for_tdef scope tdef) (gparams_of_tdef tdef)

let generalize_nested_tdef scope (enc,tdef) =
  generalize_tref (tref_for_nested_tdef scope (enc,tdef)) (gparams_of_tdef tdef)

(* -------------------------------------------------------------------- 
 * Operations on type tables.
 * -------------------------------------------------------------------- *)

exception Not_unique_type of string

let getname ltd = 
  let td = (Lazy.force ltd) in 
  let ns,n = split_type_name td.tdName in 
  (ns,n,td.tdCustomAttrs,ltd)

let add_tdef_to_tab (ns,n,cas,ltd) tab = 
  let prev = 
     (match Pmap.tryfind ns tab with 
      | None -> Hashtbl.create 1
      | Some prev -> prev) in
  if Hashtbl.mem prev n then  raise (Not_unique_type (unsplit_type_name (ns,n)));
  Hashtbl.add prev n ltd;
  Pmap.add ns prev tab

let add_lazy_tdef_to_larr ltd larr = lazy_map (fun arr -> Array.of_list (getname ltd :: Array.to_list arr)) larr

let build_tab larr = lazy_map (fun arr -> Array.fold_right add_tdef_to_tab arr Pmap.empty) larr
let build_types larr = TypeDefTable (larr, build_tab larr)

(* this is not performance critical *)
let add_tdef td (TypeDefTable (larr,ltab)) = build_types (add_lazy_tdef_to_larr (notlazy td) larr)       
let mk_tdefs l =  build_types (List.map (notlazy >> getname) l |> Array.of_list |> notlazy )
let mk_lazy_tdefs llist = build_types (lazy_map Array.of_list llist)
let dest_lazy_tdefs (TypeDefTable (larr,tab)) = Lazy.force larr |> Array.to_list
let dest_tdefs tdefs = List.map (fun (_,_,_,td) -> Lazy.force td) (dest_lazy_tdefs tdefs)
let iter_tdefs f tdefs = dest_tdefs tdefs |> List.iter f
let find_tdef x (TypeDefTable (_,m)) = 
    let ns,n = split_type_name x in 
    m |> Lazy.force |> Pmap.find ns |> (fun t -> Hashtbl.find t n) |> Lazy.force 

let replace_tdef td (TypeDefTable (larr,_)) =
    larr
    |> lazy_map (fun larr ->
           let llist = Array.to_list larr in 
           let (ns,n,_,_)  as data = getname (notlazy td) in 
           let llist = llist |> List.filter (fun (ns2,n2,_,_) -> not (deep_eq ns ns2 && string_eq n n2)) in 
           Array.of_list (data :: llist)) 
    |> build_types

(* -------------------------------------------------------------------- 
 * Operations on method tables.
 * -------------------------------------------------------------------- *)

let dest_mdefs (Methods lpmap) = fst (Lazy.force lpmap)
let add_mdef_to_tab y tab =
  let key = y.mdName in 
  let prev = Pmap.tryfind_multi key tab in 
  Pmap.add key (y::prev) tab

let add_mdef_to_pmap y (mds,tab) = y::mds,add_mdef_to_tab y tab
let add_mdef y (Methods lpmap) = Methods (lazy_map (add_mdef_to_pmap y) lpmap)

let mk_mdefs l =  Methods (notlazy (List.fold_right add_mdef_to_pmap l ([],Pmap.empty)))
let mk_lazy_mdefs l =  Methods (lazy (List.fold_right add_mdef_to_pmap (Lazy.force l) ([],Pmap.empty)))
let add_mdef_to_tdef m cd = {cd with tdMethodDefs = add_mdef m cd.tdMethodDefs }
let filter_mdefs f (Methods lpmap) = 
  Methods (lazy_map (fun (fs,_) -> 
      let l = List.filter f fs in 
      (l, List.fold_right add_mdef_to_tab l Pmap.empty)) lpmap)
let find_mdefs_by_name nm (Methods lpmap) = 
  let t = (snd (Lazy.force lpmap)) in 
  Pmap.tryfind_multi nm t 
let find_mdefs_by_arity (nm,arity) tab = 
  List.filter (fun x -> int_eq (List.length x.mdParams) arity) (find_mdefs_by_name nm tab)


let nested_of_tdef td  = td.tdNested

(* -------------------------------------------------------------------- 
 * Operations and defaults for modules, assemblies etc.
 * -------------------------------------------------------------------- *)

let default_modulSubSystem = int_to_i32 3 (* this is what comes out of ILDASM on 30/04/2001 *)
let default_modulPhysAlignment = int_to_i32 512 (* this is what comes out of ILDASM on 30/04/2001 *)
let default_modulVirtAlignment = int_to_i32 0x2000 (* this is what comes out of ILDASM on 30/04/2001 *)
let default_modulImageBase = int_to_i32 0x034f0000 (* this is what comes out of ILDASM on 30/04/2001 *)

(* -------------------------------------------------------------------- 
 * Array types
 * -------------------------------------------------------------------- *)

let rank_of_array_shape (ArrayShape l) = int_to_i32 (List.length l)
let mk_array_ty (ty,shape) = Type_array(shape,ty)
let sdshape = ArrayShape [(Some i32_zero, None)]
let mk_sdarray_ty ty = mk_array_ty (ty,sdshape)

let dest_array_ty = function 
  | Type_array (shape,ty) -> shape,ty
  | _ -> failwith "dest_array_ty: bad array type"

let is_array_ty = function 
  | Type_array _ -> true
  | _ -> false

(* -------------------------------------------------------------------- 
 * Sigs of special types built-in, e.g. those needed by the verifier
 * -------------------------------------------------------------------- *)

let mscorlib_module_name =  "CommonLanguageRuntimeLibrary"

let tname_Object = "System.Object"
let tname_String = "System.String"
let tname_AsyncCallback = "System.AsyncCallback"
let tname_IAsyncResult = "System.IAsyncResult"
let tname_IComparable = "System.IComparable"
let tname_Exception = "System.Exception"
let tname_Type = "System.Type"
let tname_Missing = "System.Reflection.Missing"
let tname_Activator = "System.Activator"
let tname_Delegate = "System.Delegate"
let tname_ValueType = "System.ValueType"
let tname_TypedReference = "System.TypedReference"
let tname_Enum = "System.Enum"
let tname_MulticastDelegate = "System.MulticastDelegate"
let tname_Array = "System.Array"

let tname_Int64 = "System.Int64"
let tname_UInt64 = "System.UInt64"
let tname_Int32 = "System.Int32"
let tname_UInt32 = "System.UInt32"
let tname_Int16 = "System.Int16"
let tname_UInt16 = "System.UInt16"
let tname_SByte = "System.SByte"
let tname_Byte = "System.Byte"
let tname_Single = "System.Single"
let tname_Double = "System.Double"
let tname_Bool = "System.Boolean"
let tname_Char = "System.Char"
let tname_IntPtr = "System.IntPtr"
let tname_UIntPtr = "System.UIntPtr"
let tname_RuntimeArgumentHandle = "System.RuntimeArgumentHandle"
let tname_RuntimeTypeHandle = "System.RuntimeTypeHandle"
let tname_RuntimeMethodHandle = "System.RuntimeMethodHandle"
let tname_RuntimeFieldHandle = "System.RuntimeFieldHandle"

type mscorlib_refs = (*F# MscorlibRefs 
and 
 (*F#
 [<StructuralEquality(false); StructuralComparison(false)>]
 F#*)
 MscorlibRefs = F#*)
  { mscorlib_scoref: scope_ref;
    mscorlib_assembly_name: string; (* adding: JAMES *)
    tref_Object: type_ref 
    ; tspec_Object: type_spec
    ; typ_Object: typ
    ; tref_String: type_ref
    ; typ_String: typ
    ; typ_AsyncCallback: typ
    ; typ_IAsyncResult: typ
    ; typ_IComparable: typ
    ; tref_Type: type_ref
    ; typ_Type: typ
    ; tref_Missing: type_ref
    ; typ_Missing: typ
    ; typ_Activator: typ
    ; typ_Delegate: typ
    ; typ_ValueType: typ
    ; typ_Enum: typ
    ; tspec_TypedReference: type_spec
    ; typ_TypedReference: typ
    ; typ_MulticastDelegate: typ
    ; typ_Array: typ
    ; tspec_Int64: type_spec
    ; tspec_UInt64: type_spec
    ; tspec_Int32: type_spec
    ; tspec_UInt32: type_spec
    ; tspec_Int16: type_spec
    ; tspec_UInt16: type_spec
    ; tspec_SByte: type_spec
    ; tspec_Byte: type_spec
    ; tspec_Single: type_spec
    ; tspec_Double: type_spec
    ; tspec_IntPtr: type_spec
    ; tspec_UIntPtr: type_spec
    ; tspec_Char: type_spec
    ; tspec_Bool: type_spec
    ; typ_int8: typ
    ; typ_int16: typ
    ; typ_int32: typ
    ; typ_int64: typ
    ; typ_uint8: typ
    ; typ_uint16: typ
    ; typ_uint32: typ
    ; typ_uint64: typ
    ; typ_float32: typ
    ; typ_float64: typ
    ; typ_bool: typ
    ; typ_char: typ
    ; typ_int: typ
    ; typ_uint: typ
    ; typ_RuntimeArgumentHandle: typ
    ; typ_RuntimeTypeHandle: typ
    ; typ_RuntimeMethodHandle: typ
    ; typ_RuntimeFieldHandle: typ

    ; typ_Byte: typ
    ; typ_Int16: typ
    ; typ_Int32: typ
    ; typ_Int64: typ
    ; typ_SByte: typ
    ; typ_UInt16: typ
    ; typ_UInt32: typ
    ; typ_UInt64: typ
    ; typ_Single: typ
    ; typ_Double: typ
    ; typ_Bool: typ
    ; typ_Char: typ
    ; typ_IntPtr: typ
    ; typ_UIntPtr: typ
    ; tspec_Exception: type_spec
    ; typ_Exception: typ }

let mk_normal_call mspec = I_call (Normalcall, mspec, None)
let mk_normal_callvirt mspec = I_callvirt (Normalcall, mspec, None)
let mk_normal_callconstraint (ty,mspec) = I_callconstraint (Normalcall, ty, mspec, None)
let mk_tail_callvirt mspec = I_callvirt (Tailcall, mspec, None)  
let mk_normal_newobj mspec =  I_newobj (mspec, None)
let ldarg_0 = I_ldarg u16_zero
let tname_CompilerGeneratedAttribute = "System.Runtime.CompilerServices.CompilerGeneratedAttribute"
let tname_DebuggableAttribute = "System.Diagnostics.DebuggableAttribute"
let tname_DebuggableAttribute_DebuggingModes = "DebuggingModes"
let tname_DebuggerHiddenAttribute = "System.Diagnostics.DebuggerHiddenAttribute"
let tname_DebuggerStepThroughAttribute = "System.Diagnostics.DebuggerStepThroughAttribute"

let mk_MscorlibRefs mscorlib_scoref mscorlib_assembly_name_option =
  let mscorlib_assembly_name =
    match mscorlib_assembly_name_option with
      | Some name -> name 
      | None      -> (match mscorlib_scoref with
                        | ScopeRef_assembly assref -> assref.assemRefName
                        | _ -> failwith "mk_MscorlibRefs: mscorlib scope_ref is not an assembly ref") in
  let tref_Object = mk_tref (mscorlib_scoref,tname_Object) in
  let tspec_Object = mk_nongeneric_tspec tref_Object in
  let typ_Object = Type_boxed tspec_Object in

  let tref_String = mk_tref (mscorlib_scoref,tname_String) in
  let tspec_String = mk_tspec(tref_String,mk_empty_gactuals) in
  let typ_String = Type_boxed tspec_String in

  let tref_AsyncCallback = mk_tref (mscorlib_scoref,tname_AsyncCallback) in
  let tspec_AsyncCallback = mk_tspec(tref_AsyncCallback,mk_empty_gactuals) in
  let typ_AsyncCallback = Type_boxed tspec_AsyncCallback in

  let tref_IAsyncResult = mk_tref (mscorlib_scoref,tname_IAsyncResult) in
  let tspec_IAsyncResult = mk_tspec(tref_IAsyncResult,mk_empty_gactuals) in
  let typ_IAsyncResult = Type_boxed tspec_IAsyncResult in

  let tref_IComparable = mk_tref (mscorlib_scoref,tname_IComparable) in
  let tspec_IComparable = mk_tspec(tref_IComparable,mk_empty_gactuals) in
  let typ_IComparable = Type_boxed tspec_IComparable in

  let tref_Exception = mk_tref (mscorlib_scoref,tname_Exception) in
  let tspec_Exception = mk_tspec(tref_Exception,mk_empty_gactuals) in
  let typ_Exception = Type_boxed tspec_Exception in

  let tref_Type = mk_tref(mscorlib_scoref,tname_Type) in
  let tspec_Type = mk_tspec(tref_Type,mk_empty_gactuals) in
  let typ_Type = Type_boxed tspec_Type in

  let tref_Missing = mk_tref(mscorlib_scoref,tname_Missing) in
  let tspec_Missing = mk_tspec(tref_Missing,mk_empty_gactuals) in
  let typ_Missing = Type_boxed tspec_Missing in


  let tref_Activator = mk_tref(mscorlib_scoref,tname_Activator) in
  let tspec_Activator = mk_tspec(tref_Activator,mk_empty_gactuals) in
  let typ_Activator = Type_boxed tspec_Activator in

  let tref_Delegate = mk_tref(mscorlib_scoref,tname_Delegate) in
  let tspec_Delegate = mk_tspec(tref_Delegate,mk_empty_gactuals) in
  let typ_Delegate = Type_boxed tspec_Delegate in

  let tref_ValueType = mk_tref (mscorlib_scoref,tname_ValueType) in
  let tspec_ValueType = mk_tspec(tref_ValueType,mk_empty_gactuals) in
  let typ_ValueType = Type_boxed tspec_ValueType in

  let tref_TypedReference = mk_tref (mscorlib_scoref,tname_TypedReference) in
  let tspec_TypedReference = mk_tspec(tref_TypedReference,mk_empty_gactuals) in
  let typ_TypedReference = Type_value tspec_TypedReference in

  let tref_Enum = mk_tref (mscorlib_scoref,tname_Enum) in
  let tspec_Enum = mk_tspec(tref_Enum,mk_empty_gactuals) in
  let typ_Enum = Type_boxed tspec_Enum in

  let tref_MulticastDelegate = mk_tref (mscorlib_scoref,tname_MulticastDelegate) in
  let tspec_MulticastDelegate = mk_tspec(tref_MulticastDelegate,mk_empty_gactuals) in
  let typ_MulticastDelegate = Type_boxed tspec_MulticastDelegate in

  let typ_Array = Type_boxed (mk_tspec(mk_tref (mscorlib_scoref,tname_Array),mk_empty_gactuals)) in

  let tref_Int64 = mk_tref (mscorlib_scoref,tname_Int64) in
  let tref_UInt64 = mk_tref (mscorlib_scoref,tname_UInt64) in
  let tref_Int32 = mk_tref (mscorlib_scoref,tname_Int32) in
  let tref_UInt32 = mk_tref (mscorlib_scoref,tname_UInt32) in
  let tref_Int16 = mk_tref (mscorlib_scoref,tname_Int16) in
  let tref_UInt16 = mk_tref (mscorlib_scoref,tname_UInt16) in
  let tref_SByte = mk_tref (mscorlib_scoref,tname_SByte) in
  let tref_Byte = mk_tref (mscorlib_scoref,tname_Byte) in
  let tref_Single = mk_tref (mscorlib_scoref,tname_Single) in
  let tref_Double = mk_tref (mscorlib_scoref,tname_Double) in
  let tref_Bool = mk_tref (mscorlib_scoref,tname_Bool) in
  let tref_Char = mk_tref (mscorlib_scoref,tname_Char) in
  let tref_IntPtr = mk_tref (mscorlib_scoref,tname_IntPtr) in
  let tref_UIntPtr = mk_tref (mscorlib_scoref,tname_UIntPtr) in

  let tspec_Int64 = mk_tspec(tref_Int64,mk_empty_gactuals) in
  let tspec_UInt64 = mk_tspec(tref_UInt64,mk_empty_gactuals) in
  let tspec_Int32 = mk_tspec(tref_Int32,mk_empty_gactuals) in
  let tspec_UInt32 = mk_tspec(tref_UInt32,mk_empty_gactuals) in
  let tspec_Int16 = mk_tspec(tref_Int16,mk_empty_gactuals) in
  let tspec_UInt16 = mk_tspec(tref_UInt16,mk_empty_gactuals) in 
  let tspec_SByte = mk_tspec(tref_SByte,mk_empty_gactuals) in
  let tspec_Byte = mk_tspec(tref_Byte,mk_empty_gactuals) in
  let tspec_Single = mk_tspec(tref_Single,mk_empty_gactuals) in
  let tspec_Double = mk_tspec(tref_Double,mk_empty_gactuals) in
  let tspec_IntPtr = mk_tspec(tref_IntPtr,mk_empty_gactuals) in
  let tspec_UIntPtr = mk_tspec(tref_UIntPtr,mk_empty_gactuals) in
  let tspec_Char = mk_tspec(tref_Char,mk_empty_gactuals) in
  let tspec_Bool = mk_tspec(tref_Bool,mk_empty_gactuals) in

  let typ_int8 = Type_value tspec_SByte  in
  let typ_int16 = Type_value tspec_Int16 in
  let typ_int32 = Type_value tspec_Int32 in
  let typ_int64 = Type_value tspec_Int64 in
  let typ_uint8 = Type_value tspec_Byte in
  let typ_uint16 = Type_value tspec_UInt16 in
  let typ_uint32 = Type_value tspec_UInt32 in 
  let typ_uint64 = Type_value tspec_UInt64 in
  let typ_float32 = Type_value tspec_Single in
  let typ_float64 = Type_value tspec_Double in
  let typ_bool = Type_value tspec_Bool in
  let typ_char = Type_value tspec_Char in
  let typ_int = Type_value tspec_IntPtr in
  let typ_uint = Type_value tspec_UIntPtr in

  let typ_SByte = Type_value tspec_SByte in 
  let typ_Int16 = Type_value tspec_Int16 in
  let typ_Int32 = Type_value tspec_Int32 in
  let typ_Int64 = Type_value tspec_Int64 in
  let typ_Byte = Type_value tspec_Byte in
  let typ_UInt16 = Type_value tspec_UInt16 in
  let typ_UInt32 = Type_value tspec_UInt32 in
  let typ_UInt64 = Type_value tspec_UInt64 in
  let typ_Single = Type_value tspec_Single in
  let typ_Double = Type_value tspec_Double in
  let typ_Bool = Type_value tspec_Bool in
  let typ_Char = Type_value tspec_Char in
  let typ_IntPtr = Type_value tspec_IntPtr in
  let typ_UIntPtr = Type_value tspec_UIntPtr in

  let tref_RuntimeArgumentHandle = mk_tref (mscorlib_scoref,tname_RuntimeArgumentHandle) in
  let tspec_RuntimeArgumentHandle = mk_tspec(tref_RuntimeArgumentHandle,mk_empty_gactuals) in
  let typ_RuntimeArgumentHandle = Type_value tspec_RuntimeArgumentHandle in
  let tref_RuntimeTypeHandle = mk_tref (mscorlib_scoref,tname_RuntimeTypeHandle) in
  let tspec_RuntimeTypeHandle = mk_tspec(tref_RuntimeTypeHandle,mk_empty_gactuals) in
  let typ_RuntimeTypeHandle = Type_value tspec_RuntimeTypeHandle in
  let tref_RuntimeMethodHandle = mk_tref (mscorlib_scoref,tname_RuntimeMethodHandle) in
  let tspec_RuntimeMethodHandle = mk_tspec(tref_RuntimeMethodHandle,mk_empty_gactuals) in
  let typ_RuntimeMethodHandle = Type_value tspec_RuntimeMethodHandle in
  let tref_RuntimeFieldHandle = mk_tref (mscorlib_scoref,tname_RuntimeFieldHandle) in
  let tspec_RuntimeFieldHandle = mk_tspec(tref_RuntimeFieldHandle,mk_empty_gactuals) in
  let typ_RuntimeFieldHandle = Type_value tspec_RuntimeFieldHandle in
  {   mscorlib_scoref            =mscorlib_scoref
    ; mscorlib_assembly_name     =mscorlib_assembly_name
    ; tref_Object                =tref_Object                  
    ; tspec_Object               =tspec_Object                 
    ; typ_Object                 =typ_Object                   
    ; tref_String                =tref_String                  
    ; typ_String                 =typ_String                   
    ; typ_AsyncCallback          =typ_AsyncCallback            
    ; typ_IAsyncResult           =typ_IAsyncResult             
    ; typ_IComparable            =typ_IComparable              
    ; typ_Activator              =typ_Activator                     
    ; tref_Type                  =tref_Type                    
    ; typ_Type                   =typ_Type                     
    ; tref_Missing               =tref_Missing                    
    ; typ_Missing                =typ_Missing                     
    ; typ_Delegate               =typ_Delegate                 
    ; typ_ValueType              =typ_ValueType                
    ; typ_Enum                   =typ_Enum                     
    ; tspec_TypedReference       =tspec_TypedReference         
    ; typ_TypedReference         =typ_TypedReference           
    ; typ_MulticastDelegate      =typ_MulticastDelegate        
    ; typ_Array                  =typ_Array                    
    ; tspec_Int64                =tspec_Int64                  
    ; tspec_UInt64               =tspec_UInt64                 
    ; tspec_Int32                =tspec_Int32                  
    ; tspec_UInt32               =tspec_UInt32                 
    ; tspec_Int16                =tspec_Int16                  
    ; tspec_UInt16               =tspec_UInt16                 
    ; tspec_SByte                =tspec_SByte                  
    ; tspec_Byte                 =tspec_Byte                   
    ; tspec_Single               =tspec_Single                 
    ; tspec_Double               =tspec_Double                 
    ; tspec_IntPtr               =tspec_IntPtr                 
    ; tspec_UIntPtr              =tspec_UIntPtr                
    ; tspec_Char                 =tspec_Char                   
    ; tspec_Bool                 =tspec_Bool                   
    ; typ_int8                   =typ_int8                     
    ; typ_int16                  =typ_int16                    
    ; typ_int32                  =typ_int32                    
    ; typ_int64                  =typ_int64                    
    ; typ_uint8                  =typ_uint8                    
    ; typ_uint16                 =typ_uint16                   
    ; typ_uint32                 =typ_uint32                   
    ; typ_uint64                 =typ_uint64                   
    ; typ_float32                =typ_float32                  
    ; typ_float64                =typ_float64                  
    ; typ_bool                   =typ_bool                     
    ; typ_char                   =typ_char                     
    ; typ_int                    =typ_int                      
    ; typ_uint                   =typ_uint                     
    ; typ_RuntimeArgumentHandle  =typ_RuntimeArgumentHandle    
    ; typ_RuntimeTypeHandle      =typ_RuntimeTypeHandle        
    ; typ_RuntimeMethodHandle    =typ_RuntimeMethodHandle      
    ; typ_RuntimeFieldHandle     =typ_RuntimeFieldHandle       
                                                                               
    ; typ_Byte                   =typ_Byte                     
    ; typ_Int16                  =typ_Int16                    
    ; typ_Int32                  =typ_Int32                    
    ; typ_Int64                  =typ_Int64                    
    ; typ_SByte                  =typ_SByte                    
    ; typ_UInt16                 =typ_UInt16                   
    ; typ_UInt32                 =typ_UInt32                   
    ; typ_UInt64                 =typ_UInt64                   
    ; typ_Single                 =typ_Single                   
    ; typ_Double                 =typ_Double                   
    ; typ_Bool                   =typ_Bool                     
    ; typ_Char                   =typ_Char                     
    ; typ_IntPtr                 =typ_IntPtr                   
    ; typ_UIntPtr                =typ_UIntPtr                  
    ; tspec_Exception            =tspec_Exception              
    ; typ_Exception              =typ_Exception                 }

        
(* NOTE: ecma_ prefix refers to the standard "mscorlib" *)
let ecma_mscorlib_assembly_name = "mscorlib"
let ecma_public_token = PublicKeyToken (Bytes.of_intarray [|0x96; 0x9D; 0xB8; 0x05; 0x3D; 0x33; 0x22; 0xAC |]) 
let ecma_mscorlib_assref = 
  { assemRefName=ecma_mscorlib_assembly_name;
    assemRefHash= None;
    assemRefPublicKeyInfo=Some ecma_public_token;
    assemRefRetargetable=true;
    assemRefVersion=None;
    assemRefLocale=None; }
    
let ecma_mscorlib_scoref = ScopeRef_assembly ecma_mscorlib_assref

let ecma_mscorlib_refs = mk_MscorlibRefs ecma_mscorlib_scoref None
   
let mspec_RuntimeHelpers_InitializeArray ilg = 
  mk_static_nongeneric_mspec_in_nongeneric_boxed_tref (mk_tref(ilg.mscorlib_scoref,"System.Runtime.CompilerServices.RuntimeHelpers"),"InitializeArray", [ilg.typ_Array;ilg.typ_RuntimeFieldHandle], Type_void)
(* e.ilg. [mk_mscorlib_exn_newobj "System.InvalidCastException"] *)
let mk_mscorlib_exn_newobj ilg eclass = 
  mk_normal_newobj (mk_nongeneric_ctor_mspec (mk_tref(ilg.mscorlib_scoref,eclass),AsObject,[]))

let mspec_Console_WriteLine ilg = mk_static_nongeneric_mspec_in_nongeneric_boxed_tref (mk_tref(ilg.mscorlib_scoref,"System.Console"),"WriteLine",[ilg.typ_String],Type_void)
let mk_writeline_call ilg = mk_normal_call (mspec_Console_WriteLine ilg)

let mspec_RunClassConstructor ilg = 
  mk_static_nongeneric_mspec_in_nongeneric_boxed_tref (mk_tref(ilg.mscorlib_scoref,"System.Runtime.CompilerServices.RuntimeHelpers"),"RunClassConstructor",[ilg.typ_RuntimeTypeHandle],Type_void)

let mk_RunClassConstructor ilg tspec =
  [ I_ldtoken (Token_type (mk_typ AsObject tspec));
    mk_normal_call (mspec_RunClassConstructor ilg) ]


let extends_of_tdef  td = td.tdExtends
let access_of_tdef td = td.tdAccess
let layout_of_tdef td = td.tdLayout
let implements_of_tdef td = td.tdImplements
let sealed_of_tdef td = td.tdSealed
let abstract_of_tdef td = td.tdAbstract
let serializable_of_tdef td = td.tdSerializable
let encoding_of_tdef td  = td.tdEncoding
let initsemantics_of_tdef td  = td.tdInitSemantics


let typ_is_boxed = function Type_boxed _ -> true | _ -> false
let typ_is_value = function Type_value _ -> true | _ -> false


let tspec_is_mscorlib ilg tspec n = 
  let tref = tref_of_tspec tspec in 
  let scoref = scoref_of_tref tref in 
  tname_eq (tname_of_tref tref) n &
  begin match scoref with
  | ScopeRef_assembly n -> string_eq n.assemRefName ilg.mscorlib_assembly_name 
  | ScopeRef_module _ -> false
  | ScopeRef_local -> true
  end

let typ_is_boxed_mscorlib_typ ilg ty n = 
  typ_is_boxed ty && tspec_is_mscorlib ilg (tspec_of_typ ty) n

let typ_is_value_mscorlib_typ ilg ty n = 
  typ_is_value ty && tspec_is_mscorlib ilg (tspec_of_typ ty) n
      
let typ_is_Object            ilg ty = typ_is_boxed_mscorlib_typ ilg ty tname_Object
(*
let typ_is_MulticastDelegate ilg ty = typ_is_boxed_mscorlib_typ ilg ty tname_MulticastDelegate
let typ_is_Delegate          ilg ty = typ_is_boxed_mscorlib_typ ilg ty tname_Delegate
let typ_is_Enum              ilg ty = typ_is_boxed_mscorlib_typ ilg ty tname_Enum
let typ_is_ValueType         ilg ty = typ_is_boxed_mscorlib_typ ilg ty tname_ValueType
*)
let typ_is_String            ilg ty = typ_is_boxed_mscorlib_typ ilg ty tname_String
let typ_is_AsyncCallback     ilg ty = typ_is_boxed_mscorlib_typ ilg ty tname_AsyncCallback
let typ_is_TypedReference    ilg ty = typ_is_value_mscorlib_typ ilg ty tname_TypedReference
let typ_is_IAsyncResult ilg ty = typ_is_boxed_mscorlib_typ ilg ty tname_IAsyncResult
let typ_is_IComparable  ilg ty = typ_is_boxed_mscorlib_typ ilg ty tname_IComparable
let typ_is_SByte        ilg ty = typ_is_value_mscorlib_typ ilg ty tname_SByte
let typ_is_Byte         ilg ty = typ_is_value_mscorlib_typ ilg ty tname_Byte
let typ_is_Int16        ilg ty = typ_is_value_mscorlib_typ ilg ty tname_Int16
let typ_is_UInt16       ilg ty = typ_is_value_mscorlib_typ ilg ty tname_UInt16
let typ_is_Int32        ilg ty = typ_is_value_mscorlib_typ ilg ty tname_Int32
let typ_is_UInt32       ilg ty = typ_is_value_mscorlib_typ ilg ty tname_UInt32
let typ_is_Int64        ilg ty = typ_is_value_mscorlib_typ ilg ty tname_Int64
let typ_is_UInt64       ilg ty = typ_is_value_mscorlib_typ ilg ty tname_UInt64
let typ_is_IntPtr       ilg ty = typ_is_value_mscorlib_typ ilg ty tname_IntPtr
let typ_is_UIntPtr      ilg ty = typ_is_value_mscorlib_typ ilg ty tname_UIntPtr
let typ_is_Bool         ilg ty = typ_is_value_mscorlib_typ ilg ty tname_Bool
let typ_is_Char         ilg ty = typ_is_value_mscorlib_typ ilg ty tname_Char
let typ_is_Single       ilg ty = typ_is_value_mscorlib_typ ilg ty tname_Single
let typ_is_Double       ilg ty = typ_is_value_mscorlib_typ ilg ty tname_Double

(* -------------------------------------------------------------------- 
 * Rescoping
 * -------------------------------------------------------------------- *)

let qrescope_scoref scoref scoref_old = 
  match scoref,scoref_old with 
  | _,ScopeRef_local -> Some scoref
  | ScopeRef_local,_ -> None
  | _,ScopeRef_module _ -> Some scoref
  | ScopeRef_module _,_ -> None
  | _ -> None
let qrescope_tref scoref x = 
  match qrescope_scoref scoref x.trefScope with 
  | None -> None
  | Some s -> Some {x with trefScope=s}

let rescope_scoref x y = match qrescope_scoref x y with Some x -> x | None -> y
let rescope_tref x y = match qrescope_tref x y with Some x -> x | None -> y

let rec qrescope_tspec scoref tspec = 

      (* ORIGINAL IMPLEMENTATION (too many allocations
           { tspecTypeRef=rescope_tref scoref tref;
             tspecInst=rescope_inst scoref tinst } *)

  let tref = tspec.tspecTypeRef in 
  let tinst = tspec.tspecInst in 
  let qtref = qrescope_tref scoref tref in 
  match tinst,qtref with 
  | [],None -> None  (* avoid reallocation in the common case *)
  | _,None -> 
      Some { tspecTypeRef=tref;
             tspecInst=rescope_inst scoref tinst }
  | _,Some tref -> 
      Some { tspecTypeRef=tref;
             tspecInst=rescope_inst scoref tinst }
and rescope_tspec x y = match qrescope_tspec x y with Some x -> x | None -> y
and rescope_typ scoref typ = 
  match typ with 
  | Type_ptr t -> Type_ptr (rescope_typ scoref t)
  | Type_fptr t -> Type_fptr (rescope_callsig scoref t)
  | Type_byref t -> Type_byref (rescope_typ scoref t)
  | Type_boxed cr -> 
      begin match qrescope_tspec scoref cr with 
      | Some res -> Type_boxed res
      | None -> typ  (* avoid reallocation in the common case *)
      end
     
  | Type_array (s,ty) -> Type_array (s,rescope_typ scoref ty)
  | Type_value cr -> 
      begin match qrescope_tspec scoref cr with 
      | Some res -> Type_value res
      | None -> typ  (* avoid reallocation in the common case *)
      end
  | Type_modified(b,tref,ty) -> Type_modified(b,rescope_tref scoref tref, rescope_typ scoref ty)
  | Type_other e -> Type_other (find_extension "typ" (fun ext -> if ext.internalTypeExtIs e then Some (ext.internalTypeExtRescope scoref e) else None) !typ_extensions)
  | x -> x
and rescope_inst scoref i = List.map (rescope_typ scoref) i
and rescope_callsig scoref  csig = 
  mk_callsig
    (csig.callsigCallconv,List.map (rescope_typ scoref) csig.callsigArgs,rescope_typ scoref csig.callsigReturn)

let rescope_dloc scoref tref = rescope_tref scoref tref 
let rescope_mref scoref x =
  { mrefParent = rescope_dloc scoref x.mrefParent;
    mrefCallconv = x.mrefCallconv;
    mrefArity=x.mrefArity;
    mrefName=x.mrefName;
    mrefArgs = List.map (rescope_typ scoref) x.mrefArgs;
    mrefReturn= rescope_typ scoref x.mrefReturn }

let rescope_fref scoref x = 
  { frefParent = rescope_tref scoref x.frefParent;
    frefName= x.frefName;
    frefType= rescope_typ scoref x.frefType }

let rescope_ospec scoref (OverridesSpec(mref,typ)) = 
  OverridesSpec (rescope_mref scoref mref,rescope_typ scoref typ)

let rescope_fspec scoref x = 
  { fspecFieldRef = rescope_fref scoref x.fspecFieldRef;
    fspecEnclosingType = rescope_typ scoref x.fspecEnclosingType }

let rescope_mspec scoref x =
  let x1,x2,x3 = dest_mspec x in 
  mk_mref_mspec_in_typ (rescope_mref scoref x1,rescope_typ scoref x2,rescope_inst scoref x3)
  

(* -------------------------------------------------------------------- 
 * Instantiate polymorphism in types
 * -------------------------------------------------------------------- *)

let rec inst_tspec_aux num_free inst tspec = 
  {tspec with tspecInst = inst_inst_aux num_free inst tspec.tspecInst }
and inst_typ_aux num_free inst typ = 
  match typ with 
  | Type_ptr t       -> Type_ptr (inst_typ_aux num_free inst t)
  | Type_fptr t      -> Type_fptr (inst_callsig_aux num_free inst t)
  | Type_array (a,t) -> Type_array (a,inst_typ_aux num_free inst t)
  | Type_byref t     -> Type_byref (inst_typ_aux num_free inst t)
  | Type_boxed cr    -> Type_boxed (inst_tspec_aux num_free inst cr)
  | Type_value cr    -> Type_value (inst_tspec_aux num_free inst cr)
  | Type_other e     -> Type_other (find_extension "type" (fun ext -> if ext.internalTypeExtIs e then Some (ext.internalTypeExtInstAux num_free inst e) else None) !typ_extensions)
  | Type_tyvar  v -> 
      let v = u16_to_int v in 
      let top = List.length inst in 
      if v < num_free then typ else
      if v - num_free >= top then Type_tyvar (int_to_u16 (v - top)) else 
      inst_read inst (int_to_u16 (v - num_free)) 
  | x -> x
and inst_genactual_aux num_free inst ty = 
  inst_typ_aux num_free inst ty
    
and inst_inst_aux num_free inst i = List.map (inst_genactual_aux num_free inst) i
and inst_callsig_aux num_free inst  csig = 
  mk_callsig 
    (csig.callsigCallconv,List.map (inst_typ_aux num_free inst) csig.callsigArgs,inst_typ_aux num_free inst csig.callsigReturn)

let inst_typ     i t = inst_typ_aux 0 i t
let inst_inst    i t = inst_inst_aux 0 i t
let inst_tspec   i t = inst_tspec_aux 0 i t
let inst_callsig i t = inst_callsig_aux 0 i t

(* --------------------------------------------------------------------
 * MS-IL: Parameters, Return types and Locals
 * -------------------------------------------------------------------- *)

let mk_param (name,ty) =
    { paramName=name;
      paramDefault=None;
      paramMarshal=None;
      paramIn=false;
      paramOut=false;
      paramOptional=false;
      paramType=ty;
      paramCustomAttrs=mk_custom_attrs [] }
let name_of_param p = p.paramName
let mk_named_param (s,ty) = mk_param (Some s,ty)
let mk_unnamed_param ty = mk_param (None,ty)
let typ_of_param p = p.paramType

let typ_of_return p = p.returnType
let mk_return ty = 
    { returnMarshal=None;
      returnType=ty;
      returnCustomAttrs=mk_custom_attrs []  }

let mk_local ty = 
    { localPinned=false;
      localType=ty; }
let typ_of_local p = p.localType

(* --------------------------------------------------------------------
 * Generics.  Return the instantiation that governss
 * a specification for a field or method.
 * -------------------------------------------------------------------- *)

let active_inst_of_mspec mspec = 
  inst_add (inst_of_typ mspec.mspecEnclosingTypeF) mspec.mspecMethodInstF

let active_inst_of_fspec fspec = 
  inst_of_typ fspec.fspecEnclosingType

let actual_ret_of_mspec mr = 
  let inst = active_inst_of_mspec mr in 
  inst_typ inst (formal_ret_of_mspec mr)
let actual_args_of_mspec mr = 
  let inst = active_inst_of_mspec mr in 
  List.map (inst_typ inst) (formal_args_of_mspec mr)
let actual_typ_of_fspec fr = 
  let env = active_inst_of_fspec fr in 
  inst_typ env (formal_typ_of_fspec fr)

let actual_callsig_of_mspec mr = 
  let inst = active_inst_of_mspec mr in 
  inst_callsig inst (formal_callsig_of_mspec mr)

(* -------------------------------------------------------------------- 
 * 
 * -------------------------------------------------------------------- *)

let mk_ldc_i32 i = I_arith (AI_ldc (DT_I4,NUM_I4 i))

(* -------------------------------------------------------------------- 
 * Make a method mbody
 * -------------------------------------------------------------------- *)

let mk_ilmbody (zeroinit,locals,maxstack,code,tag) = 
  { ilZeroInit=zeroinit;
    ilMaxStack=int_to_i32 maxstack;
    ilNoInlining=false;
    ilLocals=locals;
    ilCode= code;
    ilSource=tag }

let mk_impl info = MethodBody_il (mk_ilmbody info)

(* -------------------------------------------------------------------- 
 * Make a constructor
 * -------------------------------------------------------------------- *)

let mk_void_return = mk_return Type_void

let mk_ctor (access,args,impl) = 
  { mdName=".ctor";
    mdKind=MethodKind_ctor;
    mdCallconv=instance_callconv;
    mdParams=args;
    mdReturn= mk_void_return;
    mdAccess=access;
    mdBody= mk_mbody impl;
    mdCodeKind=MethodCodeKind_il;
      mdInternalCall=false;
    mdManaged=true;
    mdForwardRef=false;

    mdSecurityDecls=mk_security_decls [];
    mdHasSecurity=false;
    mdEntrypoint=false;

    mdGenericParams=mk_empty_gparams;
    mdExport=None;
    mdVtableEntry=None;
    mdReqSecObj=false;
    mdHideBySig=false;
    mdSpecialName=true;
    mdUnmanagedExport=false;
    mdSynchronized=false;
    mdMustRun=false;
    mdPreserveSig=false;
    mdCustomAttrs = mk_custom_attrs [];
 }

(* -------------------------------------------------------------------- 
 * Do-nothing ctor, just pass on to monomorphic superclass
 * -------------------------------------------------------------------- *)

let mk_ldargs args =
  [ ldarg_0; ] @
  list_mapi (fun i _ -> I_ldarg (int_to_u16 (i+1))) args 

let mk_call_superclass_constructor_prim args mspec =
  mk_ldargs args @ [ mk_normal_call mspec ]

let mk_nongeneric_call_superclass_constructor ((args:typ list),super_tref) =
  mk_call_superclass_constructor_prim args (mk_nongeneric_ctor_mspec (super_tref,AsObject,[]))

let mk_call_superclass_constructor ((args:typ list),tspec) =
  mk_call_superclass_constructor_prim args (mk_ctor_mspec_for_boxed_tspec (tspec,[]))

let mk_nongeneric_call_superclass_constructor2 ((args:param list),super_tref) =
  mk_call_superclass_constructor_prim args (mk_nongeneric_ctor_mspec (super_tref,AsObject,[]))

let mk_call_superclass_constructor2 ((args:param list),tspec) =
  mk_call_superclass_constructor_prim args (mk_ctor_mspec_for_boxed_tspec (tspec,[]))


let mk_normal_stfld fspec = I_stfld (Aligned,Nonvolatile,fspec)
let mk_normal_stsfld fspec = I_stsfld (Nonvolatile,fspec)
let mk_normal_ldsfld fspec = I_ldsfld (Nonvolatile,fspec)
let mk_normal_ldfld fspec = I_ldfld (Aligned,Nonvolatile,fspec)
let mk_normal_ldflda fspec = I_ldflda fspec
let mk_normal_stind dt = I_stind (Aligned,Nonvolatile,dt)
let mk_normal_ldind dt = I_ldind (Aligned,Nonvolatile,dt)
let mk_normal_ldobj dt = I_ldobj(Aligned,Nonvolatile,dt)
let mk_normal_stobj dt = I_stobj(Aligned,Nonvolatile,dt)
let mk_normal_cpind dt = [ I_ldind (Aligned,Nonvolatile,dt); 
         I_stind (Aligned,Nonvolatile,dt) ]  (* REVIEW: check me *)

let mk_nongeneric_nothing_ctor tag super_tref args = 
  mk_ctor(MemAccess_public,
    args,
    mk_impl(false,[],8,
      nonbranching_instrs_to_code  
        (mk_nongeneric_call_superclass_constructor2 (args,super_tref)),tag))

let mk_nothing_ctor tag super_tspec args = 
  mk_ctor(MemAccess_public,
    args,
    mk_impl(false,[],8,
      nonbranching_instrs_to_code 
        (mk_call_superclass_constructor2 (args,super_tspec)),tag))

(* -------------------------------------------------------------------- 
 * Make a static, top level monomophic method - very useful for
 * creating helper methods for internal use.
 * -------------------------------------------------------------------- *)
let mk_static_mdef (genparams,nm,access,args,ret,impl) = 
  { mdGenericParams=genparams;
    mdName=nm;
    mdCallconv = static_callconv;
    mdKind=MethodKind_static;
    mdParams=  args;
    mdReturn= ret;
    mdAccess=access;
    mdHasSecurity=false;
    mdSecurityDecls=mk_security_decls [];
    mdEntrypoint=false;
    mdExport=None;
    mdCustomAttrs = mk_custom_attrs [];
    mdVtableEntry=None;
    mdBody= mk_mbody impl;
    mdCodeKind=MethodCodeKind_il;
      mdInternalCall=false;
    mdManaged=true;
    mdForwardRef=false;
    mdReqSecObj=false;
    mdHideBySig=false;
    mdSpecialName=false;
    mdUnmanagedExport=false;
    mdSynchronized=false;
    mdMustRun=false;
    mdPreserveSig=false; }

let mk_static_nongeneric_mdef (nm,access,args,ret,impl) = 
  mk_static_mdef (mk_empty_gparams,nm,access,args,ret,impl)

let mk_cctor impl = 
    { mdName=".cctor";
      mdCallconv=static_callconv;
      mdGenericParams=mk_empty_gparams;
      mdKind=MethodKind_cctor;
      mdParams=[];
      mdReturn=mk_void_return;
      mdAccess=MemAccess_private; 
      mdEntrypoint=false;
      mdHasSecurity=false;
      mdSecurityDecls=mk_security_decls [];
      mdExport=None;
      mdCustomAttrs = mk_custom_attrs [];
      mdVtableEntry=None;
      mdBody= mk_mbody impl; 
      mdCodeKind=MethodCodeKind_il;
      mdInternalCall=false;
      mdManaged=true;
      mdForwardRef=false;
      mdReqSecObj=false;
      mdHideBySig=false;
      mdSpecialName=true;
      mdUnmanagedExport=false; 
      mdSynchronized=false;
      mdMustRun=false;
      mdPreserveSig=false;  } 

let mdef_is_virt mdef = 
  match mdef.mdKind with MethodKind_virtual _ -> true | _ -> false


(* -------------------------------------------------------------------- 
 * Make a virtual method, where the overriding is simply the default
 * (i.e. overrides by name/signature)
 * -------------------------------------------------------------------- *)

let mk_ospec (typ,callconv,nm,genparams,formal_args,formal_ret) =
  OverridesSpec (mk_mref (tref_of_typ typ, callconv, nm, genparams, 
        formal_args,formal_ret), typ)

let formal_mref_of_ospec (OverridesSpec(mref,_)) = mref
let enclosing_typ_of_ospec (OverridesSpec(_,ty)) = ty
let callconv_of_ospec x = (formal_mref_of_ospec >> callconv_of_mref) x
let formal_ret_of_ospec x = (formal_mref_of_ospec >> ret_of_mref) x
let formal_args_of_ospec x = (formal_mref_of_ospec >> args_of_mref) x
let tref_of_ospec x = (formal_mref_of_ospec >> tref_of_mref) x
let name_of_ospec x = (formal_mref_of_ospec >> name_of_mref) x
let formal_callsig_of_ospec x = (formal_mref_of_ospec >> callsig_of_mref) x

let active_inst_of_ospec (OverridesSpec(_,typ)) = 
  inst_of_typ typ

let actual_callsig_of_ospec mr = 
  let inst = active_inst_of_ospec mr in 
  inst_callsig inst (formal_callsig_of_ospec mr)

let mref_of_ospec (OverridesSpec(mref,_)) = mref
let tspec_of_ospec (OverridesSpec(_,tspec)) = tspec

let mk_generic_virtual_mdef (nm,access,genparams,overrides_info,actual_args,actual_ret,impl) = 
  { mdName=nm;
    mdGenericParams=genparams;
    mdCallconv=instance_callconv;
    mdKind=
    MethodKind_virtual 
      { virtFinal=false; 
        virtNewslot = false; 
        virtStrict=true;
        virtAbstract=(match impl with MethodBody_abstract -> true | _ -> false) ; 
        virtOverrides =  (match overrides_info with None -> None | Some (typ,formal_args,formal_ret) -> Some (mk_ospec (typ,instance_callconv,nm,List.length genparams,formal_args,formal_ret)))  };
    mdParams= actual_args;
    mdReturn=actual_ret;
    mdAccess=access;
    mdEntrypoint=false;
    mdHasSecurity=false;
    mdSecurityDecls=mk_security_decls [];
    mdExport=None;
    mdCustomAttrs = mk_custom_attrs [];
    mdVtableEntry=None;
    mdBody= mk_mbody impl;
    mdCodeKind=MethodCodeKind_il;
      mdInternalCall=false;
    mdManaged=true;
    mdForwardRef=false;
    mdReqSecObj=false;
    mdHideBySig=false;
    mdSpecialName=false;
    mdUnmanagedExport=false; 
    mdSynchronized=false;
    mdMustRun=false;
    mdPreserveSig=false; }
    
let mk_virtual_mdef (nm,access,overrides,args,ret,impl) =  
  mk_generic_virtual_mdef (nm,access,mk_empty_gparams,overrides,args,ret,impl)

let mk_normal_virtual_mdef (nm,access,args,ret,impl) = 
  mk_virtual_mdef(nm,access,None,args,ret,impl)


let mk_generic_instance_mdef (nm,access,genparams, actual_args,actual_ret, impl) = 
  { mdName=nm;
    mdGenericParams=genparams;
    mdCallconv=instance_callconv;
    mdKind=MethodKind_nonvirtual;
    mdParams= actual_args;
    mdReturn=actual_ret;
    mdAccess=access;
    mdEntrypoint=false;
    mdHasSecurity=false;
    mdSecurityDecls=mk_security_decls [];
    mdExport=None;
    mdCustomAttrs = mk_custom_attrs [];
    mdVtableEntry=None;
    mdBody= mk_mbody impl;
    mdCodeKind=MethodCodeKind_il;
    mdInternalCall=false;
    mdManaged=true;
    mdForwardRef=false;
    mdReqSecObj=false;
    mdHideBySig=false;
    mdSpecialName=false;
    mdUnmanagedExport=false; 
    mdSynchronized=false;
    mdMustRun=false;
    mdPreserveSig=false; }
    
let mk_instance_mdef (nm,access,args,ret,impl) =  
  mk_generic_instance_mdef (nm,access,mk_empty_gparams,args,ret,impl)


(* -------------------------------------------------------------------- 
 * Add some code to the end of the .cctor for a type.  Create a .cctor
 * if one doesn't exist already.
 * -------------------------------------------------------------------- *)

let ilmbody_code2code f il  =
  {il with ilCode = f il.ilCode}

let mdef_code2code f md  =
  let il = 
    match dest_mbody md.mdBody with 
    | MethodBody_il il-> il 
    | _ -> failwith "mdef_code2code - method not IL" in 
  let b = MethodBody_il (ilmbody_code2code f il) in 
  {md with mdBody= mk_mbody b }  

let prepend_instrs_to_code c1 c2 = 
  let internalLab = generate_code_label () in 
  join_code (check_code (mk_bblock {bblockLabel=internalLab;
          bblockInstrs=Array.of_list (c1 @ [ I_br (unique_entry_of_code c2)])})) c2

let prepend_instrs_to_ilmbody new_code ilmbody = 
  ilmbody_code2code (prepend_instrs_to_code new_code) ilmbody

let prepend_instrs_to_mdef new_code md  = 
  mdef_code2code (prepend_instrs_to_code new_code) md

let cctor_id = (".cctor",0) 
(* Creates cctor if needed *)
let cdef_cctorCode2CodeOrCreate tag f cd = 
  let mdefs = cd.tdMethodDefs in 
  let md,mdefs = 
    match find_mdefs_by_arity cctor_id mdefs with 
    | [mdef] -> mdef,filter_mdefs (fun md -> not (string_eq md.mdName (fst cctor_id))) mdefs
    | [] -> mk_cctor (mk_impl (false,[],1,nonbranching_instrs_to_code [ ],tag)), mdefs
    | _ -> failwith "bad method table: more than one .cctor found" in
  let md' = f md in
  {cd with tdMethodDefs = add_mdef md' mdefs}

let name_of_mdef m = m.mdName

let params_of_mdef m = m.mdParams

let callconv_of_mdef m = m.mdCallconv

let mdef_is_il m =
  match dest_mbody m.mdBody with
  | MethodBody_il _ -> true
  | _ -> false

let ilmbody_of_mdef m =
  match dest_mbody m.mdBody with
  | MethodBody_il il -> il
  | _ -> failwith "ilmbody_of_mdef: not IL"

let try_code_of_mdef md = 
  match dest_mbody md.mdBody with 
  | MethodBody_il il-> Some il.ilCode
  | _ -> None

let code_of_mdef md = 
  match try_code_of_mdef md with 
  | Some x -> x
  | None -> failwith "code_of_mdef: not IL" 

let entry_of_mdef md = unique_entry_of_code (code_of_mdef md)

let args_of_mdef md = List.map typ_of_param md.mdParams

let retty_of_mdef md = typ_of_return md.mdReturn

let callsig_of_mdef md =
  mk_callsig (md.mdCallconv,args_of_mdef md,retty_of_mdef md)

let mk_mref_to_mdef (tref,md) =
  mk_mref (tref,md.mdCallconv,md.mdName,List.length md.mdGenericParams,args_of_mdef md,retty_of_mdef md)

let mk_fref_to_fdef (tref,fdef) =   mk_fref_in_tref (tref, fdef.fdName, fdef.fdType)

let mref_for_mdef scope (tdefs,tdef) mdef = mk_mref_to_mdef (tref_for_nested_tdef scope (tdefs,tdef), mdef)
let fref_for_fdef scope (tdefs,tdef) fdef = mk_fref_in_tref (tref_for_nested_tdef scope (tdefs,tdef), fdef.fdName, fdef.fdType)

let mk_mspec_to_mdef (typ,md,minst) =
  mk_mspec_in_tref (tref_of_typ typ, 
      boxity_of_typ typ,
      md.mdCallconv, 
      md.mdName, 
      args_of_mdef md,
      retty_of_mdef md,
      inst_of_typ typ,
      minst)


(* Creates cctor if needed *)
let prepend_instrs_to_cctor instrs tag cd = 
  cdef_cctorCode2CodeOrCreate tag (prepend_instrs_to_mdef instrs) cd
    

let mk_fdef (isStatic,nm,ty,init,at,access) =
   { fdName=nm;
     fdType=ty;
     fdStatic = isStatic; 
     fdInit = init;
     fdData=at;
     fdOffset=None;
     fdSpecialName = false;
     fdMarshal=None; 
     fdNotSerialized=false;
     fdInitOnly = false;
     fdLiteral = false; 
     fdAccess = access; 
     fdCustomAttrs=mk_custom_attrs [] }

let mk_instance_fdef (nm,ty,init,access) = mk_fdef (false,nm,ty,init,None,access)
let mk_static_fdef (nm,ty,init,at,access) = mk_fdef (true,nm,ty,init,at,access)

(* -------------------------------------------------------------------- 
 * Scopes for allocating new temporary variables.
 * -------------------------------------------------------------------- *)

type tmps = { num_old_locals: int; newlocals: local ResizeArray.t }
let alloc_tmp tmps loc =
  let locn = int_to_u16(tmps.num_old_locals + ResizeArray.length tmps.newlocals) in 
  ResizeArray.add tmps.newlocals loc;
  locn

let get_tmps tmps = ResizeArray.to_list tmps.newlocals
let new_tmps n = { num_old_locals=n; newlocals=ResizeArray.create 10 }

(* -------------------------------------------------------------------- 
 * How many arguments does a method take?
 * -------------------------------------------------------------------- *)

let num_args md = 
  let basic = List.length md.mdParams in 
  match md.mdKind with 
    MethodKind_ctor -> basic+1
  | MethodKind_virtual _ -> basic + 1
  | MethodKind_nonvirtual -> basic+1
  | _ -> basic


(* -------------------------------------------------------------------- 
 * Add fields and types to tables, with decent error messages
 * when clashes occur...
 * -------------------------------------------------------------------- *)

exception Not_unique_field of field_def
exception Not_unique_method of method_def

let typ_of_fdef f = f.fdType
let name_of_fdef f = f.fdName 

let dest_fdefs (Fields lomm) = LazyOrderedMultiMap.dest_objs lomm
let add_fdef x (Fields lomm) = Fields (LazyOrderedMultiMap.add_obj name_of_fdef x lomm)
let mk_fdefs l =  Fields (LazyOrderedMultiMap.mk_objs name_of_fdef l)
let mk_lazy_fdefs l =  Fields (LazyOrderedMultiMap.mk_lazy_objs name_of_fdef l) 
let filter_fdefs f (Fields t) = Fields (LazyOrderedMultiMap.filter_objs name_of_fdef f t)
let find_fdefs x (Fields f) = LazyOrderedMultiMap.find_obj x f

let add_fdef_to_tdef f cd = {cd with tdFieldDefs = add_fdef f cd.tdFieldDefs }

let name_of_event e = e.eventName
let name_of_property p = p.propName

let dest_edefs (Events f) = LazyOrderedMultiMap.dest_objs f
let add_event x (Events y) = Events (LazyOrderedMultiMap.add_obj name_of_event x y)
let mk_events l =  Events (LazyOrderedMultiMap.mk_objs name_of_event l)
let mk_lazy_events l =  Events (LazyOrderedMultiMap.mk_lazy_objs name_of_event l) 
let filter_edefs f (Events t) = Events (LazyOrderedMultiMap.filter_objs name_of_event f t)
let find_edefs x (Events f) = LazyOrderedMultiMap.find_obj x f

let dest_pdefs (Properties f) = LazyOrderedMultiMap.dest_objs f
let add_property x (Properties y:properties) = Properties (LazyOrderedMultiMap.add_obj name_of_property x y)
let mk_properties l =  Properties (LazyOrderedMultiMap.mk_objs name_of_property l)
let mk_lazy_properties l =  Properties (LazyOrderedMultiMap.mk_lazy_objs name_of_property l) 
let filter_pdefs f (Properties t) = Properties (LazyOrderedMultiMap.filter_objs name_of_property f t)
let find_pdefs x (Properties f) = LazyOrderedMultiMap.find_obj x f


let dest_exported_types (ExportedTypes ltab) = Pmap.fold (fun x y r -> y::r) (Lazy.force ltab) []
let add_exported_type_to_tab y tab =
  let key = y.exportedTypeName in 
  Pmap.add key y tab
let add_exported_type y (ExportedTypes ltab) = 
  ExportedTypes (lazy_map (add_exported_type_to_tab y) ltab)
let mk_exported_types l =  
  ExportedTypes (notlazy (List.fold_right add_exported_type_to_tab l Pmap.empty))
let mk_lazy_exported_types l =  
  ExportedTypes (lazy (List.fold_right add_exported_type_to_tab (Lazy.force l) Pmap.empty))
let find_exported_type x (ExportedTypes ltab) = Pmap.find x (Lazy.force ltab)

let dest_nested_exported_types (NestedExportedTypes ltab) = Pmap.fold (fun x y r -> y::r) (Lazy.force ltab) []
let add_nested_exported_type_to_tab y tab =
  let key = y.nestedExportedTypeName in 
  Pmap.add key y tab
let add_nested_exported_type y (NestedExportedTypes ltab) = 
  NestedExportedTypes (lazy_map (add_nested_exported_type_to_tab y) ltab)
let mk_nested_exported_types l =  
  NestedExportedTypes (notlazy (List.fold_right add_nested_exported_type_to_tab l Pmap.empty))
let mk_lazy_nested_exported_types l =  
  NestedExportedTypes (lazy (List.fold_right add_nested_exported_type_to_tab (Lazy.force l) Pmap.empty))
let find_nested_exported_type x (NestedExportedTypes ltab) = Pmap.find x (Lazy.force ltab)

let dest_resources (Resources ltab) = (Lazy.force ltab)
let add_resource y (Resources ltab) = Resources (lazy_map (fun rs -> y::rs) ltab)
let mk_resources l =  Resources (notlazy l)
let mk_lazy_resources l =  Resources l

let dest_mimpls (MethodImpls ltab) = Pmap.fold (fun x y r -> y@r) (Lazy.force ltab) []
let add_mimpl_to_tab y tab =
   let key = (name_of_mref (mref_of_ospec y.mimplOverrides),List.length (args_of_mref (mref_of_ospec y.mimplOverrides))) in 
  let prev = Pmap.tryfind_multi key tab in 
  Pmap.add key (y::prev) tab
let add_mimpl y (MethodImpls ltab) = MethodImpls (lazy_map (add_mimpl_to_tab y) ltab)
let mk_mimpls l =  MethodImpls (notlazy (List.fold_right add_mimpl_to_tab l Pmap.empty))
let mk_lazy_mimpls l =  MethodImpls (lazy (List.fold_right add_mimpl_to_tab (Lazy.force l) Pmap.empty))
let filter_mimpls f (MethodImpls ltab) =  MethodImpls (lazy_map (Pmap.map (List.filter f)) ltab)


let fields_of_tdef td = td.tdFieldDefs
let methods_of_tdef td = td.tdMethodDefs
let mimpls_of_tdef td = td.tdMethodImpls
let properties_of_tdef td = td.tdProperties
let events_of_tdef td = td.tdEvents
let custom_attrs_of_tdef td = td.tdCustomAttrs
let security_decls_of_tdef td = td.tdSecurityDecls

(* -------------------------------------------------------------------- 
 * Make a constructor that simply takes its arguments and stuffs
 * them in fields.  preblock is how to call the superclass constructor....
 * -------------------------------------------------------------------- *)

let mk_storage_ctor tag preblock tspec flds = 
  mk_ctor(MemAccess_public,
          List.map mk_named_param flds,
          mk_impl
            (false,[],2,
             nonbranching_instrs_to_code
               begin 
                 (match tag with Some x -> [I_seqpoint x] | None -> []) @ 
                 preblock @
                 begin 
                   List.concat (list_mapi (fun n (nm,ty) -> 
                     [ ldarg_0;
                       I_ldarg (int_to_u16 (n+1));
                       mk_normal_stfld 
                         (mk_fspec_in_boxed_tspec (tspec,nm,ty));
                     ])  flds)
                 end
               end,tag))
    
let mk_simple_storage_ctor tag base_tspec derived_tspec flds = 
  let preblock = 
    match base_tspec with 
      None -> []
    | Some tspec -> 
        ([ ldarg_0; 
           mk_normal_call (mk_ctor_mspec_for_boxed_tspec (tspec,[])) ]) in 
  mk_storage_ctor tag preblock derived_tspec flds 


let mk_generic_class (nm,access,genparams,extends,impl,methods,fields,props,events,attrs) =
  { tdKind=TypeDef_class;
    tdName=nm;
    tdGenericParams= genparams;
    tdAccess = access;
    tdImplements = impl;
    tdAbstract = false;
    tdSealed = false;
    tdSerializable = false;
    tdComInterop=false;
    tdSpecialName=false;
    tdLayout=TypeLayout_auto;
    tdEncoding=TypeEncoding_ansi;
    tdInitSemantics=TypeInit_beforefield;
    tdExtends = Some extends;
    tdMethodDefs= methods; 
    tdFieldDefs= fields;
    tdNested=mk_tdefs [];
    tdCustomAttrs=attrs;
    tdMethodImpls=mk_mimpls [];
    tdProperties=props;
    tdEvents=events;
    tdSecurityDecls=mk_security_decls []; 
    tdHasSecurity=false;
} 
    
let mk_rawdata_vtdef ilg (nm,size,pack) =
  { tdKind=TypeDef_valuetype;
    tdName = nm;
    tdGenericParams= [];
    tdAccess = TypeAccess_private;
    tdImplements = [];
    tdAbstract = false;
    tdSealed = true;
    tdExtends = Some ilg.typ_ValueType;
    tdComInterop=false;    
    tdSerializable = false;
    tdSpecialName=false;
    tdLayout=TypeLayout_explicit { typeSize=Some size; typePack=Some pack };
    tdEncoding=TypeEncoding_ansi;
    tdInitSemantics=TypeInit_beforefield;
    tdMethodDefs= mk_mdefs []; 
    tdFieldDefs= mk_fdefs [];
    tdNested=mk_tdefs [];
    tdCustomAttrs=mk_custom_attrs [];
    tdMethodImpls=mk_mimpls [];
    tdProperties=mk_properties [];
    tdEvents=mk_events [];
    tdSecurityDecls=mk_security_decls []; 
    tdHasSecurity=false;  }


let mk_simple_tdef ilg (nm,access,methods,fields,props,events,attrs) =
  mk_generic_class (nm,access, mk_empty_gparams, ilg.typ_Object, [], methods,fields,props,events,attrs)

let mk_toplevel_tdef ilg (methods,fields) = mk_simple_tdef ilg (tname_for_toplevel,TypeAccess_public, methods,fields,mk_properties [], mk_events [], mk_custom_attrs [])

let dest_tdefs_with_toplevel_first ilg tdefs = 
  let l = dest_tdefs tdefs in 
  let top,nontop = List.partition (fun td -> tname_eq (name_of_tdef td) tname_for_toplevel) l in 
  let top2 = if isnull top then [mk_toplevel_tdef ilg (mk_mdefs [], mk_fdefs [])] else top in 
  top2@nontop

let add_toplevel_mdef ilg md modul =
  let tdef = 
    try 
      let cdef = find_tdef tname_for_toplevel modul.modulTypeDefs in
       {cdef with tdMethodDefs= add_mdef md cdef.tdMethodDefs}
    with Not_found -> 
      mk_toplevel_tdef ilg (mk_mdefs [md],mk_fdefs []) in 
  {modul with modulTypeDefs= replace_tdef tdef modul.modulTypeDefs}



let mk_genactual ty = (ty:typ)

let mk_simple_mainmod assname modname dll tdefs = 
    { modulManifest= 
      Some { manifestName=assname;
             manifestAuxModuleHashAlgorithm=Int32.of_int 0x8004;
             manifestSecurityDecls=mk_security_decls [];
             manifestPublicKey= None;
             manifestVersion= None;
             manifestLocale=None;
             manifestCustomAttrs=mk_custom_attrs [];
             manifestLongevity=LongevityUnspecified;
             manifestDisableJitOptimizations=false;
             manifestJitTracking=true;
             manifestExportedTypes=mk_exported_types [];
             manifestEntrypointElsewhere=None
           };
      modulCustomAttrs=mk_custom_attrs [];
      modulName=modname;
      modulNativeResources=None;
      
      modulTypeDefs=tdefs;
      modulSubSystem=default_modulSubSystem;
      modulDLL=dll;
      modulILonly=true;
      modul32bit=false;
      modulPhysAlignment=default_modulPhysAlignment;
      modulVirtAlignment=default_modulVirtAlignment;
      modulImageBase=default_modulImageBase;
      modulResources=mk_resources [];
(*      modulFixups=[]; *)
    }


(*-----------------------------------------------------------------------
 * Intermediate parsing structure for exception tables....
 *----------------------------------------------------------------------*)

type seh_clause = 
  | SEH_finally of (code_label * code_label)
  | SEH_fault  of (code_label * code_label)
  | SEH_filter_catch of (code_label * code_label) * (code_label * code_label)
  | SEH_type_catch of typ * (code_label * code_label)

type exception_spec = 
    { exnRange: (code_label * code_label);
      exnClauses: seh_clause list }

type exceptions = exception_spec list

(*-----------------------------------------------------------------------
 * [instructions_to_code] makes the basic block structure of code from
 * a primitive array of instructions.  We
 * do this be iterating over the instructions, pushing new basic blocks 
 * everytime we encounter an address that has been recorded in 
 * [bbstarts].
 *----------------------------------------------------------------------*)

let rec partition3 p l = 
  match l with
    [] -> ([],[],[]) 
  | h::t -> 
      if p h then 
  let (y,z) = List.partition p t in 
  ([],h::y,z) 
      else 
  let (x,y,z) = partition3 p t in 
  (h::x,y,z) 

type local_spec = 
    { locRange: (code_label * code_label);
      locInfos: local_debug_info list }

type structspec = SEH of exception_spec | LOCAL of local_spec 
            
let build_code meth_name lab2pc instrs tryspecs localspecs =

  (* Find all the interesting looking labels that form the boundaries of basic blocks. *)
  (* These are the destinations of branches and the boundaries of both exceptions and *)
  (* those blocks where locals are live. *)
  let bbstarts = 
    let res = ref CodeLabels.empty in 
    let add_range (a,b) = res := CodeLabels.insert a (CodeLabels.insert b !res) in 
    Array.iter (fun i -> res := CodeLabels.addL (destinations_of_instr i) !res) instrs;
    List.iter (fun espec -> 
      add_range espec.exnRange;
      List.iter (function 
        | SEH_finally r1 | SEH_fault r1 | SEH_type_catch (_,r1)-> add_range r1
        | SEH_filter_catch (r1,r2) -> add_range r1; add_range r2) espec.exnClauses)
      tryspecs;
    List.iter (fun l -> add_range l.locRange) localspecs;
    !res in 

  (* Construct a map that gives a unique code_label for each label that *)
  (* might be a boundary of a basic block.  These will be the labels *)
  (* for the basic blocks we end up creating. *)
  let lab2cl_map = Hashtbl.create 10 in 
  let pc2cl_map = Hashtbl.create 10 in 
  let add_bbstart_pc pc pcs cls = 
    if Hashtbl.mem pc2cl_map pc then 
      Hashtbl.find pc2cl_map pc, pcs, cls
    else 
      let cl = generate_code_label () in  
      Hashtbl.add pc2cl_map pc cl;
      cl, pc::pcs, CodeLabels.insert cl cls in 

  let bbstart_pcs, bbstart_code_labs  = 
    CodeLabels.fold
      (fun bbstart_lab (pcs, cls) -> 
        let pc = lab2pc bbstart_lab in 
        if logging then dprintf2 "bblock starts with label %s at pc %d\n" (string_of_code_label bbstart_lab) pc;
        let cl,pcs',cls' = add_bbstart_pc pc pcs cls in 
        Hashtbl.add lab2cl_map bbstart_lab cl;
        pcs',
        cls')
      bbstarts 
      ([], CodeLabels.empty) in 
  let cl0,bbstart_pcs, bbstart_code_labs = add_bbstart_pc 0 bbstart_pcs bbstart_code_labs in 

  let lab2cl bb_lab = try Hashtbl.find lab2cl_map bb_lab  with Not_found -> failwith ("basic block label "^string_of_code_label bb_lab^" not declared")  in 
  let pc2cl pc = try Hashtbl.find pc2cl_map pc with Not_found -> failwith ("internal error while mapping pc "^string_of_int pc^" to code label")  in 

  let remap_labels i =
    match i with 
    | I_leave l -> I_leave(lab2cl l)
    | I_br l -> I_br (lab2cl l)
    | I_other e -> I_other (find_extension "instr" (fun ext -> if ext.internalInstrExtIs e then Some (ext.internalInstrExtRelabel lab2cl e) else None) !instr_extensions)
    | I_brcmp (x,l1,l2) -> I_brcmp(x,lab2cl l1, lab2cl l2)
    | I_switch (ls,l) -> I_switch(List.map lab2cl ls, lab2cl l)
    | _ -> i in
  let basic_instructions = Array.map remap_labels instrs in 

  let range_inside_range (start_pc1,end_pc1) (start_pc2,end_pc2)  =
    (start_pc1:int) >= start_pc2 && start_pc1 < end_pc2 &
    (end_pc1:int) > start_pc2 && end_pc1 <= end_pc2 in 
  
(*  let restrict_range_to_range (start_pc1,end_pc1) (start_pc2,end_pc2)  =
    (max (start_pc1:int) start_pc2, min (end_pc1:int) end_pc2) in  *)
  
  let disjoint_range (start_pc1,end_pc1) (start_pc2,end_pc2) =
    ((start_pc1 : int) < start_pc2 && end_pc1 <= start_pc2) or
    (start_pc1 >= end_pc2 && end_pc1 > end_pc2) in 

  let merge_ranges (start_pc1,end_pc1) (start_pc2,end_pc2) =
    (min (start_pc1:int) start_pc2, max (end_pc1:int) end_pc2) in 
  
  let labels_to_range (l1,l2) = lab2pc l1, lab2pc l2 in 

  let lrange_inside_lrange ls1 ls2 = 
    range_inside_range (labels_to_range ls1) (labels_to_range ls2) in 
  let disjoint_lranges ls1 ls2 = 
    disjoint_range (labels_to_range ls1) (labels_to_range ls2) in 
  let lranges_of_clause cl = 
    match cl with 
    | SEH_finally r1 -> [r1]
    | SEH_fault r1 -> [r1]
    | SEH_filter_catch (r1,r2) -> [r1;r2]
    | SEH_type_catch (ty,r1) -> [r1]  in 
  let clause_inside_lrange cl lr =
    List.for_all (fun lr1 -> lrange_inside_lrange lr1 lr) (lranges_of_clause cl) in
  let clauses_inside_lrange cls lr = 
    List.for_all 
      (fun cl -> clause_inside_lrange cl lr)
      cls in 
  
  let tryspec_inside_lrange tryspec1 lr =
    (lrange_inside_lrange tryspec1.exnRange lr &
     clauses_inside_lrange tryspec1.exnClauses lr) in

  let tryspec_inside_clause tryspec1 cl =
    List.exists (fun lr -> tryspec_inside_lrange tryspec1 lr) (lranges_of_clause cl) in 

  let locspec_inside_clause locspec1 cl =
    List.exists (fun lr -> lrange_inside_lrange locspec1.locRange lr) (lranges_of_clause cl) in 

  let tryspec_inside_tryspec tryspec1 tryspec2 =
    tryspec_inside_lrange tryspec1 tryspec2.exnRange or
    List.exists (fun c2 -> tryspec_inside_clause tryspec1 c2) tryspec2.exnClauses in 
  
  let locspec_inside_tryspec locspec1 tryspec2 =
    lrange_inside_lrange locspec1.locRange tryspec2.exnRange or
    List.exists (fun c2 -> locspec_inside_clause locspec1 c2) tryspec2.exnClauses in 
  
  let tryspec_inside_locspec tryspec1 locspec2 =
    tryspec_inside_lrange tryspec1 locspec2.locRange in 
  
  let disjoint_clause_and_lrange cl lr =
    List.for_all (fun lr1 -> disjoint_lranges lr1 lr) (lranges_of_clause cl) in
  
  let disjoint_clauses_and_lrange cls lr = 
    List.for_all (fun cl -> disjoint_clause_and_lrange cl lr) cls in 
  
  let disjoint_tryspec_and_lrange tryspec1 lr =
    (disjoint_lranges tryspec1.exnRange lr &
     disjoint_clauses_and_lrange tryspec1.exnClauses lr) in
  
  let disjoint_tryspec_and_clause tryspec1 cl =
    List.for_all (fun lr -> disjoint_tryspec_and_lrange tryspec1 lr) (lranges_of_clause cl) in 

  let tryspec_disjoint_from_tryspec tryspec1 tryspec2 =
    disjoint_tryspec_and_lrange tryspec1 tryspec2.exnRange &
    List.for_all (fun c2 -> disjoint_tryspec_and_clause tryspec1 c2) tryspec2.exnClauses in 
  
  let tryspec_disjoint_from_locspec tryspec1 locspec2 =
    disjoint_tryspec_and_lrange tryspec1 locspec2.locRange in
  
  let locspec_disjoint_from_locspec locspec1 locspec2 =
    disjoint_lranges locspec1.locRange locspec2.locRange in
  
  let locspec_inside_locspec locspec1 locspec2 =
    lrange_inside_lrange locspec1.locRange locspec2.locRange in
  
  let tryspec_inside_or_disjoint_from_tryspec tryspec1 tryspec2 =
    tryspec_inside_tryspec tryspec1 tryspec2 or tryspec_disjoint_from_tryspec tryspec1 tryspec2 in 
  
  let locspec_inside_or_disjoint_from_tryspec locspec1 tryspec2 =
    locspec_inside_tryspec locspec1 tryspec2 or tryspec_disjoint_from_locspec tryspec2 locspec1 in 
  
  let locspec_inside_or_disjoint_from_locspec locspec1 locspec2 =
    locspec_inside_locspec locspec1 locspec2 or locspec_disjoint_from_locspec locspec1 locspec2 in 
  
  let tryspec_inside_or_disjoint_from_locspec tryspec1 locspec2 =
    tryspec_inside_locspec tryspec1 locspec2 or tryspec_disjoint_from_locspec tryspec1 locspec2 in 
  
  let tryspec_inside_or_disjoint_from_structspecs tryspec specs =
    List.for_all (function SEH ts' -> tryspec_inside_or_disjoint_from_tryspec tryspec ts' | LOCAL ls -> tryspec_inside_or_disjoint_from_locspec tryspec ls) specs in 

  let locspec_inside_or_disjoint_from_structspecs locspec specs =
    List.for_all (function SEH ts' -> locspec_inside_or_disjoint_from_tryspec locspec ts' | LOCAL ls -> locspec_inside_or_disjoint_from_locspec locspec ls) specs in

  let structspec_inside_structspec specA specB = (* only for sanity checks, then can be removed *)
      match specA,specB with
        | SEH   tryspecA,SEH   tryspecB -> tryspec_inside_tryspec tryspecA tryspecB
        | SEH   tryspecA,LOCAL locspecB -> tryspec_inside_locspec tryspecA locspecB
        | LOCAL locspecA,SEH   tryspecB -> locspec_inside_tryspec locspecA tryspecB
        | LOCAL locspecA,LOCAL locspecB -> locspec_inside_locspec locspecA locspecB
  in

  (* extent (or size) is the sum of range extents *)
  (* We want to build in increasing containment-order, that's a partial order. *)
  (* Size-order implies containment-order, and size-order is a total order. *)
  let extent_range (start_pc,end_pc) = end_pc - start_pc in
  let extent_lrange lrange = extent_range (labels_to_range lrange) in
  let extent_locspec locspec = extent_lrange locspec.locRange in
  let extent_list  extent_item items = List.fold_left (fun acc item -> acc + extent_item item) 0 items in
  let extent_list2 extent_item items = List.fold_left (fun acc item -> acc + extent_item item) 0 items in    
  let extent_clause cl = extent_list extent_lrange (lranges_of_clause cl) in
  let extent_tryspec tryspec = extent_lrange tryspec.exnRange + (extent_list2 extent_clause tryspec.exnClauses) in
  let extent_structspec = function LOCAL locspec -> extent_locspec locspec | SEH tryspec -> extent_tryspec tryspec in

  (* DIAGNOSTICS: START ------------------------------ *)
  let string_of_range (l1,l2) = 
    let pc1,pc2 = labels_to_range (l1,l2) in 
    string_of_code_label l1^"("^string_of_int pc1^")-"^ string_of_code_label l2^"("^string_of_int pc2^")" in
  let string_of_clause cl = String.concat "+" (List.map string_of_range (lranges_of_clause cl)) in
  let string_of_tryspec tryspec = "tryspec"^ string_of_range tryspec.exnRange ^ "--" ^ String.concat " / " (List.map string_of_clause tryspec.exnClauses) in
  let string_of_locspec locspec = "local "^(String.concat ";" (locspec.locInfos |> List.map (fun l -> l.localName)))^": "^ string_of_range locspec.locRange in
  let string_of_structspec = function SEH tryspec -> string_of_tryspec tryspec | LOCAL locspec -> string_of_locspec locspec in
  (* DIAGNOSTICS: END -------------------------------- *)

  let build_code_from_instruction_array instrs =

    (* Consume instructions until we hit the end of the basic block, either *)
    (* by hitting a control-flow instruction or by hitting the start of the *)
    (* next basic block by fall-through. *)
    let rec consume_bblock_instrs instrs rinstrs (pc:int) next_bbstart_pc =
      (* rinstrs = accumulates instructions in reverse order *)
      if int_eq pc (Array.length instrs) then begin 
        dprint_endline "*** WARNING: basic block at end of method ends without a leave, branch, return or throw. Adding throw\n";
        pc,List.rev (I_throw :: rinstrs)
      end 
      (* The next test is for drop-through at end of bblock, when we just insert *)
      (* a branch to the next bblock. *)
      else if (match next_bbstart_pc with Some pc' -> int_eq pc pc' | _ -> false) then begin
        if logging then dprintf1 "-- pushing br, pc = next_bbstart_pc = %d\n" pc;
        pc,List.rev (I_br (pc2cl pc) :: rinstrs)
      end else
        (* Otherwise bblocks end with control-flow. *)
        let i = instrs.(pc) in  
        let pc' = pc + 1 in 
          if instr_is_bblock_end i then 
            if instr_is_tailcall i then 
              if int_eq pc' (Array.length instrs) || (match instrs.(pc') with I_ret -> false | _ -> true) then 
                failwith "a tailcall must be followed by a return"
              else if (match next_bbstart_pc with Some pc'' -> int_eq pc' pc'' | _ -> false) then
                (* In this obscure case, someone branches to the return instruction *)
                (* following the tailcall, so we'd better build a basic block *)
                (* containing just that return instruction. *)
                pc', List.rev (i :: rinstrs)
              else 
                (* Otherwise skip the return instruction, but keep the tailcall. *)
                pc'+1, List.rev (i :: rinstrs)
            else 
              pc', List.rev (i :: rinstrs)
          else
            (* recursive case *)
            consume_bblock_instrs instrs (i::rinstrs) pc' next_bbstart_pc
    in
    
    (* type block = (int * int) * Code // a local type (alias) would be good, good for intelisense too *)
    let rec consume_one_bblock bbstart_pc next_bbstart_pc current_pc =
      if int_eq current_pc (Array.length instrs) then None
      else if bbstart_pc < current_pc then failwith "internal error: bad basic block structure (missing bblock start marker?)"
      else if bbstart_pc > current_pc then
        (* dprint_endline ("*** ignoring unreachable instruction in method: "^ meth_name); *)
        consume_one_bblock 
          bbstart_pc 
          next_bbstart_pc 
          (current_pc + 1)
      else
        let pc', bblock_instrs = consume_bblock_instrs instrs [] bbstart_pc next_bbstart_pc in 
        if logging then dprintf3 "-- making bblock, entry label is %s, length = %d, bbstart_pc = %d\n" (string_of_code_label (pc2cl bbstart_pc)) (List.length bblock_instrs) bbstart_pc;
        let bblock = mk_bblock {bblockLabel=pc2cl bbstart_pc; bblockInstrs=Array.of_list bblock_instrs} in 
        
        let bblock_range = (bbstart_pc, pc') in 
        (* Return the bblock and the range of instructions that the bblock covered. *)
        (* Also return any remaining instructions and the pc' for the first *)
        (* such instruction. *)
        Some ((bblock_range, bblock), pc') in 
    
    let rec fetch_bblocks bbstarts current_pc = 
      match bbstarts with 
        [] -> 
          (* if current_pc <> Array.length instrs then 
             dprint_endline ("*** ignoring instructions at end of method: "^ meth_name); *)
          []
      | h::t -> 
          let h2 = match t with [] -> None | h2:: _ -> assert (not (int_eq h h2)); Some h2 in
          match consume_one_bblock h h2 current_pc with
          | None -> []
          | Some (bblock, current_pc') -> bblock :: fetch_bblocks t current_pc' in

    let inside range (brange,_) =
      if range_inside_range brange range then true else
      if disjoint_range brange range then false else
      failwith "exception block specification overlaps the range of a basic block" in

    (* A "blocks" contain blocks, ordered on startPC.
     * Recall, a block is (range,code) where range=(pcStart,pcLast+1). *)
    let addBlock m (((startPC,endPC),code) as block) =
      match Zmap.tryfind startPC m with
        | None        -> Zmap.add startPC [block] m
        | Some blocks -> Zmap.add startPC (block :: blocks) m   (* NOTE: may reverse block *)
    in
    let addBlocks m blocks = List.fold_left addBlock m blocks in
          
    let mkBlocks blocks =
      let emptyBlocks = (Zmap.empty int_compare : (int,((int*int) * code) list) Zmap.map) in
      List.fold_left addBlock emptyBlocks blocks
    in

    let sanity_check_cover = false in  (* linear check    - REVIEW: set false and elim checks *)
    let sanity_check_order = false in  (* quadratic check - REVIEW: set false and elim checks *)

    let compute_covered_blocks ((start_pc,end_pc) as range) (blocks: (int,((int*int) * code) list) Zmap.map ) =
      (* It is assumed that scopes never overlap.
       * locinfo scopes could overlap if there is a bug elsewhere.
       * If overlaps are discovered, an exception is raised. see NOTE#overlap.
       *)
      let pcCovered,blocks = popRangeM start_pc (end_pc - 1) blocks in
      let coveredBlocks = pcCovered |> List.concat in
      (* Look for bad input, e.g. overlapping locinfo scopes. *)
      let overlapBlocks = List.filter (inside range >> not) coveredBlocks in
      if not (isNil overlapBlocks) then raise Not_found; (* see NOTE#overlap *)
      if sanity_check_cover then (
        let assertIn  block = assert (inside range block) in
        let assertOut block = assert (not (inside range block)) in
        List.iter assertIn coveredBlocks;
        Zmap.iter (fun _ bs -> List.iter assertOut bs) blocks
      );
      coveredBlocks,blocks
    in

    let rec coverage_of_codes blocks = 
      match blocks with 
        [] -> failwith "start_of_codes"
      | [(r,_)] -> r 
      | ((r,_)::t) -> merge_ranges r (coverage_of_codes t) in 
    
    (* Stage 2b - Given an innermost tryspec, collect together the *)
    (* blocks covered by it. Preserve the essential ordering of blocks. *)
    let block_for_inner_tryspec tryspec ((blocks, remaining_bblock_starts) as state0) = 
      let try_blocks, other_blocks = compute_covered_blocks (labels_to_range tryspec.exnRange) blocks in 
      if isnull try_blocks then (dprint_endline "try block specification covers no real code"; state0) else
      let get_clause r other_blocks = 
        let clause_blocks, other_blocks = 
          compute_covered_blocks (*_rough*) (labels_to_range r) other_blocks in 
        if isnull clause_blocks then 
          failwith "clause block specification covers no real code";
        (* The next line computes the code label for the entry to the clause *)
        let clause_entry_lab = lab2cl (fst r) in 
        (* Now compute the overall clause, with labels still visible. *)
        let clause_block = mk_group_block ([],List.map snd clause_blocks) in 
        (* if logging then dprintf "-- clause entry label is %s" clause_entry_lab; *)
        (clause_entry_lab, clause_blocks, clause_block), other_blocks in 
      let try_code_blocks = List.map snd try_blocks in 
      let try_entry_lab = lab2cl (fst tryspec.exnRange) in 
      let try_hidden = 
        CodeLabels.remove try_entry_lab (List.fold_right (entries_of_code' >> CodeLabels.union) try_code_blocks CodeLabels.empty)  in 
      let try_block =  mk_group_block (CodeLabels.to_list try_hidden,try_code_blocks) in 
      
      match tryspec.exnClauses with 
      |  SEH_finally _ :: _ :: _ -> failwith "finally clause combined with others"
      | [ SEH_finally r ] | [ SEH_fault r ] -> 

          let maker =       
            match tryspec.exnClauses with
              [ SEH_finally _ ] -> mk_try_finally_block 
            | [ SEH_fault _ ] -> mk_try_fault_block 
            | _ -> failwith "" in 

          let (clause_entry_lab, clause_blocks, clause_block), other_blocks = get_clause r other_blocks in 
          let newblock_range = coverage_of_codes (try_blocks@clause_blocks) in 
          (* The next construction joins the blocks together. *)
          (* It automatically hides any internal labels used in the *)
          (* clause blocks. Only the entry to the clause is kept visible. *)
          (* We hide the entries to the try block up above. *)
          let newblock =  maker (try_block,clause_entry_lab,clause_block) in 
          (* None of the entries to the clause block are visible outside the *)
          (* entire try-clause construct, nor the other entries to the try block *)
          (* apart from the one at the. top *)
          let newstarts = CodeLabels.diff remaining_bblock_starts (CodeLabels.union try_hidden (entries_of_code' clause_block)) in 
          (* Now return the new block, the remaining blocks and the new set *)
          (* of entries. *)
          addBlocks other_blocks [(newblock_range, newblock)], newstarts
      | clauses when 
        List.for_all
          (function 
            | SEH_filter_catch _ -> true
            | SEH_type_catch _ -> true | _ -> false) 
          clauses   -> 
            
            let clause_infos, other_blocks (*(prior,posterior)*) = 
              List.fold_left 
                (fun (sofar,other_blocks) cl -> 
                  match cl with 
                  | SEH_filter_catch(r1,r2) -> 
                      let ((lab1,_,bl1) as info1),other_blocks =  get_clause r1 other_blocks in 
                      let info2,other_blocks =  get_clause r2 other_blocks in 
                      (sofar@[(Choice1of2 (lab1,bl1),info2)]), other_blocks
                  | SEH_type_catch(typ,r2) -> 
                      let info2,other_blocks = get_clause r2 other_blocks in 
                      (sofar@[(Choice2of2 typ,info2)]), other_blocks
                  | _ -> failwith "internal error")
                ([],other_blocks)
                clauses in 
            let newblock_range = 
              (* Ignore filter blocks when computing this range *)
              (* REVIEW: They must always come before the catch blocks. *)
              coverage_of_codes 
                (try_blocks@
                 (map_concat (fun (_,(_,blocks2,_)) -> blocks2) clause_infos)) in 
            
            (* The next construction joins the blocks together. *)
            (* It automatically hides any internal labels used in the *)
            (* clause blocks. Only the entry to the clause is kept visible. *)
            let newblock = 
              mk_try_multi_filter_catch_block 
                (try_block,
                 List.map 
                   (fun (choice,(lab2,_,bl2)) -> choice, (lab2,bl2)) 
                   clause_infos) in 
            (* None of the entries to the filter or catch blocks are *)
            (* visible outside the entire exception construct. *)
            let newstarts =
              CodeLabels.diff remaining_bblock_starts 
                (CodeLabels.union try_hidden
                   (List.fold_right 
                      (fun (flt,(_,_,ctch_blck)) acc -> 
                        CodeLabels.union
                          (match flt with 
                           | Choice1of2 (_,flt_block) -> entries_of_code' flt_block
                           | Choice2of2 _ -> CodeLabels.empty)
                          (CodeLabels.union (entries_of_code' ctch_blck) acc)) 
                      clause_infos
                      CodeLabels.empty)) in 
            (* Now return the new block, the remaining blocks and the new set *)
            (* of entries. *)
            addBlocks other_blocks [ (newblock_range, newblock)], newstarts
      | _ -> failwith "invalid pattern of exception constructs" in 
    
    (* Stage 2b - Given an innermost tryspec, collect together the *)
    (* blocks covered by it. Preserve the essential ordering of blocks. *)
    let block_for_inner_locspec locspec ((blocks, remaining_bblock_starts) as state0) =
      let scope_blocks, other_blocks (*(prior,posterior)*) = compute_covered_blocks (labels_to_range locspec.locRange) blocks in 
      if isnull scope_blocks then (dprint_endline "scope block specification covers no real code"; state0) else
      let newblock =  mk_scope_block (locspec.locInfos,mk_group_block ([],List.map snd scope_blocks)) in 
      let newblock_range = coverage_of_codes scope_blocks in 
      addBlocks other_blocks [ (newblock_range, newblock)], remaining_bblock_starts in
      
    (* Require items by increasing inclusion-order.
     * Order by size/extent.
     * a) size-ordering implies containment-ordering.
     * b) size-ordering is total, so works with List.sort
     *)
    let build_order sA sB = int_compare (extent_structspec sA) (extent_structspec sB) in

    (* checkOrder: checking is O(n^2) *)
    let rec checkOrder = function
      | []      -> ()
      | sA::sBs -> List.iter (fun sB ->
                                if structspec_inside_structspec sB sA && not (structspec_inside_structspec sA sB) then (
                                  dprintf1 "sA = %s\n" (string_of_structspec sA);
                                  dprintf1 "sB = %s\n" (string_of_structspec sB);
                                  assert false
                                )) sBs;
                   checkOrder sBs
    in

    let do_structure structspecs block_state =
      (* List.iter (fun spec -> dprintf1 "before: %s\n" (string_of_structspec spec)) structspecs; // REVIEW: delete *)
      let structspecs = List.stable_sort build_order structspecs in
      (* List.iter (fun spec -> dprintf1 "after: %s\n" (string_of_structspec spec)) structspecs;  // REVIEW: delete *)
      if sanity_check_order then checkOrder structspecs; (* note: this check is n^2 *)
      let buildBlock block_state = function
        | SEH   tryspec -> (if logging then dprint_endline "-- checkin a tryspec";
                            block_for_inner_tryspec tryspec block_state)
        | LOCAL locspec -> (if logging then dprint_endline "-- checkin a locspec";
                            block_for_inner_locspec locspec block_state)
      in
      List.fold_left buildBlock block_state structspecs
    in

    (* Apply stage 1. Compute the blocks not taking exceptions into account. *)
    let bblocks = fetch_bblocks (List.sort int_compare bbstart_pcs) 0 in
    let bblocks = mkBlocks bblocks in
    (* Apply stage 2. Compute the overall morphed blocks. *)
    let morphed_blocks,remaining_entries = 
      let specs1 = List.map (fun x -> SEH x) tryspecs in 
      let specs2 = List.map (fun x -> LOCAL x) localspecs in 
      try do_structure (specs1 @ specs2) (bblocks,bbstart_code_labs) 
      with Not_found ->
          (* NOTE#overlap.
           * Here, "Not_found" indicates overlapping scopes were found.
           * Maybe the calling code got the locspecs scopes wrong.
           * Try recovery by discarding locspec info...
           *)
          let string_of_tryspec tryspec = "tryspec" in 
          let string_of_range (l1,l2) = 
            let pc1,pc2 = labels_to_range (l1,l2) in 
            string_of_code_label l1^"("^string_of_int pc1^")-"^ string_of_code_label l2^"("^string_of_int pc2^")" in
          let string_of_locspec locspec = "local "^(String.concat ";" (locspec.locInfos |> List.map (fun l -> l.localName)))^": "^ string_of_range locspec.locRange in  
          
          dprintf1 "\nERROR: could not find an innermost exception block or local scope, specs = \n%s\nTrying again without locals."
            (String.concat "\n" (List.map string_of_tryspec tryspecs @ List.map string_of_locspec localspecs));
         do_structure specs1 (bblocks,bbstart_code_labs) 
    in 
    let morphed_blocks = Zmap.values morphed_blocks |> List.concat in (* NOTE: may mixup order *)
    (* Now join up all the remaining blocks into one block with one entry. *)
    if logging then dprint_endline "-- computing entry label";
    if logging then dprint_endline ("-- entry label is "^string_of_code_label cl0);
    mk_group_block 
      (CodeLabels.to_list (CodeLabels.remove cl0 remaining_entries),List.map snd morphed_blocks) in 

  try build_code_from_instruction_array basic_instructions
  with e -> 
    dprint_endline ("*** error while converting instructions to code for method: " ^meth_name);
    (*F# rethrow(); F#*) raise e


(* -------------------------------------------------------------------- 
 * Detecting Delegates
 * -------------------------------------------------------------------- *)

let mk_delegate_mdefs ilg (parms,rtv) = 
  let rty = typ_of_return rtv in 
  let one nm args ret =
    let mdef = mk_virtual_mdef (nm,MemAccess_public,None,args,(mk_return ret),MethodBody_abstract) in 
    let mdef = 
      {mdef with mdKind=match mdef.mdKind with 
      | MethodKind_virtual vinfo -> MethodKind_virtual {vinfo with virtAbstract=false; } 
      | k -> k } in
    {mdef with 
      mdCodeKind=MethodCodeKind_runtime;
      mdHideBySig=true; } in 
  let ctor = mk_ctor(MemAccess_public, [ mk_named_param("object",ilg.typ_Object); mk_named_param("method",ilg.typ_IntPtr) ], MethodBody_abstract) in 
  let ctor = { ctor with  mdCodeKind=MethodCodeKind_runtime; mdHideBySig=true } in 
  [ ctor;
    one "Invoke" parms rty;
    one "BeginInvoke" (parms @ [mk_named_param("callback",ilg.typ_AsyncCallback);
                                 mk_named_param("objects",ilg.typ_Object) ] ) ilg.typ_IAsyncResult;
    one "EndInvoke" [mk_named_param("result",ilg.typ_IAsyncResult)] rty; ]
    

let is_delegate_ctor ilg md =
  string_eq md.mdName ".ctor" &
   isnull md.mdGenericParams &
  (match md.mdKind with MethodKind_ctor -> true | _ -> false) &
  begin 
    match md.mdCallconv, md.mdParams, md.mdReturn with
      cc,[arg1;arg2],rty when
      (callconv_eq cc instance_callconv &
       typ_is_Object ilg arg1.paramType && 
       (typ_is_IntPtr ilg arg2.paramType or typ_is_UIntPtr ilg arg2.paramType) &
       (match rty.returnType with Type_void -> true | _ -> false)) -> true
    | _ -> false
  end

let is_delegate_invoke (ilg:mscorlib_refs) md =
  string_eq md.mdName "Invoke" &
   isnull md.mdGenericParams &
  match md.mdKind,md.mdCallconv, md.mdParams, md.mdReturn with 
    MethodKind_virtual vinfo, cc,argtys,rty when  
      ((* vinfo.virtNewslot && -- not compulsory *)
       not vinfo.virtAbstract &
       callconv_eq cc instance_callconv) -> true
  | _ -> false

let dest_delegate_invoke (ilg:mscorlib_refs) md = 
  match md.mdCallconv, md.mdParams, md.mdReturn with
    (_,argtys,rty) -> argtys,rty

let is_delegate_begin_invoke ilg md =
  string_eq md.mdName "BeginInvoke" &
  match md.mdKind,md.mdCallconv, md.mdParams, md.mdReturn with 
    MethodKind_virtual vinfo, cc,argtys,rty when  
      vinfo.virtNewslot &
      not vinfo.virtAbstract &
      callconv_eq cc instance_callconv && 
      typ_is_IAsyncResult ilg rty.returnType &
      (match List.rev argtys with 
  (delegee :: callback :: real_args) ->
    typ_is_Object ilg delegee.paramType &
    typ_is_AsyncCallback ilg callback.paramType
      | _ -> false) -> true
  | _ -> false

let is_delegate_end_invoke ilg md =
  string_eq md.mdName "EndInvoke" &
  callconv_eq md.mdCallconv instance_callconv && 
  match md.mdKind,md.mdParams, md.mdReturn with 
    MethodKind_virtual vinfo, argtys,rty when
    vinfo.virtNewslot &
    not vinfo.virtAbstract &
    (match List.rev argtys with 
      res::rest -> typ_is_IAsyncResult ilg res.paramType 
    | _ -> false) -> true
  | _ -> false

let dest_delegate_begin_end_invoke begin_md end_md =
  let argtys = 
    match List.rev begin_md.mdParams with 
      delegee :: callback :: real_args -> List.rev real_args
    | _ -> failwith "dest_delegate_begin_end_invoke: insufficient args" in 
  let rty = end_md.mdReturn in 
  argtys,rty


type enum_info (*F# = EnumInfo and EnumInfo F#*) =  
    { enumValues: (string * field_init) list;  
      enumType: typ }

let values_of_enum_info info = info.enumValues
let typ_of_enum_info info = info.enumType

let mk_ctor_mspec_for_delegate ilg (tref,cinst,useUIntPtr) =
  let scoref = scoref_of_tref tref in 
  mk_nongeneric_instance_mspec_in_tref (tref,AsObject,".ctor",[rescope_typ scoref ilg.typ_Object; rescope_typ scoref (if useUIntPtr then ilg.typ_uint else ilg.typ_int)],Type_void,cinst) 

let info_for_enum (tdName,tdFieldDefs) = 
  match (List.partition (fun fd -> fd.fdStatic) (dest_fdefs tdFieldDefs)) with 
  | sfds,[vfd] -> 
      { enumType = vfd.fdType; 
        enumValues = List.map (fun fd -> (fd.fdName, match fd.fdInit with Some i -> i | None -> failwith ("info_of_enum_tdef: badly formed enum "^tdName^": static field does not have an default value"))) sfds }
  | _,[] -> failwith ("info_of_enum_tdef: badly formed enum "^tdName^": no non-static field found")
  | _,_ -> failwith ("info_of_enum_tdef: badly formed enum "^tdName^": more than one non-static field found")

 
(* --------------------------------------------------------------------
 * Intern tables to save space.
 * -------------------------------------------------------------------- *)

let new_intern_table ()  =
  let idx = ref 0 in 
  let t = Hashtbl.create 100 in 
  fun s -> 
    if Hashtbl.mem t s then Hashtbl.find t s   
    else let i = !idx in incr idx; (Hashtbl.add t s (s,i); (s,i))

let new_tagger idf tagf  =
  let idx = ref 0 in 
  fun s -> 
    match idf s with 
    | Some i -> s
    | None -> let i = !idx in incr idx; tagf s i

let new_idx_intern_table idf tagf  =
  let idx = ref 0 in 
  let t = Hashtbl.create 100 in 
  fun s -> 
    match idf s with 
    | Some i -> s,i
    | None -> 
      if Hashtbl.mem t s then 
        let r = Hashtbl.find t s  in 
        let i = (match idf r with Some x -> x | None -> failwith "new_idx_intern_table: internal error") in 
        r,i
      else begin
        incr idx; 
        let i = !idx in 
        let r = (tagf s i) in 
        Hashtbl.add t s r; 
        r,i
      end

type 'a idx_interner = 'a -> 'a * idx
type 'a idx_tagger = 'a -> 'a 


type manager = (*F# TermManager and TermManager = F#*)
  { bytes_interner:  bytes idx_interner;
    string_interner: string idx_interner;
    tref_interner:   type_ref idx_interner;
    tspec_interner:  type_spec idx_interner;
    mspec_interner:  method_spec idx_interner; 
    typ_interner:    typ      idx_interner;
    instr_interner:  instr    idx_interner; 
    bytes_tagger:  bytes idx_tagger;
    string_tagger: string idx_tagger;
    tref_tagger:   type_ref idx_tagger;
    tspec_tagger:  type_spec idx_tagger;
    mspec_tagger:  method_spec idx_tagger; 
    typ_tagger:    typ      idx_tagger;
    instr_tagger:  instr    idx_tagger; } 
  
let new_manager () = 
  { bytes_interner= new_intern_table();
    string_interner= new_intern_table();
    tref_interner= new_intern_table();
    tspec_interner= new_intern_table();
    mspec_interner= new_idx_intern_table(fun m -> m.mspecOptionalID) (fun m i -> {m with mspecOptionalID=Some i}); 
    typ_interner= new_intern_table();
    instr_interner= new_intern_table();
    bytes_tagger= (fun x -> x);
    string_tagger= (fun x -> x);
    tref_tagger= (fun x -> x);
    tspec_tagger= (fun x -> x);
    mspec_tagger= new_tagger (fun m -> m.mspecOptionalID) (fun m i -> {m with mspecOptionalID=Some i}); 
    typ_tagger= (fun x -> x);
    instr_tagger= (fun x -> x); }
     
let intern_bytes m x =  match m with None -> x | Some m -> fst (m.bytes_interner x)
let intern_string  m x = match m with None -> (* System.String.Intern *) x | Some m ->  fst (m.string_interner x)
let intern_instr  m x = match m with None -> x | Some m -> fst (m.instr_interner x)
let intern_typ  m x = match m with None -> x | Some m -> fst (m.typ_interner x)
let intern_tref  m x = match m with None -> x | Some m -> fst (m.tref_interner x)
let intern_tspec  m x = match m with None -> x | Some m -> fst (m.tspec_interner x)
let intern_mspec  m x = match m with None -> x | Some m -> fst (m.mspec_interner x)

let tag_bytes m x =  match m with None -> x | Some m ->  (m.bytes_tagger x)
let tag_string  m x = match m with None -> x | Some m ->  (m.string_tagger x)
let tag_instr  m x = match m with None -> x | Some m ->  (m.instr_tagger x)
let tag_typ  m x = match m with None -> x | Some m ->  (m.typ_tagger x)
let tag_tref  m x = match m with None -> x | Some m ->  (m.tref_tagger x)
let tag_tspec  m x = match m with None -> x | Some m -> (m.tspec_tagger x)
let tag_mspec  m x = match m with None -> x | Some m -> (m.mspec_tagger x)

let memoize_on keyf f = 
  let t = Hashtbl.create 1000 in
  fun x -> 
    let idx = keyf x in
    if Hashtbl.mem t idx then Hashtbl.find t idx 
    else let r = f x in Hashtbl.add t idx r;  r

let memoize f = 
  let t = Hashtbl.create 1000 in
  fun x -> 
    if Hashtbl.mem t x then Hashtbl.find t x 
    else let r = f x in Hashtbl.add t x r;  r

let mspec_idx m x = snd (m.mspec_interner x)
let mspec_optional_idx x = x.mspecOptionalID


(*---------------------------------------------------------------------
 * SHA1 hash-signing algorithm.  Used to get the public key token from
 * the public key.
 *---------------------------------------------------------------------*)



let f(t,b,c,d) = 
  if t < 20 then (b &&& c) ||| ((Int32.lognot b) &&& d) else
  if t < 40 then b ^^^ c ^^^ d else
  if t < 60 then (b &&& c) ||| (b &&& d) ||| (c &&& d) else
  b ^^^ c ^^^ d

let k0to19 = Int32.of_string "0x5A827999"      
let k20to39 = Int32.of_string "0x6ED9EBA1"     
let k40to59 = Int32.of_string "0x8F1BBCDC"     
let k60to79 = Int32.of_string "0xCA62C1D6"     

let k(t) = 
  if t < 20 then k0to19 else
  if t < 40 then k20to39 else
  if t < 60 then k40to59 else
  k60to79 

type chan = SHABytes of bytes | SHAChannel of in_channel
type sha_instream = 
    { stream: chan;
      mutable pos: int;
      mutable eof:  bool; }

(* use 64 bits to cope with large file sizes *)
let ( *** ) x y = Int64.mul x y
let ( >>>> ) x y = Int64.shift_right_logical x y 
let ( &&&& ) x y = Int64.logand x y 
let ( !!!! ) x = Int64.of_int x 
let ( ???? ) x = Int64.to_int x 

let rot_left32 x n =  (x <<< n) ||| (x lsr (32-n))

let sha_eof sha = sha.eof

(* padding and length (in bits!) recorded at end *)
let sha_after_eof sha  = 
  let n = sha.pos in 
  let len = 
    (match sha.stream with
    | SHAChannel is -> in_channel_length is
    | SHABytes s -> Bytes.length s) in 
  if int_eq n len then 0x80l
  else 
    let padded_len = (((len + 9 + 63) / 64) * 64) - 8 in  
    if n < padded_len - 8  then !!!0x0  
    else if int32_eq (!!!n &&& !!!63) !!!56 then Int64.to_int32 ((!!!!len *** !!!!8) >>>> 56) &&& !!!0xff
    else if int32_eq (!!!n &&& !!!63) !!!57 then Int64.to_int32 ((!!!!len *** !!!!8) >>>> 48) &&& !!!0xff
    else if int32_eq (!!!n &&& !!!63) !!!58 then Int64.to_int32 ((!!!!len *** !!!!8) >>>> 40) &&& !!!0xff
    else if int32_eq (!!!n &&& !!!63) !!!59 then Int64.to_int32 ((!!!!len *** !!!!8) >>>> 32) &&& !!!0xff
    else if int32_eq (!!!n &&& !!!63) !!!60 then Int64.to_int32 ((!!!!len *** !!!!8) >>>> 24) &&& !!!0xff
    else if int32_eq (!!!n &&& !!!63) !!!61 then Int64.to_int32 ((!!!!len *** !!!!8) >>>> 16) &&& !!!0xff
    else if int32_eq (!!!n &&& !!!63) !!!62 then Int64.to_int32 ((!!!!len *** !!!!8) >>>> 8) &&& !!!0xff
    else if int32_eq (!!!n &&& !!!63) !!!63 then (sha.eof <- true; Int64.to_int32 (!!!!len *** !!!!8) &&& !!!0xff)
    else !!!0x0

let sha_read8 sha = 
  let b = 
      match sha.stream with 
      | SHAChannel is -> (try !!!(input_byte is) with _ -> sha_after_eof sha)
      | SHABytes s -> if sha.pos >= Bytes.length s then sha_after_eof sha else  !!!(Bytes.get s sha.pos)  in
  sha.pos <- sha.pos + 1; 
  b
    
let sha_read32 sha  = 
  let b0 = sha_read8 sha in 
  let b1 = sha_read8 sha in 
  let b2 = sha_read8 sha in 
  let b3 = sha_read8 sha in 
  let res = (b0 <<< 24) ||| (b1 <<< 16) ||| (b2 <<< 8) ||| b3 in 
  res


let sha1_hash sha = 
  let h0 = ref (Int32.of_string "0x67452301") in
  let h1 = ref (Int32.of_string "0xEFCDAB89") in
  let h2 = ref (Int32.of_string "0x98BADCFE") in
  let h3 = ref (Int32.of_string "0x10325476") in
  let h4 = ref (Int32.of_string "0xC3D2E1F0") in
  let a = ref 0l in
  let b = ref 0l in
  let c = ref 0l in
  let d = ref 0l in
  let e = ref 0l in 
  let w = Array.create 80 !!!0x00 in 
  while (not (sha_eof sha)) do
    for i = 0 to 15 do
      w.(i) <- sha_read32 sha
    done;
    for t = 16 to 79 do
      w.(t) <- rot_left32 (w.(t-3) ^^^ w.(t-8) ^^^ w.(t-14) ^^^ w.(t-16)) 1;
    done;
    a := !h0; 
    b := !h1; 
    c := !h2; 
    d := !h3; 
    e := !h4;
    for t = 0 to 79 do
      let temp =  (rot_left32 !a 5) +++ f(t,!b,!c,!d) +++ !e +++ w.(t) +++ k(t) in 
      e := !d; 
      d := !c; 
      c :=  rot_left32 !b 30; 
      b := !a; 
      a := temp;
    done;
    h0 := !h0 +++ !a; 
    h1 := !h1 +++ !b; 
    h2 := !h2 +++ !c;  
    h3 := !h3 +++ !d; 
    h4 := !h4 +++ !e
  done;
  (!h0,!h1,!h2,!h3,!h4)

let sha1_hash_file file = 
  let is = open_in_bin file in 
  let res = sha1_hash { stream = SHAChannel is; pos = 0; eof = false } in 
  close_in is;
  res

let b0 n = Int32.to_int (n &&& 0xFFl)
let b1 n = Int32.to_int ((n lsr 8) &&& 0xFFl)
let b2 n = Int32.to_int ((n lsr 16) &&& 0xFFl)
let b3 n = Int32.to_int ((n lsr 24) &&& 0xFFl)

let sha1_hash_bytes s = 
  let (h0,h1,h2,h3,h4) = sha1_hash { stream = SHABytes s; pos = 0; eof = false } in
  Bytes.of_intarray [|  b0 h4; b1 h4; b2 h4; b3 h4; b0 h3; b1 h3; b2 h3; b3 h3; |]

(*---------------------------------------------------------------------
 * Get the public key token from the public key.
 *---------------------------------------------------------------------*)

(* Little-endian encoding of int32 *)
let token_from_public_key bytes = sha1_hash_bytes bytes 

let assref_for_manifest m = 
    { assemRefName=m.manifestName;
      assemRefHash= None; (* REVIEW: find hash?? *)
      assemRefPublicKeyInfo=(match m.manifestPublicKey with Some k -> Some (PublicKeyToken(token_from_public_key k)) | None -> None);
      assemRefRetargetable=false;
      assemRefVersion=m.manifestVersion;
      assemRefLocale=m.manifestLocale }

let assref_for_mainmod mainmod = 
  assref_for_manifest (manifest_of_mainmod mainmod)




let z_unsigned_int_size n = 
  if n <= 0x7F then 1
  else if n <= 0x3FFF then 2
  else 3

let z_unsigned_int n = 
  if n >= 0 &&  n <= 0x7F then [| n |] 
  else 
    Array.map Int32.to_int
      (if n >= 0x80 && n <= 0x3FFF then [| 0x80l ||| (!!!n lsr 8); !!!n &&& 0xFFl |] 
      else [| !!!0xc0 ||| (!!!n lsr 24); (!!!n lsr 16) &&& 0xFFl; (!!!n lsr 8) &&& 0xFFl; !!!n &&& 0xFFl |])

let string_as_utf8_intarray (s:string) = Bytes.to_intarray (Bytes.string_as_utf8_bytes s)

(* Little-endian encoding of int64 *)
let dw7 n = ???? ((n >>>> 56) &&&& (!!!! 255))
let dw6 n = ???? ((n >>>> 48) &&&& (!!!! 255))
let dw5 n = ???? ((n >>>> 40) &&&& (!!!! 255))
let dw4 n = ???? ((n >>>> 32) &&&& (!!!! 255))
let dw3 n = ???? ((n >>>> 24) &&&& (!!!! 255))
let dw2 n = ???? ((n >>>> 16) &&&& (!!!! 255))
let dw1 n = ???? ((n >>>> 8) &&&& (!!!! 255))
let dw0 n = ???? (n &&&& (!!!! 255))

let u8_as_intarray i = [| b0 !!!(u8_to_int i) |]
let u16_as_intarray x =  let n = !!!(u16_to_int x) in [| b0 n; b1 n |]
let i32_as_intarray i = [| b0 i; b1 i; b2 i; b3 i |]
let i64_as_intarray i = [| dw0 i; dw1 i; dw2 i; dw3 i; dw4 i; dw5 i; dw6 i; dw7 i |]

let i8_as_intarray i = u8_as_intarray (i8_to_u8 i)
let i16_as_intarray i = u16_as_intarray (i16_to_u16 i)
let u32_as_intarray i = i32_as_intarray (u32_to_i32 i)
let u64_as_intarray i = i64_as_intarray (u64_to_i64 i)

let ieee32_as_intarray i = i32_as_intarray (ieee32_to_bits i)
let ieee64_as_intarray i = i64_as_intarray (ieee64_to_bits i)

let rec celem_ty ilg x = 
  match x with
        | CustomElem_string _ -> ilg.typ_String
        | CustomElem_bool _ -> ilg.typ_bool
        | CustomElem_char _ -> ilg.typ_char
        | CustomElem_int8 _ -> ilg.typ_int8
        | CustomElem_int16 _ -> ilg.typ_int16
        | CustomElem_int32 _ -> ilg.typ_int32
        | CustomElem_int64 _ -> ilg.typ_int64
        | CustomElem_uint8 _ -> ilg.typ_uint8
        | CustomElem_uint16 _ -> ilg.typ_uint16
        | CustomElem_uint32 _ -> ilg.typ_uint32
        | CustomElem_uint64 _ -> ilg.typ_uint64
        | CustomElem_type _ -> ilg.typ_Type
        | CustomElem_float32 _ -> ilg.typ_float32
        | CustomElem_float64 _ -> ilg.typ_float64
        | CustomElem_array _ -> failwith "unexpected array element"

let et_END = 0x00
let et_VOID = 0x01
let et_BOOLEAN = 0x02
let et_CHAR = 0x03
let et_I1 = 0x04
let et_U1 = 0x05
let et_I2 = 0x06
let et_U2 = 0x07
let et_I4 = 0x08
let et_U4 = 0x09
let et_I8 = 0x0a
let et_U8 = 0x0b
let et_R4 = 0x0c
let et_R8 = 0x0d
let et_STRING = 0x0e
let et_PTR = 0x0f
let et_BYREF = 0x10
let et_VALUETYPE      = 0x11
let et_CLASS          = 0x12
let et_VAR            = 0x13
let et_ARRAY          = 0x14
let et_WITH           = 0x15
let et_TYPEDBYREF     = 0x16
let et_I              = 0x18
let et_U              = 0x19
let et_FNPTR          = 0x1B
let et_OBJECT         = 0x1C
let et_SZARRAY        = 0x1D
let et_MVAR           = 0x1e
let et_CMOD_REQD      = 0x1F
let et_CMOD_OPT       = 0x20

let version_to_string (a,b,c,d) = Printf.sprintf "%d.%d.%d.%d" (u16_to_int a) (u16_to_int b) (u16_to_int c) (u16_to_int d)

let qualified_name_of_aref aref = 
  String.concat ", " 
    ([ aref.assemRefName ] @
     (match aref.assemRefVersion with 
      | None -> []
      | Some v -> [ "Version="^version_to_string v ]) @
        ([ "Culture="^(match aref.assemRefLocale with None -> "neutral" | Some b -> b) ]) @
      (match aref.assemRefPublicKeyInfo with 
      | None -> [ "PublicKeyToken=null" ]
      | Some pki -> 
        let pkt = match pki with PublicKeyToken t -> t | PublicKey k -> token_from_public_key k in 
        [ "PublicKeyToken=" ^ String.concat "" (List.map (fun b -> Printf.sprintf "%02x" b) (Array.to_list (Bytes.to_intarray pkt)))]))

let qualified_name_of_scoref scoref = 
  match scoref with 
  | ScopeRef_local -> ""
  | ScopeRef_module mref -> "module "^mref.modulRefName
  | ScopeRef_assembly aref when string_eq aref.assemRefName "mscorlib" -> ""
  | ScopeRef_assembly aref -> qualified_name_of_aref aref

let qualified_name_of_tref tref = 
  let basic = String.concat "+" (enclosing_tnames_of_tref tref @ [ tname_of_tref tref ]) in 
  let sco = qualified_name_of_scoref (scoref_of_tref tref) in 
  if string_eq sco "" then basic else String.concat ", " [basic;sco]

let celem_serstring s = 
  let arr = string_as_utf8_intarray s in 
  Array.concat [ z_unsigned_int (Array.length arr); arr ]      

let rec celem_enc_ty isNamedArg x = 
  match x with
  | Type_value tspec when string_eq (tname_of_tspec tspec) "System.SByte" ->  [| et_I1 |]
  | Type_value tspec when string_eq (tname_of_tspec tspec) "System.Byte" ->  [| et_U1 |]
  | Type_value tspec when string_eq (tname_of_tspec tspec) "System.Int16" ->  [| et_I2 |]
  | Type_value tspec when string_eq (tname_of_tspec tspec) "System.UInt16" ->  [| et_U2 |]
  | Type_value tspec when string_eq (tname_of_tspec tspec) "System.Int32" ->  [| et_I4 |]
  | Type_value tspec when string_eq (tname_of_tspec tspec) "System.UInt32" ->  [| et_U4 |]
  | Type_value tspec when string_eq (tname_of_tspec tspec) "System.Int64" ->  [| et_I8 |]
  | Type_value tspec when string_eq (tname_of_tspec tspec) "System.UInt64" ->  [| et_U8 |]
  | Type_value tspec when string_eq (tname_of_tspec tspec) "System.Double" ->  [| et_R8 |]
  | Type_value tspec when string_eq (tname_of_tspec tspec) "System.Single" ->  [| et_R4 |]
  | Type_value tspec when string_eq (tname_of_tspec tspec) "System.Char" ->  [| et_CHAR |]
  | Type_value tspec when string_eq (tname_of_tspec tspec) "System.Boolean" ->  [| et_BOOLEAN |]
  | Type_boxed tspec when string_eq (tname_of_tspec tspec) "System.String" ->  [| et_STRING |]
  | Type_boxed tspec when string_eq (tname_of_tspec tspec) "System.Object" ->  [| et_OBJECT |]
  | Type_boxed tspec when string_eq (tname_of_tspec tspec) "System.Type" ->  [| 0x50 |]
  | Type_value tspec ->  
       if isNamedArg then 
           Array.append [| 0x55 |] (celem_serstring (qualified_name_of_tref (tref_of_tspec tspec)))
       else (* assume it is an enumeration *) [| et_I4 |]
  | _ ->  failwith "celem_enc_ty: unrecognized custom element type"

let rec celem_dec_ty ilg x = 
  match x with
  | x when int_eq x   et_I1 -> ilg.typ_SByte
  | x when int_eq x  et_U1 -> ilg.typ_Byte
  | x when int_eq x   et_I2 -> ilg.typ_Int16
  | x when int_eq x   et_U2 -> ilg.typ_UInt16
  | x when int_eq x   et_I4 -> ilg.typ_Int32
  | x when int_eq x   et_U4 -> ilg.typ_UInt32
  | x when int_eq x   et_I8 -> ilg.typ_Int64
  | x when int_eq x   et_U8 -> ilg.typ_UInt64
  | x when int_eq x   et_R8 -> ilg.typ_Double
  | x when int_eq x   et_R4 -> ilg.typ_Single
  | x when int_eq x  et_CHAR -> ilg.typ_Char
  | x when int_eq x   et_BOOLEAN -> ilg.typ_Bool
  | x when int_eq x   et_STRING -> ilg.typ_String
  | x when int_eq x   et_OBJECT -> ilg.typ_Object
  | x when int_eq x 0x50 -> ilg.typ_Type
  | _ ->  failwith "celem_dec_ty ilg: unrecognized custom element type"


let rec celem_val_prim ty c = 
  match c with 
  | CustomElem_bool b -> [| if b then 0x01 else 0x00 |]
  | CustomElem_string None -> [| 0xFF |]
  | CustomElem_string (Some(s)) -> celem_serstring s
  | CustomElem_char x -> u16_as_intarray (Nums.unichar_to_u16 x)
  | CustomElem_int8 x -> i8_as_intarray x
  | CustomElem_int16 x -> i16_as_intarray x
  | CustomElem_int32 x -> i32_as_intarray x
  | CustomElem_int64 x -> i64_as_intarray x
  | CustomElem_uint8 x -> u8_as_intarray x
  | CustomElem_uint16 x -> u16_as_intarray x
  | CustomElem_uint32 x -> u32_as_intarray x
  | CustomElem_uint64 x -> u64_as_intarray x
  | CustomElem_float32 x -> ieee32_as_intarray x
  | CustomElem_float64 x -> ieee64_as_intarray x
  | CustomElem_type tref -> celem_serstring (qualified_name_of_tref tref) 
  | CustomElem_array _ -> failwith "unreachable"

and celem_val ilg ty c = 
  match ty,c with 
  | Type_boxed tspec,_  when string_eq (tname_of_tspec tspec) "System.Object" ->  
     Array.concat [ celem_enc_ty false (celem_ty ilg c); celem_val_prim ty c ]
  | Type_array(shape,elemType), CustomElem_array(elems) when deep_eq shape sdshape  ->  
     Array.concat [ i32_as_intarray (!!! (List.length elems)); Array.concat (List.map (celem_val ilg elemType) elems) ]
  | _ -> 
     celem_val_prim ty c

let mk_custom_attribute_mref ilg (mspec,fixedArgs,named_args) = 
  let argtys = args_of_mref (formal_mref_of_mspec mspec) in 
  let args = 
    Array.concat
      begin 
        [ [| 0x01; 0x00; |] ] @
        List.map2 (celem_val ilg) argtys fixedArgs @
        [ u16_as_intarray (int_to_u16 (List.length named_args)) ] @
        List.map (fun (nm,ty,prop,elem) -> 
          Array.concat
            [ [| (if prop then 0x54 else 0x53) |];
              celem_enc_ty true ty;
              celem_serstring nm;
              celem_val ilg ty elem ])
          named_args
      end in
  { customMethod = mspec;
    customData = Bytes.of_intarray args }

let mk_custom_attribute ilg (tref,argtys,argvs,propvs) = 
  mk_custom_attribute_mref ilg (mk_ctor_mspec_for_nongeneric_boxed_tref (tref,argtys),argvs,propvs)

let tref_CompilerGeneratedAttribute ilg = mk_tref (ilg.mscorlib_scoref,tname_CompilerGeneratedAttribute)
let mk_CompilerGeneratedAttribute ilg = mk_custom_attribute ilg (tref_CompilerGeneratedAttribute ilg,[],[],[])
let tref_DebuggableAttribute ilg = mk_tref (ilg.mscorlib_scoref,tname_DebuggableAttribute)

let mk_DebuggableAttribute ilg (jitTracking, jitOptimizerDisabled) = 
  mk_custom_attribute ilg (tref_DebuggableAttribute ilg,[ilg.typ_Bool;ilg.typ_Bool], [CustomElem_bool jitTracking; CustomElem_bool jitOptimizerDisabled],[])
let tref_DebuggableAttribute_DebuggingModes ilg = mk_nested_tref (ilg.mscorlib_scoref,[tname_DebuggableAttribute],tname_DebuggableAttribute_DebuggingModes)

(*---------------------------------------------------------------------
 * Primitives to help read signatures.  These do not use the file cursor, but
 * pass around an int index
 *---------------------------------------------------------------------*)


let sigptr_get_byte bytes sigptr = 
  Bytes.get bytes sigptr, sigptr + 1

let sigptr_get_bool bytes sigptr = 
  let b0,sigptr = sigptr_get_byte bytes sigptr in 
  (int_eq b0 0x01) ,sigptr

let sigptr_get_u8 bytes sigptr = 
  let b0,sigptr = sigptr_get_byte bytes sigptr in 
  int_to_u8 b0,sigptr

let sigptr_get_i8 bytes sigptr = 
  let i,sigptr = sigptr_get_u8 bytes sigptr in 
  u8_to_i8 i,sigptr

let sigptr_get_u16 bytes sigptr = 
  let b0,sigptr = sigptr_get_byte bytes sigptr in 
  let b1,sigptr = sigptr_get_byte bytes sigptr in 
  i32_to_u16 (!!!b0 ||| (!!!b1 <<< 8)),sigptr

let sigptr_get_i16 bytes sigptr = 
  let u,sigptr = sigptr_get_u16 bytes sigptr in 
  u16_to_i16 u,sigptr

let sigptr_get_i32 bytes sigptr = 
  let b0,sigptr = sigptr_get_byte bytes sigptr in 
  let b1,sigptr = sigptr_get_byte bytes sigptr in 
  let b2,sigptr = sigptr_get_byte bytes sigptr in 
  let b3,sigptr = sigptr_get_byte bytes sigptr in 
  !!!b0 ||| (!!!b1 <<< 8) ||| (!!!b2 <<< 16) ||| (!!!b3 <<< 24),sigptr

let sigptr_get_u32 bytes sigptr = 
  let u,sigptr = sigptr_get_i32 bytes sigptr in 
  i32_to_u32 u,sigptr

let sigptr_get_i64 bytes sigptr = 
  let b0,sigptr = sigptr_get_byte bytes sigptr in 
  let b1,sigptr = sigptr_get_byte bytes sigptr in 
  let b2,sigptr = sigptr_get_byte bytes sigptr in 
  let b3,sigptr = sigptr_get_byte bytes sigptr in 
  let b4,sigptr = sigptr_get_byte bytes sigptr in 
  let b5,sigptr = sigptr_get_byte bytes sigptr in 
  let b6,sigptr = sigptr_get_byte bytes sigptr in 
  let b7,sigptr = sigptr_get_byte bytes sigptr in 
  !!!!b0 |||| (!!!!b1 <<<< 8) |||| (!!!!b2 <<<< 16) |||| (!!!!b3 <<<< 24) ||||
  (!!!!b4 <<<< 32) |||| (!!!!b5 <<<< 40) |||| (!!!!b6 <<<< 48) |||| (!!!!b7 <<<< 56),
  sigptr

let sigptr_get_u64 bytes sigptr = 
  let u,sigptr = sigptr_get_i64 bytes sigptr in 
  i64_to_u64 u,sigptr

let sigptr_get_ieee32 bytes sigptr = 
  let u,sigptr = sigptr_get_i32 bytes sigptr in 
  bits_to_ieee32 u,sigptr

let sigptr_get_ieee64 bytes sigptr = 
  let u,sigptr = sigptr_get_i64 bytes sigptr in 
  bits_to_ieee64 u,sigptr

let sigptr_get_intarray n bytes sigptr = 
  let res = Bytes.zero_create n in 
  for i = 0 to (n - 1) do 
    Bytes.set res i (Bytes.get bytes (sigptr + i))
  done;
  res, sigptr + n

let sigptr_get_string n bytes sigptr = 
  let intarray,sigptr = sigptr_get_intarray n bytes sigptr in 
  Bytes.utf8_bytes_as_string intarray , sigptr
   
let sigptr_get_z_i32 bytes sigptr = 
  let b0,sigptr = sigptr_get_byte bytes sigptr in 
  if b0 <= 0x7F then !!!b0, sigptr
  else if b0 <= 0xbf then 
    let b0 = !!!b0 &&& !!!0x7f in 
    let b1,sigptr = sigptr_get_byte bytes sigptr in 
       (b0 <<< 8) ||| !!!b1, sigptr
  else 
    let b0 = !!!b0 &&& !!!0x3f in 
    let b1,sigptr = sigptr_get_byte bytes sigptr in 
    let b2,sigptr = sigptr_get_byte bytes sigptr in 
    let b3,sigptr = sigptr_get_byte bytes sigptr in 
    (b0 <<< 24) ||| (!!!b1 <<< 16) ||| (!!!b2 <<< 8) ||| !!!b3, sigptr

let sigptr_get_serstring  bytes sigptr = 
  let len,sigptr = sigptr_get_z_i32 bytes sigptr  in 
  sigptr_get_string (Int32.to_int len) bytes sigptr 
  
let sigptr_get_serstring_possibly_null  bytes sigptr = 
  let b0,_ = sigptr_get_byte bytes sigptr in  (* throw away sigptr *)
  if int_eq b0 0xFF then
      None,sigptr
  else  
      let len,sigptr = sigptr_get_z_i32 bytes sigptr  in 
      let s, sigptr = sigptr_get_string (Int32.to_int len) bytes sigptr in 
      Some(s),sigptr
  

let decode_cattr_data ilg ca = 
  try 
    let bytes = ca.customData in
    let sigptr = 0 in 
    let bb0,sigptr = sigptr_get_byte bytes sigptr in 
    let bb1,sigptr = sigptr_get_byte bytes sigptr in 
    if not (int_eq bb0 0x01 && int_eq bb1 0x00) then failwith "decode_simple_cattr_data: invalid data";
    let rec parseVal argty sigptr = 
      match argty with 
      | Type_value tspec when string_eq (tname_of_tspec tspec) "System.SByte" ->  
          let n,sigptr = sigptr_get_i8 bytes sigptr in 
          CustomElem_int8 n, sigptr
      | Type_value tspec when string_eq (tname_of_tspec tspec) "System.Byte" ->  
          let n,sigptr = sigptr_get_u8 bytes sigptr in 
          CustomElem_uint8 n, sigptr
      | Type_value tspec when string_eq (tname_of_tspec tspec) "System.Int16" ->  
          let n,sigptr = sigptr_get_i16 bytes sigptr in 
          CustomElem_int16 n, sigptr
      | Type_value tspec when string_eq (tname_of_tspec tspec) "System.UInt16" ->  
          let n,sigptr = sigptr_get_u16 bytes sigptr in 
          CustomElem_uint16 n, sigptr
      | Type_value tspec when string_eq (tname_of_tspec tspec) "System.Int32" ->  
          let n,sigptr = sigptr_get_i32 bytes sigptr in 
          CustomElem_int32 n, sigptr
      | Type_value tspec when string_eq (tname_of_tspec tspec) "System.UInt32" ->  
          let n,sigptr = sigptr_get_u32 bytes sigptr in 
          CustomElem_uint32 n, sigptr
      | Type_value tspec when string_eq (tname_of_tspec tspec) "System.Int64" ->  
          let n,sigptr = sigptr_get_i64 bytes sigptr in 
          CustomElem_int64 n, sigptr
      | Type_value tspec when string_eq (tname_of_tspec tspec) "System.UInt64" ->  
          let n,sigptr = sigptr_get_u64 bytes sigptr in 
          CustomElem_uint64 n, sigptr
      | Type_value tspec when string_eq (tname_of_tspec tspec) "System.Double" ->  
          let n,sigptr = sigptr_get_ieee64 bytes sigptr in 
          CustomElem_float64 n, sigptr
      | Type_value tspec when string_eq (tname_of_tspec tspec) "System.Single" ->  
          let n,sigptr = sigptr_get_ieee32 bytes sigptr in 
          CustomElem_float32 n, sigptr
      | Type_value tspec when string_eq (tname_of_tspec tspec) "System.Char" ->  
          let n,sigptr = sigptr_get_u16 bytes sigptr in 
          CustomElem_char (Nums.u16_to_unichar n), sigptr
      | Type_value tspec when string_eq (tname_of_tspec tspec) "System.Boolean" ->  
          let n,sigptr = sigptr_get_byte bytes sigptr in 
          CustomElem_bool (not (int_eq n 0)), sigptr
      | Type_boxed tspec when string_eq (tname_of_tspec tspec) "System.String" ->  
          let n,sigptr = sigptr_get_serstring_possibly_null bytes sigptr in
          CustomElem_string n, sigptr
      | Type_boxed tspec when string_eq (tname_of_tspec tspec) "System.Type" ->  
          let n,sigptr = sigptr_get_serstring bytes sigptr in
          CustomElem_type (mk_tref(ilg.mscorlib_scoref,n)), sigptr
      | Type_boxed tspec when string_eq (tname_of_tspec tspec) "System.Object" ->  
          let et,sigptr = sigptr_get_u8 bytes sigptr in 
          let ty = celem_dec_ty ilg (u8_to_int et) in 
          parseVal ty sigptr 
      | Type_array(shape,elemTy) when deep_eq shape sdshape ->  
          let n,sigptr = sigptr_get_i32 bytes sigptr in 
          let rec parseElems acc n sigptr = 
            if int_eq n 0 then List.rev acc else
            let v,sigptr = parseVal elemTy sigptr in 
            parseElems (v ::acc) (n-1) sigptr in 
          let elems = parseElems [] (???n) sigptr in
          CustomElem_array(elems), sigptr
      | Type_value _ ->  (* assume it is an enumeration *)
          let n,sigptr = sigptr_get_i32 bytes sigptr in 
          CustomElem_int32 n, sigptr
      | _ ->  failwith "decode_simple_cattr_data: attribute data involves an enum or System.Type value" in
    let rec parseFixed argtys sigptr = 
      match argtys with 
        [] -> [],sigptr
      | h::t -> 
          let nh,sigptr = parseVal h sigptr in 
          let nt,sigptr = parseFixed t sigptr in 
          nh ::nt, sigptr in 
    let fixedArgs,sigptr = parseFixed (formal_args_of_mspec ca.customMethod) sigptr in 
    let nnamed,sigptr = sigptr_get_u16 bytes sigptr in 
    let rec parseNamed acc n sigptr = 
      if int_eq n 0 then List.rev acc else
      let isPropByte,sigptr = sigptr_get_u8 bytes sigptr in 
      let isProp = int_eq (u8_to_int isPropByte) 0x54 in
      let et,sigptr = sigptr_get_u8 bytes sigptr in 
      let ty = celem_dec_ty ilg (u8_to_int et) in 
      let nm,sigptr = sigptr_get_serstring bytes sigptr in
      let v,sigptr = parseVal ty sigptr in 
      parseNamed ((nm,ty,isProp,v) ::acc) (n-1) sigptr in 
    let named = parseNamed [] (u16_to_int nnamed) sigptr in
    fixedArgs,named
  with e -> failwith ("decode_cattr_data ilg: failure: "^Printexc.to_string e)
      

let mk_DebuggableAttribute_v2 ilg (jitTracking, ignoreSymbolStoreSequencePoints, jitOptimizerDisabled,enableEnC) = 
  mk_custom_attribute ilg 
    (tref_DebuggableAttribute ilg,[mk_nongeneric_value_typ (tref_DebuggableAttribute_DebuggingModes ilg)],
     [CustomElem_int32( 
                      (* See System.Diagnostics.DebuggableAttribute.DebuggingModes *)
                        (if jitTracking then !!!1 else 0l) |||  
                        (if jitOptimizerDisabled then !!!256 else 0l) |||  
                        (if ignoreSymbolStoreSequencePoints then !!!2 else 0l) |||
                        (if enableEnC then !!!4 else 0l))],[])




(* -------------------------------------------------------------------- 
 * Functions to collect up all the references in a full module or
 * asssembly manifest.  The process also allocates
 * a unique name to each unique internal assembly reference.
 * -------------------------------------------------------------------- *)

type refs = (*F# References and References = F#*) 
    { refsAssembly: assembly_ref list; 
      refsModul: modul_ref list; }

let insert_aref (e:assembly_ref) l = if List.mem e l.refsAssembly then l else {l with refsAssembly=e::l.refsAssembly}
let insert_mref (e:modul_ref) l = if List.mem e l.refsModul then l else {l with refsModul=e::l.refsModul}

module Hashset = struct
  type 'a t = ('a,int) Hashtbl.t
  let create n = Hashtbl.create n
  let mem t x = Hashtbl.mem t x
  let add t x = if not (mem t x) then Hashtbl.add t x 0
  let fold f t acc = Hashtbl.fold (fun x y -> f x) t acc
end

(*
module Visitset = struct
  type 'a t = 
      { name: string; 
        mutable misses: int;  
        mutable revisits: int;  
        mutable count: int;
        set: ('a,int) Hashtbl.t }
  let create nm n = { name = nm; misses = 0; revisits = 0; count = 0; set = Hashset.create n }
  let mem t x = Hashset.mem t.set x
  let add t x = if not (mem t x) then (Hashset.add t.set x)
  let fold f t acc = Hashset.fold f t.set acc
  let report t = dprintf4 "--->  %s has %d misses, %d revisits and %d elements\n" t.name t.misses t.revisits t.count
end

let visit_on idx v f = 
  match idx with 
  | None -> 
      v.Visitset.misses <- v.Visitset.misses  + 1;
      f()
  | Some i -> 
      if not (Visitset.mem v i) then begin
        Visitset.add v i;
        v.Visitset.count <- v.Visitset.count  + 1;
        f()
      end else begin
        v.Visitset.revisits <- v.Visitset.revisits  + 1
      end
*)


type refstate = 
    { refsA: assembly_ref Hashset.t; 
      refsM: modul_ref Hashset.t; 
      (* mutable mspecsVisited:  idx Visitset.t *) }

let empty_refs = 
  { refsAssembly=[];
    refsModul = []; }

let iter_option f x = match x with None -> () | Some x -> f x
let iter_pair f1 f2 (x,y) = f1 x; f2 y 

(* Now find references. *)
let refs_of_assref s x = Hashset.add s.refsA x
let refs_of_modref s x = Hashset.add s.refsM x
    
let refs_of_scoref s x = 
  match x with 
  | ScopeRef_local -> () 
  | ScopeRef_assembly assref -> refs_of_assref s assref
  | ScopeRef_module modref -> refs_of_modref s modref  
let refs_of_tref s x = refs_of_scoref s x.trefScope
  
let rec refs_of_typ s x = 
  match x with
  | Type_void | Type_other _  |  Type_tyvar _ -> ()
  | Type_modified(_,ty1,ty2) -> refs_of_tref s ty1; refs_of_typ s ty2
  | Type_array (_,ty)
  | Type_ptr ty | Type_byref ty -> refs_of_typ s ty 
  | Type_value tr | Type_boxed tr -> refs_of_tspec s tr
  | Type_fptr mref -> refs_of_callsig s mref 
and refs_of_inst s i = List.iter (refs_of_genactual s) i 
and refs_of_genactual s x = refs_of_typ s x
and refs_of_tspec s x = refs_of_tref s (tref_of_tspec x);  refs_of_inst s (inst_of_tspec x)
and refs_of_callsig s csig  = refs_of_typs s csig.callsigArgs; refs_of_typ s csig.callsigReturn
and refs_of_genparam s x = refs_of_typs s x.gpConstraints
and refs_of_genparams s b = List.iter (refs_of_genparam s) b
    
and refs_of_dloc s ts = refs_of_tref s ts
   
and refs_of_mref s x = 
  refs_of_dloc s x.mrefParent  ;
  List.iter (refs_of_typ s) x.mrefArgs;
  refs_of_typ s x.mrefReturn
    
and refs_of_fref s x = refs_of_tref s x.frefParent; refs_of_typ s x.frefType
and refs_of_ospec s (OverridesSpec(mref,ty)) = refs_of_mref s mref; refs_of_typ s ty 
and refs_of_mspec s x = 
  (* visit_on (mspec_optional_idx x)  s.mspecsVisited (fun () ->  *)
    let x1,x2,x3 = dest_mspec x in 
    refs_of_mref s x1;
    refs_of_typ s x2;
    refs_of_inst s x3
and refs_of_fspec s x =
  refs_of_fref s x.fspecFieldRef;
  refs_of_typ s x.fspecEnclosingType
and refs_of_typs s l = List.iter (refs_of_typ s) l
  
and refs_of_token s x = 
  match x with
  | Token_type ty -> refs_of_typ s ty
  | Token_method mr -> refs_of_mspec s mr
  | Token_field fr -> refs_of_fspec s fr
and refs_of_custom_attr s x = refs_of_mspec s x.customMethod
    
and refs_of_custom_attrs s cas = List.iter (refs_of_custom_attr s) (dest_custom_attrs cas)
and refs_of_varargs s tyso = iter_option (refs_of_typs s) tyso 
and refs_of_instr s x = 
  match x with
  | I_call (_,mr,varargs) | I_newobj (mr,varargs) | I_callvirt (_,mr,varargs) ->
      refs_of_mspec s mr;
      refs_of_varargs s varargs
  | I_callconstraint (_,tr,mr,varargs) -> 
      refs_of_typ s tr;
      refs_of_mspec s mr;
      refs_of_varargs s varargs
  | I_calli (_,callsig,varargs) ->  
      refs_of_callsig s callsig;  refs_of_varargs s varargs 
  | I_jmp mr | I_ldftn mr | I_ldvirtftn mr -> 
      refs_of_mspec s mr
  | I_ldsfld (_,fr) | I_ldfld (_,_,fr) | I_ldsflda fr | I_ldflda fr | I_stsfld (_,fr) | I_stfld (_,_,fr) -> 
      refs_of_fspec s fr
  | I_isinst ty | I_castclass ty | I_cpobj ty | I_initobj ty | I_ldobj (_,_,ty) 
  | I_stobj (_,_,ty) | I_box ty |I_unbox ty | I_unbox_any ty | I_sizeof ty
  | I_ldelem_any (_,ty) | I_ldelema (_,_,ty) |I_stelem_any (_,ty) | I_newarr (_,ty)
  | I_mkrefany ty | I_refanyval ty ->   refs_of_typ s ty 
  | I_ldtoken token -> refs_of_token s token 
  | I_stelem _|I_ldelem _|I_ldstr _|I_switch _|I_stloc _|I_stind _
  | I_starg _|I_ldloca _|I_ldloc _|I_ldind _
  | I_ldarga _|I_ldarg _|I_leave _|I_br _
  | I_brcmp _|I_rethrow|I_refanytype|I_ldlen|I_throw|I_initblk _ |I_cpblk _ 
  | I_localloc|I_ret |I_endfilter|I_endfinally|I_arglist
  | I_other _ | I_break|I_arith _ |I_seqpoint _ ->  ()
      
  
and refs_of_il_block s c  = 
  match c with 
  | BasicBlock bb -> Array.iter (refs_of_instr s) bb.bblockInstrs 
  | GroupBlock (_,l) -> List.iter (refs_of_il_code s) l 
  | RestrictBlock (nms,c) -> refs_of_il_code s c 
  | TryBlock (l,r) -> 
     refs_of_il_code s l;
     begin match r with 
     | FaultBlock flt -> refs_of_il_code s flt 
     | FinallyBlock flt -> refs_of_il_code s flt 
     | FilterCatchBlock clauses -> 
         List.iter 
           (fun (flt,ctch)  -> 
             refs_of_il_code s ctch;
             begin match flt with 
             | CodeFilter fltcode -> refs_of_il_code s fltcode 
             |  TypeFilter ty -> refs_of_typ s ty 
             end)
           clauses
     end
and refs_of_il_code s c  = refs_of_il_block s c 
    
and refs_of_ilmbody s il = 
  List.iter (refs_of_local s) il.ilLocals;
  refs_of_il_code s il.ilCode 
    
and refs_of_local s loc = refs_of_typ s loc.localType
    
and refs_of_mbody s x = 
  match x with 
  | MethodBody_il il -> refs_of_ilmbody s il
  | MethodBody_pinvoke (attr) -> refs_of_modref s attr.pinvokeWhere
  | _ -> ()

and refs_of_permission_value s x = 
  match x with 
  | PermissionValue_bool _ | PermissionValue_int32 _ | PermissionValue_string _ -> ()
  | PermissionValue_enum_int8 (tn,_) | PermissionValue_enum_int16 (tn,_)
  | PermissionValue_enum_int32 (tn,_) -> refs_of_tref s tn
  
and refs_of_permission s x = 
  match x with 
  | Permission (_,ty,l) -> 
      refs_of_typ s ty;
      List.iter (fun (_,v) -> refs_of_permission_value s v) l
  | PermissionSet _ -> ()
  
and refs_of_security_decls s l = List.iter (refs_of_permission s) (dest_security_decls l) 
    
and refs_of_mdef s md = 
  List.iter (refs_of_param s) md.mdParams;
  refs_of_return s md.mdReturn;
  refs_of_mbody s  (dest_mbody md.mdBody);
  refs_of_security_decls s  md.mdSecurityDecls;
  refs_of_custom_attrs s  md.mdCustomAttrs;
  refs_of_genparams s  md.mdGenericParams
    
and refs_of_param s p = refs_of_typ s p.paramType 
and refs_of_return s rt = refs_of_typ s rt.returnType
and refs_of_mdefs s fmap =  List.iter (refs_of_mdef s) (dest_mdefs fmap)
    
and refs_of_event_def s ed = 
  iter_option (refs_of_typ s)  ed.eventType ;
  refs_of_mref  s ed.eventAddOn ;
  refs_of_mref  s ed.eventRemoveOn;
  iter_option (refs_of_mref s) ed.eventFire ;
  List.iter (refs_of_mref s)  ed.eventOther ;
  refs_of_custom_attrs  s ed.eventCustomAttrs
    
and refs_of_events s fmap =  List.iter (refs_of_event_def s) (dest_edefs fmap)
    
and refs_of_property_def s pd = 
  iter_option (refs_of_mref s)  pd.propSet ;
  iter_option (refs_of_mref s)  pd.propGet ;
  refs_of_typ  s pd.propType ;
  refs_of_typs  s pd.propArgs ;
  refs_of_custom_attrs  s pd.propCustomAttrs
    
and refs_of_properties s fmap = List.iter (refs_of_property_def s) (dest_pdefs fmap)
    
and refs_of_fdef s fd = 
  refs_of_typ  s fd.fdType;
  refs_of_custom_attrs  s fd.fdCustomAttrs

and refs_of_fields s fields = List.iter (refs_of_fdef s) fields
    
and refs_of_method_impls s mimpls =  List.iter (refs_of_method_impl s) mimpls
    
and refs_of_method_impl s m = 
  refs_of_ospec s m.mimplOverrides;
  refs_of_mspec s m.mimplOverrideBy
and refs_of_tdef_kind s k =  ()
  
and refs_of_tdef s td  =  
  refs_of_types s td.tdNested;
  refs_of_genparams s  td.tdGenericParams;
  refs_of_typs  s td.tdImplements;
  iter_option (refs_of_typ s) td.tdExtends;
  refs_of_mdefs        s td.tdMethodDefs;
  refs_of_security_decls s td.tdSecurityDecls;
  refs_of_fields       s (dest_fdefs td.tdFieldDefs);
  refs_of_method_impls s (dest_mimpls td.tdMethodImpls);
  refs_of_events       s td.tdEvents;
  refs_of_tdef_kind    s td.tdKind;
  refs_of_custom_attrs s td.tdCustomAttrs;
  refs_of_properties   s td.tdProperties

and refs_of_string s _ = ()
and refs_of_types s types = List.iter  (refs_of_tdef s) (dest_tdefs types) 
    
and refs_of_exported_type s c = 
  refs_of_custom_attrs s c.exportedTypeCustomAttrs
    
and refs_of_exported_types s tab = List.iter (refs_of_exported_type s) (dest_exported_types tab)
    
and refs_of_resource_where s x = 
  match x with 
  | Resource_local _ -> ()
  | Resource_file (mref,_) -> refs_of_modref s mref
  | Resource_assembly aref -> refs_of_assref s aref
and refs_of_resource s x = 
  refs_of_resource_where s x.resourceWhere;
  refs_of_custom_attrs s x.resourceCustomAttrs
    
and refs_of_resources s tab = List.iter (refs_of_resource s) (dest_resources tab)
    
and refs_of_modul s m = 
  refs_of_types s m.modulTypeDefs;
  refs_of_resources s m.modulResources;
  iter_option (refs_of_manifest s) m.modulManifest
    
and refs_of_manifest s m = 
  refs_of_custom_attrs s m.manifestCustomAttrs;
  refs_of_exported_types s m.manifestExportedTypes;
  refs_of_security_decls s m.manifestSecurityDecls

let refs_of_module modul = 
  let s = 
    { refsA = Hashset.create 10; 
      refsM = Hashset.create 5; 
      (* mspecsVisited = Visitset.create "mspecs" 1000 *) } in 
  refs_of_modul s modul;
  (* Visitset.report s.mspecsVisited; *)
  { refsAssembly = Hashset.fold (fun x acc -> x::acc) s.refsA [];
    refsModul =  Hashset.fold (fun x acc -> x::acc) s.refsM [] }





let parse_version s = 
  try 
    let p0 = if String.contains s 'v' then String.index s 'v' + 1 else 0 in 
    let p1 = String.index s '.' in 
    let s1 = String.sub s p0 (p1-p0) in 
    let v1 = int_of_string s1 in 
    let p2 = String.index_from s (p1+1) '.' in
    let s2 = String.sub s (p1+1) (p2-(p1+1)) in 
    let v2 = int_of_string s2 in 
    let v3,v4 = 
      try 
       let p3 = if String.contains_from s (p2+1) '.' then String.index_from s (p2+1) '.' else String.length s in
       let s3 = String.sub s (p2+1) (p3-(p2+1)) in 
      let v3 = try int_of_string s3 with _ -> 0 in 
      let v4 = try if p3+1 >= String.length s then 0 else int_of_string (String.sub s (p3+1) (String.length s - (p3+1))) with _ -> 0 in     
      v3,v4 
     with _ -> 0,0 in 
    (Nums.int_to_u16 v1, Nums.int_to_u16 v2,
     Nums.int_to_u16 v3,Nums.int_to_u16 v4)
  with _ -> 
    let err = ("Invalid version format '"^s^"'.  Use major.minor.patch.qfe, e.g. 1.33.2.3 or v1.0.3705 or v1.1.4322") in 
    dprint_endline ("Error: "^ err);
    failwith err

let version_compare (a1,a2,a3,a4) ((b1,b2,b3,b4) : version_info) = 
  let c = Pervasives.compare a1 b1 in if Pervasives.(<>) c 0 then c else
  let c = Pervasives.compare a2 b2 in if Pervasives.(<>) c 0 then c else
  let c = Pervasives.compare a3 b3 in if Pervasives.(<>) c 0 then c else
  let c = Pervasives.compare a4 b4 in if Pervasives.(<>) c 0 then c else
  0


let version_max a b = if version_compare a b < 0 then b else a
let version_min a b = if version_compare a b > 0 then b else a


let mdef_for_semantic_mref td mref = 
  let args = args_of_mref mref in 
  let nargs = List.length args in 
  let nm = name_of_mref mref in 
  let mid =(nm,nargs) in 
  let possibles = find_mdefs_by_arity mid td.tdMethodDefs in 
  if isnull possibles then failwith ("no method named "^nm^" found in type "^td.tdName);
  match 
    List.filter 
      (fun md -> 
        callconv_eq (callconv_of_mref mref) md.mdCallconv &
        int_eq nargs (List.length md.mdParams) &
        List.for_all2 (fun p1 p2 -> deep_eq p1.paramType p2) md.mdParams (args_of_mref mref)) possibles with 
  | [] -> 
      failwith ("no method named "^nm^" with appropriate argument types found in type "^td.tdName);
  | [mdef] ->  mdef
  | _ -> 
      failwith ("multiple methods named "^nm^" appear with identical argument types in type "^td.tdName)
        
let modref_for_modul m =
  { modulRefName = m.modulName;
    modulRefNoMetadata = false; 
    modulRefHash = None  }


let ungenericize_tname n = 
  let sym = '`' in
  if 
    String.contains n sym && 
      (* check what comes after the symbol is a number *)
    begin
      let m = String.rindex n sym in 
      let res = ref (m < String.length n - 1) in 
      for i = m + 1 to String.length n - 1 do
        res := !res && String.get n i >= '0' && String.get n i <= '9';
      done;
      !res
    end
  then 
      let pos = String.rindex n sym in 
      String.sub n 0 pos
  else n


(* -------------------------------------------------------------------- 
 * Augmentations
 * -------------------------------------------------------------------- *)

(*F# 
type TypeRef
  with 
    member x.FullName = nested_tname_of_tref(x)
  end

type TypeSpec
  with 
    member x.Scope=x.TypeRef.Scope
    member x.Nesting=x.TypeRef.Nesting
    member x.Name=x.TypeRef.Name
    member x.FullName=x.TypeRef.FullName
  end
F#*)

type event_ref = (*F#  EventRef 
and EventRef = F#*) { erA: type_ref; erB: string }
  (*F# with 
    static member Create(a,b) = {erA=a;erB=b}
    member x.EnclosingTypeRef = x.erA
    member x.Name = x.erB
  end F#*)

type event_spec = (*F# EventSpec 
and EventSpec = F#*) { esA: event_ref; esB: typ }
  (*F# with 
     static member Create (a,b) = {esA=a;esB=b}
     member x.EventRef = x.esA
     member x.EnclosingType = x.esB
  end F#*)

type property_ref = (*F# PropertyRef 
and PropertyRef = F#*) { prA: type_ref; prB: string }
  (*F# with 
    static member Create (a,b) = {prA=a;prB=b}
    member x.EnclosingTypeRef = x.prA
    member x.Name = x.prB
  end F#*)

type property_spec =  (*F# PropertySpec 
and PropertySpec = F#*) { psA: property_ref; psB: typ }
  (*F# with 
    static member Create (a,b) = {psA=a;psB=b}
    member x.PropertyRef = x.psA
    member x.EnclosingType = x.psB
  end F#*)


let tref_of_pref x = x.prA
let tref_of_eref x = x.erA
let name_of_pref x = x.prB
let name_of_eref x = x.erB
let mk_pref (a,b) = {prA=a;prB=b}
let mk_eref (a,b) = {erA=a;erB=b}
let mk_pspec (a,b) = {psA=a;psB=b}
let mk_espec (a,b) = {esA=a;esB=b}
let enclosing_typ_of_pspec x = x.psB
let enclosing_typ_of_espec x = x.esB
let pref_of_pspec x = x.psA
let eref_of_espec x = x.esA
let eref_for_edef scope (tdefs,tdef) (x:event_def) = mk_eref (tref_for_nested_tdef scope (tdefs,tdef), x.eventName)
let pref_for_pdef scope (tdefs,tdef) (x:property_def) = mk_pref (tref_for_nested_tdef scope (tdefs,tdef), x.propName)

(*F#
type MethodDef 
  with 
      member x.Name = x.mdName
      //mdKind: method_kind;
      //Body: method_body;   
      //CodeKind: method_code_kind;   
      member x.CallingConvention = x.mdCallconv
      member x.Parameters = x.mdParams
      member x.Return = x.mdReturn
      member x.Access = x.mdAccess
      member x.IsInternalCall = x.mdInternalCall
      member x.IsManaged = x.mdManaged
      member x.IsForwardRef = x.mdForwardRef
      member x.SecurityDecls = x.mdSecurityDecls
      /// Note: some methods are marked "HasSecurity" even if there are no permissions attached, e.g. if they use SuppressUnmanagedCodeSecurityAttribute 
      member x.HasSecurity = x.mdHasSecurity
      member x.IsEntrypoint = x.mdEntrypoint
      member x.IsReqSecObj = x.mdReqSecObj
      member x.IsHideBySig = x.mdHideBySig
      /// The method is exported to unmanaged code using COM interop. 
      member x.IsUnmanagedExport = x.mdUnmanagedExport
      member x.IsSynchronized = x.mdSynchronized
      member x.IsPreserveSig = x.mdPreserveSig
      // Whidbey feature: SafeHandle finalizer must be run 
      member x.IsMustRun = x.mdMustRun
      //member Export: (i32 * string option) option; 
      //member VtableEntry: (i32 * i32) option;  
      member x.GenericParams = x.mdGenericParams
      member x.CustomAttrs = x.mdCustomAttrs
      member x.Code = try_code_of_mdef x
      member x.IsIL = match dest_mbody x.mdBody with | MethodBody_il _ -> true | _ -> false
      member x.Locals = match dest_mbody x.mdBody with | MethodBody_il il -> il.ilLocals | _ -> []
      member x.IsNoInline = let b = ilmbody_of_mdef x in b.ilNoInlining  
      member x.SourceMarker = let b = ilmbody_of_mdef x in b.ilSource
      member x.MaxStack = let b = ilmbody_of_mdef x in b.ilMaxStack  
      member x.IsZeroInit = let b = ilmbody_of_mdef x in b.ilZeroInit
      member x.IsClassInitializer = match x.mdKind with | MethodKind_cctor -> true | _ -> false
      member x.IsConstructor = match x.mdKind with | MethodKind_ctor -> true | _ -> false
      member x.IsStatic = match x.mdKind with | MethodKind_static -> true | _ -> false
      member x.IsNonVirtualInstance = match x.mdKind with | MethodKind_nonvirtual -> true | _ -> false
      member x.IsVirtual = match x.mdKind with | MethodKind_virtual _ -> true | _ -> false

      member x.IsFinal                = match x.mdKind with | MethodKind_virtual v -> v.virtFinal    | _ -> invalid_arg "MethodDef.IsFinal"
      member x.IsNewSlot              = match x.mdKind with | MethodKind_virtual v -> v.virtNewslot  | _ -> invalid_arg "MethodDef.IsNewSlot"
      member x.IsCheckAccessOnOverride= match x.mdKind with | MethodKind_virtual v -> v.virtStrict   | _ -> invalid_arg "MethodDef.IsCheckAccessOnOverride"
      member x.IsAbstract             = match x.mdKind with | MethodKind_virtual v -> v.virtAbstract | _ -> invalid_arg "MethodDef.IsAbstract"
    end

type PropertyDef 
  with 
      member x.Name = x.propName
      member x.SetMethod = x.propSet
      member x.GetMethod = x.propGet
      member x.CallingConvention = x.propCallconv
      member x.Type = x.propType
      member x.Init = x.propInit
      member x.Args = x.propArgs
      member x.CustomAttrs = x.propCustomAttrs
  end
    
type EventDef 
  with 
      member x.Type = x.eventType
      member x.Name = x.eventName
      member x.AddMethod = x.eventAddOn
      member x.RemoveMethod = x.eventRemoveOn
      member x.FireMethod = x.eventFire
      member x.OtherMethods = x.eventOther
      member x.CustomAttrs = x.eventCustomAttrs
  end
    
type TypeDefs
  with 
    member x.Details = dest_tdefs x
  end

type MethodDefs
  with 
    member x.Details = dest_mdefs x
  end

type FieldDefs
  with 
    member x.Details = dest_fdefs x
  end

type PropertyDefs
  with 
    member x.Details = dest_pdefs x
  end

type EventDefs
  with 
    member x.Details = dest_edefs x
  end

type Attributes
  with 
    member x.Details = dest_custom_attrs x
  end

type Permissions
  with 
    member x.Details = dest_security_decls x
  end

type MethodImplDefs
  with 
    member x.Details = dest_mimpls x
  end

type Resources
  with 
    member x.Details = dest_resources x
  end

type LazyMethodBody
  with 
    member x.Details = dest_mbody x
  end

type ExportedTypes
  with 
    member x.Details = dest_exported_types x
  end


type NestedExportedTypes
  with 
    member x.Details = dest_nested_exported_types x
  end

type AssemblyRef 
  with 
    member x.QualifiedName=qualified_name_of_aref x
  end 

type ArrayShape 
  with 
    member x.Rank = rank_of_array_shape x
    static member SingleDimensional = sdshape    
  end

type Attribute 
  with 
    member x.Data = x.customData
    member x.Method =x.customMethod
  end

type Parameter 
  with 
    member x.Name = x.paramName
    member x.Type = x.paramType
    member x.Default = x.paramDefault
    member x.Marshal = x.paramMarshal
    member x.IsIn = x.paramIn
    member x.IsOut = x.paramOut
    member x.IsOptional = x.paramOptional
    member x.CustomAttrs = x.paramCustomAttrs
  end
  
type ReturnValue 
  with 
        member x.Type =  x.returnType
        member x.Marshal = x.returnMarshal
        member x.CustomAttrs = x.returnCustomAttrs
  end
  
type PInvokeMethod 
  with 
        member x.Where = x.pinvokeWhere
        member x.Name = x.pinvokeName
        member x.CallingConvention = x.pinvokeCallconv
        member x.CharEncoding = x.pinvokeEncoding
        member x.NoMangle = x.pinvokeNoMangle
        member x.LastError = x.pinvokeLastErr
        member x.ThrowOnUnmappableChar = x.pinvokeThrowOnUnmappableChar
        member x.CharBestFit = x.pinvokeBestFit
  end

type OverridesSpec 
    with 
        member x.MethodRef = let (OverridesSpec(mr,ty)) = x in mr
        member x.EnclosingType = let (OverridesSpec(mr,ty)) = x in ty
    end 
  
type MethodVirtualInfo 
    with 
        member x.IsFinal = x.virtFinal
        member x.IsNewSlot = x.virtNewslot
        member x.IsCheckAccessOnOverride = x.virtStrict
        member x.IsAbstract = x.virtAbstract
        member x.OverridenMethod = x.virtOverrides
    end 

type TypeDefLayoutInfo
    with 
        member x.Size = x.typeSize
        member x.Pack = x.typePack
    end 

type ExportedType 
    with 
        member x.ModuleRef = x.exportedTypeScope
        member x.Name = x.exportedTypeName
        member x.Access = x.exportedTypeAccess
        member x.Nested = x.exportedTypeNested
        member x.CustomAttrs = x.exportedTypeCustomAttrs
    end 

type Resource
    with 
        member x.Name = x.resourceName
        member x.Location = x.resourceWhere
        member x.Access = x.resourceAccess
        member x.CustomAttrs = x.resourceCustomAttrs
    end 
      

type References 
    with 
        member x.AssemblyReferences = x.refsAssembly
        member x.ModuleReferences = x.refsModul
    end

type BasicBlock 
    with 
        member x.Label = x.bblockLabel
        member x.Instructions = x.bblockInstrs
    end 

type CallingSignature 
    with 
        member x.CallingConvention = x.callsigCallconv
        member x.ArgumentTypes = x.callsigArgs
        member x.ReturnType = x.callsigReturn
    end 

type DebugMapping 
    with 
        member x.LocalVarIndex = x.localNum
        member x.Name = x.localName
    end 

type GenericParameterDef 
    with 
        member x.Name = x.gpName
        member x.Constraints = x.gpConstraints
        member x.Variance = x.gpVariance
        member x.HasReferenceTypeConstraint = x.gpReferenceTypeConstraint
        member x.HasNotNullableValueTypeConstraint = x.gpNotNullableValueTypeConstraint
        member x.HasDefaultConstructorConstraint = x.gpDefaultConstructorConstraint
    end 

type Local 
    with 
        member x.Type = x.localType
        member x.IsPinned = x.localPinned
    end 


type Manifest 
    with   
      member x.Name = x.manifestName
      member x.AuxModuleHashAlgorithm = x.manifestAuxModuleHashAlgorithm
      member x.SecurityDecls = x.manifestSecurityDecls
      member x.PublicKey = x.manifestPublicKey
      member x.Version = x.manifestVersion
      member x.Locale = x.manifestLocale
      member x.CustomAttrs = x.manifestCustomAttrs
      member x.Longevity = x.manifestLongevity
      member x.DisableJitOptimizations = x.manifestDisableJitOptimizations
      member x.JitTracking = x.manifestJitTracking
      member x.ExportedTypes = x.manifestExportedTypes
      member x.EntrypointElsewhere = x.manifestEntrypointElsewhere
    end

#if CLI_AT_MOST_1_1
#else
open System.Runtime.CompilerServices
[<Dependency("FSharp.Core",LoadHint.Always)>] do ()
[<Dependency("FSharp.Compatibility",LoadHint.Always)>] do ()
#endif

F#*)


