// (c) Microsoft Corporation 2005-2007. 

#light

#if CLI_AT_MOST_1_1
#else

namespace Microsoft.FSharp.Collections

open Microsoft.FSharp.Core
open Microsoft.FSharp.Core.Operators
open Microsoft.FSharp.Collections
open Microsoft.FSharp.Primitives.Basics

type ResizeArray<'a> = System.Collections.Generic.List<'a>

[<CompilationRepresentation(CompilationRepresentationFlags.ModuleSuffix)>]
module ResizeArray = 

    type 'a t = ResizeArray<'a>

    let length (arr: 'a ResizeArray) =  arr.Count
    let get (arr: 'a ResizeArray) (n:int) =  arr.[n]
    let set (arr: 'a ResizeArray) (n:int) (x:'a) =  arr.[n] <- x
    let create  (n:int) x = new ResizeArray<_> { for i in 1 .. n -> x }
    let init (n:int) (f: int -> 'a) =  new ResizeArray<_> { for i in 0 .. n-1 -> f i }

    let blit (arr1:'a ResizeArray) start1 (arr2: 'a ResizeArray) start2 len =
      for i = 0 to len - 1 do 
        arr2.[start2+i] <- arr1.[start1 + i]
      done

    let concat (arrs:'a ResizeArray list) = new ResizeArray<_> { for arr in arrs for x in arr -> x }
    let append arr1 arr2 = concat [arr1; arr2]
    let sub (arr:'a ResizeArray) start len = new ResizeArray<_> { for i in start to start+len-1 -> arr.[i] }
    let fill (arr:'a ResizeArray) (start:int) (len:int) (x:'a) =
      for i = start to start + len - 1 do 
        arr.[i] <- x
      done

    let copy      (arr:'a ResizeArray) = new ResizeArray<_>(arr)
    let to_list   (arr:'a ResizeArray) = Seq.to_list arr
    let of_list l                      = new ResizeArray<_>(Seq.of_list l)
    let iter    f (arr:'a ResizeArray) = Seq.iter f arr
    let map     f (arr:'a ResizeArray) = new ResizeArray<_>(Seq.map f arr)
    let mapi    f (arr:'a ResizeArray) = new ResizeArray<_>(Seq.mapi f arr)
    let iteri   f (arr:'a ResizeArray) = Seq.iteri f arr
    let exists  f (arr:'a ResizeArray) = Seq.exists f arr
    let for_all f (arr:'a ResizeArray) = Seq.for_all f arr
    let find    f (arr:'a ResizeArray) = Seq.find f arr
    let first   f (arr:'a ResizeArray) = Seq.first f arr
    let tryfind f (arr:'a ResizeArray) = Seq.tryfind f arr
    let iter2   f (arr1:'a ResizeArray) (arr2:'b ResizeArray)  = Seq.iter2 f arr1 arr2
    let map2    f (arr1:'a ResizeArray) (arr2:'b ResizeArray) = new ResizeArray<_>(Seq.map2 f arr1 arr2)
    let choose  f (arr:'a ResizeArray) = new ResizeArray<_>(Seq.choose f arr)
    let filter  f (arr:'a ResizeArray) = new ResizeArray<_>(Seq.filter f arr)
    let partition f arr = 
      let res1 = new ResizeArray<_>() in 
      let res2 = new ResizeArray<_>() in 
      for i = 0 to length arr - 1 do 
        let x = arr.[i] in 
        if f x then res1.Add(x) else res2.Add(x)
      done;
      res1, res2

    let combine (arr1:'a ResizeArray) (arr2:'b ResizeArray) = new ResizeArray<_>(Seq.combine arr1 arr2)
    let split arr = 
      let len = length arr in 
      let res1 = new ResizeArray<_>(len) in 
      let res2 = new ResizeArray<_>(len) in 
      for i = 0 to len - 1 do 
        let x,y = arr.[i] in 
        res1.[i] <- x;
        res2.[i] <- y;
      done;
      res1,res2

    let rev arr = 
      let len = length arr in 
      let res = new ResizeArray<_>(len) in 
      for i = len downto 1 do 
        res.[(len - i) - 1] <- arr.[i]
      done;
      res

    let fold_right (f : 'a -> 'b -> 'b) (arr:'a ResizeArray) (acc: 'b) =
        let mutable res = acc in 
        let len = length arr in 
        for i = len - 1 downto 0 do 
          res <- f (get arr i) res
        done;
        res
    let fold_left (f : 'a -> 'b -> 'a) (acc: 'a) (arr:'b ResizeArray) =
        let mutable res = acc in 
        let len = length arr in 
        for i = 0 to len - 1 do 
          res <- f res (get arr i)
        done;
        res

    let to_array (arr:'a ResizeArray) = Seq.to_array arr
    let of_array (arr :  'a array) = new ResizeArray<_>(Seq.of_array arr)

#endif
