(* (c) Microsoft Corporation 2005-2006.  *)

let rec foldi f n s = if n < 0 then s else foldi f (n-1) (f n s)
let rec mapi_aux f n acc = if n < 0 then acc else mapi_aux f (n-1) (f n :: acc)
let mapi f n = mapi_aux f n []
let rec list_mapi_aux f l n = 
  match l with [] -> [] | (h::t) -> let r = f n h  in r :: list_mapi_aux f t (n+1)
let list_mapi f l =  list_mapi_aux f l 0
let rec list_iteri_aux f l n = 
  match l with [] -> () | (h::t) -> f n h; list_iteri_aux f t (n+1)
let list_iteri f l =  list_iteri_aux f l 0


type ident = string
type code = string * Lexing.position

type alphabet = LChar of char | LEof

type spec = 
    { topcode: code;
      macros: (ident * regexp) list;
      rules: (ident * ident list * clause list) list;
      botcode: code }
and clause = regexp * code
and regexp = 
  | Alt of regexp list
  | Seq of regexp list
  | Inp of alphabet
  | Star of regexp
  | Macro of ident

type node = { id: int;
              name: string;
              mutable tr: (alphabet option * node) list;
              ac: (int * int) list }

let newNfaNodeId = let n = ref 0 in fun () -> incr n; !n
let nfaNodeMap = Hashtbl.create 100
let getNfaNode nid = Hashtbl.find nfaNodeMap nid 

let newNfaNode trs ac = 
  let n = newNfaNodeId() in 
  (* List.iter (fun (inp,dest) -> Printf.eprintf "nfaNode %d: %s --> %d\n" n (match inp with None -> "<empty>" | Some (LChar c) -> String.make 1 c | Some LEof -> "eof") dest.id) trs;*)
  (* if ac <> [] then Printf.eprintf "nfaNode %d: accepting\n" n;*)
  let node = {id=n;name=string_of_int n;tr=trs;ac=ac} in 
  Hashtbl.add nfaNodeMap n node;
  node

let rec sRegexp macros re dest = 
  match re with 
  | Alt res -> 
      let trs = List.map (fun re -> (None,sRegexp macros re dest)) res in
      newNfaNode trs [] 
  | Seq res -> 
      List.fold_right (sRegexp macros) res dest 
  | Inp c -> 
      newNfaNode [(Some c, dest)] []
  | Star re -> 
      let fre = newNfaNode [(None, dest); (* (None,sre) *) ] [] in 
      let sre = sRegexp macros re fre in 
      fre.tr <- (None,sre) :: fre.tr;
      newNfaNode [(None,sre); (None,dest)] []
  | Macro m -> 
      if not (List.mem_assoc m macros) then failwith ("The macro "^m^" is not defined");
      sRegexp macros (List.assoc m macros) dest 

let sTrans mkAction macros n (regexp,code) = 
  let sAccept = newNfaNode [] [(n,mkAction code)] in 
(*   sAccept.tr <- (None,sAccept) :: sAccept.tr; *)
  sRegexp macros regexp sAccept 

let lexerStateToNfa macros clauses = 
  let actions = ref [] in 
  let mkAction code = let id = List.length !actions in actions := code :: !actions; id in 
  let trs = list_mapi (fun n x -> (None,sTrans mkAction macros n x)) clauses in
  newNfaNode trs [], List.rev !actions

(*IF-OCAML*) 
module NodeIdSetImpl = Set.Make(struct type t = int let compare (x:int) y = compare x y end)
type nodeIdSet = NodeIdSetImpl.t ref

module NodeIdMutableSet = struct
  let create() = ref NodeIdSetImpl.empty
  let mem x y = NodeIdSetImpl.mem x !y
  let add x y = y := NodeIdSetImpl.add x !y
  let fold f x z = NodeIdSetImpl.fold f !x z
  let iter f x = NodeIdSetImpl.iter f !x
  let is_empty s = NodeIdSetImpl.is_empty !s
  let elements s = NodeIdSetImpl.elements !s
  let compare s1 s2 = NodeIdSetImpl.compare !s1 !s2
end

(*ENDIF-OCAML*)
(*F# 
#if GENERICS
type nodeIdSet = System.Collections.Generic.SortedList<int,int>

module NodeIdMutableSet = begin 

  open System.Collections.Generic
  let create() : nodeIdSet = new SortedList<_,_>()
  let mem i (s : nodeIdSet) = s.ContainsKey (i)
  let add i (s : nodeIdSet) =  s.Item(i) <- i
  let fold f (s : nodeIdSet) z = Seq.fold (fun z (kvp:KeyValuePair<_,_>) -> f kvp.Key z) z s 
  let compare (s1 : nodeIdSet) (s2 : nodeIdSet)= 
    (* TODO: replace this with IEnumerable.compare Pervasives.compare s1 s2 when *)
    (* new LKG available *)
    let e1 = s1.Keys.GetEnumerator() in 
    let e2 = s2.Keys.GetEnumerator() in 
    let rec go () = 
      let e1ok = e1.MoveNext() in 
      let e2ok = e2.MoveNext() in 
      let c = Pervasives.compare e1ok e2ok in if c <> 0 then c else
      if not e1ok || not e2ok then 0 else
      let c = Pervasives.compare e1.Current e2.Current in if c <> 0 then c else
      go () in 
    go()
  let iter f (s1 : nodeIdSet) = s1.Keys |> IEnumerable.iter f
  let is_empty (s : nodeIdSet) = s.Count = 0
  let elements (s : nodeIdSet) = List.of_IEnumerable s.Keys
end
#else
let  NodeIdSetImpl = Set.Make(fun x y -> compare x y)
type nodeIdSet = Collections.Tagged.SetUntagged<node> ref

module NodeIdMutableSet = struct
  let create() = ref NodeIdSetImpl.empty
  let mem x y = NodeIdSetImpl.mem x !y
  let add x y = y := NodeIdSetImpl.add x !y
  let fold f x z = NodeIdSetImpl.fold f !x z
  let iter f x = NodeIdSetImpl.iter f !x
  let is_empty s = NodeIdSetImpl.is_empty !s
  let elements s = NodeIdSetImpl.elements !s
  let compare s1 s2 = NodeIdSetImpl.compare !s1 !s2
end
#endif

F#*)

let (>>) g f x = f(g(x))
let rec eclosure1 acc n = 
  if not (NodeIdMutableSet.mem n.id acc) then begin
    NodeIdMutableSet.add n.id acc;
    List.iter(fun (e,n2) -> match e with None -> eclosure1 acc n2 | _ -> ()) n.tr;
  end

let eclosure nset = 
  let res = NodeIdMutableSet.create() in 
  NodeIdMutableSet.iter (getNfaNode >> eclosure1 res) nset ;
  res

let move1 nset c n =
  List.iter
    (fun tr -> 
      match tr with 
      | (Some c2,dest) when c = c2 -> NodeIdMutableSet.add dest.id nset
      | _ -> ()) 
    n.tr
      
let move c nset = 
  let res = NodeIdMutableSet.create() in 
  NodeIdMutableSet.iter (getNfaNode >> move1 res c) nset;
  res

(*IF-OCAML*) 
module NodeSetMap = Map.Make(struct type t = nodeIdSet let compare = NodeIdMutableSet.compare end) 
module NodeSetSet = Set.Make(struct type t = nodeIdSet let compare = NodeIdMutableSet.compare end)
(*ENDIF-OCAML*)
(*F# 
let NodeSetMap = Map.Make(NodeIdMutableSet.compare)
let NodeSetSet = Set.Make(NodeIdMutableSet.compare)
F#*)

let nfaToDfa numDfaNodes nfaStartNode = 
  let dfaNodes = ref NodeSetMap.empty in 
  let getDfaNode nfaSet = 
    if NodeSetMap.mem nfaSet !dfaNodes then 
      NodeSetMap.find nfaSet !dfaNodes 
    else 
      let dfaNode =
        { id= !numDfaNodes; 
          name = NodeIdMutableSet.fold (fun nid s -> let n = getNfaNode nid in n.name^"-"^s) nfaSet ""; 
          tr=[];
          ac=List.concat (List.map (fun nid -> let n = getNfaNode nid in n.ac) (NodeIdMutableSet.elements nfaSet)) } in 
      (* if dfaNode.ac <> [] then Printf.eprintf "dfaNode %d: accepting\n" dfaNode.id;*)

      dfaNodes :=  NodeSetMap.add nfaSet dfaNode !dfaNodes; 
      numDfaNodes := !numDfaNodes + 1;
      dfaNode in 
  let nfaSet0 = NodeIdMutableSet.create() in 
  eclosure1 nfaSet0 nfaStartNode;
  let workList = ref [nfaSet0] in 
  let doneSet = ref NodeSetSet.empty in 
  let rec loop () = 
    match !workList with 
      [] -> ()
    | nfaSet ::t -> 
        workList := t;
        if NodeSetSet.mem nfaSet !doneSet then 
          let dfaNode = getDfaNode nfaSet in 
          loop () 
        else
          let dfaNode = getDfaNode nfaSet in 
          let doInp inp = 
            let moveSet = eclosure (move inp nfaSet) in 
            if not (NodeIdMutableSet.is_empty moveSet) then begin
              let moveSetDfaNode = getDfaNode moveSet in
              dfaNode.tr <- (Some inp, moveSetDfaNode) :: dfaNode.tr;
              (* Printf.eprintf "%d (%s) : %s --> %d (%s)\n" dfaNode.id dfaNode.name (match inp with LChar c -> String.make 1 c | LEof -> "eof") moveSetDfaNode.id moveSetDfaNode.name;*)
              let dfaNode = getDfaNode moveSet in 
              workList := moveSet :: !workList;
            end in 
          for i = 0 to 255 do 
            doInp (LChar (Char.chr i));
          done;
          doInp LEof;
          doneSet := NodeSetSet.add nfaSet !doneSet;
          loop() in 
  loop();
  getDfaNode nfaSet0, 
  List.sort (fun s1 s2 -> compare s1.id s2.id) (NodeSetMap.fold (fun _ x acc -> x :: acc) !dfaNodes [])

let accept dfaNode = 
  List.map snd (List.sort (fun (a,_) (b,_) -> compare a b) dfaNode.ac)

let compile spec = 
  let numDfaNodes = ref 0 in 
  List.fold_right
    (fun (name,args,clauses) (perRuleData,dfaNodes) -> 
      let nfa, actions = lexerStateToNfa  spec.macros clauses in 
      let ruleStartNode, ruleNodes = nfaToDfa numDfaNodes nfa in 
      (ruleStartNode,actions) :: perRuleData, ruleNodes @ dfaNodes)
    spec.rules
    ([],[])

