(* (c) Microsoft Corporation. All rights reserved *)
(* -------------------------------------------------------------------- 
 * Erase discriminated unions.
 * -------------------------------------------------------------------- *)

(*F# 
module Microsoft.Research.AbstractIL.Extensions.ILX.Cu_erase
open Microsoft.Research.AbstractIL 
open Microsoft.Research.AbstractIL.Internal 
open Microsoft.Research.AbstractIL.Extensions.ILX
module Il = Microsoft.Research.AbstractIL.IL 
module Ilx = Microsoft.Research.AbstractIL.Extensions.ILX.Types
module Ilmorph = Microsoft.Research.AbstractIL.Morphs 
module Ilprint = Microsoft.Research.AbstractIL.AsciiWriter 
module Illib = Microsoft.Research.AbstractIL.Internal.Library 
F#*)  

open Illib
open Nums
open Il
open Ilx
open Ilmorph
open List
open Ilprint
open Msilxlib
open Ilxsettings

let rec findi n f l = 
  match l with 
  | [] -> None
  | h::t -> if f h then Some (h,n) else findi (n+1) f t

type cenv  = { manager: Il.manager option; ilg: mscorlib_refs } 
let cga ilg = (if !Msilxlib.pp_implementation = NoGenerics then [] else [ mk_CompilerGeneratedAttribute ilg ])

let mk_bblock2 (a,b) = 
  mk_bblock { bblockLabel=a; bblockInstrs= Array.of_list b}

(* Settings *)
(* The default is to optimize a solitary nullary constructor to Null, *)
(* similarly a solitary non-nullary constructor to the class itself, *)
(* and to insert integer tags to help with switching *)
(* when there are [tagging_threshold] or more non-nullary constructors. *)
let prefer_discriminate_blocks_on_runtime_types = ref false
let prefer_discriminate_blocks_with_virttags = ref false

let opt_solitary_nullary = ref true
let opt_self = ref true
let tagging_threshold = ref 7

let casts () = true

let nullary a = alt_is_nullary a
let non_nullary a = not (alt_is_nullary a)

type summary = Summary of alternative array * int * bool

let optimizeAllAlternativesToConstantFieldsInRootClass (Summary (alts,gparams,_) as summary) = 
  !opt_solitary_nullary && (* shouldn't really need this - adding it for consistency with maintainPosiblyUniqueConstantFieldForAlternative below *)
  (Array.length alts > 1 && array_forall nullary alts) && 
  gparams = 0 (* can't use static fields for generic nullary constructors - though  Whidbey supports this, if we knew we were targetting generics... *)
let useRuntimeTypes (Summary(alts,gparams,_) as summary) = 
  (!prefer_discriminate_blocks_on_runtime_types || 
   (Array.length alts < !tagging_threshold)) &&
  (not (optimizeAllAlternativesToConstantFieldsInRootClass summary))
let useTags (Summary(alts,gparams,_) as summary) =  not (useRuntimeTypes summary)

let accessTagsByVirtualCall summary = 
  (not (optimizeAllAlternativesToConstantFieldsInRootClass summary)) &&
  !prefer_discriminate_blocks_with_virttags
let useTagsAccessedByVirtualCall summary = 
  useTags summary && accessTagsByVirtualCall summary
let useTagsAccessedByField summary = 
  useTags summary && not (accessTagsByVirtualCall summary)
  
(* WARNING: this must match isUnionThatUsesNullAsRepresentation in the F# compiler *) 
(* REVIEW: make this attribute controlled *)
let optimizeAlternativeToNull (Summary(alts,gparams,permitNull) as summary) alt = 
  !opt_solitary_nullary &&
  permitNull &&
  useRuntimeTypes summary && (* tags won't work if we're using "null" *)
  (* Array.length alts = 2 && *)
  array_exists_one nullary alts  &&
  array_exists (nullary >> not) alts  &&
  alt_is_nullary alt  (* is this the one? *)

let optimizingOneAlternativeToNull (Summary(alts,gparams,_) as summary) = 
  array_exists_one (optimizeAlternativeToNull summary) alts

let optimizeSingleNonNullaryAlternativeToRootClass (Summary(alts,gparams,_) as summary) alt = 
    ( !opt_self &&
     (* Check all nullary constructors are being represented without using sub-classes *)
     (array_forall (fun alt2 -> 
       not (nullary alt2) || 
       optimizeAlternativeToNull summary alt2) alts)  &&
     (* Check there is only one non-nullary constructor *)
     array_exists_one non_nullary alts  &&
     not (alt_is_nullary alt))

let optimizeAlternativeToConstantFieldInTaggedRootClass (Summary(alts,gparams,_) as summary) alt = 
    nullary alt &&
    gparams=0 &&
    not (optimizeAlternativeToNull summary alt)  &&
    useTags summary 

let optimizeAlternativeToRootClass (Summary(alts,gparams,_) as summary) alt = 
  optimizeAllAlternativesToConstantFieldsInRootClass summary ||
  optimizeAlternativeToConstantFieldInTaggedRootClass summary alt ||
  optimizeSingleNonNullaryAlternativeToRootClass summary alt
  
let maintainPosiblyUniqueConstantFieldForAlternative (Summary(alts,gparams,_) as summary) alt = 
  not (optimizeAlternativeToNull summary alt) &&
  !opt_solitary_nullary &&
  alt_is_nullary alt && 
  (match !(Msilxlib.pp_implementation) with 
   | Ilxsettings.FullGenerics -> true (* use Whidbey generic statics support *)
   | _ -> gparams = 0) (* can't use static fields for generic nullary constructors in erased code... *)

let optimizeAlternativeToUniqueConstant (Summary(alts,gparams,_) as summary) alt = 
(* Bug 255 (Optimizing nullary alternatives to unique constants doesn't work with dserialized data) *)
(* meant we couldn't optimize nullary alternatives to unique constants *)
(* However we still store a static field to avoid endlessly re-allocating *)
(* copies of the same constant. *)
  false && 
  maintainPosiblyUniqueConstantFieldForAlternative summary alt

let summary cuspec = Summary(altsarray_of_cuspec cuspec, List.length (inst_of_cuspec cuspec),nullPermitted_of_cuspec cuspec)

let getter_prop_name fields i = (List.nth fields i).fdName

let tester_name nm = "Is"^nm
let maker_name nm = nm
let tagprop_name = "Tag"

(* nb. not currently recording translation assumptions in the default external reference naming scheme *)
let tag_member_name = "tag"
let tag_member_ty ilg = ilg.typ_int32
let virt_tag_field_id ilg = tag_member_name,tag_member_ty ilg
let backend_field_id fdef = ("_"^fdef.fdName, fdef.fdType)
let backend_derived_tref tref nm = mk_tref_in_tref (tref,"_"^nm)

let ref_to_field_in_tspec tspec (nm,ty) = mk_fspec_in_boxed_tspec (tspec,nm,ty)

let const_fname nm = nm^"_uniq"
let const_fspec nm base_tspec = 
  let const_fid = (const_fname nm,
                   Type_boxed (generalize_tref (tref_of_tspec base_tspec) 
                                               (gparams_of_inst (inst_of_tspec base_tspec)))) in 
  ref_to_field_in_tspec base_tspec const_fid

let base_tspec_of_cuspec cenv cuspec = intern_tspec cenv.manager (mk_tspec (tref_of_cuspec cuspec, inst_of_cuspec cuspec))

let tspec_of_alt cenv cuspec alt =
    if optimizeAlternativeToRootClass (summary cuspec) alt then base_tspec_of_cuspec cenv cuspec
    else intern_tspec cenv.manager (mk_tspec(backend_derived_tref (tref_of_cuspec cuspec) (name_of_alt alt),inst_of_cuspec cuspec)) 
let altinfo_of_cuspec cenv cuspec cidx =
  let alt = 
    try Ilx.alt_of_cuspec cuspec cidx 
    with _ -> failwith ("alternative "^string_of_int cidx^" not found")  in 
  let tspec = tspec_of_alt cenv cuspec alt in
  name_of_alt alt,tspec,alt

let rtt_discriminate derived_tspec = 
  [ I_isinst (Type_boxed derived_tspec); I_arith AI_ldnull; I_arith AI_cgt_un ]

let ceq_then after = 
  match after with 
  | I_brcmp (BI_brfalse,a,b) -> [I_brcmp (BI_bne_un,a,b)]
  | I_brcmp (BI_brtrue,a,b) ->  [I_brcmp (BI_beq,a,b)]
  | _ -> [I_arith AI_ceq; after]

let rtt_discriminate_then derived_tspec after = 
  match after with 
  | I_brcmp (BI_brfalse,_,_) 
  | I_brcmp (BI_brtrue,_,_) -> 
      [ I_isinst (Type_boxed derived_tspec); after ]
  | _ -> rtt_discriminate derived_tspec @ [ after ]
let get_tag cenv cuspec base_tspec = 
  if  useTagsAccessedByVirtualCall (summary cuspec) then 
    let mspec = mk_nongeneric_instance_mspec_in_boxed_tspec (base_tspec,tag_member_name,[],tag_member_ty cenv.ilg) in 
    let mspec = intern_mspec cenv.manager mspec in
    [ mk_normal_callvirt mspec ]
  else
    [ mk_normal_ldfld (ref_to_field_in_tspec base_tspec (virt_tag_field_id cenv.ilg)) ]
let tag_discriminate cenv cuspec base_tspec cidx = 
  get_tag cenv cuspec base_tspec 
  @ [ mk_ldc_i32 (int_to_i32 cidx); 
      I_arith AI_ceq 
    ]
let tag_discriminate_then cenv cuspec base_tspec cidx after = 
  get_tag cenv cuspec base_tspec 
  @ [ mk_ldc_i32 (int_to_i32 cidx) ] 
  @ ceq_then after

let rec conv_instr cenv tmps inplab outlab instr = 
  match instr with 
  | I_other e when is_ilx_ext_instr e -> 
      begin match (dest_ilx_ext_instr e) with 
      |  (EI_newdata (cuspec, cidx)) ->
          let nm,derived_tspec,alt = altinfo_of_cuspec cenv cuspec cidx in 
          if optimizeAlternativeToNull (summary cuspec) alt then 
            Choice1of2 [ I_arith AI_ldnull ]
          else if maintainPosiblyUniqueConstantFieldForAlternative (summary cuspec) alt then 
            let base_tspec = base_tspec_of_cuspec cenv cuspec in 
            let const_fspec = const_fspec nm base_tspec in 
            Choice1of2 [ I_ldsfld (Nonvolatile,const_fspec) ]
          else if optimizeSingleNonNullaryAlternativeToRootClass (summary cuspec) alt then 
            let base_tspec = base_tspec_of_cuspec cenv cuspec in 
            Choice1of2 [ mk_normal_newobj(mk_ctor_mspec_for_boxed_tspec (base_tspec,map (fun fd -> fd.fdType) (Array.to_list (fdefs_of_alt alt)))) ]
          else 
            Choice1of2 [ mk_normal_newobj(mk_ctor_mspec_for_boxed_tspec (derived_tspec,map (fun fd -> fd.fdType)  (Array.to_list (fdefs_of_alt alt)))) ]

      |  (EI_stdata (cuspec, cidx,fidx)) ->
          let nm,derived_tspec,alt = altinfo_of_cuspec cenv cuspec cidx in 
          let field_id = backend_field_id (fdef_of_alt alt fidx) in 
          let field_spec = ref_to_field_in_tspec derived_tspec field_id in 
          Choice1of2 [ I_stfld (Aligned,Nonvolatile, field_spec) ] 
            
      |  (EI_lddata (cuspec,cidx,fidx)) ->
          let nm,derived_tspec,alt = altinfo_of_cuspec cenv cuspec cidx in 
          let field_id = 
            try backend_field_id (fdef_of_alt alt fidx) 
            with Failure _ -> failwith ("field "^string_of_int fidx ^"not found in alternative "^string_of_int cidx)  in 
          let field_spec = ref_to_field_in_tspec derived_tspec field_id in 
          Choice1of2 [ I_ldfld (Aligned,Nonvolatile, field_spec) ] 

      |  (EI_lddatatag cuspec) ->
          let alts = alts_of_cuspec cuspec in 
          if  useTags (summary cuspec) then 
            let base_tspec = base_tspec_of_cuspec cenv cuspec in 
            Choice1of2 (get_tag cenv cuspec base_tspec)
          else if length alts = 1 then 
            Choice1of2 [ I_arith AI_pop; I_arith (AI_ldc(DT_I4,NUM_I4(int_to_i32 0))) ] 
          else 
            let base_tspec = base_tspec_of_cuspec cenv cuspec in 
            let locn = alloc_tmp tmps (mk_local (mk_typ AsObject base_tspec)) in
            let mk_case last inplab cidx fail_lab = 
              let nm,derived_tspec,alt = altinfo_of_cuspec cenv cuspec cidx in 
              let internal_lab = generate_code_label () in 
              let cmp_null = optimizeAlternativeToNull (summary cuspec) alt in 
              if last then 
                mk_bblock2 (inplab,[ I_arith (AI_ldc(DT_I4,NUM_I4(int_to_i32 cidx))); 
                                    I_br outlab ])   
              else 
                let test = I_brcmp ((if cmp_null then BI_brtrue else BI_brfalse),fail_lab,internal_lab) in 
                let test_block = 
                  if cmp_null || optimizeSingleNonNullaryAlternativeToRootClass (summary cuspec) alt then 
                    [ test ]
                  else if optimizeAlternativeToUniqueConstant (summary cuspec) alt then 
                    let const_fspec = const_fspec nm base_tspec in 
                    [ I_ldsfld (Nonvolatile,const_fspec) ]  @
                    ceq_then test
                  else if useRuntimeTypes (summary cuspec) then 
                    rtt_discriminate_then derived_tspec test
                  else 
                    tag_discriminate_then cenv cuspec base_tspec cidx test  in 
                mk_group_block 
                  ([internal_lab],
                  [ mk_bblock2 (inplab, I_ldloc locn ::test_block);
                    mk_bblock2 (internal_lab,[I_arith (AI_ldc(DT_I4,NUM_I4(int_to_i32 cidx))); I_br outlab ]) ])  in
      (* Make the block for the last test. *)
            let last_inplab = generate_code_label () in 
            let last_block = mk_case true last_inplab 0 outlab in 
      (* Make the blocks for the remaining tests. *)
            let _,first_inplab,overall_block = 
              List.fold_right
                (fun _ (n,continue_inplab, continue_block) -> 
                  let new_inplab = generate_code_label () in 
                  n+1,
                  new_inplab,
                  mk_group_block 
                    ([continue_inplab],
                    [ mk_case false new_inplab n continue_inplab;
                      continue_block ]))
                (tl alts)
                (1,last_inplab, last_block) in
      (* Chuck on a branch to the first input label.  This gets optimized *)
      (* away by the printer/emitter. *)
            Choice2of2 
              (mk_group_block
                 ([first_inplab],
                 [ mk_bblock2 (inplab, [ I_stloc locn; I_br first_inplab ]);
               overall_block ]))
              
      |  (EI_castdata (canfail,cuspec,cidx)) ->
          let nm,derived_tspec,alt = altinfo_of_cuspec cenv cuspec cidx in 
          if optimizeAlternativeToNull (summary cuspec) alt then 
            if canfail then 
              let base_tspec = base_tspec_of_cuspec cenv cuspec in 
              let internal1 = generate_code_label () in 
              Choice2of2 
                (mk_group_block  
                   ([internal1],
                   [ mk_bblock2 (inplab,
                                [ I_arith AI_dup;
                                  I_brcmp (BI_brfalse,outlab, internal1) ]);
                     mk_bblock2 (internal1,
                                [ mk_mscorlib_exn_newobj cenv.ilg "System.InvalidCastException";
                                  I_throw ]);
                   ] ))
            else Choice1of2 [] (* If it can't fail, it's still verifiable just to leave the value on the stack unchecked *)
                
          else if optimizeSingleNonNullaryAlternativeToRootClass (summary cuspec) alt then 
            Choice1of2 []
          else if optimizeAlternativeToUniqueConstant (summary cuspec) alt then 
            if canfail then 
              let base_tspec = base_tspec_of_cuspec cenv cuspec in 
              let const_fspec = const_fspec nm base_tspec in 
              let internal1 = generate_code_label () in 
              Choice2of2 
                (mk_group_block  
                   ([internal1],
                   [ mk_bblock2 (inplab,
                                [ I_arith AI_dup;
                                  I_ldsfld (Nonvolatile,const_fspec); 
                                  I_brcmp (BI_beq,outlab, internal1) ]);
                     mk_bblock2 (internal1,
                                [ mk_mscorlib_exn_newobj cenv.ilg "System.InvalidCastException";
                                  I_throw ]);
                   ] ))
            else Choice1of2 [] (* If it can't fail, it's still verifiable just to leave the value on the stack unchecked *)
          else 
            if canfail || casts() then Choice1of2 [ I_castclass (Type_boxed derived_tspec) ] 
            else  Choice1of2 [] (* In this case we must check if generating verifiable code *)
            
      |  (EI_brisdata (cuspec,cidx,tg,fail_lab)) ->
          let nm,derived_tspec,alt = altinfo_of_cuspec cenv cuspec cidx in 
          if optimizeAlternativeToNull (summary cuspec) alt then 
            Choice1of2 [ I_brcmp (BI_brtrue,fail_lab,tg) ] 
          else if optimizeSingleNonNullaryAlternativeToRootClass (summary cuspec) alt then 
            Choice1of2 [ I_brcmp (BI_brfalse,fail_lab,tg) ] 
          else if optimizeAlternativeToUniqueConstant (summary cuspec) alt then 
            let base_tspec = base_tspec_of_cuspec cenv cuspec in 
            let const_fspec = const_fspec nm base_tspec in 
            Choice1of2 [ I_ldsfld (Nonvolatile,const_fspec); 
                         I_brcmp (BI_bne_un,fail_lab,tg) ] 
          else
            if useRuntimeTypes (summary cuspec) then 
              Choice1of2 (rtt_discriminate_then derived_tspec (I_brcmp (BI_brfalse,fail_lab,tg)))
            else 
              let base_tspec = base_tspec_of_cuspec cenv cuspec in 
              Choice1of2 (tag_discriminate_then cenv cuspec base_tspec cidx (I_brcmp (BI_brfalse,fail_lab,tg)))

      |  (EI_isdata (cuspec,cidx)) ->
          let nm,derived_tspec,alt = altinfo_of_cuspec cenv cuspec cidx in 
          if optimizeAlternativeToNull (summary cuspec) alt then 
            Choice1of2 [ I_arith AI_ldnull; I_arith AI_ceq ] 
          else if optimizeSingleNonNullaryAlternativeToRootClass (summary cuspec) alt then 
            Choice1of2 [ I_arith AI_ldnull; I_arith AI_cgt_un ] 
          else if optimizeAlternativeToUniqueConstant (summary cuspec) alt then 
            let base_tspec = base_tspec_of_cuspec cenv cuspec in 
            let const_fspec = const_fspec nm base_tspec in 
            Choice1of2 [ I_ldsfld (Nonvolatile,const_fspec); I_arith AI_ceq ] 
          else
            if useRuntimeTypes (summary cuspec) then 
              Choice1of2 (rtt_discriminate derived_tspec)
            else 
              let base_tspec = base_tspec_of_cuspec cenv cuspec in 
              Choice1of2 (tag_discriminate cenv cuspec base_tspec cidx)
            
      |  (EI_datacase (leave_on_stack,cuspec,cases,cont)) ->
          let base_tspec = base_tspec_of_cuspec cenv cuspec in 
      (* REVIEW: tag discriminate should map to a switch even for verifiable code *)
      (* when we leave the result on the stack - this will need more casts inserted *)
          if (casts() && leave_on_stack) || useRuntimeTypes (summary cuspec) then 
            let locn = alloc_tmp tmps (mk_local (mk_typ AsObject base_tspec)) in
            let mk_case last inplab (cidx,tg) fail_lab = 
              let nm,derived_tspec,alt = altinfo_of_cuspec cenv cuspec cidx in 
              let internal_lab = generate_code_label () in 
              let cmp_null = optimizeAlternativeToNull (summary cuspec) alt in 
              let use_one_block = not leave_on_stack in 
              let test = 
                let test_instr = 
                  if use_one_block then I_brcmp ((if cmp_null then BI_brfalse else BI_brtrue),tg,fail_lab) 
                  else I_brcmp ((if cmp_null then BI_brtrue else BI_brfalse),fail_lab,internal_lab) in
                [ I_ldloc locn ] @
                begin
                  if cmp_null || optimizeSingleNonNullaryAlternativeToRootClass (summary cuspec) alt then 
                    [ test_instr ]
                      
                  else if optimizeAlternativeToUniqueConstant (summary cuspec) alt then 
                    let const_fspec = const_fspec nm base_tspec in 
                    [ I_ldsfld (Nonvolatile,const_fspec) ]  @
                    ceq_then test_instr
                  else 
                    begin
                      if useRuntimeTypes (summary cuspec) then 
                        rtt_discriminate_then derived_tspec test_instr
                      else 
                        tag_discriminate_then cenv cuspec base_tspec cidx test_instr 
                    end 
                end in 
              if use_one_block then mk_bblock2 (inplab, test) 
              else
                mk_group_block 
                  ([internal_lab],
                  [ mk_bblock2 (inplab, test);
                    mk_bblock2 
                      (internal_lab,
                       (if leave_on_stack then 
                         if not cmp_null && casts () then 
                           [ I_ldloc locn; I_castclass (Type_boxed derived_tspec) ]
                         else [ I_ldloc locn ]
                       else []) @
                       [ I_br tg ]) ]) in
            
      (* Make the block for the last test. *)
            let last_inplab = generate_code_label () in 
            let last_case,first_cases = 
              let l2 = List.rev cases in List.hd l2, List.rev (List.tl l2) in 
            
            let last_block = mk_case true last_inplab last_case cont in 
      (* Make the blocks for the remaining tests. *)
            let first_inplab,overall_block = 
              List.fold_right
                (fun case_info (continue_inplab, continue_block) -> 
                  let new_inplab = generate_code_label () in 
                  new_inplab,
                  mk_group_block 
                    ([continue_inplab],
                    [ mk_case false new_inplab case_info continue_inplab;
                      continue_block ]))
                first_cases
                (last_inplab, last_block) in
      (* Chuck on a branch to the first input label.  This gets optimized *)
      (* away by the printer/emitter. *)
            Choice2of2 
              (mk_group_block
                 ([first_inplab],
                 [ mk_bblock2 (inplab, [ I_stloc locn; I_br first_inplab ]);
                   overall_block ]))
          else 	
            let mk_case i _ = 
              if mem_assoc (i) cases then assoc (i) cases else cont in 
            let dests = list_mapi mk_case (alts_of_cuspec cuspec) in 
            Choice1of2 ((if leave_on_stack then [I_arith AI_dup] else []) @
                        get_tag cenv cuspec base_tspec @ 
                        [ I_switch (dests,cont) ])
              
      | _ -> Choice1of2 [instr] 
      end

  | _ -> Choice1of2 [instr] 


let conv_ilmbody cenv il = 
  let tmps = new_tmps (List.length il.ilLocals) in  
  let code= topcode_instr2code (conv_instr cenv tmps) il.ilCode in 
  {il with
        ilLocals = il.ilLocals @ get_tmps tmps;
        ilCode=code; 
        ilMaxStack=incr_i32 (incr_i32 il.ilMaxStack) }

let conv_mdef cenv md  =
  {md with mdBody= mbody_ilmbody2ilmbody (conv_ilmbody cenv) md.mdBody }

let conv_alternative_def cenv num td cud info cuspec base_tspec alt =
  let attr = cud.cudWhere in 
  let nm = name_of_alt alt in
  let fields = Array.to_list (fdefs_of_alt alt) in
  let alt_tspec = tspec_of_alt cenv cuspec alt in
  let auxm,auxp = 
      if cud.cudHelpers then 
        let maker_meths,maker_props = 
          if alt_is_nullary alt then 
            let get_nullary_method = 
              let mdef = 
               conv_mdef cenv
                (mk_static_nongeneric_mdef
                   ("get_"^maker_name nm,
                    cud.cudHelpersAccess,[],mk_return(Type_boxed base_tspec),
                    mk_impl(true,[],length fields,
                            nonbranching_instrs_to_code 
                              (list_mapi (fun i _ -> I_ldarg (int_to_u16 i)) fields @
                               [ (mk_IlxInstr (EI_newdata (cuspec,  num)))]), attr))) in 
              { mdef with mdCustomAttrs=alt.altCustomAttrs } in
            let nullary_prop = 
              { propName=maker_name nm;
                propRTSpecialName=false;
                propSpecialName=false;
                propSet=None;
                propGet=Some(mk_mref(tref_of_tspec base_tspec,static_callconv,"get_"^maker_name nm,0,[],Type_boxed base_tspec));
                propCallconv=CC_static;
                propType=Type_boxed base_tspec;          
                propInit=None;
                propArgs=[];
                propCustomAttrs=mk_custom_attrs []; } in 
            [get_nullary_method],[nullary_prop]
            
          else
            let mdef = 
              conv_mdef cenv
                (mk_static_nongeneric_mdef
                   (maker_name nm,
                    cud.cudHelpersAccess,
                    map (fun fd -> mk_named_param (fd.fdName, fd.fdType)) fields,
                    mk_return(Type_boxed base_tspec),
                    mk_impl(true,[],length fields,
                            nonbranching_instrs_to_code 
                              (list_mapi (fun i _ -> I_ldarg (int_to_u16 i)) fields @
                               [ (mk_IlxInstr (EI_newdata (cuspec,  num)))]), attr))) in 
            (* REVIEW: rethink how and where we can attach custom attributes to discriminated unions  *)
            let mdef = { mdef with mdCustomAttrs=alt.altCustomAttrs } in
            [mdef],[] in
        let tester_meths = 
          if Array.length cud.cudAlternatives <= 1 then [] 
          else if optimizingOneAlternativeToNull info then []
          else
            [ conv_mdef cenv
                (mk_instance_mdef
                   (tester_name nm,
                    cud.cudHelpersAccess,[],
                    mk_return cenv.ilg.typ_bool,
                    mk_impl(true,[],2,nonbranching_instrs_to_code 
                              [ ldarg_0;
                                (mk_IlxInstr (EI_isdata (cuspec, num))) ], attr))) ] in 
        let getter_meths,getter_props  = 
          list_mapi
            (fun i fdef -> 
              conv_mdef cenv
                (mk_instance_mdef
                   ("get_"^getter_prop_name fields i,
                    cud.cudHelpersAccess,[],
                    mk_return fdef.fdType,
                    mk_impl(true,[],2,
                            nonbranching_instrs_to_code 
                              [ ldarg_0;
                                (mk_IlxInstr (EI_castdata (true,cuspec, num)));
                                (mk_IlxInstr (EI_lddata (cuspec, num, i)))],attr))))
            fields,
          list_mapi
            (fun i fdef -> 
              { propName=getter_prop_name fields i;
                propRTSpecialName=false;
                propSpecialName=false;
                propSet=None;
                propGet=Some(mk_mref(tref_of_tspec base_tspec,instance_callconv,"get_"^getter_prop_name fields i,0,[],fdef.fdType));
                propCallconv=CC_instance;
                propType=fdef.fdType;          
                propInit=None;
                propArgs=[];
                propCustomAttrs= fdef.fdCustomAttrs; })
            fields in 
        tester_meths @ getter_meths @ maker_meths, getter_props @ maker_props
      else 
        [],[]  in 
  let tdefs,nullary_fields = 
    if optimizeAlternativeToNull info  alt then [], []
    else if optimizeSingleNonNullaryAlternativeToRootClass info alt then [], []
    else 
      let nullary_fields = 
        if maintainPosiblyUniqueConstantFieldForAlternative info alt then 
          let basic = mk_static_fdef (const_fname nm, Type_boxed base_tspec, None, None, cud.cudReprAccess) in 
          let const_fdef = { basic with fdInitOnly=true; fdCustomAttrs = mk_custom_attrs (cga cenv.ilg) } in
          let inRootClass = optimizeAlternativeToRootClass (summary cuspec) alt in
    
          [ (info,alt, alt_tspec,num,const_fdef,inRootClass) ] 
        else [] in 
      let tdefs = 
        if optimizeAlternativeToUniqueConstant info alt then []
        else if optimizeAllAlternativesToConstantFieldsInRootClass info then []
        else if optimizeAlternativeToConstantFieldInTaggedRootClass info alt then []
        else
          
          let field_defs = 
            map 
              (fun field -> 
                let nm,ty = backend_field_id field in 
                let fdef = mk_instance_fdef (nm,ty, None, cud.cudReprAccess) in 
                { fdef with fdName=nm; fdType=ty; fdCustomAttrs=mk_custom_attrs (cga cenv.ilg) }) 
              fields in 
          
          let virttag_methods = 
            if useTagsAccessedByVirtualCall info then 
              [ mk_virtual_mdef
                  (tag_member_name,MemAccess_public,None,[],
                  mk_return (tag_member_ty cenv.ilg),
                   mk_impl (true,[],1,nonbranching_instrs_to_code [ I_arith (AI_ldc(DT_I4,NUM_I4(int_to_i32 num)));  ],attr)) ]
            else [] in 
          
          let ctor_spec = 
            mk_storage_ctor attr  
              begin 
                [ ldarg_0 ] @
                begin 
                  if  useTagsAccessedByField info then 
                    [ I_arith (AI_ldc(DT_I4,NUM_I4(int_to_i32 num)));
                      mk_normal_call (mk_ctor_mspec_for_boxed_tspec (base_tspec,[cenv.ilg.typ_int32])) ]
                  else 
                    [ mk_normal_call (mk_ctor_mspec_for_boxed_tspec (base_tspec,[])) ]
                end
              end
              alt_tspec
              (map (fun fdef -> fdef.fdName, fdef.fdType) field_defs)  in
          let derived_cdef = 
            let basic_cdef = 
              mk_generic_class (tname_of_tspec alt_tspec, 
                                TypeAccess_nested cud.cudReprAccess, 
                                td.tdGenericParams, 
                                Type_boxed base_tspec, [], 
                                mk_mdefs (virttag_methods @ [ctor_spec]), 
                                mk_fdefs field_defs,
                                mk_properties [],
                                mk_events [],
                                mk_custom_attrs []) in 
            { basic_cdef with tdSerializable=td.tdSerializable } in 
          [ derived_cdef  ] in 
      tdefs,nullary_fields in
  auxm,auxp,tdefs,nullary_fields
        
  
let rec conv_cudef cenv enc td cud = 
  let base_tspec = tspec_for_nested_tdef ScopeRef_local (enc,td) in 
  let cuspec = ClassunionSpec(ClassunionRef(tref_of_tspec base_tspec,cud.cudAlternatives,cud.cudNullPermitted), inst_of_tspec base_tspec) in
  let info = Summary(cud.cudAlternatives, (List.length td.tdGenericParams), cud.cudNullPermitted) in


  let _,aux_meths,aux_props,derived_classes,nullary_fields = 
    Array.fold_left 
      (fun (num,msofar,psofar,csofar,fsofar) alt -> 
        let ms,ps,cls,flds = conv_alternative_def cenv num td cud info cuspec base_tspec alt in 
        (num+1,msofar@ms, psofar@ps,csofar@cls,fsofar@flds)) 
      (0,[],[],[],[])
      cud.cudAlternatives in 
     
  let self_fields,self_ctor_specs,self_num = 
    match  findi 0 (optimizeSingleNonNullaryAlternativeToRootClass info) (Array.to_list cud.cudAlternatives) with 
    | Some (alt,alt_num) ->
        let fields = map backend_field_id (Array.to_list (fdefs_of_alt alt)) in 
        let ctor = 
          mk_simple_storage_ctor 
            cud.cudWhere
            (match td.tdExtends with None -> Some cenv.ilg.tspec_Object | Some typ -> Some (tspec_of_typ typ))
            base_tspec 
            fields in 
        fields,[ctor],alt_num
    |  None ->
        [],[],0 in 
  let virttag_fields = 
    if useTagsAccessedByField info then  [ virt_tag_field_id cenv.ilg ] 
    else [] in 

  let self_and_virttag_fields = 
      (self_fields @ virttag_fields) 
      |> map (fun (nm,ty)-> 
          let fdef = mk_instance_fdef (nm,ty, None, cud.cudReprAccess) in 
          { fdef with fdCustomAttrs = mk_custom_attrs (cga cenv.ilg) })  in 

  let virttag_methods = 
    if useTagsAccessedByVirtualCall info then 
      [ mk_virtual_mdef
          (tag_member_name,MemAccess_public,None,[],
           mk_return (tag_member_ty cenv.ilg),
           mk_impl (true,[],1,nonbranching_instrs_to_code [ I_arith (AI_ldc(DT_I4,NUM_I4(int_to_i32 self_num)));  ], cud.cudWhere))]
    else [] in 

  let subclass_and_nullary_ctor_specs =
    if length self_fields = 0 && length virttag_fields = 0 && length self_ctor_specs <> 0 then 
      [] (* no need to duplicate the ctor in this case *)
    else 
      [mk_simple_storage_ctor 
          cud.cudWhere
          (match td.tdExtends with None -> Some cenv.ilg.tspec_Object | Some typ -> Some (tspec_of_typ typ))
          base_tspec 
          virttag_fields] in 

  (* The following two are for the case where we're using virtual tags, and we *)
  (* need to generate a class for the nullary constructors we're optimizing to *)
  (* constant values.  These constants each carry a field giving their appropriate tag. *)
  (* i.e. we don't really use virtual dispatch to save space on these. *)
  let nullary_tspec = 
    if useTagsAccessedByVirtualCall info  && nullary_fields <> [] then 
      generalize_tref (backend_derived_tref (tref_of_tspec base_tspec) "_Simple") td.tdGenericParams
    else base_tspec in 
  let nullary_classes = 
    if useTagsAccessedByVirtualCall info && nullary_fields <> [] then 
      (* In this case the tag method just loads the sotred tag *)
      let virttag_methods = 
        [ mk_virtual_mdef
            (tag_member_name,MemAccess_public,None,[],
             mk_return (tag_member_ty cenv.ilg),
             mk_impl (true,[],1,nonbranching_instrs_to_code  
                        [ ldarg_0;
                          mk_normal_ldfld (ref_to_field_in_tspec nullary_tspec (virt_tag_field_id cenv.ilg)) ],cud.cudWhere))] in
      
      let ctor_spec = 
        mk_storage_ctor cud.cudWhere  
          [ ldarg_0;
            mk_normal_call (mk_ctor_mspec_for_boxed_tspec (base_tspec,[])) ]
          nullary_tspec
          [virt_tag_field_id cenv.ilg]  in
      
      let nullary_cdef = 
        let basic_cdef = 
          mk_generic_class (tname_of_tref (tref_of_tspec nullary_tspec), TypeAccess_nested cud.cudReprAccess, td.tdGenericParams, 
                            Type_boxed base_tspec, [], 
                            mk_mdefs (virttag_methods @ [ctor_spec]),
                            mk_fdefs [(fun (nm,ty)-> mk_instance_fdef (nm,ty, None, MemAccess_private)) (virt_tag_field_id cenv.ilg)],
                            mk_properties [],
                            mk_events [],
                            mk_custom_attrs []) in 
        { basic_cdef with tdSerializable=td.tdSerializable } in 
      [  nullary_cdef  ]
    else [] in 
  
  (* Now initialize the constant fields wherever they are stored... *)
  let add_const_field_init cd = 
    if nullary_fields = [] then cd else 
     prepend_instrs_to_cctor 
      ((map_concat
            (fun (info,alt,alt_tspec,fidx,fd,inRootClass) -> 
              let const_fid = (fd.fdName,Type_boxed base_tspec) in 
              let const_fspec = ref_to_field_in_tspec base_tspec const_fid in 

              begin 
                if useRuntimeTypes info then 
                  [ mk_normal_newobj (mk_ctor_mspec_for_boxed_tspec (alt_tspec,[])); ] 
                else if inRootClass then
                  [ I_arith (AI_ldc(DT_I4,NUM_I4(int_to_i32 fidx)));  
                    mk_normal_newobj (mk_ctor_mspec_for_boxed_tspec (alt_tspec,[cenv.ilg.typ_int32] )); ] 
                else
                  [ mk_normal_newobj (mk_ctor_mspec_for_boxed_tspec (alt_tspec,[])); ] 
              end 
              @
              [  mk_normal_stsfld const_fspec; ])
            nullary_fields)) 
      cud.cudWhere
      cd in 

  let tag_meths, tag_props, tag_fields = 
      let tag_fields = 
        Array.to_list
         (Array.mapi
          (fun num alt ->
            let fdef = mk_static_fdef("tag_"^name_of_alt alt,cenv.ilg.typ_int32,Some(FieldInit_int32(int_to_i32 num)),None,cud.cudHelpersAccess) in 
            {fdef with fdLiteral = true; fdCustomAttrs=mk_custom_attrs (cga cenv.ilg) })
          cud.cudAlternatives) in 
      let get_tag_methods,tag_props = 
        (* If we are using NULL as a representation for an element of this type then we cannot use an instance method *)
        if (optimizingOneAlternativeToNull info) then
          [ conv_mdef cenv
            (mk_static_nongeneric_mdef
               ("Get"^tagprop_name,
                cud.cudHelpersAccess,
                [mk_unnamed_param (Type_boxed base_tspec)],
                mk_return (cenv.ilg.typ_int32),
                mk_impl(true,[],2,nonbranching_instrs_to_code 
                          [ ldarg_0;
                            (mk_IlxInstr (EI_lddatatag cuspec)) ], cud.cudWhere))) ], 
          [] 
        else
          [ conv_mdef cenv
            (mk_instance_mdef
               ("get_"^tagprop_name,
                cud.cudHelpersAccess,[],
                mk_return cenv.ilg.typ_int32,
                mk_impl(true,[],2,nonbranching_instrs_to_code 
                          [ ldarg_0;
                            (mk_IlxInstr (EI_lddatatag cuspec)) ], cud.cudWhere))) ], 
          [ { propName=tagprop_name;
              propRTSpecialName=false;
              propSpecialName=false;
              propSet=None;
              propGet=Some(mk_mref(tref_of_tspec base_tspec,instance_callconv,"get_"^tagprop_name,0,[],cenv.ilg.typ_int32));
              propCallconv=CC_instance;
              propType=cenv.ilg.typ_int32;          
              propInit=None;
              propArgs=[];
              propCustomAttrs=mk_custom_attrs []; } ] in 
      get_tag_methods,tag_props,tag_fields in
      
  let base = 
    { tdName = td.tdName;
      tdNested = mk_tdefs (nullary_classes @ derived_classes @ dest_tdefs (conv_tdefs cenv (enc@[td]) td.tdNested));
      tdGenericParams= td.tdGenericParams;
      tdAccess = td.tdAccess;
      tdAbstract = false;
      tdSealed = false;
      tdSerializable=td.tdSerializable;
      tdComInterop=false;
      tdLayout=td.tdLayout; 
      tdSpecialName=td.tdSpecialName;
      tdEncoding=td.tdEncoding ;
      tdImplements = td.tdImplements;
      tdExtends= (match td.tdExtends with None -> Some cenv.ilg.typ_Object | _ -> td.tdExtends) ;
      tdMethodDefs= mk_mdefs (tag_meths @ virttag_methods @ subclass_and_nullary_ctor_specs @ self_ctor_specs @ aux_meths @ map (conv_mdef cenv) (dest_mdefs td.tdMethodDefs));
      tdSecurityDecls=td.tdSecurityDecls;
      
      tdHasSecurity=td.tdHasSecurity; 
      tdFieldDefs=mk_fdefs (List.map (fun (_,_,_,_,fdef,_) -> fdef) nullary_fields @ self_and_virttag_fields @ tag_fields @ dest_fdefs td.tdFieldDefs);
      tdMethodImpls=td.tdMethodImpls;
      tdInitSemantics=TypeInit_beforefield;
      tdEvents=td.tdEvents;
      tdProperties=mk_properties (tag_props @ aux_props @ dest_pdefs td.tdProperties);
      tdCustomAttrs=td.tdCustomAttrs;
      tdKind = TypeDef_class; } in 
  let base' = add_const_field_init base in 
  base'


and conv_tdef cenv enc td = 
  match td.tdKind with 
  | TypeDef_other e when is_ilx_ext_type_def_kind e -> 
      begin match dest_ilx_ext_type_def_kind e with 
      | ETypeDef_closure cloinfo -> 
          {td with tdNested = conv_tdefs cenv (enc@[td]) td.tdNested;
                   tdMethodDefs=mdefs_mdef2mdef (conv_mdef cenv) td.tdMethodDefs;
                   tdKind= mk_ilx_type_def_kind(ETypeDef_closure (cloinfo_ilmbody2ilmbody (conv_ilmbody cenv) cloinfo)) }
      | ETypeDef_classunion cud -> conv_cudef cenv enc td cud
      end
  | _ -> 
    {td with tdNested = conv_tdefs cenv (enc@[td]) td.tdNested;
             tdMethodDefs=mdefs_mdef2mdef (conv_mdef cenv) td.tdMethodDefs; }

and conv_tdefs cenv enc tdefs = 
  tdefs_tdef2tdef (conv_tdef cenv enc) tdefs

let conv_module ilg m modul = 
  let cenv = { ilg=ilg; manager = m } in 
  module_tdefs2tdefs (conv_tdefs cenv []) modul


       
