return {
  callbacks = {
    {
      name = "conf",
      tag = "callbacks",
      summary = "Called to read configuration settings at startup.",
      description = "The `lovr.conf` callback lets you configure default settings for LÖVR.  It is called once right before the game starts.  Make sure you put `lovr.conf` in a file called `conf.lua`, a special file that's loaded before the rest of the framework initializes.",
      key = "lovr.conf",
      module = "lovr",
      examples = {
        {
          description = "A noop conf.lua that sets all configuration settings to their defaults:",
          code = "function lovr.conf(t)\n\n  -- Set the project version and identity\n  t.version = '0.16.0'\n  t.identity = 'default'\n\n  -- Set save directory precedence\n  t.saveprecedence = true\n\n  -- Enable or disable different modules\n  t.modules.audio = true\n  t.modules.data = true\n  t.modules.event = true\n  t.modules.graphics = true\n  t.modules.headset = true\n  t.modules.math = true\n  t.modules.physics = true\n  t.modules.system = true\n  t.modules.thread = true\n  t.modules.timer = true\n\n  -- Audio\n  t.audio.spatializer = nil\n  t.audio.samplerate = 48000\n  t.audio.start = true\n\n  -- Graphics\n  t.graphics.debug = false\n  t.graphics.vsync = true\n  t.graphics.stencil = false\n  t.graphics.antialias = true\n  t.graphics.shadercache = true\n\n  -- Headset settings\n  t.headset.drivers = { 'openxr', 'desktop' }\n  t.headset.supersample = false\n  t.headset.seated = false\n  t.headset.antialias = true\n  t.headset.stencil = false\n  t.headset.submitdepth = true\n  t.headset.overlay = false\n\n  -- Math settings\n  t.math.globals = true\n\n  -- Configure the desktop window\n  t.window.width = 1080\n  t.window.height = 600\n  t.window.fullscreen = false\n  t.window.resizable = false\n  t.window.title = 'LÖVR'\n  t.window.icon = nil\nend"
        }
      },
      notes = "Disabling unused modules can improve startup time.\n\n`t.window` can be set to nil to avoid creating the window.  The window can later be opened manually using `lovr.system.openWindow`.\n\nEnabling the `t.graphics.debug` flag will add additional error checks and will send messages from the GPU driver to the `lovr.log` callback.  This will decrease performance but can help provide information on performance problems or other bugs.  It will also cause `lovr.graphics.newShader` to embed debugging information in shaders which allows inspecting variables and stepping through shaders line-by-line in tools like RenderDoc.\n\n`t.graphics.debug` can also be enabled using the `--graphics-debug` command line option.\n\nThe `headset.offset` field is a vertical offset applied to the scene for headsets that do not center their tracking origin on the floor.  This can be thought of as a \"default user height\". Setting this offset makes it easier to design experiences that work in both seated and standing VR configurations.",
      related = {
        "lovr.load"
      },
      variants = {
        {
          arguments = {
            {
              name = "t",
              type = "table",
              description = "The table to edit the configuration settings on.",
              table = {
                {
                  name = "version",
                  type = "string",
                  description = "The version of LÖVR this project targets (not used yet)."
                },
                {
                  name = "identity",
                  type = "string",
                  description = "A unique label for this project."
                },
                {
                  name = "saveprecedence",
                  type = "boolean",
                  description = "Whether the files in the save directory should have precedence over files in the source archive."
                },
                {
                  name = "modules",
                  type = "table",
                  description = "The set of enabled modules to use.",
                  table = {
                    {
                      name = "audio",
                      type = "boolean",
                      description = "Whether the audio module should be enabled."
                    },
                    {
                      name = "data",
                      type = "boolean",
                      description = "Whether the data module should be enabled."
                    },
                    {
                      name = "event",
                      type = "boolean",
                      description = "Whether the event module should be enabled."
                    },
                    {
                      name = "graphics",
                      type = "boolean",
                      description = "Whether the graphics module should be enabled."
                    },
                    {
                      name = "headset",
                      type = "boolean",
                      description = "Whether the headset module should be enabled."
                    },
                    {
                      name = "math",
                      type = "boolean",
                      description = "Whether the math module should be enabled."
                    },
                    {
                      name = "physics",
                      type = "boolean",
                      description = "Whether the physics module should be enabled."
                    },
                    {
                      name = "system",
                      type = "boolean",
                      description = "Whether the system module should be enabled."
                    },
                    {
                      name = "thread",
                      type = "boolean",
                      description = "Whether the thread module should be enabled."
                    },
                    {
                      name = "timer",
                      type = "boolean",
                      description = "Whether the timer module should be enabled."
                    }
                  }
                },
                {
                  name = "audio",
                  type = "table",
                  description = "Configuration for the audio module.",
                  table = {
                    {
                      name = "spatializer",
                      type = "string",
                      description = "An audio spatializer to use (`simple`, `oculus`, or `phonon`).  If `nil`, all of them are attempted."
                    },
                    {
                      name = "samplerate",
                      type = "number",
                      description = "The sample rate to use for audio playback."
                    },
                    {
                      name = "start",
                      type = "boolean",
                      description = "Whether the playback device should be automatically started."
                    }
                  }
                },
                {
                  name = "graphics",
                  type = "table",
                  description = "Configuration for the graphics module.",
                  table = {
                    {
                      name = "debug",
                      type = "boolean",
                      description = "Whether debug messages from the GPU should get sent to lovr.log."
                    },
                    {
                      name = "vsync",
                      type = "boolean",
                      description = "Whether vsync is enabled (forced off when VR is active)."
                    },
                    {
                      name = "stencil",
                      type = "boolean",
                      description = "Whether the desktop window should have a stencil buffer."
                    },
                    {
                      name = "antialias",
                      type = "boolean",
                      description = "Whether the desktop window rendering should be antialiased."
                    },
                    {
                      name = "shadercache",
                      type = "boolean",
                      description = "Whether the shader cache should be loaded and saved to disk."
                    }
                  }
                },
                {
                  name = "headset",
                  type = "table",
                  description = "Configuration for the headset.",
                  table = {
                    {
                      name = "drivers",
                      type = "table",
                      description = "An ordered list of preferred headset drivers."
                    },
                    {
                      name = "supersample",
                      type = "number",
                      description = "A scaling factor to apply to the headset texture.  Improves visual quality but reduces performance.  Can also be a boolean."
                    },
                    {
                      name = "seated",
                      type = "boolean",
                      description = "Whether seated mode should be used instead of standing."
                    },
                    {
                      name = "antialias",
                      type = "boolean",
                      description = "Whether headset rendering should be antialiased."
                    },
                    {
                      name = "stencil",
                      type = "boolean",
                      description = "Whether headset rendering should have a stencil buffer."
                    },
                    {
                      name = "submitdepth",
                      type = "boolean",
                      description = "Whether the depth buffer should be sent to the VR runtime (improves reprojection)."
                    },
                    {
                      name = "overlay",
                      type = "boolean",
                      description = "Whether the project should run as an overlay.  Can also be a number to control sort order against other overlays (default is zero, higher numbers go on top)."
                    }
                  }
                },
                {
                  name = "math",
                  type = "table",
                  description = "Configuration for the math module.",
                  table = {
                    {
                      name = "globals",
                      type = "boolean",
                      description = "Whether vector object functions should be added to the global scope."
                    }
                  }
                },
                {
                  name = "window",
                  type = "table",
                  description = "Configuration for the window.",
                  table = {
                    {
                      name = "width",
                      type = "number",
                      description = "The width of the window."
                    },
                    {
                      name = "height",
                      type = "number",
                      description = "The height of the window."
                    },
                    {
                      name = "fullscreen",
                      type = "boolean",
                      description = "Whether the window is fullscreen."
                    },
                    {
                      name = "resizable",
                      type = "boolean",
                      description = "Whether the window is resizable."
                    },
                    {
                      name = "title",
                      type = "string",
                      description = "The window title."
                    },
                    {
                      name = "icon",
                      type = "string",
                      description = "The path to the window icon file."
                    }
                  }
                }
              }
            }
          },
          returns = {}
        }
      }
    },
    {
      name = "draw",
      tag = "callbacks",
      summary = "Called continuously to render frames to the display.",
      description = "This callback is called every frame, and receives a `Pass` object as an argument which can be used to render graphics to the display.  If a VR headset is connected, this function renders to the headset display, otherwise it will render to the desktop window.",
      key = "lovr.draw",
      module = "lovr",
      notes = "To render to the desktop window when a VR headset is connected, use the `lovr.mirror` callback.\n\nThe display is cleared to the global background color before this callback is called, which can be changed using `lovr.graphics.setBackgroundColor`.\n\nSince the `lovr.graphics.submit` function always returns true, the following idiom can be used to submit graphics work manually and override the default submission:\n\n    function lovr.draw(pass)\n      local passes = {}\n\n      -- ... record multiple passes and add to passes table\n\n      return lovr.graphics.submit(passes)\n    end",
      related = {
        "lovr.mirror",
        "lovr.headset.getPass",
        "lovr.graphics.getWindowPass",
        "lovr.graphics.setBackgroundColor"
      },
      variants = {
        {
          arguments = {
            {
              name = "pass",
              type = "Pass",
              description = "A render pass targeting the main display (headset or window)."
            }
          },
          returns = {
            {
              name = "skip",
              type = "boolean",
              description = "If truthy, the input Pass will not be submitted to the GPU."
            }
          }
        }
      }
    },
    {
      name = "errhand",
      tag = "callbacks",
      summary = "Called when an error occurs.",
      description = "The `lovr.errhand` callback is run whenever an error occurs.  It receives a parameter containing the error message.  It should return a handler function that will run in a loop to render the error screen.\n\nThis handler function is of the same type as the one returned by `lovr.run` and has the same requirements (such as pumping events).  If an error occurs while this handler is running, the program will terminate immediately -- `lovr.errhand` will not be given a second chance.  Errors which occur in the error handler or in the handler it returns may not be cleanly reported, so be careful.\n\nA default error handler is supplied that renders the error message as text to the headset and to the window.",
      key = "lovr.errhand",
      module = "lovr",
      examples = {
        {
          description = "The default error handler.",
          code = "function lovr.errhand(message)\n  local function formatTraceback(s)\n    return s:gsub('\\n[^\\n]+$', ''):gsub('\\t', ''):gsub('stack traceback:', '\\nStack:\\n')\n  end\n\n  message = 'Error:\\n\\n' .. tostring(message) .. formatTraceback(debug.traceback('', 4))\n\n  print(message)\n\n  if not lovr.graphics or not lovr.graphics.isInitialized() then\n    return function() return 1 end\n  end\n\n  if lovr.audio then lovr.audio.stop() end\n\n  if not lovr.headset or lovr.headset.getPassthrough() == 'opaque' then\n    lovr.graphics.setBackgroundColor(.11, .10, .14)\n  else\n    lovr.graphics.setBackgroundColor(0, 0, 0, 0)\n  end\n\n  local font = lovr.graphics.getDefaultFont()\n\n  return function()\n    lovr.system.pollEvents()\n\n    for name, a in lovr.event.poll() do\n      if name == 'quit' then return a or 1\n      elseif name == 'restart' then return 'restart', lovr.restart and lovr.restart()\n      elseif name == 'keypressed' and a == 'f5' then lovr.event.restart()\n      elseif name == 'keypressed' and a == 'escape' then lovr.event.quit() end\n    end\n\n    if lovr.headset and lovr.headset.getDriver() ~= 'desktop' then\n      lovr.headset.update()\n      local pass = lovr.headset.getPass()\n      if pass then\n        font:setPixelDensity()\n\n        local scale = .35\n        local font = lovr.graphics.getDefaultFont()\n        local wrap = .7 * font:getPixelDensity()\n        local lines = font:getLines(message, wrap)\n        local width = math.min(font:getWidth(message), wrap) * scale\n        local height = .8 + #lines * font:getHeight() * scale\n        local x = -width / 2\n        local y = math.min(height / 2, 10)\n        local z = -10\n\n        pass:setColor(.95, .95, .95)\n        pass:text(message, x, y, z, scale, 0, 0, 0, 0, wrap, 'left', 'top')\n\n        lovr.graphics.submit(pass)\n        lovr.headset.submit()\n      end\n    end\n\n    if lovr.system.isWindowOpen() then\n      local pass = lovr.graphics.getWindowPass()\n      if pass then\n        local w, h = lovr.system.getWindowDimensions()\n        pass:setProjection(1, lovr.math.mat4():orthographic(0, w, 0, h, -1, 1))\n        font:setPixelDensity(1)\n\n        local scale = .6\n        local wrap = w * .8 / scale\n        local width = math.min(font:getWidth(message), wrap) * scale\n        local x = w / 2 - width / 2\n\n        pass:setColor(.95, .95, .95)\n        pass:text(message, x, h / 2, 0, scale, 0, 0, 0, 0, wrap, 'left', 'middle')\n\n        lovr.graphics.submit(pass)\n        lovr.graphics.present()\n      end\n    end\n\n    lovr.math.drain()\n  end\nend"
        }
      },
      related = {
        "lovr.quit"
      },
      variants = {
        {
          arguments = {
            {
              name = "message",
              type = "string",
              description = "The error message."
            }
          },
          returns = {
            {
              name = "handler",
              type = "function",
              description = "The error handler function.  It should return nil to continue running, \"restart\" to restart the app, or a number representing an exit status.",
              arguments = {},
              returns = {
                {
                  name = "result",
                  type = "*"
                }
              }
            }
          }
        }
      }
    },
    {
      name = "focus",
      tag = "callbacks",
      summary = "Called when the application gains or loses input focus.",
      description = "The `lovr.focus` callback is called whenever the application acquires or loses focus (for example, when opening or closing the Steam dashboard).  The callback receives a single argument, focused, which is a boolean indicating whether or not the application is now focused.  It may make sense to pause the game or reduce visual fidelity when the application loses focus.",
      key = "lovr.focus",
      module = "lovr",
      related = {
        "lovr.headset.isFocused",
        "lovr.visible"
      },
      variants = {
        {
          arguments = {
            {
              name = "focused",
              type = "boolean",
              description = "Whether the program is now focused."
            }
          },
          returns = {}
        }
      }
    },
    {
      name = "keypressed",
      tag = "callbacks",
      summary = "Called when a key is pressed.",
      description = "This callback is called when a key is pressed.",
      key = "lovr.keypressed",
      module = "lovr",
      related = {
        "lovr.system.wasKeyPressed",
        "lovr.keyreleased",
        "lovr.textinput",
        "lovr.system.isKeyDown"
      },
      variants = {
        {
          arguments = {
            {
              name = "key",
              type = "KeyCode",
              description = "The key that was pressed."
            },
            {
              name = "scancode",
              type = "number",
              description = "The id of the key (ignores keyboard layout, may vary between keyboards)."
            },
            {
              name = "repeating",
              type = "boolean",
              description = "Whether the event is the result of a key repeat instead of an actual press."
            }
          },
          returns = {}
        }
      }
    },
    {
      name = "keyreleased",
      tag = "callbacks",
      summary = "Called when a key is released.",
      description = "This callback is called when a key is released.",
      key = "lovr.keyreleased",
      module = "lovr",
      related = {
        "lovr.system.wasKeyReleased",
        "lovr.keypressed",
        "lovr.textinput",
        "lovr.system.isKeyDown"
      },
      variants = {
        {
          arguments = {
            {
              name = "key",
              type = "KeyCode",
              description = "The key that was released."
            },
            {
              name = "scancode",
              type = "number",
              description = "The id of the key (ignores keyboard layout, may vary between keyboards)."
            }
          },
          returns = {}
        }
      }
    },
    {
      name = "load",
      tag = "callbacks",
      summary = "Called once at startup.",
      description = "This callback is called once when the app starts.  It should be used to perform initial setup work, like loading resources and initializing classes and variables.",
      key = "lovr.load",
      module = "lovr",
      examples = {
        {
          code = "function lovr.load(arg)\n  model = lovr.graphics.newModel('sponza.gltf')\n  texture = lovr.graphics.newTexture('cena.png')\n  effects = lovr.graphics.newShader('vert.glsl', 'frag.glsl')\n  loadLevel(1)\nend"
        }
      },
      notes = "If the project was loaded from a restart using `lovr.event.restart`, the return value from the previously-run `lovr.restart` callback will be made available to this callback as the `restart` key in the `arg` table.\n\nThe `arg` table follows the [Lua standard](https://en.wikibooks.org/wiki/Lua_Programming/command_line_parameter).  The arguments passed in from the shell are put into a global table named `arg` and passed to `lovr.load`, but with indices offset such that the \"script\" (the project path) is at index 0.  So all arguments (if any) intended for the project are at successive indices starting with 1, and the executable and its \"internal\" arguments are in normal order but stored in negative indices.",
      related = {
        "lovr.quit"
      },
      variants = {
        {
          arguments = {
            {
              name = "arg",
              type = "table",
              description = "The command line arguments provided to the program."
            }
          },
          returns = {}
        }
      }
    },
    {
      name = "log",
      tag = "callbacks",
      summary = "Called when a message is logged.",
      description = "This callback is called when a message is logged.  The default implementation of this callback prints the message to the console using `print`, but it's possible to override this callback to render messages in VR, write them to a file, filter messages, and more.\n\nThe message can have a \"tag\" that is a short string representing the sender, and a \"level\" indicating how severe the message is.\n\nThe `t.graphics.debug` flag in `lovr.conf` can be used to get log messages from the GPU driver (tagged as `GPU`).  It is also possible to emit customlog messages using `lovr.event.push`, or by calling the callback.",
      key = "lovr.log",
      module = "lovr",
      related = {
        "Pass:text"
      },
      variants = {
        {
          arguments = {
            {
              name = "message",
              type = "string",
              description = "The log message.  It may end in a newline."
            },
            {
              name = "level",
              type = "string",
              description = "The log level (`debug`, `info`, `warn`, or `error`)."
            },
            {
              name = "tag",
              type = "string",
              description = "The log tag."
            }
          },
          returns = {}
        }
      }
    },
    {
      name = "mirror",
      tag = "callbacks",
      summary = "Called to render content to the desktop window.",
      description = "This callback is called every frame after rendering to the headset and is usually used to render a mirror of the headset display onto the desktop window.  It can be overridden for custom mirroring behavior.  For example, a stereo view could be drawn instead of a single eye or a 2D HUD could be rendered.",
      key = "lovr.mirror",
      module = "lovr",
      examples = {
        {
          description = "The default `lovr.mirror` implementation draws the headset mirror texture to the window if the headset is active, or just calls `lovr.draw` if there isn't a headset.",
          code = "function lovr.mirror(pass)\n  if lovr.headset then\n    local texture = lovr.headset.getTexture()\n    if texture then\n      pass:fill(texture)\n    else\n      return true\n    end\n  else\n    return lovr.draw and lovr.draw(pass)\n  end\nend"
        }
      },
      related = {
        "lovr.system.openWindow",
        "lovr.draw"
      },
      variants = {
        {
          arguments = {
            {
              name = "pass",
              type = "Pass",
              description = "A render pass targeting the window."
            }
          },
          returns = {
            {
              name = "skip",
              type = "boolean",
              description = "If truthy, the input Pass will not be submitted to the GPU."
            }
          }
        }
      }
    },
    {
      name = "mousemoved",
      tag = "callbacks",
      summary = "Called when the mouse is moved.",
      description = "This callback is called when the mouse is moved.",
      key = "lovr.mousemoved",
      module = "lovr",
      related = {
        "lovr.mousepressed",
        "lovr.mousereleased",
        "lovr.wheelmoved",
        "lovr.system.getMouseX",
        "lovr.system.getMouseY",
        "lovr.system.getMousePosition"
      },
      variants = {
        {
          arguments = {
            {
              name = "x",
              type = "number",
              description = "The new x position of the mouse."
            },
            {
              name = "y",
              type = "number",
              description = "The new y position of the mouse."
            },
            {
              name = "dx",
              type = "number",
              description = "The movement on the x axis since the last mousemove event."
            },
            {
              name = "dy",
              type = "number",
              description = "The movement on the y axis since the last mousemove event."
            }
          },
          returns = {}
        }
      }
    },
    {
      name = "mousepressed",
      tag = "callbacks",
      summary = "Called when a mouse button is pressed.",
      description = "This callback is called when a mouse button is pressed.",
      key = "lovr.mousepressed",
      module = "lovr",
      related = {
        "lovr.mousereleased",
        "lovr.mousemoved",
        "lovr.wheelmoved",
        "lovr.system.isMouseDown"
      },
      variants = {
        {
          arguments = {
            {
              name = "x",
              type = "number",
              description = "The x position of the mouse when the button was pressed."
            },
            {
              name = "y",
              type = "number",
              description = "The y position of the mouse when the button was pressed."
            },
            {
              name = "button",
              type = "number",
              description = "The button that was pressed.  Will be 1 for the primary button, 2 for the secondary button, or 3 for the middle mouse button."
            }
          },
          returns = {}
        }
      }
    },
    {
      name = "mousereleased",
      tag = "callbacks",
      summary = "Called when a mouse button is released.",
      description = "This callback is called when a mouse button is released.",
      key = "lovr.mousereleased",
      module = "lovr",
      related = {
        "lovr.mousepressed",
        "lovr.mousemoved",
        "lovr.wheelmoved",
        "lovr.system.isMouseDown"
      },
      variants = {
        {
          arguments = {
            {
              name = "x",
              type = "number",
              description = "The x position of the mouse when the button was released."
            },
            {
              name = "y",
              type = "number",
              description = "The y position of the mouse when the button was released."
            },
            {
              name = "button",
              type = "number",
              description = "The button that was released.  Will be 1 for the primary button, 2 for the secondary button, or 3 for the middle mouse button."
            }
          },
          returns = {}
        }
      }
    },
    {
      name = "permission",
      tag = "callbacks",
      summary = "Called when a permission request is answered.",
      description = "This callback contains a permission response previously requested with `lovr.system.requestPermission`.  The callback contains information on whether permission was granted or denied.",
      key = "lovr.permission",
      module = "lovr",
      related = {
        "lovr.system.requestPermission"
      },
      variants = {
        {
          arguments = {
            {
              name = "permission",
              type = "Permission",
              description = "The type of permission."
            },
            {
              name = "granted",
              type = "boolean",
              description = "Whether permission was granted or denied."
            }
          },
          returns = {}
        }
      }
    },
    {
      name = "quit",
      tag = "callbacks",
      summary = "Called before quitting.",
      description = "This callback is called right before the application is about to quit.  Use it to perform any necessary cleanup work.  A truthy value can be returned from this callback to abort quitting.",
      key = "lovr.quit",
      module = "lovr",
      examples = {
        {
          code = "function lovr.quit()\n  if shouldQuit() then\n    return false\n  else\n    return true\n  end\nend"
        }
      },
      related = {
        "lovr.event.quit",
        "lovr.load"
      },
      variants = {
        {
          arguments = {},
          returns = {
            {
              name = "abort",
              type = "boolean",
              description = "Whether quitting should be aborted."
            }
          }
        }
      }
    },
    {
      name = "recenter",
      tag = "callbacks",
      summary = "Called when the user recenters the coordinate space.",
      description = "The `lovr.recenter` callback is called whenever the user performs a \"recenter\" gesture to realign the virtual coordinate space.  On most VR systems this will move the origin to the user's current position and rotate its yaw to match the view direction.  The y=0 position will always be on the floor or at eye level, depending on whether `t.headset.seated` was set in `lovr.conf`.",
      key = "lovr.recenter",
      module = "lovr",
      notes = "Note that the pose of the `floor` device will not always be at the origin of the coordinate space.  It uses a fixed position on the floor in the real world, usually the center of the configured play area.  This allows virtual objects to be positioned in a room without having them jump around after a recenter.",
      related = {
        "lovr.headset.getBoundsWidth",
        "lovr.headset.getBoundsDepth",
        "lovr.headset.getBoundsDimensions",
        "lovr.headset.getBoundsGeometry",
        "lovr.headset.isSeated"
      },
      variants = {
        {
          arguments = {},
          returns = {}
        }
      }
    },
    {
      name = "resize",
      tag = "callbacks",
      summary = "Called when the window is resized.",
      description = "This callback is called when the desktop window is resized.",
      key = "lovr.resize",
      module = "lovr",
      related = {
        "Pass:getDimensions",
        "Pass:getWidth",
        "Pass:getHeight",
        "lovr.headset.getDisplayDimensions",
        "lovr.conf"
      },
      variants = {
        {
          arguments = {
            {
              name = "width",
              type = "number",
              description = "The new width of the window."
            },
            {
              name = "height",
              type = "number",
              description = "The new height of the window."
            }
          },
          returns = {}
        }
      }
    },
    {
      name = "restart",
      tag = "callbacks",
      summary = "Called when restarting.",
      description = "This callback is called when a restart from `lovr.event.restart` is happening.  A value can be returned to send it to the next LÖVR instance, available as the `restart` key in the argument table passed to `lovr.load`.  Object instances can not be used as the restart value, since they are destroyed as part of the cleanup process.",
      key = "lovr.restart",
      module = "lovr",
      examples = {
        {
          code = "function lovr.restart()\n  return currentLevel:getName()\nend"
        }
      },
      notes = "Only nil, booleans, numbers, and strings are supported types for the return value.",
      related = {
        "lovr.event.restart",
        "lovr.load",
        "lovr.quit"
      },
      variants = {
        {
          arguments = {},
          returns = {
            {
              name = "cookie",
              type = "*",
              description = "The value to send to the next `lovr.load`."
            }
          }
        }
      }
    },
    {
      name = "run",
      tag = "callbacks",
      summary = "The main entry point.",
      description = "This callback is the main entry point for a LÖVR program.  It calls `lovr.load` and returns a function that will be called every frame.",
      key = "lovr.run",
      module = "lovr",
      examples = {
        {
          description = "The default `lovr.run`:",
          code = "function lovr.run()\n  if lovr.timer then lovr.timer.step() end\n  if lovr.load then lovr.load(arg) end\n  return function()\n    if lovr.system then lovr.system.pollEvents() end\n    if lovr.event then\n      for name, a, b, c, d in lovr.event.poll() do\n        if name == 'restart' then\n          local cookie = lovr.restart and lovr.restart()\n          return 'restart', cookie\n        elseif name == 'quit' and (not lovr.quit or not lovr.quit(a)) then\n          return a or 0\n        end\n        if lovr.handlers[name] then lovr.handlers[name](a, b, c, d) end\n      end\n    end\n    local dt = 0\n    if lovr.timer then dt = lovr.timer.step() end\n    if lovr.headset then dt = lovr.headset.update() end\n    if lovr.update then lovr.update(dt) end\n    if lovr.graphics then\n      local headset = lovr.headset and lovr.headset.getPass()\n      if headset and (not lovr.draw or lovr.draw(headset)) then headset = nil end\n      local window = lovr.graphics.getWindowPass()\n      if window and (not lovr.mirror or lovr.mirror(window)) then window = nil end\n      lovr.graphics.submit(headset, window)\n      lovr.graphics.present()\n    end\n    if lovr.headset then lovr.headset.submit() end\n    if lovr.math then lovr.math.drain() end\n  end\nend"
        }
      },
      notes = "The main loop function can return one of the following values:\n\n- Returning `nil` will keep the main loop running.\n- Returning the string 'restart' plus an optional value will restart LÖVR.  The value can be\n  accessed in the `restart` key of the `arg` global.\n- Returning a number will exit LÖVR using the number as the exit code (0 means success).\n\nCare should be taken when overriding this callback.  For example, if the main loop does not call `lovr.system.pollEvents` then the OS will think LÖVR is unresponsive, or if the quit event is not handled then closing the window won't work.",
      related = {
        "lovr.load",
        "lovr.quit"
      },
      variants = {
        {
          arguments = {},
          returns = {
            {
              name = "loop",
              type = "function",
              description = "The main loop function.",
              arguments = {},
              returns = {
                {
                  name = "result",
                  type = "*"
                }
              }
            }
          }
        }
      }
    },
    {
      name = "textinput",
      tag = "callbacks",
      summary = "Called when text has been entered.",
      description = "This callback is called when text has been entered.\n\nFor example, when `shift + 1` is pressed on an American keyboard, `lovr.textinput` will be called with `!`.",
      key = "lovr.textinput",
      module = "lovr",
      notes = "Some characters in UTF-8 unicode take multiple bytes to encode.  Due to the way Lua works, the length of these strings will be bigger than 1 even though they are just a single character.  The `utf8` library included with LÖVR can be used to manipulate UTF-8 strings.  `Pass:text` will also correctly handle UTF-8.",
      related = {
        "lovr.keypressed",
        "lovr.keyreleased"
      },
      variants = {
        {
          arguments = {
            {
              name = "text",
              type = "string",
              description = "The UTF-8 encoded character."
            },
            {
              name = "code",
              type = "number",
              description = "The integer codepoint of the character."
            }
          },
          returns = {}
        }
      }
    },
    {
      name = "threaderror",
      tag = "callbacks",
      summary = "Called when an error occurs in a thread.",
      description = "The `lovr.threaderror` callback is called whenever an error occurs in a Thread.  It receives the Thread object where the error occurred and an error message.\n\nThe default implementation of this callback will call `lovr.errhand` with the error.",
      key = "lovr.threaderror",
      module = "lovr",
      related = {
        "Thread",
        "Thread:getError",
        "lovr.errhand"
      },
      variants = {
        {
          arguments = {
            {
              name = "thread",
              type = "Thread",
              description = "The Thread that errored."
            },
            {
              name = "message",
              type = "string",
              description = "The error message."
            }
          },
          returns = {}
        }
      }
    },
    {
      name = "update",
      tag = "callbacks",
      summary = "Called every frame to update the application logic.",
      description = "The `lovr.update` callback should be used to update your game's logic.  It receives a single parameter, `dt`, which represents the amount of elapsed time between frames.  You can use this value to scale timers, physics, and animations in your game so they play at a smooth, consistent speed.",
      key = "lovr.update",
      module = "lovr",
      examples = {
        {
          code = "function lovr.update(dt)\n  ball.vy = ball.vy + ball.gravity * dt\n  ball.y = ball.y + ball.vy * dt\nend"
        }
      },
      related = {
        "lovr.timer.getDelta"
      },
      variants = {
        {
          arguments = {
            {
              name = "dt",
              type = "number",
              description = "The number of seconds elapsed since the last update."
            }
          },
          returns = {}
        }
      }
    },
    {
      name = "visible",
      tag = "callbacks",
      summary = "Called when the application gains or loses visibility.",
      description = "The `lovr.visible` callback is called whenever the application becomes visible or invisible. `lovr.draw` may still be called even while invisible to give the VR runtime timing info.  If the VR runtime decides the application doesn't need to render anymore, LÖVR will detect this and stop calling `lovr.draw`.",
      key = "lovr.visible",
      module = "lovr",
      related = {
        "lovr.headset.isVisible",
        "lovr.focus"
      },
      variants = {
        {
          arguments = {
            {
              name = "visible",
              type = "boolean",
              description = "Whether the application is visible in the headset display."
            }
          },
          returns = {}
        }
      }
    },
    {
      name = "wheelmoved",
      tag = "callbacks",
      summary = "Called when a mouse wheel is moved.",
      description = "This callback is called on scroll action, from a mouse wheel or a touchpad",
      key = "lovr.wheelmoved",
      module = "lovr",
      related = {
        "lovr.mousepressed",
        "lovr.mousereleased",
        "lovr.mousemoved",
        "lovr.system.isMouseDown"
      },
      variants = {
        {
          arguments = {
            {
              name = "deltaX",
              type = "number",
              description = "The relative horizontal motion; rightward movement resuts in positive values."
            },
            {
              name = "deltaY",
              type = "number",
              description = "The relative vertical motion; upward movement results in positive values."
            }
          },
          returns = {}
        }
      }
    }
  },
  modules = {
    {
      name = "enet",
      tag = "libraries",
      summary = "UDP networking library.",
      description = "ENet is a UDP networking plugin bundled with LÖVR that can be used for networking and multiplayer experiences.  ENet allows messages to be marked for reliable and in-order delivery, allowing the speed of UDP to be used without sacrificing reliability.\n\nThe full documentation and examples can be found on the [lua-enet](http://leafo.net/lua-enet/) page.  LÖVE also has lua-enet documentation [here](https://love2d.org/wiki/lua-enet).",
      key = "enet",
      enums = {},
      examples = {
        {
          description = "Here's a simple echo server example. The client sends a message to the server and waits for a response. The server waits for a message and sends it back to the client.",
          code = "-- client/main.lua\nlocal enet = require 'enet'\n\nfunction lovr.load()\n  local host = enet.host_create()\n  local server = host:connect('localhost:6789')\n\n  local done = false\n  while not done do\n    local event = host:service(100)\n    if event then\n      if event.type == 'connect' then\n        print('Connected to', event.peer)\n        event.peer:send('hello world')\n      elseif event.type == 'receive' then\n        print('Got message: ', event.data, event.peer)\n        done = true\n      end\n    end\n  end\n\n  server:disconnect()\n  host:flush()\nend\n\n-- server/main.lua\nlocal enet = require 'enet'\n\nfunction lovr.load()\n  local host = enet.host_create('localhost:6789')\n  while true do\n    local event = host:service(100)\n    if event and event.type == 'receive' then\n      print('Got message: ', event.data, event.peer)\n      event.peer:send(event.data)\n    end\n  end\nend"
        }
      },
      external = true,
      functions = {},
      objects = {}
    },
    {
      name = "http",
      tag = "libraries",
      summary = "HTTP(S) requests.",
      description = "The [lovr-http](https://github.com/bjornbytes/lovr-http) plugin performs HTTP requests.\n\nFirst, `require` the plugin and save it into a variable: `local http = require 'http'`.\n\nThe module has one function:\n\n    status, data, headers = http.request(url, [options])\n\nThis will perform an HTTP request and block until the request is complete.\n\n### Arguments\n\n`url` is the URL to request.  If it doesn't have a protocol, then `http://` will be added.\n\n`options` is optional, and is used for advanced request settings.\n\n`options.method` is the HTTP method to use, also called the verb.  `GET` is used by default if there's no data in the request, otherwise it defauls to `POST`.  It will be converted to all-caps.\n\n`options.data` is the data to send to the server, also called the body.  It can be a few different types:\n\n- When `data` is nil, no request body will be sent (and `method` will default to `GET`).\n- When `data` is a string, the string will be used directly as the request body.\n- When `data` is a table, then pairs in the table will be URL encoded and concatenated together\n  to form an `application/x-www-form-urlencoded` body.  For example, if data is\n  `{ n = 10, k = 'v!' }`, then the request body will be something like `k=v%21&n=10`. Table\n  pairs will only be used if the key is a string and the value is a string or number.\n- When `data` is a lightuserdata, the data pointed to by the lightuserdata will be used as the\n  request body. Additionally, the `datasize` option should be an integer indicating how big the\n  request body is, in bytes.\n\nWhen `options.data` is set, the `Content-Type` request header will default to `application/x-www-urlencoded` unless it's set to something else.\n\n`options.headers` is a table of request headers to send to the server.  Pairs in the table will only be used if the key is a string and the value is a string or number.\n\n### Returns\n\nIf an error occurs, the function returns `nil, errormessage`.\n\nOtherwise, 3 values are returned:\n\n- `status` is an integer with the HTTP status code (200 is OK, 404 is Not Found, etc.).\n- `data` is a string with the data sent by the server (HTML, JSON, binary, etc.).\n- `headers` is a table of response headers.",
      key = "http",
      enums = {},
      examples = {
        {
          code = "local http = require 'http'\n\nlocal status, data, headers = http.request('https://zombo.com')\n\nprint('welcome')\nprint(status)\nprint(data)\nprint('headers:')\nfor k, v in pairs(headers) do\n  print('\\t' .. k, v)\nend"
        }
      },
      external = true,
      functions = {},
      objects = {}
    },
    {
      name = "lovr",
      summary = "In the beginning, there was nothing.",
      description = "`lovr` is the single global table that is exposed to every LÖVR app. It contains a set of **modules** and a set of **callbacks**.",
      key = "lovr",
      enums = {},
      functions = {
        {
          name = "getVersion",
          tag = "version",
          summary = "Get the current version.",
          description = "Get the current major, minor, and patch version of LÖVR.",
          key = "lovr.getVersion",
          module = "lovr",
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "major",
                  type = "number",
                  description = "The major version."
                },
                {
                  name = "minor",
                  type = "number",
                  description = "The minor version."
                },
                {
                  name = "patch",
                  type = "number",
                  description = "The patch number."
                }
              }
            }
          }
        }
      },
      objects = {
        {
          name = "Object",
          summary = "The base object.",
          description = "The superclass of all LÖVR objects.  All objects have these methods.",
          key = "Object",
          module = "lovr",
          methods = {
            {
              name = "release",
              summary = "Immediately release the Lua reference to an object.",
              description = "Immediately destroys Lua's reference to the object it's called on.  After calling this function on an object, it is an error to do anything with the object from Lua (call methods on it, pass it to other functions, etc.).  If nothing else is using the object, it will be destroyed immediately, which can be used to destroy something earlier than it would normally be garbage collected in order to reduce memory.",
              key = "Object:release",
              module = "lovr",
              notes = "The object may not be destroyed immediately if something else is referring to it (e.g. it is pushed to a Channel or exists in the payload of a pending event).",
              variants = {
                {
                  arguments = {},
                  returns = {}
                }
              }
            },
            {
              name = "type",
              summary = "Get the type name of the object.",
              description = "Returns the name of the object's type as a string.",
              key = "Object:type",
              module = "lovr",
              examples = {
                {
                  code = "function isTexture(obj)\n  return type(obj) == 'userdata' and obj:type() == 'Texture'\nend"
                }
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "type",
                      type = "string",
                      description = "The type of the object."
                    }
                  }
                }
              }
            }
          },
          notes = "Note that the functions here don't apply to any vector objects, see `Vectors`."
        }
      },
      sections = {
        {
          name = "Modules",
          tag = "modules",
          description = "Modules are the **what** of your app; you can use the functions in modules to tell LÖVR to do things. For example, you can draw things on the screen, figure out what buttons on a controller are pressed, or load a 3D model from a file.  Each module does what it says on the tin, so the `lovr.graphics` module deals with rendering graphics and `lovr.headset` allows you to interact with VR hardware."
        },
        {
          name = "Callbacks",
          tag = "callbacks",
          description = "Callbacks are the **when** of the application; you write code inside callbacks which LÖVR then calls at certain points in time.  For example, the `lovr.load` callback is called once at startup, and `lovr.focus` is called when the VR application gains or loses input focus."
        },
        {
          name = "Version",
          tag = "version",
          description = "This function can be used to get the current version of LÖVR."
        },
        {
          name = "Libraries",
          tag = "libraries",
          description = "LÖVR bundles a few third-party modules by default."
        }
      }
    },
    {
      name = "audio",
      tag = "modules",
      summary = "Plays sound.",
      description = "The `lovr.audio` module is responsible for playing sound effects and music.  To play a sound, create a `Source` object and call `Source:play` on it.  Currently ogg, wav, and mp3 audio formats are supported.",
      key = "lovr.audio",
      enums = {
        {
          name = "AudioMaterial",
          summary = "Different types of audio materials.",
          description = "Different types of audio material presets, for use with `lovr.audio.setGeometry`.",
          key = "AudioMaterial",
          module = "lovr.audio",
          values = {
            {
              name = "generic",
              description = "Generic default audio material."
            },
            {
              name = "brick",
              description = "Brick."
            },
            {
              name = "carpet",
              description = "Carpet."
            },
            {
              name = "ceramic",
              description = "Ceramic."
            },
            {
              name = "concrete",
              description = "Concrete."
            },
            {
              name = "glass",
              description = "Glass."
            },
            {
              name = "gravel",
              description = "Gravel."
            },
            {
              name = "metal",
              description = "Metal."
            },
            {
              name = "plaster",
              description = "Plaster."
            },
            {
              name = "rock",
              description = "Rock."
            },
            {
              name = "wood",
              description = "Wood."
            }
          }
        },
        {
          name = "AudioShareMode",
          summary = "How audio devices are shared on the system.",
          description = "Audio devices can be created in shared mode or exclusive mode.  In exclusive mode, the audio device is the only one active on the system, which gives better performance and lower latency. However, exclusive devices aren't always supported and might not be allowed, so there is a higher chance that creating one will fail.",
          key = "AudioShareMode",
          module = "lovr.audio",
          related = {
            "lovr.audio.setDevice"
          },
          values = {
            {
              name = "shared",
              description = "Shared mode."
            },
            {
              name = "exclusive",
              description = "Exclusive mode."
            }
          }
        },
        {
          name = "AudioType",
          summary = "Different types of audio devices",
          description = "When referencing audio devices, this indicates whether it's the playback or capture device.",
          key = "AudioType",
          module = "lovr.audio",
          related = {
            "lovr.audio.getDevices",
            "lovr.audio.setDevice",
            "lovr.audio.start",
            "lovr.audio.stop",
            "lovr.audio.isStarted"
          },
          values = {
            {
              name = "playback",
              description = "The playback device (speakers, headphones)."
            },
            {
              name = "capture",
              description = "The capture device (microphone)."
            }
          }
        },
        {
          name = "Effect",
          summary = "Different types of Source effects.",
          description = "Different types of effects that can be applied with `Source:setEffectEnabled`.",
          key = "Effect",
          module = "lovr.audio",
          notes = "The active spatializer will determine which effects are supported.  If an unsupported effect is enabled on a Source, no error will be reported.  Instead, it will be silently ignored.\n\nSee `lovr.audio.getSpatializer` for a table of the supported effects for each spatializer.",
          values = {
            {
              name = "absorption",
              description = "Models absorption as sound travels through the air, water, etc."
            },
            {
              name = "attenuation",
              description = "Decreases audio volume with distance (1 / max(distance, 1))."
            },
            {
              name = "occlusion",
              description = "Causes audio to drop off when the Source is occluded by geometry."
            },
            {
              name = "reverb",
              description = "Models reverb caused by audio bouncing off of geometry."
            },
            {
              name = "spatialization",
              description = "Spatializes the Source using either simple panning or an HRTF."
            },
            {
              name = "transmission",
              description = "Causes audio to be heard through walls when occluded, based on audio materials."
            }
          }
        },
        {
          name = "TimeUnit",
          summary = "Time units for sound samples.",
          description = "When figuring out how long a Source is or seeking to a specific position in the sound file, units can be expressed in terms of seconds or in terms of frames.  A frame is one set of samples for each channel (one sample for mono, two samples for stereo).",
          key = "TimeUnit",
          module = "lovr.audio",
          values = {
            {
              name = "seconds",
              description = "Seconds."
            },
            {
              name = "frames",
              description = "Frames."
            }
          }
        },
        {
          name = "VolumeUnit",
          summary = "Different units of volume.",
          description = "When accessing the volume of Sources or the audio listener, this can be done in linear units with a 0 to 1 range, or in decibels with a range of -∞ to 0.",
          key = "VolumeUnit",
          module = "lovr.audio",
          values = {
            {
              name = "linear",
              description = "Linear volume range."
            },
            {
              name = "db",
              description = "Decibels."
            }
          }
        }
      },
      functions = {
        {
          name = "getAbsorption",
          summary = "Get the absorption coefficients.",
          description = "Returns the global air absorption coefficients for the medium.  This affects Sources that have the `absorption` effect enabled, causing audio volume to drop off with distance as it is absorbed by the medium it's traveling through (air, water, etc.).  The difference between absorption and the attenuation effect is that absorption is more subtle and is frequency-dependent, so higher-frequency bands can get absorbed more quickly than lower ones. This can be used to apply \"underwater\" effects and stuff.",
          key = "lovr.audio.getAbsorption",
          module = "lovr.audio",
          notes = "Absorption is currently only supported by the phonon spatializer.\n\nThe frequency bands correspond to `400Hz`, `2.5KHz`, and `15KHz`.\n\nThe default coefficients are `.0002`, `.0017`, and `.0182` for low, mid, and high.",
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "low",
                  type = "number",
                  description = "The absorption coefficient for the low frequency band."
                },
                {
                  name = "mid",
                  type = "number",
                  description = "The absorption coefficient for the mid frequency band."
                },
                {
                  name = "high",
                  type = "number",
                  description = "The absorption coefficient for the high frequency band."
                }
              }
            }
          }
        },
        {
          name = "getDevice",
          tag = "devices",
          summary = "Switch audio devices.",
          description = "Returns information about the active playback or capture device.",
          key = "lovr.audio.getDevice",
          module = "lovr.audio",
          notes = "If no device has been set yet, this function returns `nil`.\n\nThe device doesn't need to be started.",
          related = {
            "lovr.audio.getDevices",
            "lovr.audio.setDevice"
          },
          variants = {
            {
              arguments = {
                {
                  name = "type",
                  type = "AudioType",
                  description = "The type of device to query.",
                  default = "'playback'"
                }
              },
              returns = {
                {
                  name = "name",
                  type = "string",
                  description = "The name of the device."
                },
                {
                  name = "id",
                  type = "userdata",
                  description = "The opaque id of the device."
                }
              }
            }
          }
        },
        {
          name = "getDevices",
          tag = "devices",
          summary = "Get a list of audio devices.",
          description = "Returns a list of playback or capture devices.  Each device has an `id`, `name`, and a `default` flag indicating whether it's the default device.\n\nTo use a specific device id for playback or capture, pass it to `lovr.audio.setDevice`.",
          key = "lovr.audio.getDevices",
          module = "lovr.audio",
          related = {
            "lovr.audio.setDevice",
            "lovr.audio.getDevice",
            "lovr.audio.start",
            "lovr.audio.stop"
          },
          variants = {
            {
              arguments = {
                {
                  name = "type",
                  type = "AudioType",
                  description = "The type of devices to query (playback or capture).",
                  default = "'playback'"
                }
              },
              returns = {
                {
                  name = "devices",
                  type = "table",
                  description = "The list of devices.",
                  table = {
                    {
                      name = "[].id",
                      type = "userdata",
                      description = "A unique, opaque id for the device."
                    },
                    {
                      name = "[].name",
                      type = "string",
                      description = "A human readable name for the device."
                    },
                    {
                      name = "[].default",
                      type = "boolean",
                      description = "Whether the device is the default audio device."
                    }
                  }
                }
              }
            }
          }
        },
        {
          name = "getOrientation",
          tag = "listener",
          summary = "Get the orientation of the listener.",
          description = "Returns the orientation of the virtual audio listener in angle/axis representation.",
          key = "lovr.audio.getOrientation",
          module = "lovr.audio",
          related = {
            "lovr.audio.getPosition",
            "lovr.audio.getPose",
            "Source:getOrientation"
          },
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "angle",
                  type = "number",
                  description = "The number of radians the listener is rotated around its axis of rotation."
                },
                {
                  name = "ax",
                  type = "number",
                  description = "The x component of the axis of rotation."
                },
                {
                  name = "ay",
                  type = "number",
                  description = "The y component of the axis of rotation."
                },
                {
                  name = "az",
                  type = "number",
                  description = "The z component of the axis of rotation."
                }
              }
            }
          }
        },
        {
          name = "getPose",
          tag = "listener",
          summary = "Get the pose of the listener.",
          description = "Returns the position and orientation of the virtual audio listener.",
          key = "lovr.audio.getPose",
          module = "lovr.audio",
          related = {
            "lovr.audio.getPosition",
            "lovr.audio.getOrientation",
            "Source:getPose"
          },
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "x",
                  type = "number",
                  description = "The x position of the listener, in meters."
                },
                {
                  name = "y",
                  type = "number",
                  description = "The y position of the listener, in meters."
                },
                {
                  name = "z",
                  type = "number",
                  description = "The z position of the listener, in meters."
                },
                {
                  name = "angle",
                  type = "number",
                  description = "The number of radians the listener is rotated around its axis of rotation."
                },
                {
                  name = "ax",
                  type = "number",
                  description = "The x component of the axis of rotation."
                },
                {
                  name = "ay",
                  type = "number",
                  description = "The y component of the axis of rotation."
                },
                {
                  name = "az",
                  type = "number",
                  description = "The z component of the axis of rotation."
                }
              }
            }
          }
        },
        {
          name = "getPosition",
          tag = "listener",
          summary = "Get the position of the listener.",
          description = "Returns the position of the virtual audio listener, in meters.",
          key = "lovr.audio.getPosition",
          module = "lovr.audio",
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "x",
                  type = "number",
                  description = "The x position of the listener."
                },
                {
                  name = "y",
                  type = "number",
                  description = "The y position of the listener."
                },
                {
                  name = "z",
                  type = "number",
                  description = "The z position of the listener."
                }
              }
            }
          }
        },
        {
          name = "getSampleRate",
          tag = "devices",
          summary = "Get the playback device sample rate.",
          description = "Returns the sample rate used by the playback device.  This can be changed using `lovr.conf`.",
          key = "lovr.audio.getSampleRate",
          module = "lovr.audio",
          related = {
            "lovr.conf"
          },
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "rate",
                  type = "number",
                  description = "The sample rate of the playback device, in Hz."
                }
              }
            }
          }
        },
        {
          name = "getSpatializer",
          tag = "listener",
          summary = "Get the name of the active spatializer",
          description = "Returns the name of the active spatializer (`simple`, `oculus`, or `phonon`).\n\nThe `t.audio.spatializer` setting in `lovr.conf` can be used to express a preference for a particular spatializer.  If it's `nil`, all spatializers will be tried in the following order: `phonon`, `oculus`, `simple`.",
          key = "lovr.audio.getSpatializer",
          module = "lovr.audio",
          notes = "Using a feature or effect that is not supported by the current spatializer will not error, it just won't do anything.\n\n<table>\n  <thead>\n    <tr>\n      <td>Feature</td>\n      <td>simple</td>\n      <td>phonon</td>\n      <td>oculus</td>\n    </tr>\n  </thead>\n  <tbody>\n    <tr>\n      <td>Effect: Spatialization</td>\n      <td>x</td>\n      <td>x</td>\n      <td>x</td>\n    </tr>\n    <tr>\n      <td>Effect: Attenuation</td>\n      <td>x</td>\n      <td>x</td>\n      <td>x</td>\n    </tr>\n    <tr>\n      <td>Effect: Absorption</td>\n      <td></td>\n      <td>x</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td>Effect: Occlusion</td>\n      <td></td>\n      <td>x</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td>Effect: Transmission</td>\n      <td></td>\n      <td>x</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td>Effect: Reverb</td>\n      <td></td>\n      <td>x</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td>lovr.audio.setGeometry</td>\n      <td></td>\n      <td>x</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td>Source:setDirectivity</td>\n      <td>x</td>\n      <td>x</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td>Source:setRadius</td>\n      <td></td>\n      <td>x</td>\n      <td></td>\n    </tr>\n  </tbody> </table>",
          related = {
            "lovr.conf"
          },
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "spatializer",
                  type = "string",
                  description = "The name of the active spatializer."
                }
              }
            }
          }
        },
        {
          name = "getVolume",
          tag = "listener",
          summary = "Get the master volume.",
          description = "Returns the master volume.  All audio sent to the playback device has its volume multiplied by this factor.",
          key = "lovr.audio.getVolume",
          module = "lovr.audio",
          notes = "The default volume is 1.0 (0 dB).",
          variants = {
            {
              arguments = {
                {
                  name = "units",
                  type = "VolumeUnit",
                  description = "The units to return (linear or db).",
                  default = "'linear'"
                }
              },
              returns = {
                {
                  name = "volume",
                  type = "number",
                  description = "The master volume."
                }
              }
            }
          }
        },
        {
          name = "isStarted",
          tag = "devices",
          summary = "Check if an audio device is started.",
          description = "Returns whether an audio device is started.",
          key = "lovr.audio.isStarted",
          module = "lovr.audio",
          related = {
            "lovr.audio.start",
            "lovr.audio.stop"
          },
          variants = {
            {
              arguments = {
                {
                  name = "type",
                  type = "AudioType",
                  description = "The type of device to check.",
                  default = "'playback'"
                }
              },
              returns = {
                {
                  name = "started",
                  type = "boolean",
                  description = "Whether the device is active."
                }
              }
            }
          }
        },
        {
          name = "newSource",
          tag = "sources",
          summary = "Create a new Source.",
          description = "Creates a new Source from an ogg, wav, or mp3 file.",
          key = "lovr.audio.newSource",
          module = "lovr.audio",
          examples = {
            {
              code = "function lovr.load()\n  sandstorm = lovr.audio.newSource('darude.ogg', {\n    decode = false,\n    effects = { 'spatialization', attenuation = false, reverb = true }\n  })\n\n  sandstorm:play()\nend"
            }
          },
          related = {
            "Source:clone"
          },
          variants = {
            {
              arguments = {
                {
                  name = "filename",
                  type = "string",
                  description = "The filename of the sound to load."
                },
                {
                  name = "options",
                  type = "table",
                  description = "Optional options.",
                  table = {
                    {
                      name = "decode",
                      type = "boolean",
                      description = "Whether to immediately decode compressed sounds, instead of progressively decoding as the Source plays.  Enabling this will use more memory but reduce CPU overhead during playback.  Recommended for short sound effects.",
                      default = "false"
                    },
                    {
                      name = "pitchable",
                      type = "boolean",
                      description = "Whether the pitch of the Source can be changed with `Source:setPitch`.  Setting this to false will improve performance slightly.",
                      default = "true"
                    },
                    {
                      name = "spatial",
                      type = "boolean",
                      description = "Whether the Source should use spatial effects.  Non-spatial sources will get routed directly to the speakers without further processing.  Enabling an effect on a non-spatial source will raise an error.",
                      default = "true"
                    },
                    {
                      name = "effects",
                      type = "table",
                      description = "A table of `Effect`s to enable on the Source.  This can be a list (numeric keys, effect name values) or a map (effect name keys, boolean values) or a mix of the two.  Effects can also be enabled later using `Source:setEffectEnabled`.  If nil, all effects will be enabled.  Ignored if the `spatial` flag is false.",
                      default = "nil"
                    }
                  }
                }
              },
              returns = {
                {
                  name = "source",
                  type = "Source",
                  description = "The new Source."
                }
              }
            },
            {
              arguments = {
                {
                  name = "blob",
                  type = "Blob",
                  description = "The Blob containing the Source data."
                },
                {
                  name = "options",
                  type = "table",
                  description = "Optional options.",
                  table = {
                    {
                      name = "decode",
                      type = "boolean",
                      description = "Whether to immediately decode compressed sounds, instead of progressively decoding as the Source plays.  Enabling this will use more memory but reduce CPU overhead during playback.  Recommended for short sound effects.",
                      default = "false"
                    },
                    {
                      name = "pitchable",
                      type = "boolean",
                      description = "Whether the pitch of the Source can be changed with `Source:setPitch`.  Setting this to false will improve performance slightly.",
                      default = "true"
                    },
                    {
                      name = "spatial",
                      type = "boolean",
                      description = "Whether the Source should use spatial effects.  Non-spatial sources will get routed directly to the speakers without further processing.  Enabling an effect on a non-spatial source will raise an error.",
                      default = "true"
                    },
                    {
                      name = "effects",
                      type = "table",
                      description = "A table of `Effect`s to enable on the Source.  This can be a list (numeric keys, effect name values) or a map (effect name keys, boolean values) or a mix of the two.  Effects can also be enabled later using `Source:setEffectEnabled`.  If nil, all effects will be enabled.  Ignored if the `spatial` flag is false.",
                      default = "nil"
                    }
                  }
                }
              },
              returns = {
                {
                  name = "source",
                  type = "Source",
                  description = "The new Source."
                }
              }
            },
            {
              arguments = {
                {
                  name = "sound",
                  type = "Sound",
                  description = "The Sound containing raw audio samples to play."
                },
                {
                  name = "options",
                  type = "table",
                  description = "Optional options.",
                  table = {
                    {
                      name = "decode",
                      type = "boolean",
                      description = "Whether to immediately decode compressed sounds, instead of progressively decoding as the Source plays.  Enabling this will use more memory but reduce CPU overhead during playback.  Recommended for short sound effects.",
                      default = "false"
                    },
                    {
                      name = "pitchable",
                      type = "boolean",
                      description = "Whether the pitch of the Source can be changed with `Source:setPitch`.  Setting this to false will improve performance slightly.",
                      default = "true"
                    },
                    {
                      name = "spatial",
                      type = "boolean",
                      description = "Whether the Source should use spatial effects.  Non-spatial sources will get routed directly to the speakers without further processing.  Enabling an effect on a non-spatial source will raise an error.",
                      default = "true"
                    },
                    {
                      name = "effects",
                      type = "table",
                      description = "A table of `Effect`s to enable on the Source.  This can be a list (numeric keys, effect name values) or a map (effect name keys, boolean values) or a mix of the two.  Effects can also be enabled later using `Source:setEffectEnabled`.  If nil, all effects will be enabled.  Ignored if the `spatial` flag is false.",
                      default = "nil"
                    }
                  }
                }
              },
              returns = {
                {
                  name = "source",
                  type = "Source",
                  description = "The new Source."
                }
              }
            }
          }
        },
        {
          name = "setAbsorption",
          summary = "Set the absorption coefficients.",
          description = "Sets the global air absorption coefficients for the medium.  This affects Sources that have the `absorption` effect enabled, causing audio volume to drop off with distance as it is absorbed by the medium it's traveling through (air, water, etc.).  The difference between absorption and the attenuation effect is that absorption is more subtle and is frequency-dependent, so higher-frequency bands can get absorbed more quickly than lower ones.  This can be used to apply \"underwater\" effects and stuff.",
          key = "lovr.audio.setAbsorption",
          module = "lovr.audio",
          notes = "Absorption is currently only supported by the phonon spatializer.\n\nThe frequency bands correspond to `400Hz`, `2.5KHz`, and `15KHz`.\n\nThe default coefficients are `.0002`, `.0017`, and `.0182` for low, mid, and high.",
          variants = {
            {
              arguments = {
                {
                  name = "low",
                  type = "number",
                  description = "The absorption coefficient for the low frequency band."
                },
                {
                  name = "mid",
                  type = "number",
                  description = "The absorption coefficient for the mid frequency band."
                },
                {
                  name = "high",
                  type = "number",
                  description = "The absorption coefficient for the high frequency band."
                }
              },
              returns = {}
            }
          }
        },
        {
          name = "setDevice",
          tag = "devices",
          summary = "Switch audio devices.",
          description = "Switches either the playback or capture device to a new one.\n\nIf a device for the given type is already active, it will be stopped and destroyed.  The new device will not be started automatically, use `lovr.audio.start` to start it.\n\nA device id (previously retrieved using `lovr.audio.getDevices`) can be given to use a specific audio device, or `nil` can be used for the id to use the default audio device.\n\nA sink can be also be provided when changing the device.  A sink is an audio stream (`Sound` object with a `stream` type) that will receive all audio samples played (for playback) or all audio samples captured (for capture).  When an audio device with a sink is started, be sure to periodically call `Sound:read` on the sink to read audio samples from it, otherwise it will overflow and discard old data.  The sink can have any format, data will be converted as needed. Using a sink for the playback device will reduce performance, but this isn't the case for capture devices.\n\nAudio devices can be started in `shared` or `exclusive` mode.  Exclusive devices may have lower latency than shared devices, but there's a higher chance that requesting exclusive access to an audio device will fail (either because it isn't supported or allowed).  One strategy is to first try the device in exclusive mode, switching to shared if it doesn't work.",
          key = "lovr.audio.setDevice",
          module = "lovr.audio",
          related = {
            "lovr.audio.getDevice",
            "lovr.audio.getDevices",
            "lovr.audio.start",
            "lovr.audio.stop"
          },
          variants = {
            {
              arguments = {
                {
                  name = "type",
                  type = "AudioType",
                  description = "The device to switch.",
                  default = "'playback'"
                },
                {
                  name = "id",
                  type = "userdata",
                  description = "The id of the device to use, or `nil` to use the default device.",
                  default = "nil"
                },
                {
                  name = "sink",
                  type = "Sound",
                  description = "An optional audio stream to use as a sink for the device.",
                  default = "nil"
                },
                {
                  name = "mode",
                  type = "AudioShareMode",
                  description = "The sharing mode for the device.",
                  default = "shared"
                }
              },
              returns = {
                {
                  name = "success",
                  type = "boolean",
                  description = "Whether creating the audio device succeeded."
                }
              }
            }
          }
        },
        {
          name = "setGeometry",
          tag = "listener",
          summary = "Set the geometry for audio effects.",
          description = "Sets a mesh of triangles to use for modeling audio effects, using a table of vertices or a Model.  When the appropriate effects are enabled, audio from `Source` objects will correctly be occluded by walls and bounce around to create realistic reverb.\n\nAn optional `AudioMaterial` may be provided to specify the acoustic properties of the geometry.",
          key = "lovr.audio.setGeometry",
          module = "lovr.audio",
          notes = "This is currently only supported/used by the `phonon` spatializer.\n\nThe `Effect`s that use geometry are:\n\n- `occlusion`\n- `reverb`\n- `transmission`\n\nIf an existing geometry has been set, this function will replace it.\n\nThe triangles must use counterclockwise winding.",
          related = {
            "lovr.audio.getSpatializer",
            "Source:setEffectEnabled"
          },
          variants = {
            {
              arguments = {
                {
                  name = "vertices",
                  type = "table",
                  description = "A flat table of vertices.  Each vertex is 3 numbers representing its x, y, and z position. The units used for audio coordinates are up to you, but meters are recommended."
                },
                {
                  name = "indices",
                  type = "table",
                  description = "A list of indices, indicating how the vertices are connected into triangles.  Indices are 1-indexed and are 32 bits (they can be bigger than 65535)."
                },
                {
                  name = "material",
                  type = "AudioMaterial",
                  description = "The acoustic material to use.",
                  default = "'generic'"
                }
              },
              returns = {
                {
                  name = "success",
                  type = "boolean",
                  description = "Whether audio geometry is supported by the current spatializer and the geometry was loaded successfully."
                }
              }
            },
            {
              arguments = {
                {
                  name = "model",
                  type = "Model",
                  description = "A model to use for the audio geometry."
                },
                {
                  name = "material",
                  type = "AudioMaterial",
                  description = "The acoustic material to use.",
                  default = "'generic'"
                }
              },
              returns = {
                {
                  name = "success",
                  type = "boolean",
                  description = "Whether audio geometry is supported by the current spatializer and the geometry was loaded successfully."
                }
              }
            }
          }
        },
        {
          name = "setOrientation",
          tag = "listener",
          summary = "Set the orientation of the listener.",
          description = "Sets the orientation of the virtual audio listener in angle/axis representation.",
          key = "lovr.audio.setOrientation",
          module = "lovr.audio",
          related = {
            "lovr.audio.setPosition",
            "lovr.audio.setPose",
            "Source:setOrientation"
          },
          variants = {
            {
              description = "Set the listener orientation using numbers.",
              arguments = {
                {
                  name = "angle",
                  type = "number",
                  description = "The number of radians the listener should be rotated around its rotation axis."
                },
                {
                  name = "ax",
                  type = "number",
                  description = "The x component of the axis of rotation."
                },
                {
                  name = "ay",
                  type = "number",
                  description = "The y component of the axis of rotation."
                },
                {
                  name = "az",
                  type = "number",
                  description = "The z component of the axis of rotation."
                }
              },
              returns = {}
            },
            {
              description = "Set the listener orientation using a vector.",
              arguments = {
                {
                  name = "orientation",
                  type = "Quat",
                  description = "The orientation of the listener."
                }
              },
              returns = {}
            }
          }
        },
        {
          name = "setPose",
          tag = "listener",
          summary = "Set the pose of the listener.",
          description = "Sets the position and orientation of the virtual audio listener.",
          key = "lovr.audio.setPose",
          module = "lovr.audio",
          notes = "The position of the listener doesn't use any specific units, but usually they can be thought of as meters to match the headset module.",
          related = {
            "lovr.audio.setPosition",
            "lovr.audio.setOrientation",
            "Source:setPose"
          },
          variants = {
            {
              description = "Set the pose of the listener using numbers.",
              arguments = {
                {
                  name = "x",
                  type = "number",
                  description = "The x position of the listener."
                },
                {
                  name = "y",
                  type = "number",
                  description = "The y position of the listener."
                },
                {
                  name = "z",
                  type = "number",
                  description = "The z position of the listener."
                },
                {
                  name = "angle",
                  type = "number",
                  description = "The number of radians the listener is rotated around its axis of rotation."
                },
                {
                  name = "ax",
                  type = "number",
                  description = "The x component of the axis of rotation."
                },
                {
                  name = "ay",
                  type = "number",
                  description = "The y component of the axis of rotation."
                },
                {
                  name = "az",
                  type = "number",
                  description = "The z component of the axis of rotation."
                }
              },
              returns = {}
            },
            {
              description = "Set the pose of the listener using vector types.",
              arguments = {
                {
                  name = "position",
                  type = "Vec3",
                  description = "The position of the listener."
                },
                {
                  name = "orientation",
                  type = "Quat",
                  description = "The orientation of the listener."
                }
              },
              returns = {}
            }
          }
        },
        {
          name = "setPosition",
          tag = "listener",
          summary = "Set the position of the listener.",
          description = "Sets the position of the virtual audio listener.  The position doesn't have any specific units, but usually they can be thought of as meters, to match the headset module.",
          key = "lovr.audio.setPosition",
          module = "lovr.audio",
          related = {
            "lovr.audio.setOrientation",
            "lovr.audio.setPose",
            "Source:setPosition"
          },
          variants = {
            {
              description = "Set the listener position using numbers.",
              arguments = {
                {
                  name = "x",
                  type = "number",
                  description = "The x position of the listener."
                },
                {
                  name = "y",
                  type = "number",
                  description = "The y position of the listener."
                },
                {
                  name = "z",
                  type = "number",
                  description = "The z position of the listener."
                }
              },
              returns = {}
            },
            {
              description = "Set the listener position using a vector.",
              arguments = {
                {
                  name = "position",
                  type = "Vec3",
                  description = "The listener position."
                }
              },
              returns = {}
            }
          }
        },
        {
          name = "setVolume",
          tag = "listener",
          summary = "Set the master volume.",
          description = "Sets the master volume.  All audio sent to the playback device has its volume multiplied by this factor.",
          key = "lovr.audio.setVolume",
          module = "lovr.audio",
          notes = "The volume will be clamped to a 0-1 range (0 dB).",
          variants = {
            {
              arguments = {
                {
                  name = "volume",
                  type = "number",
                  description = "The master volume."
                },
                {
                  name = "units",
                  type = "VolumeUnit",
                  description = "The units of the value.",
                  default = "'linear'"
                }
              },
              returns = {}
            }
          }
        },
        {
          name = "start",
          tag = "devices",
          summary = "Start an audio device.",
          description = "Starts the active playback or capture device.  By default the playback device is initialized and started, but this can be controlled using the `t.audio.start` flag in `lovr.conf`.",
          key = "lovr.audio.start",
          module = "lovr.audio",
          notes = "Starting an audio device may fail if:\n\n- The device is already started\n- No device was initialized with `lovr.audio.setDevice`\n- Lack of `audiocapture` permission on Android (see `lovr.system.requestPermission`)\n- Some other problem accessing the audio device",
          related = {
            "lovr.audio.getDevices",
            "lovr.audio.setDevice",
            "lovr.audio.stop",
            "lovr.audio.isStarted",
            "lovr.system.requestPermission",
            "lovr.permission"
          },
          variants = {
            {
              arguments = {
                {
                  name = "type",
                  type = "AudioType",
                  description = "The type of device to start.",
                  default = "'playback'"
                }
              },
              returns = {
                {
                  name = "started",
                  type = "boolean",
                  description = "Whether the device was successfully started."
                }
              }
            }
          }
        },
        {
          name = "stop",
          tag = "devices",
          summary = "Stop an audio device.",
          description = "Stops the active playback or capture device.  This may fail if:\n\n- The device is not started\n- No device was initialized with `lovr.audio.setDevice`",
          key = "lovr.audio.stop",
          module = "lovr.audio",
          notes = "Switching devices with `lovr.audio.setDevice` will stop the existing one.",
          related = {
            "lovr.audio.getDevices",
            "lovr.audio.setDevice",
            "lovr.audio.start",
            "lovr.audio.isStarted"
          },
          variants = {
            {
              arguments = {
                {
                  name = "type",
                  type = "AudioType",
                  description = "The type of device to stop.",
                  default = "'playback'"
                }
              },
              returns = {
                {
                  name = "stopped",
                  type = "boolean",
                  description = "Whether the device was successfully stopped."
                }
              }
            }
          }
        }
      },
      objects = {
        {
          name = "Source",
          summary = "A playable sound object.",
          description = "A Source is an object representing a single sound.  Currently ogg, wav, and mp3 formats are supported.\n\nWhen a Source is playing, it will send audio to the speakers.  Sources do not play automatically when they are created.  Instead, the `play`, `pause`, and `stop` functions can be used to control when they should play.\n\n`Source:seek` and `Source:tell` can be used to control the playback position of the Source.  A Source can be set to loop when it reaches the end using `Source:setLooping`.",
          key = "Source",
          module = "lovr.audio",
          constructors = {
            "lovr.audio.newSource",
            "Source:clone"
          },
          methods = {
            {
              name = "clone",
              tag = "sourceUtility",
              summary = "Create an identical copy of the Source.",
              description = "Creates a copy of the Source, referencing the same `Sound` object and inheriting all of the settings of this Source.  However, it will be created in the stopped state and will be rewound to the beginning.",
              key = "Source:clone",
              module = "lovr.audio",
              notes = "This is a good way to create multiple Sources that play the same sound, since the audio data won't be loaded multiple times and can just be reused.  You can also create multiple `Source` objects and pass in the same `Sound` object for each one, which will have the same effect.",
              related = {
                "lovr.audio.newSource"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "source",
                      type = "Source",
                      description = "A genetically identical copy of the Source."
                    }
                  }
                }
              }
            },
            {
              name = "getDirectivity",
              tag = "sourceEffects",
              summary = "Get the directivity of the Source.",
              description = "Returns the directivity settings for the Source.\n\nThe directivity is controlled by two parameters: the weight and the power.\n\nThe weight is a number between 0 and 1 controlling the general \"shape\" of the sound emitted. 0.0 results in a completely omnidirectional sound that can be heard from all directions.  1.0 results in a full dipole shape that can be heard only from the front and back.  0.5 results in a cardioid shape that can only be heard from one direction.  Numbers in between will smoothly transition between these.\n\nThe power is a number that controls how \"focused\" or sharp the shape is.  Lower power values can be heard from a wider set of angles.  It is an exponent, so it can get arbitrarily large.  Note that a power of zero will still result in an omnidirectional source, regardless of the weight.",
              key = "Source:getDirectivity",
              module = "lovr.audio",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "weight",
                      type = "number",
                      description = "The dipole weight.  0.0 is omnidirectional, 1.0 is a dipole, 0.5 is cardioid."
                    },
                    {
                      name = "power",
                      type = "number",
                      description = "The dipole power, controlling how focused the directivity shape is."
                    }
                  }
                }
              }
            },
            {
              name = "getDuration",
              tag = "sourcePlayback",
              summary = "Get the duration of the Source.",
              description = "Returns the duration of the Source.",
              key = "Source:getDuration",
              module = "lovr.audio",
              related = {
                "Sound:getDuration"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "unit",
                      type = "TimeUnit",
                      description = "The unit to return.",
                      default = "'seconds'"
                    }
                  },
                  returns = {
                    {
                      name = "duration",
                      type = "number",
                      description = "The duration of the Source."
                    }
                  }
                }
              }
            },
            {
              name = "getOrientation",
              tag = "sourceEffects",
              summary = "Get the orientation of the Source.",
              description = "Returns the orientation of the Source, in angle/axis representation.",
              key = "Source:getOrientation",
              module = "lovr.audio",
              related = {
                "Source:getPosition",
                "Source:getPose",
                "lovr.audio.getOrientation"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the Source is rotated around its axis of rotation."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    }
                  }
                }
              }
            },
            {
              name = "getPitch",
              tag = "sourcePlayback",
              summary = "Get the pitch of the Source.",
              description = "Returns the pitch of the Source.",
              key = "Source:getPitch",
              module = "lovr.audio",
              notes = "The default pitch is 1.  Every doubling/halving of the pitch will raise/lower the pitch by one octave.  Changing the pitch also changes the playback speed.",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "pitch",
                      type = "number",
                      description = "The pitch."
                    }
                  }
                }
              }
            },
            {
              name = "getPose",
              tag = "sourceEffects",
              summary = "Get the pose of the Source.",
              description = "Returns the position and orientation of the Source.",
              key = "Source:getPose",
              module = "lovr.audio",
              related = {
                "Source:getPosition",
                "Source:getOrientation",
                "lovr.audio.getPose"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x position of the Source, in meters."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y position of the Source, in meters."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z position of the Source, in meters."
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the Source is rotated around its axis of rotation."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    }
                  }
                }
              }
            },
            {
              name = "getPosition",
              tag = "sourceEffects",
              summary = "Get the position of the Source.",
              description = "Returns the position of the Source, in meters.  Setting the position will cause the Source to be distorted and attenuated based on its position relative to the listener.",
              key = "Source:getPosition",
              module = "lovr.audio",
              related = {
                "Source:getOrientation",
                "Source:getPose",
                "lovr.audio.getPosition"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate."
                    }
                  }
                }
              }
            },
            {
              name = "getRadius",
              tag = "sourceEffects",
              summary = "Get the radius of the Source.",
              description = "Returns the radius of the Source, in meters.\n\nThis does not control falloff or attenuation.  It is only used for smoothing out occlusion.  If a Source doesn't have a radius, then when it becomes occluded by a wall its volume will instantly drop.  Giving the Source a radius that approximates its emitter's size will result in a smooth transition between audible and occluded, improving realism.",
              key = "Source:getRadius",
              module = "lovr.audio",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "radius",
                      type = "number",
                      description = "The radius of the Source, in meters."
                    }
                  }
                }
              }
            },
            {
              name = "getSound",
              tag = "sourceUtility",
              summary = "Get the Sound object backing the Source.",
              description = "Returns the `Sound` object backing the Source.  Multiple Sources can share one Sound, allowing its data to only be loaded once.  An easy way to do this sharing is by using `Source:clone`.",
              key = "Source:getSound",
              module = "lovr.audio",
              related = {
                "Source:clone",
                "lovr.audio.newSource"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "sound",
                      type = "Sound",
                      description = "The Sound object."
                    }
                  }
                }
              }
            },
            {
              name = "getVolume",
              tag = "sourcePlayback",
              summary = "Get the volume of the Source.",
              description = "Returns the current volume factor for the Source.",
              key = "Source:getVolume",
              module = "lovr.audio",
              variants = {
                {
                  arguments = {
                    {
                      name = "units",
                      type = "VolumeUnit",
                      description = "The units to return (linear or db).",
                      default = "'linear'"
                    }
                  },
                  returns = {
                    {
                      name = "volume",
                      type = "number",
                      description = "The volume of the Source."
                    }
                  }
                }
              }
            },
            {
              name = "isEffectEnabled",
              tag = "sourceEffects",
              summary = "Check if an effect is enabled.",
              description = "Returns whether a given `Effect` is enabled for the Source.",
              key = "Source:isEffectEnabled",
              module = "lovr.audio",
              notes = "The active spatializer will determine which effects are supported.  If an unsupported effect is enabled on a Source, no error will be reported.  Instead, it will be silently ignored.  See `lovr.audio.getSpatializer` for a table showing the effects supported by each spatializer.\n\nCalling this function on a non-spatial Source will always return false.",
              related = {
                "Source:isSpatial"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "effect",
                      type = "Effect",
                      description = "The effect."
                    }
                  },
                  returns = {
                    {
                      name = "enabled",
                      type = "boolean",
                      description = "Whether the effect is enabled."
                    }
                  }
                }
              }
            },
            {
              name = "isLooping",
              tag = "sourcePlayback",
              summary = "Check if the Source is looping.",
              description = "Returns whether or not the Source will loop when it finishes.",
              key = "Source:isLooping",
              module = "lovr.audio",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "looping",
                      type = "boolean",
                      description = "Whether or not the Source is looping."
                    }
                  }
                }
              }
            },
            {
              name = "isPlaying",
              tag = "sourcePlayback",
              summary = "Check if the Source is playing.",
              description = "Returns whether or not the Source is playing.",
              key = "Source:isPlaying",
              module = "lovr.audio",
              related = {
                "Source:play",
                "Source:pause",
                "Source:stop"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "playing",
                      type = "boolean",
                      description = "Whether the Source is playing."
                    }
                  }
                }
              }
            },
            {
              name = "isSpatial",
              tag = "sourceEffects",
              summary = "Check if the Source is spatial.",
              description = "Returns whether the Source was created with the `spatial` flag.  Non-spatial sources are routed directly to the speakers and can not use effects.",
              key = "Source:isSpatial",
              module = "lovr.audio",
              related = {
                "Source:isEffectEnabled",
                "Source:setEffectEnabled"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "spatial",
                      type = "boolean",
                      description = "Whether the source is spatial."
                    }
                  }
                }
              }
            },
            {
              name = "pause",
              tag = "sourcePlayback",
              summary = "Pause the Source.",
              description = "Pauses the source.  It can be resumed with `Source:resume` or `Source:play`. If a paused source is rewound, it will remain paused.",
              key = "Source:pause",
              module = "lovr.audio",
              variants = {
                {
                  arguments = {},
                  returns = {}
                }
              }
            },
            {
              name = "play",
              tag = "sourcePlayback",
              summary = "Play the Source.",
              description = "Plays the Source.  This doesn't do anything if the Source is already playing.",
              key = "Source:play",
              module = "lovr.audio",
              notes = "There is a maximum of 64 Sources that can be playing at once.  If 64 Sources are already playing, this function will return `false`.",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "success",
                      type = "boolean",
                      description = "Whether the Source successfully started playing."
                    }
                  }
                }
              }
            },
            {
              name = "seek",
              tag = "sourcePlayback",
              summary = "Set the playback position of the Source.",
              description = "Seeks the Source to the specified position.",
              key = "Source:seek",
              module = "lovr.audio",
              notes = "Seeking a Source backed by a stream `Sound` has no meaningful effect.",
              variants = {
                {
                  arguments = {
                    {
                      name = "position",
                      type = "number",
                      description = "The position to seek to."
                    },
                    {
                      name = "unit",
                      type = "TimeUnit",
                      description = "The units for the seek position.",
                      default = "'seconds'"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setDirectivity",
              tag = "sourceEffects",
              summary = "Set the directivity of the Source.",
              description = "Sets the directivity settings for the Source.\n\nThe directivity is controlled by two parameters: the weight and the power.\n\nThe weight is a number between 0 and 1 controlling the general \"shape\" of the sound emitted. 0.0 results in a completely omnidirectional sound that can be heard from all directions.  1.0 results in a full dipole shape that can be heard only from the front and back.  0.5 results in a cardioid shape that can only be heard from one direction.  Numbers in between will smoothly transition between these.\n\nThe power is a number that controls how \"focused\" or sharp the shape is.  Lower power values can be heard from a wider set of angles.  It is an exponent, so it can get arbitrarily large.  Note that a power of zero will still result in an omnidirectional source, regardless of the weight.",
              key = "Source:setDirectivity",
              module = "lovr.audio",
              variants = {
                {
                  arguments = {
                    {
                      name = "weight",
                      type = "number",
                      description = "The dipole weight.  0.0 is omnidirectional, 1.0 is a dipole, 0.5 is cardioid."
                    },
                    {
                      name = "power",
                      type = "number",
                      description = "The dipole power, controlling how focused the directivity shape is."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setEffectEnabled",
              tag = "sourceEffects",
              summary = "Enable or disable an effect.",
              description = "Enables or disables an effect on the Source.",
              key = "Source:setEffectEnabled",
              module = "lovr.audio",
              notes = "The active spatializer will determine which effects are supported.  If an unsupported effect is enabled on a Source, no error will be reported.  Instead, it will be silently ignored.  See `lovr.audio.getSpatializer` for a table showing the effects supported by each spatializer.\n\nCalling this function on a non-spatial Source will throw an error.",
              related = {
                "Source:isSpatial"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "effect",
                      type = "Effect",
                      description = "The effect."
                    },
                    {
                      name = "enable",
                      type = "boolean",
                      description = "Whether the effect should be enabled."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setLooping",
              tag = "sourcePlayback",
              summary = "Set whether or not the Source loops.",
              description = "Sets whether or not the Source loops.",
              key = "Source:setLooping",
              module = "lovr.audio",
              notes = "Attempting to loop a Source backed by a stream `Sound` will cause an error.",
              variants = {
                {
                  arguments = {
                    {
                      name = "loop",
                      type = "boolean",
                      description = "Whether or not the Source will loop."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setOrientation",
              tag = "sourceEffects",
              summary = "Set the orientation of the Source.",
              description = "Sets the orientation of the Source in angle/axis representation.",
              key = "Source:setOrientation",
              module = "lovr.audio",
              related = {
                "Source:setPosition",
                "Source:setPose",
                "lovr.audio.setOrientation"
              },
              variants = {
                {
                  description = "Set the orientation using angle/axis numbers.",
                  arguments = {
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the Source should be rotated around its rotation axis."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    }
                  },
                  returns = {}
                },
                {
                  description = "Set the orientation using a quaternion.",
                  arguments = {
                    {
                      name = "orientation",
                      type = "Quat",
                      description = "The orientation."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setPitch",
              tag = "sourcePlayback",
              summary = "Set the pitch of the Source.",
              description = "Sets the pitch of the Source.",
              key = "Source:setPitch",
              module = "lovr.audio",
              notes = "The default pitch is 1.  Every doubling/halving of the pitch will raise/lower the pitch by one octave.  Changing the pitch also changes the playback speed.",
              variants = {
                {
                  arguments = {
                    {
                      name = "pitch",
                      type = "number",
                      description = "The new pitch."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setPose",
              tag = "sourceEffects",
              summary = "Set the pose of the Source.",
              description = "Sets the position and orientation of the Source.",
              key = "Source:setPose",
              module = "lovr.audio",
              notes = "The position doesn't have any defined units, but meters are used by convention.",
              related = {
                "Source:setPosition",
                "Source:setOrientation",
                "lovr.audio.setPose"
              },
              variants = {
                {
                  description = "Set the pose using numbers.",
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x position of the Source."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y position of the Source."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z position of the Source."
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the Source is rotated around its axis of rotation."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    }
                  },
                  returns = {}
                },
                {
                  description = "Set the pose using vector types.",
                  arguments = {
                    {
                      name = "position",
                      type = "Vec3",
                      description = "The position."
                    },
                    {
                      name = "orientation",
                      type = "Quat",
                      description = "The orientation."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setPosition",
              tag = "sourceEffects",
              summary = "Set the position of the Source.",
              description = "Sets the position of the Source.  Setting the position will cause the Source to be distorted and attenuated based on its position relative to the listener.\n\nOnly mono sources can be positioned.  Setting the position of a stereo Source will cause an error.",
              key = "Source:setPosition",
              module = "lovr.audio",
              notes = "The position doesn't have any defined units, but meters are used by convention.",
              related = {
                "Source:setOrientation",
                "Source:setPose"
              },
              variants = {
                {
                  description = "Set the position using numbers.",
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate of the position."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate of the position."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate of the position."
                    }
                  },
                  returns = {}
                },
                {
                  description = "Set the position using a vector.",
                  arguments = {
                    {
                      name = "position",
                      type = "Vec3",
                      description = "The position."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setRadius",
              tag = "sourceEffects",
              summary = "Set the radius of the Source.",
              description = "Sets the radius of the Source, in meters.\n\nThis does not control falloff or attenuation.  It is only used for smoothing out occlusion.  If a Source doesn't have a radius, then when it becomes occluded by a wall its volume will instantly drop.  Giving the Source a radius that approximates its emitter's size will result in a smooth transition between audible and occluded, improving realism.",
              key = "Source:setRadius",
              module = "lovr.audio",
              variants = {
                {
                  arguments = {
                    {
                      name = "radius",
                      type = "number",
                      description = "The new radius of the Source, in meters."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setVolume",
              tag = "sourcePlayback",
              summary = "Set the volume of the Source.",
              description = "Sets the current volume factor for the Source.",
              key = "Source:setVolume",
              module = "lovr.audio",
              notes = "The volume will be clamped to a 0-1 range (0 dB).",
              variants = {
                {
                  arguments = {
                    {
                      name = "volume",
                      type = "number",
                      description = "The new volume."
                    },
                    {
                      name = "units",
                      type = "VolumeUnit",
                      description = "The units of the value.",
                      default = "'linear'"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "stop",
              tag = "sourcePlayback",
              summary = "Stop the Source.",
              description = "Stops the source, also rewinding it to the beginning.",
              key = "Source:stop",
              module = "lovr.audio",
              related = {
                "Source:play",
                "Source:pause",
                "Source:isPlaying"
              },
              variants = {
                {
                  arguments = {},
                  returns = {}
                }
              }
            },
            {
              name = "tell",
              tag = "sourcePlayback",
              summary = "Get the playback position of the Source.",
              description = "Returns the current playback position of the Source.",
              key = "Source:tell",
              module = "lovr.audio",
              notes = "The return value for Sources backed by a stream `Sound` has no meaning.",
              variants = {
                {
                  arguments = {
                    {
                      name = "unit",
                      type = "TimeUnit",
                      description = "The unit to return.",
                      default = "'seconds'"
                    }
                  },
                  returns = {
                    {
                      name = "position",
                      type = "number",
                      description = "The current playback position."
                    }
                  }
                }
              }
            }
          },
          sections = {
            {
              name = "Playback",
              tag = "sourcePlayback"
            },
            {
              name = "Spatial Effects",
              tag = "sourceEffects"
            },
            {
              name = "Utility",
              tag = "sourceUtility"
            }
          }
        }
      },
      sections = {
        {
          name = "Sources",
          tag = "sources",
          description = "Sources are objects that represent a single sound instance."
        },
        {
          name = "Listener",
          tag = "listener",
          description = "The listener is a virtual object in 3D space that \"hears\" all the sounds that are playing. It can be positioned and oriented in 3D space, which controls how Sources in the world are heard.  Usually this would be locked to the headset pose."
        },
        {
          name = "Devices",
          tag = "devices",
          description = "It's possible to list the available audio devices on the system, and pick a specific device to use for either playback or capture.  Devices can also be manually started and stopped. Other useful features of `lovr.audio.setDevice` include the ability to stream all audio data to a custom sink and the option to create a device in exclusive mode for higher performance. By default, the default playback device is automatically initialized and started, but this can be configured using `lovr.conf`."
        }
      }
    },
    {
      name = "data",
      tag = "modules",
      summary = "Exposes low level functions for working with data.",
      description = "The `lovr.data` module provides functions for accessing underlying data representations for several LÖVR objects.",
      key = "lovr.data",
      enums = {
        {
          name = "AnimationProperty",
          summary = "Different animated properties.",
          description = "This indicates the different node properties that can be animated.",
          key = "AnimationProperty",
          module = "lovr.data",
          values = {
            {
              name = "translation",
              description = "Node translation."
            },
            {
              name = "rotation",
              description = "Node rotation."
            },
            {
              name = "scale",
              description = "Node scale."
            },
            {
              name = "weights",
              description = "Node blend shape weights."
            }
          }
        },
        {
          name = "AttributeType",
          summary = "Data types for vertex attributes in meshes.",
          description = "These are the data types that can be used by vertex data in meshes.",
          key = "AttributeType",
          module = "lovr.data",
          values = {
            {
              name = "i8",
              description = "Signed 8 bit integers (-128 to 127)."
            },
            {
              name = "u8",
              description = "Unsigned 8 bit integers (0 to 255)."
            },
            {
              name = "i16",
              description = "Signed 16 bit integers (-32768 to 32767)."
            },
            {
              name = "u16",
              description = "Unsigned 16 bit integers (0 to 65535)."
            },
            {
              name = "i32",
              description = "Signed 32 bit integers (-2147483648 to 2147483647)."
            },
            {
              name = "u32",
              description = "Unsigned 32 bit integers (0 to 429467295)."
            },
            {
              name = "f32",
              description = "Floating point numbers."
            }
          }
        },
        {
          name = "ChannelLayout",
          summary = "Different channel layouts for Sounds.",
          description = "Sounds can have different numbers of channels, and those channels can map to various speaker layouts.",
          key = "ChannelLayout",
          module = "lovr.data",
          related = {
            "lovr.data.newSound",
            "Sound:getFormat"
          },
          values = {
            {
              name = "mono",
              description = "1 channel."
            },
            {
              name = "stereo",
              description = "2 channels.  The first channel is for the left speaker and the second is for the right."
            },
            {
              name = "ambisonic",
              description = "4 channels.  Ambisonic channels don't map directly to speakers but instead represent directions in 3D space, sort of like the images of a skybox.  Currently, ambisonic sounds can only be loaded, not played."
            }
          }
        },
        {
          name = "DefaultAttribute",
          summary = "Attributes that can be loaded from a model.",
          description = "These are the different types of attributes that may be present in meshes loaded from models.",
          key = "DefaultAttribute",
          module = "lovr.data",
          values = {
            {
              name = "position",
              description = "Vertex positions."
            },
            {
              name = "normal",
              description = "Vertex normal vectors."
            },
            {
              name = "uv",
              description = "Vertex texture coordinates."
            },
            {
              name = "color",
              description = "Vertex colors."
            },
            {
              name = "tangent",
              description = "Vertex tangent vectors."
            },
            {
              name = "joints",
              description = "Vertex joint indices."
            },
            {
              name = "weights",
              description = "Vertex joint weights."
            }
          }
        },
        {
          name = "ModelDrawMode",
          summary = "Different draw modes for meshes in ModelDatas.",
          description = "The DrawMode of a mesh determines how its vertices are connected together.",
          key = "ModelDrawMode",
          module = "lovr.data",
          related = {
            "ModelData:getMeshDrawMode"
          },
          values = {
            {
              name = "points",
              description = "Each vertex is draw as a single point."
            },
            {
              name = "lines",
              description = "Every pair of vertices is drawn as a line."
            },
            {
              name = "linestrip",
              description = "Draws a single line through all of the vertices."
            },
            {
              name = "lineloop",
              description = "Draws a single line through all of the vertices, then connects back to the first vertex."
            },
            {
              name = "strip",
              description = "Vertices are rendered as triangles.  After the first 3 vertices, each subsequent vertex connects to the previous two."
            },
            {
              name = "triangles",
              description = "Every 3 vertices forms a triangle."
            },
            {
              name = "fan",
              description = "Vertices are rendered as triangles.  After the first 3 vertices, each subsequent vertex is connected to the previous vertex and the first vertex."
            }
          }
        },
        {
          name = "SampleFormat",
          summary = "Different data types for samples in a Sound.",
          description = "Sounds can store audio samples as 16 bit integers or 32 bit floats.",
          key = "SampleFormat",
          module = "lovr.data",
          related = {
            "lovr.data.newSound",
            "Sound:getFormat"
          },
          values = {
            {
              name = "f32",
              description = "32 bit floating point samples (between -1.0 and 1.0)."
            },
            {
              name = "i16",
              description = "16 bit integer samples (between -32768 and 32767)."
            }
          }
        },
        {
          name = "SmoothMode",
          summary = "Different ways to interpolate between animation keyframes.",
          description = "Different ways to interpolate between animation keyframes.",
          key = "SmoothMode",
          module = "lovr.data",
          values = {
            {
              name = "step",
              description = "The animated property will snap to the nearest keyframe."
            },
            {
              name = "linear",
              description = "The animated property will linearly interpolate between keyframes."
            },
            {
              name = "cubic",
              description = "The animated property will follow a smooth curve between nearby keyframes."
            }
          }
        },
        {
          name = "TextureFormat",
          description = "Different data layouts for pixels in `Image` and `Texture` objects.\n\nFormats starting with `d` are depth formats, used for depth/stencil render targets.\n\nFormats starting with `bc` and `astc` are compressed formats.  Compressed formats have better performance since they stay compressed on the CPU and GPU, reducing the amount of memory bandwidth required to look up all the pixels needed for shading.\n\nFormats without the `f` suffix are unsigned normalized formats, which store values in the range `[0,1]`.  The `f` suffix indicates a floating point format which can store values outside this range, and is used for HDR rendering or storing data in a texture.",
          key = "TextureFormat",
          module = "lovr.data",
          values = {
            {
              name = "r8",
              description = "One 8-bit channel.  1 byte per pixel."
            },
            {
              name = "rg8",
              description = "Two 8-bit channels.  2 bytes per pixel."
            },
            {
              name = "rgba8",
              description = "Four 8-bit channels.  4 bytes per pixel."
            },
            {
              name = "r16",
              description = "One 16-bit channel.  2 bytes per pixel."
            },
            {
              name = "rg16",
              description = "Two 16-bit channels.  4 bytes per pixel."
            },
            {
              name = "rgba16",
              description = "Four 16-bit channels.  8 bytes per pixel."
            },
            {
              name = "r16f",
              description = "One 16-bit floating point channel.  2 bytes per pixel."
            },
            {
              name = "rg16f",
              description = "Two 16-bit floating point channels.  4 bytes per pixel."
            },
            {
              name = "rgba16f",
              description = "Four 16-bit floating point channels.  8 bytes per pixel."
            },
            {
              name = "r32f",
              description = "One 32-bit floating point channel.  4 bytes per pixel."
            },
            {
              name = "rg32f",
              description = "Two 32-bit floating point channels.  8 bytes per pixel."
            },
            {
              name = "rgba32f",
              description = "Four 32-bit floating point channels.  16 bytes per pixel."
            },
            {
              name = "rgb565",
              description = "Packs three channels into 16 bits.  2 bytes per pixel."
            },
            {
              name = "rgb5a1",
              description = "Packs four channels into 16 bits, with \"cutout\" alpha.  2 bytes per pixel."
            },
            {
              name = "rgb10a2",
              description = "Packs four channels into 32 bits.  4 bytes per pixel."
            },
            {
              name = "rg11b10f",
              description = "Packs three unsigned floating point channels into 32 bits.  4 bytes per pixel."
            },
            {
              name = "d16",
              description = "One 16-bit depth channel.  2 bytes per pixel."
            },
            {
              name = "d24s8",
              description = "One 24-bit depth channel and one 8-bit stencil channel.  4 bytes per pixel."
            },
            {
              name = "d32f",
              description = "One 32-bit floating point depth channel.  4 bytes per pixel."
            },
            {
              name = "d32fs8",
              description = "One 32-bit floating point depth channel and one 8-bit stencil channel.  5 bytes per pixel."
            },
            {
              name = "bc1",
              description = "3 channels.  8 bytes per 4x4 block, or 0.5 bytes per pixel.  Good for opaque images."
            },
            {
              name = "bc2",
              description = "Four channels.  16 bytes per 4x4 block or 1 byte per pixel.  Not good for anything, because it only has 16 distinct levels of alpha."
            },
            {
              name = "bc3",
              description = "Four channels.  16 bytes per 4x4 block or 1 byte per pixel.  Good for color images with transparency."
            },
            {
              name = "bc4u",
              description = "One unsigned normalized channel.  8 bytes per 4x4 block or 0.5 bytes per pixel.  Good for grayscale images, like heightmaps."
            },
            {
              name = "bc4s",
              description = "One signed normalized channel.  8 bytes per 4x4 block or 0.5 bytes per pixel.  Similar to bc4u but has a range of -1 to 1."
            },
            {
              name = "bc5u",
              description = "Two unsigned normalized channels.  16 bytes per 4x4 block, or 1 byte per pixel.  Good for normal maps."
            },
            {
              name = "bc5s",
              description = "Two signed normalized channels.  16 bytes per 4x4 block or 1 byte per pixel.  Good for normal maps."
            },
            {
              name = "bc6uf",
              description = "Three unsigned floating point channels.  16 bytes per 4x4 block or 1 byte per pixel.  Good for HDR images."
            },
            {
              name = "bc6sf",
              description = "Three floating point channels.  16 bytes per 4x4 block or 1 byte per pixel.  Good for HDR images."
            },
            {
              name = "bc7",
              description = "Four channels.  16 bytes per 4x4 block or 1 byte per pixel.  High quality.  Good for most color images, including transparency."
            },
            {
              name = "astc4x4",
              description = "Four channels, 16 bytes per 4x4 block or 1 byte per pixel."
            },
            {
              name = "astc5x4",
              description = "Four channels, 16 bytes per 5x4 block or 0.80 bytes per pixel."
            },
            {
              name = "astc5x5",
              description = "Four channels, 16 bytes per 5x5 block or 0.64 bytes per pixel."
            },
            {
              name = "astc6x5",
              description = "Four channels, 16 bytes per 6x5 block or 0.53 bytes per pixel."
            },
            {
              name = "astc6x6",
              description = "Four channels, 16 bytes per 6x6 block or 0.44 bytes per pixel."
            },
            {
              name = "astc8x5",
              description = "Four channels, 16 bytes per 8x5 block or 0.40 bytes per pixel."
            },
            {
              name = "astc8x6",
              description = "Four channels, 16 bytes per 8x6 block or 0.33 bytes per pixel."
            },
            {
              name = "astc8x8",
              description = "Four channels, 16 bytes per 8x8 block or 0.25 bytes per pixel."
            },
            {
              name = "astc10x5",
              description = "Four channels, 16 bytes per 10x5 block or 0.32 bytes per pixel."
            },
            {
              name = "astc10x6",
              description = "Four channels, 16 bytes per 10x6 block or 0.27 bytes per pixel."
            },
            {
              name = "astc10x8",
              description = "Four channels, 16 bytes per 10x8 block or 0.20 bytes per pixel."
            },
            {
              name = "astc10x10",
              description = "Four channels, 16 bytes per 10x10 block or 0.16 bytes per pixel."
            },
            {
              name = "astc12x10",
              description = "Four channels, 16 bytes per 12x10 block or 0.13 bytes per pixel."
            },
            {
              name = "astc12x12",
              description = "Four channels, 16 bytes per 12x12 block or 0.11 bytes per pixel."
            }
          }
        }
      },
      functions = {
        {
          name = "newBlob",
          summary = "Create a new Blob.",
          description = "Creates a new Blob.",
          key = "lovr.data.newBlob",
          module = "lovr.data",
          related = {
            "lovr.filesystem.newBlob"
          },
          variants = {
            {
              arguments = {
                {
                  name = "size",
                  type = "number",
                  description = "The amount of data to allocate for the Blob, in bytes.  All of the bytes will be filled with zeroes."
                },
                {
                  name = "name",
                  type = "string",
                  description = "A name for the Blob (used in error messages)",
                  default = "''"
                }
              },
              returns = {
                {
                  name = "blob",
                  type = "Blob",
                  description = "The new Blob."
                }
              }
            },
            {
              arguments = {
                {
                  name = "contents",
                  type = "string",
                  description = "A string to use for the Blob's contents."
                },
                {
                  name = "name",
                  type = "string",
                  description = "A name for the Blob (used in error messages)",
                  default = "''"
                }
              },
              returns = {
                {
                  name = "blob",
                  type = "Blob",
                  description = "The new Blob."
                }
              }
            },
            {
              arguments = {
                {
                  name = "source",
                  type = "Blob",
                  description = "A Blob to copy the contents from."
                },
                {
                  name = "name",
                  type = "string",
                  description = "A name for the Blob (used in error messages)",
                  default = "''"
                }
              },
              returns = {
                {
                  name = "blob",
                  type = "Blob",
                  description = "The new Blob."
                }
              }
            }
          }
        },
        {
          name = "newImage",
          summary = "Create a new Image.",
          description = "Creates a new Image.  Image data can be loaded and decoded from an image file, or a raw block of pixels with a specified width, height, and format can be created.",
          key = "lovr.data.newImage",
          module = "lovr.data",
          notes = "The supported image file formats are png, jpg, hdr, dds (DXT1, DXT3, DXT5), ktx, and astc.\n\nOnly 2D textures are supported for DXT/ASTC.\n\nCurrently textures loaded as KTX need to be in DXT/ASTC formats.",
          variants = {
            {
              description = "Load image data from a file.",
              arguments = {
                {
                  name = "filename",
                  type = "string",
                  description = "The filename of the image to load."
                },
                {
                  name = "flip",
                  type = "boolean",
                  description = "Whether to vertically flip the image on load.  This should be true for normal textures, and false for textures that are going to be used in a cubemap.",
                  default = "true"
                }
              },
              returns = {
                {
                  name = "image",
                  type = "Image",
                  description = "The new Image."
                }
              }
            },
            {
              description = "Create an Image with a given size and pixel format.",
              arguments = {
                {
                  name = "width",
                  type = "number",
                  description = "The width of the texture."
                },
                {
                  name = "height",
                  type = "number",
                  description = "The height of the texture."
                },
                {
                  name = "format",
                  type = "TextureFormat",
                  description = "The format of the texture's pixels.",
                  default = "rgba8"
                },
                {
                  name = "data",
                  type = "Blob",
                  description = "Raw pixel values to use as the contents.  If `nil`, the data will all be zero.",
                  default = "nil"
                }
              },
              returns = {
                {
                  name = "image",
                  type = "Image",
                  description = "The new Image."
                }
              }
            },
            {
              description = "Clone an existing Image.",
              arguments = {
                {
                  name = "source",
                  type = "Image",
                  description = "The Image to clone."
                }
              },
              returns = {
                {
                  name = "image",
                  type = "Image",
                  description = "The new Image."
                }
              }
            },
            {
              description = "Decode image data from a Blob.",
              arguments = {
                {
                  name = "blob",
                  type = "Blob",
                  description = "The Blob containing image data to decode."
                },
                {
                  name = "flip",
                  type = "boolean",
                  description = "Whether to vertically flip the image on load.  This should be true for normal textures, and false for textures that are going to be used in a cubemap.",
                  default = "true"
                }
              },
              returns = {
                {
                  name = "image",
                  type = "Image",
                  description = "The new Image."
                }
              }
            }
          }
        },
        {
          name = "newModelData",
          summary = "Create a new ModelData.",
          description = "Loads a 3D model from a file.  The supported 3D file formats are OBJ and glTF.",
          key = "lovr.data.newModelData",
          module = "lovr.data",
          variants = {
            {
              arguments = {
                {
                  name = "filename",
                  type = "string",
                  description = "The filename of the model to load."
                }
              },
              returns = {
                {
                  name = "modelData",
                  type = "ModelData",
                  description = "The new ModelData."
                }
              }
            },
            {
              arguments = {
                {
                  name = "blob",
                  type = "Blob",
                  description = "The Blob containing data for a model to decode."
                }
              },
              returns = {
                {
                  name = "modelData",
                  type = "ModelData",
                  description = "The new ModelData."
                }
              }
            }
          }
        },
        {
          name = "newRasterizer",
          summary = "Create a new Rasterizer.",
          description = "Creates a new Rasterizer from a TTF file.",
          key = "lovr.data.newRasterizer",
          module = "lovr.data",
          variants = {
            {
              description = "Create a Rasterizer for the default font included with LÖVR (Varela Round).",
              arguments = {
                {
                  name = "size",
                  type = "number",
                  description = "The resolution to render the fonts at, in pixels.  Higher resolutions use more memory and processing power but may provide better quality results for some fonts/situations.",
                  default = "32"
                }
              },
              returns = {
                {
                  name = "rasterizer",
                  type = "Rasterizer",
                  description = "The new Rasterizer."
                }
              }
            },
            {
              arguments = {
                {
                  name = "filename",
                  type = "string",
                  description = "The filename of the font file to load."
                },
                {
                  name = "size",
                  type = "number",
                  description = "The resolution to render the fonts at, in pixels.  Higher resolutions use more memory and processing power but may provide better quality results for some fonts/situations.",
                  default = "32"
                }
              },
              returns = {
                {
                  name = "rasterizer",
                  type = "Rasterizer",
                  description = "The new Rasterizer."
                }
              }
            },
            {
              arguments = {
                {
                  name = "blob",
                  type = "Blob",
                  description = "The Blob containing font data."
                },
                {
                  name = "size",
                  type = "number",
                  description = "The resolution to render the fonts at, in pixels.  Higher resolutions use more memory and processing power but may provide better quality results for some fonts/situations.",
                  default = "32"
                }
              },
              returns = {
                {
                  name = "rasterizer",
                  type = "Rasterizer",
                  description = "The new Rasterizer."
                }
              }
            }
          }
        },
        {
          name = "newSound",
          summary = "Create a new Sound.",
          description = "Creates a new Sound.  A sound can be loaded from an audio file, or it can be created empty with capacity for a certain number of audio frames.\n\nWhen loading audio from a file, use the `decode` option to control whether compressed audio should remain compressed or immediately get decoded to raw samples.\n\nWhen creating an empty sound, the `contents` parameter can be set to `'stream'` to create an audio stream.  On streams, `Sound:setFrames` will always write to the end of the stream, and `Sound:getFrames` will always read the oldest samples from the beginning.  The number of frames in the sound is the total capacity of the stream's buffer.",
          key = "lovr.data.newSound",
          module = "lovr.data",
          notes = "It is highly recommended to use an audio format that matches the format of the audio module: `f32` sample formats at a sample rate of 48000, with 1 channel for spatialized sources or 2 channels for unspatialized sources.  This will avoid the need to convert audio during playback, which boosts performance of the audio thread.\n\nThe WAV importer supports 16, 24, and 32 bit integer data and 32 bit floating point data.  The data must be mono, stereo, or 4-channel full-sphere ambisonic.  The `WAVE_FORMAT_EXTENSIBLE` extension is supported.\n\nAmbisonic channel layouts are supported for import (but not yet for playback).  Ambisonic data can be loaded from WAV files.  It must be first-order full-sphere ambisonic data with 4 channels.  If the WAV has a `WAVE_FORMAT_EXTENSIBLE` chunk with an `AMBISONIC_B_FORMAT` format GUID, then the data is understood as using the AMB format with Furse-Malham channel ordering and normalization.  *All other* 4-channel files are assumed to be using the AmbiX format with ACN channel ordering and SN3D normalization.  AMB files will get automatically converted to AmbiX on import, so ambisonic Sounds will always be in a consistent format.\n\nOGG and MP3 files will always have the `f32` format when loaded.",
          variants = {
            {
              description = "Create a raw or stream Sound from a frame count and format info:",
              arguments = {
                {
                  name = "frames",
                  type = "number",
                  description = "The number of frames the Sound can hold."
                },
                {
                  name = "format",
                  type = "SampleFormat",
                  description = "The sample data type.",
                  default = "'f32'"
                },
                {
                  name = "channels",
                  type = "ChannelLayout",
                  description = "The channel layout.",
                  default = "'stereo'"
                },
                {
                  name = "sampleRate",
                  type = "number",
                  description = "The sample rate, in Hz.",
                  default = "48000"
                },
                {
                  name = "contents",
                  type = "*",
                  description = "A Blob containing raw audio samples to use as the initial contents, 'stream' to create an audio stream, or `nil` to leave the data initialized to zero.",
                  default = "nil"
                }
              },
              returns = {
                {
                  name = "sound",
                  type = "Sound",
                  description = "Sounds good."
                }
              }
            },
            {
              description = "Load a sound from a file.  Compressed audio formats (OGG, MP3) can optionally be decoded into raw sounds.",
              arguments = {
                {
                  name = "filename",
                  type = "string",
                  description = "The filename of a sound to load."
                },
                {
                  name = "decode",
                  type = "boolean",
                  description = "Whether compressed audio files should be immediately decoded."
                }
              },
              returns = {
                {
                  name = "sound",
                  type = "Sound",
                  description = "Sounds good."
                }
              }
            },
            {
              description = "Load a sound from a Blob containing the data of an audio file.  Compressed audio formats (OGG, MP3) can optionally be decoded into raw sounds.\n\nIf the Blob contains raw audio samples, use the first variant instead of this one.",
              arguments = {
                {
                  name = "blob",
                  type = "Blob",
                  description = "The Blob containing audio file data to load."
                },
                {
                  name = "decode",
                  type = "boolean",
                  description = "Whether compressed audio files should be immediately decoded."
                }
              },
              returns = {
                {
                  name = "sound",
                  type = "Sound",
                  description = "Sounds good."
                }
              }
            }
          }
        }
      },
      objects = {
        {
          name = "Blob",
          summary = "A chunk of binary data.",
          description = "A Blob is an object that holds binary data.  It can be passed to most functions that take filename arguments, like `lovr.graphics.newModel` or `lovr.audio.newSource`.  Blobs aren't usually necessary for simple projects, but they can be really helpful if:\n\n- You need to work with low level binary data, potentially using the LuaJIT FFI for increased\n  performance.\n- You are working with data that isn't stored as a file, such as programmatically generated data\n  or a string from a network request.\n- You want to load data from a file once and then use it to create many different objects.\n\nA Blob's size cannot be changed once it is created.",
          key = "Blob",
          module = "lovr.data",
          constructors = {
            "lovr.data.newBlob",
            "lovr.filesystem.newBlob"
          },
          methods = {
            {
              name = "getSize",
              summary = "Get the size of the Blob, in bytes.",
              description = "Returns the size of the Blob's contents, in bytes.",
              key = "Blob:getSize",
              module = "lovr.data",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "bytes",
                      type = "number",
                      description = "The size of the Blob, in bytes."
                    }
                  }
                }
              }
            },
            {
              name = "getName",
              summary = "Get the label of the Blob.",
              description = "Returns the filename the Blob was loaded from, or the custom name given to it when it was created.  This label is also used in error messages.",
              key = "Blob:getName",
              module = "lovr.data",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the Blob."
                    }
                  }
                }
              }
            },
            {
              name = "getPointer",
              summary = "Get a raw pointer to the Blob's data.",
              description = "Returns a raw pointer to the Blob's data.  This can be used to interface with other C libraries using the LuaJIT FFI.  Use this only if you know what you're doing!",
              key = "Blob:getPointer",
              module = "lovr.data",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "pointer",
                      type = "userdata",
                      description = "A pointer to the data."
                    }
                  }
                }
              }
            },
            {
              name = "getString",
              summary = "Get the Blob's contents as a string.",
              description = "Returns a binary string containing the Blob's data.",
              key = "Blob:getString",
              module = "lovr.data",
              examples = {
                {
                  description = "Print each byte of the main.lua file:",
                  code = "blob = lovr.filesystem.newBlob('main.lua')\nstr = blob:getString()\n\nfor i = 1, #str do\n  print(string.byte(str, i))\nend"
                }
              },
              notes = "This effectively allocates a new copy of the Blob as a Lua string, so this should be avoided for really big Blobs!",
              variants = {
                {
                  arguments = {
                    {
                      name = "offset",
                      type = "number",
                      description = "A byte offset into the Blob where the string will start.",
                      default = "0"
                    },
                    {
                      name = "size",
                      type = "number",
                      description = "The number of bytes the string will contain.  If nil, the rest of the data in the Blob will be used, based on the `offset` parameter.",
                      default = "nil"
                    }
                  },
                  returns = {
                    {
                      name = "data",
                      type = "string",
                      description = "The Blob's data."
                    }
                  }
                }
              }
            },
            {
              name = "getI8",
              summary = "Unpack signed 8-bit integers from the Blob.",
              description = "Returns signed 8-bit integers from the data in the Blob.",
              key = "Blob:getI8",
              module = "lovr.data",
              related = {
                "Blob:getU8",
                "Blob:getI16",
                "Blob:getU16",
                "Blob:getI32",
                "Blob:getU32",
                "Blob:getF32",
                "Blob:getF64"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "offset",
                      type = "number",
                      description = "A non-negative byte offset to read from.",
                      default = "0"
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "The number of integers to read.",
                      default = "1"
                    }
                  },
                  returns = {
                    {
                      name = "...",
                      type = "number",
                      description = "`count` signed 8-bit integers, from -128 to 127."
                    }
                  }
                }
              }
            },
            {
              name = "getU8",
              summary = "Unpack unsigned 8-bit integers from the Blob.",
              description = "Returns unsigned 8-bit integers from the data in the Blob.",
              key = "Blob:getU8",
              module = "lovr.data",
              related = {
                "Blob:getI8",
                "Blob:getI16",
                "Blob:getU16",
                "Blob:getI32",
                "Blob:getU32",
                "Blob:getF32",
                "Blob:getF64"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "offset",
                      type = "number",
                      description = "A non-negative byte offset to read from.",
                      default = "0"
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "The number of integers to read.",
                      default = "1"
                    }
                  },
                  returns = {
                    {
                      name = "...",
                      type = "number",
                      description = "`count` unsigned 8-bit integers, from 0 to 255."
                    }
                  }
                }
              }
            },
            {
              name = "getI16",
              summary = "Unpack signed 16-bit integers from the Blob.",
              description = "Returns signed 16-bit integers from the data in the Blob.",
              key = "Blob:getI16",
              module = "lovr.data",
              related = {
                "Blob:getI8",
                "Blob:getU8",
                "Blob:getU16",
                "Blob:getI32",
                "Blob:getU32",
                "Blob:getF32",
                "Blob:getF64"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "offset",
                      type = "number",
                      description = "A non-negative byte offset to read from.",
                      default = "0"
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "The number of integers to read.",
                      default = "1"
                    }
                  },
                  returns = {
                    {
                      name = "...",
                      type = "number",
                      description = "`count` signed 16-bit integers, from -32768 to 32767."
                    }
                  }
                }
              }
            },
            {
              name = "getU16",
              summary = "Unpack unsigned 16-bit integers from the Blob.",
              description = "Returns unsigned 16-bit integers from the data in the Blob.",
              key = "Blob:getU16",
              module = "lovr.data",
              related = {
                "Blob:getI8",
                "Blob:getU8",
                "Blob:getI16",
                "Blob:getI32",
                "Blob:getU32",
                "Blob:getF32",
                "Blob:getF64"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "offset",
                      type = "number",
                      description = "A non-negative byte offset to read from.",
                      default = "0"
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "The number of integers to read.",
                      default = "1"
                    }
                  },
                  returns = {
                    {
                      name = "...",
                      type = "number",
                      description = "`count` unsigned 16-bit integers, from 0 to 65535."
                    }
                  }
                }
              }
            },
            {
              name = "getI32",
              summary = "Unpack signed 32-bit integers from the Blob.",
              description = "Returns signed 32-bit integers from the data in the Blob.",
              key = "Blob:getI32",
              module = "lovr.data",
              related = {
                "Blob:getI8",
                "Blob:getU8",
                "Blob:getI16",
                "Blob:getU16",
                "Blob:getU32",
                "Blob:getF32",
                "Blob:getF64"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "offset",
                      type = "number",
                      description = "A non-negative byte offset to read from.",
                      default = "0"
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "The number of integers to read.",
                      default = "1"
                    }
                  },
                  returns = {
                    {
                      name = "...",
                      type = "number",
                      description = "`count` signed 32-bit integers, from -2147483648 to 2147483647."
                    }
                  }
                }
              }
            },
            {
              name = "getU32",
              summary = "Unpack unsigned 32-bit integers from the Blob.",
              description = "Returns unsigned 32-bit integers from the data in the Blob.",
              key = "Blob:getU32",
              module = "lovr.data",
              related = {
                "Blob:getI8",
                "Blob:getU8",
                "Blob:getI16",
                "Blob:getU16",
                "Blob:getI32",
                "Blob:getF32",
                "Blob:getF64"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "offset",
                      type = "number",
                      description = "A non-negative byte offset to read from.",
                      default = "0"
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "The number of integers to read.",
                      default = "1"
                    }
                  },
                  returns = {
                    {
                      name = "...",
                      type = "number",
                      description = "`count` unsigned 32-bit integers, from 0 to 4294967296."
                    }
                  }
                }
              }
            },
            {
              name = "getF32",
              summary = "Unpack 32-bit floating point numbers from the Blob.",
              description = "Returns 32-bit floating point numbers from the data in the Blob.",
              key = "Blob:getF32",
              module = "lovr.data",
              related = {
                "Blob:getI8",
                "Blob:getU8",
                "Blob:getI16",
                "Blob:getU16",
                "Blob:getI32",
                "Blob:getU32",
                "Blob:getF64"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "offset",
                      type = "number",
                      description = "A non-negative byte offset to read from.",
                      default = "0"
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "The number of floats to read.",
                      default = "1"
                    }
                  },
                  returns = {
                    {
                      name = "...",
                      type = "number",
                      description = "`count` 32-bit floats."
                    }
                  }
                }
              }
            },
            {
              name = "getF64",
              summary = "Unpack 64-bit floating point numbers from the Blob.",
              description = "Returns 64-bit floating point numbers from the data in the Blob.",
              key = "Blob:getF64",
              module = "lovr.data",
              related = {
                "Blob:getI8",
                "Blob:getU8",
                "Blob:getI16",
                "Blob:getU16",
                "Blob:getI32",
                "Blob:getU32",
                "Blob:getF32"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "offset",
                      type = "number",
                      description = "A non-negative byte offset to read from.",
                      default = "0"
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "The number of doubles to read.",
                      default = "1"
                    }
                  },
                  returns = {
                    {
                      name = "...",
                      type = "number",
                      description = "`count` 64-bit doubles."
                    }
                  }
                }
              }
            }
          }
        },
        {
          name = "Image",
          summary = "An object that stores pixel data for Textures.",
          description = "An Image stores raw 2D pixel info for `Texture`s.  It has a width, height, and format.  The Image can be initialized with the contents of an image file or it can be created with uninitialized contents.  The supported image formats are `png`, `jpg`, `hdr`, `dds`, `ktx`, and `astc`.\n\nUsually you can just use Textures, but Image can be useful if you want to manipulate individual pixels, load Textures in a background thread, or use the FFI to efficiently access the raw image data.",
          key = "Image",
          module = "lovr.data",
          constructors = {
            "lovr.data.newImage"
          },
          methods = {
            {
              name = "encode",
              summary = "Encode the Image as png.",
              description = "Encodes the Image to an uncompressed png.  This intended mainly for debugging.",
              key = "Image:encode",
              module = "lovr.data",
              related = {
                "lovr.filesystem.write"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "blob",
                      type = "Blob",
                      description = "A new Blob containing the PNG image data."
                    }
                  }
                }
              }
            },
            {
              name = "getBlob",
              summary = "Get the bytes backing this Image as a `Blob`.",
              description = "Returns a Blob containing the raw bytes of the Image.",
              key = "Image:getBlob",
              module = "lovr.data",
              related = {
                "Blob:getPointer",
                "Sound:getBlob"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "blob",
                      type = "Blob",
                      description = "The Blob instance containing the bytes for the `Image`."
                    }
                  }
                }
              }
            },
            {
              name = "getDimensions",
              summary = "Get the dimensions of the Image.",
              description = "Returns the dimensions of the Image, in pixels.",
              key = "Image:getDimensions",
              module = "lovr.data",
              related = {
                "Image:getWidth",
                "Image:getHeight",
                "Texture:getDimensions"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "width",
                      type = "number",
                      description = "The width of the Image, in pixels."
                    },
                    {
                      name = "height",
                      type = "number",
                      description = "The height of the Image, in pixels."
                    }
                  }
                }
              }
            },
            {
              name = "getFormat",
              summary = "Get the pixel format of the Image.",
              description = "Returns the format of the Image.",
              key = "Image:getFormat",
              module = "lovr.data",
              related = {
                "TextureFormat",
                "Texture:getFormat"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "format",
                      type = "TextureFormat",
                      description = "The format of the pixels in the Image."
                    }
                  }
                }
              }
            },
            {
              name = "getHeight",
              summary = "Get the height of the Image.",
              description = "Returns the height of the Image, in pixels.",
              key = "Image:getHeight",
              module = "lovr.data",
              related = {
                "Image:getWidth",
                "Image:getDimensions",
                "Texture:getHeight"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "height",
                      type = "number",
                      description = "The height of the Image, in pixels."
                    }
                  }
                }
              }
            },
            {
              name = "getPixel",
              summary = "Get the value of a pixel of the Image.",
              description = "Returns the value of a pixel of the Image.",
              key = "Image:getPixel",
              module = "lovr.data",
              notes = "The following texture formats are supported: `r8`, `rg8`, `rgba8`, `r16`, `rg16`, `rgba16`, `r32f`, `rg32f`, `rgba32f`.",
              related = {
                "Image:setPixel",
                "Image:mapPixel",
                "TextureFormat",
                "Texture:getPixels",
                "Texture:setPixels",
                "Texture:newReadback"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate of the pixel to get (0-indexed)."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate of the pixel to get (0-indexed)."
                    }
                  },
                  returns = {
                    {
                      name = "r",
                      type = "number",
                      description = "The red component of the pixel, from 0.0 to 1.0."
                    },
                    {
                      name = "g",
                      type = "number",
                      description = "The green component of the pixel, from 0.0 to 1.0."
                    },
                    {
                      name = "b",
                      type = "number",
                      description = "The blue component of the pixel, from 0.0 to 1.0."
                    },
                    {
                      name = "a",
                      type = "number",
                      description = "The alpha component of the pixel, from 0.0 to 1.0."
                    }
                  }
                }
              }
            },
            {
              name = "getWidth",
              summary = "Get the width of the Image.",
              description = "Returns the width of the Image, in pixels.",
              key = "Image:getWidth",
              module = "lovr.data",
              related = {
                "Image:getHeight",
                "Image:getDimensions",
                "Texture:getWidth"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "width",
                      type = "number",
                      description = "The width of the Image, in pixels."
                    }
                  }
                }
              }
            },
            {
              name = "mapPixel",
              summary = "Transform an Image by applying a function to every pixel.",
              description = "Transforms pixels in the Image using a function.\n\nThe callback function passed to this function will be called once for each pixel.  For each pixel, the function will be called with its x and y coordinate and the red, green, blue, and alpha components of its color.  Whatever the function returns will be used as the new color for the pixel.\n\nThe callback function will potentially be called thousands of times, so it's best to keep the amount of code in there small and fast.",
              key = "Image:mapPixel",
              module = "lovr.data",
              examples = {
                {
                  description = "Convert an Image to grayscale.",
                  code = "image:mapPixel(function(x, y, r, g, b, a)\n  local brightness = .21 * r + .72 * g + .07 * b\n  return brightness, brightness, brightness, a\nend)"
                },
                {
                  description = "Efficient Image updates using FFI.  Due to the low-level nature, this will be a lot faster, but it's specialized to the `rgba8` image format and risks crashing if used improperly.",
                  code = "local ffi = require 'ffi'\n\nfunction lovr.load()\n  local w, h = 256, 256\n\n  image = lovr.data.newImage(w, h)\n\n  local pointer = ffi.cast('uint8_t*', image:getPointer())\n\n  for y = 0, h - 1 do\n    for x = 0, w - 1 do\n      pointer[(y * w + x) * 4 + 0] = (x / w) * 255\n      pointer[(y * w + x) * 4 + 1] = (y / h) * 255\n      pointer[(y * w + x) * 4 + 2] = 255\n      pointer[(y * w + x) * 4 + 3] = 255\n    end\n  end\n\n  texture = lovr.graphics.newTexture(image)\nend\n\nfunction lovr.draw(pass)\n  pass:fill(texture)\nend"
                }
              },
              notes = "The following texture formats are supported: `r8`, `rg8`, `rgba8`, `r16`, `rg16`, `rgba16`, `r32f`, `rg32f`, `rgba32f`.",
              related = {
                "Image:setPixel",
                "Image:getPixel",
                "TextureFormat",
                "Texture:setPixels"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "callback",
                      type = "function",
                      description = "The function that will be called for each pixel."
                    },
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate of the upper-left corner of the area of the Image to affect.",
                      default = "0"
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate of the upper-left corner of the area of the Image to affect.",
                      default = "0"
                    },
                    {
                      name = "w",
                      type = "number",
                      description = "The width of the area to affect.",
                      default = "image:getWidth()"
                    },
                    {
                      name = "h",
                      type = "number",
                      description = "The height of the area to affect.",
                      default = "image:getHeight()"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "paste",
              summary = "Copy pixels from another Image to this one.",
              description = "Copies a rectangle of pixels from one Image to this one.",
              key = "Image:paste",
              module = "lovr.data",
              notes = "The two Images must have the same pixel format.\n\nCompressed images cannot be copied.\n\nThe rectangle cannot go outside the dimensions of the source or destination textures.",
              related = {
                "Image:getPixel",
                "Image:setPixel",
                "Texture:setPixels"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "source",
                      type = "Image",
                      description = "The Image to copy pixels from."
                    },
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate to paste to (0-indexed).",
                      default = "0"
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate to paste to (0-indexed).",
                      default = "0"
                    },
                    {
                      name = "fromX",
                      type = "number",
                      description = "The x coordinate in the source to paste from (0-indexed).",
                      default = "0"
                    },
                    {
                      name = "fromY",
                      type = "number",
                      description = "The y coordinate in the source to paste from (0-indexed).",
                      default = "0"
                    },
                    {
                      name = "width",
                      type = "number",
                      description = "The width of the region to copy.",
                      default = "source:getWidth()"
                    },
                    {
                      name = "height",
                      type = "number",
                      description = "The height of the region to copy.",
                      default = "source:getHeight()"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setPixel",
              summary = "Set the value of a pixel of the Image.",
              description = "Sets the value of a single pixel of the Image.\n\nIf you need to change a bunch of pixels, consider using `Image:mapPixel`.",
              key = "Image:setPixel",
              module = "lovr.data",
              notes = "The following texture formats are supported: `r8`, `rg8`, `rgba8`, `r16`, `rg16`, `rgba16`, `r32f`, `rg32f`, `rgba32f`.",
              related = {
                "Image:mapPixel",
                "Image:getPixel",
                "TextureFormat",
                "Texture:setPixels"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate of the pixel to set (0-indexed)."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate of the pixel to set (0-indexed)."
                    },
                    {
                      name = "r",
                      type = "number",
                      description = "The red component of the pixel, from 0.0 to 1.0."
                    },
                    {
                      name = "g",
                      type = "number",
                      description = "The green component of the pixel, from 0.0 to 1.0."
                    },
                    {
                      name = "b",
                      type = "number",
                      description = "The blue component of the pixel, from 0.0 to 1.0."
                    },
                    {
                      name = "a",
                      type = "number",
                      description = "The alpha component of the pixel, from 0.0 to 1.0.",
                      default = "1.0"
                    }
                  },
                  returns = {}
                }
              }
            }
          }
        },
        {
          name = "ModelData",
          summary = "An object that loads and stores data for 3D models.",
          description = "A ModelData is a container object that loads and holds data contained in 3D model files.  This can include a variety of things like the node structure of the asset, the vertex data it contains, contains, the `Image` and `Material` properties, and any included animations.\n\nThe current supported formats are OBJ, glTF, and STL.\n\nUsually you can just load a `Model` directly, but using a `ModelData` can be helpful if you want to load models in a thread or access more low-level information about the Model.",
          key = "ModelData",
          module = "lovr.data",
          constructors = {
            "lovr.data.newModelData"
          },
          methods = {
            {
              name = "getAnimationChannelCount",
              summary = "Get the number of channels in an animation.",
              description = "Returns the number of channels in an animation.\n\nA channel is a set of keyframes for a single property of a node.",
              key = "ModelData:getAnimationChannelCount",
              module = "lovr.data",
              related = {
                "ModelData:getAnimationNode",
                "ModelData:getAnimationProperty"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of an animation."
                    }
                  },
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of channels in the animation."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of an animation."
                    }
                  },
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of channels in the animation."
                    }
                  }
                }
              }
            },
            {
              name = "getAnimationCount",
              summary = "Get the number of animations in the model.",
              description = "Returns the number of animations in the model.",
              key = "ModelData:getAnimationCount",
              module = "lovr.data",
              related = {
                "Model:getAnimationCount"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of animations in the model."
                    }
                  }
                }
              }
            },
            {
              name = "getAnimationDuration",
              summary = "Get the duration of an animation.",
              description = "Returns the duration of an animation.",
              key = "ModelData:getAnimationDuration",
              module = "lovr.data",
              notes = "The duration of the animation is calculated as the latest timestamp of all of its channels.",
              related = {
                "Model:getAnimationDuration"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the animation."
                    }
                  },
                  returns = {
                    {
                      name = "duration",
                      type = "number",
                      description = "The duration of the animation, in seconds."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the animation."
                    }
                  },
                  returns = {
                    {
                      name = "duration",
                      type = "number",
                      description = "The duration of the animation, in seconds."
                    }
                  }
                }
              }
            },
            {
              name = "getAnimationKeyframe",
              summary = "Get a keyframe in a channel of an animation.",
              description = "Returns a single keyframe in a channel of an animation.",
              key = "ModelData:getAnimationKeyframe",
              module = "lovr.data",
              related = {
                "ModelData:getAnimationSmoothMode",
                "ModelData:getAnimationKeyframeCount"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of an animation."
                    },
                    {
                      name = "channel",
                      type = "number",
                      description = "The index of a channel in the animation."
                    },
                    {
                      name = "keyframe",
                      type = "number",
                      description = "The index of a keyframe in the channel."
                    }
                  },
                  returns = {
                    {
                      name = "time",
                      type = "number",
                      description = "The timestamp of the keyframe."
                    },
                    {
                      name = "...",
                      type = "number",
                      description = "The data for the keyframe (either 3 or 4 numbers depending on the property)."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of an animation."
                    },
                    {
                      name = "channel",
                      type = "number",
                      description = "The index of a channel in the animation."
                    },
                    {
                      name = "keyframe",
                      type = "number",
                      description = "The index of a keyframe in the channel."
                    }
                  },
                  returns = {
                    {
                      name = "time",
                      type = "number",
                      description = "The timestamp of the keyframe."
                    },
                    {
                      name = "...",
                      type = "number",
                      description = "The data for the keyframe (either 3 or 4 numbers depending on the property)."
                    }
                  }
                }
              }
            },
            {
              name = "getAnimationKeyframeCount",
              summary = "Get the number of keyframes in a channel of an animation.",
              description = "Returns the number of keyframes in a channel of an animation.",
              key = "ModelData:getAnimationKeyframeCount",
              module = "lovr.data",
              related = {
                "ModelData:getAnimationSmoothMode",
                "ModelData:getAnimationKeyframe"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of an animation."
                    },
                    {
                      name = "channel",
                      type = "number",
                      description = "The index of a channel in the animation."
                    }
                  },
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of keyframes in the channel."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of an animation."
                    },
                    {
                      name = "channel",
                      type = "number",
                      description = "The index of a channel in the animation."
                    }
                  },
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of keyframes in the channel."
                    }
                  }
                }
              }
            },
            {
              name = "getAnimationName",
              summary = "Get the name of an animation.",
              description = "Returns the name of an animation.",
              key = "ModelData:getAnimationName",
              module = "lovr.data",
              notes = "If the animation does not have a name, this function returns `nil`.",
              related = {
                "Model:getAnimationName"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the animation."
                    }
                  },
                  returns = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the animation."
                    }
                  }
                }
              }
            },
            {
              name = "getAnimationNode",
              summary = "Get the node targeted by the channel of an animation.",
              description = "Returns the index of a node targeted by an animation's channel.",
              key = "ModelData:getAnimationNode",
              module = "lovr.data",
              related = {
                "ModelData:getAnimationProperty",
                "ModelData:getAnimationSmoothMode"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of an animation."
                    },
                    {
                      name = "channel",
                      type = "number",
                      description = "The index of a channel in the animation."
                    }
                  },
                  returns = {
                    {
                      name = "node",
                      type = "number",
                      description = "The index of the node targeted by the channel."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of an animation."
                    },
                    {
                      name = "channel",
                      type = "number",
                      description = "The index of a channel in the animation."
                    }
                  },
                  returns = {
                    {
                      name = "node",
                      type = "number",
                      description = "The index of the node targeted by the channel."
                    }
                  }
                }
              }
            },
            {
              name = "getAnimationProperty",
              summary = "Get the property targeted by the channel of an animation.",
              description = "Returns the property targeted by an animation's channel.",
              key = "ModelData:getAnimationProperty",
              module = "lovr.data",
              related = {
                "ModelData:getAnimationNode",
                "ModelData:getAnimationSmoothMode"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of an animation."
                    },
                    {
                      name = "channel",
                      type = "number",
                      description = "The index of a channel in the animation."
                    }
                  },
                  returns = {
                    {
                      name = "property",
                      type = "AnimationProperty",
                      description = "The property (translation, rotation, scale, weights) affected by the keyframes."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of an animation."
                    },
                    {
                      name = "channel",
                      type = "number",
                      description = "The index of a channel in the animation."
                    }
                  },
                  returns = {
                    {
                      name = "property",
                      type = "AnimationProperty",
                      description = "The property (translation, rotation, scale, weights) affected by the keyframes."
                    }
                  }
                }
              }
            },
            {
              name = "getAnimationSmoothMode",
              summary = "Get the smooth mode of a channel in an animation.",
              description = "Returns the smooth mode of a channel in an animation.",
              key = "ModelData:getAnimationSmoothMode",
              module = "lovr.data",
              related = {
                "ModelData:getAnimationNode",
                "ModelData:getAnimationProperty"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of an animation."
                    },
                    {
                      name = "channel",
                      type = "number",
                      description = "The index of a channel in the animation."
                    }
                  },
                  returns = {
                    {
                      name = "smooth",
                      type = "SmoothMode",
                      description = "The smooth mode of the keyframes."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of an animation."
                    },
                    {
                      name = "channel",
                      type = "number",
                      description = "The index of a channel in the animation."
                    }
                  },
                  returns = {
                    {
                      name = "smooth",
                      type = "SmoothMode",
                      description = "The smooth mode of the keyframes."
                    }
                  }
                }
              }
            },
            {
              name = "getBlendShapeCount",
              summary = "Get the number of blend shapes in the model.",
              description = "Returns the number of blend shapes in the model.",
              key = "ModelData:getBlendShapeCount",
              module = "lovr.data",
              related = {
                "ModelData:getBlendShapeName",
                "Model:getBlendShapeCount"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of blend shapes in the model."
                    }
                  }
                }
              }
            },
            {
              name = "getBlendShapeName",
              summary = "Get the name of a blend shape in the model.",
              description = "Returns the name of a blend shape in the model.",
              key = "ModelData:getBlendShapeName",
              module = "lovr.data",
              notes = "This function will throw an error if the blend shape index is invalid.",
              related = {
                "ModelData:getBlendShapeCount",
                "Model:getBlendShapeName"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of a blend shape."
                    }
                  },
                  returns = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the blend shape."
                    }
                  }
                }
              }
            },
            {
              name = "getBlob",
              summary = "Get a Blob in the model.",
              description = "Returns one of the Blobs in the model, by index.",
              key = "ModelData:getBlob",
              module = "lovr.data",
              related = {
                "ModelData:getBlobCount",
                "ModelData:getImage"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the Blob to get."
                    }
                  },
                  returns = {
                    {
                      name = "blob",
                      type = "Blob",
                      description = "The Blob object."
                    }
                  }
                }
              }
            },
            {
              name = "getBlobCount",
              summary = "Get the number of Blobs stored in the model.",
              description = "Returns the number of Blobs in the model.",
              key = "ModelData:getBlobCount",
              module = "lovr.data",
              related = {
                "ModelData:getBlob",
                "ModelData:getImageCount"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of Blobs in the model."
                    }
                  }
                }
              }
            },
            {
              name = "getBoundingBox",
              summary = "Get the bounding box of the model.",
              description = "Returns the 6 values of the model's axis-aligned bounding box.",
              key = "ModelData:getBoundingBox",
              module = "lovr.data",
              related = {
                "ModelData:getWidth",
                "ModelData:getHeight",
                "ModelData:getDepth",
                "ModelData:getDimensions",
                "ModelData:getCenter",
                "ModelData:getBoundingSphere",
                "Model:getBoundingBox"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "minx",
                      type = "number",
                      description = "The minimum x coordinate of the vertices in the model."
                    },
                    {
                      name = "maxx",
                      type = "number",
                      description = "The maximum x coordinate of the vertices in the model."
                    },
                    {
                      name = "miny",
                      type = "number",
                      description = "The minimum y coordinate of the vertices in the model."
                    },
                    {
                      name = "maxy",
                      type = "number",
                      description = "The maximum y coordinate of the vertices in the model."
                    },
                    {
                      name = "minz",
                      type = "number",
                      description = "The minimum z coordinate of the vertices in the model."
                    },
                    {
                      name = "maxz",
                      type = "number",
                      description = "The maximum z coordinate of the vertices in the model."
                    }
                  }
                }
              }
            },
            {
              name = "getBoundingSphere",
              summary = "Get the bounding sphere of the model.",
              description = "Returns a sphere approximately enclosing the vertices in the model.",
              key = "ModelData:getBoundingSphere",
              module = "lovr.data",
              related = {
                "ModelData:getWidth",
                "ModelData:getHeight",
                "ModelData:getDepth",
                "ModelData:getDimensions",
                "ModelData:getCenter",
                "ModelData:getBoundingBox",
                "Model:getBoundingSphere"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate of the position of the sphere."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate of the position of the sphere."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate of the position of the sphere."
                    },
                    {
                      name = "radius",
                      type = "number",
                      description = "The radius of the bounding sphere."
                    }
                  }
                }
              }
            },
            {
              name = "getCenter",
              summary = "Get the center of the model's bounding box.",
              description = "Returns the center of the model's axis-aligned bounding box, relative to the model's origin.",
              key = "ModelData:getCenter",
              module = "lovr.data",
              related = {
                "ModelData:getWidth",
                "ModelData:getHeight",
                "ModelData:getDepth",
                "ModelData:getDimensions",
                "ModelData:getBoundingBox",
                "Model:getCenter"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x offset of the center of the bounding box."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y offset of the center of the bounding box."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z offset of the center of the bounding box."
                    }
                  }
                }
              }
            },
            {
              name = "getDepth",
              summary = "Get the depth of the model.",
              description = "Returns the depth of the model, computed from its axis-aligned bounding box.",
              key = "ModelData:getDepth",
              module = "lovr.data",
              related = {
                "ModelData:getWidth",
                "ModelData:getHeight",
                "ModelData:getDimensions",
                "ModelData:getCenter",
                "ModelData:getBoundingBox",
                "Model:getDepth"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "depth",
                      type = "number",
                      description = "The depth of the model."
                    }
                  }
                }
              }
            },
            {
              name = "getDimensions",
              summary = "Get the dimensions of the model.",
              description = "Returns the width, height, and depth of the model, computed from its axis-aligned bounding box.",
              key = "ModelData:getDimensions",
              module = "lovr.data",
              related = {
                "ModelData:getWidth",
                "ModelData:getHeight",
                "ModelData:getDepth",
                "ModelData:getCenter",
                "ModelData:getBoundingBox",
                "Model:getDimensions"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "width",
                      type = "number",
                      description = "The width of the model."
                    },
                    {
                      name = "height",
                      type = "number",
                      description = "The height of the model."
                    },
                    {
                      name = "depth",
                      type = "number",
                      description = "The depth of the model."
                    }
                  }
                }
              }
            },
            {
              name = "getHeight",
              summary = "Get the height of the model.",
              description = "Returns the height of the model, computed from its axis-aligned bounding box.",
              key = "ModelData:getHeight",
              module = "lovr.data",
              related = {
                "ModelData:getWidth",
                "ModelData:getDepth",
                "ModelData:getDimensions",
                "ModelData:getCenter",
                "ModelData:getBoundingBox",
                "Model:getHeight"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "height",
                      type = "number",
                      description = "The height of the model."
                    }
                  }
                }
              }
            },
            {
              name = "getImage",
              summary = "Get an Image in the model.",
              description = "Returns one of the Images in the model, by index.",
              key = "ModelData:getImage",
              module = "lovr.data",
              related = {
                "ModelData:getImageCount",
                "ModelData:getBlob"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the Image to get."
                    }
                  },
                  returns = {
                    {
                      name = "image",
                      type = "Image",
                      description = "The Image object."
                    }
                  }
                }
              }
            },
            {
              name = "getImageCount",
              summary = "Get the number of Images stored in the model.",
              description = "Returns the number of Images in the model.",
              key = "ModelData:getImageCount",
              module = "lovr.data",
              related = {
                "ModelData:getImage",
                "ModelData:getBlobCount"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of Images in the model."
                    }
                  }
                }
              }
            },
            {
              name = "getMaterial",
              summary = "Get the material properties for a material in the model.",
              description = "Returns a table with all of the properties of a material.",
              key = "ModelData:getMaterial",
              module = "lovr.data",
              notes = "All images are optional and may be `nil`.",
              related = {
                "ModelData:getMaterialCount",
                "ModelData:getMeshMaterial",
                "lovr.graphics.newMaterial",
                "Model:getMaterial"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of a material."
                    }
                  },
                  returns = {
                    {
                      name = "properties",
                      type = "table",
                      description = "The material properties.",
                      table = {
                        {
                          name = "color",
                          type = "table",
                          description = "The color of the material.  The table contains the `r`, `g`, `b`, and `a` components of the color, from 0 to 1."
                        },
                        {
                          name = "glow",
                          type = "table",
                          description = "The glow color of the material (sometimes called emissive).  The table contains the `r`, `g`, and `b` components of the color from 0 to 1, and a fourth number indicating the strength of the glow."
                        },
                        {
                          name = "uvShift",
                          type = "table",
                          description = "A table with 2 numbers indicating an offset to apply to UVs."
                        },
                        {
                          name = "uvScale",
                          type = "table",
                          description = "A table with 2 numbers indicating a scale to apply to UVs.  By default, shaders apply the UV scale before the UV offset."
                        },
                        {
                          name = "metalness",
                          type = "number",
                          description = "The metalness parameter of the material.  This is typically 0 or 1.  By default, shaders multiply this property with the value from the metalness texture (when present) to get the final metalness used for shading."
                        },
                        {
                          name = "roughness",
                          type = "number",
                          description = "The roughness parameter of the material.  By default, shaders multiply this property with the value from the roughness texture (when present) to get the final roughness used for shading."
                        },
                        {
                          name = "clearcoat",
                          type = "number",
                          description = "The clearcoat parameter of the material."
                        },
                        {
                          name = "clearcoatRoughness",
                          type = "number",
                          description = "The roughness of the clearcoat layer."
                        },
                        {
                          name = "occlusionStrength",
                          type = "number",
                          description = "A number multiplied by the value from the ambient occlusion texture to control how strong the occlusion effect is."
                        },
                        {
                          name = "normalScale",
                          type = "number",
                          description = "A number multiplied by the value from the normal texture to control how strong the normal mapping effect is."
                        },
                        {
                          name = "alphaCutoff",
                          type = "number",
                          description = "If a pixel has an alpha value less than the alpha cutoff, it will be discarded, which prevents it from occluding things behind it.  This is sometimes called \"holepunch\" or \"cutout\" alpha.  It's useful for textures with transparency."
                        },
                        {
                          name = "texture",
                          type = "number",
                          description = "The index of the Image used for the color texture."
                        },
                        {
                          name = "glowTexture",
                          type = "number",
                          description = "The index of the Image used for the glow texture."
                        },
                        {
                          name = "occlusionTexture",
                          type = "number",
                          description = "The index of the Image used for the ambient occlusion texture.  The red channel of the texture is used for ambient occlusion, allowing multiple parameters to use the same texture."
                        },
                        {
                          name = "metalnessTexture",
                          type = "number",
                          description = "The index of the Image used for the metalness texture.  The blue channel of the texture is used for metalness, allowing multiple parameters to use the same texture."
                        },
                        {
                          name = "roughnessTexture",
                          type = "number",
                          description = "The index of the Image to use for the roughness texture.  The green channel of the texture is used for roughness, allowing multiple parameters to use the same texture."
                        },
                        {
                          name = "clearcoatTexture",
                          type = "number",
                          description = "The index of the Image to use for the clearcoat texture.  The red channel of the texture is used for the clearcoat parameter, allowing multiple parameters to use the same texture."
                        },
                        {
                          name = "normalTexture",
                          type = "number",
                          description = "The index of the Image to use for the normal map."
                        }
                      }
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of a material."
                    }
                  },
                  returns = {
                    {
                      name = "properties",
                      type = "table",
                      description = "The material properties.",
                      table = {
                        {
                          name = "color",
                          type = "table",
                          description = "The color of the material.  The table contains the `r`, `g`, `b`, and `a` components of the color, from 0 to 1."
                        },
                        {
                          name = "glow",
                          type = "table",
                          description = "The glow color of the material (sometimes called emissive).  The table contains the `r`, `g`, and `b` components of the color from 0 to 1, and a fourth number indicating the strength of the glow."
                        },
                        {
                          name = "uvShift",
                          type = "table",
                          description = "A table with 2 numbers indicating an offset to apply to UVs."
                        },
                        {
                          name = "uvScale",
                          type = "table",
                          description = "A table with 2 numbers indicating a scale to apply to UVs.  By default, shaders apply the UV scale before the UV offset."
                        },
                        {
                          name = "metalness",
                          type = "number",
                          description = "The metalness parameter of the material.  This is typically 0 or 1.  By default, shaders multiply this property with the value from the metalness texture (when present) to get the final metalness used for shading."
                        },
                        {
                          name = "roughness",
                          type = "number",
                          description = "The roughness parameter of the material.  By default, shaders multiply this property with the value from the roughness texture (when present) to get the final roughness used for shading."
                        },
                        {
                          name = "clearcoat",
                          type = "number",
                          description = "The clearcoat parameter of the material."
                        },
                        {
                          name = "clearcoatRoughness",
                          type = "number",
                          description = "The roughness of the clearcoat layer."
                        },
                        {
                          name = "occlusionStrength",
                          type = "number",
                          description = "A number multiplied by the value from the ambient occlusion texture to control how strong the occlusion effect is."
                        },
                        {
                          name = "normalScale",
                          type = "number",
                          description = "A number multiplied by the value from the normal texture to control how strong the normal mapping effect is."
                        },
                        {
                          name = "alphaCutoff",
                          type = "number",
                          description = "If a pixel has an alpha value less than the alpha cutoff, it will be discarded, which prevents it from occluding things behind it.  This is sometimes called \"holepunch\" or \"cutout\" alpha.  It's useful for textures with transparency."
                        },
                        {
                          name = "texture",
                          type = "number",
                          description = "The index of the Image used for the color texture."
                        },
                        {
                          name = "glowTexture",
                          type = "number",
                          description = "The index of the Image used for the glow texture."
                        },
                        {
                          name = "occlusionTexture",
                          type = "number",
                          description = "The index of the Image used for the ambient occlusion texture.  The red channel of the texture is used for ambient occlusion, allowing multiple parameters to use the same texture."
                        },
                        {
                          name = "metalnessTexture",
                          type = "number",
                          description = "The index of the Image used for the metalness texture.  The blue channel of the texture is used for metalness, allowing multiple parameters to use the same texture."
                        },
                        {
                          name = "roughnessTexture",
                          type = "number",
                          description = "The index of the Image to use for the roughness texture.  The green channel of the texture is used for roughness, allowing multiple parameters to use the same texture."
                        },
                        {
                          name = "clearcoatTexture",
                          type = "number",
                          description = "The index of the Image to use for the clearcoat texture.  The red channel of the texture is used for the clearcoat parameter, allowing multiple parameters to use the same texture."
                        },
                        {
                          name = "normalTexture",
                          type = "number",
                          description = "The index of the Image to use for the normal map."
                        }
                      }
                    }
                  }
                }
              }
            },
            {
              name = "getMaterialCount",
              summary = "Get the number of materials in the model.",
              description = "Returns the number of materials in the model.",
              key = "ModelData:getMaterialCount",
              module = "lovr.data",
              related = {
                "ModelData:getMaterialName",
                "ModelData:getMeshMaterial",
                "ModelData:getMaterial",
                "Model:getMaterialCount"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of materials in the model."
                    }
                  }
                }
              }
            },
            {
              name = "getMaterialName",
              summary = "Get the name of a material in the model.",
              description = "Returns the name of a material in the model.",
              key = "ModelData:getMaterialName",
              module = "lovr.data",
              related = {
                "ModelData:getMaterialCount",
                "ModelData:getMeshMaterial",
                "ModelData:getMaterial",
                "Model:getMaterialName"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of a material."
                    }
                  },
                  returns = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the material, or nil if the material does not have a name."
                    }
                  }
                }
              }
            },
            {
              name = "getMeshCount",
              summary = "Get the number of meshes in the model.",
              description = "Returns the number of meshes in the model.",
              key = "ModelData:getMeshCount",
              module = "lovr.data",
              related = {
                "ModelData:getNodeMeshes"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of meshes in the model."
                    }
                  }
                }
              }
            },
            {
              name = "getMeshDrawMode",
              summary = "Get the draw mode of a mesh.",
              description = "Returns the draw mode of a mesh.  This controls how its vertices are connected together (points, lines, or triangles).",
              key = "ModelData:getMeshDrawMode",
              module = "lovr.data",
              variants = {
                {
                  arguments = {
                    {
                      name = "mesh",
                      type = "number",
                      description = "The index of a mesh."
                    }
                  },
                  returns = {
                    {
                      name = "mode",
                      type = "ModelDrawMode",
                      description = "The draw mode of the mesh."
                    }
                  }
                }
              }
            },
            {
              name = "getMeshIndex",
              summary = "Get one of the vertex indices in a mesh.",
              description = "Returns one of the vertex indices in a mesh.  If a mesh has vertex indices, they define the order and connectivity of the vertices in the mesh, allowing a vertex to be reused multiple times without duplicating its data.",
              key = "ModelData:getMeshIndex",
              module = "lovr.data",
              related = {
                "ModelData:getMeshIndexFormat",
                "ModelData:getMeshIndexCount",
                "ModelData:getMeshVertex",
                "ModelData:getTriangles"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "mesh",
                      type = "number",
                      description = "The index of a mesh to get the vertex from."
                    },
                    {
                      name = "index",
                      type = "number",
                      description = "The index of a vertex index in the mesh to retrieve."
                    }
                  },
                  returns = {
                    {
                      name = "vertexindex",
                      type = "number",
                      description = "The vertex index.  Like all indices in Lua, this is 1-indexed."
                    }
                  }
                }
              }
            },
            {
              name = "getMeshIndexCount",
              summary = "Get the number of vertex indices in a mesh.",
              description = "Returns the number of vertex indices in a mesh.  Vertex indices allow for vertices to be reused when defining triangles.",
              key = "ModelData:getMeshIndexCount",
              module = "lovr.data",
              notes = "This may return zero if the mesh does not use indices.",
              variants = {
                {
                  arguments = {
                    {
                      name = "mesh",
                      type = "number",
                      description = "The index of a mesh."
                    }
                  },
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of vertex indices in the mesh."
                    }
                  }
                }
              }
            },
            {
              name = "getMeshIndexFormat",
              summary = "Get the data format of vertex indices in a mesh.",
              description = "Returns the data format of vertex indices in a mesh.  If a mesh doesn't use vertex indices, this function returns nil.",
              key = "ModelData:getMeshIndexFormat",
              module = "lovr.data",
              related = {
                "ModelData:getMeshVertexFormat"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "mesh",
                      type = "number",
                      description = "The index of a mesh."
                    }
                  },
                  returns = {
                    {
                      name = "type",
                      type = "AttributeType",
                      description = "The data type of each vertex index (always u16 or u32)."
                    },
                    {
                      name = "blob",
                      type = "number",
                      description = "The index of a Blob in the mesh where the binary data is stored."
                    },
                    {
                      name = "offset",
                      type = "number",
                      description = "A byte offset into the Blob's data where the index data starts."
                    },
                    {
                      name = "stride",
                      type = "number",
                      description = "The number of bytes between subsequent vertex indices.  Indices are always tightly packed, so this will always be 2 or 4 depending on the data type."
                    }
                  }
                }
              }
            },
            {
              name = "getMeshMaterial",
              summary = "Get the index of the material applied to a mesh.",
              description = "Returns the index of the material applied to a mesh.",
              key = "ModelData:getMeshMaterial",
              module = "lovr.data",
              variants = {
                {
                  arguments = {
                    {
                      name = "mesh",
                      type = "number",
                      description = "The index of a mesh."
                    }
                  },
                  returns = {
                    {
                      name = "material",
                      type = "number",
                      description = "The index of the material applied to the mesh, or nil if the mesh does not have a material."
                    }
                  }
                }
              }
            },
            {
              name = "getMeshVertex",
              summary = "Get the data for a single vertex in a mesh.",
              description = "Returns the data for a single vertex in a mesh.  The data returned depends on the vertex format of a mesh, which is given by `ModelData:getMeshVertexFormat`.",
              key = "ModelData:getMeshVertex",
              module = "lovr.data",
              related = {
                "ModelData:getMeshVertexFormat",
                "ModelData:getMeshVertexCount",
                "ModelData:getMeshIndex",
                "ModelData:getTriangles"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "mesh",
                      type = "number",
                      description = "The index of a mesh to get the vertex from."
                    },
                    {
                      name = "vertex",
                      type = "number",
                      description = "The index of a vertex in the mesh to retrieve."
                    }
                  },
                  returns = {
                    {
                      name = "...",
                      type = "number",
                      description = "The data for all of the attributes of the vertex."
                    }
                  }
                }
              }
            },
            {
              name = "getMeshVertexCount",
              summary = "Get the number of vertices in a mesh.",
              description = "Returns the number of vertices in a mesh.",
              key = "ModelData:getMeshVertexCount",
              module = "lovr.data",
              related = {
                "ModelData:getMeshIndexCount"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "mesh",
                      type = "number",
                      description = "The index of a mesh."
                    }
                  },
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of vertices in the mesh."
                    }
                  }
                }
              }
            },
            {
              name = "getMeshVertexFormat",
              summary = "Get the vertex format of a mesh.",
              description = "Returns the vertex format of a mesh.  The vertex format defines the properties associated with each vertex (position, color, etc.), including their types and binary data layout.",
              key = "ModelData:getMeshVertexFormat",
              module = "lovr.data",
              notes = "The format is given as a table of vertex attributes.  Each attribute is a table containing the following:\n\n    { name, type, components, blob, offset, stride }\n\n- The `name` will be a `DefaultAttribute`.\n- The `type` will be an `AttributeType`.\n- The `component` count will be 1-4.\n- The `blob` is an index of one of the Blobs in the model (see `ModelData:getBlob`).\n- The `offset` is a byte offset from the start of the Blob where the attribute's data starts.\n- The `stride` is the number of bytes between consecutive values.",
              related = {
                "ModelData:getMeshIndexFormat"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "mesh",
                      type = "number",
                      description = "The index of a mesh."
                    }
                  },
                  returns = {
                    {
                      name = "format",
                      type = "table",
                      description = "The vertex format of the mesh."
                    }
                  }
                }
              }
            },
            {
              name = "getMetadata",
              summary = "Get extra information from the model file.",
              description = "Returns extra information stored in the model file.  Currently this is only implemented for glTF models and returns the JSON string from the glTF or glb file.  The metadata can be used to get application-specific data or add support for glTF extensions not supported by LÖVR.",
              key = "ModelData:getMetadata",
              module = "lovr.data",
              related = {
                "Model:getMetadata"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "metadata",
                      type = "string",
                      description = "The metadata from the model file."
                    }
                  }
                }
              }
            },
            {
              name = "getNodeChildren",
              summary = "Get the children of a node.",
              description = "Given a parent node, this function returns a table with the indices of its children.",
              key = "ModelData:getNodeChildren",
              module = "lovr.data",
              notes = "If the node does not have any children, this function returns an empty table.",
              related = {
                "ModelData:getNodeParent",
                "ModelData:getRootNode",
                "Model:getNodeChildren"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the parent node."
                    }
                  },
                  returns = {
                    {
                      name = "children",
                      type = "table",
                      description = "A table containing a node index for each child of the node."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the parent node."
                    }
                  },
                  returns = {
                    {
                      name = "children",
                      type = "table",
                      description = "A table containing a node index for each child of the node."
                    }
                  }
                }
              }
            },
            {
              name = "getNodeCount",
              summary = "Get the number of nodes in the model.",
              description = "Returns the number of nodes in the model.",
              key = "ModelData:getNodeCount",
              module = "lovr.data",
              related = {
                "Model:getNodeCount"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of nodes in the model."
                    }
                  }
                }
              }
            },
            {
              name = "getNodeMeshes",
              summary = "Get the indices of meshes attached to a node.",
              description = "Returns a table of mesh indices attached to a node.  Meshes define the geometry and materials of a model, as opposed to the nodes which define the transforms and hierarchy.  A node can have multiple meshes, and meshes can be reused in multiple nodes.",
              key = "ModelData:getNodeMeshes",
              module = "lovr.data",
              related = {
                "ModelData:getMeshCount"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the node."
                    }
                  },
                  returns = {
                    {
                      name = "meshes",
                      type = "table",
                      description = "A table with the node's mesh indices."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the node."
                    }
                  },
                  returns = {
                    {
                      name = "meshes",
                      type = "table",
                      description = "A table with the node's mesh indices."
                    }
                  }
                }
              }
            },
            {
              name = "getNodeName",
              summary = "Get the name of a node.",
              description = "Returns the name of a node.",
              key = "ModelData:getNodeName",
              module = "lovr.data",
              notes = "If the node does not have a name, this function returns `nil`.",
              related = {
                "Model:getNodeName"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the node."
                    }
                  },
                  returns = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the node."
                    }
                  }
                }
              }
            },
            {
              name = "getNodeOrientation",
              summary = "Get the local orientation of a node.",
              description = "Returns local orientation of a node, relative to its parent.",
              key = "ModelData:getNodeOrientation",
              module = "lovr.data",
              related = {
                "ModelData:getNodePosition",
                "ModelData:getNodeScale",
                "ModelData:getNodePose",
                "ModelData:getNodeTransform"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the node."
                    }
                  },
                  returns = {
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the node is rotated around its axis of rotation."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the node."
                    }
                  },
                  returns = {
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the node is rotated around its axis of rotation."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    }
                  }
                }
              }
            },
            {
              name = "getNodeParent",
              summary = "Get the parent of a node.",
              description = "Given a child node, this function returns the index of its parent.",
              key = "ModelData:getNodeParent",
              module = "lovr.data",
              related = {
                "ModelData:getNodeChildren",
                "ModelData:getRootNode",
                "Model:getNodeParent"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the child node."
                    }
                  },
                  returns = {
                    {
                      name = "parent",
                      type = "number",
                      description = "The index of the parent."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the child node."
                    }
                  },
                  returns = {
                    {
                      name = "parent",
                      type = "number",
                      description = "The index of the parent."
                    }
                  }
                }
              }
            },
            {
              name = "getNodePose",
              summary = "Get the local pose of a node.",
              description = "Returns local pose (position and orientation) of a node, relative to its parent.",
              key = "ModelData:getNodePose",
              module = "lovr.data",
              related = {
                "ModelData:getNodePosition",
                "ModelData:getNodeOrientation",
                "ModelData:getNodeScale",
                "ModelData:getNodeTransform"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the node."
                    }
                  },
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate."
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the node is rotated around its axis of rotation."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the node."
                    }
                  },
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate."
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the node is rotated around its axis of rotation."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    }
                  }
                }
              }
            },
            {
              name = "getNodePosition",
              summary = "Get the local position of a node.",
              description = "Returns local position of a node, relative to its parent.",
              key = "ModelData:getNodePosition",
              module = "lovr.data",
              related = {
                "ModelData:getNodeOrientation",
                "ModelData:getNodeScale",
                "ModelData:getNodePose",
                "ModelData:getNodeTransform"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the node."
                    }
                  },
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the node."
                    }
                  },
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate."
                    }
                  }
                }
              }
            },
            {
              name = "getNodeScale",
              summary = "Get the local scale of a node.",
              description = "Returns local scale of a node, relative to its parent.",
              key = "ModelData:getNodeScale",
              module = "lovr.data",
              related = {
                "ModelData:getNodePosition",
                "ModelData:getNodeOrientation",
                "ModelData:getNodePose",
                "ModelData:getNodeTransform"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the node."
                    }
                  },
                  returns = {
                    {
                      name = "sx",
                      type = "number",
                      description = "The x scale."
                    },
                    {
                      name = "sy",
                      type = "number",
                      description = "The y scale."
                    },
                    {
                      name = "sz",
                      type = "number",
                      description = "The z scale."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the node."
                    }
                  },
                  returns = {
                    {
                      name = "sx",
                      type = "number",
                      description = "The x scale."
                    },
                    {
                      name = "sy",
                      type = "number",
                      description = "The y scale."
                    },
                    {
                      name = "sz",
                      type = "number",
                      description = "The z scale."
                    }
                  }
                }
              }
            },
            {
              name = "getNodeSkin",
              summary = "Get the index of the skin used by a node.",
              description = "Returns the index of the skin used by a node.  Skins are collections of joints used for skeletal animation.  A model can have multiple skins, and each node can use at most one skin to drive the animation of its meshes.",
              key = "ModelData:getNodeSkin",
              module = "lovr.data",
              related = {
                "ModelData:getSkinCount"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the node."
                    }
                  },
                  returns = {
                    {
                      name = "skin",
                      type = "number",
                      description = "The index of the node's skin, or nil if the node isn't skeletally animated."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the node."
                    }
                  },
                  returns = {
                    {
                      name = "skin",
                      type = "number",
                      description = "The index of the node's skin, or nil if the node isn't skeletally animated."
                    }
                  }
                }
              }
            },
            {
              name = "getNodeTransform",
              summary = "Get the local transform of a node.",
              description = "Returns local transform (position, orientation, and scale) of a node, relative to its parent.",
              key = "ModelData:getNodeTransform",
              module = "lovr.data",
              notes = "For best results when animating, it's recommended to keep the 3 components of the scale the same.",
              related = {
                "ModelData:getNodePosition",
                "ModelData:getNodeOrientation",
                "ModelData:getNodeScale",
                "ModelData:getNodePose"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the node."
                    }
                  },
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate."
                    },
                    {
                      name = "sx",
                      type = "number",
                      description = "The x scale."
                    },
                    {
                      name = "sy",
                      type = "number",
                      description = "The y scale."
                    },
                    {
                      name = "sz",
                      type = "number",
                      description = "The z scale."
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the node is rotated around its axis of rotation."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the node."
                    }
                  },
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate."
                    },
                    {
                      name = "sx",
                      type = "number",
                      description = "The x scale."
                    },
                    {
                      name = "sy",
                      type = "number",
                      description = "The y scale."
                    },
                    {
                      name = "sz",
                      type = "number",
                      description = "The z scale."
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the node is rotated around its axis of rotation."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    }
                  }
                }
              }
            },
            {
              name = "getRootNode",
              summary = "Get the index of the root node.",
              description = "Returns the index of the model's root node.",
              key = "ModelData:getRootNode",
              module = "lovr.data",
              related = {
                "ModelData:getNodeCount",
                "ModelData:getNodeParent",
                "Model:getRootNode"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "root",
                      type = "number",
                      description = "The index of the root node."
                    }
                  }
                }
              }
            },
            {
              name = "getSkinCount",
              summary = "Get the number of skins in the model.",
              description = "Returns the number of skins in the model.  A skin is a collection of joints targeted by an animation.",
              key = "ModelData:getSkinCount",
              module = "lovr.data",
              notes = "There is currently a maximum of 256 skins.",
              related = {
                "Model:hasJoints"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of skins in the model."
                    }
                  }
                }
              }
            },
            {
              name = "getSkinInverseBindMatrix",
              summary = "Get the inverse bind matrix for a joint in the skin.",
              description = "Returns the inverse bind matrix for a joint in the skin.",
              key = "ModelData:getSkinInverseBindMatrix",
              module = "lovr.data",
              variants = {
                {
                  arguments = {
                    {
                      name = "skin",
                      type = "number",
                      description = "The index of a skin."
                    },
                    {
                      name = "joint",
                      type = "number",
                      description = "The index of a joint in the skin."
                    }
                  },
                  returns = {
                    {
                      name = "...",
                      type = "number",
                      description = "The 16 components of the 4x4 inverse bind matrix, in column-major order."
                    }
                  }
                }
              }
            },
            {
              name = "getSkinJoints",
              summary = "Get the joints in a skin.",
              description = "Returns a table with the node indices of the joints in a skin.",
              key = "ModelData:getSkinJoints",
              module = "lovr.data",
              variants = {
                {
                  arguments = {
                    {
                      name = "skin",
                      type = "number",
                      description = "The index of a skin."
                    }
                  },
                  returns = {
                    {
                      name = "joints",
                      type = "table",
                      description = "The joints in the skin."
                    }
                  }
                }
              }
            },
            {
              name = "getTriangleCount",
              summary = "Get the total number of triangles in the model.",
              description = "Returns the total number of triangles in the model.  This count includes meshes that are attached to multiple nodes, and the count corresponds to the triangles returned by `ModelData:getTriangles`.",
              key = "ModelData:getTriangleCount",
              module = "lovr.data",
              related = {
                "ModelData:getTriangles",
                "ModelData:getVertexCount",
                "Model:getTriangleCount"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The total number of triangles in the model."
                    }
                  }
                }
              }
            },
            {
              name = "getTriangles",
              summary = "Get all the triangles in the model.",
              description = "Returns the data for all triangles in the model.  There are a few differences between this and the mesh-specific functions like `ModelData:getMeshVertex` and `ModelData:getMeshIndex`:\n\n- Only vertex positions are returned, not other vertex attributes.\n- Positions are relative to the origin of the whole model, instead of local to a node.\n- If a mesh is attached to more than one node, its vertices will be in the table multiple times.\n- Vertex indices will be relative to the whole triangle list instead of a mesh.",
              key = "ModelData:getTriangles",
              module = "lovr.data",
              notes = "After this function is called on a ModelData once, the result is cached.",
              related = {
                "ModelData:getTriangleCount",
                "ModelData:getVertexCount",
                "Model:getTriangles"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "vertices",
                      type = "table",
                      description = "The triangle vertex positions, returned as a flat (non-nested) table of numbers.  The position of each vertex is given as an x, y, and z coordinate."
                    },
                    {
                      name = "indices",
                      type = "table",
                      description = "The vertex indices.  Every 3 indices describes a triangle."
                    }
                  }
                }
              }
            },
            {
              name = "getVertexCount",
              summary = "Get the total vertex count of the model.",
              description = "Returns the total vertex count of a model.  This count includes meshes that are attached to multiple nodes, and the count corresponds to the vertices returned by `ModelData:getTriangles`.",
              key = "ModelData:getVertexCount",
              module = "lovr.data",
              related = {
                "ModelData:getTriangles",
                "ModelData:getTriangleCount",
                "Model:getVertexCount"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The total number of vertices in the model."
                    }
                  }
                }
              }
            },
            {
              name = "getWidth",
              summary = "Get the width of the model.",
              description = "Returns the width of the model, computed from its axis-aligned bounding box.",
              key = "ModelData:getWidth",
              module = "lovr.data",
              related = {
                "ModelData:getHeight",
                "ModelData:getDepth",
                "ModelData:getDimensions",
                "ModelData:getCenter",
                "ModelData:getBoundingBox",
                "Model:getWidth"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "width",
                      type = "number",
                      description = "The width of the model."
                    }
                  }
                }
              }
            }
          }
        },
        {
          name = "Rasterizer",
          summary = "An object that rasterizes glyphs from font files.",
          description = "A Rasterizer is an object that parses a TTF file, decoding and rendering glyphs from it.\n\nUsually you can just use `Font` objects.",
          key = "Rasterizer",
          module = "lovr.data",
          constructors = {
            "lovr.data.newRasterizer"
          },
          methods = {
            {
              name = "getAdvance",
              summary = "Get the advance of a glyph.",
              description = "Returns the advance metric for a glyph, in pixels.  The advance is the horizontal distance to advance the cursor after rendering the glyph.",
              key = "Rasterizer:getAdvance",
              module = "lovr.data",
              variants = {
                {
                  arguments = {
                    {
                      name = "character",
                      type = "string",
                      description = "A character."
                    }
                  },
                  returns = {
                    {
                      name = "advance",
                      type = "number",
                      description = "The advance of the glyph, in pixels."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "codepoint",
                      type = "number",
                      description = "A codepoint."
                    }
                  },
                  returns = {
                    {
                      name = "advance",
                      type = "number",
                      description = "The advance of the glyph, in pixels."
                    }
                  }
                }
              }
            },
            {
              name = "getAscent",
              summary = "Get the ascent of the font.",
              description = "Returns the ascent metric of the font, in pixels.  The ascent represents how far any glyph of the font ascends above the baseline.",
              key = "Rasterizer:getAscent",
              module = "lovr.data",
              related = {
                "Rasterizer:getDescent",
                "Font:getAscent"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "ascent",
                      type = "number",
                      description = "The ascent of the font, in pixels."
                    }
                  }
                }
              }
            },
            {
              name = "getBearing",
              summary = "Get the bearing of a glyph.",
              description = "Returns the bearing metric for a glyph, in pixels.  The bearing is the horizontal distance from the cursor to the edge of the glyph.",
              key = "Rasterizer:getBearing",
              module = "lovr.data",
              variants = {
                {
                  arguments = {
                    {
                      name = "character",
                      type = "string",
                      description = "A character."
                    }
                  },
                  returns = {
                    {
                      name = "bearing",
                      type = "number",
                      description = "The bearing of the glyph, in pixels."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "codepoint",
                      type = "number",
                      description = "A codepoint."
                    }
                  },
                  returns = {
                    {
                      name = "bearing",
                      type = "number",
                      description = "The bearing of the glyph, in pixels."
                    }
                  }
                }
              }
            },
            {
              name = "getBoundingBox",
              summary = "Get the bounding box of a glyph, or the font.",
              description = "Returns the bounding box of a glyph, or the bounding box surrounding all glyphs.  Note that font coordinates use a cartesian \"y up\" coordinate system.",
              key = "Rasterizer:getBoundingBox",
              module = "lovr.data",
              related = {
                "Rasterizer:getWidth",
                "Rasterizer:getHeight",
                "Rasterizer:getDimensions"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "character",
                      type = "string",
                      description = "A character."
                    }
                  },
                  returns = {
                    {
                      name = "x1",
                      type = "number",
                      description = "The left edge of the bounding box, in pixels."
                    },
                    {
                      name = "y1",
                      type = "number",
                      description = "The bottom edge of the bounding box, in pixels."
                    },
                    {
                      name = "x2",
                      type = "number",
                      description = "The right edge of the bounding box, in pixels."
                    },
                    {
                      name = "y2",
                      type = "number",
                      description = "The top edge of the bounding box, in pixels."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "codepoint",
                      type = "number",
                      description = "A codepoint."
                    }
                  },
                  returns = {
                    {
                      name = "x1",
                      type = "number",
                      description = "The left edge of the bounding box, in pixels."
                    },
                    {
                      name = "y1",
                      type = "number",
                      description = "The bottom edge of the bounding box, in pixels."
                    },
                    {
                      name = "x2",
                      type = "number",
                      description = "The right edge of the bounding box, in pixels."
                    },
                    {
                      name = "y2",
                      type = "number",
                      description = "The top edge of the bounding box, in pixels."
                    }
                  }
                },
                {
                  arguments = {},
                  returns = {
                    {
                      name = "x1",
                      type = "number",
                      description = "The left edge of the bounding box, in pixels."
                    },
                    {
                      name = "y1",
                      type = "number",
                      description = "The bottom edge of the bounding box, in pixels."
                    },
                    {
                      name = "x2",
                      type = "number",
                      description = "The right edge of the bounding box, in pixels."
                    },
                    {
                      name = "y2",
                      type = "number",
                      description = "The top edge of the bounding box, in pixels."
                    }
                  }
                }
              }
            },
            {
              name = "getCurves",
              summary = "Get the bezier curves defining a glyph.",
              description = "Returns the bezier curve control points defining the shape of a glyph.",
              key = "Rasterizer:getCurves",
              module = "lovr.data",
              related = {
                "Curve",
                "Rasterizer:newImage"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "character",
                      type = "string",
                      description = "A character."
                    },
                    {
                      name = "three",
                      type = "boolean",
                      description = "Whether the control points should be 3D or 2D."
                    }
                  },
                  returns = {
                    {
                      name = "curves",
                      type = "table",
                      description = "A table of curves.  Each curve is a table of numbers representing the control points (2 for a line, 3 for a quadratic curve, etc.)."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "codepoint",
                      type = "number",
                      description = "A codepoint."
                    },
                    {
                      name = "three",
                      type = "boolean",
                      description = "Whether the control points should be 3D or 2D."
                    }
                  },
                  returns = {
                    {
                      name = "curves",
                      type = "table",
                      description = "A table of curves.  Each curve is a table of numbers representing the control points (2 for a line, 3 for a quadratic curve, etc.)."
                    }
                  }
                }
              }
            },
            {
              name = "getDescent",
              summary = "Get the descent of the font.",
              description = "Returns the descent metric of the font, in pixels.  The descent represents how far any glyph of the font descends below the baseline.",
              key = "Rasterizer:getDescent",
              module = "lovr.data",
              related = {
                "Rasterizer:getAscent",
                "Font:getDescent"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "descent",
                      type = "number",
                      description = "The descent of the font, in pixels."
                    }
                  }
                }
              }
            },
            {
              name = "getDimensions",
              summary = "Get the dimensions of a glyph, or the font.",
              description = "Returns the dimensions of a glyph, or the dimensions of any glyph.",
              key = "Rasterizer:getDimensions",
              module = "lovr.data",
              related = {
                "Rasterizer:getWidth",
                "Rasterizer:getHeight",
                "Rasterizer:getBoundingBox"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "character",
                      type = "string",
                      description = "A character."
                    }
                  },
                  returns = {
                    {
                      name = "width",
                      type = "number",
                      description = "The width, in pixels."
                    },
                    {
                      name = "height",
                      type = "number",
                      description = "The height, in pixels."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "codepoint",
                      type = "number",
                      description = "A codepoint."
                    }
                  },
                  returns = {
                    {
                      name = "width",
                      type = "number",
                      description = "The width, in pixels."
                    },
                    {
                      name = "height",
                      type = "number",
                      description = "The height, in pixels."
                    }
                  }
                },
                {
                  arguments = {},
                  returns = {
                    {
                      name = "width",
                      type = "number",
                      description = "The width, in pixels."
                    },
                    {
                      name = "height",
                      type = "number",
                      description = "The height, in pixels."
                    }
                  }
                }
              }
            },
            {
              name = "getFontSize",
              summary = "Get the size of the font.",
              description = "Returns the size of the font, in pixels.  This is the size the rasterizer was created with, and defines the size of images it rasterizes.",
              key = "Rasterizer:getFontSize",
              module = "lovr.data",
              related = {
                "Rasterizer:getHeight"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "size",
                      type = "number",
                      description = "The font size, in pixels."
                    }
                  }
                }
              }
            },
            {
              name = "getGlyphCount",
              summary = "Get the number of glyphs stored in the font file.",
              description = "Returns the number of glyphs stored in the font file.",
              key = "Rasterizer:getGlyphCount",
              module = "lovr.data",
              related = {
                "Rasterizer:hasGlyphs"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of glyphs stored in the font file."
                    }
                  }
                }
              }
            },
            {
              name = "getHeight",
              summary = "Get the height of a glyph, or the font.",
              description = "Returns the height of a glyph, or the maximum height of any glyph.",
              key = "Rasterizer:getHeight",
              module = "lovr.data",
              related = {
                "Rasterizer:getWidth",
                "Rasterizer:getDimensions",
                "Rasterizer:getBoundingBox"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "character",
                      type = "string",
                      description = "A character."
                    }
                  },
                  returns = {
                    {
                      name = "height",
                      type = "number",
                      description = "The height, in pixels."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "codepoint",
                      type = "number",
                      description = "A codepoint."
                    }
                  },
                  returns = {
                    {
                      name = "height",
                      type = "number",
                      description = "The height, in pixels."
                    }
                  }
                },
                {
                  arguments = {},
                  returns = {
                    {
                      name = "height",
                      type = "number",
                      description = "The height, in pixels."
                    }
                  }
                }
              }
            },
            {
              name = "getKerning",
              summary = "Get the kerning between two glyphs.",
              description = "Returns the kerning between 2 glyphs, in pixels.  Kerning is a slight horizontal adjustment between 2 glyphs to improve the visual appearance.  It will often be negative.",
              key = "Rasterizer:getKerning",
              module = "lovr.data",
              related = {
                "Font:getKerning"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "first",
                      type = "string",
                      description = "The first character."
                    },
                    {
                      name = "second",
                      type = "string",
                      description = "The second character."
                    }
                  },
                  returns = {
                    {
                      name = "keming",
                      type = "number",
                      description = "The kerning between the two glyphs."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "firstCodepoint",
                      type = "number",
                      description = "The first codepoint."
                    },
                    {
                      name = "second",
                      type = "string",
                      description = "The second character."
                    }
                  },
                  returns = {
                    {
                      name = "keming",
                      type = "number",
                      description = "The kerning between the two glyphs."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "first",
                      type = "string",
                      description = "The first character."
                    },
                    {
                      name = "secondCodepoint",
                      type = "number",
                      description = "The second codepoint."
                    }
                  },
                  returns = {
                    {
                      name = "keming",
                      type = "number",
                      description = "The kerning between the two glyphs."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "firstCodepoint",
                      type = "number",
                      description = "The first codepoint."
                    },
                    {
                      name = "secondCodepoint",
                      type = "number",
                      description = "The second codepoint."
                    }
                  },
                  returns = {
                    {
                      name = "keming",
                      type = "number",
                      description = "The kerning between the two glyphs."
                    }
                  }
                }
              }
            },
            {
              name = "getLeading",
              summary = "Get the leading of the font.",
              description = "Returns the leading metric of the font, in pixels.  This is the full amount of space between lines.",
              key = "Rasterizer:getLeading",
              module = "lovr.data",
              related = {
                "Rasterizer:getAscent",
                "Rasterizer:getDescent"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "leading",
                      type = "number",
                      description = "The font leading, in pixels."
                    }
                  }
                }
              }
            },
            {
              name = "getWidth",
              summary = "Get the width of a glyph, or the font.",
              description = "Returns the width of a glyph, or the maximum width of any glyph.",
              key = "Rasterizer:getWidth",
              module = "lovr.data",
              related = {
                "Rasterizer:getHeight",
                "Rasterizer:getDimensions",
                "Rasterizer:getBoundingBox"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "character",
                      type = "string",
                      description = "A character."
                    }
                  },
                  returns = {
                    {
                      name = "width",
                      type = "number",
                      description = "The width, in pixels."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "codepoint",
                      type = "number",
                      description = "A codepoint."
                    }
                  },
                  returns = {
                    {
                      name = "width",
                      type = "number",
                      description = "The width, in pixels."
                    }
                  }
                },
                {
                  arguments = {},
                  returns = {
                    {
                      name = "width",
                      type = "number",
                      description = "The width, in pixels."
                    }
                  }
                }
              }
            },
            {
              name = "hasGlyphs",
              summary = "Get whether the Rasterizer can rasterize a set of glyphs.",
              description = "Returns whether the Rasterizer can rasterize a set of glyphs.",
              key = "Rasterizer:hasGlyphs",
              module = "lovr.data",
              related = {
                "Rasterizer:getGlyphCount"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "...",
                      type = "*",
                      description = "Strings (sets of characters) or numbers (character codes) to check for."
                    }
                  },
                  returns = {
                    {
                      name = "hasGlyphs",
                      type = "boolean",
                      description = "true if the Rasterizer can rasterize all of the supplied characters, false otherwise."
                    }
                  }
                }
              }
            },
            {
              name = "newImage",
              summary = "Get an Image of a rasterized glyph.",
              description = "Returns an `Image` containing a rasterized glyph.",
              key = "Rasterizer:newImage",
              module = "lovr.data",
              related = {
                "Rasterizer:getCurves"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "character",
                      type = "string",
                      description = "A character."
                    },
                    {
                      name = "spread",
                      type = "number",
                      description = "The width of the distance field, for signed distance field rasterization.",
                      default = "4.0"
                    },
                    {
                      name = "padding",
                      type = "number",
                      description = "The number of pixels of padding to add at the edges of the image.",
                      default = "spread / 2"
                    }
                  },
                  returns = {
                    {
                      name = "image",
                      type = "Image",
                      description = "The glyph image.  It will be in the `rgba32f` format."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "codepoint",
                      type = "number",
                      description = "A codepoint."
                    },
                    {
                      name = "spread",
                      type = "number",
                      description = "The width of the distance field, for signed distance field rasterization.",
                      default = "4.0"
                    },
                    {
                      name = "padding",
                      type = "number",
                      description = "The number of pixels of padding to add at the edges of the image.",
                      default = "spread / 2"
                    }
                  },
                  returns = {
                    {
                      name = "image",
                      type = "Image",
                      description = "The glyph image.  It will be in the `rgba32f` format."
                    }
                  }
                }
              }
            }
          }
        },
        {
          name = "Sound",
          summary = "An object that holds raw audio samples.",
          description = "A Sound stores the data for a sound.  The supported sound formats are OGG, WAV, and MP3.  Sounds cannot be played directly.  Instead, there are `Source` objects in `lovr.audio` that are used for audio playback.  All Source objects are backed by one of these Sounds, and multiple Sources can share a single Sound to reduce memory usage.\n\nMetadata\n---\n\nSounds hold a fixed number of frames.  Each frame contains one audio sample for each channel. The `SampleFormat` of the Sound is the data type used for each sample (floating point, integer, etc.).  The Sound has a `ChannelLayout`, representing the number of audio channels and how they map to speakers (mono, stereo, etc.).  The sample rate of the Sound indicates how many frames should be played per second.  The duration of the sound (in seconds) is the number of frames divided by the sample rate.\n\nCompression\n---\n\nSounds can be compressed.  Compressed sounds are stored compressed in memory and are decoded as they are played.  This uses a lot less memory but increases CPU usage during playback.  OGG and MP3 are compressed audio formats.  When creating a sound from a compressed format, there is an option to immediately decode it, storing it uncompressed in memory.  It can be a good idea to decode short sound effects, since they won't use very much memory even when uncompressed and it will improve CPU usage.  Compressed sounds can not be written to using `Sound:setFrames`.\n\nStreams\n---\n\nSounds can be created as a stream by passing `'stream'` as their contents when creating them. Audio frames can be written to the end of the stream, and read from the beginning.  This works well for situations where data is being generated in real time or streamed in from some other data source.\n\nSources can be backed by a stream and they'll just play whatever audio is pushed to the stream. The audio module also lets you use a stream as a \"sink\" for an audio device.  For playback devices, this works like loopback, so the mixed audio from all playing Sources will get written to the stream.  For capture devices, all the microphone input will get written to the stream. Conversion between sample formats, channel layouts, and sample rates will happen automatically.\n\nKeep in mind that streams can still only hold a fixed number of frames.  If too much data is written before it is read, older frames will start to get overwritten.  Similary, it's possible to read too much data without writing fast enough.\n\nAmbisonics\n---\n\nAmbisonic sounds can be imported from WAVs, but can not yet be played.  Sounds with a `ChannelLayout` of `ambisonic` are stored as first-order full-sphere ambisonics using the AmbiX format (ACN channel ordering and SN3D channel normalization).  The AMB format is supported for import and will automatically get converted to AmbiX.  See `lovr.data.newSound` for more info.",
          key = "Sound",
          module = "lovr.data",
          constructors = {
            "lovr.data.newSound"
          },
          methods = {
            {
              name = "getBlob",
              summary = "Get the bytes backing this Sound as a Blob.",
              description = "Returns a Blob containing the raw bytes of the Sound.",
              key = "Sound:getBlob",
              module = "lovr.data",
              notes = "Samples for each channel are stored interleaved.  The data type of each sample is given by `Sound:getFormat`.",
              related = {
                "Blob:getPointer",
                "Image:getBlob"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "blob",
                      type = "Blob",
                      description = "The Blob instance containing the bytes for the `Sound`."
                    }
                  }
                }
              }
            },
            {
              name = "getCapacity",
              summary = "Get the number of frames that can be written to the Sound.",
              description = "Returns the number of frames that can be written to the Sound.  For stream sounds, this is the number of frames that can be written without overwriting existing data.  For normal sounds, this returns the same value as `Sound:getFrameCount`.",
              key = "Sound:getCapacity",
              module = "lovr.data",
              related = {
                "Sound:getFrameCount",
                "Sound:getSampleCount",
                "Source:getDuration"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "capacity",
                      type = "number",
                      description = "The number of frames that can be written to the Sound."
                    }
                  }
                }
              }
            },
            {
              name = "getChannelCount",
              summary = "Get the number of channels in the Sound.",
              description = "Returns the number of channels in the Sound.  Mono sounds have 1 channel, stereo sounds have 2 channels, and ambisonic sounds have 4 channels.",
              key = "Sound:getChannelCount",
              module = "lovr.data",
              related = {
                "Sound:getChannelLayout"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "channels",
                      type = "number",
                      description = "The number of channels in the sound."
                    }
                  }
                }
              }
            },
            {
              name = "getChannelLayout",
              summary = "Get the channel layout of the Sound.",
              description = "Returns the channel layout of the Sound.",
              key = "Sound:getChannelLayout",
              module = "lovr.data",
              related = {
                "Sound:getChannelCount"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "channels",
                      type = "ChannelLayout",
                      description = "The channel layout."
                    }
                  }
                }
              }
            },
            {
              name = "getDuration",
              summary = "Get the duration of the Sound.",
              description = "Returns the duration of the Sound, in seconds.",
              key = "Sound:getDuration",
              module = "lovr.data",
              notes = "This can be computed as `(frameCount / sampleRate)`.",
              related = {
                "Sound:getFrameCount",
                "Sound:getSampleCount",
                "Sound:getSampleRate",
                "Source:getDuration"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "duration",
                      type = "number",
                      description = "The duration of the Sound, in seconds."
                    }
                  }
                }
              }
            },
            {
              name = "getFormat",
              summary = "Get the sample format of the Sound.",
              description = "Returns the sample format of the Sound.",
              key = "Sound:getFormat",
              module = "lovr.data",
              related = {
                "Sound:getChannelLayout",
                "Sound:getSampleRate"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "format",
                      type = "SampleFormat",
                      description = "The data type of each sample."
                    }
                  }
                }
              }
            },
            {
              name = "getFrameCount",
              summary = "Get the number of frames in the Sound.",
              description = "Returns the number of frames in the Sound.  A frame stores one sample for each channel.",
              key = "Sound:getFrameCount",
              module = "lovr.data",
              notes = "For streams, this returns the number of frames in the stream's buffer.",
              related = {
                "Sound:getDuration",
                "Sound:getSampleCount",
                "Sound:getChannelCount"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "frames",
                      type = "number",
                      description = "The number of frames in the Sound."
                    }
                  }
                }
              }
            },
            {
              name = "getFrames",
              summary = "Read frames from the Sound.",
              description = "Reads frames from the Sound into a table, Blob, or another Sound.",
              key = "Sound:getFrames",
              module = "lovr.data",
              variants = {
                {
                  arguments = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of frames to read.  If nil, reads as many frames as possible.\n\nCompressed sounds will automatically be decoded.\n\nReading from a stream will ignore the source offset and read the oldest frames.",
                      default = "nil"
                    },
                    {
                      name = "srcOffset",
                      type = "number",
                      description = "A frame offset to apply to the sound when reading frames.",
                      default = "0"
                    }
                  },
                  returns = {
                    {
                      name = "t",
                      type = "table",
                      description = "A table containing audio frames."
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "The number of frames read."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "t",
                      type = "table",
                      description = "An existing table to read frames into."
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "The number of frames to read.  If nil, reads as many frames as possible.\n\nCompressed sounds will automatically be decoded.\n\nReading from a stream will ignore the source offset and read the oldest frames.",
                      default = "nil"
                    },
                    {
                      name = "srcOffset",
                      type = "number",
                      description = "A frame offset to apply to the sound when reading frames.",
                      default = "0"
                    },
                    {
                      name = "dstOffset",
                      type = "number",
                      description = "An offset to apply to the destination when writing frames (indices for tables, bytes for Blobs, frames for Sounds).",
                      default = "0"
                    }
                  },
                  returns = {
                    {
                      name = "t",
                      type = "table",
                      description = "A table containing audio frames."
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "The number of frames read."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "blob",
                      type = "Blob",
                      description = "A Blob to read frames into."
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "The number of frames to read.  If nil, reads as many frames as possible.\n\nCompressed sounds will automatically be decoded.\n\nReading from a stream will ignore the source offset and read the oldest frames.",
                      default = "nil"
                    },
                    {
                      name = "srcOffset",
                      type = "number",
                      description = "A frame offset to apply to the sound when reading frames.",
                      default = "0"
                    },
                    {
                      name = "dstOffset",
                      type = "number",
                      description = "An offset to apply to the destination when writing frames (indices for tables, bytes for Blobs, frames for Sounds).",
                      default = "0"
                    }
                  },
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of frames read."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "sound",
                      type = "Sound",
                      description = "Another Sound to copy frames into."
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "The number of frames to read.  If nil, reads as many frames as possible.\n\nCompressed sounds will automatically be decoded.\n\nReading from a stream will ignore the source offset and read the oldest frames.",
                      default = "nil"
                    },
                    {
                      name = "srcOffset",
                      type = "number",
                      description = "A frame offset to apply to the sound when reading frames.",
                      default = "0"
                    },
                    {
                      name = "dstOffset",
                      type = "number",
                      description = "An offset to apply to the destination when writing frames (indices for tables, bytes for Blobs, frames for Sounds).",
                      default = "0"
                    }
                  },
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of frames read."
                    }
                  }
                }
              }
            },
            {
              name = "getSampleCount",
              summary = "Get the number of samples in the Sound.",
              description = "Returns the total number of samples in the Sound.",
              key = "Sound:getSampleCount",
              module = "lovr.data",
              notes = "For streams, this returns the number of samples in the stream's buffer.",
              related = {
                "Sound:getDuration",
                "Sound:getFrameCount",
                "Sound:getChannelCount"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "samples",
                      type = "number",
                      description = "The total number of samples in the Sound."
                    }
                  }
                }
              }
            },
            {
              name = "getSampleRate",
              summary = "Get the sample rate of the Sound.",
              description = "Returns the sample rate of the Sound, in Hz.  This is the number of frames that are played every second.  It's usually a high number like 48000.",
              key = "Sound:getSampleRate",
              module = "lovr.data",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "frequency",
                      type = "number",
                      description = "The number of frames per second in the Sound."
                    }
                  }
                }
              }
            },
            {
              name = "isCompressed",
              summary = "Check if the Sound is compressed.",
              description = "Returns whether the Sound is compressed.  Compressed sounds are loaded from compressed audio formats like MP3 and OGG.  They use a lot less memory but require some extra CPU work during playback.  Compressed sounds can not be modified using `Sound:setFrames`.",
              key = "Sound:isCompressed",
              module = "lovr.data",
              related = {
                "Sound:isStream",
                "lovr.data.newSound"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "compressed",
                      type = "boolean",
                      description = "Whether the Sound is compressed."
                    }
                  }
                }
              }
            },
            {
              name = "isStream",
              summary = "Check if the Sound is a stream.",
              description = "Returns whether the Sound is a stream.",
              key = "Sound:isStream",
              module = "lovr.data",
              related = {
                "Sound:isCompressed",
                "lovr.data.newSound"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "stream",
                      type = "boolean",
                      description = "Whether the Sound is a stream."
                    }
                  }
                }
              }
            },
            {
              name = "setFrames",
              summary = "Write frames to the Sound.",
              description = "Writes frames to the Sound.",
              key = "Sound:setFrames",
              module = "lovr.data",
              examples = {
                {
                  description = "Generate a sine wave.",
                  code = "function lovr.load()\n  local length = 1\n  local rate = 48000\n  local frames = length * rate\n  local frequency = 440\n  local volume = 1.0\n\n  sound = lovr.data.newSound(frames, 'f32', 'stereo', rate)\n\n  local data = {}\n  for i = 1, frames do\n    local amplitude = math.sin((i - 1) * frequency / rate * (2 * math.pi)) * volume\n    data[2 * i - 1] = amplitude\n    data[2 * i - 0] = amplitude\n  end\n\n  sound:setFrames(data)\n\n  source = lovr.audio.newSource(sound)\n  source:setLooping(true)\n  source:play()\nend"
                }
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "t",
                      type = "table",
                      description = "A table containing frames to write."
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "How many frames to write.  If nil, writes as many as possible.",
                      default = "nil"
                    },
                    {
                      name = "dstOffset",
                      type = "number",
                      description = "A frame offset to apply when writing the frames.",
                      default = "0"
                    },
                    {
                      name = "srcOffset",
                      type = "number",
                      description = "A frame, byte, or index offset to apply when reading frames from the source.",
                      default = "0"
                    }
                  },
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of frames written."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "blob",
                      type = "Blob",
                      description = "A Blob containing frames to write."
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "How many frames to write.  If nil, writes as many as possible.",
                      default = "nil"
                    },
                    {
                      name = "dstOffset",
                      type = "number",
                      description = "A frame offset to apply when writing the frames.",
                      default = "0"
                    },
                    {
                      name = "srcOffset",
                      type = "number",
                      description = "A frame, byte, or index offset to apply when reading frames from the source.",
                      default = "0"
                    }
                  },
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of frames written."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "sound",
                      type = "Sound",
                      description = "Another Sound to copy frames from."
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "How many frames to write.  If nil, writes as many as possible.",
                      default = "nil"
                    },
                    {
                      name = "dstOffset",
                      type = "number",
                      description = "A frame offset to apply when writing the frames.",
                      default = "0"
                    },
                    {
                      name = "srcOffset",
                      type = "number",
                      description = "A frame, byte, or index offset to apply when reading frames from the source.",
                      default = "0"
                    }
                  },
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of frames written."
                    }
                  }
                }
              }
            }
          }
        }
      }
    },
    {
      name = "event",
      tag = "modules",
      summary = "Handles events from the operating system.",
      description = "The `lovr.event` module handles events from the operating system.\n\nDue to its low-level nature, it's rare to use `lovr.event` in simple projects.",
      key = "lovr.event",
      enums = {
        {
          name = "KeyCode",
          summary = "Keys that can be pressed.",
          description = "Keys that can be pressed on a keyboard.  Notably, numpad keys are missing right now.",
          key = "KeyCode",
          module = "lovr.event",
          related = {
            "lovr.keypressed",
            "lovr.keyreleased"
          },
          values = {
            {
              name = "a",
              description = "The A key."
            },
            {
              name = "b",
              description = "The B key."
            },
            {
              name = "c",
              description = "The C key."
            },
            {
              name = "d",
              description = "The D key."
            },
            {
              name = "e",
              description = "The E key."
            },
            {
              name = "f",
              description = "The F key."
            },
            {
              name = "g",
              description = "The G key."
            },
            {
              name = "h",
              description = "The H key."
            },
            {
              name = "i",
              description = "The I key."
            },
            {
              name = "j",
              description = "The J key."
            },
            {
              name = "k",
              description = "The K key."
            },
            {
              name = "l",
              description = "The L key."
            },
            {
              name = "m",
              description = "The M key."
            },
            {
              name = "n",
              description = "The N key."
            },
            {
              name = "o",
              description = "The O key."
            },
            {
              name = "p",
              description = "The P key."
            },
            {
              name = "q",
              description = "The Q key."
            },
            {
              name = "r",
              description = "The R key."
            },
            {
              name = "s",
              description = "The S key."
            },
            {
              name = "t",
              description = "The T key."
            },
            {
              name = "u",
              description = "The U key."
            },
            {
              name = "v",
              description = "The V key."
            },
            {
              name = "w",
              description = "The W key."
            },
            {
              name = "x",
              description = "The X key."
            },
            {
              name = "y",
              description = "The Y key."
            },
            {
              name = "z",
              description = "The Z key."
            },
            {
              name = "0",
              description = "The 0 key."
            },
            {
              name = "1",
              description = "The 1 key."
            },
            {
              name = "2",
              description = "The 2 key."
            },
            {
              name = "3",
              description = "The 3 key."
            },
            {
              name = "4",
              description = "The 4 key."
            },
            {
              name = "5",
              description = "The 5 key."
            },
            {
              name = "6",
              description = "The 6 key."
            },
            {
              name = "7",
              description = "The 7 key."
            },
            {
              name = "8",
              description = "The 8 key."
            },
            {
              name = "9",
              description = "The 9 key."
            },
            {
              name = "space",
              description = "The space bar."
            },
            {
              name = "return",
              description = "The enter key."
            },
            {
              name = "tab",
              description = "The tab key."
            },
            {
              name = "escape",
              description = "The escape key."
            },
            {
              name = "backspace",
              description = "The backspace key."
            },
            {
              name = "up",
              description = "The up arrow key."
            },
            {
              name = "down",
              description = "The down arrow key."
            },
            {
              name = "left",
              description = "The left arrow key."
            },
            {
              name = "right",
              description = "The right arrow key."
            },
            {
              name = "home",
              description = "The home key."
            },
            {
              name = "end",
              description = "The end key."
            },
            {
              name = "pageup",
              description = "The page up key."
            },
            {
              name = "pagedown",
              description = "The page down key."
            },
            {
              name = "insert",
              description = "The insert key."
            },
            {
              name = "delete",
              description = "The delete key."
            },
            {
              name = "f1",
              description = "The F1 key."
            },
            {
              name = "f2",
              description = "The F2 key."
            },
            {
              name = "f3",
              description = "The F3 key."
            },
            {
              name = "f4",
              description = "The F4 key."
            },
            {
              name = "f5",
              description = "The F5 key."
            },
            {
              name = "f6",
              description = "The F6 key."
            },
            {
              name = "f7",
              description = "The F7 key."
            },
            {
              name = "f8",
              description = "The F8 key."
            },
            {
              name = "f9",
              description = "The F9 key."
            },
            {
              name = "f10",
              description = "The F10 key."
            },
            {
              name = "f11",
              description = "The F11 key."
            },
            {
              name = "f12",
              description = "The F12 key."
            },
            {
              name = "`",
              description = "The backtick/backquote/grave accent key."
            },
            {
              name = "-",
              description = "The dash/hyphen/minus key."
            },
            {
              name = "=",
              description = "The equal sign key."
            },
            {
              name = "[",
              description = "The left bracket key."
            },
            {
              name = "]",
              description = "The right bracket key."
            },
            {
              name = "\\",
              description = "The backslash key."
            },
            {
              name = ";",
              description = "The semicolon key."
            },
            {
              name = "'",
              description = "The single quote key."
            },
            {
              name = ",",
              description = "The comma key."
            },
            {
              name = ".",
              description = "The period key."
            },
            {
              name = "/",
              description = "The slash key."
            },
            {
              name = "lctrl",
              description = "The left control key."
            },
            {
              name = "lshift",
              description = "The left shift key."
            },
            {
              name = "lalt",
              description = "The left alt key."
            },
            {
              name = "lgui",
              description = "The left OS key (windows, command, super)."
            },
            {
              name = "rctrl",
              description = "The right control key."
            },
            {
              name = "rshift",
              description = "The right shift key."
            },
            {
              name = "ralt",
              description = "The right alt key."
            },
            {
              name = "rgui",
              description = "The right OS key (windows, command, super)."
            },
            {
              name = "capslock",
              description = "The caps lock key."
            },
            {
              name = "scrolllock",
              description = "The scroll lock key."
            },
            {
              name = "numlock",
              description = "The numlock key."
            }
          }
        }
      },
      examples = {
        {
          description = "Adding a custom event.",
          code = "function lovr.load()\n  lovr.handlers['customevent'] = function(a, b, c)\n    print('custom event handled with args:', a, b, c)\n  end\n\n  lovr.event.push('customevent', 1, 2, 3)\nend"
        }
      },
      functions = {
        {
          name = "clear",
          summary = "Clear the event queue.",
          description = "Clears the event queue, removing any unprocessed events.",
          key = "lovr.event.clear",
          module = "lovr.event",
          variants = {
            {
              arguments = {},
              returns = {}
            }
          }
        },
        {
          name = "poll",
          summary = "Iterate over unprocessed events in the queue.",
          description = "This function returns a Lua iterator for all of the unprocessed items in the event queue.  Each event consists of a name as a string, followed by event-specific arguments.  This function is called in the default implementation of `lovr.run`, so it is normally not necessary to poll for events yourself.",
          key = "lovr.event.poll",
          module = "lovr.event",
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "iterator",
                  type = "function",
                  description = "The iterator function, usable in a for loop.",
                  arguments = {},
                  returns = {}
                }
              }
            }
          }
        },
        {
          name = "push",
          summary = "Manually push an event onto the queue.",
          description = "Pushes an event onto the event queue.  It will be processed the next time `lovr.event.poll` is called.  For an event to be processed properly, there needs to be a function in the `lovr.handlers` table with a key that's the same as the event name.",
          key = "lovr.event.push",
          module = "lovr.event",
          notes = "Only nil, booleans, numbers, strings, and LÖVR objects are supported types for event data.",
          related = {
            "lovr.event.poll",
            "lovr.event.quit"
          },
          variants = {
            {
              arguments = {
                {
                  name = "name",
                  type = "string",
                  description = "The name of the event."
                },
                {
                  name = "...",
                  type = "*",
                  description = "The arguments for the event.  Currently, up to 4 are supported."
                }
              },
              returns = {}
            }
          }
        },
        {
          name = "quit",
          summary = "Quit the application.",
          description = "Pushes an event to quit.  An optional number can be passed to set the exit code for the application.  An exit code of zero indicates normal termination, whereas a nonzero exit code indicates that an error occurred.",
          key = "lovr.event.quit",
          module = "lovr.event",
          notes = "This function is equivalent to calling `lovr.event.push('quit', <args>)`.\n\nThe event won't be processed until the next time `lovr.event.poll` is called.\n\nThe `lovr.quit` callback will be called when the event is processed, which can be used to do any cleanup work.  The callback can also return `false` to abort the quitting process.",
          related = {
            "lovr.quit",
            "lovr.event.poll",
            "lovr.event.restart"
          },
          variants = {
            {
              arguments = {
                {
                  name = "code",
                  type = "number",
                  description = "The exit code of the program.",
                  default = "0"
                }
              },
              returns = {}
            }
          }
        },
        {
          name = "restart",
          summary = "Restart the application.",
          description = "Pushes an event to restart the framework.",
          key = "lovr.event.restart",
          module = "lovr.event",
          notes = "The event won't be processed until the next time `lovr.event.poll` is called.\n\nThe `lovr.restart` callback can be used to persist a value between restarts.",
          related = {
            "lovr.restart",
            "lovr.event.poll",
            "lovr.event.quit"
          },
          variants = {
            {
              arguments = {},
              returns = {}
            }
          }
        }
      },
      notes = "You can define your own custom events by adding a function to the `lovr.handlers` table with a key of the name of the event you want to add.  Then, push the event using `lovr.event.push`.",
      objects = {}
    },
    {
      name = "filesystem",
      tag = "modules",
      summary = "Provides access to the filesystem.",
      description = "The `lovr.filesystem` module provides access to the filesystem.\n\nAll files written will go in a special folder called the \"save directory\".  The location of the save directory is platform-specific:\n\n<table>\n  <tr>\n    <td>Windows</td>\n    <td><code>C:\\Users\\&lt;user&gt;\\AppData\\Roaming\\LOVR\\&lt;identity&gt;</code></td>\n  </tr>\n  <tr>\n    <td>macOS</td>\n    <td><code>/Users/&lt;user&gt;/Library/Application Support/LOVR/&lt;identity&gt;</code></td>\n  </tr>\n  <tr>\n    <td>Linux</td>\n    <td><code>/home/&lt;user&gt;/.local/share/LOVR/&lt;identity&gt;</code></td>\n  </tr>\n  <tr>\n    <td>Android</td>\n    <td><code>/sdcard/Android/data/&lt;identity&gt;/files</code></td>\n  </tr> </table>\n\n`<identity>` is a unique identifier for the project, and can be set in `lovr.conf`.  On Android, the identity can not be changed and will always be the package id (e.g. `org.lovr.app`).\n\nWhen files are read, they will be searched for in multiple places.  By default, the save directory is checked first, then the project source (folder or zip).  That way, when data is written to a file, any future reads will see the new data.  The `t.saveprecedence` conf setting can be used to change this precedence.\n\nConceptually, `lovr.filesystem` uses a \"virtual filesystem\", which is an ordered list of folders and zip files that are merged into a single filesystem hierarchy.  Folders and archives in the list can be added and removed with `lovr.filesystem.mount` and `lovr.filesystem.unmount`.\n\nLÖVR extends Lua's `require` function to look for modules in the virtual filesystem.  The search patterns can be changed with `lovr.filesystem.setRequirePath`, similar to `package.path`.",
      key = "lovr.filesystem",
      enums = {},
      functions = {
        {
          name = "append",
          tag = "filesystem-files",
          summary = "Append content to the end of a file.",
          description = "Appends content to the end of a file.",
          key = "lovr.filesystem.append",
          module = "lovr.filesystem",
          notes = "If the file does not exist, it is created.",
          variants = {
            {
              arguments = {
                {
                  name = "filename",
                  type = "string",
                  description = "The file to append to."
                },
                {
                  name = "content",
                  type = "string",
                  description = "A string to write to the end of the file."
                }
              },
              returns = {
                {
                  name = "bytes",
                  type = "number",
                  description = "The number of bytes actually appended to the file."
                }
              }
            },
            {
              arguments = {
                {
                  name = "filename",
                  type = "string",
                  description = "The file to append to."
                },
                {
                  name = "blob",
                  type = "Blob",
                  description = "A Blob containing data to append to the file."
                }
              },
              returns = {
                {
                  name = "bytes",
                  type = "number",
                  description = "The number of bytes actually appended to the file."
                }
              }
            }
          }
        },
        {
          name = "createDirectory",
          tag = "filesystem-files",
          summary = "Create a directory.",
          description = "Creates a directory in the save directory.  Any parent directories that don't exist will also be created.",
          key = "lovr.filesystem.createDirectory",
          module = "lovr.filesystem",
          variants = {
            {
              arguments = {
                {
                  name = "path",
                  type = "string",
                  description = "The directory to create, recursively."
                }
              },
              returns = {
                {
                  name = "success",
                  type = "boolean",
                  description = "Whether the directory was created."
                }
              }
            }
          }
        },
        {
          name = "getAppdataDirectory",
          tag = "filesystem-paths",
          summary = "Get the application data directory.",
          description = "Returns the application data directory.  This will be something like:\n\n- `C:\\Users\\user\\AppData\\Roaming` on Windows.\n- `/home/user/.config` on Linux.\n- `/Users/user/Library/Application Support` on macOS.",
          key = "lovr.filesystem.getAppdataDirectory",
          module = "lovr.filesystem",
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "path",
                  type = "string",
                  description = "The absolute path to the appdata directory."
                }
              }
            }
          }
        },
        {
          name = "getDirectoryItems",
          tag = "filesystem-files",
          summary = "Get a list of files in a directory.",
          description = "Returns a sorted table containing all files and folders in a single directory.",
          key = "lovr.filesystem.getDirectoryItems",
          module = "lovr.filesystem",
          notes = "This function calls `table.sort` to sort the results, so if `table.sort` is not available in the global scope the results are not guaranteed to be sorted.",
          variants = {
            {
              arguments = {
                {
                  name = "path",
                  type = "string",
                  description = "The directory."
                }
              },
              returns = {
                {
                  name = "table",
                  type = "items",
                  description = "A table with a string for each file and subfolder in the directory."
                }
              }
            }
          }
        },
        {
          name = "getExecutablePath",
          tag = "filesystem-paths",
          summary = "Get the path of the LÖVR executable.",
          description = "Returns the absolute path of the LÖVR executable.",
          key = "lovr.filesystem.getExecutablePath",
          module = "lovr.filesystem",
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "path",
                  type = "string",
                  description = "The absolute path of the LÖVR executable, or `nil` if it is unknown."
                }
              }
            }
          }
        },
        {
          name = "getIdentity",
          tag = "filesystem-virtual",
          summary = "Get the name of the save directory.",
          description = "Returns the identity of the game, which is used as the name of the save directory.  The default is `default`.  It can be changed using `t.identity` in `lovr.conf`.",
          key = "lovr.filesystem.getIdentity",
          module = "lovr.filesystem",
          notes = "On Android, this is always the package id (like `org.lovr.app`).",
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "identity",
                  type = "string",
                  description = "The name of the save directory, or `nil` if it isn't set."
                }
              }
            }
          }
        },
        {
          name = "getLastModified",
          tag = "filesystem-files",
          summary = "Get the modification time of a file.",
          description = "Returns when a file was last modified, since some arbitrary time in the past.",
          key = "lovr.filesystem.getLastModified",
          module = "lovr.filesystem",
          variants = {
            {
              arguments = {
                {
                  name = "path",
                  type = "string",
                  description = "The file to check."
                }
              },
              returns = {
                {
                  name = "time",
                  type = "number",
                  description = "The modification time of the file, in seconds, or `nil` if it's unknown."
                }
              }
            }
          }
        },
        {
          name = "getRealDirectory",
          tag = "filesystem-virtual",
          summary = "Get the absolute path to a file.",
          description = "Get the absolute path of the mounted archive containing a path in the virtual filesystem.  This can be used to determine if a file is in the game's source directory or the save directory.",
          key = "lovr.filesystem.getRealDirectory",
          module = "lovr.filesystem",
          variants = {
            {
              arguments = {
                {
                  name = "path",
                  type = "string",
                  description = "The path to check."
                }
              },
              returns = {
                {
                  name = "realpath",
                  type = "string",
                  description = "The absolute path of the mounted archive containing `path`."
                }
              }
            }
          }
        },
        {
          name = "getRequirePath",
          tag = "filesystem-lua",
          summary = "Get the require path.",
          description = "Returns the require path.  The require path is a semicolon-separated list of patterns that LÖVR will use to search for files when they are `require`d.  Any question marks in the pattern will be replaced with the module that is being required.  It is similar to Lua\\'s `package.path` variable, but the main difference is that the patterns are relative to the virtual filesystem.",
          key = "lovr.filesystem.getRequirePath",
          module = "lovr.filesystem",
          notes = "The default reqiure path is '?.lua;?/init.lua'.",
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "path",
                  type = "string",
                  description = "The semicolon separated list of search patterns."
                }
              }
            }
          }
        },
        {
          name = "getSaveDirectory",
          tag = "filesystem-paths",
          summary = "Get the location of the save directory.",
          description = "Returns the absolute path to the save directory.",
          key = "lovr.filesystem.getSaveDirectory",
          module = "lovr.filesystem",
          notes = "The save directory takes the following form:\n\n    <appdata>/LOVR/<identity>\n\nWhere `<appdata>` is `lovr.filesystem.getAppdataDirectory` and `<identity>` is `lovr.filesystem.getIdentity` and can be customized using `lovr.conf`.",
          related = {
            "lovr.filesystem.getIdentity",
            "lovr.filesystem.getAppdataDirectory"
          },
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "path",
                  type = "string",
                  description = "The absolute path to the save directory."
                }
              }
            }
          }
        },
        {
          name = "getSize",
          tag = "filesystem-files",
          summary = "Get the size of a file.",
          description = "Returns the size of a file, in bytes.",
          key = "lovr.filesystem.getSize",
          module = "lovr.filesystem",
          notes = "If the file does not exist, an error is thrown.",
          variants = {
            {
              arguments = {
                {
                  name = "file",
                  type = "string",
                  description = "The file."
                }
              },
              returns = {
                {
                  name = "size",
                  type = "number",
                  description = "The size of the file, in bytes."
                }
              }
            }
          }
        },
        {
          name = "getSource",
          tag = "filesystem-paths",
          summary = "Get the location of the project source.",
          description = "Get the absolute path of the project's source directory or archive.",
          key = "lovr.filesystem.getSource",
          module = "lovr.filesystem",
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "path",
                  type = "string",
                  description = "The absolute path of the project's source, or `nil` if it's unknown."
                }
              }
            }
          }
        },
        {
          name = "getUserDirectory",
          tag = "filesystem-paths",
          summary = "Get the location of the user's home directory.",
          description = "Returns the absolute path of the user's home directory.",
          key = "lovr.filesystem.getUserDirectory",
          module = "lovr.filesystem",
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "path",
                  type = "string",
                  description = "The absolute path of the user's home directory."
                }
              }
            }
          }
        },
        {
          name = "getWorkingDirectory",
          tag = "filesystem-paths",
          summary = "Get the current working directory.",
          description = "Returns the absolute path of the working directory.  Usually this is where the executable was started from.",
          key = "lovr.filesystem.getWorkingDirectory",
          module = "lovr.filesystem",
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "path",
                  type = "string",
                  description = "The current working directory, or `nil` if it's unknown."
                }
              }
            }
          }
        },
        {
          name = "isDirectory",
          tag = "filesystem-files",
          summary = "Check whether a path is a directory.",
          description = "Check if a path exists and is a directory.",
          key = "lovr.filesystem.isDirectory",
          module = "lovr.filesystem",
          related = {
            "lovr.filesystem.isFile"
          },
          variants = {
            {
              arguments = {
                {
                  name = "path",
                  type = "string",
                  description = "The path to check."
                }
              },
              returns = {
                {
                  name = "isDirectory",
                  type = "boolean",
                  description = "Whether or not the path is a directory."
                }
              }
            }
          }
        },
        {
          name = "isFile",
          tag = "filesystem-files",
          summary = "Check whether a path is a file.",
          description = "Check if a path exists and is a file.",
          key = "lovr.filesystem.isFile",
          module = "lovr.filesystem",
          related = {
            "lovr.filesystem.isDirectory"
          },
          variants = {
            {
              arguments = {
                {
                  name = "path",
                  type = "string",
                  description = "The path to check."
                }
              },
              returns = {
                {
                  name = "isFile",
                  type = "boolean",
                  description = "Whether or not the path is a file."
                }
              }
            }
          }
        },
        {
          name = "isFused",
          tag = "filesystem-virtual",
          summary = "Check if the project is fused.",
          description = "Returns whether the current project source is fused to the executable.",
          key = "lovr.filesystem.isFused",
          module = "lovr.filesystem",
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "fused",
                  type = "boolean",
                  description = "Whether or not the project is fused."
                }
              }
            }
          }
        },
        {
          name = "load",
          tag = "filesystem-lua",
          summary = "Load a file as Lua code.",
          description = "Load a file containing Lua code, returning a Lua chunk that can be run.",
          key = "lovr.filesystem.load",
          module = "lovr.filesystem",
          examples = {
            {
              description = "Safely loading code:",
              code = "local success, chunk = pcall(lovr.filesystem.load, filename)\nif not success then\n  print('Oh no! There was an error: ' .. tostring(chunk))\nelse\n  local success, result = pcall(chunk)\n  print(success, result)\nend"
            }
          },
          notes = "An error is thrown if the file contains syntax errors.",
          variants = {
            {
              arguments = {
                {
                  name = "filename",
                  type = "string",
                  description = "The file to load."
                },
                {
                  name = "mode",
                  type = "string",
                  description = "The type of code that can be loaded.  `t` allows text, `b` allows binary, and `bt` allows both.",
                  default = "'bt'"
                }
              },
              returns = {
                {
                  name = "chunk",
                  type = "function",
                  description = "The runnable chunk."
                }
              }
            }
          }
        },
        {
          name = "mount",
          tag = "filesystem-virtual",
          summary = "Mount a directory or archive.",
          description = "Mounts a directory or `.zip` archive, adding it to the virtual filesystem.  This allows you to read files from it.",
          key = "lovr.filesystem.mount",
          module = "lovr.filesystem",
          examples = {
            {
              description = "Mount `data.zip` with a file `images/background.png`:",
              code = "lovr.filesystem.mount('data.zip', 'assets')\nprint(lovr.filesystem.isFile('assets/images/background.png')) -- true"
            }
          },
          notes = "The `append` option lets you control the priority of the archive's files in the event of naming collisions.\n\nThis function is not thread safe.  Mounting or unmounting an archive while other threads call lovr.filesystem functions is not supported.",
          related = {
            "lovr.filesystem.unmount"
          },
          variants = {
            {
              arguments = {
                {
                  name = "path",
                  type = "string",
                  description = "The path to mount."
                },
                {
                  name = "mountpoint",
                  type = "string",
                  description = "The path in the virtual filesystem to mount to.",
                  default = "'/'"
                },
                {
                  name = "append",
                  type = "boolean",
                  description = "Whether the archive will be added to the end or the beginning of the search path.",
                  default = "false"
                },
                {
                  name = "root",
                  type = "string",
                  description = "A subdirectory inside the archive to use as the root.  If `nil`, the actual root of the archive is used.",
                  default = "nil"
                }
              },
              returns = {
                {
                  name = "success",
                  type = "boolean",
                  description = "Whether the archive was successfully mounted."
                }
              }
            }
          }
        },
        {
          name = "newBlob",
          tag = "filesystem-files",
          summary = "Create a new Blob from a file.",
          description = "Creates a new Blob that contains the contents of a file.",
          key = "lovr.filesystem.newBlob",
          module = "lovr.filesystem",
          related = {
            "lovr.data.newBlob",
            "Blob"
          },
          variants = {
            {
              arguments = {
                {
                  name = "filename",
                  type = "string",
                  description = "The file to load."
                }
              },
              returns = {
                {
                  name = "blob",
                  type = "Blob",
                  description = "The new Blob."
                }
              }
            }
          }
        },
        {
          name = "read",
          tag = "filesystem-files",
          summary = "Read a file.",
          description = "Read the contents of a file.",
          key = "lovr.filesystem.read",
          module = "lovr.filesystem",
          notes = "If the file does not exist or cannot be read, nil is returned.",
          variants = {
            {
              arguments = {
                {
                  name = "filename",
                  type = "string",
                  description = "The name of the file to read."
                },
                {
                  name = "bytes",
                  type = "number",
                  description = "The number of bytes to read (if -1, all bytes will be read).",
                  default = "-1"
                }
              },
              returns = {
                {
                  name = "contents",
                  type = "string",
                  description = "The contents of the file."
                },
                {
                  name = "bytes",
                  type = "number",
                  description = "The number of bytes read from the file."
                }
              }
            }
          }
        },
        {
          name = "remove",
          tag = "filesystem-files",
          summary = "Remove a file or directory.",
          description = "Remove a file or directory in the save directory.",
          key = "lovr.filesystem.remove",
          module = "lovr.filesystem",
          notes = "A directory can only be removed if it is empty.\n\nTo recursively remove a folder, use this function with `lovr.filesystem.getDirectoryItems`.",
          variants = {
            {
              arguments = {
                {
                  name = "path",
                  type = "string",
                  description = "The file or directory to remove."
                }
              },
              returns = {
                {
                  name = "success",
                  type = "boolean",
                  description = "Whether the path was removed."
                }
              }
            }
          }
        },
        {
          name = "setIdentity",
          tag = "filesystem-virtual",
          summary = "Set the name of the save directory.",
          description = "Set the name of the save directory.  This function can only be called once and is called automatically at startup, so this function normally isn't called manually.  However, the identity can be changed by setting the `t.identity` option in `lovr.conf`.",
          key = "lovr.filesystem.setIdentity",
          module = "lovr.filesystem",
          related = {
            "lovr.conf",
            "lovr.filesystem.getSaveDirectory"
          },
          variants = {
            {
              arguments = {
                {
                  name = "identity",
                  type = "string",
                  description = "The name of the save directory."
                }
              },
              returns = {}
            }
          }
        },
        {
          name = "setRequirePath",
          tag = "filesystem-lua",
          summary = "Set the require path.",
          description = "Sets the require path.  The require path is a semicolon-separated list of patterns that LÖVR will use to search for files when they are `require`d.  Any question marks in the pattern will be replaced with the module that is being required.  It is similar to Lua\\'s `package.path` variable, except the patterns will be checked using `lovr.filesystem` APIs. This allows `require` to work even when the project is packaged into a zip archive, or when the project is launched from a different directory.",
          key = "lovr.filesystem.setRequirePath",
          module = "lovr.filesystem",
          notes = "The default reqiure path is '?.lua;?/init.lua'.",
          variants = {
            {
              arguments = {
                {
                  name = "path",
                  type = "string",
                  description = "An optional semicolon separated list of search patterns.",
                  default = "nil"
                }
              },
              returns = {}
            }
          }
        },
        {
          name = "unmount",
          tag = "filesystem-virtual",
          summary = "Unmount a mounted archive.",
          description = "Unmounts a directory or archive previously mounted with `lovr.filesystem.mount`.",
          key = "lovr.filesystem.unmount",
          module = "lovr.filesystem",
          notes = "This function is not thread safe.  Mounting or unmounting an archive while other threads call lovr.filesystem functions is not supported.",
          related = {
            "lovr.filesystem.mount"
          },
          variants = {
            {
              arguments = {
                {
                  name = "path",
                  type = "string",
                  description = "The path to unmount."
                }
              },
              returns = {
                {
                  name = "success",
                  type = "boolean",
                  description = "Whether the archive was unmounted."
                }
              }
            }
          }
        },
        {
          name = "write",
          tag = "filesystem-files",
          summary = "Write to a file.",
          description = "Write to a file in the save directory.",
          key = "lovr.filesystem.write",
          module = "lovr.filesystem",
          notes = "If the file does not exist, it is created.\n\nIf the file already has data in it, it will be replaced with the new content.\n\nIf the path contains subdirectories, all of the parent directories need to exist first or the write will fail.  Use `lovr.filesystem.createDirectory` to make sure they're created first.",
          related = {
            "lovr.filesystem.append",
            "lovr.filesystem.getSaveDirectory",
            "lovr.filesystem.read"
          },
          variants = {
            {
              arguments = {
                {
                  name = "filename",
                  type = "string",
                  description = "The file to write to."
                },
                {
                  name = "content",
                  type = "string",
                  description = "A string to write to the file."
                }
              },
              returns = {
                {
                  name = "success",
                  type = "boolean",
                  description = "Whether the write was successful."
                }
              }
            },
            {
              arguments = {
                {
                  name = "filename",
                  type = "string",
                  description = "The file to write to."
                },
                {
                  name = "blob",
                  type = "Blob",
                  description = "A Blob containing data to write to the file."
                }
              },
              returns = {
                {
                  name = "success",
                  type = "boolean",
                  description = "Whether the write was successful."
                }
              }
            }
          }
        }
      },
      objects = {},
      sections = {
        {
          name = "Files",
          tag = "filesystem-files",
          description = "Operations for reading/writing files and querying their metadata."
        },
        {
          name = "Virtual Filesystem",
          tag = "filesystem-virtual"
        },
        {
          name = "Paths",
          tag = "filesystem-paths",
          description = "Useful system paths."
        },
        {
          name = "Lua",
          tag = "filesystem-lua"
        }
      }
    },
    {
      name = "graphics",
      tag = "modules",
      summary = "Renders graphics using the GPU.",
      description = "The graphics module renders graphics and performs computation using the GPU.\n\nMost of the graphics functions are on the `Pass` object.",
      key = "lovr.graphics",
      enums = {
        {
          name = "BlendAlphaMode",
          summary = "Whether premultiplied alpha is enabled.",
          description = "Controls whether premultiplied alpha is enabled.",
          key = "BlendAlphaMode",
          module = "lovr.graphics",
          notes = "The premultiplied mode should be used if pixels being drawn have already been blended, or \"pre-multiplied\", by the alpha channel.  This happens when rendering to a texture that contains pixels with transparent alpha values, since the stored color values have already been faded by alpha and don't need to be faded a second time with the alphamultiply blend mode.",
          related = {
            "BlendMode",
            "Pass:setBlendMode"
          },
          values = {
            {
              name = "alphamultiply",
              description = "Color channel values are multiplied by the alpha channel during blending."
            },
            {
              name = "premultiplied",
              description = "Color channel values are not multiplied by the alpha.  Instead, it's assumed that the colors have already been multiplied by the alpha.  This should be used if the pixels being drawn have already been blended, or \"pre-multiplied\"."
            }
          }
        },
        {
          name = "BlendMode",
          summary = "Blend modes.",
          description = "Different ways pixels can blend with the pixels behind them.",
          key = "BlendMode",
          module = "lovr.graphics",
          related = {
            "BlendAlphaMode",
            "Pass:setBlendMode"
          },
          values = {
            {
              name = "alpha",
              description = "Colors will be mixed based on alpha."
            },
            {
              name = "add",
              description = "Colors will be added to the existing color, alpha will not be changed."
            },
            {
              name = "subtract",
              description = "Colors will be subtracted from the existing color, alpha will not be changed."
            },
            {
              name = "multiply",
              description = "All color channels will be multiplied together, producing a darkening effect."
            },
            {
              name = "lighten",
              description = "The maximum value of each color channel will be used."
            },
            {
              name = "darken",
              description = "The minimum value of each color channel will be used."
            },
            {
              name = "screen",
              description = "The opposite of multiply: the pixel colors are inverted, multiplied, and inverted again, producing a lightening effect."
            }
          }
        },
        {
          name = "CompareMode",
          summary = "Different ways of performing comparisons.",
          description = "The method used to compare depth and stencil values when performing the depth and stencil tests. Also used for compare modes in `Sampler`s.",
          key = "CompareMode",
          module = "lovr.graphics",
          notes = "This type can also be specified using mathematical notation, e.g. `=`, `>`, `<=`, etc. `notequal` can be provided as `~=` or `!=`.",
          related = {
            "Pass:setDepthTest",
            "Pass:setStencilTest",
            "Pass:setDepthWrite",
            "Pass:setStencilWrite"
          },
          values = {
            {
              name = "none",
              description = "The test does not take place, and acts as though it always passes."
            },
            {
              name = "equal",
              description = "The test passes if the values are equal."
            },
            {
              name = "notequal",
              description = "The test passes if the values are not equal."
            },
            {
              name = "less",
              description = "The test passes if the value is less than the existing one."
            },
            {
              name = "lequal",
              description = "The test passes if the value is less than or equal to the existing one."
            },
            {
              name = "greater",
              description = "The test passes if the value is greater than the existing one."
            },
            {
              name = "gequal",
              description = "The test passes if the value is greater than or equal to the existing one."
            }
          }
        },
        {
          name = "CullMode",
          summary = "Different ways of doing face culling.",
          description = "The different ways of doing triangle backface culling.",
          key = "CullMode",
          module = "lovr.graphics",
          related = {
            "Winding",
            "Pass:setCullMode",
            "Pass:setWinding"
          },
          values = {
            {
              name = "none",
              description = "Both sides of triangles will be drawn."
            },
            {
              name = "back",
              description = "Skips rendering the back side of triangles."
            },
            {
              name = "front",
              description = "Skips rendering the front side of triangles."
            }
          }
        },
        {
          name = "DataLayout",
          description = "The different ways to pack Buffer fields into memory.\n\nThe default is `packed`, which is suitable for vertex buffers and index buffers.  It doesn't add any padding between elements, and so it doesn't waste any space.  However, this layout won't necessarily work for uniform buffers and storage buffers.\n\nThe `std140` layout corresponds to the std140 layout used for uniform buffers in GLSL.  It adds the most padding between fields, and requires the stride to be a multiple of 16.  Example:\n\n    layout(std140) uniform ObjectScales { float scales[64]; };\n\nThe `std430` layout corresponds to the std430 layout used for storage buffers in GLSL.  It adds some padding between certain types, and may round up the stride.  Example:\n\n    layout(std430) buffer TileSizes { vec2 sizes[]; }",
          key = "DataLayout",
          module = "lovr.graphics",
          related = {
            "lovr.graphics.newBuffer",
            "Buffer:getFormat",
            "Buffer:getStride",
            "DataType"
          },
          values = {
            {
              name = "packed",
              description = "The packed layout, without any padding."
            },
            {
              name = "std140",
              description = "The std140 layout."
            },
            {
              name = "std430",
              description = "The std430 layout."
            }
          }
        },
        {
          name = "DataType",
          description = "Different types for `Buffer` fields.  These are scalar, vector, or matrix types, usually packed into small amounts of space to reduce the amount of memory they occupy.\n\nThe names are encoded as follows:\n\n- The data type:\n  - `i` for signed integer\n  - `u` for unsigned integer\n  - `sn` for signed normalized (-1 to 1)\n  - `un` for unsigned normalized (0 to 1)\n  - `f` for floating point\n- The bit depth of each component\n- The letter `x` followed by the component count (for vectors)",
          key = "DataType",
          module = "lovr.graphics",
          notes = "In addition to these values, the following aliases can be used:\n\n<table>\n  <thead>\n    <tr>\n      <td>Alias</td>\n      <td>Maps to</td>\n    </tr>\n  </thead>\n  <tbody>\n    <tr>\n      <td><code>vec2</code></td>\n      <td><code>f32x2</code></td>\n    </tr>\n    <tr>\n      <td><code>vec3</code></td>\n      <td><code>f32x3</code></td>\n    </tr>\n    <tr>\n      <td><code>vec4</code></td>\n      <td><code>f32x4</code></td>\n    </tr>\n    <tr>\n      <td><code>int</code></td>\n      <td><code>i32</code></td>\n    </tr>\n    <tr>\n      <td><code>uint</code></td>\n      <td><code>u32</code></td>\n    </tr>\n    <tr>\n      <td><code>float</code></td>\n      <td><code>f32</code></td>\n    </tr>\n    <tr>\n      <td><code>color</code></td>\n      <td><code>un8x4</code></td>\n    </tr>\n  </tbody> </table>\n\nAdditionally, the following convenience rules apply:\n\n- Field types can end in an `s`, which will be stripped off.\n- Field types can end in `x1`, which will be stripped off.\n\nSo you can write, e.g. `lovr.graphics.newBuffer(4, 'floats')`, which is cute!",
          related = {
            "lovr.graphics.newBuffer",
            "Buffer:getFormat"
          },
          values = {
            {
              name = "i8x4",
              description = "Four 8-bit signed integers."
            },
            {
              name = "u8x4",
              description = "Four 8-bit unsigned integers."
            },
            {
              name = "sn8x4",
              description = "Four 8-bit signed normalized values."
            },
            {
              name = "un8x4",
              description = "Four 8-bit unsigned normalized values (aka `color`)."
            },
            {
              name = "un10x3",
              description = "Three 10-bit unsigned normalized values, and 2 padding bits (aka `normal`)."
            },
            {
              name = "i16",
              description = "One 16-bit signed integer."
            },
            {
              name = "i16x2",
              description = "Two 16-bit signed integers."
            },
            {
              name = "i16x4",
              description = "Four 16-bit signed integers."
            },
            {
              name = "u16",
              description = "One 16-bit unsigned integer."
            },
            {
              name = "u16x2",
              description = "Two 16-bit unsigned integers."
            },
            {
              name = "u16x4",
              description = "Four 16-bit unsigned integers."
            },
            {
              name = "sn16x2",
              description = "Two 16-bit signed normalized values."
            },
            {
              name = "sn16x4",
              description = "Four 16-bit signed normalized values."
            },
            {
              name = "un16x2",
              description = "Two 16-bit unsigned normalized values."
            },
            {
              name = "un16x4",
              description = "Four 16-bit unsigned normalized values."
            },
            {
              name = "i32",
              description = "One 32-bit signed integer (aka `int`)."
            },
            {
              name = "i32x2",
              description = "Two 32-bit signed integers."
            },
            {
              name = "i32x2",
              description = "Two 32-bit signed integers."
            },
            {
              name = "i32x3",
              description = "Three 32-bit signed integers."
            },
            {
              name = "i32x4",
              description = "Four 32-bit signed integers."
            },
            {
              name = "u32",
              description = "One 32-bit unsigned integer (aka `uint`)."
            },
            {
              name = "u32x2",
              description = "Two 32-bit unsigned integers."
            },
            {
              name = "u32x3",
              description = "Three 32-bit unsigned integers."
            },
            {
              name = "u32x4",
              description = "Four 32-bit unsigned integers."
            },
            {
              name = "f16x2",
              description = "Two 16-bit floating point numbers."
            },
            {
              name = "f16x4",
              description = "Four 16-bit floating point numbers."
            },
            {
              name = "f32",
              description = "One 32-bit floating point number (aka `float`)."
            },
            {
              name = "f32x2",
              description = "Two 32-bit floating point numbers (aka `vec2`)."
            },
            {
              name = "f32x3",
              description = "Three 32-bit floating point numbers (aka `vec3`)."
            },
            {
              name = "f32x4",
              description = "Four 32-bit floating point numbers (aka `vec4`)."
            },
            {
              name = "mat2",
              description = "A 2x2 matrix containing four 32-bit floats."
            },
            {
              name = "mat3",
              description = "A 3x3 matrix containing nine 32-bit floats."
            },
            {
              name = "mat4",
              description = "A 4x4 matrix containing sixteen 32-bit floats."
            },
            {
              name = "index16",
              description = "Like u16, but 1-indexed."
            },
            {
              name = "index32",
              description = "Like u32, but 1-indexed."
            }
          }
        },
        {
          name = "DefaultShader",
          summary = "Built-in shaders.",
          description = "The set of shaders built in to LÖVR.  These can be passed to `Pass:setShader` or `lovr.graphics.newShader` instead of writing GLSL code.  The shaders can be further customized by using the `flags` option to change their behavior.  If the active shader is set to `nil`, LÖVR picks one of these shaders to use.",
          key = "DefaultShader",
          module = "lovr.graphics",
          values = {
            {
              name = "unlit",
              description = "Basic shader without lighting that uses colors and a texture."
            },
            {
              name = "normal",
              description = "Shades triangles based on their normal, resulting in a cool rainbow effect."
            },
            {
              name = "font",
              description = "Renders font glyphs."
            },
            {
              name = "cubemap",
              description = "Renders cubemaps."
            },
            {
              name = "equirect",
              description = "Renders spherical textures."
            },
            {
              name = "fill",
              description = "Renders a fullscreen triangle."
            }
          }
        },
        {
          name = "DrawMode",
          summary = "Different ways to draw mesh vertices.",
          description = "Different ways vertices in a mesh can be connected together and filled in with pixels.",
          key = "DrawMode",
          module = "lovr.graphics",
          values = {
            {
              name = "points",
              description = "Each vertex is rendered as a single point.  The size of the point can be controlled using the `pointSize` shader flag, or by writing to the `PointSize` variable in shaders.  The maximum point size is given by the `pointSize` limit from `lovr.graphics.getLimits`."
            },
            {
              name = "lines",
              description = "Pairs of vertices are connected with line segments.  To draw a single line through all of the vertices, an index buffer can be used to repeat vertices.  It is not currently possible to change the width of the lines, although cylinders or capsules can be used as an alternative."
            },
            {
              name = "triangles",
              description = "Every 3 vertices form a triangle, which is filled in with pixels (unless `Pass:setWireframe` is used).  This mode is the most commonly used."
            }
          }
        },
        {
          name = "DrawStyle",
          summary = "Different styles to draw shapes.",
          description = "Whether a shape should be drawn filled or outlined.",
          key = "DrawStyle",
          module = "lovr.graphics",
          related = {
            "Pass:plane",
            "Pass:cube",
            "Pass:box",
            "Pass:circle"
          },
          values = {
            {
              name = "fill",
              description = "The shape will be filled in (the default)."
            },
            {
              name = "line",
              description = "The shape will be outlined."
            }
          }
        },
        {
          name = "FilterMode",
          summary = "Different ways to smooth textures.",
          description = "Controls how `Sampler` objects smooth pixels in textures.",
          key = "FilterMode",
          module = "lovr.graphics",
          related = {
            "lovr.graphics.newSampler",
            "Sampler:getFilter",
            "Texture:setPixels"
          },
          values = {
            {
              name = "nearest",
              description = "A pixelated appearance where the \"nearest neighbor\" pixel is used."
            },
            {
              name = "linear",
              description = "A smooth appearance where neighboring pixels are averaged."
            }
          }
        },
        {
          name = "HorizontalAlign",
          summary = "Different ways to horizontally align text.",
          description = "Different ways to horizontally align text with `Pass:text`.",
          key = "HorizontalAlign",
          module = "lovr.graphics",
          related = {
            "VerticalAlign",
            "Pass:text",
            "Font:getVertices"
          },
          values = {
            {
              name = "left",
              description = "Left-aligned text."
            },
            {
              name = "center",
              description = "Centered text."
            },
            {
              name = "right",
              description = "Right-aligned text."
            }
          }
        },
        {
          name = "MeshStorage",
          summary = "Whether a Mesh stores its data on the CPU or GPU.",
          description = "Whether a Mesh stores its data on the CPU or GPU.",
          key = "MeshStorage",
          module = "lovr.graphics",
          notes = "There are some significant differences and tradeoffs between the two modes:\n\n- CPU meshes store a second copy of the vertices in RAM, which can be expensive for large\n  meshes.\n- When vertices are modified, CPU meshes will update the CPU copy, and only upload to the GPU\n  the next time the Mesh is drawn.  GPU meshes, on the other hand, will immediately upload\n  modified vertices to the GPU.  This means that calling `Mesh:setVertices` multiple times per\n  frame will be faster with a CPU mesh.\n- CPU meshes have an internal vertex buffer that can't be accessed from Lua.\n- CPU meshes can compute their bounding box using `Mesh:computeBoundingBox`.  GPU meshes can't.",
          related = {
            "lovr.graphics.newMesh"
          },
          values = {
            {
              name = "cpu",
              description = "The Mesh will store a copy of the vertices on the CPU."
            },
            {
              name = "gpu",
              description = "The Mesh will not keep a CPU copy, only storing vertices on the GPU."
            }
          }
        },
        {
          name = "OriginType",
          summary = "Different coordinate spaces for nodes in a Model.",
          description = "Different coordinate spaces for nodes in a `Model`.",
          key = "OriginType",
          module = "lovr.graphics",
          related = {
            "Model:getNodePosition",
            "Model:getNodeOrientation",
            "Model:getNodeScale",
            "Model:getNodePose",
            "Model:getNodeTransform",
            "Model:getRootNode",
            "Model:getNodeParent"
          },
          values = {
            {
              name = "root",
              description = "Transforms are relative to the origin (root) of the Model."
            },
            {
              name = "parent",
              description = "Transforms are relative to the parent of the node."
            }
          }
        },
        {
          name = "PassType",
          summary = "Different types of Passes.",
          description = "The three different types of `Pass` objects.  Each Pass has a single type, which determines the type of work it does and which functions can be called on it.",
          key = "PassType",
          module = "lovr.graphics",
          deprecated = true,
          related = {
            "lovr.graphics.getPass",
            "lovr.graphics.submit",
            "Pass:getType"
          },
          values = {
            {
              name = "render",
              description = "A render pass renders graphics to a set of up to four color textures and an optional depth texture.  The textures all need to have the same dimensions and sample counts.  The textures can have multiple layers, and all rendering work will be broadcast to each layer.  Each layer can use a different camera pose, which is used for stereo rendering."
            },
            {
              name = "compute",
              description = "A compute pass runs compute shaders.  Compute passes usually only call `Pass:setShader`, `Pass:send`, and `Pass:compute`.  All of the compute work in a single compute pass is run in parallel, so multiple compute passes should be used if one compute pass needs to happen after a different one."
            },
            {
              name = "transfer",
              description = "A transfer pass copies data to and from GPU memory in `Buffer` and `Texture` objects. Transfer passes use `Pass:copy`, `Pass:clear`, `Pass:blit`, `Pass:mipmap`, and `Pass:read`. Similar to compute passes, all the work in a transfer pass happens in parallel, so multiple passes should be used if the transfers need to be ordered."
            }
          }
        },
        {
          name = "ShaderStage",
          summary = "Different shader stages.",
          description = "Different shader stages.  Graphics shaders have a `vertex` and `fragment` stage, and compute shaders have a single `compute` stage.",
          key = "ShaderStage",
          module = "lovr.graphics",
          values = {
            {
              name = "vertex",
              description = "The vertex stage, which computes transformed vertex positions."
            },
            {
              name = "fragment",
              description = "The fragment stage, which computes pixel colors."
            },
            {
              name = "compute",
              description = "The compute stage, which performs arbitrary computation."
            }
          }
        },
        {
          name = "ShaderType",
          summary = "Different types of Shaders.",
          description = "The two types of shaders that can be created.",
          key = "ShaderType",
          module = "lovr.graphics",
          related = {
            "lovr.graphics.newShader",
            "Shader:getType",
            "ShaderStage"
          },
          values = {
            {
              name = "graphics",
              description = "A graphics shader with a vertex and pixel stage."
            },
            {
              name = "compute",
              description = "A compute shader with a single compute stage."
            }
          }
        },
        {
          name = "StackType",
          summary = "Types of stacks that can be pushed and popped.",
          description = "Different types of stacks that can be pushed and popped with `Pass:push` and `Pass:pop`.",
          key = "StackType",
          module = "lovr.graphics",
          values = {
            {
              name = "transform",
              description = "The transform stack (`Pass:transform`, `Pass:translate`, etc.)."
            },
            {
              name = "state",
              description = "Graphics state, like `Pass:setColor`, `Pass:setFont`, etc.  Notably this does not include camera poses/projections or shader variables changed with `Pass:send`."
            }
          }
        },
        {
          name = "StencilAction",
          summary = "Different ways of updating the stencil buffer.",
          description = "Different ways of updating the stencil buffer with `Pass:setStencilWrite`.",
          key = "StencilAction",
          module = "lovr.graphics",
          related = {
            "Pass:setStencilWrite",
            "Pass:setStencilTest",
            "Pass:setColorWrite"
          },
          values = {
            {
              name = "keep",
              description = "Stencil buffer pixels will not be changed by draws."
            },
            {
              name = "zero",
              description = "Stencil buffer pixels will be set to zero."
            },
            {
              name = "replace",
              description = "Stencil buffer pixels will be replaced with a custom value."
            },
            {
              name = "increment",
              description = "Stencil buffer pixels will be incremented each time they're rendered to."
            },
            {
              name = "decrement",
              description = "Stencil buffer pixels will be decremented each time they're rendered to."
            },
            {
              name = "incrementwrap",
              description = "Similar to increment, but will wrap around to 0 when it exceeds 255."
            },
            {
              name = "decrementwrap",
              description = "Similar to decrement, but will wrap around to 255 when it goes below 0."
            },
            {
              name = "invert",
              description = "The bits in the stencil buffer pixels will be inverted."
            }
          }
        },
        {
          name = "TextureFeature",
          summary = "Different ways Textures can be used.",
          description = "These are the different ways `Texture` objects can be used.  These are passed in to `lovr.graphics.isFormatSupported` to see which texture operations are supported by the GPU for a given format.",
          key = "TextureFeature",
          module = "lovr.graphics",
          values = {
            {
              name = "sample",
              description = "The Texture can be sampled (e.g. a `texture2D` or `sampler2D` variable in shaders)."
            },
            {
              name = "filter",
              description = "The Texture can be used with a `Sampler` using a `FilterMode` of `linear`."
            },
            {
              name = "render",
              description = "The Texture can be rendered to by using it as a target in a render `Pass`."
            },
            {
              name = "blend",
              description = "Blending can be enabled when rendering to this format in a render pass."
            },
            {
              name = "storage",
              description = "The Texture can be sent to an image variable in shaders (e.g. `image2D`)."
            },
            {
              name = "atomic",
              description = "Atomic operations can be used on storage textures with this format."
            },
            {
              name = "blitsrc",
              description = "Source textures in `Pass:blit` can use this format."
            },
            {
              name = "blitdst",
              description = "Destination textures in `Pass:blit` can use this format."
            }
          }
        },
        {
          name = "TextureType",
          description = "Different types of textures.  Textures are multidimensional blocks of GPU memory, and the texture's type determines how many dimensions there are, and adds some semantics about what the 3rd dimension means.",
          key = "TextureType",
          module = "lovr.graphics",
          values = {
            {
              name = "2d",
              description = "A single 2D image, the most common type."
            },
            {
              name = "3d",
              description = "A 3D image, where a sequence of 2D images defines a 3D volume.  Each mipmap level of a 3D texture gets smaller in the x, y, and z axes, unlike cubemap and array textures."
            },
            {
              name = "cube",
              description = "Six square 2D images with the same dimensions that define the faces of a cubemap, used for skyboxes or other \"directional\" images."
            },
            {
              name = "array",
              description = "Array textures are sequences of distinct 2D images that all have the same dimensions."
            }
          }
        },
        {
          name = "TextureUsage",
          description = "These are the different things `Texture`s can be used for.  When creating a Texture, a set of these flags can be provided, restricting what operations are allowed on the texture.  Using a smaller set of flags may improve performance.  If none are provided, the only usage flag applied is `sample`.",
          key = "TextureUsage",
          module = "lovr.graphics",
          values = {
            {
              name = "sample",
              description = "Whether the texture can be sampled from in Shaders (i.e. used in a material, or bound to a variable with a `texture` type, like `texture2D`)."
            },
            {
              name = "render",
              description = "Whether the texture can be rendered to (i.e. by using it as a render target in `lovr.graphics.pass`)."
            },
            {
              name = "storage",
              description = "Whether the texture can be used as a storage texture for compute operations (i.e. bound to a variable with an `image` type, like `image2D`)."
            },
            {
              name = "transfer",
              description = "Whether the texture can be used in a transfer pass."
            }
          }
        },
        {
          name = "VerticalAlign",
          summary = "Different ways to vertically align text.",
          description = "Different ways to vertically align text with `Pass:text`.",
          key = "VerticalAlign",
          module = "lovr.graphics",
          related = {
            "HorizontalAlign",
            "Pass:text",
            "Font:getVertices"
          },
          values = {
            {
              name = "top",
              description = "Top-aligned text."
            },
            {
              name = "middle",
              description = "Centered text."
            },
            {
              name = "bottom",
              description = "Bottom-aligned text."
            }
          }
        },
        {
          name = "Winding",
          summary = "Different triangle windings.",
          description = "Indicates whether the front face of a triangle uses the clockwise or counterclockwise vertex order.",
          key = "Winding",
          module = "lovr.graphics",
          related = {
            "Pass:setWinding",
            "Pass:setCullMode"
          },
          values = {
            {
              name = "clockwise",
              description = "Clockwise winding."
            },
            {
              name = "counterclockwise",
              description = "Counterclockwise winding."
            }
          }
        },
        {
          name = "WrapMode",
          summary = "Different ways to wrap textures.",
          description = "Controls how `Sampler` objects wrap textures.",
          key = "WrapMode",
          module = "lovr.graphics",
          values = {
            {
              name = "clamp",
              description = "Pixels will be clamped to the edge, with pixels outside the 0-1 uv range using colors from the nearest edge."
            },
            {
              name = "repeat",
              description = "Tiles the texture."
            }
          }
        }
      },
      functions = {
        {
          name = "compileShader",
          tag = "graphics-objects",
          summary = "Compile shader code to bytecode.",
          description = "Compiles shader code to SPIR-V bytecode.  The bytecode can be passed to `lovr.graphics.newShader` to create shaders, which will be faster than creating it from GLSL. The bytecode is portable, so bytecode compiled on one platform will work on other platforms. This allows shaders to be precompiled in a build step.",
          key = "lovr.graphics.compileShader",
          module = "lovr.graphics",
          notes = "The input can be GLSL or SPIR-V.  If it's SPIR-V, it will be returned unchanged as a Blob.\n\nIf the shader fails to compile, an error will be thrown with the error message.",
          related = {
            "lovr.graphics.newShader",
            "Shader"
          },
          variants = {
            {
              arguments = {
                {
                  name = "stage",
                  type = "ShaderStage",
                  description = "The type of shader to compile."
                },
                {
                  name = "source",
                  type = "string",
                  description = "A string or filename with shader code."
                }
              },
              returns = {
                {
                  name = "bytecode",
                  type = "Blob",
                  description = "A Blob containing compiled SPIR-V code."
                }
              }
            },
            {
              arguments = {
                {
                  name = "stage",
                  type = "ShaderStage",
                  description = "The type of shader to compile."
                },
                {
                  name = "blob",
                  type = "Blob",
                  description = "A Blob containing shader code."
                }
              },
              returns = {
                {
                  name = "bytecode",
                  type = "Blob",
                  description = "A Blob containing compiled SPIR-V code."
                }
              }
            }
          }
        },
        {
          name = "getBackgroundColor",
          tag = "graphics-global",
          summary = "Get the background color.",
          description = "Returns the global background color.  The textures in a render pass will be cleared to this color at the beginning of the pass if no other clear option is specified.  Additionally, the headset and window will be cleared to this color before rendering.",
          key = "lovr.graphics.getBackgroundColor",
          module = "lovr.graphics",
          notes = "Setting the background color in `lovr.draw` will apply on the following frame, since the default pass is cleared before `lovr.draw` is called.\n\nInternally, this color is applied to the default pass objects when retrieving one of them using `lovr.headset.getPass` or `lovr.graphics.getWindowPass`.  Both are called automatically by the default `lovr.run` implementation.\n\nUsing the background color to clear the display is expected to be more efficient than manually clearing after a render pass begins, especially on mobile GPUs.",
          related = {
            "lovr.graphics.newPass",
            "Pass:setClear",
            "Texture:clear",
            "Pass:fill"
          },
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "r",
                  type = "number",
                  description = "The red component of the background color."
                },
                {
                  name = "g",
                  type = "number",
                  description = "The green component of the background color."
                },
                {
                  name = "b",
                  type = "number",
                  description = "The blue component of the background color."
                },
                {
                  name = "a",
                  type = "number",
                  description = "The alpha component of the background color."
                }
              }
            }
          }
        },
        {
          name = "getBuffer",
          tag = "graphics-objects",
          summary = "Get a temporary Buffer.",
          description = "Returns a temporary Buffer.",
          key = "lovr.graphics.getBuffer",
          module = "lovr.graphics",
          deprecated = true,
          examples = {
            {
              description = "Examples of different buffer formats.",
              code = "-- 2 matrices\nlovr.graphics.getBuffer('mat4', 2)\n\n-- 3 integers, with initial data\nlovr.graphics.getBuffer('int', { 1, 2, 3 })\n\n-- a simple mesh:\nlovr.graphics.getBuffer({\n  { name = 'VertexPosition', type = 'vec3' },\n  { name = 'VertexColor', type = 'color' }\n}, 4)\n\n-- a uniform buffer with vec3's, using the std140 packing\nlovr.graphics.getBuffer({ 'vec3', layout = 'std140' }, data)\n\n-- a uniform buffer with key-value fields\nlovr.graphics.getBuffer({\n  { 'AmbientColor', 'vec3' },\n  { 'LightPosition', 'vec3' },\n  { 'LightType', 'u32' },\n  { 'LightColor', 'vec4' },\n  layout = 'std140'\n})\n\n-- a buffer with nested structure and array types\nlovr.graphics.getBuffer({\n  { 'globals', {\n    { 'ObjectCount', 'int' },\n    { 'WorldSize', 'vec2' },\n    { 'Scale', 'float' }\n  }},\n  { 'materials', {\n    { 'Color', 'vec4' },\n    { 'Glow', 'vec3' },\n    { 'Roughness', 'float' }\n  }, length = 32 },\n  layout = 'std430'\n})\n\n-- a buffer using a variable from a shader:\nlovr.graphics.getBuffer(shader:getBufferFormat('transforms'))"
            }
          },
          notes = "The format table can contain a list of `DataType`s or a list of tables to provide extra information about each field.  Each inner table has the following keys:\n\n- `type` is the `DataType` of the field and is required.\n- `name` is the name of the field, used to match table keys and vertex attribute names.\n- `offset` is the byte offset of the field.  Any fields with a `nil` offset will be placed next\n  to each other sequentially in memory, subject to any padding required by the Buffer's layout.\n  In practice this means that you probably want to provide an `offset` for either all of the\n  fields or none of them.\n- `length` is the array size of the field.\n\nAs a shorthand, the name, type, and optionally the length of a field can be provided as a list instead of using keys.\n\nIf no table or Blob is used to define the initial Buffer contents, its data will be undefined.",
          variants = {
            {
              arguments = {
                {
                  name = "size",
                  type = "number",
                  description = "The size of the Buffer, in bytes."
                }
              },
              returns = {
                {
                  name = "buffer",
                  type = "Buffer",
                  description = "The new Buffer."
                }
              }
            },
            {
              arguments = {
                {
                  name = "blob",
                  type = "Blob",
                  description = "A Blob with the initial contents of the Buffer.  The size of the Blob will be used to determine the length of the Buffer."
                }
              },
              returns = {
                {
                  name = "buffer",
                  type = "Buffer",
                  description = "The new Buffer."
                }
              }
            },
            {
              arguments = {
                {
                  name = "format",
                  type = "table",
                  description = "A list of fields in the Buffer.",
                  table = {
                    {
                      name = "layout",
                      type = "DataLayout",
                      description = "How to lay out the Buffer fields in memory.",
                      default = "packed"
                    },
                    {
                      name = "stride",
                      type = "number",
                      description = "The stride of the Buffer, in bytes.  When `nil`, the stride will be automatically computed based on the fields.  The stride can not be zero or smaller than the max byte occupied by one of the fields.  The layout of the Buffer may adjust the stride."
                    }
                  }
                },
                {
                  name = "length",
                  type = "number",
                  description = "The length of the Buffer.",
                  default = "1"
                }
              },
              returns = {
                {
                  name = "buffer",
                  type = "Buffer",
                  description = "The new Buffer."
                }
              }
            },
            {
              arguments = {
                {
                  name = "format",
                  type = "table",
                  description = "A list of fields in the Buffer.",
                  table = {
                    {
                      name = "layout",
                      type = "DataLayout",
                      description = "How to lay out the Buffer fields in memory.",
                      default = "packed"
                    },
                    {
                      name = "stride",
                      type = "number",
                      description = "The stride of the Buffer, in bytes.  When `nil`, the stride will be automatically computed based on the fields.  The stride can not be zero or smaller than the max byte occupied by one of the fields.  The layout of the Buffer may adjust the stride."
                    }
                  }
                },
                {
                  name = "data",
                  type = "table",
                  description = "The initial data to put into the Buffer.  The length of the Buffer will be determined by the contents of the table.  The contents can be a mix of tables, numbers, and vectors, but the length calculation requires each field to consistently use one type of data."
                }
              },
              returns = {
                {
                  name = "buffer",
                  type = "Buffer",
                  description = "The new Buffer."
                }
              }
            },
            {
              arguments = {
                {
                  name = "format",
                  type = "table",
                  description = "A list of fields in the Buffer.",
                  table = {
                    {
                      name = "layout",
                      type = "DataLayout",
                      description = "How to lay out the Buffer fields in memory.",
                      default = "packed"
                    },
                    {
                      name = "stride",
                      type = "number",
                      description = "The stride of the Buffer, in bytes.  When `nil`, the stride will be automatically computed based on the fields.  The stride can not be zero or smaller than the max byte occupied by one of the fields.  The layout of the Buffer may adjust the stride."
                    }
                  }
                },
                {
                  name = "blob",
                  type = "Blob",
                  description = "A Blob with the initial contents of the Buffer.  The size of the Blob will be used to determine the length of the Buffer."
                }
              },
              returns = {
                {
                  name = "buffer",
                  type = "Buffer",
                  description = "The new Buffer."
                }
              }
            },
            {
              arguments = {
                {
                  name = "type",
                  type = "DataType",
                  description = "The type of each item in the Buffer."
                },
                {
                  name = "length",
                  type = "number",
                  description = "The length of the Buffer.",
                  default = "1"
                }
              },
              returns = {
                {
                  name = "buffer",
                  type = "Buffer",
                  description = "The new Buffer."
                }
              }
            },
            {
              arguments = {
                {
                  name = "type",
                  type = "DataType",
                  description = "The type of each item in the Buffer."
                },
                {
                  name = "data",
                  type = "table",
                  description = "The initial data to put into the Buffer.  The length of the Buffer will be determined by the contents of the table.  The contents can be a mix of tables, numbers, and vectors, but the length calculation requires each field to consistently use one type of data."
                }
              },
              returns = {
                {
                  name = "buffer",
                  type = "Buffer",
                  description = "The new Buffer."
                }
              }
            },
            {
              arguments = {
                {
                  name = "type",
                  type = "DataType",
                  description = "The type of each item in the Buffer."
                },
                {
                  name = "blob",
                  type = "Blob",
                  description = "A Blob with the initial contents of the Buffer.  The size of the Blob will be used to determine the length of the Buffer."
                }
              },
              returns = {
                {
                  name = "buffer",
                  type = "Buffer",
                  description = "The new Buffer."
                }
              }
            },
            {
              arguments = {
                {
                  name = "length",
                  type = "number",
                  description = "The length of the Buffer.",
                  default = "1"
                },
                {
                  name = "type",
                  type = "DataType",
                  description = "The type of each item in the Buffer."
                }
              },
              returns = {
                {
                  name = "buffer",
                  type = "Buffer",
                  description = "The new Buffer."
                }
              },
              deprecated = true
            },
            {
              arguments = {
                {
                  name = "data",
                  type = "table",
                  description = "The initial data to put into the Buffer.  The length of the Buffer will be determined by the contents of the table.  The contents can be a mix of tables, numbers, and vectors, but the length calculation requires each field to consistently use one type of data."
                },
                {
                  name = "type",
                  type = "DataType",
                  description = "The type of each item in the Buffer."
                }
              },
              returns = {
                {
                  name = "buffer",
                  type = "Buffer",
                  description = "The new Buffer."
                }
              },
              deprecated = true
            },
            {
              arguments = {
                {
                  name = "length",
                  type = "number",
                  description = "The length of the Buffer.",
                  default = "1"
                },
                {
                  name = "format",
                  type = "table",
                  description = "A list of fields in the Buffer.",
                  table = {
                    {
                      name = "layout",
                      type = "DataLayout",
                      description = "How to lay out the Buffer fields in memory.",
                      default = "packed"
                    },
                    {
                      name = "stride",
                      type = "number",
                      description = "The stride of the Buffer, in bytes.  When `nil`, the stride will be automatically computed based on the fields.  The stride can not be zero or smaller than the max byte occupied by one of the fields.  The layout of the Buffer may adjust the stride."
                    }
                  }
                }
              },
              returns = {
                {
                  name = "buffer",
                  type = "Buffer",
                  description = "The new Buffer."
                }
              },
              deprecated = true
            },
            {
              arguments = {
                {
                  name = "data",
                  type = "table",
                  description = "The initial data to put into the Buffer.  The length of the Buffer will be determined by the contents of the table.  The contents can be a mix of tables, numbers, and vectors, but the length calculation requires each field to consistently use one type of data."
                },
                {
                  name = "format",
                  type = "table",
                  description = "A list of fields in the Buffer.",
                  table = {
                    {
                      name = "layout",
                      type = "DataLayout",
                      description = "How to lay out the Buffer fields in memory.",
                      default = "packed"
                    },
                    {
                      name = "stride",
                      type = "number",
                      description = "The stride of the Buffer, in bytes.  When `nil`, the stride will be automatically computed based on the fields.  The stride can not be zero or smaller than the max byte occupied by one of the fields.  The layout of the Buffer may adjust the stride."
                    }
                  }
                }
              },
              returns = {
                {
                  name = "buffer",
                  type = "Buffer",
                  description = "The new Buffer."
                }
              },
              deprecated = true
            },
            {
              arguments = {
                {
                  name = "blob",
                  type = "Blob",
                  description = "A Blob with the initial contents of the Buffer.  The size of the Blob will be used to determine the length of the Buffer."
                },
                {
                  name = "type",
                  type = "DataType",
                  description = "The type of each item in the Buffer."
                }
              },
              returns = {
                {
                  name = "buffer",
                  type = "Buffer",
                  description = "The new Buffer."
                }
              },
              deprecated = true
            },
            {
              arguments = {
                {
                  name = "blob",
                  type = "Blob",
                  description = "A Blob with the initial contents of the Buffer.  The size of the Blob will be used to determine the length of the Buffer."
                },
                {
                  name = "format",
                  type = "table",
                  description = "A list of fields in the Buffer.",
                  table = {
                    {
                      name = "layout",
                      type = "DataLayout",
                      description = "How to lay out the Buffer fields in memory.",
                      default = "packed"
                    },
                    {
                      name = "stride",
                      type = "number",
                      description = "The stride of the Buffer, in bytes.  When `nil`, the stride will be automatically computed based on the fields.  The stride can not be zero or smaller than the max byte occupied by one of the fields.  The layout of the Buffer may adjust the stride."
                    }
                  }
                }
              },
              returns = {
                {
                  name = "buffer",
                  type = "Buffer",
                  description = "The new Buffer."
                }
              },
              deprecated = true
            }
          }
        },
        {
          name = "getDefaultFont",
          tag = "graphics-objects",
          summary = "Get the default Font.",
          description = "Returns the default Font.  The default font is Varela Round, created at 32px with a spread value of `4.0`.  It's used by `Pass:text` if no Font is provided.",
          key = "lovr.graphics.getDefaultFont",
          module = "lovr.graphics",
          related = {
            "Pass:text",
            "lovr.graphics.newFont"
          },
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "font",
                  type = "Font",
                  description = "The default Font object."
                }
              }
            }
          }
        },
        {
          name = "getDevice",
          tag = "graphics-misc",
          summary = "Get information about the graphics device and driver.",
          description = "Returns information about the graphics device and driver.",
          key = "lovr.graphics.getDevice",
          module = "lovr.graphics",
          notes = "The device and vendor ID numbers will usually be PCI IDs, which are standardized numbers consisting of 4 hex digits.  Various online databases and system utilities can be used to look up these numbers.  Here are some example vendor IDs for a few popular GPU manufacturers:\n\n<table>\n  <thead>\n    <tr>\n      <td>ID</td>\n      <td>Vendor</td>\n    </tr>\n  </thead>\n  <tbody>\n    <tr>\n      <td><code>0x1002</code></td>\n      <td>Advanced Micro Devices, Inc.</td>\n    </tr>\n    <tr>\n      <td><code>0x8086</code></td>\n      <td>Intel Corporation</td>\n    </tr>\n    <tr>\n      <td><code>0x10de</code></td>\n      <td>NVIDIA Corporation</td>\n    </tr>\n  </tbody> </table>\n\nIt is not currently possible to get the version of the driver, although this could be added.\n\nRegarding multiple GPUs: If OpenXR is enabled, the OpenXR runtime has control over which GPU is used, which ensures best compatibility with the VR headset.  Otherwise, the \"first\" GPU returned by the renderer will be used.  There is currently no other way to pick a GPU to use.",
          related = {
            "lovr.graphics.getFeatures",
            "lovr.graphics.getLimits"
          },
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "device",
                  type = "table",
                  table = {
                    {
                      name = "id",
                      type = "number",
                      description = "The vendor-unique number for this GPU."
                    },
                    {
                      name = "vendor",
                      type = "number",
                      description = "The identifier of the GPU vendor."
                    },
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the GPU."
                    },
                    {
                      name = "renderer",
                      type = "string",
                      description = "The renderer in use, currently either \"Vulkan\" or \"WebGPU\"."
                    },
                    {
                      name = "subgroupSize",
                      type = "number",
                      description = "The number of threads that run in a single GPU compute unit.  This is usually 32 or 64, and is sometimes called the \"wave\" or \"warp\" size.  This can be used to optimize compute shaders for the current hardware."
                    },
                    {
                      name = "discrete",
                      type = "boolean",
                      description = "Whether the GPU is a discrete graphics card."
                    }
                  }
                }
              }
            }
          }
        },
        {
          name = "getFeatures",
          tag = "graphics-misc",
          summary = "Get the supported GPU features.",
          description = "Returns a table indicating which features are supported by the GPU.",
          key = "lovr.graphics.getFeatures",
          module = "lovr.graphics",
          related = {
            "lovr.graphics.isFormatSupported",
            "lovr.graphics.getDevice",
            "lovr.graphics.getLimits"
          },
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "features",
                  type = "table",
                  description = "",
                  table = {
                    {
                      name = "textureBC",
                      type = "boolean",
                      description = "Whether `TextureFormat`s starting with `bc` are supported. This will almost always be `true` on desktop GPUs and will almost always be `false` on mobile GPUs."
                    },
                    {
                      name = "textureASTC",
                      type = "boolean",
                      description = "Whether `TextureFormat`s beginning with `astc` are supported.  This will almost always be `true` on mobile GPUs and will almost always be `false` on desktop GPUs."
                    },
                    {
                      name = "wireframe",
                      type = "boolean",
                      description = "When supported, `Pass:setWireframe` will work, otherwise it will do nothing. This will always be `true` when using Vulkan, and will always be `false` when using WebGPU."
                    },
                    {
                      name = "depthClamp",
                      type = "boolean",
                      description = "When supported, `Pass:setDepthClamp` will work, otherwise it will do nothing."
                    },
                    {
                      name = "depthResolve",
                      type = "boolean",
                      description = "When supported, multisampled render passes can use a non-multisampled depth texture. Otherwise, the depth texture sample count needs to match the render pass sample count."
                    },
                    {
                      name = "indirectDrawFirstInstance",
                      type = "boolean",
                      description = "Whether indirect draws can set the firstInstance property of buffer memory to something other than zero."
                    },
                    {
                      name = "float64",
                      type = "boolean",
                      description = "Whether shader code can use doubles."
                    },
                    {
                      name = "int64",
                      type = "boolean",
                      description = "Whether shader code can use signed and unsigned 64-bit integers."
                    },
                    {
                      name = "int16",
                      type = "boolean",
                      description = "Whether shader code can use signed and unsigned 16-bit integers."
                    }
                  }
                }
              }
            }
          }
        },
        {
          name = "getLimits",
          tag = "graphics-misc",
          summary = "Get the limits of the current GPU.",
          description = "Returns limits of the current GPU.",
          key = "lovr.graphics.getLimits",
          module = "lovr.graphics",
          notes = "The limit ranges are as follows:\n\n<table>\n  <thead>\n    <tr>\n      <td>Limit</td>\n      <td>Minimum</td>\n      <td>Maximum</td>\n    </tr>\n  </thead>\n  <tbody>\n    <tr>\n      <td><code>textureSize2D</code></td>\n      <td>4096</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td><code>textureSize3D</code></td>\n      <td>256</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td><code>textureSizeCube</code></td>\n      <td>4096</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td><code>textureLayers</code></td>\n      <td>256</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td><code>renderSize</code></td>\n      <td>{ 4096, 4096, 6 }</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td><code>uniformBuffersPerStage</code></td>\n      <td>9</td>\n      <td>32*</td>\n    </tr>\n    <tr>\n      <td><code>storageBuffersPerStage</code></td>\n      <td>4</td>\n      <td>32*</td>\n    </tr>\n    <tr>\n      <td><code>sampledTexturesPerStage</code></td>\n      <td>32</td>\n      <td>32*</td>\n    </tr>\n    <tr>\n      <td><code>storageTexturesPerStage</code></td>\n      <td>4</td>\n      <td>32*</td>\n    </tr>\n    <tr>\n      <td><code>samplersPerStage</code></td>\n      <td>15</td>\n      <td>32*</td>\n    </tr>\n    <tr>\n      <td><code>resourcesPerShader</code></td>\n      <td>32</td>\n      <td>32*</td>\n    </tr>\n    <tr>\n      <td><code>uniformBufferRange</code></td>\n      <td>65536</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td><code>storageBufferRange</code></td>\n      <td>134217728 (128MB)</td>\n      <td>1073741824 (1GB)*</td>\n    </tr>\n    <tr>\n      <td><code>uniformBufferAlign</code></td>\n      <td></td>\n      <td>256</td>\n    </tr>\n    <tr>\n      <td><code>storageBufferAlign</code></td>\n      <td></td>\n      <td>64</td>\n    </tr>\n    <tr>\n      <td><code>vertexAttributes</code></td>\n      <td>16</td>\n      <td>16*</td>\n    </tr>\n    <tr>\n      <td><code>vertexBufferStride</code></td>\n      <td>2048</td>\n      <td>65535*</td>\n    </tr>\n    <tr>\n      <td><code>vertexShaderOutputs</code></td>\n      <td>64</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td><code>clipDistances</code></td>\n      <td>0</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td><code>cullDistances</code></td>\n      <td>0</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td><code>clipAndCullDistances</code></td>\n      <td>0</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td><code>computeDispatchCount</code></td>\n      <td>{ 65536, 65536, 65536 }</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td><code>computeWorkgroupSize</code></td>\n      <td>{ 128, 128, 64 }</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td><code>computeWorkgroupVolume</code></td>\n      <td>128</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td><code>computeSharedMemory</code></td>\n      <td>16384 (16KB)</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td><code>pushConstantSize</code></td>\n      <td>128</td>\n      <td>256*</td>\n    </tr>\n    <tr>\n      <td><code>indirectDrawCount</code></td>\n      <td>1</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td><code>instances</code></td>\n      <td>134217727</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td><code>anisotropy</code></td>\n      <td>0.0</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td><code>pointSize</code></td>\n      <td>1.0</td>\n      <td></td>\n    </tr>\n  </tbody> </table>\n\nNote: in the table above, `*` means that LÖVR itself is imposing a cap on the limit, instead of the GPU.",
          related = {
            "lovr.graphics.isFormatSupported",
            "lovr.graphics.getDevice",
            "lovr.graphics.getFeatures"
          },
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "limits",
                  type = "table",
                  description = "",
                  table = {
                    {
                      name = "textureSize2D",
                      type = "number",
                      description = "The maximum width/height of `2d` and `array` textures."
                    },
                    {
                      name = "textureSize3D",
                      type = "number",
                      description = "The maximum width/height/depth of `3d` textures."
                    },
                    {
                      name = "textureSizeCube",
                      type = "number",
                      description = "The maximum width/height of `cube` textures."
                    },
                    {
                      name = "textureLayers",
                      type = "number",
                      description = "The maximum depth of `array` textures."
                    },
                    {
                      name = "renderSize",
                      type = "table",
                      description = "The maximum width, height, and layer count of a texture (or texture view) used as a render target."
                    },
                    {
                      name = "uniformBuffersPerStage",
                      type = "number",
                      description = "The maximum number of uniform buffers in a shader stage."
                    },
                    {
                      name = "storageBuffersPerStage",
                      type = "number",
                      description = "The maximum number of storage buffers in a shader stage."
                    },
                    {
                      name = "sampledTexturesPerStage",
                      type = "number",
                      description = "The maximum number of sampled textures in a shader stage."
                    },
                    {
                      name = "storageTexturesPerStage",
                      type = "number",
                      description = "The maximum number of storage textures in a shader stage."
                    },
                    {
                      name = "samplersPerStage",
                      type = "number",
                      description = "The maximum number of samplers in a shader stage."
                    },
                    {
                      name = "resourcesPerShader",
                      type = "number",
                      description = "The maximum combined number of buffers, textures, and sampler variables in a Shader."
                    },
                    {
                      name = "uniformBufferRange",
                      type = "number",
                      description = "The maximum range of bytes that can be bound to a uniform buffer in a shader."
                    },
                    {
                      name = "storageBufferRange",
                      type = "number",
                      description = "The maximum range of bytes that can be bound to a storage buffer in a shader."
                    },
                    {
                      name = "uniformBufferAlign",
                      type = "number",
                      description = "When binding a range of bytes to a uniform buffer binding in a shader, the byte offset of the range must be a multiple of this limit's value."
                    },
                    {
                      name = "storageBufferAlign",
                      type = "number",
                      description = "When binding a range of bytes to a storage buffer binding in a shader, the byte offset of the range must be a multiple of this limit's value."
                    },
                    {
                      name = "vertexAttributes",
                      type = "number",
                      description = "The maximum number of input attributes in a vertex shader."
                    },
                    {
                      name = "vertexBufferStride",
                      type = "number",
                      description = "The maximum stride of a buffer used as a vertex buffer, in bytes."
                    },
                    {
                      name = "vertexShaderOutputs",
                      type = "number",
                      description = "The maximum number of components output from a vertex shader."
                    },
                    {
                      name = "clipDistances",
                      type = "number",
                      description = "The maximum number of clipping planes declared by a shader."
                    },
                    {
                      name = "cullDistances",
                      type = "number",
                      description = "The maximum number of cull distances declared by a shader."
                    },
                    {
                      name = "clipAndCullDistances",
                      type = "number",
                      description = "The maximum number of clipping planes and cull distances declared by a shader."
                    },
                    {
                      name = "workgroupCount",
                      type = "table",
                      description = "The maximum values of `x`, `y`, and `z` in `Pass:compute`."
                    },
                    {
                      name = "workgroupSize",
                      type = "table",
                      description = "The maximum values of `local_size_x`, `local_size_y`, and `local_size_z` declared in a compute shader."
                    },
                    {
                      name = "totalWorkgroupSize",
                      type = "number",
                      description = "The maximum product of `local_size_x`, `local_size_y`, and `local_size_z` in a compute shader."
                    },
                    {
                      name = "computeSharedMemory",
                      type = "number",
                      description = "The maximum number of bytes used by `shared` variables in compute shaders."
                    },
                    {
                      name = "shaderConstantSize",
                      type = "number",
                      description = "The maximum number of bytes of push constants that can be in a Shader.  Push constants are shared between stages, so the stage with the largest amount of push constant data will count towards this limit."
                    },
                    {
                      name = "indirectDrawCount",
                      type = "number",
                      description = "The maximum number of draws that can be issued by an indirect draw call."
                    },
                    {
                      name = "instances",
                      type = "number",
                      description = "The maximum number of instances that can be rendered in a draw call."
                    },
                    {
                      name = "anisotropy",
                      type = "number",
                      description = "The maximum value of the `anisotropy` parameter in `lovr.graphics.newSampler`."
                    },
                    {
                      name = "pointSize",
                      type = "number",
                      description = "The maximum point size."
                    }
                  }
                }
              }
            }
          }
        },
        {
          name = "getPass",
          tag = "graphics-objects",
          summary = "Get a temporary Pass.",
          description = "Creates and returns a temporary Pass object.",
          key = "lovr.graphics.getPass",
          module = "lovr.graphics",
          deprecated = true,
          notes = "Fun facts about render passes:\n\n- Textures must have been created with the `render` `TextureUsage`.\n- Textures must have the same dimensions, layer counts, and sample counts.\n- When rendering to textures with multiple layers, each draw will be broadcast to all layers.\n  Render passes have multiple \"views\" (cameras), and each layer uses a corresponding view,\n  allowing each layer to be rendered from a different viewpoint.  This enables fast stereo\n  rendering, but can also be used to efficiently render to cubemaps.  The `ViewIndex` variable\n  can also be used in shaders to set up any desired per-view behavior.\n- Mipmaps will automatically be generated for textures at the end of the render pass.\n- It's okay to have zero color textures, but in this case there must be a depth texture.\n- It's possible to render to a specific mipmap level of a Texture, or a subset of its layers, by\n  rendering to texture views, see `Texture:newView`.\n\nFor `compute` passes, all of the commands in the pass act as though they run in parallel.  This means that writing to the same element of a buffer twice, or writing to it and reading from it again is not guaranteed to work properly on all GPUs.  If compute or transfers need to be sequenced, multiple passes should be used.  It is, however, completely fine to read and write to non-overlapping regions of the same buffer or texture.",
          related = {
            "lovr.graphics.submit",
            "lovr.graphics.getWindowPass",
            "lovr.headset.getPass"
          },
          variants = {
            {
              description = "Create a compute pass.",
              arguments = {
                {
                  name = "type",
                  type = "PassType",
                  description = "The type of pass to create."
                }
              },
              returns = {
                {
                  name = "pass",
                  type = "Pass",
                  description = "The new Pass."
                }
              }
            },
            {
              description = "Create a render pass.",
              arguments = {
                {
                  name = "type",
                  type = "PassType",
                  description = "The type of pass to create."
                },
                {
                  name = "texture",
                  type = "Texture",
                  description = "The texture the render pass will render to.  Ignored for non-render passes."
                }
              },
              returns = {
                {
                  name = "pass",
                  type = "Pass",
                  description = "The new Pass."
                }
              }
            },
            {
              description = "Create a render pass, with options.",
              arguments = {
                {
                  name = "type",
                  type = "PassType",
                  description = "The type of pass to create."
                },
                {
                  name = "canvas",
                  type = "table",
                  description = "Render pass configuration.  Up to 4 textures can be provided in table keys 1 through 4. Ignored for non-render passes.",
                  table = {
                    {
                      name = "depth",
                      type = "table",
                      description = "Depth/stencil buffer configuration.  In addition to a table, it can be a `Texture`, a `TextureFormat`, or `false` to disable the depth buffer.",
                      table = {
                        {
                          name = "format",
                          type = "TextureFormat",
                          description = "The format of the depth buffer texture, which must be a depth format (the ones that start with `d`).  LÖVR will create or reuse an internal depth buffer with this format.",
                          default = "'d32f'"
                        },
                        {
                          name = "texture",
                          type = "Texture",
                          description = "A Texture to use as the depth buffer.  Takes precedence over `format`."
                        }
                      }
                    },
                    {
                      name = "samples",
                      type = "number",
                      description = "The number of multisamples to use.  Can be 4 for antialiasing, or 1 to disable antialiasing.",
                      default = "4"
                    }
                  }
                }
              },
              returns = {
                {
                  name = "pass",
                  type = "Pass",
                  description = "The new Pass."
                }
              }
            }
          }
        },
        {
          name = "getWindowPass",
          tag = "graphics-objects",
          summary = "Get the window pass.",
          description = "Returns the window pass.  This is a builtin render `Pass` object that renders to the desktop window texture.  If the desktop window was not open when the graphics module was initialized, this function will return `nil`.",
          key = "lovr.graphics.getWindowPass",
          module = "lovr.graphics",
          notes = "`lovr.conf` may be used to change the settings for the pass:  `t.graphics.antialias` enables antialiasing, and `t.graphics.stencil` enables the stencil buffer.\n\nThis pass clears the window texture to the background color, which can be changed using `lovr.graphics.setBackgroundColor`.",
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "pass",
                  type = "Pass",
                  description = "The window pass, or `nil` if there is no window."
                }
              }
            }
          }
        },
        {
          name = "isFormatSupported",
          tag = "graphics-misc",
          summary = "Check if a Texture format is supported.",
          description = "Returns the type of operations the GPU supports for a texture format, if any.",
          key = "lovr.graphics.isFormatSupported",
          module = "lovr.graphics",
          related = {
            "lovr.graphics.getDevice",
            "lovr.graphics.getFeatures",
            "lovr.graphics.getLimits"
          },
          variants = {
            {
              arguments = {
                {
                  name = "format",
                  type = "TextureFormat",
                  description = "The texture format to query."
                },
                {
                  name = "...features",
                  type = "TextureFeature",
                  description = "Zero or more features to check.  If no features are given, this function will return whether the GPU supports *any* feature for this format.  Otherwise, this function will only return true if *all* of the input features are supported."
                }
              },
              returns = {
                {
                  name = "linear",
                  type = "boolean",
                  description = "Whether the GPU supports these operations for textures with this format, when created with the `linear` flag set to `true`."
                },
                {
                  name = "srgb",
                  type = "boolean",
                  description = "Whether the GPU supports these operations for textures with this format, when created with the `linear` flag set to `false`."
                }
              }
            }
          }
        },
        {
          name = "isTimingEnabled",
          tag = "graphics-global",
          summary = "Check if timing stats are enabled.",
          description = "Returns whether timing stats are enabled.  When enabled, `Pass:getStats` will return `submitTime` and `gpuTime` durations.  Timing is enabled by default when `t.graphics.debug` is set in `lovr.conf`.  Timing has a small amount of overhead, so it should only be enabled when needed.",
          key = "lovr.graphics.isTimingEnabled",
          module = "lovr.graphics",
          related = {
            "Pass:getStats"
          },
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "enabled",
                  type = "boolean",
                  description = "Whether timing is enabled."
                }
              }
            }
          }
        },
        {
          name = "newBuffer",
          tag = "graphics-objects",
          summary = "Create a new Buffer.",
          description = "Creates a Buffer.",
          key = "lovr.graphics.newBuffer",
          module = "lovr.graphics",
          examples = {
            {
              description = "Examples of different buffer formats.",
              code = "-- 2 matrices\nlovr.graphics.newBuffer('mat4', 2)\n\n-- 3 integers, with initial data\nlovr.graphics.newBuffer('int', { 1, 2, 3 })\n\n-- a simple mesh:\nlovr.graphics.newBuffer({\n  { name = 'VertexPosition', type = 'vec3' },\n  { name = 'VertexColor', type = 'color' }\n}, 4)\n\n-- a uniform buffer with vec3's, using the std140 packing\nlovr.graphics.newBuffer({ 'vec3', layout = 'std140' }, data)\n\n-- a uniform buffer with key-value fields\nlovr.graphics.newBuffer({\n  { 'AmbientColor', 'vec3' },\n  { 'LightPosition', 'vec3' },\n  { 'LightType', 'u32' },\n  { 'LightColor', 'vec4' },\n  layout = 'std140'\n})\n\n-- a buffer with nested structure and array types\nlovr.graphics.newBuffer({\n  { 'globals', {\n    { 'ObjectCount', 'int' },\n    { 'WorldSize', 'vec2' },\n    { 'Scale', 'float' }\n  }},\n  { 'materials', {\n    { 'Color', 'vec4' },\n    { 'Glow', 'vec3' },\n    { 'Roughness', 'float' }\n  }, length = 32 },\n  layout = 'std430'\n})\n\n-- a buffer using a variable from a shader:\nlovr.graphics.newBuffer(shader:getBufferFormat('transforms'))"
            }
          },
          notes = "The format table can contain a list of `DataType`s or a list of tables to provide extra information about each field.  Each inner table has the following keys:\n\n- `type` is the `DataType` of the field and is required.\n- `name` is the name of the field, used to match table keys and vertex attribute names.\n- `offset` is the byte offset of the field.  Any fields with a `nil` offset will be placed next\n  to each other sequentially in memory, subject to any padding required by the Buffer's layout.\n  In practice this means that you probably want to provide an `offset` for either all of the\n  fields or none of them.\n- `length` is the array size of the field.\n\nAs a shorthand, the name, type, and optionally the length of a field can be provided as a list instead of using keys.\n\nIf no table or Blob is used to define the initial Buffer contents, its data will be undefined.",
          related = {
            "Shader:getBufferFormat"
          },
          variants = {
            {
              arguments = {
                {
                  name = "size",
                  type = "number",
                  description = "The size of the Buffer, in bytes."
                }
              },
              returns = {
                {
                  name = "buffer",
                  type = "Buffer",
                  description = "The new Buffer."
                }
              }
            },
            {
              arguments = {
                {
                  name = "blob",
                  type = "Blob",
                  description = "A Blob with the initial contents of the Buffer.  The size of the Blob will be used to determine the length of the Buffer."
                }
              },
              returns = {
                {
                  name = "buffer",
                  type = "Buffer",
                  description = "The new Buffer."
                }
              }
            },
            {
              arguments = {
                {
                  name = "format",
                  type = "table",
                  description = "A list of fields in the Buffer.",
                  table = {
                    {
                      name = "layout",
                      type = "DataLayout",
                      description = "How to lay out the Buffer fields in memory.",
                      default = "packed"
                    },
                    {
                      name = "stride",
                      type = "number",
                      description = "The stride of the Buffer, in bytes.  When `nil`, the stride will be automatically computed based on the fields.  The stride can not be zero or smaller than the max byte occupied by one of the fields.  The layout of the Buffer may adjust the stride."
                    }
                  }
                },
                {
                  name = "length",
                  type = "number",
                  description = "The length of the Buffer.",
                  default = "1"
                }
              },
              returns = {
                {
                  name = "buffer",
                  type = "Buffer",
                  description = "The new Buffer."
                }
              }
            },
            {
              arguments = {
                {
                  name = "format",
                  type = "table",
                  description = "A list of fields in the Buffer.",
                  table = {
                    {
                      name = "layout",
                      type = "DataLayout",
                      description = "How to lay out the Buffer fields in memory.",
                      default = "packed"
                    },
                    {
                      name = "stride",
                      type = "number",
                      description = "The stride of the Buffer, in bytes.  When `nil`, the stride will be automatically computed based on the fields.  The stride can not be zero or smaller than the max byte occupied by one of the fields.  The layout of the Buffer may adjust the stride."
                    }
                  }
                },
                {
                  name = "data",
                  type = "table",
                  description = "The initial data to put into the Buffer.  The length of the Buffer will be determined by the contents of the table.  The contents can be a mix of tables, numbers, and vectors, but the length calculation requires each field to consistently use one type of data."
                }
              },
              returns = {
                {
                  name = "buffer",
                  type = "Buffer",
                  description = "The new Buffer."
                }
              }
            },
            {
              arguments = {
                {
                  name = "format",
                  type = "table",
                  description = "A list of fields in the Buffer.",
                  table = {
                    {
                      name = "layout",
                      type = "DataLayout",
                      description = "How to lay out the Buffer fields in memory.",
                      default = "packed"
                    },
                    {
                      name = "stride",
                      type = "number",
                      description = "The stride of the Buffer, in bytes.  When `nil`, the stride will be automatically computed based on the fields.  The stride can not be zero or smaller than the max byte occupied by one of the fields.  The layout of the Buffer may adjust the stride."
                    }
                  }
                },
                {
                  name = "blob",
                  type = "Blob",
                  description = "A Blob with the initial contents of the Buffer.  The size of the Blob will be used to determine the length of the Buffer."
                }
              },
              returns = {
                {
                  name = "buffer",
                  type = "Buffer",
                  description = "The new Buffer."
                }
              }
            },
            {
              arguments = {
                {
                  name = "type",
                  type = "DataType",
                  description = "The type of each item in the Buffer."
                },
                {
                  name = "length",
                  type = "number",
                  description = "The length of the Buffer.",
                  default = "1"
                }
              },
              returns = {
                {
                  name = "buffer",
                  type = "Buffer",
                  description = "The new Buffer."
                }
              }
            },
            {
              arguments = {
                {
                  name = "type",
                  type = "DataType",
                  description = "The type of each item in the Buffer."
                },
                {
                  name = "data",
                  type = "table",
                  description = "The initial data to put into the Buffer.  The length of the Buffer will be determined by the contents of the table.  The contents can be a mix of tables, numbers, and vectors, but the length calculation requires each field to consistently use one type of data."
                }
              },
              returns = {
                {
                  name = "buffer",
                  type = "Buffer",
                  description = "The new Buffer."
                }
              }
            },
            {
              arguments = {
                {
                  name = "type",
                  type = "DataType",
                  description = "The type of each item in the Buffer."
                },
                {
                  name = "blob",
                  type = "Blob",
                  description = "A Blob with the initial contents of the Buffer.  The size of the Blob will be used to determine the length of the Buffer."
                }
              },
              returns = {
                {
                  name = "buffer",
                  type = "Buffer",
                  description = "The new Buffer."
                }
              }
            },
            {
              arguments = {
                {
                  name = "length",
                  type = "number",
                  description = "The length of the Buffer.",
                  default = "1"
                },
                {
                  name = "type",
                  type = "DataType",
                  description = "The type of each item in the Buffer."
                }
              },
              returns = {
                {
                  name = "buffer",
                  type = "Buffer",
                  description = "The new Buffer."
                }
              },
              deprecated = true
            },
            {
              arguments = {
                {
                  name = "data",
                  type = "table",
                  description = "The initial data to put into the Buffer.  The length of the Buffer will be determined by the contents of the table.  The contents can be a mix of tables, numbers, and vectors, but the length calculation requires each field to consistently use one type of data."
                },
                {
                  name = "type",
                  type = "DataType",
                  description = "The type of each item in the Buffer."
                }
              },
              returns = {
                {
                  name = "buffer",
                  type = "Buffer",
                  description = "The new Buffer."
                }
              },
              deprecated = true
            },
            {
              arguments = {
                {
                  name = "length",
                  type = "number",
                  description = "The length of the Buffer.",
                  default = "1"
                },
                {
                  name = "format",
                  type = "table",
                  description = "A list of fields in the Buffer.",
                  table = {
                    {
                      name = "layout",
                      type = "DataLayout",
                      description = "How to lay out the Buffer fields in memory.",
                      default = "packed"
                    },
                    {
                      name = "stride",
                      type = "number",
                      description = "The stride of the Buffer, in bytes.  When `nil`, the stride will be automatically computed based on the fields.  The stride can not be zero or smaller than the max byte occupied by one of the fields.  The layout of the Buffer may adjust the stride."
                    }
                  }
                }
              },
              returns = {
                {
                  name = "buffer",
                  type = "Buffer",
                  description = "The new Buffer."
                }
              },
              deprecated = true
            },
            {
              arguments = {
                {
                  name = "data",
                  type = "table",
                  description = "The initial data to put into the Buffer.  The length of the Buffer will be determined by the contents of the table.  The contents can be a mix of tables, numbers, and vectors, but the length calculation requires each field to consistently use one type of data."
                },
                {
                  name = "format",
                  type = "table",
                  description = "A list of fields in the Buffer.",
                  table = {
                    {
                      name = "layout",
                      type = "DataLayout",
                      description = "How to lay out the Buffer fields in memory.",
                      default = "packed"
                    },
                    {
                      name = "stride",
                      type = "number",
                      description = "The stride of the Buffer, in bytes.  When `nil`, the stride will be automatically computed based on the fields.  The stride can not be zero or smaller than the max byte occupied by one of the fields.  The layout of the Buffer may adjust the stride."
                    }
                  }
                }
              },
              returns = {
                {
                  name = "buffer",
                  type = "Buffer",
                  description = "The new Buffer."
                }
              },
              deprecated = true
            },
            {
              arguments = {
                {
                  name = "blob",
                  type = "Blob",
                  description = "A Blob with the initial contents of the Buffer.  The size of the Blob will be used to determine the length of the Buffer."
                },
                {
                  name = "type",
                  type = "DataType",
                  description = "The type of each item in the Buffer."
                }
              },
              returns = {
                {
                  name = "buffer",
                  type = "Buffer",
                  description = "The new Buffer."
                }
              },
              deprecated = true
            },
            {
              arguments = {
                {
                  name = "blob",
                  type = "Blob",
                  description = "A Blob with the initial contents of the Buffer.  The size of the Blob will be used to determine the length of the Buffer."
                },
                {
                  name = "format",
                  type = "table",
                  description = "A list of fields in the Buffer.",
                  table = {
                    {
                      name = "layout",
                      type = "DataLayout",
                      description = "How to lay out the Buffer fields in memory.",
                      default = "packed"
                    },
                    {
                      name = "stride",
                      type = "number",
                      description = "The stride of the Buffer, in bytes.  When `nil`, the stride will be automatically computed based on the fields.  The stride can not be zero or smaller than the max byte occupied by one of the fields.  The layout of the Buffer may adjust the stride."
                    }
                  }
                }
              },
              returns = {
                {
                  name = "buffer",
                  type = "Buffer",
                  description = "The new Buffer."
                }
              },
              deprecated = true
            }
          }
        },
        {
          name = "newFont",
          tag = "graphics-objects",
          summary = "Create a new Font.",
          description = "Creates a new Font.",
          key = "lovr.graphics.newFont",
          module = "lovr.graphics",
          related = {
            "lovr.graphics.getDefaultFont",
            "lovr.data.newRasterizer",
            "Pass:text"
          },
          variants = {
            {
              description = "Creates a new Font from a TTF file.",
              arguments = {
                {
                  name = "filename",
                  type = "string",
                  description = "A path to a TTF file."
                },
                {
                  name = "size",
                  type = "number",
                  description = "The size of the Font in pixels.  Larger sizes are slower to initialize and use more memory, but have better quality.",
                  default = "32"
                },
                {
                  name = "spread",
                  type = "number",
                  description = "For signed distance field fonts (currently all fonts), the width of the SDF, in pixels.  The greater the distance the font is viewed from, the larger this value needs to be for the font to remain properly antialiased.  Increasing this will have a performance penalty similar to increasing the size of the font.",
                  default = "4"
                }
              },
              returns = {
                {
                  name = "font",
                  type = "Font",
                  description = "The new Font."
                }
              }
            },
            {
              description = "Creates a new Font from TTF data.",
              arguments = {
                {
                  name = "blob",
                  type = "Blob",
                  description = "A Blob containing TTF file data."
                },
                {
                  name = "size",
                  type = "number",
                  description = "The size of the Font in pixels.  Larger sizes are slower to initialize and use more memory, but have better quality.",
                  default = "32"
                },
                {
                  name = "spread",
                  type = "number",
                  description = "For signed distance field fonts (currently all fonts), the width of the SDF, in pixels.  The greater the distance the font is viewed from, the larger this value needs to be for the font to remain properly antialiased.  Increasing this will have a performance penalty similar to increasing the size of the font.",
                  default = "4"
                }
              },
              returns = {
                {
                  name = "font",
                  type = "Font",
                  description = "The new Font."
                }
              }
            },
            {
              description = "Creates a new Font using the default typeface (Varela Round).",
              arguments = {
                {
                  name = "size",
                  type = "number",
                  description = "The size of the Font in pixels.  Larger sizes are slower to initialize and use more memory, but have better quality.",
                  default = "32"
                },
                {
                  name = "spread",
                  type = "number",
                  description = "For signed distance field fonts (currently all fonts), the width of the SDF, in pixels.  The greater the distance the font is viewed from, the larger this value needs to be for the font to remain properly antialiased.  Increasing this will have a performance penalty similar to increasing the size of the font.",
                  default = "4"
                }
              },
              returns = {
                {
                  name = "font",
                  type = "Font",
                  description = "The new Font."
                }
              }
            },
            {
              description = "Creates a new Font from an existing Rasterizer.",
              arguments = {
                {
                  name = "rasterizer",
                  type = "Rasterizer",
                  description = "An existing Rasterizer to use to load glyph images."
                },
                {
                  name = "spread",
                  type = "number",
                  description = "For signed distance field fonts (currently all fonts), the width of the SDF, in pixels.  The greater the distance the font is viewed from, the larger this value needs to be for the font to remain properly antialiased.  Increasing this will have a performance penalty similar to increasing the size of the font.",
                  default = "4"
                }
              },
              returns = {
                {
                  name = "font",
                  type = "Font",
                  description = "The new Font."
                }
              }
            }
          }
        },
        {
          name = "newMaterial",
          tag = "graphics-objects",
          summary = "Create a new Material.",
          description = "Creates a new Material from a table of properties and textures.  All fields are optional.  Once a Material is created, its properties can not be changed.  Instead, a new Material should be created with the updated properties.",
          key = "lovr.graphics.newMaterial",
          module = "lovr.graphics",
          notes = "The non-texture material properties can be accessed in shaders using `Material.<property>`, where the property is the same as the Lua table key.  The textures use capitalized names in shader code, e.g. `ColorTexture`.",
          variants = {
            {
              arguments = {
                {
                  name = "properties",
                  type = "table",
                  description = "Material properties.",
                  table = {
                    {
                      name = "color",
                      type = "Vec4",
                      description = "The base color of the surface.  Can be a `Vec3`, `Vec4`, table of numbers, or hexcode. Can be toggled in shaders using the `materialColor` flag, which defaults to `true`.",
                      default = "{ 1, 1, 1, 1 }"
                    },
                    {
                      name = "glow",
                      type = "Vec4",
                      description = "The glow color of the surface, sometimes called \"emissive\".  The glow is not affected by lighting, so it's a good fit for e.g. headlights on a car or LED lights on a panel.  The alpha of the glow color is used as the glow strength.  Can be a `Vec3`, `Vec4`, table of numbers, or hexcode.  Can be toggled in shaders using the `glow` flag, which defaults to `false`.",
                      default = "{ 0, 0, 0, 0 }"
                    },
                    {
                      name = "uvShift",
                      type = "Vec2",
                      description = "An offset to apply to the UV coordinates used to sample textures.  The offset is not affected by `uvScale`.  This can be used to map UV coordinates to a sub-rectangle of a texture atlas.  Can be a `Vec2`, table of numbers, or a single number which gets assigned to both axes.  Can be toggled in shaders using the `uvTransform` flag, which defaults to `true`.",
                      default = "{ 0, 0 }"
                    },
                    {
                      name = "uvScale",
                      type = "Vec2",
                      description = "A scale factor to apply to the UV coordinates used to sample textures.  The scale is not affected by `uvOffset`.  This can be used to map UV coordinates to a sub-rectangle of a texture atlas, or repeat a texture multiple times across a surface.  Can be a `Vec2`, table of numbers, or a single number which gets assigned to both axes. Can be toggled in shaders using the `uvTransform` flag, which defaults to `true`.",
                      default = "{ 1, 1 }"
                    },
                    {
                      name = "metalness",
                      type = "number",
                      description = "The metalness the surface, used for physically-based rendering.  1.0 means the surface is metallic (conductor), and 0.0 means the surface is non-metallic (dielectric).  Values in between are seldom used and are only used in textures to transition between a metallic and non-metallic surface.  Metals reflect light differently than non-metals. Used by the lighting helper functions `initSurface` and `getLighting`.",
                      default = "0"
                    },
                    {
                      name = "roughness",
                      type = "number",
                      description = "The roughness of the surface, used for physically-based rendering.  1.0 means the surface is rough (blurry reflections), and 0.0 means the surface is smooth (sharp reflections).  Used by the lighting helper functions `initSurface` and `getLighting`.",
                      default = "0"
                    },
                    {
                      name = "clearcoat",
                      type = "number",
                      description = "The clearcoat factor.  Not currently used by LÖVR.",
                      default = "0"
                    },
                    {
                      name = "clearcoatRoughness",
                      type = "number",
                      description = "The roughness of the clearcoat layer.  Not currently used by LÖVR.",
                      default = "0"
                    },
                    {
                      name = "occlusionStrength",
                      type = "number",
                      description = "The strength of the ambient occlusion effect.  Ambient occlusion only affects indirect lighting.  Used by the lighting helper functions `initSurface` and `getIndirectLighting`.  Can be toggled in shaders using the `ambientOcclusion` flag, which defaults to `true`.",
                      default = "1"
                    },
                    {
                      name = "normalScale",
                      type = "number",
                      description = "The strength of the normal map.  Used by the `initSurface` function to bend the surface normal.  Can be toggled in shaders using the `normalMap` flag, which defaults to `false`.",
                      default = "1"
                    },
                    {
                      name = "alphaCutoff",
                      type = "number",
                      description = "The alpha cutoff.  At the end of the fragment shader, if the alpha of the final color is below the alpha cutoff, then the pixel will be \"discarded\" which means that it won't write a depth value.  Often used for transparent textures, especially with the \"alpha to coverage\" state set by `Pass:setAlphaToCoverage`.  Can be toggled in shaders using the `alphaCutoff` flag, which defaults to `false`.",
                      default = "0"
                    },
                    {
                      name = "texture",
                      type = "Texture",
                      description = "The base color texture.  In shaders this gets multiplied with the `color` property to get the base color of the pixel.  Can be toggled in shaders using the `colorTexture` flag, which defaults to `true`."
                    },
                    {
                      name = "glowTexture",
                      type = "Texture",
                      description = "The glow color texture.  In shaders, samples from this texture get multiplied with the `glow` property to get the glow color of the pixel.  Can be toggled in shaders using the `glowTexture` flag, which defaults to `true` (also requires the `glow` flag to be enabled)."
                    },
                    {
                      name = "metalnessTexture",
                      type = "Texture",
                      description = "The metalness texture.  In shaders, samples from the blue channel of this texture get multiplied with the `metalness` property to get the metalness value of the pixel.  Can be toggled in shaders using the `metalnessTexture` flag, which defaults to `true`."
                    },
                    {
                      name = "roughnessTexture",
                      type = "Texture",
                      description = "The roughness texture.  In shaders, samples from the green channel of this texture get multiplied with the `roughness` property to get the roughness value of the pixel.  Can be toggled in shaders using the `roughnessTexture` flag, which defaults to `true`."
                    },
                    {
                      name = "clearcoatTexture",
                      type = "Texture",
                      description = "Not currently used by LÖVR."
                    },
                    {
                      name = "occlusionTexture",
                      type = "Texture",
                      description = "The ambient occlusion texture.  In shaders, samples from the red channel of this texture get multiplied with the `occlusionStrength` property to get the ambient occlusion value of the pixel. Used by the lighting helper functions `initSurface` and `getIndirectLighting`.  Can be toggled in shaders using the `ambientOcclusion` flag, which defaults to `true`."
                    },
                    {
                      name = "normalTexture",
                      type = "Texture",
                      description = "The normal map, used to apply details to a surface without adding mesh geometry.  The `normalScale` property can be used to control how strong the effect is.  Can be toggled in shaders using the `normalMap` flag, which defaults to `false`."
                    }
                  }
                }
              },
              returns = {
                {
                  name = "material",
                  type = "Material",
                  description = "The new material."
                }
              }
            }
          }
        },
        {
          name = "newMesh",
          tag = "graphics-objects",
          summary = "Create a new Mesh.",
          description = "Creates a Mesh.  The capacity of the Mesh must be provided upfront, using either a vertex count or the vertex data itself.  A custom vertex format can be given to specify the set of attributes in each vertex, which get sent to the vertex shader.  If the format isn't given, the default vertex format will be used:\n\n    {\n      { 'VertexPosition', 'vec3' },\n      { 'VertexNormal', 'vec3' },\n      { 'VertexUV', 'vec2' }\n    }",
          key = "lovr.graphics.newMesh",
          module = "lovr.graphics",
          examples = {
            {
              code = "function lovr.load()\n  mesh = lovr.graphics.newMesh({\n    { 'VertexPosition', 'vec3' },\n    { 'VertexColor', 'vec4' }\n  }, {\n    {   0,  .4, 0 , 1, 0, 0, 1 },\n    { -.5, -.4, 0 , 0, 1, 0, 1 },\n    {  .5, -.4, 0 , 0, 0, 1, 1 }\n  })\nend\n\nfunction lovr.draw(pass)\n  pass:draw(mesh, 0, 1.7, -1)\nend"
            }
          },
          notes = "The Mesh will always use the `gpu` storage mode if it's created from a vertex buffer.",
          related = {
            "lovr.graphics.newBuffer",
            "lovr.graphics.newModel"
          },
          variants = {
            {
              arguments = {
                {
                  name = "count",
                  type = "number",
                  description = "The number of vertices in the Mesh."
                },
                {
                  name = "storage",
                  type = "MeshStorage",
                  description = "The storage mode of the Mesh.",
                  default = "'cpu'"
                }
              },
              returns = {
                {
                  name = "mesh",
                  type = "Mesh",
                  description = "The new Mesh."
                }
              }
            },
            {
              arguments = {
                {
                  name = "vertices",
                  type = "table",
                  description = "A table of vertices, formatted according to the vertex format.  The length of the table will be used to set the vertex count of the Mesh."
                },
                {
                  name = "storage",
                  type = "MeshStorage",
                  description = "The storage mode of the Mesh.",
                  default = "'cpu'"
                }
              },
              returns = {
                {
                  name = "mesh",
                  type = "Mesh",
                  description = "The new Mesh."
                }
              }
            },
            {
              arguments = {
                {
                  name = "blob",
                  type = "Blob",
                  description = "A Blob containing vertex data, formatted according to the vertex format.  The size of the Blob will be used to set the vertex count of the Mesh, and must be a multiple of the vertex size."
                },
                {
                  name = "storage",
                  type = "MeshStorage",
                  description = "The storage mode of the Mesh.",
                  default = "'cpu'"
                }
              },
              returns = {
                {
                  name = "mesh",
                  type = "Mesh",
                  description = "The new Mesh."
                }
              }
            },
            {
              arguments = {
                {
                  name = "format",
                  type = "table",
                  description = "A table of attributes describing the format of each vertex.  Each attribute is a table that must have `name` and `type` keys, where the name is a string and the type is a `DataType`. Attributes can also have an `offset` key, which is a byte offset relative to the start of the vertex.  As a shorthand, the name and type can be given as a pair without keys. Additionally, the format can have a `stride` key to set the number of bytes between subsequent vertices."
                },
                {
                  name = "count",
                  type = "number",
                  description = "The number of vertices in the Mesh."
                },
                {
                  name = "storage",
                  type = "MeshStorage",
                  description = "The storage mode of the Mesh.",
                  default = "'cpu'"
                }
              },
              returns = {
                {
                  name = "mesh",
                  type = "Mesh",
                  description = "The new Mesh."
                }
              }
            },
            {
              arguments = {
                {
                  name = "format",
                  type = "table",
                  description = "A table of attributes describing the format of each vertex.  Each attribute is a table that must have `name` and `type` keys, where the name is a string and the type is a `DataType`. Attributes can also have an `offset` key, which is a byte offset relative to the start of the vertex.  As a shorthand, the name and type can be given as a pair without keys. Additionally, the format can have a `stride` key to set the number of bytes between subsequent vertices."
                },
                {
                  name = "vertices",
                  type = "table",
                  description = "A table of vertices, formatted according to the vertex format.  The length of the table will be used to set the vertex count of the Mesh."
                },
                {
                  name = "storage",
                  type = "MeshStorage",
                  description = "The storage mode of the Mesh.",
                  default = "'cpu'"
                }
              },
              returns = {
                {
                  name = "mesh",
                  type = "Mesh",
                  description = "The new Mesh."
                }
              }
            },
            {
              arguments = {
                {
                  name = "format",
                  type = "table",
                  description = "A table of attributes describing the format of each vertex.  Each attribute is a table that must have `name` and `type` keys, where the name is a string and the type is a `DataType`. Attributes can also have an `offset` key, which is a byte offset relative to the start of the vertex.  As a shorthand, the name and type can be given as a pair without keys. Additionally, the format can have a `stride` key to set the number of bytes between subsequent vertices."
                },
                {
                  name = "blob",
                  type = "Blob",
                  description = "A Blob containing vertex data, formatted according to the vertex format.  The size of the Blob will be used to set the vertex count of the Mesh, and must be a multiple of the vertex size."
                },
                {
                  name = "storage",
                  type = "MeshStorage",
                  description = "The storage mode of the Mesh.",
                  default = "'cpu'"
                }
              },
              returns = {
                {
                  name = "mesh",
                  type = "Mesh",
                  description = "The new Mesh."
                }
              }
            },
            {
              arguments = {
                {
                  name = "buffer",
                  type = "Buffer",
                  description = "A Buffer containing vertex data.  Its length will be used as the vertex count, and its format will be used as the vertex format."
                }
              },
              returns = {
                {
                  name = "mesh",
                  type = "Mesh",
                  description = "The new Mesh."
                }
              }
            }
          }
        },
        {
          name = "newModel",
          tag = "graphics-objects",
          summary = "Create a new Model.",
          description = "Loads a 3D model from a file.  Currently, OBJ, glTF, and binary STL files are supported.",
          key = "lovr.graphics.newModel",
          module = "lovr.graphics",
          notes = "Currently, the following features are not supported by the model importer:\n\n- glTF: Only the default scene is loaded.\n- glTF: Currently, each skin in a Model can have up to 256 joints.\n- glTF: Meshes can't appear multiple times in the node hierarchy with different skins, they need\n  to use 1 skin consistently.\n- glTF: `KHR_texture_transform` is supported, but all textures in a material will use the same\n  transform.\n- STL: ASCII STL files are not supported.\n\nDiffuse and emissive textures will be loaded using sRGB encoding, all other textures will be loaded as linear.\n\nLoading a model file will fail if the asset references textures or other files using absolute paths.  Relative paths should be used instead, and will be relative to the model file within the virtual filesystem.",
          related = {
            "lovr.data.newModelData",
            "Pass:draw"
          },
          variants = {
            {
              arguments = {
                {
                  name = "filename",
                  type = "string",
                  description = "The path to model file."
                },
                {
                  name = "options",
                  type = "table",
                  description = "Model options.",
                  table = {
                    {
                      name = "mipmaps",
                      type = "boolean",
                      description = "Whether the textures created for the Model should have mipmaps generated.",
                      default = "true"
                    },
                    {
                      name = "materials",
                      type = "boolean",
                      description = "Whether the textures and materials in the Model should be loaded.  When false, the model will use the material set with `Pass:setMaterial`, although it will apply to all nodes.",
                      default = "true"
                    }
                  }
                }
              },
              returns = {
                {
                  name = "model",
                  type = "Model",
                  description = "The new Model."
                }
              }
            },
            {
              arguments = {
                {
                  name = "blob",
                  type = "Blob",
                  description = "A Blob containing 3D model data."
                },
                {
                  name = "options",
                  type = "table",
                  description = "Model options.",
                  table = {
                    {
                      name = "mipmaps",
                      type = "boolean",
                      description = "Whether the textures created for the Model should have mipmaps generated.",
                      default = "true"
                    },
                    {
                      name = "materials",
                      type = "boolean",
                      description = "Whether the textures and materials in the Model should be loaded.  When false, the model will use the material set with `Pass:setMaterial`, although it will apply to all nodes.",
                      default = "true"
                    }
                  }
                }
              },
              returns = {
                {
                  name = "model",
                  type = "Model",
                  description = "The new Model."
                }
              }
            },
            {
              arguments = {
                {
                  name = "modelData",
                  type = "ModelData",
                  description = "An existing ModelData object to use for the Model."
                },
                {
                  name = "options",
                  type = "table",
                  description = "Model options.",
                  table = {
                    {
                      name = "mipmaps",
                      type = "boolean",
                      description = "Whether the textures created for the Model should have mipmaps generated.",
                      default = "true"
                    },
                    {
                      name = "materials",
                      type = "boolean",
                      description = "Whether the textures and materials in the Model should be loaded.  When false, the model will use the material set with `Pass:setMaterial`, although it will apply to all nodes.",
                      default = "true"
                    }
                  }
                }
              },
              returns = {
                {
                  name = "model",
                  type = "Model",
                  description = "The new Model."
                }
              }
            }
          }
        },
        {
          name = "newPass",
          tag = "graphics-objects",
          summary = "Create a new Pass.",
          description = "Creates and returns a new Pass object.  The canvas (the set of textures the Pass renders to) can be specified when creating the Pass, or later using `Pass:setCanvas`.",
          key = "lovr.graphics.newPass",
          module = "lovr.graphics",
          notes = "Fun facts about render passes:\n\n- Textures must have been created with the `render` `TextureUsage`.\n- Textures must have the same dimensions, layer counts, and sample counts.\n- When rendering to textures with multiple layers, each draw will be broadcast to all layers.\n  Render passes have multiple \"views\" (cameras), and each layer uses a corresponding view,\n  allowing each layer to be rendered from a different viewpoint.  This enables fast stereo\n  rendering, but can also be used to efficiently render to cubemaps.  The `ViewIndex` variable\n  can also be used in shaders to set up any desired per-view behavior.\n- Mipmaps will automatically be generated for textures at the end of the render pass.\n- It's okay to have zero color textures, but in this case there must be a depth texture.\n- It's possible to render to a specific mipmap level of a Texture, or a subset of its layers, by\n  rendering to texture views, see `Texture:newView`.",
          related = {
            "lovr.graphics.submit",
            "lovr.graphics.getWindowPass",
            "lovr.headset.getPass"
          },
          variants = {
            {
              description = "Create a pass that renders to a set of textures.",
              arguments = {
                {
                  name = "...textures",
                  type = "Texture",
                  description = "One or more textures the pass will render to.  This can be changed later using `Pass:setCanvas`."
                }
              },
              returns = {
                {
                  name = "pass",
                  type = "Pass",
                  description = "The new Pass."
                }
              }
            },
            {
              description = "Create a pass, with extra canvas settings.",
              arguments = {
                {
                  name = "canvas",
                  type = "table",
                  description = "Render target configuration.  Up to 4 textures can be provided in table keys 1 through 4, as well as the following keys:",
                  table = {
                    {
                      name = "depth",
                      type = "table",
                      description = "Depth/stencil buffer settings.  In addition to a table, it can be a `Texture`, a `TextureFormat`, or `false` to disable the depth buffer.",
                      table = {
                        {
                          name = "format",
                          type = "TextureFormat",
                          description = "The format of the depth buffer texture, which must be a depth format (the ones that start with `d`).  LÖVR will create or reuse an internal depth buffer with this format.",
                          default = "'d32f'"
                        },
                        {
                          name = "texture",
                          type = "Texture",
                          description = "A Texture to use as the depth buffer.  Takes precedence over `format`."
                        }
                      }
                    },
                    {
                      name = "samples",
                      type = "number",
                      description = "The number of multisamples to use.  Can be 4 for antialiasing, or 1 to disable antialiasing.",
                      default = "4"
                    }
                  }
                }
              },
              returns = {
                {
                  name = "pass",
                  type = "Pass",
                  description = "The new Pass."
                }
              }
            },
            {
              description = "Create an empty Pass without a canvas.",
              arguments = {},
              returns = {
                {
                  name = "pass",
                  type = "Pass",
                  description = "The new Pass."
                }
              }
            }
          }
        },
        {
          name = "newSampler",
          tag = "graphics-objects",
          summary = "Create a new Sampler.",
          description = "Creates a new Sampler.  Samplers are immutable, meaning their parameters can not be changed after the sampler is created.  Instead, a new sampler should be created with the updated properties.",
          key = "lovr.graphics.newSampler",
          module = "lovr.graphics",
          related = {
            "Pass:setSampler"
          },
          variants = {
            {
              arguments = {
                {
                  name = "parameters",
                  type = "table",
                  description = "Parameters for the sampler.",
                  table = {
                    {
                      name = "filter",
                      type = "table",
                      description = "How the sampler smooths texture pixels.  Can be a table of 3 FilterModes, or a single FilterMode to use for all three.",
                      default = "'linear'",
                      table = {
                        {
                          name = "[1]",
                          type = "FilterMode",
                          description = "The filter mode to use when minifying a texture (drawing it at a smaller size than its native pixel resolution)."
                        },
                        {
                          name = "[2]",
                          type = "FilterMode",
                          description = "The filter mode to use when magnifying a texture (drawing it at a larger size than its native pixel resolution)."
                        },
                        {
                          name = "[3]",
                          type = "FilterMode",
                          description = "The filter mode used to smooth between mipmap levels in a texture."
                        }
                      }
                    },
                    {
                      name = "wrap",
                      type = "table",
                      description = "How the sampler behaves when wrapping UVs outside the 0-1 range.  Can be a table of 3 WrapModes, or a single WrapMode to use for all three axes.",
                      default = "'repeat'",
                      table = {
                        {
                          name = "[1]",
                          type = "WrapMode",
                          description = "The horizontal wrap mode."
                        },
                        {
                          name = "[2]",
                          type = "WrapMode",
                          description = "The vertical wrap mode."
                        },
                        {
                          name = "[3]",
                          type = "FilterMode",
                          description = "The \"z\" wrap mode for 3D textures."
                        }
                      }
                    },
                    {
                      name = "compare",
                      type = "CompareMode",
                      description = "The compare mode of the sampler (for shadow samplers).",
                      default = "'none'"
                    },
                    {
                      name = "anisotropy",
                      type = "number",
                      description = "The maximum amount of anisotropic filtering to use.",
                      default = "1"
                    },
                    {
                      name = "mipmaprange",
                      type = "table",
                      description = "A table of 2 mipmap levels the sampler will clamp to."
                    }
                  }
                }
              },
              returns = {
                {
                  name = "sampler",
                  type = "Sampler",
                  description = "The new sampler."
                }
              }
            }
          }
        },
        {
          name = "newShader",
          tag = "graphics-objects",
          summary = "Create a new Shader.",
          description = "Creates a Shader, which is a small program that runs on the GPU.\n\nShader code is usually written in GLSL and compiled to SPIR-V bytecode.  SPIR-V is faster to load but requires a build step.  Either form can be used to create a shader.",
          key = "lovr.graphics.newShader",
          module = "lovr.graphics",
          related = {
            "lovr.graphics.compileShader",
            "ShaderType",
            "ShaderStage"
          },
          variants = {
            {
              description = "Create a graphics shader.  It has a vertex stage that computes vertex positions, and a fragment stage that computes pixel colors.",
              arguments = {
                {
                  name = "vertex",
                  type = "string",
                  description = "A string, path to a file, or Blob containing GLSL or SPIR-V code for the vertex stage.  Can also be a `DefaultShader` to use that shader's vertex code."
                },
                {
                  name = "fragment",
                  type = "string",
                  description = "A string, path to a file, or Blob containing GLSL or SPIR-V code for the fragment stage. Can also be a `DefaultShader` to use that shader's fragment code."
                },
                {
                  name = "options",
                  type = "table",
                  description = "Shader options.",
                  table = {
                    {
                      name = "flags",
                      type = "table",
                      description = "A table of shader flags.  The keys of the table should be flag names or flag ID numbers. The values can be numbers or booleans, depending on the type of the flag as declared in the shader."
                    },
                    {
                      name = "label",
                      type = "string",
                      description = "A label to use for the shader in debugging tools."
                    }
                  }
                }
              },
              returns = {
                {
                  name = "shader",
                  type = "Shader",
                  description = "The new shader."
                }
              }
            },
            {
              description = "Create a compute shader.",
              arguments = {
                {
                  name = "compute",
                  type = "string",
                  description = "A string, path to a file, or Blob containing GLSL or SPIR-V code for the compute stage."
                },
                {
                  name = "options",
                  type = "table",
                  description = "Shader options.",
                  table = {
                    {
                      name = "flags",
                      type = "table",
                      description = "A table of shader flags.  The keys of the table should be flag names or flag ID numbers. The values can be numbers or booleans, depending on the type of the flag as declared in the shader."
                    },
                    {
                      name = "label",
                      type = "string",
                      description = "A label to use for the shader in debugging tools."
                    }
                  }
                }
              },
              returns = {
                {
                  name = "shader",
                  type = "Shader",
                  description = "The new shader."
                }
              }
            },
            {
              description = "Create a copy of one of the default shaders (used to provide different flags).",
              arguments = {
                {
                  name = "default",
                  type = "DefaultShader",
                  description = "The default shader to use."
                },
                {
                  name = "options",
                  type = "table",
                  description = "Shader options.",
                  table = {
                    {
                      name = "flags",
                      type = "table",
                      description = "A table of shader flags.  The keys of the table should be flag names or flag ID numbers. The values can be numbers or booleans, depending on the type of the flag as declared in the shader."
                    },
                    {
                      name = "label",
                      type = "string",
                      description = "A label to use for the shader in debugging tools."
                    }
                  }
                }
              },
              returns = {
                {
                  name = "shader",
                  type = "Shader",
                  description = "The new shader."
                }
              }
            }
          }
        },
        {
          name = "newTexture",
          tag = "graphics-objects",
          summary = "Create a new Texture.",
          description = "Creates a new Texture.  Image filenames or `Image` objects can be used to provide the initial pixel data and the dimensions, format, and type.  Alternatively, dimensions can be provided, which will create an empty texture.",
          key = "lovr.graphics.newTexture",
          module = "lovr.graphics",
          notes = "If no `type` is provided in the options table, LÖVR will guess the `TextureType` of the Texture based on the number of layers:\n\n- If there's only 1 layer, the type will be `2d`.\n- If there are 6 images provided, the type will be `cube`.\n- Otherwise, the type will be `array`.\n\nNote that an Image can contain multiple layers and mipmaps.  When a single Image is provided, its layer count will be used as the Texture's layer count.\n\nIf multiple Images are used to initialize the Texture, they must all have a single layer, and their dimensions, format, and mipmap counts must match.\n\nWhen providing cubemap images in a table, they can be in one of the following forms:\n\n    { 'px.png', 'nx.png', 'py.png', 'ny.png', 'pz.png', 'nz.png' }\n    { right = 'px.png', left = 'nx.png', top = 'py.png', bottom = 'ny.png', back = 'pz.png', front = 'nz.png' }\n    { px = 'px.png', nx = 'nx.png', py = 'py.png', ny = 'ny.png', pz = 'pz.png', nz = 'nz.png' }\n\n(Where 'p' stands for positive and 'n' stands for negative).\n\nIf no `usage` is provided in the options table, LÖVR will guess the `TextureUsage` of the Texture.  The `sample` usage is always included, but if the texture was created without any images then the texture will have the `render` usage as well.\n\nThe supported image formats are png, jpg, hdr, dds, ktx1, ktx2, and astc.\n\nIf image data is provided, mipmaps will be generated for any missing mipmap levels.",
          related = {
            "Texture:newView"
          },
          variants = {
            {
              arguments = {
                {
                  name = "filename",
                  type = "string",
                  description = "The filename of an image to load."
                },
                {
                  name = "options",
                  type = "table",
                  description = "Texture options.",
                  table = {
                    {
                      name = "type",
                      type = "TextureType",
                      description = "The type of the texture."
                    },
                    {
                      name = "format",
                      type = "TextureFormat",
                      description = "The format of the texture (ignored when images are provided).",
                      default = "'rgba8'"
                    },
                    {
                      name = "linear",
                      type = "boolean",
                      description = "Whether the texture is in linear color space instead of sRGB.  Linear textures should be used for non-color data, like normal maps.",
                      default = "false"
                    },
                    {
                      name = "samples",
                      type = "number",
                      description = "The number of samples in the texture, used for multisample antialiasing.  Currently must be 1 or 4.  Ignored when images are provided.",
                      default = "1"
                    },
                    {
                      name = "mipmaps",
                      type = "*",
                      description = "The number of mipmap levels in the texture, or a boolean.  If true, a full mipmap chain will be created.  If false, the texture will only have a single mipmap.",
                      default = "true"
                    },
                    {
                      name = "usage",
                      type = "table",
                      description = "A list of `TextureUsage` indicating how the texture will be used."
                    },
                    {
                      name = "label",
                      type = "string",
                      description = "A label for the Texture that will show up in debugging tools."
                    }
                  }
                }
              },
              returns = {
                {
                  name = "texture",
                  type = "Texture",
                  description = "The new Texture."
                }
              }
            },
            {
              arguments = {
                {
                  name = "width",
                  type = "number",
                  description = "The width of the Texture, in pixels."
                },
                {
                  name = "height",
                  type = "number",
                  description = "The height of the Texture, in pixels."
                },
                {
                  name = "options",
                  type = "table",
                  description = "Texture options.",
                  table = {
                    {
                      name = "type",
                      type = "TextureType",
                      description = "The type of the texture."
                    },
                    {
                      name = "format",
                      type = "TextureFormat",
                      description = "The format of the texture (ignored when images are provided).",
                      default = "'rgba8'"
                    },
                    {
                      name = "linear",
                      type = "boolean",
                      description = "Whether the texture is in linear color space instead of sRGB.  Linear textures should be used for non-color data, like normal maps.",
                      default = "false"
                    },
                    {
                      name = "samples",
                      type = "number",
                      description = "The number of samples in the texture, used for multisample antialiasing.  Currently must be 1 or 4.  Ignored when images are provided.",
                      default = "1"
                    },
                    {
                      name = "mipmaps",
                      type = "*",
                      description = "The number of mipmap levels in the texture, or a boolean.  If true, a full mipmap chain will be created.  If false, the texture will only have a single mipmap.",
                      default = "true"
                    },
                    {
                      name = "usage",
                      type = "table",
                      description = "A list of `TextureUsage` indicating how the texture will be used."
                    },
                    {
                      name = "label",
                      type = "string",
                      description = "A label for the Texture that will show up in debugging tools."
                    }
                  }
                }
              },
              returns = {
                {
                  name = "texture",
                  type = "Texture",
                  description = "The new Texture."
                }
              }
            },
            {
              arguments = {
                {
                  name = "width",
                  type = "number",
                  description = "The width of the Texture, in pixels."
                },
                {
                  name = "height",
                  type = "number",
                  description = "The height of the Texture, in pixels."
                },
                {
                  name = "layers",
                  type = "number",
                  description = "The number of layers in the Texture."
                },
                {
                  name = "options",
                  type = "table",
                  description = "Texture options.",
                  table = {
                    {
                      name = "type",
                      type = "TextureType",
                      description = "The type of the texture."
                    },
                    {
                      name = "format",
                      type = "TextureFormat",
                      description = "The format of the texture (ignored when images are provided).",
                      default = "'rgba8'"
                    },
                    {
                      name = "linear",
                      type = "boolean",
                      description = "Whether the texture is in linear color space instead of sRGB.  Linear textures should be used for non-color data, like normal maps.",
                      default = "false"
                    },
                    {
                      name = "samples",
                      type = "number",
                      description = "The number of samples in the texture, used for multisample antialiasing.  Currently must be 1 or 4.  Ignored when images are provided.",
                      default = "1"
                    },
                    {
                      name = "mipmaps",
                      type = "*",
                      description = "The number of mipmap levels in the texture, or a boolean.  If true, a full mipmap chain will be created.  If false, the texture will only have a single mipmap.",
                      default = "true"
                    },
                    {
                      name = "usage",
                      type = "table",
                      description = "A list of `TextureUsage` indicating how the texture will be used."
                    },
                    {
                      name = "label",
                      type = "string",
                      description = "A label for the Texture that will show up in debugging tools."
                    }
                  }
                }
              },
              returns = {
                {
                  name = "texture",
                  type = "Texture",
                  description = "The new Texture."
                }
              }
            },
            {
              arguments = {
                {
                  name = "image",
                  type = "string",
                  description = "An Image object holding pixel data to load into the Texture."
                },
                {
                  name = "options",
                  type = "table",
                  description = "Texture options.",
                  table = {
                    {
                      name = "type",
                      type = "TextureType",
                      description = "The type of the texture."
                    },
                    {
                      name = "format",
                      type = "TextureFormat",
                      description = "The format of the texture (ignored when images are provided).",
                      default = "'rgba8'"
                    },
                    {
                      name = "linear",
                      type = "boolean",
                      description = "Whether the texture is in linear color space instead of sRGB.  Linear textures should be used for non-color data, like normal maps.",
                      default = "false"
                    },
                    {
                      name = "samples",
                      type = "number",
                      description = "The number of samples in the texture, used for multisample antialiasing.  Currently must be 1 or 4.  Ignored when images are provided.",
                      default = "1"
                    },
                    {
                      name = "mipmaps",
                      type = "*",
                      description = "The number of mipmap levels in the texture, or a boolean.  If true, a full mipmap chain will be created.  If false, the texture will only have a single mipmap.",
                      default = "true"
                    },
                    {
                      name = "usage",
                      type = "table",
                      description = "A list of `TextureUsage` indicating how the texture will be used."
                    },
                    {
                      name = "label",
                      type = "string",
                      description = "A label for the Texture that will show up in debugging tools."
                    }
                  }
                }
              },
              returns = {
                {
                  name = "texture",
                  type = "Texture",
                  description = "The new Texture."
                }
              }
            },
            {
              arguments = {
                {
                  name = "images",
                  type = "table",
                  description = "A table of filenames or Images to load into the Texture."
                },
                {
                  name = "options",
                  type = "table",
                  description = "Texture options.",
                  table = {
                    {
                      name = "type",
                      type = "TextureType",
                      description = "The type of the texture."
                    },
                    {
                      name = "format",
                      type = "TextureFormat",
                      description = "The format of the texture (ignored when images are provided).",
                      default = "'rgba8'"
                    },
                    {
                      name = "linear",
                      type = "boolean",
                      description = "Whether the texture is in linear color space instead of sRGB.  Linear textures should be used for non-color data, like normal maps.",
                      default = "false"
                    },
                    {
                      name = "samples",
                      type = "number",
                      description = "The number of samples in the texture, used for multisample antialiasing.  Currently must be 1 or 4.  Ignored when images are provided.",
                      default = "1"
                    },
                    {
                      name = "mipmaps",
                      type = "*",
                      description = "The number of mipmap levels in the texture, or a boolean.  If true, a full mipmap chain will be created.  If false, the texture will only have a single mipmap.",
                      default = "true"
                    },
                    {
                      name = "usage",
                      type = "table",
                      description = "A list of `TextureUsage` indicating how the texture will be used."
                    },
                    {
                      name = "label",
                      type = "string",
                      description = "A label for the Texture that will show up in debugging tools."
                    }
                  }
                }
              },
              returns = {
                {
                  name = "texture",
                  type = "Texture",
                  description = "The new Texture."
                }
              }
            },
            {
              arguments = {
                {
                  name = "blob",
                  type = "Blob",
                  description = "A Blob object holding pixel data to load into the Texture."
                },
                {
                  name = "options",
                  type = "table",
                  description = "Texture options.",
                  table = {
                    {
                      name = "type",
                      type = "TextureType",
                      description = "The type of the texture."
                    },
                    {
                      name = "format",
                      type = "TextureFormat",
                      description = "The format of the texture (ignored when images are provided).",
                      default = "'rgba8'"
                    },
                    {
                      name = "linear",
                      type = "boolean",
                      description = "Whether the texture is in linear color space instead of sRGB.  Linear textures should be used for non-color data, like normal maps.",
                      default = "false"
                    },
                    {
                      name = "samples",
                      type = "number",
                      description = "The number of samples in the texture, used for multisample antialiasing.  Currently must be 1 or 4.  Ignored when images are provided.",
                      default = "1"
                    },
                    {
                      name = "mipmaps",
                      type = "*",
                      description = "The number of mipmap levels in the texture, or a boolean.  If true, a full mipmap chain will be created.  If false, the texture will only have a single mipmap.",
                      default = "true"
                    },
                    {
                      name = "usage",
                      type = "table",
                      description = "A list of `TextureUsage` indicating how the texture will be used."
                    },
                    {
                      name = "label",
                      type = "string",
                      description = "A label for the Texture that will show up in debugging tools."
                    }
                  }
                }
              },
              returns = {
                {
                  name = "texture",
                  type = "Texture",
                  description = "The new Texture."
                }
              }
            }
          }
        },
        {
          name = "present",
          tag = "work-submission",
          summary = "Update the desktop window contents.",
          description = "Presents the window texture to the desktop window.  This function is called automatically by the default implementation of `lovr.run`, so it normally does not need to be called.",
          key = "lovr.graphics.present",
          module = "lovr.graphics",
          notes = "This should be called after submitting the window pass (`lovr.graphics.getWindowPass`).  If the window texture has not been rendered to since the last present, this function does nothing.",
          related = {
            "lovr.graphics.submit",
            "lovr.graphics.getWindowPass"
          },
          variants = {
            {
              arguments = {},
              returns = {}
            }
          }
        },
        {
          name = "setBackgroundColor",
          tag = "graphics-global",
          summary = "Set the background color.",
          description = "Changes the global background color.  The textures in a render pass will be cleared to this color at the beginning of the pass if no other clear option is specified.  Additionally, the headset and window will be cleared to this color before rendering.",
          key = "lovr.graphics.setBackgroundColor",
          module = "lovr.graphics",
          notes = "Setting the background color in `lovr.draw` will apply on the following frame, since the default pass is cleared before `lovr.draw` is called.\n\nInternally, this color is applied to the default pass objects when retrieving one of them using `lovr.headset.getPass` or `lovr.graphics.getWindowPass`.  Both are called automatically by the default `lovr.run` implementation.\n\nUsing the background color to clear the display is expected to be more efficient than manually clearing after a render pass begins, especially on mobile GPUs.",
          related = {
            "lovr.graphics.newPass",
            "Pass:setClear",
            "Texture:clear",
            "Pass:fill"
          },
          variants = {
            {
              arguments = {
                {
                  name = "r",
                  type = "number",
                  description = "The red component of the background color."
                },
                {
                  name = "g",
                  type = "number",
                  description = "The green component of the background color."
                },
                {
                  name = "b",
                  type = "number",
                  description = "The blue component of the background color."
                },
                {
                  name = "a",
                  type = "number",
                  description = "The alpha component of the background color.",
                  default = "1.0"
                }
              },
              returns = {}
            },
            {
              arguments = {
                {
                  name = "hex",
                  type = "number",
                  description = "A hexcode (like `0xffffff`) to use for the background color (does not support alpha)."
                },
                {
                  name = "a",
                  type = "number",
                  description = "The alpha component of the background color.",
                  default = "1.0"
                }
              },
              returns = {}
            },
            {
              arguments = {
                {
                  name = "table",
                  type = "table",
                  description = "A table containing 3 or 4 color components."
                }
              },
              returns = {}
            }
          }
        },
        {
          name = "setTimingEnabled",
          tag = "graphics-global",
          summary = "Enable or disable timing stats.",
          description = "Enables or disables timing stats.  When enabled, `Pass:getStats` will return `submitTime` and `gpuTime` durations.  Timing is enabled by default when `t.graphics.debug` is set in `lovr.conf`.  Timing has a small amount of overhead, so it should only be enabled when needed.",
          key = "lovr.graphics.setTimingEnabled",
          module = "lovr.graphics",
          related = {
            "Pass:getStats"
          },
          variants = {
            {
              arguments = {
                {
                  name = "enable",
                  type = "boolean",
                  description = "Whether timing should be enabled."
                }
              },
              returns = {}
            }
          }
        },
        {
          name = "submit",
          tag = "work-submission",
          summary = "Submit recorded graphics work to the GPU.",
          description = "Submits work to the GPU.",
          key = "lovr.graphics.submit",
          module = "lovr.graphics",
          notes = "The submitted `Pass` objects will run in the order specified.  Commands within a single Pass do not have any ordering guarantees.\n\nSubmitting work to the GPU is not thread safe.  No other `lovr.graphics` or `Pass` functions may run at the same time as `lovr.graphics.submit`.\n\nCalling this function will invalidate any temporary buffers or passes that were created during the frame.\n\nSubmitting work to the GPU is a relatively expensive operation.  It's a good idea to batch all `Pass` objects into 1 submission if possible, unless there's a good reason not to.  One such reason would be that the frame has so much work that some of it needs to be submitted early to prevent the GPU from running out of things to do.  Another would be for `Readback` objects.\n\nBy default, this function is called with the default pass at the end of `lovr.draw` and `lovr.mirror`.\n\nIt is valid to submit zero passes.  This will send an empty batch of work to the GPU.",
          related = {
            "lovr.graphics.wait"
          },
          variants = {
            {
              arguments = {
                {
                  name = "...",
                  type = "Pass",
                  description = "The pass objects to submit.  Falsy values will be skipped."
                }
              },
              returns = {
                {
                  name = "true",
                  type = "boolean",
                  description = "Always returns true, for convenience when returning from `lovr.draw`."
                }
              }
            },
            {
              arguments = {
                {
                  name = "t",
                  type = "table",
                  description = "A table of passes to submit.  Falsy values will be skipped."
                }
              },
              returns = {
                {
                  name = "true",
                  type = "boolean",
                  description = "Always returns true, for convenience when returning from `lovr.draw`."
                }
              }
            }
          }
        },
        {
          name = "wait",
          tag = "work-submission",
          summary = "Stall the CPU until all submitted GPU work is finished.",
          description = "Waits for all submitted GPU work to finish.  A normal application that is trying to render graphics at a high framerate should never use this function, since waiting like this prevents the CPU from doing other useful work.  Otherwise, reasons to use this function might be for debugging or to force a `Readback` to finish immediately.",
          key = "lovr.graphics.wait",
          module = "lovr.graphics",
          related = {
            "lovr.graphics.submit"
          },
          variants = {
            {
              arguments = {},
              returns = {}
            }
          }
        }
      },
      objects = {
        {
          name = "Buffer",
          summary = "A block of memory on the GPU.",
          description = "A Buffer is a block of memory on the GPU.  It's like a GPU version of a `Blob`.  Lua code can write data to the buffer which uploads to VRAM, and shaders read buffer data during rendering. Compute shaders can also write to buffers.\n\nThe **size** of a Buffer is the number of bytes of VRAM it occupies.  It's set when the Buffer is created and can't be changed afterwards.\n\nBuffers can optionally have a **format**, which defines the type of data stored in the buffer. The format determines how Lua values are converted into binary.  Like the size, it can't change after the buffer is created.  `Shader:getBufferFormat` returns the format of a variable in a `Shader`.\n\nWhen a Buffer has a format, it also has a **length**, which is the number of items it holds, and a **stride**, which is the number of bytes between each item.\n\n`Buffer:setData` is used to upload data to the Buffer.  `Buffer:clear` can also be used to efficiently zero out a Buffer.\n\n`Buffer:getData` can be used to download data from the Buffer, but be aware that it stalls the GPU until the download is complete, which is very slow!  `Buffer:newReadback` will instead download the data in the background, which avoids costly stalls.\n\nBuffers are often used for mesh data.  Vertices stored in buffers can be drawn using `Pass:mesh`.  `Mesh` objects can also be used, which wrap Buffers along with some extra metadata.\n\nBuffers can be \"bound\" to a variable in a Shader using `Pass:send`.  That means that the next time the shader runs, the data from the Buffer will be used for the stuff in the variable.\n\nIt's important to understand that data from a Buffer will only be used at the point when graphics commands are actually submitted.  This example records 2 draws, changing the buffer data between each one:\n\n    buffer:setData(data1)\n    pass:mesh(buffer)\n    buffer:setData(data2)\n    pass:mesh(buffer)\n    lovr.graphics.submit(pass)\n\n**Both** draws will use `data2` here!  That's because `lovr.graphics.submit` is where the draws actually get processed, so they both see the \"final\" state of the buffer.  The data in a Buffer can't be 2 things at once!  If you need multiple versions of data, it's best to use a bigger buffer with offsets (or multiple buffers).",
          key = "Buffer",
          module = "lovr.graphics",
          constructors = {
            "lovr.graphics.newBuffer",
            "lovr.graphics.getBuffer"
          },
          methods = {
            {
              name = "clear",
              tag = "buffer-transfer",
              summary = "Clear data in the Buffer.",
              description = "Clears a range of data in the Buffer to a value.",
              key = "Buffer:clear",
              module = "lovr.graphics",
              related = {
                "Texture:clear"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "offset",
                      type = "number",
                      description = "The offset of the range of the Buffer to clear, in bytes.  Must be a multiple of 4.",
                      default = "0"
                    },
                    {
                      name = "extent",
                      type = "number",
                      description = "The number of bytes to clear.  If `nil`, clears to the end of the Buffer.  Must be a multiple of 4.",
                      default = "nil"
                    },
                    {
                      name = "value",
                      type = "number",
                      description = "The value to clear to.  This will be interpreted as a 32 bit number, which will be repeated across the clear range.",
                      default = "0x00000000"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "getData",
              tag = "buffer-transfer",
              summary = "Get the data in the Buffer.",
              description = "Downloads the Buffer's data from VRAM and returns it as a table.  This function is very very slow because it stalls the CPU until the data is finished downloading, so it should only be used for debugging or non-interactive scripts.  `Buffer:newReadback` is an alternative that returns a `Readback` object, which will not block the CPU.",
              key = "Buffer:getData",
              module = "lovr.graphics",
              notes = "The length of the table will equal the number of items read.  Here are some examples of how the table is formatted:\n\n    buffer = lovr.graphics.newBuffer('int', { 7 })\n    buffer:getData() --> returns { 7 }\n\n    buffer = lovr.graphics.newBuffer('vec3', { 7, 8, 9 })\n    buffer:getData() --> returns {{ 7, 8, 9 }}\n\n    buffer = lovr.graphics.newBuffer('int', { 1, 2, 3 })\n    buffer:getData() --> returns { 1, 2, 3 }\n\n    buffer = lovr.graphics.newBuffer({ 'vec2', 'vec2' }, {\n      vec2(1,2), vec2(3,4),\n      vec2(5,6), vec2(7,8)\n    })\n    buffer:getData() --> returns { { 1, 2, 3, 4 }, { 5, 6, 7, 8 } }\n\n    buffer = lovr.graphics.newBuffer({\n      { 'a', 'float' },\n      { 'b', 'float' }\n    }, { a = 1, b = 2 })\n    buffer:getData() --> returns { { 1, 2 } }\n\n    buffer = lovr.graphics.newBuffer({\n      { 'x', 'int', 3 }\n    }, { x = { 1, 2, 3 } })\n    buffer:getData() --> returns { { x = { 1, 2, 3 } } }\n\n    buffer = lovr.graphics.newBuffer({\n      { 'lights', {\n        { 'pos', 'vec3' },\n        { 'size', 'float' },\n      }, 10}\n    }, data)\n    buffer:getData() --> returns { { lights = { { pos = ..., size = ... }, ... } } }\n\nIn summary, each individual item is wrapped in a table, except if the format is a single number. If the format has nested types or arrays then the tables will be key-value, otherwise they will use numeric keys.",
              related = {
                "Buffer:newReadback",
                "Buffer:mapData",
                "Readback:getData"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the first item to read.",
                      default = "1"
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "The number of items to read.  If nil, reads the remainder of the buffer.",
                      default = "nil"
                    }
                  },
                  returns = {
                    {
                      name = "t",
                      type = "table",
                      description = "The table with the Buffer's data."
                    }
                  }
                }
              }
            },
            {
              name = "getFormat",
              tag = "buffer-metadata",
              summary = "Get the format of the Buffer.",
              description = "Returns the format the Buffer was created with.",
              key = "Buffer:getFormat",
              module = "lovr.graphics",
              examples = {
                {
                  code = "function lovr.load()\n  buffer = lovr.graphics.newBuffer({\n    { 'a', 'float' },\n    { 'b', 'un16x2' }\n  })\n\n  for i, field in ipairs(buffer:getFormat()) do\n    print(('%s: %s'):format(field.name, field.type))\n  end\n\n  -- prints the following:\n  -- a: f32\n  -- b: un16x2\nend"
                }
              },
              related = {
                "Buffer:getSize",
                "Buffer:getLength",
                "Buffer:getStride"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "format",
                      type = "table",
                      description = "A list of fields comprising the format.",
                      table = {
                        {
                          name = "[].name",
                          type = "string",
                          description = "The name of the field (if fields were created with names)."
                        },
                        {
                          name = "[].type",
                          type = "*",
                          description = "The `DataType` of the field, or a recursive table with the sub-format."
                        },
                        {
                          name = "[].offset",
                          type = "number",
                          description = "The offset of the field relative to its parent, in bytes."
                        },
                        {
                          name = "[].length",
                          type = "number",
                          description = "The array length of the field, or `nil` if it's not an array."
                        },
                        {
                          name = "[].stride",
                          type = "number",
                          description = "The stride of the field in bytes, or `nil` if it's not an array."
                        }
                      }
                    }
                  }
                }
              }
            },
            {
              name = "getLength",
              tag = "buffer-metadata",
              summary = "Get the length of the Buffer.",
              description = "Returns the length of the Buffer, or `nil` if the Buffer was not created with a format.",
              key = "Buffer:getLength",
              module = "lovr.graphics",
              related = {
                "Buffer:getSize",
                "Buffer:getStride"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "length",
                      type = "number",
                      description = "The length of the Buffer."
                    }
                  }
                }
              }
            },
            {
              name = "getPointer",
              tag = "buffer-transfer",
              summary = "Get a writable pointer to the Buffer's memory.",
              description = "Returns a pointer to GPU memory and schedules a copy from this pointer to the buffer's data. The data in the pointer will replace the data in the buffer.  This is intended for use with the LuaJIT FFI or for passing to C libraries.",
              key = "Buffer:getPointer",
              module = "lovr.graphics",
              deprecated = true,
              notes = "The pointer remains valid until the next call to `lovr.graphics.submit`, during which the data in the pointer will be uploaded to the buffer.\n\nThe initial contents of the pointer are undefined.\n\nSpecial care should be taken when writing data:\n\n- Reading data from the pointer will be very slow on some systems, and should be avoided.\n- It is better to write data to the pointer sequentially.  Random access may be slower.",
              related = {
                "Blob:getPointer",
                "Buffer:mapData"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "offset",
                      type = "number",
                      description = "A byte offset in the buffer to write to.",
                      default = "0"
                    },
                    {
                      name = "extent",
                      type = "number",
                      description = "The number of bytes to replace.  If nil, writes to the rest of the buffer.",
                      default = "nil"
                    }
                  },
                  returns = {
                    {
                      name = "pointer",
                      type = "lightuserdata",
                      description = "A pointer to the Buffer's memory."
                    }
                  }
                }
              }
            },
            {
              name = "getSize",
              tag = "buffer-metadata",
              summary = "Get the size of the Buffer, in bytes.",
              description = "Returns the size of the Buffer in VRAM, in bytes.  This is the same as `length * stride`.\n\nThe size of the Buffer can't change after it's created.",
              key = "Buffer:getSize",
              module = "lovr.graphics",
              related = {
                "Buffer:getLength",
                "Buffer:getStride"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "size",
                      type = "number",
                      description = "The size of the Buffer, in bytes."
                    }
                  }
                }
              }
            },
            {
              name = "getStride",
              tag = "buffer-metadata",
              summary = "Get the stride of the Buffer, in bytes.",
              description = "Returns the distance between each item in the Buffer, in bytes, or `nil` if the Buffer was not created with a format.",
              key = "Buffer:getStride",
              module = "lovr.graphics",
              notes = "When a Buffer is created, the stride can be set explicitly, otherwise it will be automatically computed based on the fields in the Buffer.\n\nStrides can not be zero, and can not be smaller than the size of a single item.",
              related = {
                "Buffer:getSize",
                "Buffer:getLength"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "stride",
                      type = "number",
                      description = "The stride of the Buffer, in bytes."
                    }
                  }
                }
              }
            },
            {
              name = "isTemporary",
              tag = "buffer-metadata",
              summary = "Check if the Buffer is temporary.",
              description = "Returns whether the Buffer is temporary.",
              key = "Buffer:isTemporary",
              module = "lovr.graphics",
              deprecated = true,
              related = {
                "lovr.graphics.getBuffer"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "temporary",
                      type = "boolean",
                      description = "Whether the Buffer is temporary."
                    }
                  }
                }
              }
            },
            {
              name = "mapData",
              tag = "buffer-transfer",
              summary = "Get a writable pointer to the Buffer's memory.",
              description = "Returns a pointer to GPU memory and schedules a copy from this pointer to the buffer's data. The data in the pointer will replace the data in the buffer.  This is intended for use with the LuaJIT FFI or for passing to C libraries.",
              key = "Buffer:mapData",
              module = "lovr.graphics",
              notes = "The pointer remains valid until the next call to `lovr.graphics.submit`, during which the data in the pointer will be uploaded to the buffer.\n\nThe initial contents of the pointer are undefined.\n\nSpecial care should be taken when writing data:\n\n- Reading data from the pointer will be very slow on some systems, and should be avoided.\n- It is better to write data to the pointer sequentially.  Random access may be slower.",
              related = {
                "Blob:getPointer",
                "Buffer:getPointer"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "offset",
                      type = "number",
                      description = "A byte offset in the buffer to write to.",
                      default = "0"
                    },
                    {
                      name = "extent",
                      type = "number",
                      description = "The number of bytes to replace.  If nil, writes to the rest of the buffer.",
                      default = "nil"
                    }
                  },
                  returns = {
                    {
                      name = "pointer",
                      type = "lightuserdata",
                      description = "A pointer to the Buffer's memory."
                    }
                  }
                }
              }
            },
            {
              name = "newReadback",
              tag = "buffer-transfer",
              summary = "Read back the contents of the Buffer asynchronously.",
              description = "Creates and returns a new `Readback` that will download the data in the Buffer from VRAM. Once the readback is complete, `Readback:getData` returns the data as a table, or `Readback:getBlob` returns the data as a `Blob`.",
              key = "Buffer:newReadback",
              module = "lovr.graphics",
              notes = "The offset and extent arguments must be a multiple of the Buffer's stride (unless it was created without a format).",
              related = {
                "Buffer:getData",
                "Texture:newReadback"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "offset",
                      type = "number",
                      description = "A byte offset to read from.",
                      default = "0"
                    },
                    {
                      name = "extent",
                      type = "number",
                      description = "The number of bytes to read.  If nil, reads the rest of the buffer.",
                      default = "nil"
                    }
                  },
                  returns = {
                    {
                      name = "readback",
                      type = "Readback",
                      description = "A new Readback object."
                    }
                  }
                }
              }
            },
            {
              name = "setData",
              tag = "buffer-transfer",
              summary = "Change the data in the Buffer.",
              description = "Copies data to the Buffer from either a table, `Blob`, or `Buffer`.",
              key = "Buffer:setData",
              module = "lovr.graphics",
              examples = {
                {
                  description = "Various examples of copying different formats.",
                  code = "function lovr.load()\n  buffer = lovr.graphics.newBuffer('int', 3)\n  buffer:setData({ 1, 2, 3 })\n\n  buffer = lovr.graphics.newBuffer('vec3', 2)\n  buffer:setData({ 1,2,3, 4,5,6 })\n  buffer:setData({ { 1, 2, 3 }, { 4, 5, 6 } })\n  buffer:setData({ vec3(1, 2, 3), vec3(4, 5, 6) })\n\n  -- When the Buffer's length is 1, wrapping in table is optional:\n  buffer = lovr.graphics.newBuffer('vec3')\n  buffer:setData(1, 2, 3)\n  buffer:setData(vec3(1, 2, 3))\n\n  -- Same for key-value structs\n  buffer = lovr.graphics.newBuffer({\n    { 'x', 'float' },\n    { 'y', 'float' }\n  })\n  buffer:setData({ x = 1, y = 2 })\n\n  -- Key/value formats\n  buffer = lovr.graphics.newBuffer({\n    { 'x', 'float' },\n    { 'y', 'float' }\n  }, 3)\n  buffer:setData({\n    { x = 1, y = 2 },\n    { x = 3, y = 4 },\n    { x = 5, y = 6 }\n  })\n  buffer:setData({ 1, 2, 3, 4, 5, 6 })\n  buffer:setData({ { 1, 2 }, { 3, 4 }, { 5, 6 } })\n\n  -- Nested formats\n  buffer = lovr.graphics.newBuffer({\n    { 'a', {\n      {'b', {\n        'c', 'float'\n      }}\n    }}\n  })\n  buffer:setData({ a = { b = { c = 42 } } })\n\n  -- Inner arrays\n  buffer = lovr.graphics.newBuffer({\n    { 'positions', 'vec3', 10 },\n    { 'sizes', 'float', 10 },\n    layout = 'std140'\n  })\n  local data = { positions = {}, sizes = {} }\n  for i = 1, buffer:getLength() do\n    data.positions[i] = vec3(i, i, i)\n    data.sizes[i] = i\n  end\n  buffer:setData(data)\nend"
                }
              },
              notes = "One gotcha is that unspecified fields will be set to zero.  Here's an example:\n\n    buffer = lovr.graphics.newBuffer({{ 'x', 'int' }, { 'y', 'int' }})\n    buffer:setData({ x = 1, y = 1 }) -- set the data\n    buffer:setData({ x = 1 }) -- set the data, partially\n    -- buffer data is now {x=1, y=0}!\n\nThis doesn't apply to separate items in the buffer.  For example, if the Buffer's length is 2 and only the 1st item is set, the second item will remain undisturbed:\n\n    buffer = lovr.graphics.newBuffer({{ 'x', 'int' }, { 'y', 'int' }}, 2)\n    buffer:setData({{ x = 1, y = 1 }, { x = 2, y = 2 }}) -- set the data\n    buffer:setData({{ x = 1 }}) -- set one item, partially\n    -- buffer data is now {{x=1, y=0}, {x=2, y=2}}",
              variants = {
                {
                  arguments = {
                    {
                      name = "table",
                      type = "table",
                      description = "A flat or nested table of items to copy to the Buffer (see notes for format)."
                    },
                    {
                      name = "destinationIndex",
                      type = "number",
                      description = "The index of the first value in the Buffer to update.",
                      default = "1"
                    },
                    {
                      name = "sourceIndex",
                      type = "number",
                      description = "The index in the table to copy from.",
                      default = "1"
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "The number of items to copy.  `nil` will copy as many items as possible, based on the lengths of the source and destination.",
                      default = "nil"
                    }
                  },
                  returns = {}
                },
                {
                  description = "Copies a single field to a buffer with numbers (buffer length must be 1).",
                  arguments = {
                    {
                      name = "...numbers",
                      type = "number",
                      description = "Numerical components to copy to the buffer."
                    }
                  },
                  returns = {}
                },
                {
                  description = "Copies a single vector to a buffer (buffer length must be 1).",
                  arguments = {
                    {
                      name = "vector",
                      type = "*",
                      description = "Vector objects to copy to the buffer."
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "blob",
                      type = "Blob",
                      description = "The Blob to copy data from."
                    },
                    {
                      name = "destinationOffset",
                      type = "number",
                      description = "The byte offset to copy to.",
                      default = "0"
                    },
                    {
                      name = "sourceOffset",
                      type = "number",
                      description = "The byte offset to copy from.",
                      default = "0"
                    },
                    {
                      name = "size",
                      type = "number",
                      description = "The number of bytes to copy.  If nil, copies as many bytes as possible.",
                      default = "nil"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "buffer",
                      type = "Buffer",
                      description = "The Buffer to copy data from."
                    },
                    {
                      name = "destinationOffset",
                      type = "number",
                      description = "The byte offset to copy to.",
                      default = "0"
                    },
                    {
                      name = "sourceOffset",
                      type = "number",
                      description = "The byte offset to copy from.",
                      default = "0"
                    },
                    {
                      name = "size",
                      type = "number",
                      description = "The number of bytes to copy.  If nil, copies as many bytes as possible.",
                      default = "nil"
                    }
                  },
                  returns = {}
                }
              }
            }
          },
          sections = {
            {
              name = "Metadata",
              tag = "buffer-metadata"
            },
            {
              name = "Transfers",
              tag = "buffer-transfer"
            }
          }
        },
        {
          name = "Font",
          summary = "A Font used to render text.",
          description = "Font objects are used to render text with `Pass:text`.  The active font can be changed using `Pass:setFont`.  The default font is Varela Round, which is used when no font is active, and can be retrieved using `lovr.graphics.getDefaultFont`.  Custom fonts can be loaded from TTF files using `lovr.graphics.newFont`.\n\nEach Font uses a `Rasterizer` to load the TTF file and create images for each glyph. As text is drawn, the Font uploads images from the Rasterizer to a GPU texture atlas as needed.  The Font also performs text layout and mesh generation for strings of text.\n\nLÖVR uses a text rendering technique called \"multichannel signed distance fields\" (MSDF), which makes the font rendering remain crisp when text is viewed up close.",
          key = "Font",
          module = "lovr.graphics",
          constructors = {
            "lovr.graphics.newFont",
            "lovr.graphics.getDefaultFont"
          },
          methods = {
            {
              name = "getAscent",
              summary = "Get the ascent of the Font.",
              description = "Returns the ascent of the font.  The ascent is the maximum amount glyphs ascend above the baseline.  The units depend on the font's pixel density.  With the default density, the units correspond to meters.",
              key = "Font:getAscent",
              module = "lovr.graphics",
              related = {
                "Rasterizer:getAscent",
                "Font:getDescent",
                "Font:getHeight",
                "Font:getKerning",
                "Font:getWidth"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "ascent",
                      type = "number",
                      description = "The ascent of the font."
                    }
                  }
                }
              }
            },
            {
              name = "getDescent",
              summary = "Get the descent of the Font.",
              description = "Returns the descent of the font.  The descent is the maximum amount glyphs descend below the baseline.  The units depend on the font's pixel density.  With the default density, the units correspond to meters.",
              key = "Font:getDescent",
              module = "lovr.graphics",
              related = {
                "Rasterizer:getDescent",
                "Font:getAscent",
                "Font:getHeight",
                "Font:getKerning",
                "Font:getWidth"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "descent",
                      type = "number",
                      description = "The descent of the font."
                    }
                  }
                }
              }
            },
            {
              name = "getHeight",
              summary = "Get the height of the Font.",
              description = "Returns the height of the font, sometimes also called the leading.  This is the full height of a line of text, including the space between lines.  Each line of a multiline string is separated on the y axis by this height, multiplied by the font's line spacing.  The units depend on the font's pixel density.  With the default density, the units correspond to meters.",
              key = "Font:getHeight",
              module = "lovr.graphics",
              related = {
                "Rasterizer:getLeading",
                "Font:getLineSpacing",
                "Font:setLineSpacing",
                "Font:getAscent",
                "Font:getDescent",
                "Font:getKerning",
                "Font:getWidth",
                "Font:getLines"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "height",
                      type = "number",
                      description = "The height of the font."
                    }
                  }
                }
              }
            },
            {
              name = "getKerning",
              summary = "Get the kerning between 2 glyphs.",
              description = "Returns the kerning between 2 glyphs.  Kerning is a slight horizontal adjustment between 2 glyphs to improve the visual appearance.  It will often be negative.  The units depend on the font's pixel density.  With the default density, the units correspond to meters.",
              key = "Font:getKerning",
              module = "lovr.graphics",
              related = {
                "Rasterizer:getKerning",
                "Font:getAscent",
                "Font:getDescent",
                "Font:getHeight",
                "Font:getWidth"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "first",
                      type = "string",
                      description = "The first character."
                    },
                    {
                      name = "second",
                      type = "string",
                      description = "The second character."
                    }
                  },
                  returns = {
                    {
                      name = "keming",
                      type = "number",
                      description = "The kerning between the two glyphs."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "firstCodepoint",
                      type = "number",
                      description = "The first codepoint."
                    },
                    {
                      name = "second",
                      type = "string",
                      description = "The second character."
                    }
                  },
                  returns = {
                    {
                      name = "keming",
                      type = "number",
                      description = "The kerning between the two glyphs."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "first",
                      type = "string",
                      description = "The first character."
                    },
                    {
                      name = "secondCodepoint",
                      type = "number",
                      description = "The second codepoint."
                    }
                  },
                  returns = {
                    {
                      name = "keming",
                      type = "number",
                      description = "The kerning between the two glyphs."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "firstCodepoint",
                      type = "number",
                      description = "The first codepoint."
                    },
                    {
                      name = "secondCodepoint",
                      type = "number",
                      description = "The second codepoint."
                    }
                  },
                  returns = {
                    {
                      name = "keming",
                      type = "number",
                      description = "The kerning between the two glyphs."
                    }
                  }
                }
              }
            },
            {
              name = "getLineSpacing",
              summary = "Get the line spacing of the Font.",
              description = "Returns the line spacing of the Font.  When spacing out lines, the height of the font is multiplied the line spacing to get the final spacing value.  The default is 1.0.",
              key = "Font:getLineSpacing",
              module = "lovr.graphics",
              related = {
                "Font:getHeight"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "spacing",
                      type = "number",
                      description = "The line spacing of the font."
                    }
                  }
                }
              }
            },
            {
              name = "getLines",
              summary = "Wrap a string into a sequence of lines.",
              description = "Returns a table of wrapped lines for a piece of text, given a line length limit.  Newlines are handled correctly.  The wrap limit units depend on the pixel density of the font.  With the default pixel density, the units correspond to meters when the font is drawn at 1.0 scale.",
              key = "Font:getLines",
              module = "lovr.graphics",
              related = {
                "Font:getWidth",
                "Font:getHeight",
                "Pass:text"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "string",
                      type = "string",
                      description = "The text to wrap."
                    },
                    {
                      name = "wrap",
                      type = "number",
                      description = "The line length to wrap at."
                    }
                  },
                  returns = {
                    {
                      name = "lines",
                      type = "table",
                      description = "A table strings, one for each wrapped line (without any color information)."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "strings",
                      type = "table",
                      description = "A table of colored strings, each given as a `{ color, string }` pair.  The color can be a `Vec3`, `Vec4`, table, or hexcode."
                    },
                    {
                      name = "wrap",
                      type = "number",
                      description = "The line length to wrap at."
                    }
                  },
                  returns = {
                    {
                      name = "lines",
                      type = "table",
                      description = "A table strings, one for each wrapped line (without any color information)."
                    }
                  }
                }
              }
            },
            {
              name = "getPixelDensity",
              summary = "Get the pixel density of the Font.",
              description = "Returns the pixel density of the font.  The density is a \"pixels per world unit\" factor that controls how the pixels in the font's texture are mapped to units in the coordinate space.\n\nThe default pixel density is set to the height of the font.  This means that lines of text rendered with a scale of 1.0 come out to 1 unit (meter) tall.  However, if this font was drawn to a 2D texture where the units are in pixels, the font would still be drawn 1 unit (pixel) tall!  Scaling the coordinate space or the size of the text by the height of the font would fix this.  However, a more convenient option is to set the pixel density of the font to 1.0 when doing 2D rendering to make the font's size match up with the pixels of the canvas.",
              key = "Font:getPixelDensity",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "density",
                      type = "number",
                      description = "The pixel density of the font."
                    }
                  }
                }
              }
            },
            {
              name = "getRasterizer",
              summary = "Get the Font's Rasterizer.",
              description = "Returns the Rasterizer object backing the Font.",
              key = "Font:getRasterizer",
              module = "lovr.graphics",
              related = {
                "lovr.graphics.newFont",
                "lovr.data.newRasterizer"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "rasterizer",
                      type = "Rasterizer",
                      description = "The Rasterizer."
                    }
                  }
                }
              }
            },
            {
              name = "getVertices",
              summary = "Get the vertices for a piece of text.",
              description = "Returns a table of vertices for a piece of text, along with a Material to use when rendering it. The Material returned by this function may not be the same if the Font's texture atlas needs to be recreated with a bigger size to make room for more glyphs.",
              key = "Font:getVertices",
              module = "lovr.graphics",
              notes = "Each vertex is a table of 4 floating point numbers with the following data:\n\n    { x, y, u, v }\n\nThese could be placed in a vertex buffer using the following buffer format:\n\n    { 'vec2:VertexPosition', 'vec2:VertexUV' }",
              variants = {
                {
                  arguments = {
                    {
                      name = "halign",
                      type = "HorizontalAlign",
                      description = "The horizontal align."
                    },
                    {
                      name = "valign",
                      type = "VerticalAlign",
                      description = "The vertical align."
                    }
                  },
                  returns = {
                    {
                      name = "vertices",
                      type = "table",
                      description = "The table of vertices.  See below for the format of each vertex."
                    },
                    {
                      name = "material",
                      type = "Material",
                      description = "A Material to use when rendering the vertices."
                    }
                  }
                }
              }
            },
            {
              name = "getWidth",
              summary = "Get the width of rendered text.",
              description = "Returns the maximum width of a piece of text.  This function does not perform wrapping but does respect newlines in the text.",
              key = "Font:getWidth",
              module = "lovr.graphics",
              related = {
                "Font:getAscent",
                "Font:getDescent",
                "Font:getHeight",
                "Font:getKerning",
                "Font:getLines"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "string",
                      type = "string",
                      description = "The text to measure."
                    }
                  },
                  returns = {
                    {
                      name = "width",
                      type = "number",
                      description = "The maximum width of the text."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "strings",
                      type = "table",
                      description = "A table of colored strings, each given as a `{ color, string }` pair.  The color can be a `Vec3`, `Vec4`, table, or hexcode."
                    }
                  },
                  returns = {
                    {
                      name = "width",
                      type = "number",
                      description = "The maximum width of the text."
                    }
                  }
                }
              }
            },
            {
              name = "setLineSpacing",
              summary = "Set the line spacing of the Font.",
              description = "Sets the line spacing of the Font.  When spacing out lines, the height of the font is multiplied by the line spacing to get the final spacing value.  The default is 1.0.",
              key = "Font:setLineSpacing",
              module = "lovr.graphics",
              related = {
                "Font:getHeight"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "spacing",
                      type = "number",
                      description = "The new line spacing."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setPixelDensity",
              summary = "Set the pixel density of the Font.",
              description = "Sets the pixel density of the font.  The density is a \"pixels per world unit\" factor that controls how the pixels in the font's texture are mapped to units in the coordinate space.\n\nThe default pixel density is set to the height of the font.  This means that lines of text rendered with a scale of 1.0 come out to 1 unit (meter) tall.  However, if this font was drawn to a 2D texture where the units are in pixels, the font would still be drawn 1 unit (pixel) tall!  Scaling the coordinate space or the size of the text by the height of the font would fix this.  However, a more convenient option is to set the pixel density of the font to 1.0 when doing 2D rendering to make the font's size match up with the pixels of the canvas.",
              key = "Font:setPixelDensity",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {
                    {
                      name = "density",
                      type = "number",
                      description = "The new pixel density of the font."
                    }
                  },
                  returns = {}
                },
                {
                  description = "Resets the pixel density to the default, which is given by `Font:getHeight`.",
                  arguments = {},
                  returns = {}
                }
              }
            }
          },
          notes = "MSDF text requires a special shader to work.  LÖVR will automatically switch to this shader if no shader is active on the `Pass`.  This font shader is also available as a `DefaultShader`."
        },
        {
          name = "Material",
          summary = "A set of properties and textures that define the properties of a surface.",
          description = "Materials are a set of properties and textures that define the properties of a surface, like what color it is, how bumpy or shiny it is, etc. `Shader` code can use the data from a material to compute lighting.\n\nMaterials are immutable, and can't be changed after they are created.  Instead, a new Material should be created with the updated properties.\n\n`Pass:setMaterial` changes the active material, causing it to affect rendering until the active material is changed again.\n\nUsing material objects is optional.  `Pass:setMaterial` can take a `Texture`, and `Pass:setColor` can change the color of objects, so basic tinting and texturing of surfaces does not require a full material to be created.  Also, a custom material system could be developed by sending textures and other data to shaders manually.\n\n`Model` objects will create materials for all of the materials defined in the model file.\n\nIn shader code, non-texture material properties can be accessed as `Material.<property>`, and material textures can be accessed as `<Type>Texture`, e.g. `RoughnessTexture`.",
          key = "Material",
          module = "lovr.graphics",
          constructors = {
            "lovr.graphics.newMaterial"
          },
          methods = {
            {
              name = "getProperties",
              summary = "Get the properties of the Material.",
              description = "Returns the properties of the Material in a table.",
              key = "Material:getProperties",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "properties",
                      type = "table",
                      description = "The Material properties."
                    }
                  }
                }
              }
            }
          }
        },
        {
          name = "Mesh",
          summary = "A drawable triangle mesh.",
          description = "Meshes store arbitrary geometry data, and can be drawn with `Pass:draw`.\n\nMeshes hold a list of **vertices**.  The number of vertices is declared upfront when the Mesh is created, and it can not be resized afterwards.\n\nThe Mesh has a **vertex format**, which is a set of **attributes** comprising each vertex, like a `position`, `color`, etc.\n\nThe **vertex indices** in the Mesh describe the order that the vertices are rendered in.  This is an optimization that allows vertices to be reused if they are used for multiple triangles, without duplicating all of their data.\n\nThe Mesh has a **draw mode**, which controls how the vertices are connected together to create pixels.  It can either be `points`, `lines`, or `triangles`.\n\nThe Mesh can have a `Material` applied, which defines colors, textures, and other properties of its surface.\n\nThe **draw range** of the Mesh defines a subset of the vertices to render when the Mesh is drawn.\n\nThe **bounding box** of the Mesh allows LÖVR to skip rendering it when it's out of view.",
          key = "Mesh",
          module = "lovr.graphics",
          constructors = {
            "lovr.graphics.newMesh"
          },
          methods = {
            {
              name = "computeBoundingBox",
              summary = "Compute the bounding box of the Mesh.",
              description = "Computes the axis-aligned bounding box of the Mesh from its vertices.\n\nIf the Mesh was created with the `gpu` storage mode, this function will do nothing and return `false`.\n\nIf the Mesh does not have an attribute named `VertexPosition` with the `f32x3` (aka `vec3`) type, this function will do nothing and return `false`.\n\nOtherwise, the bounding box will be set and the return value will be `true`.\n\nThe bounding box can also be assigned manually using `Mesh:setBoundingBox`, which can be used to set the bounding box on a `gpu` mesh or for cases where the bounding box is already known.\n\nPasses will use the bounding box of a Mesh to cull it against the cameras when `Pass:setViewCull` is enabled, which avoids rendering it when it's out of view.",
              key = "Mesh:computeBoundingBox",
              module = "lovr.graphics",
              related = {
                "Mesh:getBoundingBox",
                "Mesh:setBoundingBox",
                "Pass:setViewCull",
                "Collider:getAABB",
                "Shape:getAABB",
                "Model:getBoundingBox",
                "ModelData:getBoundingBox"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "updated",
                      type = "boolean",
                      description = "Whether the bounding box was updated."
                    }
                  }
                }
              }
            },
            {
              name = "getBoundingBox",
              summary = "Get the bounding box of the Mesh.",
              description = "Returns the axis-aligned bounding box of the Mesh, or `nil` if the Mesh doesn't have a bounding box.\n\nMeshes with the `cpu` storage mode can compute their bounding box automatically using `Mesh:computeBoundingBox`.  The bounding box can also be set manually using `Mesh:setBoundingBox`.\n\nPasses will use the bounding box of a Mesh to cull it against the cameras when `Pass:setViewCull` is enabled, which avoids rendering it when it's out of view.",
              key = "Mesh:getBoundingBox",
              module = "lovr.graphics",
              related = {
                "Mesh:computeBoundingBox",
                "Pass:setViewCull",
                "Collider:getAABB",
                "Shape:getAABB",
                "Model:getBoundingBox",
                "ModelData:getBoundingBox"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "minx",
                      type = "number",
                      description = "The minimum x coordinate of the bounding box."
                    },
                    {
                      name = "maxx",
                      type = "number",
                      description = "The maximum x coordinate of the bounding box."
                    },
                    {
                      name = "miny",
                      type = "number",
                      description = "The minimum y coordinate of the bounding box."
                    },
                    {
                      name = "maxy",
                      type = "number",
                      description = "The maximum y coordinate of the bounding box."
                    },
                    {
                      name = "minz",
                      type = "number",
                      description = "The minimum z coordinate of the bounding box."
                    },
                    {
                      name = "maxz",
                      type = "number",
                      description = "The maximum z coordinate of the bounding box."
                    }
                  }
                }
              }
            },
            {
              name = "getDrawMode",
              summary = "Get the draw mode of the Mesh.",
              description = "Returns the `DrawMode` of the mesh, which controls how the vertices in the Mesh are connected together to create pixels.  The default is `triangles`.",
              key = "Mesh:getDrawMode",
              module = "lovr.graphics",
              related = {
                "Pass:setMeshMode"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "mode",
                      type = "DrawMode",
                      description = "The current draw mode."
                    }
                  }
                }
              }
            },
            {
              name = "getIndexBuffer",
              summary = "Get the Buffer backing the vertex indices of the Mesh.",
              description = "Returns the `Buffer` object that holds the data for the vertex indices in the Mesh.\n\nThis can be `nil` if the Mesh doesn't have any indices.\n\nIf a Mesh uses the `cpu` storage mode, the index buffer is internal to the `Mesh` and this function will return `nil`.  This ensures that the CPU data for the Mesh does not get out of sync with the GPU data in the Buffer.",
              key = "Mesh:getIndexBuffer",
              module = "lovr.graphics",
              related = {
                "Mesh:getIndices",
                "Mesh:setIndices",
                "Mesh:getVertexBuffer"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "buffer",
                      type = "Buffer",
                      description = "The index buffer."
                    }
                  }
                }
              }
            },
            {
              name = "getIndices",
              summary = "Get the vertex indices in the Mesh.",
              description = "Returns a table with the Mesh's vertex indices.",
              key = "Mesh:getIndices",
              module = "lovr.graphics",
              notes = "This function will be very very slow if the Mesh's storage is `gpu`, because the data needs to be downloaded from the GPU.",
              related = {
                "Mesh:getIndexBuffer",
                "Mesh:setIndexBuffer"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "t",
                      type = "table",
                      description = "A table of numbers with the 1-based vertex indices."
                    }
                  }
                }
              }
            },
            {
              name = "getMaterial",
              summary = "Get the Material applied to the Mesh.",
              description = "Returns the `Material` applied to the Mesh.",
              key = "Mesh:getMaterial",
              module = "lovr.graphics",
              related = {
                "Pass:setMaterial",
                "Model:getMaterial",
                "lovr.graphics.newMaterial"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "material",
                      type = "Material",
                      description = "The material."
                    }
                  }
                }
              }
            },
            {
              name = "getVertexBuffer",
              summary = "Get the Buffer backing the vertices of the Mesh.",
              description = "Returns the `Buffer` object that holds the data for the vertices in the Mesh.\n\nIf a Mesh uses the `cpu` storage mode, the vertex buffer is internal to the `Mesh` and this function will return `nil`.  This ensures that the CPU data for the Mesh does not get out of sync with the GPU data in the Buffer.",
              key = "Mesh:getVertexBuffer",
              module = "lovr.graphics",
              related = {
                "Mesh:getVertices",
                "Mesh:setVertices",
                "Mesh:getIndexBuffer"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "buffer",
                      type = "Buffer",
                      description = "The vertex buffer."
                    }
                  }
                }
              }
            },
            {
              name = "getVertexCount",
              summary = "Get the number of vertices in the Mesh.",
              description = "Returns the number of vertices in the Mesh.  The vertex count is set when the Mesh is created and can't change afterwards.",
              key = "Mesh:getVertexCount",
              module = "lovr.graphics",
              related = {
                "Mesh:getVertexStride",
                "Mesh:getVertexFormat",
                "lovr.graphics.newMesh"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of vertices in the Mesh."
                    }
                  }
                }
              }
            },
            {
              name = "getVertexFormat",
              summary = "Get the vertex format of the Mesh.",
              description = "Returns the vertex format of the Mesh, which is a list of \"attributes\" that make up the data for each vertex (position, color, UV, etc.).",
              key = "Mesh:getVertexFormat",
              module = "lovr.graphics",
              notes = "If no vertex format is given when the Mesh is created, it will use a default format:\n\n    {\n      { 'VertexPosition', 'vec3', 0 },\n      { 'VertexNormal', 'vec3', 12 },\n      { 'VertexUV', 'vec2', 24 }\n    }\n\nThe name of the vertex attribute corresponds to an `in` input variable in a vertex shader.\n\nThere are a few built-in attributes that all shaders will understand and use by default:\n\n<table>\n  <thead>\n    <tr>\n      <td>Name</td>\n      <td>Description</td>\n    </tr>\n  </thead>\n  <tbody>\n    <tr>\n      <td><code>VertexPosition</code></td>\n      <td>The position of the vertex.</td>\n    </tr>\n    <tr>\n      <td><code>VertexNormal</code></td>\n      <td>The normal vector of the vertex.</td>\n    </tr>\n    <tr>\n      <td><code>VertexUV</code></td>\n      <td>The texture coordinate of the vertex.</td>\n    </tr>\n    <tr>\n      <td><code>VertexColor</code></td>\n      <td>The color of the vertex (linear color space).</td>\n    </tr>\n    <tr>\n      <td><code>VertexTangent</code></td>\n      <td>The tangent vector of the vertex.</td>\n    </tr>\n  </tbody> </table>\n\nSee the `Shaders` and `Meshes` guides for more info.",
              related = {
                "Mesh:getVertexCount",
                "Mesh:getVertexStride",
                "lovr.graphics.newMesh"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "format",
                      type = "table",
                      description = "The vertex format.",
                      table = {
                        {
                          name = "[][1]",
                          type = "string",
                          description = "The name of the attribute."
                        },
                        {
                          name = "[][2]",
                          type = "DataType",
                          description = "The type of the attribute."
                        },
                        {
                          name = "[][3]",
                          type = "number",
                          description = "The byte offset of the attribute."
                        }
                      }
                    }
                  }
                }
              }
            },
            {
              name = "getVertexStride",
              summary = "Get the size of each vertex in the Mesh.",
              description = "Returns the stride of the Mesh, which is the number of bytes used by each vertex.",
              key = "Mesh:getVertexStride",
              module = "lovr.graphics",
              related = {
                "Mesh:getVertexCount",
                "Mesh:getVertexFormat",
                "lovr.graphics.newMesh"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "stride",
                      type = "number",
                      description = "The stride of the Mesh, in bytes."
                    }
                  }
                }
              }
            },
            {
              name = "getVertices",
              summary = "Get the vertices in the Mesh.",
              description = "Returns the vertices in the Mesh.",
              key = "Mesh:getVertices",
              module = "lovr.graphics",
              notes = "> **This function will be very very slow if the storage mode of the Mesh is `gpu`, because the\n> data will be downloaded from VRAM.  A better option is to call `Buffer:newReadback` on the\n> Mesh's underlying vertex buffer (`Mesh:getVertexBuffer`), which will download in the\n> background instead of waiting for it to complete.**",
              related = {
                "Mesh:getVertexBuffer",
                "Mesh:getVertexFormat",
                "Mesh:getIndices",
                "Mesh:setIndices"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the first vertex to return.",
                      default = "1"
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "The number of vertices to return.  If nil, returns the \"rest\" of the vertices, based on the `index` argument.",
                      default = "nil"
                    }
                  },
                  returns = {
                    {
                      name = "vertices",
                      type = "table",
                      description = "A table of vertices.  Each vertex is a table of numbers for each vertex attribute, given by the vertex format of the Mesh."
                    }
                  }
                }
              }
            },
            {
              name = "setBoundingBox",
              summary = "Set or remove the bounding box of the Mesh.",
              description = "Sets or removes the axis-aligned bounding box of the Mesh.\n\nMeshes with the `cpu` storage mode can compute their bounding box automatically using `Mesh:computeBoundingBox`.\n\nPasses will use the bounding box of a Mesh to cull it against the cameras when `Pass:setViewCull` is enabled, which avoids rendering it when it's out of view.",
              key = "Mesh:setBoundingBox",
              module = "lovr.graphics",
              related = {
                "Mesh:computeBoundingBox",
                "Pass:setViewCull",
                "Collider:getAABB",
                "Shape:getAABB",
                "Model:getBoundingBox",
                "ModelData:getBoundingBox"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "minx",
                      type = "number",
                      description = "The minimum x coordinate of the bounding box."
                    },
                    {
                      name = "maxx",
                      type = "number",
                      description = "The maximum x coordinate of the bounding box."
                    },
                    {
                      name = "miny",
                      type = "number",
                      description = "The minimum y coordinate of the bounding box."
                    },
                    {
                      name = "maxy",
                      type = "number",
                      description = "The maximum y coordinate of the bounding box."
                    },
                    {
                      name = "minz",
                      type = "number",
                      description = "The minimum z coordinate of the bounding box."
                    },
                    {
                      name = "maxz",
                      type = "number",
                      description = "The maximum z coordinate of the bounding box."
                    }
                  },
                  returns = {}
                },
                {
                  description = "Remove the bounding box.",
                  arguments = {},
                  returns = {}
                }
              }
            },
            {
              name = "setDrawMode",
              summary = "Set the draw mode of the Mesh.",
              description = "Changes the `DrawMode` of the mesh, which controls how the vertices in the Mesh are connected together to create pixels.  The default is `triangles`.",
              key = "Mesh:setDrawMode",
              module = "lovr.graphics",
              related = {
                "Pass:setMeshMode"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "mode",
                      type = "DrawMode",
                      description = "The current draw mode."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setIndexBuffer",
              summary = "Set a Buffer for the Mesh to use for vertex indices.",
              description = "Sets a `Buffer` object the Mesh will use for vertex indices.\n\nThis can only be used if the Mesh uses the `gpu` storage mode.\n\nThe Buffer must have a single field with the `u16`, `u32`, `index16`, or `index32` type.",
              key = "Mesh:setIndexBuffer",
              module = "lovr.graphics",
              examples = {
                {
                  description = "Use an index buffer to draw a plane.",
                  code = "function lovr.load()\n  mesh = lovr.graphics.newMesh({\n    { -1,  1, 0 }, -- upper left\n    {  1,  1, 0 }, -- upper right\n    { -1, -1, 0 }, -- lower left\n    {  1, -1, 0 }, -- lower right\n  }, 'gpu')\n\n  -- 2 triangles\n  local indices = { 1,3,2, 2,3,4 }\n\n  local indexBuffer = lovr.graphics.newBuffer('index16', indices)\n  mesh:setIndexBuffer(indexBuffer)\nend\n\nfunction lovr.draw(pass)\n  pass:draw(mesh, 0, 1.7, -2)\nend"
                }
              },
              notes = "The index buffer stores a list of numbers where each number is the index of a vertex in the Mesh.  When drawing the Mesh, the data from the vertex corresponding to the index is used.  This can be used to reorder or reuse vertices, which uses less data than repeating a vertex multiple times in the Mesh.",
              related = {
                "Mesh:getIndices",
                "Mesh:setIndices",
                "Mesh:getVertexBuffer"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "buffer",
                      type = "Buffer",
                      description = "The index buffer."
                    }
                  }
                }
              }
            },
            {
              name = "setIndices",
              summary = "Set the vertex indices of the Mesh.",
              description = "Sets or clears the vertex indices of the Mesh.  Vertex indices define the list of triangles in the mesh.  They allow vertices to be reused multiple times without duplicating all their data, which can save a lot of memory and processing time if a vertex is used for multiple triangles.\n\nIf a Mesh doesn't have vertex indices, then the vertices are rendered in order.",
              key = "Mesh:setIndices",
              module = "lovr.graphics",
              related = {
                "Mesh:getIndexBuffer",
                "Mesh:setIndexBuffer",
                "Mesh:setVertices"
              },
              variants = {
                {
                  description = "Set vertex indices using a table.",
                  arguments = {
                    {
                      name = "t",
                      type = "table",
                      description = "A list of numbers (1-based)."
                    }
                  },
                  returns = {}
                },
                {
                  description = "Set vertex indices using a Blob.",
                  arguments = {
                    {
                      name = "blob",
                      type = "Blob",
                      description = "The Blob with index data."
                    },
                    {
                      name = "type",
                      type = "DataType",
                      description = "The type of index data in the Blob.  Must be `u16` or `u32`."
                    }
                  },
                  returns = {}
                },
                {
                  description = "Disable vertex indices.",
                  arguments = {},
                  returns = {}
                }
              }
            },
            {
              name = "setMaterial",
              summary = "Set a Material to use when drawing the Mesh.",
              description = "Sets a `Material` to use when drawing the Mesh.",
              key = "Mesh:setMaterial",
              module = "lovr.graphics",
              related = {
                "Pass:setMaterial",
                "Model:getMaterial",
                "lovr.graphics.newMaterial"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "material",
                      type = "Material",
                      description = "The material to use."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setVertices",
              summary = "Set vertices in the Mesh.",
              description = "Sets the data for vertices in the Mesh.",
              key = "Mesh:setVertices",
              module = "lovr.graphics",
              notes = "CPU meshes will write the data to CPU memory and upload any changes to the GPU before the Mesh is drawn.  GPU meshes don't store this CPU copy of the data, and will immediately upload the new vertex data to VRAM.  This means that multiple calls to this function might be slower on a `gpu` mesh.",
              related = {
                "Mesh:getVertexBuffer",
                "Mesh:getVertexFormat",
                "Mesh:getIndices",
                "Mesh:setIndices"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "vertices",
                      type = "table",
                      description = "A table of vertices, where each vertex is a table of numbers matching the vertex format of the Mesh."
                    },
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the first vertex to return.",
                      default = "1"
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "The number of vertices to return.  If nil, returns the \"rest\" of the vertices, based on the `index` argument.",
                      default = "nil"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "blob",
                      type = "Blob",
                      description = "A Blob containing binary vertex data."
                    },
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the first vertex to return.",
                      default = "1"
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "The number of vertices to return.  If nil, returns the \"rest\" of the vertices, based on the `index` argument.",
                      default = "nil"
                    }
                  },
                  returns = {}
                }
              }
            }
          }
        },
        {
          name = "Model",
          summary = "A 3D model.",
          description = "Models are 3D model assets loaded from files.  Currently, OBJ, glTF, and binary STL files are supported.\n\nA model can be drawn using `Pass:draw`.\n\nThe raw CPU data for a model is held in a `ModelData` object, which can be loaded on threads or reused for multiple Model instances.\n\nModels have a hierarchy of nodes which can have their transforms modified.  Meshes are attached to these nodes.  The same mesh can be attached to multiple nodes, allowing it to be drawn multiple times while only storing a single copy of its data.\n\nModels can have animations.  Animations have keyframes which affect the transforms of nodes. Right now each model can only be drawn with a single animated pose per frame.\n\nModels can have materials, which are collections of properties and textures that define how its surface is affected by lighting.  Each mesh in the model can use a single material.",
          key = "Model",
          module = "lovr.graphics",
          constructors = {
            "lovr.graphics.newModel",
            "lovr.headset.newModel"
          },
          methods = {
            {
              name = "animate",
              summary = "Animate the Model.",
              description = "Animates a Model by setting or blending the transforms of nodes using data stored in the keyframes of an animation.\n\nThe animation from the model file is evaluated at the timestamp, resulting in a set of node properties.  These properties are then applied to the nodes in the model, using an optional blend factor.  If the animation doesn't have keyframes that target a given node, the node will remain unchanged.",
              key = "Model:animate",
              module = "lovr.graphics",
              notes = "If the timestamp is larger than the duration of the animation, it will wrap back around to zero, so looping an animation doesn't require using the modulo operator.\n\nTo change the speed of the animation, multiply the timestamp by a speed factor.\n\nFor each animated property in the animation, if the timestamp used for the animation is less than the timestamp of the first keyframe, the data of the first keyframe will be used.\n\nThis function can be called multiple times to layer and blend animations.  The model joints will be drawn in the final resulting pose.\n\n`Model:resetNodeTransforms` can be used to reset the model nodes to their initial transforms, which is helpful to ensure animating starts from a clean slate.",
              related = {
                "Model:resetNodeTransforms",
                "Model:getAnimationCount",
                "Model:getAnimationName",
                "Model:getAnimationDuration",
                "Model:getNodePosition",
                "Model:setNodePosition",
                "Model:getNodeOrientation",
                "Model:setNodeOrientation",
                "Model:getNodeScale",
                "Model:setNodeScale",
                "Model:getNodeTransform",
                "Model:setNodeTransform"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of an animation in the model file."
                    },
                    {
                      name = "time",
                      type = "number",
                      description = "The timestamp to evaluate the keyframes at, in seconds."
                    },
                    {
                      name = "blend",
                      type = "number",
                      description = "How much of the animation's pose to blend into the nodes, from 0 to 1.",
                      default = "1.0"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of an animation in the model file."
                    },
                    {
                      name = "time",
                      type = "number",
                      description = "The timestamp to evaluate the keyframes at, in seconds."
                    },
                    {
                      name = "blend",
                      type = "number",
                      description = "How much of the animation's pose to blend into the nodes, from 0 to 1.",
                      default = "1.0"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "clone",
              summary = "Return a lightweight copy of the Model with its own animation state.",
              description = "Returns a lightweight copy of a Model.  Most of the data will be shared between the two copies of the model, like the materials, textures, and metadata.  However, the clone has its own set of node transforms, allowing it to be animated separately from its parent.  This allows a single model to be rendered in multiple different animation poses in a frame.",
              key = "Model:clone",
              module = "lovr.graphics",
              notes = "The node transforms of the clone will be reset to their initial setup poses.",
              related = {
                "lovr.graphics.newModel"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "model",
                      type = "Model",
                      description = "A genetically identical copy of the Model."
                    }
                  }
                }
              }
            },
            {
              name = "getAnimationCount",
              summary = "Get the number of animations in the Model.",
              description = "Returns the number of animations in the Model.",
              key = "Model:getAnimationCount",
              module = "lovr.graphics",
              related = {
                "Model:getAnimationName",
                "Model:getAnimationDuration",
                "Model:animate"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of animations in the Model."
                    }
                  }
                }
              }
            },
            {
              name = "getAnimationDuration",
              summary = "Get the duration of an animation in the Model.",
              description = "Returns the duration of an animation in the Model, in seconds.",
              key = "Model:getAnimationDuration",
              module = "lovr.graphics",
              notes = "The duration of an animation is calculated as the largest timestamp of all of its keyframes.",
              related = {
                "Model:getAnimationCount",
                "Model:getAnimationName",
                "Model:animate"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The animation index."
                    }
                  },
                  returns = {
                    {
                      name = "duration",
                      type = "number",
                      description = "The duration of the animation, in seconds."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the animation."
                    }
                  },
                  returns = {
                    {
                      name = "duration",
                      type = "number",
                      description = "The duration of the animation, in seconds."
                    }
                  }
                }
              }
            },
            {
              name = "getAnimationName",
              summary = "Get the name of an animation in the Model.",
              description = "Returns the name of an animation in the Model.",
              key = "Model:getAnimationName",
              module = "lovr.graphics",
              related = {
                "Model:getAnimationCount",
                "Model:getAnimationDuration"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of an animation."
                    }
                  },
                  returns = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the animation."
                    }
                  }
                }
              }
            },
            {
              name = "getBlendShapeCount",
              summary = "Get the number of blend shapes in the model.",
              description = "Returns the number of blend shapes in the model.",
              key = "Model:getBlendShapeCount",
              module = "lovr.graphics",
              related = {
                "Model:getBlendShapeName",
                "ModelData:getBlendShapeCount"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of blend shapes in the model."
                    }
                  }
                }
              }
            },
            {
              name = "getBlendShapeName",
              summary = "Get the name of a blend shape in the model.",
              description = "Returns the name of a blend shape in the model.",
              key = "Model:getBlendShapeName",
              module = "lovr.graphics",
              notes = "This function will throw an error if the blend shape index is invalid.",
              related = {
                "Model:getBlendShapeCount",
                "ModelData:getBlendShapeName"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of a blend shape."
                    }
                  },
                  returns = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the blend shape."
                    }
                  }
                }
              }
            },
            {
              name = "getBlendShapeWeight",
              summary = "Get the weight of a blend shape.",
              description = "Returns the weight of a blend shape.  A blend shape contains offset values for the vertices of one of the meshes in a Model.  Whenever the Model is drawn, the offsets are multiplied by the weight of the blend shape, allowing for smooth blending between different meshes.  A weight of zero won't apply any displacement and will skip processing of the blend shape.",
              key = "Model:getBlendShapeWeight",
              module = "lovr.graphics",
              notes = "The initial weights are declared in the model file.\n\nWeights can be any number, but usually they're kept between 0 and 1.\n\nThis function will throw an error if the blend shape name or index doesn't exist.",
              related = {
                "Model:getBlendShapeCount",
                "Model:getBlendShapeName"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of a blend shape."
                    }
                  },
                  returns = {
                    {
                      name = "weight",
                      type = "number",
                      description = "The weight of the blend shape."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of a blend shape."
                    }
                  },
                  returns = {
                    {
                      name = "weight",
                      type = "number",
                      description = "The weight of the blend shape."
                    }
                  }
                }
              }
            },
            {
              name = "getBoundingBox",
              summary = "Get the bounding box of the Model.",
              description = "Returns the 6 values of the Model's axis-aligned bounding box.",
              key = "Model:getBoundingBox",
              module = "lovr.graphics",
              related = {
                "Model:getWidth",
                "Model:getHeight",
                "Model:getDepth",
                "Model:getDimensions",
                "Model:getCenter",
                "Model:getBoundingSphere",
                "ModelData:getBoundingBox",
                "Collider:getAABB"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "minx",
                      type = "number",
                      description = "The minimum x coordinate of the vertices in the Model."
                    },
                    {
                      name = "maxx",
                      type = "number",
                      description = "The maximum x coordinate of the vertices in the Model."
                    },
                    {
                      name = "miny",
                      type = "number",
                      description = "The minimum y coordinate of the vertices in the Model."
                    },
                    {
                      name = "maxy",
                      type = "number",
                      description = "The maximum y coordinate of the vertices in the Model."
                    },
                    {
                      name = "minz",
                      type = "number",
                      description = "The minimum z coordinate of the vertices in the Model."
                    },
                    {
                      name = "maxz",
                      type = "number",
                      description = "The maximum z coordinate of the vertices in the Model."
                    }
                  }
                }
              }
            },
            {
              name = "getBoundingSphere",
              summary = "Get the bounding sphere of the Model.",
              description = "Returns a sphere approximately enclosing the vertices in the Model.",
              key = "Model:getBoundingSphere",
              module = "lovr.graphics",
              related = {
                "Model:getWidth",
                "Model:getHeight",
                "Model:getDepth",
                "Model:getDimensions",
                "Model:getCenter",
                "Model:getBoundingBox",
                "ModelData:getBoundingSphere"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate of the position of the sphere."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate of the position of the sphere."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate of the position of the sphere."
                    },
                    {
                      name = "radius",
                      type = "number",
                      description = "The radius of the bounding sphere."
                    }
                  }
                }
              }
            },
            {
              name = "getCenter",
              summary = "Get the center of the Model's bounding box.",
              description = "Returns the center of the Model's axis-aligned bounding box, relative to the Model's origin.",
              key = "Model:getCenter",
              module = "lovr.graphics",
              related = {
                "Model:getWidth",
                "Model:getHeight",
                "Model:getDepth",
                "Model:getDimensions",
                "Model:getBoundingBox",
                "ModelData:getCenter"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x offset of the center of the bounding box."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y offset of the center of the bounding box."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z offset of the center of the bounding box."
                    }
                  }
                }
              }
            },
            {
              name = "getData",
              summary = "Get the ModelData backing the Model.",
              description = "Returns the ModelData this Model was created from.",
              key = "Model:getData",
              module = "lovr.graphics",
              related = {
                "lovr.data.newModelData"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "data",
                      type = "ModelData",
                      description = "The ModelData."
                    }
                  }
                }
              }
            },
            {
              name = "getDepth",
              summary = "Get the depth of the Model.",
              description = "Returns the depth of the Model, computed from its axis-aligned bounding box.",
              key = "Model:getDepth",
              module = "lovr.graphics",
              related = {
                "Model:getWidth",
                "Model:getHeight",
                "Model:getDimensions",
                "Model:getCenter",
                "Model:getBoundingBox",
                "ModelData:getDepth"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "depth",
                      type = "number",
                      description = "The depth of the Model."
                    }
                  }
                }
              }
            },
            {
              name = "getDimensions",
              summary = "Get the dimensions of the Model.",
              description = "Returns the width, height, and depth of the Model, computed from its axis-aligned bounding box.",
              key = "Model:getDimensions",
              module = "lovr.graphics",
              related = {
                "Model:getWidth",
                "Model:getHeight",
                "Model:getDepth",
                "Model:getCenter",
                "Model:getBoundingBox",
                "ModelData:getDimensions"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "width",
                      type = "number",
                      description = "The width of the Model."
                    },
                    {
                      name = "height",
                      type = "number",
                      description = "The height of the Model."
                    },
                    {
                      name = "depth",
                      type = "number",
                      description = "The depth of the Model."
                    }
                  }
                }
              }
            },
            {
              name = "getHeight",
              summary = "Get the height of the Model.",
              description = "Returns the height of the Model, computed from its axis-aligned bounding box.",
              key = "Model:getHeight",
              module = "lovr.graphics",
              related = {
                "Model:getWidth",
                "Model:getDepth",
                "Model:getDimensions",
                "Model:getCenter",
                "Model:getBoundingBox",
                "ModelData:getHeight"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "height",
                      type = "number",
                      description = "The height of the Model."
                    }
                  }
                }
              }
            },
            {
              name = "getIndexBuffer",
              summary = "Get a Buffer containing the triangle indices in the Model.",
              description = "Returns the index buffer used by the Model.  The index buffer describes the order used to draw the vertices in each mesh.",
              key = "Model:getIndexBuffer",
              module = "lovr.graphics",
              related = {
                "Model:getVertexBuffer"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "buffer",
                      type = "Buffer",
                      description = "The index buffer."
                    }
                  }
                }
              }
            },
            {
              name = "getMaterial",
              summary = "Get a Material from the Model.",
              description = "Returns a `Material` loaded from the Model.",
              key = "Model:getMaterial",
              module = "lovr.graphics",
              related = {
                "Model:getMaterialCount",
                "Model:getMaterialName",
                "Model:getNodeDraw"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the Material to return."
                    }
                  },
                  returns = {
                    {
                      name = "material",
                      type = "Material",
                      description = "The material."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the Material to return."
                    }
                  },
                  returns = {
                    {
                      name = "material",
                      type = "Material",
                      description = "The material."
                    }
                  }
                }
              }
            },
            {
              name = "getMaterialCount",
              summary = "Get the number of materials in the Model.",
              description = "Returns the number of materials in the Model.",
              key = "Model:getMaterialCount",
              module = "lovr.graphics",
              related = {
                "Model:getMaterialName",
                "Model:getMaterial"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of materials in the Model."
                    }
                  }
                }
              }
            },
            {
              name = "getMaterialName",
              summary = "Get the name of a material in the Model.",
              description = "Returns the name of a material in the Model.",
              key = "Model:getMaterialName",
              module = "lovr.graphics",
              related = {
                "Model:getMaterialCount",
                "Model:getMaterial"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of a material."
                    }
                  },
                  returns = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the material."
                    }
                  }
                }
              }
            },
            {
              name = "getMetadata",
              summary = "Get extra information from the model file.",
              description = "Returns extra information stored in the model file.  Currently this is only implemented for glTF models and returns the JSON string from the glTF or glb file.  The metadata can be used to get application-specific data or add support for glTF extensions not supported by LÖVR.",
              key = "Model:getMetadata",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "metadata",
                      type = "string",
                      description = "The metadata from the model file."
                    }
                  }
                }
              }
            },
            {
              name = "getNodeChildren",
              summary = "Get the children of a node.",
              description = "Given a parent node, this function returns a table with the indices of its children.",
              key = "Model:getNodeChildren",
              module = "lovr.graphics",
              notes = "If the node does not have any children, this function returns an empty table.",
              related = {
                "Model:getNodeParent",
                "Model:getRootNode"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the parent node."
                    }
                  },
                  returns = {
                    {
                      name = "children",
                      type = "table",
                      description = "A table containing a node index for each child of the node."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the parent node."
                    }
                  },
                  returns = {
                    {
                      name = "children",
                      type = "table",
                      description = "A table containing a node index for each child of the node."
                    }
                  }
                }
              }
            },
            {
              name = "getNodeCount",
              summary = "Get the number of nodes in the model.",
              description = "Returns the number of nodes in the model.",
              key = "Model:getNodeCount",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of nodes in the model."
                    }
                  }
                }
              }
            },
            {
              name = "getNodeDraw",
              summary = "Get the information needed to draw one mesh in the model.",
              description = "Returns the draw mode, material, and vertex range of a mesh in the model.",
              key = "Model:getNodeDraw",
              module = "lovr.graphics",
              examples = {
                {
                  code = "function lovr.load()\n  local m = lovr.graphics.newModel('enraged-gorilla.gltf')\n\n  model = {\n    object = m,\n    data = m:getData(),\n    vertices = m:getVertexBuffer(),\n    indices = m:getIndexBuffer()\n  }\nend\n\nlocal function drawNode(model, pass, i)\n  for j = 1, model.object:getNodeDrawCount(i) do\n    local mode, material, start, count, base = model.object:getNodeDraw(i, j)\n    local transform = mat4(model.object:getNodeTransform(i))\n\n    pass:setMeshMode(mode)\n    pass:setMaterial(material)\n\n    if base then\n      pass:mesh(model.vertices, model.indices, transform, start, count, 1, base)\n    else\n      pass:mesh(model.vertices, transform, start, count)\n    end\n  end\n\n  for _, index in ipairs(model.data:getNodeChildren(i)) do\n    drawNode(model, pass, index)\n  end\nend\n\nfunction lovr.draw(pass)\n  drawNode(model, pass, model.data:getRootNode())\nend"
                }
              },
              related = {
                "Pass:setMeshMode",
                "Pass:setMaterial",
                "Pass:mesh",
                "Model:getVertexBuffer",
                "Model:getIndexBuffer"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "node",
                      type = "number",
                      description = "The index of the node."
                    },
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the draw."
                    }
                  },
                  returns = {
                    {
                      name = "mode",
                      type = "DrawMode",
                      description = "Whether the vertices are points, lines, or triangles."
                    },
                    {
                      name = "material",
                      type = "Material",
                      description = "The Material used by the draw."
                    },
                    {
                      name = "start",
                      type = "number",
                      description = "The offset of the first vertex in the draw."
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "The number of vertices in the draw."
                    },
                    {
                      name = "base",
                      type = "number",
                      description = "The base vertex of the draw (added to each instance value), or nil if the draw does not use an index buffer."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the node."
                    },
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the draw."
                    }
                  },
                  returns = {
                    {
                      name = "mode",
                      type = "DrawMode",
                      description = "Whether the vertices are points, lines, or triangles."
                    },
                    {
                      name = "material",
                      type = "Material",
                      description = "The Material used by the draw."
                    },
                    {
                      name = "start",
                      type = "number",
                      description = "The offset of the first vertex in the draw."
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "The number of vertices in the draw."
                    },
                    {
                      name = "base",
                      type = "number",
                      description = "The base vertex of the draw (added to each instance value), or nil if the draw does not use an index buffer."
                    }
                  }
                }
              }
            },
            {
              name = "getNodeDrawCount",
              summary = "Get the number of meshes attached to a node.",
              description = "Returns the number of meshes attached to a node.  Each mesh is drawn individually.",
              key = "Model:getNodeDrawCount",
              module = "lovr.graphics",
              related = {
                "ModelData:getMeshCount",
                "Model:getNodeDraw"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of a node."
                    }
                  },
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of draws in the node."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of a node."
                    }
                  },
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of draws in the node."
                    }
                  }
                }
              }
            },
            {
              name = "getNodeName",
              summary = "Get the name of a node in the Model.",
              description = "Returns the name of a node.",
              key = "Model:getNodeName",
              module = "lovr.graphics",
              related = {
                "Model:getNodeCount",
                "Model:getAnimationName",
                "Model:getMaterialName"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the node."
                    }
                  },
                  returns = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the node."
                    }
                  }
                }
              }
            },
            {
              name = "getNodeOrientation",
              summary = "Get the orientation of a node.",
              description = "Returns the orientation of a node.",
              key = "Model:getNodeOrientation",
              module = "lovr.graphics",
              related = {
                "Model:getNodePosition",
                "Model:setNodePosition",
                "Model:getNodeScale",
                "Model:setNodeScale",
                "Model:getNodePose",
                "Model:setNodePose",
                "Model:getNodeTransform",
                "Model:setNodeTransform",
                "Model:animate"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the node."
                    },
                    {
                      name = "origin",
                      type = "OriginType",
                      description = "Whether the orientation should be returned relative to the root node or the node's parent.",
                      default = "'root'"
                    }
                  },
                  returns = {
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the node is rotated around its axis of rotation."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the node."
                    },
                    {
                      name = "origin",
                      type = "OriginType",
                      description = "Whether the orientation should be returned relative to the root node or the node's parent.",
                      default = "'root'"
                    }
                  },
                  returns = {
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the node is rotated around its axis of rotation."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    }
                  }
                }
              }
            },
            {
              name = "getNodeParent",
              summary = "Get the parent of a node.",
              description = "Given a child node, this function returns the index of its parent.",
              key = "Model:getNodeParent",
              module = "lovr.graphics",
              related = {
                "Model:getNodeChildren",
                "Model:getRootNode"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the child node."
                    }
                  },
                  returns = {
                    {
                      name = "parent",
                      type = "number",
                      description = "The index of the parent."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the child node."
                    }
                  },
                  returns = {
                    {
                      name = "parent",
                      type = "number",
                      description = "The index of the parent."
                    }
                  }
                }
              }
            },
            {
              name = "getNodePose",
              summary = "Get the pose of a node.",
              description = "Returns the pose (position and orientation) of a node.",
              key = "Model:getNodePose",
              module = "lovr.graphics",
              related = {
                "Model:getNodePosition",
                "Model:setNodePosition",
                "Model:getNodeOrientation",
                "Model:setNodeOrientation",
                "Model:getNodeScale",
                "Model:setNodeScale",
                "Model:getNodeTransform",
                "Model:setNodeTransform",
                "Model:animate"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of a node."
                    },
                    {
                      name = "origin",
                      type = "OriginType",
                      description = "Whether the pose should be returned relative to the root node or the node's parent.",
                      default = "'root'"
                    }
                  },
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x position of the node."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y position of the node."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z position of the node."
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the node is rotated around its axis of rotation."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of a node."
                    },
                    {
                      name = "origin",
                      type = "OriginType",
                      description = "Whether the pose should be returned relative to the root node or the node's parent.",
                      default = "'root'"
                    }
                  },
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x position of the node."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y position of the node."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z position of the node."
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the node is rotated around its axis of rotation."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    }
                  }
                }
              }
            },
            {
              name = "getNodePosition",
              summary = "Get the position of a node.",
              description = "Returns the position of a node.",
              key = "Model:getNodePosition",
              module = "lovr.graphics",
              related = {
                "Model:getNodeOrientation",
                "Model:setNodeOrientation",
                "Model:getNodeScale",
                "Model:setNodeScale",
                "Model:getNodePose",
                "Model:setNodePose",
                "Model:getNodeTransform",
                "Model:setNodeTransform",
                "Model:animate"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the node."
                    },
                    {
                      name = "space",
                      type = "OriginType",
                      description = "Whether the position should be returned relative to the root node or the node's parent.",
                      default = "'root'"
                    }
                  },
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the node."
                    },
                    {
                      name = "space",
                      type = "OriginType",
                      description = "Whether the position should be returned relative to the root node or the node's parent.",
                      default = "'root'"
                    }
                  },
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate."
                    }
                  }
                }
              }
            },
            {
              name = "getNodeScale",
              summary = "Get the scale of a node.",
              description = "Returns the scale of a node.",
              key = "Model:getNodeScale",
              module = "lovr.graphics",
              related = {
                "Model:getNodePosition",
                "Model:setNodePosition",
                "Model:getNodeOrientation",
                "Model:setNodeOrientation",
                "Model:getNodePose",
                "Model:setNodePose",
                "Model:getNodeTransform",
                "Model:setNodeTransform",
                "Model:animate"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the node."
                    },
                    {
                      name = "origin",
                      type = "OriginType",
                      description = "Whether the scale should be returned relative to the root node or the node's parent.",
                      default = "'root'"
                    }
                  },
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x scale."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y scale."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z scale."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the node."
                    },
                    {
                      name = "origin",
                      type = "OriginType",
                      description = "Whether the scale should be returned relative to the root node or the node's parent.",
                      default = "'root'"
                    }
                  },
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x scale."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y scale."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z scale."
                    }
                  }
                }
              }
            },
            {
              name = "getNodeTransform",
              summary = "Get the transform of a node.",
              description = "Returns the transform (position, scale, and rotation) of a node.",
              key = "Model:getNodeTransform",
              module = "lovr.graphics",
              related = {
                "Model:getNodePosition",
                "Model:setNodePosition",
                "Model:getNodeOrientation",
                "Model:setNodeOrientation",
                "Model:getNodeScale",
                "Model:setNodeScale",
                "Model:getNodePose",
                "Model:setNodePose",
                "Model:animate"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of a node."
                    },
                    {
                      name = "origin",
                      type = "OriginType",
                      description = "Whether the transform should be returned relative to the root node or the node's parent.",
                      default = "'root'"
                    }
                  },
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x position of the node."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y position of the node."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z position of the node."
                    },
                    {
                      name = "sx",
                      type = "number",
                      description = "The x scale of the node."
                    },
                    {
                      name = "sy",
                      type = "number",
                      description = "The y scale of the node."
                    },
                    {
                      name = "sz",
                      type = "number",
                      description = "The z scale of the node."
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the node is rotated around its axis of rotation."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of a node."
                    },
                    {
                      name = "origin",
                      type = "OriginType",
                      description = "Whether the transform should be returned relative to the root node or the node's parent.",
                      default = "'root'"
                    }
                  },
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x position of the node."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y position of the node."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z position of the node."
                    },
                    {
                      name = "sx",
                      type = "number",
                      description = "The x scale of the node."
                    },
                    {
                      name = "sy",
                      type = "number",
                      description = "The y scale of the node."
                    },
                    {
                      name = "sz",
                      type = "number",
                      description = "The z scale of the node."
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the node is rotated around its axis of rotation."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    }
                  }
                }
              }
            },
            {
              name = "getRootNode",
              summary = "Get the index of the root node.",
              description = "Returns the index of the model's root node.",
              key = "Model:getRootNode",
              module = "lovr.graphics",
              related = {
                "Model:getNodeCount",
                "Model:getNodeParent"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "root",
                      type = "number",
                      description = "The index of the root node."
                    }
                  }
                }
              }
            },
            {
              name = "getTexture",
              summary = "Get one of the textures in the Model.",
              description = "Returns one of the textures in the Model.",
              key = "Model:getTexture",
              module = "lovr.graphics",
              related = {
                "Model:getTextureCount",
                "Model:getMaterial"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the texture to get."
                    }
                  },
                  returns = {
                    {
                      name = "texture",
                      type = "Texture",
                      description = "The texture."
                    }
                  }
                }
              }
            },
            {
              name = "getTextureCount",
              summary = "Get the number of textures in the Model.",
              description = "Returns the number of textures in the Model.",
              key = "Model:getTextureCount",
              module = "lovr.graphics",
              related = {
                "Model:getTexture"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of textures in the Model."
                    }
                  }
                }
              }
            },
            {
              name = "getTriangleCount",
              summary = "Get the total number of triangles in the Model.",
              description = "Returns the total number of triangles in the Model.",
              key = "Model:getTriangleCount",
              module = "lovr.graphics",
              notes = "This isn't always related to the length of the vertex buffer, since a mesh in the Model could be drawn by multiple nodes.",
              related = {
                "Model:getTriangles",
                "Model:getVertexCount",
                "ModelData:getTriangleCount"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The total number of triangles in the Model."
                    }
                  }
                }
              }
            },
            {
              name = "getTriangles",
              summary = "Get all the triangles in the Model.",
              description = "Returns 2 tables containing mesh data for the Model.\n\nThe first table is a list of vertex positions and contains 3 numbers for the x, y, and z coordinate of each vertex.  The second table is a list of triangles and contains 1-based indices into the first table representing the first, second, and third vertices that make up each triangle.\n\nThe vertex positions will be affected by node transforms.",
              key = "Model:getTriangles",
              module = "lovr.graphics",
              notes = "After this function is called on a Model once, the result is cached (in its ModelData).",
              related = {
                "Model:getTriangleCount",
                "Model:getVertexCount",
                "ModelData:getTriangles"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "vertices",
                      type = "table",
                      description = "The triangle vertex positions, returned as a flat (non-nested) table of numbers.  The position of each vertex is given as an x, y, and z coordinate."
                    },
                    {
                      name = "indices",
                      type = "table",
                      description = "The vertex indices.  Every 3 indices describes a triangle."
                    }
                  }
                }
              }
            },
            {
              name = "getVertexBuffer",
              summary = "Get a Buffer containing the vertices in the Model.",
              description = "Returns a `Buffer` that holds the vertices of all of the meshes in the Model.",
              key = "Model:getVertexBuffer",
              module = "lovr.graphics",
              related = {
                "Model:getIndexBuffer"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "buffer",
                      type = "Buffer",
                      description = "The vertex buffer."
                    }
                  }
                }
              }
            },
            {
              name = "getVertexCount",
              summary = "Get the total vertex count of the Model.",
              description = "Returns the total vertex count of the Model.",
              key = "Model:getVertexCount",
              module = "lovr.graphics",
              notes = "This isn't always the same as the length of the vertex buffer, since a mesh in the Model could be drawn by multiple nodes.",
              related = {
                "Model:getTriangles",
                "Model:getTriangleCount",
                "ModelData:getVertexCount"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The total number of vertices."
                    }
                  }
                }
              }
            },
            {
              name = "getWidth",
              summary = "Get the width of the Model.",
              description = "Returns the width of the Model, computed from its axis-aligned bounding box.",
              key = "Model:getWidth",
              module = "lovr.graphics",
              related = {
                "Model:getHeight",
                "Model:getDepth",
                "Model:getDimensions",
                "Model:getCenter",
                "Model:getBoundingBox",
                "ModelData:getWidth"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "width",
                      type = "number",
                      description = "The width of the Model."
                    }
                  }
                }
              }
            },
            {
              name = "hasJoints",
              summary = "Check if the Model uses joints for skeletal animation.",
              description = "Returns whether the Model has any skeletal animations.",
              key = "Model:hasJoints",
              module = "lovr.graphics",
              notes = "This will return when there's at least one skin in the model, as returned by `ModelData:getSkinCount`.\n\nEven if this function returns true, the model could still have non-skeletal animations.\n\nRight now a model can only be drawn with one skeletal pose per frame.",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "jointed",
                      type = "boolean",
                      description = "Whether the animation uses joint nodes for skeletal animation."
                    }
                  }
                }
              }
            },
            {
              name = "resetNodeTransforms",
              summary = "Reset node transforms.",
              description = "Resets node transforms to the original ones defined in the model file.",
              key = "Model:resetNodeTransforms",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {},
                  returns = {}
                }
              }
            },
            {
              name = "setBlendShapeWeight",
              summary = "Set the weight of a blend shape.",
              description = "Sets the weight of a blend shape.  A blend shape contains offset values for the vertices of one of the meshes in a Model.  Whenever the Model is drawn, the offsets are multiplied by the weight of the blend shape, allowing for smooth blending between different meshes.  A weight of zero won't apply any displacement and will skip processing of the blend shape.",
              key = "Model:setBlendShapeWeight",
              module = "lovr.graphics",
              notes = "The initial weights are declared in the model file.\n\nWeights can be any number, but usually they're kept between 0 and 1.\n\nThis function will throw an error if the blend shape name or index doesn't exist.",
              related = {
                "Model:getBlendShapeCount",
                "Model:getBlendShapeName"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of a blend shape."
                    },
                    {
                      name = "weight",
                      type = "number",
                      description = "The new weight for the blend shape."
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of a blend shape."
                    },
                    {
                      name = "weight",
                      type = "number",
                      description = "The new weight for the blend shape."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setNodeOrientation",
              summary = "Set or blend the orientation of a node.",
              description = "Sets or blends the orientation of a node to a new orientation.  This sets the local orientation of the node, relative to its parent.",
              key = "Model:setNodeOrientation",
              module = "lovr.graphics",
              related = {
                "Model:getNodePosition",
                "Model:setNodePosition",
                "Model:getNodeScale",
                "Model:setNodeScale",
                "Model:getNodePose",
                "Model:setNodePose",
                "Model:getNodeTransform",
                "Model:setNodeTransform",
                "Model:animate"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the node."
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the node should be rotated around its rotation axis."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    },
                    {
                      name = "blend",
                      type = "number",
                      description = "A number from 0 to 1 indicating how much of the target orientation to blend in.  A value of 0 will not change the node's orientation at all, whereas 1 will fully blend to the target orientation.",
                      default = "1.0"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the node."
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the node should be rotated around its rotation axis."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    },
                    {
                      name = "blend",
                      type = "number",
                      description = "A number from 0 to 1 indicating how much of the target orientation to blend in.  A value of 0 will not change the node's orientation at all, whereas 1 will fully blend to the target orientation.",
                      default = "1.0"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the node."
                    },
                    {
                      name = "orientation",
                      type = "Quat",
                      description = "The orientation."
                    },
                    {
                      name = "blend",
                      type = "number",
                      description = "A number from 0 to 1 indicating how much of the target orientation to blend in.  A value of 0 will not change the node's orientation at all, whereas 1 will fully blend to the target orientation.",
                      default = "1.0"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the node."
                    },
                    {
                      name = "orientation",
                      type = "Quat",
                      description = "The orientation."
                    },
                    {
                      name = "blend",
                      type = "number",
                      description = "A number from 0 to 1 indicating how much of the target orientation to blend in.  A value of 0 will not change the node's orientation at all, whereas 1 will fully blend to the target orientation.",
                      default = "1.0"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setNodePose",
              summary = "Set or blend the pose of a node.",
              description = "Sets or blends the pose (position and orientation) of a node to a new pose.  This sets the local pose of the node, relative to its parent.  The scale will remain unchanged.",
              key = "Model:setNodePose",
              module = "lovr.graphics",
              related = {
                "Model:getNodePosition",
                "Model:setNodePosition",
                "Model:getNodeOrientation",
                "Model:setNodeOrientation",
                "Model:getNodeScale",
                "Model:setNodeScale",
                "Model:getNodeTransform",
                "Model:setNodeTransform",
                "Model:animate"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the node."
                    },
                    {
                      name = "x",
                      type = "number",
                      description = "The x component of the position."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y component of the position."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z component of the position."
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the node should be rotated around its rotation axis."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    },
                    {
                      name = "blend",
                      type = "number",
                      description = "A number from 0 to 1 indicating how much of the target pose to blend in.  A value of 0 will not change the node's pose at all, whereas 1 will fully blend to the target pose.",
                      default = "1.0"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the node."
                    },
                    {
                      name = "x",
                      type = "number",
                      description = "The x component of the position."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y component of the position."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z component of the position."
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the node should be rotated around its rotation axis."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    },
                    {
                      name = "blend",
                      type = "number",
                      description = "A number from 0 to 1 indicating how much of the target pose to blend in.  A value of 0 will not change the node's pose at all, whereas 1 will fully blend to the target pose.",
                      default = "1.0"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the node."
                    },
                    {
                      name = "position",
                      type = "Vec3",
                      description = "The target position.  Can also be provided as 3 numbers."
                    },
                    {
                      name = "orientation",
                      type = "Quat",
                      description = "The target orientation.  Can also be provided as 4 numbers in angle-axis form."
                    },
                    {
                      name = "blend",
                      type = "number",
                      description = "A number from 0 to 1 indicating how much of the target pose to blend in.  A value of 0 will not change the node's pose at all, whereas 1 will fully blend to the target pose.",
                      default = "1.0"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the node."
                    },
                    {
                      name = "position",
                      type = "Vec3",
                      description = "The target position.  Can also be provided as 3 numbers."
                    },
                    {
                      name = "orientation",
                      type = "Quat",
                      description = "The target orientation.  Can also be provided as 4 numbers in angle-axis form."
                    },
                    {
                      name = "blend",
                      type = "number",
                      description = "A number from 0 to 1 indicating how much of the target pose to blend in.  A value of 0 will not change the node's pose at all, whereas 1 will fully blend to the target pose.",
                      default = "1.0"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setNodePosition",
              summary = "Set or blend the position of a node.",
              description = "Sets or blends the position of a node.  This sets the local position of the node, relative to its parent.",
              key = "Model:setNodePosition",
              module = "lovr.graphics",
              related = {
                "Model:getNodeOrientation",
                "Model:setNodeOrientation",
                "Model:getNodeScale",
                "Model:setNodeScale",
                "Model:getNodePose",
                "Model:setNodePose",
                "Model:getNodeTransform",
                "Model:setNodeTransform",
                "Model:animate"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the node."
                    },
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate of the new position."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate of the new position."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate of the new position."
                    },
                    {
                      name = "blend",
                      type = "number",
                      description = "A number from 0 to 1 indicating how much of the new position to blend in.  A value of 0 will not change the node's position at all, whereas 1 will fully blend to the target position.",
                      default = "1.0"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the node."
                    },
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate of the new position."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate of the new position."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate of the new position."
                    },
                    {
                      name = "blend",
                      type = "number",
                      description = "A number from 0 to 1 indicating how much of the new position to blend in.  A value of 0 will not change the node's position at all, whereas 1 will fully blend to the target position.",
                      default = "1.0"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the node."
                    },
                    {
                      name = "position",
                      type = "Vec3",
                      description = "The new position."
                    },
                    {
                      name = "blend",
                      type = "number",
                      description = "A number from 0 to 1 indicating how much of the new position to blend in.  A value of 0 will not change the node's position at all, whereas 1 will fully blend to the target position.",
                      default = "1.0"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the node."
                    },
                    {
                      name = "position",
                      type = "Vec3",
                      description = "The new position."
                    },
                    {
                      name = "blend",
                      type = "number",
                      description = "A number from 0 to 1 indicating how much of the new position to blend in.  A value of 0 will not change the node's position at all, whereas 1 will fully blend to the target position.",
                      default = "1.0"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setNodeScale",
              summary = "Set or blend the scale of a node.",
              description = "Sets or blends the scale of a node to a new scale.  This sets the local scale of the node, relative to its parent.",
              key = "Model:setNodeScale",
              module = "lovr.graphics",
              notes = "For best results when animating, it's recommended to keep the 3 scale components the same.",
              related = {
                "Model:getNodePosition",
                "Model:setNodePosition",
                "Model:getNodeOrientation",
                "Model:setNodeOrientation",
                "Model:getNodePose",
                "Model:setNodePose",
                "Model:getNodeTransform",
                "Model:setNodeTransform",
                "Model:animate"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the node."
                    },
                    {
                      name = "sx",
                      type = "number",
                      description = "The x scale."
                    },
                    {
                      name = "sy",
                      type = "number",
                      description = "The y scale."
                    },
                    {
                      name = "sz",
                      type = "number",
                      description = "The z scale."
                    },
                    {
                      name = "blend",
                      type = "number",
                      description = "A number from 0 to 1 indicating how much of the new scale to blend in.  A value of 0 will not change the node's scale at all, whereas 1 will fully blend to the target scale.",
                      default = "1.0"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the node."
                    },
                    {
                      name = "sx",
                      type = "number",
                      description = "The x scale."
                    },
                    {
                      name = "sy",
                      type = "number",
                      description = "The y scale."
                    },
                    {
                      name = "sz",
                      type = "number",
                      description = "The z scale."
                    },
                    {
                      name = "blend",
                      type = "number",
                      description = "A number from 0 to 1 indicating how much of the new scale to blend in.  A value of 0 will not change the node's scale at all, whereas 1 will fully blend to the target scale.",
                      default = "1.0"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the node."
                    },
                    {
                      name = "scale",
                      type = "Vec3",
                      description = "The new scale."
                    },
                    {
                      name = "blend",
                      type = "number",
                      description = "A number from 0 to 1 indicating how much of the new scale to blend in.  A value of 0 will not change the node's scale at all, whereas 1 will fully blend to the target scale.",
                      default = "1.0"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the node."
                    },
                    {
                      name = "scale",
                      type = "Vec3",
                      description = "The new scale."
                    },
                    {
                      name = "blend",
                      type = "number",
                      description = "A number from 0 to 1 indicating how much of the new scale to blend in.  A value of 0 will not change the node's scale at all, whereas 1 will fully blend to the target scale.",
                      default = "1.0"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setNodeTransform",
              summary = "Set or blend the transform of a node.",
              description = "Sets or blends the transform of a node to a new transform.  This sets the local transform of the node, relative to its parent.",
              key = "Model:setNodeTransform",
              module = "lovr.graphics",
              notes = "For best results when animating, it's recommended to keep the 3 components of the scale the same.\n\nEven though the translation, scale, and rotation parameters are given in TSR order, they are applied in the normal TRS order.",
              related = {
                "Model:getNodePosition",
                "Model:setNodePosition",
                "Model:getNodeOrientation",
                "Model:setNodeOrientation",
                "Model:getNodeScale",
                "Model:setNodeScale",
                "Model:getNodePose",
                "Model:setNodePose",
                "Model:animate"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the node."
                    },
                    {
                      name = "x",
                      type = "number",
                      description = "The x component of the position."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y component of the position."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z component of the position."
                    },
                    {
                      name = "sx",
                      type = "number",
                      description = "The x component of the scale."
                    },
                    {
                      name = "sy",
                      type = "number",
                      description = "The y component of the scale."
                    },
                    {
                      name = "sz",
                      type = "number",
                      description = "The z component of the scale."
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the node should be rotated around its rotation axis."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    },
                    {
                      name = "blend",
                      type = "number",
                      description = "A number from 0 to 1 indicating how much of the target transform to blend in.  A value of 0 will not change the node's transform at all, whereas 1 will fully blend to the target transform.",
                      default = "1.0"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the node."
                    },
                    {
                      name = "x",
                      type = "number",
                      description = "The x component of the position."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y component of the position."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z component of the position."
                    },
                    {
                      name = "sx",
                      type = "number",
                      description = "The x component of the scale."
                    },
                    {
                      name = "sy",
                      type = "number",
                      description = "The y component of the scale."
                    },
                    {
                      name = "sz",
                      type = "number",
                      description = "The z component of the scale."
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the node should be rotated around its rotation axis."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    },
                    {
                      name = "blend",
                      type = "number",
                      description = "A number from 0 to 1 indicating how much of the target transform to blend in.  A value of 0 will not change the node's transform at all, whereas 1 will fully blend to the target transform.",
                      default = "1.0"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the node."
                    },
                    {
                      name = "position",
                      type = "Vec3",
                      description = "The position."
                    },
                    {
                      name = "scale",
                      type = "Vec3",
                      description = "The scale."
                    },
                    {
                      name = "orientation",
                      type = "Quat",
                      description = "The orientation."
                    },
                    {
                      name = "blend",
                      type = "number",
                      description = "A number from 0 to 1 indicating how much of the target transform to blend in.  A value of 0 will not change the node's transform at all, whereas 1 will fully blend to the target transform.",
                      default = "1.0"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the node."
                    },
                    {
                      name = "position",
                      type = "Vec3",
                      description = "The position."
                    },
                    {
                      name = "scale",
                      type = "Vec3",
                      description = "The scale."
                    },
                    {
                      name = "orientation",
                      type = "Quat",
                      description = "The orientation."
                    },
                    {
                      name = "blend",
                      type = "number",
                      description = "A number from 0 to 1 indicating how much of the target transform to blend in.  A value of 0 will not change the node's transform at all, whereas 1 will fully blend to the target transform.",
                      default = "1.0"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the node."
                    },
                    {
                      name = "transform",
                      type = "Mat4",
                      description = "The transform."
                    },
                    {
                      name = "blend",
                      type = "number",
                      description = "A number from 0 to 1 indicating how much of the target transform to blend in.  A value of 0 will not change the node's transform at all, whereas 1 will fully blend to the target transform.",
                      default = "1.0"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the node."
                    },
                    {
                      name = "transform",
                      type = "Mat4",
                      description = "The transform."
                    },
                    {
                      name = "blend",
                      type = "number",
                      description = "A number from 0 to 1 indicating how much of the target transform to blend in.  A value of 0 will not change the node's transform at all, whereas 1 will fully blend to the target transform.",
                      default = "1.0"
                    }
                  },
                  returns = {}
                }
              }
            }
          }
        },
        {
          name = "Pass",
          summary = "A stream of graphics commands.",
          description = "Pass objects record work for the GPU.  They contain a list of things to draw and a list of compute shaders to run.\n\nMethods like `Pass:sphere` will \"record\" a draw on the Pass, which adds it to the list.  Other methods like `Pass:setBlendMode` or `Pass:setShader` will change the way the next draws are processed.\n\nOnce all of the work has been recorded to a Pass, it can be sent to the GPU using `lovr.graphics.submit`, which will start processing all of the compute work and draws (in that order).\n\nA Pass can have a canvas, which is a set of textures that the draws will render to.\n\n`Pass:reset` is used to clear all of the computes and draws, putting the Pass in a fresh state.\n\n`lovr.draw` is called every frame with a `Pass` that is configured to render to either the headset or the window.  The Pass will automatically get submitted afterwards.",
          key = "Pass",
          module = "lovr.graphics",
          constructors = {
            "lovr.graphics.newPass",
            "lovr.graphics.getWindowPass",
            "lovr.headset.getPass",
            "lovr.graphics.getPass"
          },
          methods = {
            {
              name = "barrier",
              tag = "compute",
              summary = "Synchronize compute work.",
              description = "Synchronizes compute work.\n\nBy default, within a single Pass, multiple calls to `Pass:compute` can run on the GPU in any order, or all at the same time.  This is great because it lets the GPU process the work as efficiently as possible, but sometimes multiple compute dispatches need to be sequenced.\n\nCalling this function will insert a barrier.  All compute operations on the Pass after the barrier will only start once all of the previous compute operations on the Pass are finished.",
              key = "Pass:barrier",
              module = "lovr.graphics",
              examples = {
                {
                  code = "pass = lovr.graphics.newPass()\npass:setShader(computeShader)\n\npass:compute(x, y, z)\npass:compute(x, y, z)\npass:barrier()\npass:compute(x, y, z) --> waits for the previous 2 :computes to complete"
                }
              },
              notes = "It's only necessary to use a barrier if a compute shader is reading/writing the same bytes of memory that a previous compute operation in the same Pass read/wrote.\n\nBarriers will slow things down because they reduce parallelism by causing the GPU to wait. Strategic reordering of non-dependent :compute calls around the barrier can help.\n\nCalling this function before recording any :computes will do nothing, and calling it after the last :compute will do nothing.",
              related = {
                "Pass:compute"
              },
              variants = {
                {
                  arguments = {},
                  returns = {}
                }
              }
            },
            {
              name = "beginTally",
              tag = "tally",
              summary = "Begin a tally.",
              description = "Begins a new tally.  The tally will count the number of pixels touched by any draws that occur while the tally is active.  If a pixel fails the depth test or stencil test then it won't be counted, so the tally is a way to detect if objects are visible.\n\nThe results for all the tallies in the pass can be copied to a `Buffer` when the Pass finishes by setting a buffer with `Pass:setTallyBuffer`.",
              key = "Pass:beginTally",
              module = "lovr.graphics",
              notes = "There is currently a maximum of 256 tallies per pass.\n\nIf a tally is already active, this function will error.",
              related = {
                "Pass:finishTally"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the tally that was started."
                    }
                  }
                }
              }
            },
            {
              name = "box",
              tag = "drawing",
              summary = "Draw a box.",
              description = "Draw a box.  This is like `Pass:cube`, except it takes 3 separate values for the scale.",
              key = "Pass:box",
              module = "lovr.graphics",
              related = {
                "Pass:cube"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate of the center of the box.",
                      default = "0"
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate of the center of the box.",
                      default = "0"
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate of the center of the box.",
                      default = "0"
                    },
                    {
                      name = "width",
                      type = "number",
                      description = "The width of the box.",
                      default = "1"
                    },
                    {
                      name = "height",
                      type = "number",
                      description = "The height of the box.",
                      default = "1"
                    },
                    {
                      name = "depth",
                      type = "number",
                      description = "The depth of the box.",
                      default = "1"
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The rotation of the box around its rotation axis, in radians.",
                      default = "0"
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation.",
                      default = "0"
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation.",
                      default = "1"
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation.",
                      default = "0"
                    },
                    {
                      name = "style",
                      type = "DrawStyle",
                      description = "Whether the box should be drawn filled or outlined.",
                      default = "'fill'"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "position",
                      type = "Vec3",
                      description = "The position of the box."
                    },
                    {
                      name = "size",
                      type = "Vec3",
                      description = "The size of the box."
                    },
                    {
                      name = "orientation",
                      type = "Quat",
                      description = "The orientation of the box."
                    },
                    {
                      name = "style",
                      type = "DrawStyle",
                      description = "Whether the box should be drawn filled or outlined.",
                      default = "'fill'"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "transform",
                      type = "Mat4",
                      description = "The transform of the box."
                    },
                    {
                      name = "style",
                      type = "DrawStyle",
                      description = "Whether the box should be drawn filled or outlined.",
                      default = "'fill'"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "capsule",
              tag = "drawing",
              summary = "Draw a capsule.",
              description = "Draws a capsule.  A capsule is shaped like a cylinder with a hemisphere on each end.",
              key = "Pass:capsule",
              module = "lovr.graphics",
              notes = "The length of the capsule does not include the end caps.  The local origin of the capsule is in the center, and the local z axis points towards the end caps.",
              variants = {
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate of the center of the capsule.",
                      default = "0"
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate of the center of the capsule.",
                      default = "0"
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate of the center of the capsule.",
                      default = "0"
                    },
                    {
                      name = "radius",
                      type = "number",
                      description = "The radius of the capsule.",
                      default = "1.0"
                    },
                    {
                      name = "length",
                      type = "number",
                      description = "The length of the capsule.",
                      default = "1"
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The rotation of the capsule around its rotation axis, in radians.",
                      default = "0"
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation.",
                      default = "0"
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation.",
                      default = "1"
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation.",
                      default = "0"
                    },
                    {
                      name = "segments",
                      type = "number",
                      description = "The number of circular segments to render.",
                      default = "32"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "position",
                      type = "Vec3",
                      description = "The position of the center of the capsule."
                    },
                    {
                      name = "scale",
                      type = "Vec3",
                      description = "The size of the capsule (x and y scale the radius, z is the length)."
                    },
                    {
                      name = "orientation",
                      type = "Quat",
                      description = "The orientation of the capsule."
                    },
                    {
                      name = "segments",
                      type = "number",
                      description = "The number of circular segments to render.",
                      default = "32"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "transform",
                      type = "Mat4",
                      description = "The transform of the capsule."
                    },
                    {
                      name = "segments",
                      type = "number",
                      description = "The number of circular segments to render.",
                      default = "32"
                    }
                  },
                  returns = {}
                },
                {
                  description = "Draws a capsule between two points.",
                  arguments = {
                    {
                      name = "p1",
                      type = "Vec3",
                      description = "The starting point of the capsule."
                    },
                    {
                      name = "p2",
                      type = "Vec3",
                      description = "The ending point of the capsule."
                    },
                    {
                      name = "radius",
                      type = "number",
                      description = "The radius of the capsule.",
                      default = "1.0"
                    },
                    {
                      name = "segments",
                      type = "number",
                      description = "The number of circular segments to render.",
                      default = "32"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "circle",
              tag = "drawing",
              summary = "Draw a circle.",
              description = "Draws a circle.",
              key = "Pass:circle",
              module = "lovr.graphics",
              notes = "The local origin of the circle is in its center.  The local z axis is perpendicular to the circle.",
              variants = {
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate of the center of the circle.",
                      default = "0"
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate of the center of the circle.",
                      default = "0"
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate of the center of the circle.",
                      default = "0"
                    },
                    {
                      name = "radius",
                      type = "number",
                      description = "The radius of the circle.",
                      default = "1"
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The rotation of the circle around its rotation axis, in radians.",
                      default = "0"
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation.",
                      default = "0"
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation.",
                      default = "1"
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation.",
                      default = "0"
                    },
                    {
                      name = "style",
                      type = "DrawStyle",
                      description = "Whether the circle should be filled or outlined.",
                      default = "'fill'"
                    },
                    {
                      name = "angle1",
                      type = "number",
                      description = "The angle of the beginning of the arc.",
                      default = "0"
                    },
                    {
                      name = "angle2",
                      type = "number",
                      description = "angle of the end of the arc.",
                      default = "2 * math.pi"
                    },
                    {
                      name = "segments",
                      type = "number",
                      description = "The number of segments to render.",
                      default = "64"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "position",
                      type = "Vec3",
                      description = "The position of the circle."
                    },
                    {
                      name = "radius",
                      type = "number",
                      description = "The radius of the circle.",
                      default = "1"
                    },
                    {
                      name = "orientation",
                      type = "Quat",
                      description = "The orientation of the circle."
                    },
                    {
                      name = "style",
                      type = "DrawStyle",
                      description = "Whether the circle should be filled or outlined.",
                      default = "'fill'"
                    },
                    {
                      name = "angle1",
                      type = "number",
                      description = "The angle of the beginning of the arc.",
                      default = "0"
                    },
                    {
                      name = "angle2",
                      type = "number",
                      description = "angle of the end of the arc.",
                      default = "2 * math.pi"
                    },
                    {
                      name = "segments",
                      type = "number",
                      description = "The number of segments to render.",
                      default = "64"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "transform",
                      type = "Mat4",
                      description = "The transform of the circle."
                    },
                    {
                      name = "style",
                      type = "DrawStyle",
                      description = "Whether the circle should be filled or outlined.",
                      default = "'fill'"
                    },
                    {
                      name = "angle1",
                      type = "number",
                      description = "The angle of the beginning of the arc.",
                      default = "0"
                    },
                    {
                      name = "angle2",
                      type = "number",
                      description = "angle of the end of the arc.",
                      default = "2 * math.pi"
                    },
                    {
                      name = "segments",
                      type = "number",
                      description = "The number of segments to render.",
                      default = "64"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "compute",
              tag = "compute",
              summary = "Run a compute shader.",
              description = "Runs a compute shader.  There must be an active compute shader set using `Pass:setShader`.\n\nAll of the compute shader dispatches in a Pass will run **before** all of the draws in the Pass (if any).  They will also run at the same time in parallel, unless `Pass:barrier` is used to control the order.",
              key = "Pass:compute",
              module = "lovr.graphics",
              examples = {
                {
                  description = "A compute shader that makes a texture grayscale.",
                  code = "function lovr.load()\n  shader = lovr.graphics.newShader([[\n    layout(local_size_x = 8, local_size_y = 8) in;\n    layout(set = 0, binding = 0, rgba8) uniform image2D image;\n\n    void lovrmain() {\n      ivec2 size = imageSize(image);\n      ivec2 pixel = ivec2(GlobalThreadID.xy);\n\n      if (pixel.x >= size.x || pixel.y >= size.y) {\n        return;\n      }\n\n      vec4 color = imageLoad(image, pixel);\n      color.rgb = vec3(color.r * .2126 + color.g * .7512 + color.b * .0722);\n      imageStore(image, pixel, color);\n    }\n  ]])\n\n  texture = lovr.graphics.newTexture('image.png', {\n    usage = { 'storage', 'sample', 'transfer' },\n    linear = true -- srgb textures don't always support storage usage\n  })\n\n  local tw, th = texture:getDimensions()\n  local sx, sy = shader:getWorkgroupSize()\n  local gx, gy = math.ceil(tw / sx), math.ceil(th / sy)\n\n  local computer = lovr.graphics.newPass()\n\n  computer:setShader(shader)\n  computer:send('image', texture)\n  computer:compute(gx, gy)\n  lovr.graphics.submit(computer)\n\n  texture:generateMipmaps()\nend\n\nfunction lovr.draw(pass)\n  pass:draw(texture, 0, 1.7, -1)\nend"
                }
              },
              notes = "Compute shaders are usually run once for each pixel in an image, once per particle, once per object, etc.  The 3 arguments represent how many times to run, or \"dispatch\", the compute shader, in up to 3 dimensions.  Each element of this grid is called a **workgroup**.\n\nTo make things even more complicated, each workgroup itself is made up of a set of \"mini GPU threads\", which are called **local workgroups**.  Like workgroups, the local workgroup size can also be 3D.  It's declared in the shader code, like this:\n\n    layout(local_size_x = w, local_size_y = h, local_size_z = d) in;\n\nAll these 3D grids can get confusing, but the basic idea is to make the local workgroup size a small block of e.g. 32 particles or 8x8 pixels or 4x4x4 voxels, and then dispatch however many workgroups are needed to cover a list of particles, image, voxel field, etc.\n\nThe reason to do it this way is that the GPU runs its threads in little fixed-size bundles called subgroups.  Subgroups are usually 32 or 64 threads (the exact size is given by the `subgroupSize` property of `lovr.graphics.getDevice`) and all run together.  If the local workgroup size was `1x1x1`, then the GPU would only run 1 thread per subgroup and waste the other 31 or 63.  So for the best performance, be sure to set a local workgroup size bigger than 1!\n\nInside the compute shader, a few builtin variables can be used to figure out which workgroup is running:\n\n- `uvec3 WorkgroupCount` is the workgroup count per axis (the `Pass:compute` arguments).\n- `uvec3 WorkgroupSize` is the local workgroup size (declared in the shader).\n- `uvec3 WorkgroupID` is the index of the current (global) workgroup.\n- `uvec3 LocalThreadID` is the index of the local workgroup inside its workgroup.\n- `uint LocalThreadIndex` is a 1D version of `LocalThreadID`.\n- `uvec3 GlobalThreadID` is the unique identifier for a thread within all workgroups in a\n  dispatch. It's equivalent to `WorkgroupID * WorkgroupSize + LocalThreadID` (usually what you\n  want!)\n\nIndirect compute dispatches are useful to \"chain\" compute shaders together, while keeping all of the data on the GPU.  The first dispatch can do some computation and write some results to buffers, then the second indirect dispatch can use the data in those buffers to know how many times it should run.  An example would be a compute shader that does some sort of object culling, writing the number of visible objects to a buffer along with the IDs of each one. Subsequent compute shaders can be indirectly dispatched to perform extra processing on the visible objects.  Finally, an indirect draw can be used to render them.",
              related = {
                "Pass:barrier",
                "Pass:setShader",
                "Pass:send"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The number of workgroups to dispatch in the x dimension.",
                      default = "1"
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The number of workgroups to dispatch in the y dimension.",
                      default = "1"
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The number of workgroups to dispatch in the z dimension.",
                      default = "1"
                    }
                  },
                  returns = {}
                },
                {
                  description = "Perform an \"indirect\" dispatch.  Instead of passing in the workgroup counts directly from Lua, the workgroup counts are read from a `Buffer` object at a particular byte offset. Each count should be a 4-byte integer, so in total 12 bytes will be read from the buffer.",
                  arguments = {
                    {
                      name = "buffer",
                      type = "Buffer",
                      description = "A Buffer object containing the x, y, and z workgroup counts, stored as 4 byte unsigned integers."
                    },
                    {
                      name = "offset",
                      type = "number",
                      description = "The byte offset to read the workgroup counts from in the Buffer.",
                      default = "0"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "cone",
              tag = "drawing",
              summary = "Draw a cone.",
              description = "Draws a cone.",
              key = "Pass:cone",
              module = "lovr.graphics",
              notes = "The local origin is at the center of the base of the cone, and the negative z axis points towards the tip.",
              variants = {
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate of the center of the base of the cone.",
                      default = "0"
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate of the center of the base of the cone.",
                      default = "0"
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate of the center of the base of the cone.",
                      default = "0"
                    },
                    {
                      name = "radius",
                      type = "number",
                      description = "The radius of the cone.",
                      default = "1"
                    },
                    {
                      name = "length",
                      type = "number",
                      description = "The length of the cone.",
                      default = "1"
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The rotation of the cone around its rotation axis, in radians.",
                      default = "0"
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation.",
                      default = "0"
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation.",
                      default = "1"
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation.",
                      default = "0"
                    },
                    {
                      name = "segments",
                      type = "number",
                      description = "The number of segments in the cone.",
                      default = "64"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "position",
                      type = "Vec3",
                      description = "The position of the center of the base of the cone."
                    },
                    {
                      name = "scale",
                      type = "Vec3",
                      description = "The size of the cone (x and y scale the radius, z is the length)."
                    },
                    {
                      name = "orientation",
                      type = "Quat",
                      description = "The orientation of the cone."
                    },
                    {
                      name = "segments",
                      type = "number",
                      description = "The number of segments in the cone.",
                      default = "64"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "transform",
                      type = "Mat4",
                      description = "The transform of the cone."
                    },
                    {
                      name = "segments",
                      type = "number",
                      description = "The number of segments in the cone.",
                      default = "64"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "p1",
                      type = "Vec3",
                      description = "The position of the base of the cone."
                    },
                    {
                      name = "p2",
                      type = "Vec3",
                      description = "The position of the tip of the cone."
                    },
                    {
                      name = "radius",
                      type = "number",
                      description = "The radius of the cone.",
                      default = "1"
                    },
                    {
                      name = "segments",
                      type = "number",
                      description = "The number of segments in the cone.",
                      default = "64"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "cube",
              tag = "drawing",
              summary = "Draw a cube.",
              description = "Draws a cube.",
              key = "Pass:cube",
              module = "lovr.graphics",
              notes = "The local origin is in the center of the cube.",
              variants = {
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate of the center of the cube.",
                      default = "0"
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate of the center of the cube.",
                      default = "0"
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate of the center of the cube.",
                      default = "0"
                    },
                    {
                      name = "size",
                      type = "number",
                      description = "The size of the cube.",
                      default = "1"
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The rotation of the cube around its rotation axis, in radians.",
                      default = "0"
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation.",
                      default = "0"
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation.",
                      default = "1"
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation.",
                      default = "0"
                    },
                    {
                      name = "style",
                      type = "DrawStyle",
                      description = "Whether the cube should be drawn filled or outlined.",
                      default = "'fill'"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "position",
                      type = "Vec3",
                      description = "The position of the cube."
                    },
                    {
                      name = "size",
                      type = "number",
                      description = "The size of the cube.",
                      default = "1"
                    },
                    {
                      name = "orientation",
                      type = "Quat",
                      description = "The orientation of the cube."
                    },
                    {
                      name = "style",
                      type = "DrawStyle",
                      description = "Whether the cube should be drawn filled or outlined.",
                      default = "'fill'"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "transform",
                      type = "Mat4",
                      description = "The transform of the cube."
                    },
                    {
                      name = "style",
                      type = "DrawStyle",
                      description = "Whether the cube should be drawn filled or outlined.",
                      default = "'fill'"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "cylinder",
              tag = "drawing",
              summary = "Draw a cylinder.",
              description = "Draws a cylinder.",
              key = "Pass:cylinder",
              module = "lovr.graphics",
              notes = "The local origin is in the center of the cylinder, and the length of the cylinder is along the z axis.",
              variants = {
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate of the center of the cylinder.",
                      default = "0"
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate of the center of the cylinder.",
                      default = "0"
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate of the center of the cylinder.",
                      default = "0"
                    },
                    {
                      name = "radius",
                      type = "number",
                      description = "The radius of the cylinder.",
                      default = "1"
                    },
                    {
                      name = "length",
                      type = "number",
                      description = "The length of the cylinder.",
                      default = "1"
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The rotation of the cylinder around its rotation axis, in radians.",
                      default = "0"
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation.",
                      default = "0"
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation.",
                      default = "1"
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation.",
                      default = "0"
                    },
                    {
                      name = "capped",
                      type = "boolean",
                      description = "Whether the tops and bottoms of the cylinder should be rendered.",
                      default = "true"
                    },
                    {
                      name = "angle1",
                      type = "number",
                      description = "The angle of the beginning of the arc.",
                      default = "0"
                    },
                    {
                      name = "angle2",
                      type = "number",
                      description = "angle of the end of the arc.",
                      default = "2 * math.pi"
                    },
                    {
                      name = "segments",
                      type = "number",
                      description = "The number of circular segments to render.",
                      default = "64"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "position",
                      type = "Vec3",
                      description = "The position of the center of the cylinder."
                    },
                    {
                      name = "scale",
                      type = "Vec3",
                      description = "The size of the cylinder (x and y scale the radius, z is the length)."
                    },
                    {
                      name = "orientation",
                      type = "Quat",
                      description = "The orientation of the cylinder."
                    },
                    {
                      name = "capped",
                      type = "boolean",
                      description = "Whether the tops and bottoms of the cylinder should be rendered.",
                      default = "true"
                    },
                    {
                      name = "angle1",
                      type = "number",
                      description = "The angle of the beginning of the arc.",
                      default = "0"
                    },
                    {
                      name = "angle2",
                      type = "number",
                      description = "angle of the end of the arc.",
                      default = "2 * math.pi"
                    },
                    {
                      name = "segments",
                      type = "number",
                      description = "The number of circular segments to render.",
                      default = "64"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "transform",
                      type = "Mat4",
                      description = "The transform of the cylinder."
                    },
                    {
                      name = "capped",
                      type = "boolean",
                      description = "Whether the tops and bottoms of the cylinder should be rendered.",
                      default = "true"
                    },
                    {
                      name = "angle1",
                      type = "number",
                      description = "The angle of the beginning of the arc.",
                      default = "0"
                    },
                    {
                      name = "angle2",
                      type = "number",
                      description = "angle of the end of the arc.",
                      default = "2 * math.pi"
                    },
                    {
                      name = "segments",
                      type = "number",
                      description = "The number of circular segments to render.",
                      default = "64"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "p1",
                      type = "Vec3",
                      description = "The starting point of the cylinder."
                    },
                    {
                      name = "p2",
                      type = "Vec3",
                      description = "The ending point of the cylinder."
                    },
                    {
                      name = "radius",
                      type = "number",
                      description = "The radius of the cylinder.",
                      default = "1"
                    },
                    {
                      name = "capped",
                      type = "boolean",
                      description = "Whether the tops and bottoms of the cylinder should be rendered.",
                      default = "true"
                    },
                    {
                      name = "angle1",
                      type = "number",
                      description = "The angle of the beginning of the arc.",
                      default = "0"
                    },
                    {
                      name = "angle2",
                      type = "number",
                      description = "angle of the end of the arc.",
                      default = "2 * math.pi"
                    },
                    {
                      name = "segments",
                      type = "number",
                      description = "The number of circular segments to render.",
                      default = "64"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "draw",
              tag = "drawing",
              summary = "Draw a Model, Mesh, or Texture.",
              description = "Draws a `Model`, `Mesh`, or `Texture`.",
              key = "Pass:draw",
              module = "lovr.graphics",
              notes = "`Model:getMesh` can be used to draw individual meshes of a model.\n\nTextures ignore the `instances` parameter.\n\nWhen drawing a Texture, the plane will be 1 meter wide at 1.0 scale and the height will be adjusted based on the Texture's aspect ratio.",
              variants = {
                {
                  arguments = {
                    {
                      name = "object",
                      type = "*",
                      description = "The Model, Mesh, or Texture to draw."
                    },
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate to draw the object at.",
                      default = "0"
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate to draw the object at.",
                      default = "0"
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate to draw the object at.",
                      default = "0"
                    },
                    {
                      name = "scale",
                      type = "number",
                      description = "The scale of the object.",
                      default = "1"
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The rotation of the object around its rotation axis, in radians.",
                      default = "0"
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation.",
                      default = "0"
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation.",
                      default = "1"
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation.",
                      default = "0"
                    },
                    {
                      name = "instances",
                      type = "number",
                      description = "The number of instances to draw.",
                      default = "1"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "object",
                      type = "*",
                      description = "The Model, Mesh, or Texture to draw."
                    },
                    {
                      name = "position",
                      type = "Vec3",
                      description = "The position to draw the object at."
                    },
                    {
                      name = "scale",
                      type = "number",
                      description = "The scale of the object.",
                      default = "1"
                    },
                    {
                      name = "orientation",
                      type = "Quat",
                      description = "The orientation of the object."
                    },
                    {
                      name = "instances",
                      type = "number",
                      description = "The number of instances to draw.",
                      default = "1"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "object",
                      type = "*",
                      description = "The Model, Mesh, or Texture to draw."
                    },
                    {
                      name = "transform",
                      type = "Mat4",
                      description = "The transform of the object."
                    },
                    {
                      name = "instances",
                      type = "number",
                      description = "The number of instances to draw.",
                      default = "1"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "fill",
              tag = "drawing",
              summary = "Draw a fullscreen triangle.",
              description = "Draws a fullscreen triangle.  The `fill` shader is used, which stretches the triangle across the screen.",
              key = "Pass:fill",
              module = "lovr.graphics",
              notes = "This function has some special behavior for array textures:\n\n- Filling a single-layer texture to a multi-layer canvas will mirror the texture to all layers,\n  just like regular drawing.\n- Filling a 2-layer texture to a mono canvas will render the 2 layers side-by-side.\n- Filling a multi-layer texture to a multi-layer canvas will do a layer-by-layer fill (the layer\n  counts must match).",
              variants = {
                {
                  arguments = {
                    {
                      name = "texture",
                      type = "Texture",
                      description = "The texture to fill.  If nil, the texture from the active material is used."
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {},
                  returns = {}
                }
              }
            },
            {
              name = "finishTally",
              tag = "tally",
              summary = "Finish a tally.",
              description = "Finishes a tally that was previously started with `Pass:beginTally`.  This will stop counting the number of pixels affected by draws.\n\nThe results for all the tallies in the pass can be copied to a `Buffer` when the Pass finishes by setting a buffer with `Pass:setTallyBuffer`.",
              key = "Pass:finishTally",
              module = "lovr.graphics",
              notes = "There is currently a maximum of 256 tallies per pass.\n\nIf no tally is active, this function will error.",
              related = {
                "Pass:beginTally"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the tally that was finished."
                    }
                  }
                }
              }
            },
            {
              name = "getCanvas",
              tag = "canvas",
              summary = "Get the Pass's canvas.",
              description = "Returns the Pass's canvas, or `nil` if the Pass doesn't have a canvas.  The canvas is a set of textures that the Pass will draw to when it's submitted.",
              key = "Pass:getCanvas",
              module = "lovr.graphics",
              notes = "If the Pass has multiple color textures, a fragment shader should be used to write a different color to each texture.  Here's an example that writes red to the first texture and blue to the second texture:\n\n    // Declare an output variable for the second texture\n    layout(location = 1) out vec4 secondColor;\n\n    vec4 lovrmain() {\n      secondColor = vec4(0, 0, 1, 1);\n      return vec4(1, 0, 0, 1);\n    }",
              related = {
                "Pass:getClear",
                "Pass:setClear",
                "Pass:getWidth",
                "Pass:getHeight",
                "Pass:getDimensions"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "canvas",
                      type = "table",
                      description = "The canvas.  Numeric keys will contain the color Textures, along with the following keys:",
                      table = {
                        {
                          name = "depth",
                          type = "*",
                          description = "A `Texture` or `TextureFormat` with the depth buffer."
                        },
                        {
                          name = "samples",
                          type = "number",
                          description = "The number of multisamples used for antialiasing (either 1 or 4)."
                        }
                      }
                    }
                  }
                },
                {
                  description = "This function returns nil when a canvas hasn't been set.",
                  arguments = {},
                  returns = {}
                }
              }
            },
            {
              name = "getClear",
              tag = "canvas",
              summary = "Return the clear values of the Pass.",
              description = "Returns the clear values of the pass.",
              key = "Pass:getClear",
              module = "lovr.graphics",
              notes = "The default clear color is transparent black.",
              related = {
                "Pass:getCanvas"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "clears",
                      type = "table",
                      description = "The clear values for the pass.  Each color texture's clear value is stored at its index, as either a 4-number rgba table or a boolean.  If the pass has a depth texture, there will also be a `depth` key with its clear value as a number or boolean."
                    }
                  }
                }
              }
            },
            {
              name = "getDimensions",
              tag = "canvas",
              summary = "Get the dimensions of the Pass's canvas.",
              description = "Returns the dimensions of the textures of the Pass's canvas, in pixels.",
              key = "Pass:getDimensions",
              module = "lovr.graphics",
              notes = "If the pass doesn't have a canvas, this function returns zeros.",
              related = {
                "Pass:getWidth",
                "Pass:getHeight",
                "Pass:getViewCount",
                "Pass:getCanvas",
                "Pass:setCanvas",
                "lovr.system.getWindowDimensions",
                "lovr.headset.getDisplayDimensions"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "width",
                      type = "number",
                      description = "The texture width."
                    },
                    {
                      name = "height",
                      type = "number",
                      description = "The texture height."
                    }
                  }
                }
              }
            },
            {
              name = "getHeight",
              tag = "canvas",
              summary = "Get the height of the Pass's canvas.",
              description = "Returns the height of the textures of the Pass's canvas, in pixels.",
              key = "Pass:getHeight",
              module = "lovr.graphics",
              notes = "If the pass doesn't have a canvas, this function returns zero.",
              related = {
                "Pass:getWidth",
                "Pass:getDimensions",
                "Pass:getViewCount",
                "Pass:getCanvas",
                "Pass:setCanvas",
                "lovr.system.getWindowHeight",
                "lovr.headset.getDisplayHeight"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "height",
                      type = "number",
                      description = "The texture height."
                    }
                  }
                }
              }
            },
            {
              name = "getProjection",
              tag = "camera",
              summary = "Get the field of view.",
              description = "Returns the projection for a single view.",
              key = "Pass:getProjection",
              module = "lovr.graphics",
              related = {
                "lovr.headset.getViewAngles",
                "lovr.headset.getViewCount",
                "Pass:getViewPose",
                "Pass:setViewPose"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "view",
                      type = "number",
                      description = "The view index."
                    }
                  },
                  returns = {
                    {
                      name = "left",
                      type = "number",
                      description = "The left field of view angle, in radians."
                    },
                    {
                      name = "right",
                      type = "number",
                      description = "The right field of view angle, in radians."
                    },
                    {
                      name = "up",
                      type = "number",
                      description = "The top field of view angle, in radians."
                    },
                    {
                      name = "down",
                      type = "number",
                      description = "The bottom field of view angle, in radians."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "view",
                      type = "number",
                      description = "The view index."
                    },
                    {
                      name = "matrix",
                      type = "Mat4",
                      description = "The matrix to fill with the projection."
                    }
                  },
                  returns = {
                    {
                      name = "matrix",
                      type = "Mat4",
                      description = "The matrix containing the projection."
                    }
                  }
                }
              }
            },
            {
              name = "getSampleCount",
              tag = "canvas",
              summary = "Get the antialiasing setting of a render pass.",
              description = "Returns the antialiasing setting of a render pass.",
              key = "Pass:getSampleCount",
              module = "lovr.graphics",
              deprecated = true,
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "samples",
                      type = "number",
                      description = "The number of samples used for rendering.  Currently, will be 1 or 4."
                    }
                  }
                }
              }
            },
            {
              name = "getScissor",
              tag = "camera",
              summary = "Get the scissor rectangle.",
              description = "Returns the scissor rectangle, or `nil` if no scissor is set.  Any pixels outside the scissor rectangle will not be drawn.",
              key = "Pass:getScissor",
              module = "lovr.graphics",
              notes = "The scissor will apply to all draws in a Pass when the pass is submitted.\n\nThe default scissor rectangle covers the entire dimensions of the render pass textures.",
              related = {
                "Pass:getViewport",
                "Pass:setViewport"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate of the upper-left corner of the scissor rectangle."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate of the upper-left corner of the scissor rectangle."
                    },
                    {
                      name = "w",
                      type = "number",
                      description = "The width of the scissor rectangle."
                    },
                    {
                      name = "h",
                      type = "number",
                      description = "The height of the scissor rectangle."
                    }
                  }
                }
              }
            },
            {
              name = "getStats",
              tag = "pass-misc",
              summary = "Get statistics for the Pass.",
              description = "Returns statistics for the Pass.",
              key = "Pass:getStats",
              module = "lovr.graphics",
              examples = {
                {
                  description = "See how long it takes the GPU to render a cube.",
                  code = "lovr.graphics.setTimingEnabled(true)\n\nfunction lovr.draw(pass)\n  pass:cube(0, 1.7, -1, .5, lovr.timer.getTime() * .2, 0, 1, 0)\n\n  local stats = pass:getStats()\n  print(('Rendering a cube takes %f microseconds'):format(stats.gpuTime * 1e6))\nend"
                }
              },
              related = {
                "lovr.graphics.isTimingEnabled",
                "lovr.graphics.setTimingEnabled",
                "Pass:setViewCull"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "stats",
                      type = "table",
                      description = "A table with statistics.",
                      table = {
                        {
                          name = "draws",
                          type = "number",
                          description = "The number of draws added to the Pass."
                        },
                        {
                          name = "computes",
                          type = "number",
                          description = "The number of compute calls added to the Pass."
                        },
                        {
                          name = "drawsCulled",
                          type = "number",
                          description = "The number of draw calls that were culled the last time the Pass was submitted."
                        },
                        {
                          name = "cpuMemoryReserved",
                          type = "number",
                          description = "The amount of CPU memory the Pass has reserved, in bytes."
                        },
                        {
                          name = "cpuMemoryUsed",
                          type = "number",
                          description = "The amount of CPU memory the Pass is currently using, in bytes."
                        },
                        {
                          name = "submitTime",
                          type = "number",
                          description = "The amount of time taken on the CPU to submit the Pass the last time it was submitted, in seconds.  Only updates when timing stats have been enabled with `lovr.graphics.setTimingEnabled`, and has a few frames of delay."
                        },
                        {
                          name = "gpuTime",
                          type = "number",
                          description = "The amount of time taken on the GPU to process the Pass, in seconds.  Only updates when timing stats have been enabled with `lovr.graphics.setTimingEnabled`, and has a few frames of delay."
                        }
                      }
                    }
                  }
                }
              }
            },
            {
              name = "getTallyBuffer",
              tag = "tally",
              summary = "Get the Buffer that tally results will be written to.",
              description = "Returns the Buffer that tally results will be written to.  Each time the render pass finishes, the results of all the tallies will be copied to the Buffer at the specified offset.  The buffer can be used in a later pass in a compute shader, or the data in the buffer can be read back using e.g. `Buffer:newReadback`.\n\nIf no buffer has been set, this function will return `nil`.",
              key = "Pass:getTallyBuffer",
              module = "lovr.graphics",
              related = {
                "Pass:beginTally",
                "Pass:finishTally"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "buffer",
                      type = "Buffer",
                      description = "The buffer."
                    },
                    {
                      name = "offset",
                      type = "number",
                      description = "An offset in the buffer where results will be written."
                    }
                  }
                }
              }
            },
            {
              name = "getTarget",
              tag = "canvas",
              summary = "Get the textures a render pass is rendering to.",
              description = "Returns the textures a render pass is rendering to.",
              key = "Pass:getTarget",
              module = "lovr.graphics",
              deprecated = true,
              related = {
                "Pass:getClear"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "target",
                      type = "table",
                      description = "A table of the color textures targeted by the pass, with an additional `depth` key if the pass has a depth texture."
                    }
                  }
                }
              }
            },
            {
              name = "getType",
              tag = "pass-misc",
              summary = "Get the type of the Pass.",
              description = "Returns the type of the pass (render, compute, or transfer).  The type restricts what kinds of functions can be called on the pass.",
              key = "Pass:getType",
              module = "lovr.graphics",
              deprecated = true,
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "type",
                      type = "PassType",
                      description = "The type of the Pass."
                    }
                  }
                }
              }
            },
            {
              name = "getViewCount",
              tag = "camera",
              summary = "Returns the view count of a render pass.",
              description = "Returns the view count of a render pass.  This is the layer count of the textures it is rendering to.",
              key = "Pass:getViewCount",
              module = "lovr.graphics",
              notes = "A render pass has one \"camera\" for each view.  Whenever something is drawn, it is broadcast to each view (layer) of each texture, using the corresponding camera.",
              related = {
                "Pass:getViewPose",
                "Pass:setViewPose",
                "Pass:getProjection",
                "Pass:setProjection",
                "lovr.headset.getViewCount"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "views",
                      type = "number",
                      description = "The view count."
                    }
                  }
                }
              }
            },
            {
              name = "getViewPose",
              tag = "camera",
              summary = "Get the camera pose.",
              description = "Get the pose of a single view.",
              key = "Pass:getViewPose",
              module = "lovr.graphics",
              related = {
                "lovr.headset.getViewPose",
                "lovr.headset.getViewCount",
                "Pass:getProjection",
                "Pass:setProjection"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "view",
                      type = "number",
                      description = "The view index."
                    }
                  },
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x position of the viewer, in meters."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y position of the viewer, in meters."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z position of the viewer, in meters."
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the viewer is rotated around its axis of rotation."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "view",
                      type = "number",
                      description = "The view index."
                    },
                    {
                      name = "matrix",
                      type = "Mat4",
                      description = "The matrix to fill with the view pose."
                    },
                    {
                      name = "invert",
                      type = "boolean",
                      description = "Whether the matrix should be inverted."
                    }
                  },
                  returns = {
                    {
                      name = "matrix",
                      type = "Mat4",
                      description = "The matrix containing the view pose."
                    }
                  }
                }
              }
            },
            {
              name = "getViewport",
              tag = "camera",
              summary = "Get the viewport.",
              description = "Returns the viewport, or `nil` if no viewport is set.  Everything rendered will get mapped to the rectangle defined by the viewport.  More specifically, this defines the transformation from normalized device coordinates to pixel coordinates.",
              key = "Pass:getViewport",
              module = "lovr.graphics",
              notes = "The viewport will apply to all draws in a Pass when the pass is submitted.\n\nThe viewport rectangle can use floating point numbers.\n\nA negative viewport height (with a y coordinate equal to the bottom of the viewport) can be used to flip the rendering vertically.\n\nThe default viewport extends from `(0, 0)` to the dimensions of the target textures, with min depth and max depth respectively set to 0 and 1.",
              related = {
                "Pass:getScissor",
                "Pass:setScissor",
                "Pass:getDimensions"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate of the upper-left corner of the viewport."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate of the upper-left corner of the viewport."
                    },
                    {
                      name = "w",
                      type = "number",
                      description = "The width of the viewport."
                    },
                    {
                      name = "h",
                      type = "number",
                      description = "The height of the viewport.  May be negative."
                    },
                    {
                      name = "dmin",
                      type = "number",
                      description = "The min component of the depth range.",
                      default = "0.0"
                    },
                    {
                      name = "dmax",
                      type = "number",
                      description = "The max component of the depth range.",
                      default = "1.0"
                    }
                  }
                }
              }
            },
            {
              name = "getWidth",
              tag = "canvas",
              summary = "Get the width of the Pass's canvas.",
              description = "Returns the width of the textures of the Pass's canvas, in pixels.",
              key = "Pass:getWidth",
              module = "lovr.graphics",
              notes = "If the pass doesn't have a canvas, this function returns zero.",
              related = {
                "Pass:getHeight",
                "Pass:getDimensions",
                "Pass:getViewCount",
                "Pass:getCanvas",
                "Pass:setCanvas",
                "lovr.system.getWindowWidth",
                "lovr.headset.getDisplayWidth"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "width",
                      type = "number",
                      description = "The texture width."
                    }
                  }
                }
              }
            },
            {
              name = "line",
              tag = "drawing",
              summary = "Draw a line.",
              description = "Draws a line between points.  `Pass:mesh` can also be used to draw line segments using the `line` `DrawMode`.",
              key = "Pass:line",
              module = "lovr.graphics",
              notes = "There is currently no way to increase line thickness.",
              variants = {
                {
                  arguments = {
                    {
                      name = "x1",
                      type = "number",
                      description = "The x coordinate of the first point."
                    },
                    {
                      name = "y1",
                      type = "number",
                      description = "The y coordinate of the first point."
                    },
                    {
                      name = "z1",
                      type = "number",
                      description = "The z coordinate of the first point."
                    },
                    {
                      name = "x2",
                      type = "number",
                      description = "The x coordinate of the next point."
                    },
                    {
                      name = "y2",
                      type = "number",
                      description = "The y coordinate of the next point."
                    },
                    {
                      name = "z2",
                      type = "number",
                      description = "The z coordinate of the next point."
                    },
                    {
                      name = "...",
                      type = "*",
                      description = "More points to add to the line."
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "t",
                      type = "table",
                      description = "A table of numbers or `Vec3` objects (not a mix) representing points of the line."
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "v1",
                      type = "Vec3",
                      description = "A vector containing the position of the first point of the line."
                    },
                    {
                      name = "v2",
                      type = "Vec3",
                      description = "A vector containing the position of the next point on the line."
                    },
                    {
                      name = "...",
                      type = "*",
                      description = "More points to add to the line."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "mesh",
              tag = "drawing",
              summary = "Draw a mesh.",
              description = "Draws a mesh.",
              key = "Pass:mesh",
              module = "lovr.graphics",
              examples = {
                {
                  code = "function lovr.load()\n  local vertices = {\n    vec3(  0,  .4, 0), vec4(1, 0, 0, 1),\n    vec3(-.5, -.4, 0), vec4(0, 1, 0, 1),\n    vec3( .5, -.4, 0), vec4(0, 0, 1, 1)\n  }\n\n  local format = {\n    { name = 'VertexPosition', type = 'vec3' },\n    { name = 'VertexColor', type = 'vec4' }\n  }\n\n  triangle = lovr.graphics.newBuffer(format, vertices)\nend\n\nfunction lovr.draw(pass)\n  pass:mesh(triangle, 0, 1.7, -1)\nend"
                }
              },
              notes = "The index buffer defines the order the vertices are drawn in.  It can be used to reorder, reuse, or omit vertices from the mesh.\n\nWhen drawing without a vertex buffer, the `VertexIndex` variable can be used in shaders to compute the position of each vertex, possibly by reading data from other `Buffer` or `Texture` resources.\n\nThe active `DrawMode` controls whether the vertices are drawn as points, lines, or triangles.\n\nThe active `Material` is applied to the mesh.",
              variants = {
                {
                  description = "Draw a range of vertices from a Buffer, using numbers for the transform.",
                  arguments = {
                    {
                      name = "vertices",
                      type = "Buffer",
                      description = "The buffer containing the vertices to draw.",
                      default = "nil"
                    },
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate of the position to draw the mesh at.",
                      default = "0"
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate of the position to draw the mesh at.",
                      default = "0"
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate of the position to draw the mesh at.",
                      default = "0"
                    },
                    {
                      name = "scale",
                      type = "number",
                      description = "The scale of the mesh.",
                      default = "1"
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the mesh is rotated around its rotational axis.",
                      default = "0"
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation.",
                      default = "0"
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation.",
                      default = "1"
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation.",
                      default = "0"
                    },
                    {
                      name = "start",
                      type = "number",
                      description = "The 1-based index of the first vertex to render from the vertex buffer (or the first index, when using an index buffer).",
                      default = "1"
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "The number of vertices to render (or the number of indices, when using an index buffer). When `nil`, as many vertices or indices as possible will be drawn (based on the length of the Buffers and `start`).",
                      default = "nil"
                    },
                    {
                      name = "instances",
                      type = "number",
                      description = "The number of copies of the mesh to render.",
                      default = "1"
                    }
                  },
                  returns = {}
                },
                {
                  description = "Draw a range of vertices from a Buffer, using vector types for the transform.",
                  arguments = {
                    {
                      name = "vertices",
                      type = "Buffer",
                      description = "The buffer containing the vertices to draw.",
                      default = "nil"
                    },
                    {
                      name = "position",
                      type = "Vec3",
                      description = "The position to draw the mesh at."
                    },
                    {
                      name = "scales",
                      type = "Vec3",
                      description = "The scale of the mesh."
                    },
                    {
                      name = "orientation",
                      type = "Quat",
                      description = "The orientation of the mesh."
                    },
                    {
                      name = "start",
                      type = "number",
                      description = "The 1-based index of the first vertex to render from the vertex buffer (or the first index, when using an index buffer).",
                      default = "1"
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "The number of vertices to render (or the number of indices, when using an index buffer). When `nil`, as many vertices or indices as possible will be drawn (based on the length of the Buffers and `start`).",
                      default = "nil"
                    },
                    {
                      name = "instances",
                      type = "number",
                      description = "The number of copies of the mesh to render.",
                      default = "1"
                    }
                  },
                  returns = {}
                },
                {
                  description = "Draw a range of vertices from a Buffer, using a matrix for the transform.",
                  arguments = {
                    {
                      name = "vertices",
                      type = "Buffer",
                      description = "The buffer containing the vertices to draw.",
                      default = "nil"
                    },
                    {
                      name = "transform",
                      type = "Mat4",
                      description = "The transform to apply to the mesh."
                    },
                    {
                      name = "start",
                      type = "number",
                      description = "The 1-based index of the first vertex to render from the vertex buffer (or the first index, when using an index buffer).",
                      default = "1"
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "The number of vertices to render (or the number of indices, when using an index buffer). When `nil`, as many vertices or indices as possible will be drawn (based on the length of the Buffers and `start`).",
                      default = "nil"
                    },
                    {
                      name = "instances",
                      type = "number",
                      description = "The number of copies of the mesh to render.",
                      default = "1"
                    }
                  },
                  returns = {}
                },
                {
                  description = "Draw a mesh using a vertex buffer and an index buffer, using numbers for the transform.",
                  arguments = {
                    {
                      name = "vertices",
                      type = "Buffer",
                      description = "The buffer containing the vertices to draw.",
                      default = "nil"
                    },
                    {
                      name = "indices",
                      type = "Buffer",
                      description = "The buffer containing the vertex indices to draw."
                    },
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate of the position to draw the mesh at.",
                      default = "0"
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate of the position to draw the mesh at.",
                      default = "0"
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate of the position to draw the mesh at.",
                      default = "0"
                    },
                    {
                      name = "scale",
                      type = "number",
                      description = "The scale of the mesh.",
                      default = "1"
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the mesh is rotated around its rotational axis.",
                      default = "0"
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation.",
                      default = "0"
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation.",
                      default = "1"
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation.",
                      default = "0"
                    },
                    {
                      name = "start",
                      type = "number",
                      description = "The 1-based index of the first vertex to render from the vertex buffer (or the first index, when using an index buffer).",
                      default = "1"
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "The number of vertices to render (or the number of indices, when using an index buffer). When `nil`, as many vertices or indices as possible will be drawn (based on the length of the Buffers and `start`).",
                      default = "nil"
                    },
                    {
                      name = "instances",
                      type = "number",
                      description = "The number of copies of the mesh to render.",
                      default = "1"
                    },
                    {
                      name = "base",
                      type = "number",
                      description = "A base offset to apply to vertex indices.",
                      default = "0"
                    }
                  },
                  returns = {}
                },
                {
                  description = "Draw a mesh using a vertex buffer and an index buffer, using vector types for the transform.",
                  arguments = {
                    {
                      name = "vertices",
                      type = "Buffer",
                      description = "The buffer containing the vertices to draw.",
                      default = "nil"
                    },
                    {
                      name = "indices",
                      type = "Buffer",
                      description = "The buffer containing the vertex indices to draw."
                    },
                    {
                      name = "position",
                      type = "Vec3",
                      description = "The position to draw the mesh at."
                    },
                    {
                      name = "scales",
                      type = "Vec3",
                      description = "The scale of the mesh."
                    },
                    {
                      name = "orientation",
                      type = "Quat",
                      description = "The orientation of the mesh."
                    },
                    {
                      name = "start",
                      type = "number",
                      description = "The 1-based index of the first vertex to render from the vertex buffer (or the first index, when using an index buffer).",
                      default = "1"
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "The number of vertices to render (or the number of indices, when using an index buffer). When `nil`, as many vertices or indices as possible will be drawn (based on the length of the Buffers and `start`).",
                      default = "nil"
                    },
                    {
                      name = "instances",
                      type = "number",
                      description = "The number of copies of the mesh to render.",
                      default = "1"
                    },
                    {
                      name = "base",
                      type = "number",
                      description = "A base offset to apply to vertex indices.",
                      default = "0"
                    }
                  },
                  returns = {}
                },
                {
                  description = "Draw a mesh using a vertex buffer and an index buffer, using a matrix for the transform.",
                  arguments = {
                    {
                      name = "vertices",
                      type = "Buffer",
                      description = "The buffer containing the vertices to draw.",
                      default = "nil"
                    },
                    {
                      name = "indices",
                      type = "Buffer",
                      description = "The buffer containing the vertex indices to draw."
                    },
                    {
                      name = "transform",
                      type = "Mat4",
                      description = "The transform to apply to the mesh."
                    },
                    {
                      name = "start",
                      type = "number",
                      description = "The 1-based index of the first vertex to render from the vertex buffer (or the first index, when using an index buffer).",
                      default = "1"
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "The number of vertices to render (or the number of indices, when using an index buffer). When `nil`, as many vertices or indices as possible will be drawn (based on the length of the Buffers and `start`).",
                      default = "nil"
                    },
                    {
                      name = "instances",
                      type = "number",
                      description = "The number of copies of the mesh to render.",
                      default = "1"
                    },
                    {
                      name = "base",
                      type = "number",
                      description = "A base offset to apply to vertex indices.",
                      default = "0"
                    }
                  },
                  returns = {}
                },
                {
                  description = "Perform indirect draws.  `drawcount` meshes from the vertex and index buffer will be drawn, using parameters starting from `offset` bytes in the `draws` buffer.",
                  arguments = {
                    {
                      name = "vertices",
                      type = "Buffer",
                      description = "The buffer containing the vertices to draw.",
                      default = "nil"
                    },
                    {
                      name = "indices",
                      type = "Buffer",
                      description = "The buffer containing the vertex indices to draw."
                    },
                    {
                      name = "draws",
                      type = "Buffer",
                      description = "The buffer containing indirect draw commands."
                    },
                    {
                      name = "drawcount",
                      type = "number",
                      description = "The number of indirect draws to draw."
                    },
                    {
                      name = "offset",
                      type = "number",
                      description = "A byte offset into the draw buffer."
                    },
                    {
                      name = "stride",
                      type = "number",
                      description = "The number of bytes between consecutive elements in the draw buffer."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "origin",
              tag = "transform",
              summary = "Reset the transform to the origin.",
              description = "Resets the transform back to the origin.",
              key = "Pass:origin",
              module = "lovr.graphics",
              related = {
                "Pass:translate",
                "Pass:rotate",
                "Pass:scale",
                "Pass:transform",
                "Pass:push",
                "Pass:pop"
              },
              variants = {
                {
                  arguments = {},
                  returns = {}
                }
              }
            },
            {
              name = "plane",
              tag = "drawing",
              summary = "Draw a plane.",
              description = "Draws a plane.",
              key = "Pass:plane",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate of the center of the plane.",
                      default = "0"
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate of the center of the plane.",
                      default = "0"
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate of the center of the plane.",
                      default = "0"
                    },
                    {
                      name = "width",
                      type = "number",
                      description = "The width of the plane.",
                      default = "1"
                    },
                    {
                      name = "height",
                      type = "number",
                      description = "The height of the plane.",
                      default = "1"
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The rotation of the plane around its rotation axis, in radians.",
                      default = "0"
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation.",
                      default = "0"
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation.",
                      default = "1"
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation.",
                      default = "0"
                    },
                    {
                      name = "style",
                      type = "DrawStyle",
                      description = "Whether the plane should be drawn filled or outlined.",
                      default = "'fill'"
                    },
                    {
                      name = "columns",
                      type = "number",
                      description = "The number of horizontal segments in the plane.",
                      default = "1"
                    },
                    {
                      name = "rows",
                      type = "number",
                      description = "The number of vertical segments in the plane.",
                      default = "columns"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "position",
                      type = "Vec3",
                      description = "The position of the plane."
                    },
                    {
                      name = "size",
                      type = "Vec2",
                      description = "The size of the plane."
                    },
                    {
                      name = "orientation",
                      type = "Quat",
                      description = "The orientation of the plane."
                    },
                    {
                      name = "style",
                      type = "DrawStyle",
                      description = "Whether the plane should be drawn filled or outlined.",
                      default = "'fill'"
                    },
                    {
                      name = "columns",
                      type = "number",
                      description = "The number of horizontal segments in the plane.",
                      default = "1"
                    },
                    {
                      name = "rows",
                      type = "number",
                      description = "The number of vertical segments in the plane.",
                      default = "columns"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "transform",
                      type = "Mat4",
                      description = "The transform of the plane."
                    },
                    {
                      name = "style",
                      type = "DrawStyle",
                      description = "Whether the plane should be drawn filled or outlined.",
                      default = "'fill'"
                    },
                    {
                      name = "columns",
                      type = "number",
                      description = "The number of horizontal segments in the plane.",
                      default = "1"
                    },
                    {
                      name = "rows",
                      type = "number",
                      description = "The number of vertical segments in the plane.",
                      default = "columns"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "points",
              tag = "drawing",
              summary = "Draw points.",
              description = "Draws points.  `Pass:mesh` can also be used to draw points using a `Buffer`.",
              key = "Pass:points",
              module = "lovr.graphics",
              notes = "To change the size of points, set the `pointSize` shader flag in `lovr.graphics.newShader` or write to the `PointSize` variable in the vertex shader.  Points are always the same size on the screen, regardless of distance, and the units are in pixels.",
              variants = {
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate of the first point."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate of the first point."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate of the first point."
                    },
                    {
                      name = "...",
                      type = "*",
                      description = "More points."
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "t",
                      type = "table",
                      description = "A table of numbers or Vec3 objects (not both) representing point positions."
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "v",
                      type = "Vec3",
                      description = "A vector containing the position of the first point to draw."
                    },
                    {
                      name = "...",
                      type = "*",
                      description = "More points."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "pop",
              tag = "transform",
              summary = "Pop one of the stacks.",
              description = "Pops the transform or render state stack, restoring it to the state it was in when it was last pushed.",
              key = "Pass:pop",
              module = "lovr.graphics",
              notes = "If a stack is popped without a corresponding push, the stack \"underflows\" which causes an error.",
              related = {
                "Pass:push",
                "StackType"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "stack",
                      type = "StackType",
                      description = "The type of stack to pop.",
                      default = "'transform'"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "push",
              tag = "transform",
              summary = "Push state onto a stack.",
              description = "Saves a copy of the transform or render states.  Further changes can be made to the transform or render states, and afterwards `Pass:pop` can be used to restore the original state.  Pushes and pops can be nested, but it's an error to pop without a corresponding push.",
              key = "Pass:push",
              module = "lovr.graphics",
              notes = "Each stack has a limit of the number of copies it can store.  There can be 16 transforms and 4 render states saved.\n\nThe `state` stack does not save the camera info or shader variables changed with `Pass:send`.",
              related = {
                "Pass:pop",
                "StackType"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "stack",
                      type = "StackType",
                      description = "The type of stack to push.",
                      default = "'transform'"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "reset",
              tag = "pass-misc",
              summary = "Reset the Pass.",
              description = "Resets the Pass, clearing all of its draws and computes and resetting all of its state to the default values.",
              key = "Pass:reset",
              module = "lovr.graphics",
              notes = "The following things won't be reset:\n\n- Pass canvas, set with `Pass:setCanvas`.\n- Pass clears, set with `Pass:setClear`.\n- The tally buffer, set with `Pass:setTallyBuffer`.",
              variants = {
                {
                  arguments = {},
                  returns = {}
                }
              }
            },
            {
              name = "rotate",
              tag = "transform",
              summary = "Rotate the coordinate system.",
              description = "Rotates the coordinate system.",
              key = "Pass:rotate",
              module = "lovr.graphics",
              related = {
                "Pass:translate",
                "Pass:scale",
                "Pass:transform",
                "Pass:origin",
                "Pass:push",
                "Pass:pop"
              },
              variants = {
                {
                  description = "Rotate the coordinate system using numbers.",
                  arguments = {
                    {
                      name = "angle",
                      type = "number",
                      description = "The amount to rotate the coordinate system by, in radians."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    }
                  },
                  returns = {}
                },
                {
                  description = "Rotate the coordinate system using a quaternion.",
                  arguments = {
                    {
                      name = "rotation",
                      type = "Quat",
                      description = "A quaternion containing the rotation to apply."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "roundrect",
              tag = "drawing",
              summary = "Draw a rounded rectangle.",
              description = "Draws a rounded rectangle.",
              key = "Pass:roundrect",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate of the center of the rectangle.",
                      default = "0"
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate of the center of the rectangle.",
                      default = "0"
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate of the center of the rectangle.",
                      default = "0"
                    },
                    {
                      name = "width",
                      type = "number",
                      description = "The width of the rectangle.",
                      default = "1"
                    },
                    {
                      name = "height",
                      type = "number",
                      description = "The height of the rectangle.",
                      default = "1"
                    },
                    {
                      name = "thickness",
                      type = "number",
                      description = "The thickness of the rectangle.",
                      default = "1"
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The rotation of the rectangle around its rotation axis, in radians.",
                      default = "0"
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation.",
                      default = "0"
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation.",
                      default = "1"
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation.",
                      default = "0"
                    },
                    {
                      name = "radius",
                      type = "number",
                      description = "The radius of the rectangle corners.  If the radius is zero or negative, the rectangle will have sharp corners.",
                      default = "0"
                    },
                    {
                      name = "segments",
                      type = "number",
                      description = "The number of circular segments to use for each corner.  This increases the smoothness, but increases the number of vertices in the mesh.",
                      default = "8"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "position",
                      type = "Vec3",
                      description = "The position of the rectangle."
                    },
                    {
                      name = "size",
                      type = "Vec3",
                      description = "The size of the rectangle (width, height, thickness)."
                    },
                    {
                      name = "orientation",
                      type = "Quat",
                      description = "The orientation of the rectangle."
                    },
                    {
                      name = "radius",
                      type = "number",
                      description = "The radius of the rectangle corners.  If the radius is zero or negative, the rectangle will have sharp corners.",
                      default = "0"
                    },
                    {
                      name = "segments",
                      type = "number",
                      description = "The number of circular segments to use for each corner.  This increases the smoothness, but increases the number of vertices in the mesh.",
                      default = "8"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "transform",
                      type = "Mat4",
                      description = "The transform of the rectangle."
                    },
                    {
                      name = "radius",
                      type = "number",
                      description = "The radius of the rectangle corners.  If the radius is zero or negative, the rectangle will have sharp corners.",
                      default = "0"
                    },
                    {
                      name = "segments",
                      type = "number",
                      description = "The number of circular segments to use for each corner.  This increases the smoothness, but increases the number of vertices in the mesh.",
                      default = "8"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "scale",
              tag = "transform",
              summary = "Scale the coordinate system.",
              description = "Scales the coordinate system.",
              key = "Pass:scale",
              module = "lovr.graphics",
              related = {
                "Pass:translate",
                "Pass:rotate",
                "Pass:transform",
                "Pass:origin",
                "Pass:push",
                "Pass:pop"
              },
              variants = {
                {
                  description = "Scale the coordinate system using numbers.",
                  arguments = {
                    {
                      name = "sx",
                      type = "number",
                      description = "The x component of the scale."
                    },
                    {
                      name = "sy",
                      type = "number",
                      description = "The y component of the scale.",
                      default = "sx"
                    },
                    {
                      name = "sz",
                      type = "number",
                      description = "The z component of the scale.",
                      default = "sx"
                    }
                  },
                  returns = {}
                },
                {
                  description = "Scale the coordinate system using a vector.",
                  arguments = {
                    {
                      name = "scale",
                      type = "Vec3",
                      description = "The scale to apply."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "send",
              tag = "shaders",
              summary = "Set the value of a shader variable.",
              description = "Sends a value to a variable in the Pass's active `Shader`.  The active shader is changed using `Pass:setShader`.",
              key = "Pass:send",
              module = "lovr.graphics",
              examples = {
                {
                  code = "function lovr.load()\n  shader = lovr.graphics.newShader([[\n    layout(set = 2, binding = 0) uniform sampler mySampler;\n    layout(set = 2, binding = 1) uniform Colors { vec4 colors[256]; };\n    layout(set = 2, binding = 2) uniform texture2D rocks;\n\n    Constants {\n      uint constant;\n    };\n\n    vec4 lovrmain() {\n      return DefaultPosition;\n    }\n  ]], 'unlit')\n\n  clampler = lovr.graphics.newSampler({ wrap = 'clamp' })\n  colorBuffer = lovr.graphics.newBuffer(256, 'vec4')\n  rockTexture = lovr.graphics.newTexture('rocks.jpg')\nend\n\nfunction lovr.draw(pass)\n  pass:setShader(shader)\n  pass:send('mySampler', clampler)\n  pass:send('Colors', colorBuffer)\n  pass:send('rocks', rockTexture)\n  pass:send('constant', 42)\n  -- Draw\nend"
                }
              },
              notes = "Shader variables can be in different \"sets\".  Variables changed by this function must be in set #2, because LÖVR uses set #0 and set #1 internally.\n\nThe new value will persist until a new shader is set that uses a different \"type\" for the binding number of the variable.  See `Pass:setShader` for more details.",
              variants = {
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the Shader variable."
                    },
                    {
                      name = "buffer",
                      type = "Buffer",
                      description = "The Buffer to assign."
                    },
                    {
                      name = "offset",
                      type = "number",
                      description = "An offset from the start of the buffer where data will be read, in bytes.",
                      default = "0"
                    },
                    {
                      name = "extent",
                      type = "number",
                      description = "The number of bytes that will be available for reading.  If zero, as much data as possible will be bound, depending on the offset, buffer size, and the `uniformBufferRange` or `storageBufferRange` limit.",
                      default = "0"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the Shader variable."
                    },
                    {
                      name = "texture",
                      type = "Texture",
                      description = "The Texture to assign."
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the Shader variable."
                    },
                    {
                      name = "sampler",
                      type = "Sampler",
                      description = "The Sampler to assign."
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the Shader variable."
                    },
                    {
                      name = "constant",
                      type = "*",
                      description = "Numbers, vectors, or tables to assign to the constant or uniform buffer."
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "binding",
                      type = "number",
                      description = "The binding number of the Shader variable."
                    },
                    {
                      name = "buffer",
                      type = "Buffer",
                      description = "The Buffer to assign."
                    },
                    {
                      name = "offset",
                      type = "number",
                      description = "An offset from the start of the buffer where data will be read, in bytes.",
                      default = "0"
                    },
                    {
                      name = "extent",
                      type = "number",
                      description = "The number of bytes that will be available for reading.  If zero, as much data as possible will be bound, depending on the offset, buffer size, and the `uniformBufferRange` or `storageBufferRange` limit.",
                      default = "0"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "binding",
                      type = "number",
                      description = "The binding number of the Shader variable."
                    },
                    {
                      name = "texture",
                      type = "Texture",
                      description = "The Texture to assign."
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "binding",
                      type = "number",
                      description = "The binding number of the Shader variable."
                    },
                    {
                      name = "sampler",
                      type = "Sampler",
                      description = "The Sampler to assign."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setAlphaToCoverage",
              tag = "pipeline",
              summary = "Enable or disable alpha to coverage.",
              description = "Sets whether alpha to coverage is enabled.  Alpha to coverage factors the alpha of a pixel into antialiasing calculations.  It can be used to get antialiased edges on textures with transparency.  It's often used for foliage.",
              key = "Pass:setAlphaToCoverage",
              module = "lovr.graphics",
              notes = "By default, alpha to coverage is disabled.",
              variants = {
                {
                  arguments = {
                    {
                      name = "enable",
                      type = "boolean",
                      description = "Whether alpha to coverage should be enabled."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setBlendMode",
              tag = "pipeline",
              summary = "Set the blend mode.",
              description = "Sets the blend mode.  When a pixel is drawn, the blend mode controls how it is mixed with the color and alpha of the pixel underneath it.",
              key = "Pass:setBlendMode",
              module = "lovr.graphics",
              notes = "The default blend mode is `alpha` with the `alphamultiply` alpha mode.",
              variants = {
                {
                  description = "Sets the blend mode for all canvas textures.",
                  arguments = {
                    {
                      name = "blend",
                      type = "BlendMode",
                      description = "The blend mode."
                    },
                    {
                      name = "alphaBlend",
                      type = "BlendAlphaMode",
                      description = "The alpha blend mode, used to control premultiplied alpha."
                    }
                  },
                  returns = {}
                },
                {
                  description = "Disables blending.  When something is drawn, its pixel colors will replace any existing color in the target texture.  This can work okay for opaque objects, but won't render text or transparency properly.",
                  arguments = {},
                  returns = {}
                },
                {
                  description = "Sets the blend mode for a single canvas texture.",
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the canvas texture that will use the new blend mode."
                    },
                    {
                      name = "blend",
                      type = "BlendMode",
                      description = "The blend mode."
                    },
                    {
                      name = "alphaBlend",
                      type = "BlendAlphaMode",
                      description = "The alpha blend mode, used to control premultiplied alpha."
                    }
                  },
                  returns = {}
                },
                {
                  description = "Disables blending for a single canvas texture.",
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the canvas texture that will use the new blend mode."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setCanvas",
              tag = "canvas",
              summary = "Set the Pass's canvas.",
              description = "Sets the Pass's canvas.  The canvas is a set of textures that the Pass will draw to when it's submitted, along with configuration for the depth buffer and antialiasing.",
              key = "Pass:setCanvas",
              module = "lovr.graphics",
              notes = "Changing the canvas will reset the pass, as though `Pass:reset` was called.\n\nAll textures must have the same dimensions, layer counts, and multisample counts.  They also must have been created with the `render` usage flag.\n\nThe number of layers in the textures determines how many views (cameras) the pass has.  Each draw will be rendered to all texture layers, as seen from the corresponding camera.  For example, VR rendering will use a canvas texture with 2 layers, one for each eye.\n\nTo render to a specific mipmap level or layer of a texture, use texture views (`Texture:newView`).\n\nMipmaps will be regenerated for all of canvas textures at the end of a render pass.\n\nIf the Pass has multiple color textures, a fragment shader should be used to write a different color to each texture.  Here's an example that writes red to the first texture and blue to the second texture:\n\n    // Declare an output variable for the second texture\n    layout(location = 1) out vec4 secondColor;\n\n    vec4 lovrmain() {\n      secondColor = vec4(0, 0, 1, 1);\n      return vec4(1, 0, 0, 1);\n    }",
              related = {
                "Pass:getClear",
                "Pass:setClear",
                "Pass:getWidth",
                "Pass:getHeight",
                "Pass:getDimensions"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "...textures",
                      type = "Texture",
                      description = "One or more color textures the pass will render to."
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "canvas",
                      type = "table",
                      description = "The canvas.  Each numeric key is a color texture to render to (up to 4), along with the following keys to control depth buffer and antialiasing settings:",
                      table = {
                        {
                          name = "depth",
                          type = "*",
                          description = "A `Texture` or `TextureFormat` with the depth buffer.",
                          default = "d32f"
                        },
                        {
                          name = "samples",
                          type = "number",
                          description = "The number of multisamples used for antialiasing (either 1 or 4).",
                          default = "4"
                        }
                      }
                    }
                  },
                  returns = {}
                },
                {
                  description = "Disable the canvas.  Any draws in the Pass will be skipped when it is submitted (compute shaders will still run though).",
                  arguments = {},
                  returns = {}
                }
              }
            },
            {
              name = "setClear",
              tag = "canvas",
              summary = "Set the clear values of the Pass.",
              description = "Sets the clear values of the pass.  This controls the initial colors of the canvas texture pixels at the beginning of the render pass.  For each color texture, it can be one of the following:\n\n- A specific RGBA color value (or number for the depth texture).\n- `true`, to do a \"fast clear\" to undefined values.  This is useful if the pass is going to end\n  up drawing to all of the texture's pixels.\n- `false`, to avoid clearing and load the texture's existing pixels.  This can be slow on mobile\n  GPUs.",
              key = "Pass:setClear",
              module = "lovr.graphics",
              notes = "If the depth clear is not given, it will be set to 0.\n\nAll clear colors will default to transparent black (all zeros) when the Pass is created.",
              related = {
                "Pass:setCanvas",
                "Texture:clear"
              },
              variants = {
                {
                  description = "Set the clear color for all color textures, using a hexcode.",
                  arguments = {
                    {
                      name = "hex",
                      type = "number",
                      description = "A hexcode color to clear all color textures to."
                    }
                  },
                  returns = {}
                },
                {
                  description = "Set the clear color for all color textures, using numbers.",
                  arguments = {
                    {
                      name = "r",
                      type = "number",
                      description = "The red component of the clear color."
                    },
                    {
                      name = "g",
                      type = "number",
                      description = "The green component of the clear color."
                    },
                    {
                      name = "b",
                      type = "number",
                      description = "The blue component of the clear color."
                    },
                    {
                      name = "a",
                      type = "number",
                      description = "The alpha component of the clear color.",
                      default = "1.0"
                    }
                  },
                  returns = {}
                },
                {
                  description = "Set the clear color for all color textures, using a boolean.",
                  arguments = {
                    {
                      name = "clear",
                      type = "boolean",
                      description = "Whether color textures should be cleared."
                    }
                  },
                  returns = {}
                },
                {
                  description = "Set the clear color for all color textures using a table, or set clear values for individual textures.",
                  arguments = {
                    {
                      name = "t",
                      type = "table",
                      description = "A table of clear values.  This can be a table of 4 numbers to use for all color textures, or it can be a list of boolean and/or RGBA tables to use for each individual color texture.  It can also have a `depth` key with a boolean/number for the depth texture's clear."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setColor",
              tag = "pipeline",
              summary = "Set the color.",
              description = "Sets the color used for drawing.  Color components are from 0 to 1.",
              key = "Pass:setColor",
              module = "lovr.graphics",
              notes = "The default color is `(1, 1, 1, 1)`.",
              variants = {
                {
                  arguments = {
                    {
                      name = "r",
                      type = "number",
                      description = "The red component of the color."
                    },
                    {
                      name = "g",
                      type = "number",
                      description = "The green component of the color."
                    },
                    {
                      name = "b",
                      type = "number",
                      description = "The blue component of the color."
                    },
                    {
                      name = "a",
                      type = "number",
                      description = "The alpha component of the color.",
                      default = "1.0"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "t",
                      type = "table",
                      descriptioin = "A table of 3 or 4 color components."
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "hex",
                      type = "number",
                      description = "A hexcode."
                    },
                    {
                      name = "a",
                      type = "number",
                      description = "The alpha component of the color.",
                      default = "1.0"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setColorWrite",
              tag = "pipeline",
              summary = "Change the color channels affected by drawing.",
              description = "Sets the color channels affected by drawing, on a per-channel basis.  Disabling color writes is often used to render to the depth or stencil buffer without affecting existing pixel colors.",
              key = "Pass:setColorWrite",
              module = "lovr.graphics",
              notes = "By default, color writes are enabled for all channels.",
              related = {
                "Pass:setDepthWrite",
                "Pass:setStencilWrite"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "enable",
                      type = "boolean",
                      description = "Whether all color components should be affected by draws."
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "r",
                      type = "boolean",
                      description = "Whether the red component should be affected by draws."
                    },
                    {
                      name = "g",
                      type = "boolean",
                      description = "Whether the green component should be affected by draws."
                    },
                    {
                      name = "b",
                      type = "boolean",
                      description = "Whether the blue component should be affected by draws."
                    },
                    {
                      name = "a",
                      type = "boolean",
                      description = "Whether the alpha component should be affected by draws."
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the canvas texture to update."
                    },
                    {
                      name = "enable",
                      type = "boolean",
                      description = "Whether all color components should be affected by draws."
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the canvas texture to update."
                    },
                    {
                      name = "r",
                      type = "boolean",
                      description = "Whether the red component should be affected by draws."
                    },
                    {
                      name = "g",
                      type = "boolean",
                      description = "Whether the green component should be affected by draws."
                    },
                    {
                      name = "b",
                      type = "boolean",
                      description = "Whether the blue component should be affected by draws."
                    },
                    {
                      name = "a",
                      type = "boolean",
                      description = "Whether the alpha component should be affected by draws."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setCullMode",
              tag = "pipeline",
              summary = "Control triangle face culling.",
              description = "Sets whether the front or back faces of triangles are culled.",
              key = "Pass:setCullMode",
              module = "lovr.graphics",
              notes = "The default cull mode is `none`.",
              related = {
                "Pass:setViewCull",
                "Pass:setWinding"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "mode",
                      type = "CullMode",
                      description = "Whether `front` faces, `back` faces, or `none` of the faces should be culled.",
                      default = "'none'"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setDepthClamp",
              tag = "pipeline",
              summary = "Enable or disable depth clamp.",
              description = "Enables or disables depth clamp.  Normally, when pixels fall outside of the clipping planes, they are clipped (not rendered).  Depth clamp will instead render these pixels, clamping their depth on to the clipping planes.",
              key = "Pass:setDepthClamp",
              module = "lovr.graphics",
              notes = "This isn\\'t supported on all GPUs.  Use the `depthClamp` feature of `lovr.graphics.getFeatures` to check for support.  If depth clamp is enabled when unsupported, it will silently fall back to depth clipping.\n\nDepth clamping is not enabled by default.",
              related = {
                "Pass:setDepthTest",
                "Pass:setDepthWrite",
                "Pass:setDepthOffset"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "enable",
                      type = "boolean",
                      description = "Whether depth clamp should be enabled."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setDepthOffset",
              tag = "pipeline",
              summary = "Configure the depth offset.",
              description = "Set the depth offset.  This is a constant offset added to the depth value of pixels.  It can be used to fix Z fighting when rendering decals or other nearly-overlapping objects.",
              key = "Pass:setDepthOffset",
              module = "lovr.graphics",
              notes = "The default depth offset is zero for both values.",
              related = {
                "Pass:setDepthTest",
                "Pass:setDepthWrite"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "offset",
                      type = "number",
                      description = "The depth offset.",
                      default = "0.0"
                    },
                    {
                      name = "sloped",
                      type = "number",
                      description = "The sloped depth offset.",
                      default = "0.0"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setDepthTest",
              tag = "pipeline",
              summary = "Configure the depth test.",
              description = "Sets the depth test.",
              key = "Pass:setDepthTest",
              module = "lovr.graphics",
              notes = "When using LÖVR's default projection (reverse Z with infinite far plane) the default depth test is `gequal`, depth values of 0.0 are on the far plane and depth values of 1.0 are on the near plane, closer to the camera.\n\nA depth buffer must be present to use the depth test, but this is enabled by default.",
              related = {
                "Pass:setDepthWrite",
                "Pass:setDepthOffset",
                "Pass:setDepthClamp",
                "Pass:setStencilTest"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "test",
                      type = "CompareMode",
                      description = "The new depth test to use."
                    }
                  },
                  returns = {}
                },
                {
                  description = "Disable the depth test.",
                  arguments = {},
                  returns = {}
                }
              }
            },
            {
              name = "setDepthWrite",
              tag = "pipeline",
              summary = "Set whether draws write to the depth buffer.",
              description = "Sets whether draws write to the depth buffer.  When a pixel is drawn, if depth writes are enabled and the pixel passes the depth test, the depth buffer will be updated with the pixel's depth value.",
              key = "Pass:setDepthWrite",
              module = "lovr.graphics",
              notes = "The default depth write is `true`.",
              related = {
                "Pass:setStencilWrite",
                "Pass:setColorWrite",
                "Pass:setDepthTest"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "write",
                      type = "boolean",
                      description = "Whether the depth buffer should be affected by draws."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setFont",
              tag = "pipeline",
              summary = "Set the font.",
              description = "Sets the font used for `Pass:text`.",
              key = "Pass:setFont",
              module = "lovr.graphics",
              related = {
                "Pass:text",
                "lovr.graphics.newFont",
                "lovr.graphics.getDefaultFont"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "font",
                      type = "Font",
                      description = "The Font to use when rendering text."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setMaterial",
              tag = "pipeline",
              summary = "Set the material.",
              description = "Sets the material.  This will apply to most drawing, except for text, skyboxes, and models, which use their own materials.",
              key = "Pass:setMaterial",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {
                    {
                      name = "material",
                      type = "Material",
                      description = "The material to use for drawing."
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "texture",
                      type = "Texture",
                      description = "The texture to use as the material."
                    }
                  },
                  returns = {}
                },
                {
                  description = "Use the default material.",
                  arguments = {},
                  returns = {}
                }
              }
            },
            {
              name = "setMeshMode",
              tag = "pipeline",
              summary = "Change the way vertices are connected together.",
              description = "Changes the way vertices are connected together when drawing using `Pass:mesh`.",
              key = "Pass:setMeshMode",
              module = "lovr.graphics",
              notes = "The default mesh mode is `triangles`.",
              variants = {
                {
                  arguments = {
                    {
                      name = "mode",
                      type = "DrawMode",
                      description = "The mesh mode to use."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setProjection",
              tag = "camera",
              summary = "Set the field of view.",
              description = "Sets the projection for a single view.  4 field of view angles can be used, similar to the field of view returned by `lovr.headset.getViewAngles`.  Alternatively, a projection matrix can be used for other types of projections like orthographic, oblique, etc.\n\nUp to 6 views are supported.  The Pass returned by `lovr.headset.getPass` will have its views automatically configured to match the headset.",
              key = "Pass:setProjection",
              module = "lovr.graphics",
              notes = "A far clipping plane of 0.0 can be used for an infinite far plane with reversed Z range.  This is the default because it improves depth precision and reduces Z fighting.  Using a non-infinite far plane requires the depth buffer to be cleared to 1.0 instead of 0.0 and the default depth test to be changed to `lequal` instead of `gequal`.",
              related = {
                "lovr.headset.getViewAngles",
                "lovr.headset.getViewCount",
                "Pass:getViewPose",
                "Pass:setViewPose"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "view",
                      type = "number",
                      description = "The index of the view to update."
                    },
                    {
                      name = "left",
                      type = "number",
                      description = "The left field of view angle, in radians."
                    },
                    {
                      name = "right",
                      type = "number",
                      description = "The right field of view angle, in radians."
                    },
                    {
                      name = "up",
                      type = "number",
                      description = "The top field of view angle, in radians."
                    },
                    {
                      name = "down",
                      type = "number",
                      description = "The bottom field of view angle, in radians."
                    },
                    {
                      name = "near",
                      type = "number",
                      description = "The near clipping plane distance, in meters.",
                      default = ".01"
                    },
                    {
                      name = "far",
                      type = "number",
                      description = "The far clipping plane distance, in meters.",
                      default = "0.0"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "view",
                      type = "number",
                      description = "The index of the view to update."
                    },
                    {
                      name = "matrix",
                      type = "Mat4",
                      description = "The projection matrix for the view."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setSampler",
              tag = "pipeline",
              summary = "Set the sampler.",
              description = "Sets the default `Sampler` to use when sampling textures.  It is also possible to send a custom sampler to a shader using `Pass:send` and use that instead, which allows customizing the sampler on a per-texture basis.",
              key = "Pass:setSampler",
              module = "lovr.graphics",
              examples = {
                {
                  code = "function lovr.draw(pass)\n  pass:setSampler('nearest') -- activate minecraft mode\n  pass:setMaterial(rock)\n  pass:cube(x, y, z)\nend"
                }
              },
              notes = "The `getPixel` shader helper function will use this sampler.",
              variants = {
                {
                  arguments = {
                    {
                      name = "filter",
                      type = "FilterMode",
                      description = "The default filter mode to use when sampling textures (the `repeat` wrap mode will be used).",
                      default = "'linear'"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "sampler",
                      type = "Sampler",
                      description = "The default sampler shaders will use when reading from textures."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setScissor",
              tag = "camera",
              summary = "Set the scissor rectangle.",
              description = "Sets the scissor rectangle.  Any pixels outside the scissor rectangle will not be drawn.",
              key = "Pass:setScissor",
              module = "lovr.graphics",
              notes = "The scissor will apply to all draws in a Pass when the pass is submitted, even if this function is called after adding the draws.\n\n`x` and `y` can not be negative.  `w` and `h` must be positive.\n\nBy default, the scissor is disabled and will cover the entire render area.",
              related = {
                "Pass:getViewport",
                "Pass:setViewport"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate of the upper-left corner of the scissor rectangle."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate of the upper-left corner of the scissor rectangle."
                    },
                    {
                      name = "w",
                      type = "number",
                      description = "The width of the scissor rectangle."
                    },
                    {
                      name = "h",
                      type = "number",
                      description = "The height of the scissor rectangle."
                    }
                  },
                  returns = {}
                },
                {
                  description = "Disable the scissor.",
                  arguments = {},
                  returns = {}
                }
              }
            },
            {
              name = "setShader",
              tag = "shaders",
              summary = "Set the active Shader.",
              description = "Sets the active shader.  In a render pass, the Shader will affect all drawing operations until it is changed again.  In a compute pass, the Shader will be run when `Pass:compute` is called.",
              key = "Pass:setShader",
              module = "lovr.graphics",
              notes = "Changing the shader will preserve resource bindings (the ones set using `Pass:send`) **unless** the new shader declares a resource for a binding number using a different type than the current shader.  In this case, the resource \"type\" means one of the following:\n\n- Uniform buffer (`uniform`).\n- Storage buffer (`buffer`).\n- Sampled texture, (`uniform texture<type>`).\n- Storage texture, (`uniform image<type>`).\n- Sampler (`uniform sampler`).\n\nIf the new shader doesn't declare a resource in a particular binding number, any resource there will be preserved.\n\nIf there's a clash in resource types like this, the variable will be \"cleared\".  Using a buffer variable that has been cleared is not well-defined, and may return random data or even crash the GPU.  For textures, white pixels will be returned.  Samplers will use `linear` filtering and the `repeat` wrap mode.\n\nChanging the shader will not clear push constants set in the `Constants` block.",
              related = {
                "Pass:send",
                "Pass:compute"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "shader",
                      type = "Shader",
                      description = "The shader to use."
                    }
                  },
                  returns = {}
                },
                {
                  description = "Use one of the default shaders for drawing.",
                  arguments = {
                    {
                      name = "default",
                      type = "DefaultShader",
                      description = "One of the default shaders to use."
                    }
                  },
                  returns = {}
                },
                {
                  description = "Switch back to using an automatic shader for drawing.",
                  arguments = {},
                  returns = {}
                }
              }
            },
            {
              name = "setStencilTest",
              tag = "pipeline",
              summary = "Configure the stencil test.",
              description = "Sets the stencil test.  Any pixels that fail the stencil test won't be drawn.  For example, setting the stencil test to `('equal', 1)` will only draw pixels that have a stencil value of 1. The stencil buffer can be modified by drawing while stencil writes are enabled with `lovr.graphics.setStencilWrite`.",
              key = "Pass:setStencilTest",
              module = "lovr.graphics",
              notes = "The stencil test is disabled by default.\n\nSetting the stencil test requires the `Pass` to have a depth texture with the `d24s8` or `d32fs8` format (the `s` means \"stencil\").  The `t.graphics.stencil` and `t.headset.stencil` flags in `lovr.conf` can be used to request a stencil format for the default window and headset passes, respectively.",
              related = {
                "Pass:setStencilWrite",
                "Pass:setDepthTest"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "test",
                      type = "CompareMode",
                      description = "The new stencil test to use."
                    },
                    {
                      name = "value",
                      type = "number",
                      description = "The stencil value to compare against."
                    },
                    {
                      name = "mask",
                      type = "number",
                      description = "An optional mask to apply to stencil values before the comparison.",
                      default = "0xff"
                    }
                  },
                  returns = {}
                },
                {
                  description = "Disable the stencil test.",
                  arguments = {},
                  returns = {}
                }
              }
            },
            {
              name = "setStencilWrite",
              tag = "pipeline",
              summary = "Set whether draws write to the stencil buffer.",
              description = "Sets or disables stencil writes.  When stencil writes are enabled, any pixels drawn will update the values in the stencil buffer using the `StencilAction` set.",
              key = "Pass:setStencilWrite",
              module = "lovr.graphics",
              notes = "By default, stencil writes are disabled.\n\nSetting the stencil test requires the `Pass` to have a depth texture with the `d24s8` or `d32fs8` format (the `s` means \"stencil\").  The `t.graphics.stencil` and `t.headset.stencil` flags in `lovr.conf` can be used to request a stencil format for the default window and headset passes, respectively.",
              related = {
                "Pass:setStencilTest",
                "Pass:setDepthTest"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "action",
                      type = "StencilAction",
                      description = "How pixels drawn will update the stencil buffer."
                    },
                    {
                      name = "value",
                      type = "number",
                      description = "When using the 'replace' action, this is the value to replace with.",
                      default = "1"
                    },
                    {
                      name = "mask",
                      type = "number",
                      description = "An optional mask to apply to stencil values before writing.",
                      default = "0xff"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "actions",
                      type = "table",
                      description = "A list of 3 stencil actions, used when a pixel fails the stencil test, fails the depth test, or passes the stencil test, respectively."
                    },
                    {
                      name = "value",
                      type = "number",
                      description = "When using the 'replace' action, this is the value to replace with.",
                      default = "1"
                    },
                    {
                      name = "mask",
                      type = "number",
                      description = "An optional mask to apply to stencil values before writing.",
                      default = "0xff"
                    }
                  },
                  returns = {}
                },
                {
                  description = "Disables stencil writing.",
                  arguments = {},
                  returns = {}
                }
              }
            },
            {
              name = "setTallyBuffer",
              tag = "tally",
              summary = "Set the Buffer that tally results will be written to.",
              description = "Sets the Buffer where tally results will be written to.  Each time the render pass finishes, the results of all the tallies will be copied to the Buffer at the specified offset.  The buffer can be used in a later pass in a compute shader, or the data in the buffer can be read back using e.g. `Buffer:newReadback`.",
              key = "Pass:setTallyBuffer",
              module = "lovr.graphics",
              notes = "Each tally result is a 4-byte unsigned integer with the number of samples that passed the depth and stencil tests.\n\nIf the buffer doesn't have enough room to store all the tallies, the number of tallies copied will be clamped to the minimum number that will fit.",
              related = {
                "Pass:beginTally",
                "Pass:finishTally"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "buffer",
                      type = "Buffer",
                      description = "The buffer."
                    },
                    {
                      name = "offset",
                      type = "number",
                      description = "A byte offset where results will be written.  Must be a multiple of 4."
                    }
                  },
                  returns = {}
                },
                {
                  description = "Unset the tally buffer.",
                  arguments = {},
                  returns = {}
                }
              }
            },
            {
              name = "setViewCull",
              tag = "pipeline",
              summary = "Enable or disable view frustum culling.",
              description = "Enables or disables view frustum culling.  When enabled, if an object is drawn outside of the camera view, the draw will be skipped.  This can improve performance.",
              key = "Pass:setViewCull",
              module = "lovr.graphics",
              notes = "View frustum culling is disabled by default.\n\nObjects will be culled against all views in the Pass.  The pose and projection for these views is controlled using `Pass:setViewPose` and `Pass:setProjection`.\n\nView frustum culling will increase CPU usage, but will reduce GPU usage depending on how many objects get culled and how many vertices they have.\n\nFor most scenes that draw objects all around the camera, frustum culling will usually result in large speedups.  However, it's always good to measure to be sure.  For example, if every object drawn is in view, then frustum culling will only make things slower, because LÖVR will spend time checking if objects are in view without actually culling any of them.\n\n`Pass:getStats` will return `draws` and `drawsCulled` fields.  The `submitTime` and `gpuTime` fields (with `lovr.graphics.setTimingEnabled`) are a good way to measure the impact of culling.\n\nTo cull an object against a view frustum, LÖVR needs to know the object's bounding box.  The following types of draws have bounding boxes:\n\n- `Pass:plane`\n- `Pass:roundrect`\n- `Pass:cube`\n- `Pass:box`\n- `Pass:circle`\n- `Pass:sphere`\n- `Pass:cylinder`\n- `Pass:cone`\n- `Pass:capsule`\n- `Pass:torus`\n- `Pass:draw` (see notes below for `Model` and `Mesh` objects)\n\nThe following draws do **not** currently have bounding boxes, and will not be culled:\n\n- `Pass:points`\n- `Pass:line`\n- `Pass:text`\n- `Pass:skybox`\n- `Pass:fill`\n- `Pass:mesh`\n\n`Model` objects only compute their bounding box when they're loaded, using the initial node transforms. If a node's transform changes, either manually with `Model:setNodeTransform` or from an animation, then the bounding box will become out of sync and culling will not work properly. View culling should be disabled when rendering these models.\n\n`Mesh` objects will not have a bounding box by default.  Meshes with a storage type of `cpu` can compute their bounding boxes using `Mesh:computeBoundingBox`, which should be called after creating the Mesh or whenever its vertices change.  Any type of Mesh can have its bounding box set manually using `Mesh:setBoundingBox`.  This can be faster than `Mesh:computeBoundingBox` if the bounding box is already known, and is the only way to give a `gpu` Mesh a bounding box.",
              related = {
                "Pass:setCullMode",
                "Mesh:computeBoundingBox",
                "Mesh:setBoundingBox",
                "Pass:setViewPose",
                "Pass:setProjection"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "enable",
                      type = "boolean",
                      description = "Whether frustum culling should be enabled."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setViewPose",
              tag = "camera",
              summary = "Set the camera pose.",
              description = "Sets the pose for a single view.  Objects rendered in this view will appear as though the camera is positioned using the given pose.\n\nUp to 6 views are supported.  When rendering to the headset, views are changed to match the eye positions.  These view poses are also available using `lovr.headset.getViewPose`.",
              key = "Pass:setViewPose",
              module = "lovr.graphics",
              related = {
                "lovr.headset.getViewPose",
                "lovr.headset.getViewCount",
                "Pass:getProjection",
                "Pass:setProjection"
              },
              variants = {
                {
                  description = "Set the pose of the view using numbers.",
                  arguments = {
                    {
                      name = "view",
                      type = "number",
                      description = "The index of the view to update."
                    },
                    {
                      name = "x",
                      type = "number",
                      description = "The x position of the viewer, in meters."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y position of the viewer, in meters."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z position of the viewer, in meters."
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the viewer is rotated around its axis of rotation."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    }
                  },
                  returns = {}
                },
                {
                  description = "Set the pose of the view using vectors.",
                  arguments = {
                    {
                      name = "view",
                      type = "number",
                      description = "The index of the view to update."
                    },
                    {
                      name = "position",
                      type = "Vec3",
                      description = "The position of the viewer, in meters."
                    },
                    {
                      name = "orientation",
                      type = "Quat",
                      description = "The orientation of the viewer."
                    }
                  },
                  returns = {}
                },
                {
                  description = "Set the pose of the view using a matrix.",
                  arguments = {
                    {
                      name = "view",
                      type = "number",
                      description = "The index of the view to update."
                    },
                    {
                      name = "matrix",
                      type = "Mat4",
                      description = "A matrix containing the viewer pose."
                    },
                    {
                      name = "inverted",
                      type = "boolean",
                      description = "Whether the matrix is an inverted pose (a view matrix)."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setViewport",
              tag = "camera",
              summary = "Set the viewport.",
              description = "Sets the viewport.  Everything rendered will get mapped to the rectangle defined by the viewport.  More specifically, this defines the transformation from normalized device coordinates to pixel coordinates.",
              key = "Pass:setViewport",
              module = "lovr.graphics",
              notes = "The viewport will apply to all draws in a Pass when the pass is submitted, even if this function is called after adding the draws.\n\nThe viewport rectangle can use floating point numbers.\n\nA negative viewport height (with a y coordinate equal to the bottom of the viewport) can be used to flip the rendering vertically.\n\nThe default viewport extends from `(0, 0)` to the dimensions of the target textures, with min depth and max depth respectively set to 0 and 1.",
              related = {
                "Pass:getScissor",
                "Pass:setScissor",
                "Pass:getDimensions"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate of the upper-left corner of the viewport."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate of the upper-left corner of the viewport."
                    },
                    {
                      name = "w",
                      type = "number",
                      description = "The width of the viewport.  Must be positive."
                    },
                    {
                      name = "h",
                      type = "number",
                      description = "The height of the viewport.  May be negative."
                    },
                    {
                      name = "dmin",
                      type = "number",
                      description = "The min component of the depth range, between 0 and 1.",
                      default = "0.0"
                    },
                    {
                      name = "dmax",
                      type = "number",
                      description = "The max component of the depth range, between 0 and 1.",
                      default = "1.0"
                    }
                  },
                  returns = {}
                },
                {
                  description = "Disable the viewport.",
                  arguments = {},
                  returns = {}
                }
              }
            },
            {
              name = "setWinding",
              tag = "pipeline",
              summary = "Set the winding direction of triangle vertices.",
              description = "Sets whether vertices in the clockwise or counterclockwise order vertices are considered the \"front\" face of a triangle.  This is used for culling with `Pass:setCullMode`.",
              key = "Pass:setWinding",
              module = "lovr.graphics",
              notes = "The default winding is counterclockwise.  LÖVR's builtin shapes are wound counterclockwise.",
              related = {
                "Pass:setCullMode"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "winding",
                      type = "Winding",
                      description = "Whether triangle vertices are ordered `clockwise` or `counterclockwise`."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setWireframe",
              tag = "pipeline",
              summary = "Enable or disable wireframe rendering.",
              description = "Enables or disables wireframe rendering.  This will draw all triangles as lines while active. It's intended to be used for debugging, since it usually has a performance cost.",
              key = "Pass:setWireframe",
              module = "lovr.graphics",
              notes = "Wireframe rendering is disabled by default.\n\nThere is currently no way to change the thickness of the lines.",
              related = {
                "Pass:setMeshMode"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "enable",
                      type = "boolean",
                      description = "Whether wireframe rendering should be enabled."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "skybox",
              tag = "drawing",
              summary = "Draw a skybox.",
              description = "Draws a skybox.",
              key = "Pass:skybox",
              module = "lovr.graphics",
              notes = "The skybox will be rotated based on the camera rotation.\n\nThe skybox is drawn using a fullscreen triangle.\n\nThe skybox uses a custom shader, so set the shader to `nil` before calling this function (unless explicitly using a custom shader).",
              variants = {
                {
                  arguments = {
                    {
                      name = "skybox",
                      type = "Texture",
                      description = "The skybox to render.  Its `TextureType` can be `cube` to render as a cubemap, or `2d` to render as an equirectangular (spherical) 2D image."
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {},
                  returns = {}
                }
              }
            },
            {
              name = "sphere",
              tag = "drawing",
              summary = "Draw a sphere.",
              description = "Draws a sphere",
              key = "Pass:sphere",
              module = "lovr.graphics",
              notes = "The local origin of the sphere is in its center.",
              variants = {
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate of the center of the sphere.",
                      default = "0"
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate of the center of the sphere.",
                      default = "0"
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate of the center of the sphere.",
                      default = "0"
                    },
                    {
                      name = "radius",
                      type = "number",
                      description = "The radius of the sphere.",
                      default = "1"
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The rotation of the sphere around its rotation axis, in radians.",
                      default = "0"
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation.",
                      default = "0"
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation.",
                      default = "1"
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation.",
                      default = "0"
                    },
                    {
                      name = "longitudes",
                      type = "number",
                      description = "The number of \"horizontal\" segments.",
                      default = "48"
                    },
                    {
                      name = "latitudes",
                      type = "number",
                      description = "The number of \"vertical\" segments.",
                      default = "longitudes / 2"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "position",
                      type = "Vec3",
                      description = "The position of the sphere."
                    },
                    {
                      name = "radius",
                      type = "number",
                      description = "The radius of the sphere.",
                      default = "1"
                    },
                    {
                      name = "orientation",
                      type = "Quat",
                      description = "The orientation of the sphere."
                    },
                    {
                      name = "longitudes",
                      type = "number",
                      description = "The number of \"horizontal\" segments.",
                      default = "48"
                    },
                    {
                      name = "latitudes",
                      type = "number",
                      description = "The number of \"vertical\" segments.",
                      default = "longitudes / 2"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "transform",
                      type = "Mat4",
                      description = "The transform of the sphere."
                    },
                    {
                      name = "longitudes",
                      type = "number",
                      description = "The number of \"horizontal\" segments.",
                      default = "48"
                    },
                    {
                      name = "latitudes",
                      type = "number",
                      description = "The number of \"vertical\" segments.",
                      default = "longitudes / 2"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "text",
              tag = "drawing",
              summary = "Draw text.",
              description = "Draws text.  The font can be changed using `Pass:setFont`.",
              key = "Pass:text",
              module = "lovr.graphics",
              notes = "UTF-8 encoded strings are supported.\n\nNewlines will start a new line of text.  Tabs will be rendered as four spaces.  Carriage returns are ignored.\n\nWith the default font pixel density, a scale of 1.0 makes the text height 1 meter.\n\nThe wrap value does not take into account the text's scale.\n\nText rendering requires a special shader, which will only be automatically used when the active shader is set to `nil`.\n\nBlending should be enabled when rendering text (it's on by default).\n\nThis function can draw up to 16384 visible characters at a time, and will currently throw an error if the string is too long.",
              related = {
                "Pass:setFont",
                "lovr.graphics.getDefaultFont",
                "Pass:setShader",
                "Font:getWidth",
                "Font:getHeight",
                "Font:getLines",
                "Font:getVertices",
                "Font"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "text",
                      type = "string",
                      description = "The text to render."
                    },
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate of the text origin.",
                      default = "0"
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate of the text origin.",
                      default = "0"
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate of the text origin.",
                      default = "0"
                    },
                    {
                      name = "scale",
                      type = "number",
                      description = "The scale of the text (with the default pixel density, units are meters).",
                      default = "1"
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The rotation of the text around its rotation axis, in radians.",
                      default = "0"
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation.",
                      default = "0"
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation.",
                      default = "1"
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation.",
                      default = "0"
                    },
                    {
                      name = "wrap",
                      type = "number",
                      description = "The maximum width of each line in meters (before scale is applied).  When zero, the text will not wrap.",
                      default = "0"
                    },
                    {
                      name = "halign",
                      type = "HorizontalAlign",
                      description = "The horizontal alignment relative to the text origin.",
                      default = "'center'"
                    },
                    {
                      name = "valign",
                      type = "VerticalAlign",
                      description = "The vertical alignment relative to the text origin.",
                      default = "'middle'"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "text",
                      type = "string",
                      description = "The text to render."
                    },
                    {
                      name = "position",
                      type = "Vec3",
                      description = "The position of the text."
                    },
                    {
                      name = "scale",
                      type = "number",
                      description = "The scale of the text (with the default pixel density, units are meters).",
                      default = "1"
                    },
                    {
                      name = "orientation",
                      type = "Quat",
                      description = "The orientation of the text."
                    },
                    {
                      name = "wrap",
                      type = "number",
                      description = "The maximum width of each line in meters (before scale is applied).  When zero, the text will not wrap.",
                      default = "0"
                    },
                    {
                      name = "halign",
                      type = "HorizontalAlign",
                      description = "The horizontal alignment relative to the text origin.",
                      default = "'center'"
                    },
                    {
                      name = "valign",
                      type = "VerticalAlign",
                      description = "The vertical alignment relative to the text origin.",
                      default = "'middle'"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "text",
                      type = "string",
                      description = "The text to render."
                    },
                    {
                      name = "transform",
                      type = "Mat4",
                      description = "The transform of the text."
                    },
                    {
                      name = "wrap",
                      type = "number",
                      description = "The maximum width of each line in meters (before scale is applied).  When zero, the text will not wrap.",
                      default = "0"
                    },
                    {
                      name = "halign",
                      type = "HorizontalAlign",
                      description = "The horizontal alignment relative to the text origin.",
                      default = "'center'"
                    },
                    {
                      name = "valign",
                      type = "VerticalAlign",
                      description = "The vertical alignment relative to the text origin.",
                      default = "'middle'"
                    }
                  },
                  returns = {}
                },
                {
                  description = "Renders multicolor text.",
                  arguments = {
                    {
                      name = "colortext",
                      type = "table",
                      description = "A table of strings with colors to render, in the form `{ color1, string1, color2, string2 }`, where color is a `Vec3`, `Vec4`, hexcode, or table of numbers."
                    },
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate of the text origin.",
                      default = "0"
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate of the text origin.",
                      default = "0"
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate of the text origin.",
                      default = "0"
                    },
                    {
                      name = "scale",
                      type = "number",
                      description = "The scale of the text (with the default pixel density, units are meters).",
                      default = "1"
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The rotation of the text around its rotation axis, in radians.",
                      default = "0"
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation.",
                      default = "0"
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation.",
                      default = "1"
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation.",
                      default = "0"
                    },
                    {
                      name = "wrap",
                      type = "number",
                      description = "The maximum width of each line in meters (before scale is applied).  When zero, the text will not wrap.",
                      default = "0"
                    },
                    {
                      name = "halign",
                      type = "HorizontalAlign",
                      description = "The horizontal alignment relative to the text origin.",
                      default = "'center'"
                    },
                    {
                      name = "valign",
                      type = "VerticalAlign",
                      description = "The vertical alignment relative to the text origin.",
                      default = "'middle'"
                    }
                  },
                  returns = {}
                },
                {
                  description = "Renders multicolor text.",
                  arguments = {
                    {
                      name = "colortext",
                      type = "table",
                      description = "A table of strings with colors to render, in the form `{ color1, string1, color2, string2 }`, where color is a `Vec3`, `Vec4`, hexcode, or table of numbers."
                    },
                    {
                      name = "position",
                      type = "Vec3",
                      description = "The position of the text."
                    },
                    {
                      name = "scale",
                      type = "number",
                      description = "The scale of the text (with the default pixel density, units are meters).",
                      default = "1"
                    },
                    {
                      name = "orientation",
                      type = "Quat",
                      description = "The orientation of the text."
                    },
                    {
                      name = "wrap",
                      type = "number",
                      description = "The maximum width of each line in meters (before scale is applied).  When zero, the text will not wrap.",
                      default = "0"
                    },
                    {
                      name = "halign",
                      type = "HorizontalAlign",
                      description = "The horizontal alignment relative to the text origin.",
                      default = "'center'"
                    },
                    {
                      name = "valign",
                      type = "VerticalAlign",
                      description = "The vertical alignment relative to the text origin.",
                      default = "'middle'"
                    }
                  },
                  returns = {}
                },
                {
                  description = "Renders multicolor text.",
                  arguments = {
                    {
                      name = "colortext",
                      type = "table",
                      description = "A table of strings with colors to render, in the form `{ color1, string1, color2, string2 }`, where color is a `Vec3`, `Vec4`, hexcode, or table of numbers."
                    },
                    {
                      name = "transform",
                      type = "Mat4",
                      description = "The transform of the text."
                    },
                    {
                      name = "wrap",
                      type = "number",
                      description = "The maximum width of each line in meters (before scale is applied).  When zero, the text will not wrap.",
                      default = "0"
                    },
                    {
                      name = "halign",
                      type = "HorizontalAlign",
                      description = "The horizontal alignment relative to the text origin.",
                      default = "'center'"
                    },
                    {
                      name = "valign",
                      type = "VerticalAlign",
                      description = "The vertical alignment relative to the text origin.",
                      default = "'middle'"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "torus",
              tag = "drawing",
              summary = "Draw a donut.",
              description = "Draws a torus.",
              key = "Pass:torus",
              module = "lovr.graphics",
              notes = "The local origin is in the center of the torus, and the torus forms a circle around the local Z axis.",
              variants = {
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate of the center of the torus.",
                      default = "0"
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate of the center of the torus.",
                      default = "0"
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate of the center of the torus.",
                      default = "0"
                    },
                    {
                      name = "radius",
                      type = "number",
                      description = "The radius of the torus.",
                      default = "1"
                    },
                    {
                      name = "thickness",
                      type = "number",
                      description = "The thickness of the torus.",
                      default = "1"
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The rotation of the torus around its rotation axis, in radians.",
                      default = "0"
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation.",
                      default = "0"
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation.",
                      default = "1"
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation.",
                      default = "0"
                    },
                    {
                      name = "tsegments",
                      type = "number",
                      description = "The number of toroidal (circular) segments to render.",
                      default = "64"
                    },
                    {
                      name = "psegments",
                      type = "number",
                      description = "The number of poloidal (tubular) segments to render.",
                      default = "32"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "position",
                      type = "Vec3",
                      description = "The position of the center of the torus."
                    },
                    {
                      name = "scale",
                      type = "Vec3",
                      description = "The size of the torus (x and y scale the radius, z is the thickness)."
                    },
                    {
                      name = "orientation",
                      type = "Quat",
                      description = "The orientation of the torus."
                    },
                    {
                      name = "tsegments",
                      type = "number",
                      description = "The number of toroidal (circular) segments to render.",
                      default = "64"
                    },
                    {
                      name = "psegments",
                      type = "number",
                      description = "The number of poloidal (tubular) segments to render.",
                      default = "32"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "transform",
                      type = "Mat4",
                      description = "The transform of the torus."
                    },
                    {
                      name = "tsegments",
                      type = "number",
                      description = "The number of toroidal (circular) segments to render.",
                      default = "64"
                    },
                    {
                      name = "psegments",
                      type = "number",
                      description = "The number of poloidal (tubular) segments to render.",
                      default = "32"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "transform",
              tag = "transform",
              summary = "Apply a general transform to the coordinate system.",
              description = "Transforms the coordinate system.",
              key = "Pass:transform",
              module = "lovr.graphics",
              related = {
                "Pass:translate",
                "Pass:rotate",
                "Pass:scale",
                "Pass:origin",
                "Pass:push",
                "Pass:pop"
              },
              variants = {
                {
                  description = "Transform the coordinate system using numbers.",
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x component of the translation."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y component of the translation."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z component of the translation."
                    },
                    {
                      name = "sx",
                      type = "number",
                      description = "The x component of the scale."
                    },
                    {
                      name = "sy",
                      type = "number",
                      description = "The y component of the scale."
                    },
                    {
                      name = "sz",
                      type = "number",
                      description = "The z component of the scale."
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The amount to rotate the coordinate system by, in radians."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    }
                  },
                  returns = {}
                },
                {
                  description = "Transform the coordinate system using vector types.",
                  arguments = {
                    {
                      name = "translation",
                      type = "Vec3",
                      description = "The translation to apply."
                    },
                    {
                      name = "scale",
                      type = "Vec3",
                      description = "The scale to apply."
                    },
                    {
                      name = "rotation",
                      type = "Quat",
                      description = "A quaternion containing the rotation to apply."
                    }
                  },
                  returns = {}
                },
                {
                  description = "Transform the coordinate system using a matrix.",
                  arguments = {
                    {
                      name = "transform",
                      type = "Mat4",
                      description = "A matrix containing the transformation to apply."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "translate",
              tag = "transform",
              summary = "Translate the coordinate system.",
              description = "Translates the coordinate system.",
              key = "Pass:translate",
              module = "lovr.graphics",
              notes = "Order matters when scaling, translating, and rotating the coordinate system.",
              related = {
                "Pass:rotate",
                "Pass:scale",
                "Pass:transform",
                "Pass:origin",
                "Pass:push",
                "Pass:pop"
              },
              variants = {
                {
                  description = "Translate the coordinate system using numbers.",
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x component of the translation."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y component of the translation."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z component of the translation."
                    }
                  },
                  returns = {}
                },
                {
                  description = "Translate the coordinate system using a vector.",
                  arguments = {
                    {
                      name = "translation",
                      type = "Vec3",
                      description = "The translation."
                    }
                  },
                  returns = {}
                }
              }
            }
          },
          sections = {
            {
              name = "Drawing",
              tag = "drawing",
              description = "Draw objects and shapes."
            },
            {
              name = "Coordinate System",
              tag = "transform",
              description = "Manipulate the 3D coordinate system."
            },
            {
              name = "Render States",
              tag = "pipeline",
              description = "Set render states that change the way draws appear.  Render states can be saved and restored using `Pass:push` and `Pass:pop`."
            },
            {
              name = "Shaders",
              tag = "shaders",
              description = "Change the shader used for draws or computes, and set variables in the active shader."
            },
            {
              name = "Compute",
              tag = "compute"
            },
            {
              name = "Tally",
              tag = "tally",
              description = "Tallies count the number of pixels that were visible for a draw."
            },
            {
              name = "Camera",
              tag = "camera"
            },
            {
              name = "Canvas",
              tag = "canvas"
            },
            {
              name = "Miscellaneous",
              tag = "pass-misc"
            }
          }
        },
        {
          name = "Readback",
          summary = "An asynchronous read of a GPU resource.",
          description = "Readbacks track the progress of an asynchronous read of a `Buffer` or `Texture`.  Once a Readback is created in a transfer pass, and the transfer pass is submitted, the Readback can be polled for completion or the CPU can wait for it to finish using `Readback:wait`.",
          key = "Readback",
          module = "lovr.graphics",
          constructors = {
            "Buffer:newReadback",
            "Texture:newReadback"
          },
          methods = {
            {
              name = "getBlob",
              summary = "Get Readback's data as a Blob.",
              description = "Returns the Readback's data as a Blob.",
              key = "Readback:getBlob",
              module = "lovr.graphics",
              notes = "If the Readback is reading back a Texture, returns `nil`.",
              related = {
                "Readback:getData",
                "Readback:getImage"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "blob",
                      type = "Blob",
                      description = "The Blob."
                    }
                  }
                }
              }
            },
            {
              name = "getData",
              summary = "Get Readback's data as a table.",
              description = "Returns the data from the Readback, as a table.  See `Buffer:getData` for the way the table is structured.",
              key = "Readback:getData",
              module = "lovr.graphics",
              notes = "This returns `nil` for readbacks of `Texture` objects.",
              related = {
                "Readback:getBlob",
                "Readback:getImage"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "data",
                      type = "table",
                      description = "A table containing the data that was read back."
                    }
                  }
                }
              }
            },
            {
              name = "getImage",
              summary = "Get Readback's data as an Image.",
              description = "Returns the Readback's data as an Image.",
              key = "Readback:getImage",
              module = "lovr.graphics",
              notes = "If the Readback is not reading back a Texture, returns `nil`.",
              related = {
                "Readback:getData",
                "Readback:getBlob"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "image",
                      type = "Image",
                      description = "The Image."
                    }
                  }
                }
              }
            },
            {
              name = "isComplete",
              summary = "Check if a Readback is complete.",
              description = "Returns whether the Readback has completed on the GPU and its data is available.",
              key = "Readback:isComplete",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "complete",
                      type = "boolean",
                      description = "Whether the readback is complete."
                    }
                  }
                }
              }
            },
            {
              name = "wait",
              summary = "Wait for the Readback to finish.",
              description = "Blocks the CPU until the Readback is finished on the GPU.",
              key = "Readback:wait",
              module = "lovr.graphics",
              notes = "If the transfer pass that created the readback has not been submitted yet, no wait will occur and this function will return `false`.",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "waited",
                      type = "boolean",
                      description = "Whether the CPU had to be blocked for waiting."
                    }
                  }
                }
              }
            }
          }
        },
        {
          name = "Sampler",
          summary = "An object that controls how texture pixels are read.",
          description = "Samplers are objects that control how pixels are read from a texture.  They can control whether the pixels are smoothed, whether the texture wraps at the edge of its UVs, and more.\n\nEach `Pass` has a default sampler that will be used by default, which can be changed using `Pass:setSampler`.  Also, samplers can be declared in shaders using the following syntax:\n\n    layout(set = 2, binding = X) uniform sampler mySampler;\n\nA Sampler can be sent to the variable using `Pass:send('mySampler', sampler)`.\n\nThe properties of a Sampler are immutable, and can't be changed after it's created.",
          key = "Sampler",
          module = "lovr.graphics",
          constructors = {
            "lovr.graphics.newSampler"
          },
          methods = {
            {
              name = "getAnisotropy",
              summary = "Get the anisotropy level of the Sampler.",
              description = "Returns the anisotropy level of the Sampler.  Anisotropy smooths out a texture's appearance when viewed at grazing angles.",
              key = "Sampler:getAnisotropy",
              module = "lovr.graphics",
              notes = "Not all GPUs support anisotropy.  The maximum anisotropy level is given by the `anisotropy` limit of `lovr.graphics.getLimits`, which may be zero.  It's very common for the maximum to be 16, however.",
              related = {
                "Sampler:getFilter",
                "Sampler:getWrap",
                "Sampler:getCompareMode",
                "Sampler:getMipmapRange"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "anisotropy",
                      type = "number",
                      description = "The anisotropy level of the sampler."
                    }
                  }
                }
              }
            },
            {
              name = "getCompareMode",
              summary = "Get the compare mode of the Sampler.",
              description = "Returns the compare mode of the Sampler.  This is a feature typically only used for shadow mapping.  Using a sampler with a compare mode requires it to be declared in a shader as a `samplerShadow` instead of a `sampler` variable, and used with a texture that has a depth format.  The result of sampling a depth texture with a shadow sampler is a number between 0 and 1, indicating the percentage of sampled pixels that passed the comparison.",
              key = "Sampler:getCompareMode",
              module = "lovr.graphics",
              related = {
                "Sampler:getFilter",
                "Sampler:getWrap",
                "Sampler:getAnisotropy",
                "Sampler:getMipmapRange"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "compare",
                      type = "CompareMode",
                      description = "The compare mode of the sampler."
                    }
                  }
                }
              }
            },
            {
              name = "getFilter",
              summary = "Get the filter mode of the Sampler.",
              description = "Returns the filter mode of the Sampler.",
              key = "Sampler:getFilter",
              module = "lovr.graphics",
              related = {
                "Sampler:getWrap",
                "Sampler:getCompareMode",
                "Sampler:getAnisotropy",
                "Sampler:getMipmapRange"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "min",
                      type = "FilterMode",
                      description = "The filter mode used when the texture is minified."
                    },
                    {
                      name = "mag",
                      type = "FilterMode",
                      description = "The filter mode used when the texture is magnified."
                    },
                    {
                      name = "mip",
                      type = "FilterMode",
                      description = "The filter mode used to select a mipmap level."
                    }
                  }
                }
              }
            },
            {
              name = "getMipmapRange",
              summary = "Get the mipmap range of the Sampler.",
              description = "Returns the mipmap range of the Sampler.  This is used to clamp the range of mipmap levels that can be accessed from a texture.",
              key = "Sampler:getMipmapRange",
              module = "lovr.graphics",
              related = {
                "Sampler:getFilter",
                "Sampler:getWrap",
                "Sampler:getCompareMode",
                "Sampler:getAnisotropy"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "min",
                      type = "number",
                      description = "The minimum mipmap level that will be sampled (0 is the largest image)."
                    },
                    {
                      name = "max",
                      type = "number",
                      description = "The maximum mipmap level that will be sampled."
                    }
                  }
                }
              }
            },
            {
              name = "getWrap",
              summary = "Get the wrap mode of the Sampler.",
              description = "Returns the wrap mode of the sampler, used to wrap or clamp texture coordinates when the extend outside of the 0-1 range.",
              key = "Sampler:getWrap",
              module = "lovr.graphics",
              related = {
                "Sampler:getFilter",
                "Sampler:getCompareMode",
                "Sampler:getAnisotropy",
                "Sampler:getMipmapRange"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "x",
                      type = "WrapMode",
                      description = "The wrap mode used in the horizontal direction."
                    },
                    {
                      name = "y",
                      type = "WrapMode",
                      description = "The wrap mode used in the vertical direction."
                    },
                    {
                      name = "z",
                      type = "WrapMode",
                      description = "The wrap mode used in the \"z\" direction, for 3D textures only."
                    }
                  }
                }
              }
            }
          }
        },
        {
          name = "Shader",
          summary = "GPU program.",
          description = "Shaders are small GPU programs.  See the `Shaders` guide for a full introduction to Shaders.",
          key = "Shader",
          module = "lovr.graphics",
          constructors = {
            "lovr.graphics.newShader",
            "Shader:clone"
          },
          methods = {
            {
              name = "clone",
              summary = "Clone a Shader.",
              description = "Clones a shader.  This creates an inexpensive copy of it with different flags.  It can be used to create several variants of a shader with different behavior.",
              key = "Shader:clone",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {
                    {
                      name = "source",
                      type = "Shader",
                      description = "The Shader to clone."
                    },
                    {
                      name = "flags",
                      type = "table",
                      description = "The flags used by the clone."
                    }
                  },
                  returns = {
                    {
                      name = "shader",
                      type = "Shader",
                      description = "The new Shader."
                    }
                  }
                }
              }
            },
            {
              name = "getBufferFormat",
              summary = "Get the format of a buffer in the Shader.",
              description = "Returns the format of a buffer declared in shader code.  The return type matches the same syntax used by `lovr.graphics.newBuffer` and `Buffer:getFormat`, so it can be used to quickly create a Buffer that matches a variable from a Shader.",
              key = "Shader:getBufferFormat",
              module = "lovr.graphics",
              examples = {
                {
                  code = "shader = lovr.graphics.newShader([[\n  layout(set = 2, binding = 0) uniform Transforms {\n    mat4 transforms[32];\n  };\n\n  vec4 lovrmain() {\n    return Projection * View * transforms[InstanceIndex] * VertexPosition;\n  }\n]], 'unlit')\n\nlocal format, length = shader:getBufferFormat('Transforms')\nprint(length) --> 32\nprint(format[1].name) --> 'transforms'\nprint(format[1].type) --> 'mat4'\n\n-- Can pass the 2 results directly to newBuffer:\ntransforms = lovr.graphics.newBuffer(shader:getBufferFormat('Transforms'))\n\n-- Or override the length with some initial data:\ntransforms = lovr.graphics.newBuffer(shader:getBufferFormat('Transforms'), objects)"
                }
              },
              notes = "If the buffer only has a single array field, the format will be \"unwrapped\" to an array instead of a single-field struct with an array in it.  Example:\n\n    shader = lovr.graphics.newShader([[\n      layout(set = 0, binding = 0) buffer Numbers {\n        uint numbers[64];\n      };\n\n      void lovrmain(){}\n    ]])\n\n    shader:getBufferFormat('Numbers')\n    -- returns {{ name = 'numbers', type = 'u32' }}, 64\n    -- not     {{ name = 'numbers', type = 'u32', length = 64 }}, 1\n\nSimilarly, if the buffer only has a single struct field, the format will be \"unwrapped\" to the inner struct.  This lets you use a struct for a Buffer's data without having to wrap everything in an extra namespace.  Example:\n\n    shader = lovr.graphics.newShader([[\n      struct HandParams {\n        vec3 pos;\n        float grip;\n      };\n\n      layout(set = 0, binding = 0) buffer Hand {\n        HandParams params;\n      };\n\n      void lovrmain(){}\n    ]])\n\n    shader:getBufferFormat('Hand')\n    -- returns {{ name = 'pos', type = 'vec3' }, { name = 'grip', type = 'float' }}, 1\n    -- not     {{ name = 'params', type = {...}}}, 1",
              related = {
                "lovr.graphics.newBuffer",
                "Buffer:getFormat"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the buffer variable to return the format of."
                    }
                  },
                  returns = {
                    {
                      name = "format",
                      type = "table",
                      description = "A list of fields that match the type declaration of the buffer in the shader code.  Each field has `name`, `type`, and `offset` keys.  If the field is an array, it will have `length` and `stride` keys as well.  The top-level table also has a `stride` key.  Offsets and strides are in bytes."
                    },
                    {
                      name = "length",
                      type = "number",
                      description = "The number of items in the buffer (or 1 if the buffer is not an array)."
                    }
                  }
                }
              }
            },
            {
              name = "getType",
              summary = "Get the type of the Shader.",
              description = "Returns whether the shader is a graphics or compute shader.",
              key = "Shader:getType",
              module = "lovr.graphics",
              related = {
                "Shader:hasStage",
                "lovr.graphics.newShader"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "type",
                      type = "ShaderType",
                      description = "The type of the Shader."
                    }
                  }
                }
              }
            },
            {
              name = "getWorkgroupSize",
              summary = "Get the workgroup size of a compute shader.",
              description = "Returns the workgroup size of a compute shader.  The workgroup size defines how many times a compute shader is invoked for each workgroup dispatched by `Pass:compute`.",
              key = "Shader:getWorkgroupSize",
              module = "lovr.graphics",
              notes = "For example, if the workgroup size is `8x8x1` and `16x16x16` workgroups are dispatched, then the compute shader will run `16 * 16 * 16 * (8 * 8 * 1) = 262144` times.",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x size of a workgroup."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y size of a workgroup."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z size of a workgroup."
                    }
                  }
                }
              }
            },
            {
              name = "hasAttribute",
              summary = "Check if the Shader has a given vertex attribute.",
              description = "Returns whether the Shader has a vertex attribute, by name or location.",
              key = "Shader:hasAttribute",
              module = "lovr.graphics",
              examples = {
                {
                  code = "function lovr.load()\n  shader = lovr.graphics.newShader([[\n    layout(location = 7) in uint coolAttribute;\n\n    vec4 lovrmain() {\n      return DefaultPosition;\n    }\n  ]], [[\n    vec4 lovrmain() {\n      return DefaultColor;\n    }\n  ]])\n\n  print(shader:hasAttribute('coolAttribute')) --> true\n  print(shader:hasAttribute(7)) --> true\n  print(shader:hasAttribute(8)) --> false\nend"
                }
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of an attribute."
                    }
                  },
                  returns = {
                    {
                      name = "exists",
                      type = "boolean",
                      description = "Whether the Shader has the attribute."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "location",
                      type = "number",
                      description = "The location of an attribute."
                    }
                  },
                  returns = {
                    {
                      name = "exists",
                      type = "boolean",
                      description = "Whether the Shader has the attribute."
                    }
                  }
                }
              }
            },
            {
              name = "hasStage",
              summary = "Check if the Shader has a given stage.",
              description = "Returns whether the Shader has a given stage.",
              key = "Shader:hasStage",
              module = "lovr.graphics",
              related = {
                "Shader:getType"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "stage",
                      type = "ShaderStage",
                      description = "The stage."
                    }
                  },
                  returns = {
                    {
                      name = "exists",
                      type = "boolean",
                      description = "Whether the Shader has the stage."
                    }
                  }
                }
              }
            }
          }
        },
        {
          name = "Texture",
          summary = "A multidimensional block of memory on the GPU.",
          description = "Textures are multidimensional blocks of memory on the GPU, contrasted with `Buffer` objects which are one-dimensional.  Textures are used as the destination for rendering operations, and textures loaded from images provide surface data to `Material` objects.",
          key = "Texture",
          module = "lovr.graphics",
          constructors = {
            "lovr.graphics.newTexture",
            "Texture:newView"
          },
          methods = {
            {
              name = "clear",
              tag = "texture-transfer",
              summary = "Clear the Texture to a color.",
              description = "Clears layers and mipmaps in a texture to a given color.\n\nWhen a Texture is being used as a canvas for a `Pass`, the clear color can be set with `Pass:setClear`, which a more efficient way to clear the texture before rendering.",
              key = "Texture:clear",
              module = "lovr.graphics",
              notes = "The texture must have been created with the `transfer` usage to clear it.\n\nThe clear color will be interpreted as a linear color for sRGB formats.",
              related = {
                "Buffer:clear",
                "Texture:setPixels",
                "Pass:setClear"
              },
              variants = {
                {
                  description = "Clear the whole texture to zero (transparent black).",
                  arguments = {},
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "hex",
                      type = "number",
                      description = "The hexcode color to clear to."
                    },
                    {
                      name = "layer",
                      type = "number",
                      description = "The index of the first layer to clear.",
                      default = "1"
                    },
                    {
                      name = "layerCount",
                      type = "number",
                      description = "The number of layers to clear.  If nil, clears the rest of the layers.",
                      default = "nil"
                    },
                    {
                      name = "mipmap",
                      type = "number",
                      description = "The index of the first mipmap to clear.",
                      default = "1"
                    },
                    {
                      name = "mipmapCount",
                      type = "number",
                      description = "The number of mipmaps to clear.  If nil, clears the rest of the mipmaps.",
                      default = "nil"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "r",
                      type = "number",
                      description = "The red component of the clear color."
                    },
                    {
                      name = "g",
                      type = "number",
                      description = "The green component of the clear color."
                    },
                    {
                      name = "b",
                      type = "number",
                      description = "The blue component of the clear color."
                    },
                    {
                      name = "a",
                      type = "number",
                      description = "The alpha component of the clear color."
                    },
                    {
                      name = "layer",
                      type = "number",
                      description = "The index of the first layer to clear.",
                      default = "1"
                    },
                    {
                      name = "layerCount",
                      type = "number",
                      description = "The number of layers to clear.  If nil, clears the rest of the layers.",
                      default = "nil"
                    },
                    {
                      name = "mipmap",
                      type = "number",
                      description = "The index of the first mipmap to clear.",
                      default = "1"
                    },
                    {
                      name = "mipmapCount",
                      type = "number",
                      description = "The number of mipmaps to clear.  If nil, clears the rest of the mipmaps.",
                      default = "nil"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "t",
                      type = "table",
                      description = "A table with color components."
                    },
                    {
                      name = "layer",
                      type = "number",
                      description = "The index of the first layer to clear.",
                      default = "1"
                    },
                    {
                      name = "layerCount",
                      type = "number",
                      description = "The number of layers to clear.  If nil, clears the rest of the layers.",
                      default = "nil"
                    },
                    {
                      name = "mipmap",
                      type = "number",
                      description = "The index of the first mipmap to clear.",
                      default = "1"
                    },
                    {
                      name = "mipmapCount",
                      type = "number",
                      description = "The number of mipmaps to clear.  If nil, clears the rest of the mipmaps.",
                      default = "nil"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "v3",
                      type = "Vec3",
                      description = "A vec3 with the clear color."
                    },
                    {
                      name = "layer",
                      type = "number",
                      description = "The index of the first layer to clear.",
                      default = "1"
                    },
                    {
                      name = "layerCount",
                      type = "number",
                      description = "The number of layers to clear.  If nil, clears the rest of the layers.",
                      default = "nil"
                    },
                    {
                      name = "mipmap",
                      type = "number",
                      description = "The index of the first mipmap to clear.",
                      default = "1"
                    },
                    {
                      name = "mipmapCount",
                      type = "number",
                      description = "The number of mipmaps to clear.  If nil, clears the rest of the mipmaps.",
                      default = "nil"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "v4",
                      type = "Vec4",
                      description = "A vec4 with the clear color."
                    },
                    {
                      name = "layer",
                      type = "number",
                      description = "The index of the first layer to clear.",
                      default = "1"
                    },
                    {
                      name = "layerCount",
                      type = "number",
                      description = "The number of layers to clear.  If nil, clears the rest of the layers.",
                      default = "nil"
                    },
                    {
                      name = "mipmap",
                      type = "number",
                      description = "The index of the first mipmap to clear.",
                      default = "1"
                    },
                    {
                      name = "mipmapCount",
                      type = "number",
                      description = "The number of mipmaps to clear.  If nil, clears the rest of the mipmaps.",
                      default = "nil"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "generateMipmaps",
              tag = "texture-transfer",
              summary = "Regenerate mipmaps for a Texture.",
              description = "Regenerates mipmap levels of a texture.  This downscales pixels from the texture to progressively smaller sizes and saves them.  If the texture is drawn at a smaller scale later, the mipmaps are used, which smooths out the appearance and improves performance.",
              key = "Texture:generateMipmaps",
              module = "lovr.graphics",
              notes = "Mipmaps will automatically be regenerated for textures after rendering to them in a `Pass`. This can be disabled by rendering to a single-level texture view instead.\n\nThe texture must have been created with the `transfer` usage to mipmap it.\n\nThe texture can not be multisampled.\n\nTexture views can not currently be mipmapped.",
              related = {
                "Texture:setPixels",
                "Texture:getMipmapCount"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "base",
                      type = "number",
                      description = "The base mipmap level which will be used to generate subsequent mipmaps.",
                      default = "1"
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "The number of mipmap levels to generate.  If nil, the rest of the mipmaps will be generated.",
                      default = "nil"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "getDimensions",
              tag = "texture-metadata",
              summary = "Get the dimensions of the Texture.",
              description = "Returns the width, height, and depth of the Texture.",
              key = "Texture:getDimensions",
              module = "lovr.graphics",
              related = {
                "Texture:getWidth",
                "Texture:getHeight",
                "Texture:getLayerCount"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "width",
                      type = "number",
                      description = "The width of the Texture."
                    },
                    {
                      name = "height",
                      type = "number",
                      description = "The height of the Texture."
                    },
                    {
                      name = "layers",
                      type = "number",
                      description = "The number of layers in the Texture."
                    }
                  }
                }
              }
            },
            {
              name = "getFormat",
              tag = "texture-metadata",
              summary = "Get the format of the Texture.",
              description = "Returns the format of the texture.",
              key = "Texture:getFormat",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "format",
                      type = "TextureFormat",
                      description = "The format of the Texture."
                    }
                  }
                }
              }
            },
            {
              name = "getHeight",
              tag = "texture-metadata",
              summary = "Get the height of the Texture, in pixels.",
              description = "Returns the height of the Texture, in pixels.",
              key = "Texture:getHeight",
              module = "lovr.graphics",
              related = {
                "Texture:getWidth",
                "Texture:getLayerCount",
                "Texture:getDimensions"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "height",
                      type = "number",
                      description = "The height of the Texture, in pixels."
                    }
                  }
                }
              }
            },
            {
              name = "getLayerCount",
              tag = "texture-metadata",
              summary = "Get the layer count of the Texture.",
              description = "Returns the layer count of the Texture.  2D textures always have 1 layer and cubemaps always have 6 layers.  3D and array textures have a variable number of layers.",
              key = "Texture:getLayerCount",
              module = "lovr.graphics",
              related = {
                "Texture:getWidth",
                "Texture:getHeight",
                "Texture:getDimensions"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "layers",
                      type = "number",
                      description = "The layer count of the Texture."
                    }
                  }
                }
              }
            },
            {
              name = "getMipmapCount",
              tag = "texture-metadata",
              summary = "Get the number of mipmap levels in the Texture.",
              description = "Returns the number of mipmap levels in the Texture.",
              key = "Texture:getMipmapCount",
              module = "lovr.graphics",
              related = {
                "lovr.graphics.newTexture",
                "Sampler:getMipmapRange",
                "Texture:generateMipmaps"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "mipmaps",
                      type = "number",
                      description = "The number of mipmap levels in the Texture."
                    }
                  }
                }
              }
            },
            {
              name = "getParent",
              tag = "texture-view",
              summary = "Get the parent of a texture view.",
              description = "Returns the parent of a Texture view, which is the Texture that it references.  Returns `nil` if the Texture is not a view.",
              key = "Texture:getParent",
              module = "lovr.graphics",
              related = {
                "Texture:isView",
                "Texture:newView"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "parent",
                      type = "Texture",
                      description = "The parent of the texture, or `nil` if the texture is not a view."
                    }
                  }
                }
              }
            },
            {
              name = "getPixels",
              tag = "texture-transfer",
              summary = "Get the pixels of the Texture.",
              description = "Creates and returns a new `Image` object with the current pixels of the Texture.  This function is very very slow because it stalls the CPU until the download is complete.  It should only be used for debugging, non-interactive scripts, etc.  For an asynchronous version that doesn't stall the CPU, see `Texture:newReadback`.",
              key = "Texture:getPixels",
              module = "lovr.graphics",
              notes = "The texture must have been created with the `transfer` usage.\n\nMultisampled textures can not be read back.\n\nIt is not currently possible to read back a texture view.",
              related = {
                "Texture:newReadback"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x offset of the region to download.",
                      default = "0"
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y offset of the region to download.",
                      default = "0"
                    },
                    {
                      name = "layer",
                      type = "number",
                      description = "The index of the layer to download.",
                      default = "1"
                    },
                    {
                      name = "mipmap",
                      type = "number",
                      description = "The index of the mipmap level to download.",
                      default = "1"
                    },
                    {
                      name = "width",
                      type = "number",
                      description = "The width of the pixel rectangle to download.  If nil, the \"rest\" of the width will be used, based on the texture width and x offset.",
                      default = "nil"
                    },
                    {
                      name = "height",
                      type = "number",
                      description = "The height of the pixel rectangle to download.  If nil, the \"rest\" of the height will be used, based on the texture height and y offset.",
                      default = "nil"
                    }
                  },
                  returns = {
                    {
                      name = "image",
                      type = "Image",
                      description = "The new image with the pixels."
                    }
                  }
                }
              }
            },
            {
              name = "getSampleCount",
              tag = "texture-metadata",
              summary = "Get the number of MSAA samples in the Texture.",
              description = "Returns the number of samples in the texture.  Multiple samples are used for multisample antialiasing when rendering to the texture.  Currently, the sample count is either 1 (not antialiased) or 4 (antialiased).",
              key = "Texture:getSampleCount",
              module = "lovr.graphics",
              related = {
                "lovr.graphics.newTexture",
                "Pass:getSampleCount"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "samples",
                      type = "number",
                      description = "The number of samples in the Texture."
                    }
                  }
                }
              }
            },
            {
              name = "getType",
              tag = "texture-metadata",
              summary = "Get the type of the Texture.",
              description = "Returns the type of the texture.",
              key = "Texture:getType",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "type",
                      type = "TextureType",
                      description = "The type of the Texture."
                    }
                  }
                }
              }
            },
            {
              name = "getWidth",
              tag = "texture-metadata",
              summary = "Get the width of the Texture, in pixels.",
              description = "Returns the width of the Texture, in pixels.",
              key = "Texture:getWidth",
              module = "lovr.graphics",
              related = {
                "Texture:getHeight",
                "Texture:getLayerCount",
                "Texture:getDimensions"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "width",
                      type = "number",
                      description = "The width of the Texture, in pixels."
                    }
                  }
                }
              }
            },
            {
              name = "hasUsage",
              tag = "texture-metadata",
              summary = "Check if a Texture was created with a set of usage flags.",
              description = "Returns whether a Texture was created with a set of `TextureUsage` flags.  Usage flags are specified when the Texture is created, and restrict what you can do with a Texture object.  By default, only the `sample` usage is enabled.  Applying a smaller set of usage flags helps LÖVR optimize things better.",
              key = "Texture:hasUsage",
              module = "lovr.graphics",
              related = {
                "lovr.graphics.newTexture"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "...",
                      type = "TextureUsage",
                      description = "One or more usage flags."
                    }
                  },
                  returns = {
                    {
                      name = "supported",
                      type = "boolean",
                      description = "Whether the Texture has all the provided usage flags."
                    }
                  }
                }
              }
            },
            {
              name = "isView",
              tag = "texture-view",
              summary = "Check if a Texture is a texture view.",
              description = "Returns whether a Texture is a texture view, created with `Texture:newView`.",
              key = "Texture:isView",
              module = "lovr.graphics",
              related = {
                "Texture:getParent",
                "Texture:newView"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "view",
                      type = "boolean",
                      description = "Whether the Texture is a texture view."
                    }
                  }
                }
              }
            },
            {
              name = "newReadback",
              tag = "texture-transfer",
              summary = "Read back the contents of the Texture asynchronously.",
              description = "Creates and returns a new `Readback` that will download the pixels in the Texture from VRAM. Once the readback is complete, `Readback:getImage` returns an `Image` with a CPU copy of the data.",
              key = "Texture:newReadback",
              module = "lovr.graphics",
              examples = {
                {
                  description = "Take a screenshot when pressing a key.  This uses an intermediate texture and render pass, to work around the fact that the window/headset textures don't support transfers.",
                  code = "local screenshot = false\nlocal readback\nlocal texture\nlocal pass\n\nfunction lovr.keypressed(key)\n  if key == 'p' then screenshot = true end\nend\n\nfunction lovr.load()\n  local width, height = lovr.headset.getDisplayDimensions()\n\n  texture = lovr.graphics.newTexture(width, height, {\n    usage = { 'render', 'transfer', 'sample' }\n  })\n\n  pass = lovr.graphics.newPass(texture)\nend\n\nfunction lovr.update()\n  pass:reset()\n  for i = 1, lovr.headset.getViewCount() do\n    pass:setViewPose(i, lovr.headset.getViewPose(i))\n    pass:setProjection(i, lovr.headset.getViewAngles(i))\n  end\n  pass:text('hellooo', 0, 1.7, -1, .1)\n  lovr.graphics.submit(pass)\n\n  if screenshot and not readback then\n    readback = texture:newReadback()\n    screenshot = false\n  end\n\n  if readback and readback:isComplete() then\n    local filename = 'screenshot.png'\n    lovr.filesystem.write(filename, readback:getImage():encode())\n    print('saved screenshot to ' .. filename)\n    readback = nil\n  end\nend\n\nfunction lovr.draw(p)\n  p:fill(texture)\nend"
                }
              },
              notes = "The texture must have been created with the `transfer` usage.\n\nMultisampled textures can not be read back.\n\nIt is not currently possible to read back a texture view.",
              related = {
                "Texture:getPixels",
                "Buffer:newReadback"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x offset of the region to download.",
                      default = "0"
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y offset of the region to download.",
                      default = "0"
                    },
                    {
                      name = "layer",
                      type = "number",
                      description = "The index of the layer to download.",
                      default = "1"
                    },
                    {
                      name = "mipmap",
                      type = "number",
                      description = "The index of the mipmap level to download.",
                      default = "1"
                    },
                    {
                      name = "width",
                      type = "number",
                      description = "The width of the pixel rectangle to download.  If nil, the \"rest\" of the width will be used, based on the texture width and x offset.",
                      default = "nil"
                    },
                    {
                      name = "height",
                      type = "number",
                      description = "The height of the pixel rectangle to download.  If nil, the \"rest\" of the height will be used, based on the texture height and y offset.",
                      default = "nil"
                    }
                  },
                  returns = {
                    {
                      name = "readback",
                      type = "Readback",
                      description = "A new Readback object."
                    }
                  }
                }
              }
            },
            {
              name = "newView",
              tag = "texture-view",
              summary = "Create a texture view referencing a parent Texture.",
              description = "Creates a new Texture view.  A texture view does not store any pixels on its own, but instead uses the pixel data of a \"parent\" Texture object.  The width, height, format, sample count, and usage flags all match the parent.  The view may have a different `TextureType` from the parent, and it may reference a subset of the parent texture's layers and mipmap levels.\n\nTexture views can be used as render targets in a render pass and they can be bound to Shaders. They can not currently be used for transfer operations.  They are used for:\n\n- Reinterpretation of texture contents.  For example, a cubemap can be treated as\n  an array texture.\n- Rendering to a particular image or mipmap level of a texture.\n- Binding a particular image or mipmap level to a shader.",
              key = "Texture:newView",
              module = "lovr.graphics",
              related = {
                "Texture:isView",
                "Texture:getParent",
                "lovr.graphics.newTexture"
              },
              variants = {
                {
                  description = "Create a 2D texture view referencing a single layer and mipmap.",
                  arguments = {
                    {
                      name = "layer",
                      type = "number",
                      description = "The index of the first layer in the view.",
                      default = "1"
                    },
                    {
                      name = "mipmap",
                      type = "number",
                      description = "The index of the first mipmap in the view.",
                      default = "1"
                    }
                  },
                  returns = {
                    {
                      name = "view",
                      type = "Texture",
                      description = "The new texture view."
                    }
                  }
                },
                {
                  description = "Create a texture view with an explicit type, layer range, and mipmap range.",
                  arguments = {
                    {
                      name = "type",
                      type = "TextureType",
                      description = "The texture type of the view."
                    },
                    {
                      name = "layer",
                      type = "number",
                      description = "The index of the first layer in the view.",
                      default = "1"
                    },
                    {
                      name = "layerCount",
                      type = "number",
                      description = "The number of layers in the view, or `nil` to use all remaining layers.",
                      default = "nil"
                    },
                    {
                      name = "mipmap",
                      type = "number",
                      description = "The index of the first mipmap in the view.",
                      default = "1"
                    },
                    {
                      name = "mipmapCount",
                      type = "number",
                      description = "The number of mipmaps in the view, or `nil` to use all remaining mipmaps.",
                      default = "nil"
                    }
                  },
                  returns = {
                    {
                      name = "view",
                      type = "Texture",
                      description = "The new texture view."
                    }
                  }
                }
              }
            },
            {
              name = "setPixels",
              tag = "texture-transfer",
              summary = "Replace pixels in the Texture.",
              description = "Sets pixels in the texture.  The source data can be an `Image` with the pixels to upload, or another `Texture` object to copy from.",
              key = "Texture:setPixels",
              module = "lovr.graphics",
              notes = "The destination and source textures must have been created with the `transfer` usage.\n\nMultisampled textures can not be copied.\n\nIt is not currently possible to copy to or from a texture view.\n\nCopying between textures requires them to have the same format.\n\nWhen using different region sizes in a texture-to-texture copy:\n\n- It is not possible to mix 3D with non-3D textures.\n- Not every texture format is supported, use `lovr.graphics.isFormatSupported` to check.",
              related = {
                "Texture:newReadback",
                "Image:paste"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "image",
                      type = "Image",
                      description = "The image to copy to the texture."
                    },
                    {
                      name = "dstx",
                      type = "number",
                      description = "The x offset to copy to.",
                      default = "0"
                    },
                    {
                      name = "dsty",
                      type = "number",
                      description = "The y offset to copy to.",
                      default = "0"
                    },
                    {
                      name = "dstlayer",
                      type = "number",
                      description = "The index of the layer to copy to.",
                      default = "1"
                    },
                    {
                      name = "dstmipmap",
                      type = "number",
                      description = "The index of the mipmap level to copy to.",
                      default = "1"
                    },
                    {
                      name = "srcx",
                      type = "number",
                      description = "The x offset to copy from.",
                      default = "0"
                    },
                    {
                      name = "srcy",
                      type = "number",
                      description = "The y offset to copy from.",
                      default = "0"
                    },
                    {
                      name = "srclayer",
                      type = "number",
                      description = "The index of the layer to copy from.",
                      default = "1"
                    },
                    {
                      name = "srcmipmap",
                      type = "number",
                      description = "The index of the mipmap level to copy from.",
                      default = "1"
                    },
                    {
                      name = "width",
                      type = "number",
                      description = "The width of the region of pixels to copy.  If nil, the maximum possible width will be used, based on the widths of the source/destination and the offset parameters.",
                      default = "nil"
                    },
                    {
                      name = "height",
                      type = "number",
                      description = "The height of the region of pixels to copy.  If nil, the maximum possible height will be used, based on the heights of the source/destination and the offset parameters.",
                      default = "nil"
                    },
                    {
                      name = "layers",
                      type = "number",
                      description = "The number of layers to copy.  If nil, copies as many layers as possible.",
                      default = "nil"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "texture",
                      type = "Texture",
                      description = "The texture to copy from."
                    },
                    {
                      name = "dstx",
                      type = "number",
                      description = "The x offset to copy to.",
                      default = "0"
                    },
                    {
                      name = "dsty",
                      type = "number",
                      description = "The y offset to copy to.",
                      default = "0"
                    },
                    {
                      name = "dstlayer",
                      type = "number",
                      description = "The index of the layer to copy to.",
                      default = "1"
                    },
                    {
                      name = "dstmipmap",
                      type = "number",
                      description = "The index of the mipmap level to copy to.",
                      default = "1"
                    },
                    {
                      name = "srcx",
                      type = "number",
                      description = "The x offset to copy from.",
                      default = "0"
                    },
                    {
                      name = "srcy",
                      type = "number",
                      description = "The y offset to copy from.",
                      default = "0"
                    },
                    {
                      name = "srclayer",
                      type = "number",
                      description = "The index of the layer to copy from.",
                      default = "1"
                    },
                    {
                      name = "srcmipmap",
                      type = "number",
                      description = "The index of the mipmap level to copy from.",
                      default = "1"
                    },
                    {
                      name = "width",
                      type = "number",
                      description = "The width of the region of pixels to copy.  If nil, the maximum possible width will be used, based on the widths of the source/destination and the offset parameters.",
                      default = "nil"
                    },
                    {
                      name = "height",
                      type = "number",
                      description = "The height of the region of pixels to copy.  If nil, the maximum possible height will be used, based on the heights of the source/destination and the offset parameters.",
                      default = "nil"
                    },
                    {
                      name = "layers",
                      type = "number",
                      description = "The number of layers to copy.  If nil, copies as many layers as possible.",
                      default = "nil"
                    },
                    {
                      name = "srcwidth",
                      type = "number",
                      description = "The width of the region in the source texture to copy.  If it doesn't match `width`, the copy will be scaled up or down to fit.",
                      default = "width"
                    },
                    {
                      name = "srcheight",
                      type = "number",
                      description = "The height of the region in the source texture to copy.  If it doesn't match `height`, the copy will be scaled up or down to fit.",
                      default = "width"
                    },
                    {
                      name = "srcdepth",
                      type = "number",
                      description = "The depth of the region in the source texture to copy (`3d` textures only).",
                      default = "layers"
                    },
                    {
                      name = "filter",
                      type = "FilterMode",
                      description = "The filtering mode used to scale the copy when the source and destination sizes don't match.",
                      default = "'linear'"
                    }
                  },
                  returns = {}
                }
              }
            }
          },
          sections = {
            {
              name = "Metadata",
              tag = "texture-metadata"
            },
            {
              name = "Transfers",
              tag = "texture-transfer"
            },
            {
              name = "Texture Views",
              tag = "texture-view"
            }
          }
        }
      },
      sections = {
        {
          name = "Objects",
          tag = "graphics-objects"
        },
        {
          name = "Global State",
          tag = "graphics-global"
        },
        {
          name = "Work Submission",
          tag = "work-submission",
          description = "The only way to get the GPU to do anything is to submit `Pass` objects to it.  LÖVR submits the default pass automatically at the end of `lovr.draw`, but work can also be submitted manually."
        },
        {
          name = "System Info",
          tag = "graphics-misc",
          description = "Information about the GPU hardware and the features it supports."
        }
      }
    },
    {
      name = "headset",
      tag = "modules",
      summary = "Connects to VR hardware.",
      description = "The `lovr.headset` module is where all the magical VR functionality is.  With it, you can access connected VR hardware and get information about the available space the player has.  Note that all units are reported in meters.  Position `(0, 0, 0)` is on the floor in the center of the play area.",
      key = "lovr.headset",
      enums = {
        {
          name = "Device",
          description = "Different types of input devices supported by the `lovr.headset` module.",
          key = "Device",
          module = "lovr.headset",
          notes = "The `grip` pose is used to render an object held in the user's hand.  It's positioned at the surface of the palm.  The X axis of the grip orientation is perpendicular to the palm, pointing away from the left palm or into the right palm.  If you imagine the hand holding a stick, the Z axis will be parallel to the stick.\n\n<img src=\"/img/grip.svg\" width=\"600\" alt=\"Hand Grip Pose\" class=\"flat\"/>\n\n*Image from the [OpenXR Specification](https://registry.khronos.org/OpenXR/specs/1.0/html/xrspec.html#_grip_pose)*\n\n---\n\nThe `point` pose is used to aim or point at objects.  It's usually positioned slightly in front of the hand or controller, and is oriented so the -Z axis points in a natural forward direction.\n\n<img src=\"/img/aim.svg\" width=\"600\" alt=\"Hand Point Pose\" class=\"flat\"/>\n\n*Image from the [OpenXR Specification](https://registry.khronos.org/OpenXR/specs/1.0/html/xrspec.html#_aim_pose)*\n\n---\n\nThe `pinch` pose is a stable point between the thumb and index finger on a hand, or a point slightly in front of a controller.  The -Z axis will point forward, away from the hand.  It's good for precise, close-range interaction.\n\n<img src=\"/img/pinch.svg\" width=\"600\" alt=\"Hand Pinch Pose\" class=\"flat\"/>\n\n*Image from the [OpenXR Specification](https://registry.khronos.org/OpenXR/specs/1.0/html/xrspec.html#_pinch_pose)*\n\n---\n\nThe `poke` pose is a position located at the tip of the index finger, or a point slightly in front of a controller.  The -Z axis will point forward out of the tip of the finger, the +Y axis will be perpendicular to the fingernail.\n\n<img src=\"/img/poke.svg\" width=\"600\" alt=\"Hand Poke Pose\" class=\"flat\"/>\n\n*Image from the [OpenXR Specification](https://registry.khronos.org/OpenXR/specs/1.0/html/xrspec.html#_poke_pose)*\n\n---\n\nThese \"hand pose devices\" do not report any button input with e.g. `lovr.headset.isDown`.  The main `hand/left` and `hand/right` devices should be used for buttons and haptics.",
          related = {
            "DeviceAxis",
            "DeviceButton",
            "lovr.headset.getPose",
            "lovr.headset.getPosition",
            "lovr.headset.getOrientation",
            "lovr.headset.getVelocity",
            "lovr.headset.getAngularVelocity",
            "lovr.headset.getSkeleton",
            "lovr.headset.isTracked",
            "lovr.headset.isDown",
            "lovr.headset.isTouched",
            "lovr.headset.wasPressed",
            "lovr.headset.wasReleased",
            "lovr.headset.getAxis",
            "lovr.headset.vibrate",
            "lovr.headset.animate"
          },
          values = {
            {
              name = "head",
              description = "The headset