// Copyright The OpenTelemetry Authors
// SPDX-License-Identifier: Apache-2.0

package pdata // import "go.opentelemetry.io/collector/internal/cmd/pdatagen/internal/pdata"
import (
	"go.opentelemetry.io/collector/internal/cmd/pdatagen/internal/proto"
)

var pcommon = &Package{
	info: &PackageInfo{
		name: "pcommon",
		path: "pcommon",
		imports: []string{
			`"encoding/binary"`,
			`"fmt"`,
			`"iter"`,
			`"math"`,
			`"sort"`,
			`"sync"`,
			``,
			`"go.opentelemetry.io/collector/pdata/internal"`,
			`"go.opentelemetry.io/collector/pdata/internal/json"`,
			`"go.opentelemetry.io/collector/pdata/internal/proto"`,
		},
		testImports: []string{
			`"strconv"`,
			`"testing"`,
			``,
			`"github.com/stretchr/testify/assert"`,
			`"github.com/stretchr/testify/require"`,
			`"google.golang.org/protobuf/proto"`,
			`gootlpcommon "go.opentelemetry.io/proto/slim/otlp/common/v1"`,
			`gootlpresource "go.opentelemetry.io/proto/slim/otlp/resource/v1"`,
			``,
			`"go.opentelemetry.io/collector/internal/testutil"`,
			`"go.opentelemetry.io/collector/pdata/internal"`,
			`"go.opentelemetry.io/collector/pdata/internal/json"`,
		},
	},
	structs: []baseStruct{
		anyValueStruct,
		arrayValueStruct,
		keyValueStruct,
		keyValueListStruct,
		anyValueSlice,
		scope,
		resource,
		byteSlice,
		float64Slice,
		uInt64Slice,
		int64Slice,
		int32Slice,
		stringSlice,
	},
}

var scope = &messageStruct{
	structName:    "InstrumentationScope",
	packageName:   "pcommon",
	description:   "// InstrumentationScope is a message representing the instrumentation scope information.",
	protoName:     "InstrumentationScope",
	upstreamProto: "gootlpcommon.InstrumentationScope",
	fields: []Field{
		&PrimitiveField{
			fieldName: "Name",
			protoID:   1,
			protoType: proto.TypeString,
		},
		&PrimitiveField{
			fieldName: "Version",
			protoID:   2,
			protoType: proto.TypeString,
		},
		&SliceField{
			fieldName:   "Attributes",
			protoType:   proto.TypeMessage,
			protoID:     3,
			returnSlice: mapStruct,
		},
		&PrimitiveField{
			fieldName: "DroppedAttributesCount",
			protoID:   4,
			protoType: proto.TypeUint32,
		},
	},
}

// This will not be generated by this class.
// Defined here just to be available as returned message for the fields.
var mapStruct = &messageSlice{
	structName:      "Map",
	packageName:     "pcommon",
	elementNullable: false,
	element:         keyValueStruct,
}

var keyValueStruct = &messageStruct{
	structName:    "KeyValue",
	protoName:     "KeyValue",
	upstreamProto: "gootlpcommon.KeyValue",
	fields: []Field{
		&PrimitiveField{
			fieldName: "Key",
			protoID:   1,
			protoType: proto.TypeString,
		},
		&MessageField{
			fieldName:     "Value",
			protoID:       2,
			returnMessage: anyValueClone,
		},
	},
	hasOnlyInternal: true,
}

var anyValueClone = &messageStruct{
	structName: "Value",
	protoName:  "AnyValue",
}

// anyValueStruct needs to be different from anyValue because otherwise we cause initialization circular deps with mapStruct.
var anyValueStruct = &messageStruct{
	structName:    "Value",
	packageName:   "pcommon",
	protoName:     "AnyValue",
	upstreamProto: "gootlpcommon.AnyValue",
	fields: []Field{
		&OneOfField{
			typeName:                   "ValueType",
			originFieldName:            "Value",
			testValueIdx:               1, //
			omitOriginFieldNameInNames: true,
			values: []oneOfValue{
				&OneOfPrimitiveValue{
					fieldName:       "StringValue",
					protoID:         1,
					originFieldName: "StringValue",
					protoType:       proto.TypeString,
				},
				&OneOfPrimitiveValue{
					fieldName:       "BoolValue",
					protoID:         2,
					originFieldName: "BoolValue",
					protoType:       proto.TypeBool,
				},
				&OneOfPrimitiveValue{
					fieldName:       "IntValue",
					protoID:         3,
					originFieldName: "IntValue",
					protoType:       proto.TypeInt64,
				},
				&OneOfPrimitiveValue{
					fieldName:       "DoubleValue",
					protoID:         4,
					originFieldName: "DoubleValue",
					protoType:       proto.TypeDouble,
				},
				&OneOfMessageValue{
					fieldName:     "ArrayValue",
					protoID:       5,
					returnMessage: arrayValueStruct,
				},
				&OneOfMessageValue{
					fieldName:     "KvlistValue",
					protoID:       6,
					returnMessage: keyValueListStruct,
				},
				&OneOfPrimitiveValue{
					fieldName:       "BytesValue",
					protoID:         7,
					originFieldName: "BytesValue",
					protoType:       proto.TypeBytes,
				},
			},
		},
	},
	hasOnlyInternal: true,
}

var keyValueListStruct = &messageStruct{
	structName:    "KeyValueList",
	description:   "// KeyValueList is a list of KeyValue messages. We need KeyValueList as a message since oneof in AnyValue does not allow repeated fields.",
	protoName:     "KeyValueList",
	upstreamProto: "gootlpcommon.KeyValueList",
	fields: []Field{
		&SliceField{
			fieldName:   "Values",
			protoID:     1,
			protoType:   proto.TypeMessage,
			returnSlice: mapStruct,
		},
	},
	hasOnlyInternal: true,
}

var arrayValueStruct = &messageStruct{
	structName:    "ArrayValue",
	description:   "// ArrayValue is a list of AnyValue messages. We need ArrayValue as a message since oneof in AnyValue does not allow repeated fields.",
	protoName:     "ArrayValue",
	upstreamProto: "gootlpcommon.ArrayValue",
	fields: []Field{
		&SliceField{
			fieldName:   "Values",
			protoID:     1,
			protoType:   proto.TypeMessage,
			returnSlice: anyValueSlice,
		},
	},
	hasOnlyInternal: true,
}

var anyValueSlice = &messageSlice{
	structName:      "Slice",
	packageName:     "pcommon",
	elementNullable: false,
	element:         anyValueClone,
}

var traceState = &messageStruct{
	structName:  "TraceState",
	packageName: "pcommon",
	protoName:   "TraceState", // Fake name to generate correct CopyOrig* name.
}

var timestampType = &TypedType{
	structName:  "Timestamp",
	packageName: "pcommon",
	protoType:   proto.TypeFixed64,
	defaultVal:  "0",
	testVal:     "1234567890",
}

var traceIDType = &TypedType{
	structName:  "TraceID",
	packageName: "pcommon",
	protoType:   proto.TypeMessage,
	messageName: "TraceID",
	defaultVal:  "TraceID([16]byte{})",
	testVal:     "TraceID([16]byte{1, 2, 3, 4, 5, 6, 7, 8, 8, 7, 6, 5, 4, 3, 2, 1})",
}

var spanIDType = &TypedType{
	structName:  "SpanID",
	packageName: "pcommon",
	protoType:   proto.TypeMessage,
	messageName: "SpanID",
	defaultVal:  "SpanID([8]byte{})",
	testVal:     "SpanID([8]byte{8, 7, 6, 5, 4, 3, 2, 1})",
}

var resource = &messageStruct{
	structName:    "Resource",
	packageName:   "pcommon",
	description:   "// Resource is a message representing the resource information.",
	protoName:     "Resource",
	upstreamProto: "gootlpresource.Resource",
	fields: []Field{
		&SliceField{
			fieldName:   "Attributes",
			protoType:   proto.TypeMessage,
			protoID:     1,
			returnSlice: mapStruct,
		},
		&PrimitiveField{
			fieldName: "DroppedAttributesCount",
			protoID:   2,
			protoType: proto.TypeUint32,
		},
		&SliceField{
			fieldName:   "EntityRefs",
			protoType:   proto.TypeMessage,
			protoID:     3,
			returnSlice: entityRefSlice,
			// Hide accessors for this field from 1.x public API since the proto field is experimental.
			// It's available via the xpdata/entity.ResourceEntityRefs.
			hideAccessors: true,
		},
	},
}

var byteSlice = &primitiveSliceStruct{
	structName:           "ByteSlice",
	packageName:          "pcommon",
	itemType:             "byte",
	testOrigVal:          "1, 2, 3",
	testInterfaceOrigVal: []any{1, 2, 3},
	testSetVal:           "5",
	testNewVal:           "1, 5, 3",
}

var float64Slice = &primitiveSliceStruct{
	structName:           "Float64Slice",
	packageName:          "pcommon",
	itemType:             "float64",
	testOrigVal:          "1.1, 2.2, 3.3",
	testInterfaceOrigVal: []any{1.1, 2.2, 3.3},
	testSetVal:           "5.5",
	testNewVal:           "1.1, 5.5, 3.3",
}

var uInt64Slice = &primitiveSliceStruct{
	structName:           "UInt64Slice",
	packageName:          "pcommon",
	itemType:             "uint64",
	testOrigVal:          "1, 2, 3",
	testInterfaceOrigVal: []any{1, 2, 3},
	testSetVal:           "5",
	testNewVal:           "1, 5, 3",
}

var int64Slice = &primitiveSliceStruct{
	structName:           "Int64Slice",
	packageName:          "pcommon",
	itemType:             "int64",
	testOrigVal:          "1, 2, 3",
	testInterfaceOrigVal: []any{1, 2, 3},
	testSetVal:           "5",
	testNewVal:           "1, 5, 3",
}

var int32Slice = &primitiveSliceStruct{
	structName:           "Int32Slice",
	packageName:          "pcommon",
	itemType:             "int32",
	testOrigVal:          "1, 2, 3",
	testInterfaceOrigVal: []any{1, 2, 3},
	testSetVal:           "5",
	testNewVal:           "1, 5, 3",
}

var stringSlice = &primitiveSliceStruct{
	structName:           "StringSlice",
	packageName:          "pcommon",
	itemType:             "string",
	testOrigVal:          `"a", "b", "c"`,
	testInterfaceOrigVal: []any{`"a"`, `"b"`, `"c"`},
	testSetVal:           `"d"`,
	testNewVal:           `"a", "d", "c"`,
}
