// Copyright The OpenTelemetry Authors
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

// Package telemetry controls the telemetry settings to be used in the collector.
package telemetry

import (
	"flag"

	"go.opentelemetry.io/collector/internal/version"
)

const (
	metricsAddrCfg   = "metrics-addr"
	metricsPrefixCfg = "metrics-prefix"
)

var (
	// Command-line flags that control publication of telemetry data.
	metricsAddrPtr   *string
	metricsPrefixPtr *string

	addInstanceIDPtr *bool
)

func Flags(flags *flag.FlagSet) {
	// At least until we can use a generic, i.e.: OpenCensus, metrics exporter
	// we default to Prometheus at port 8888, if not otherwise specified.
	metricsAddrPtr = flags.String(
		metricsAddrCfg,
		GetMetricsAddrDefault(),
		"[address]:port for exposing collector telemetry.")

	metricsPrefixPtr = flags.String(
		metricsPrefixCfg,
		"otelcol",
		"Prefix to the metrics generated by the collector.")

	addInstanceIDPtr = flags.Bool(
		"add-instance-id",
		true,
		"Flag to control the addition of 'service.instance.id' to the collector metrics.")
}

// GetMetricsAddrDefault returns the default metrics bind address and port depending on
// the current build type.
func GetMetricsAddrDefault() string {
	if version.IsDevBuild() {
		// Listen on localhost by default for dev builds to avoid security prompts.
		return "localhost:8888"
	}
	return ":8888"
}

func GetAddInstanceID() bool {
	return *addInstanceIDPtr
}

func GetMetricsAddr() string {
	return *metricsAddrPtr
}

func GetMetricsPrefix() string {
	return *metricsPrefixPtr
}
