// Copyright 2025 Google LLC.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// Code generated file. DO NOT EDIT.

// Package merchantapi provides access to the Merchant API.
//
// For product documentation, see: https://developers.google.com/merchant/api
//
// # Library status
//
// These client libraries are officially supported by Google. However, this
// library is considered complete and is in maintenance mode. This means
// that we will address critical bugs and security issues but will not add
// any new features.
//
// When possible, we recommend using our newer
// [Cloud Client Libraries for Go](https://pkg.go.dev/cloud.google.com/go)
// that are still actively being worked and iterated on.
//
// # Creating a client
//
// Usage example:
//
//	import "google.golang.org/api/merchantapi/lfp_v1beta"
//	...
//	ctx := context.Background()
//	merchantapiService, err := merchantapi.NewService(ctx)
//
// In this example, Google Application Default Credentials are used for
// authentication. For information on how to create and obtain Application
// Default Credentials, see https://developers.google.com/identity/protocols/application-default-credentials.
//
// # Other authentication options
//
// To use an API key for authentication (note: some APIs do not support API
// keys), use [google.golang.org/api/option.WithAPIKey]:
//
//	merchantapiService, err := merchantapi.NewService(ctx, option.WithAPIKey("AIza..."))
//
// To use an OAuth token (e.g., a user token obtained via a three-legged OAuth
// flow, use [google.golang.org/api/option.WithTokenSource]:
//
//	config := &oauth2.Config{...}
//	// ...
//	token, err := config.Exchange(ctx, ...)
//	merchantapiService, err := merchantapi.NewService(ctx, option.WithTokenSource(config.TokenSource(ctx, token)))
//
// See [google.golang.org/api/option.ClientOption] for details on options.
package merchantapi // import "google.golang.org/api/merchantapi/lfp_v1beta"

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"log/slog"
	"net/http"
	"net/url"
	"strconv"
	"strings"

	"github.com/googleapis/gax-go/v2/internallog"
	googleapi "google.golang.org/api/googleapi"
	internal "google.golang.org/api/internal"
	gensupport "google.golang.org/api/internal/gensupport"
	option "google.golang.org/api/option"
	internaloption "google.golang.org/api/option/internaloption"
	htransport "google.golang.org/api/transport/http"
)

// Always reference these packages, just in case the auto-generated code
// below doesn't.
var _ = bytes.NewBuffer
var _ = strconv.Itoa
var _ = fmt.Sprintf
var _ = json.NewDecoder
var _ = io.Copy
var _ = url.Parse
var _ = gensupport.MarshalJSON
var _ = googleapi.Version
var _ = errors.New
var _ = strings.Replace
var _ = context.Canceled
var _ = internaloption.WithDefaultEndpoint
var _ = internal.Version
var _ = internallog.New

const apiId = "merchantapi:lfp_v1beta"
const apiName = "merchantapi"
const apiVersion = "lfp_v1beta"
const basePath = "https://merchantapi.googleapis.com/"
const basePathTemplate = "https://merchantapi.UNIVERSE_DOMAIN/"
const mtlsBasePath = "https://merchantapi.mtls.googleapis.com/"

// OAuth2 scopes used by this API.
const (
	// Manage your product listings and accounts for Google Shopping
	ContentScope = "https://www.googleapis.com/auth/content"
)

// NewService creates a new Service.
func NewService(ctx context.Context, opts ...option.ClientOption) (*Service, error) {
	scopesOption := internaloption.WithDefaultScopes(
		"https://www.googleapis.com/auth/content",
	)
	// NOTE: prepend, so we don't override user-specified scopes.
	opts = append([]option.ClientOption{scopesOption}, opts...)
	opts = append(opts, internaloption.WithDefaultEndpoint(basePath))
	opts = append(opts, internaloption.WithDefaultEndpointTemplate(basePathTemplate))
	opts = append(opts, internaloption.WithDefaultMTLSEndpoint(mtlsBasePath))
	opts = append(opts, internaloption.EnableNewAuthLibrary())
	client, endpoint, err := htransport.NewClient(ctx, opts...)
	if err != nil {
		return nil, err
	}
	s := &Service{client: client, BasePath: basePath, logger: internaloption.GetLogger(opts)}
	s.Accounts = NewAccountsService(s)
	if endpoint != "" {
		s.BasePath = endpoint
	}
	return s, nil
}

// New creates a new Service. It uses the provided http.Client for requests.
//
// Deprecated: please use NewService instead.
// To provide a custom HTTP client, use option.WithHTTPClient.
// If you are using google.golang.org/api/googleapis/transport.APIKey, use option.WithAPIKey with NewService instead.
func New(client *http.Client) (*Service, error) {
	if client == nil {
		return nil, errors.New("client is nil")
	}
	return NewService(context.TODO(), option.WithHTTPClient(client))
}

type Service struct {
	client    *http.Client
	logger    *slog.Logger
	BasePath  string // API endpoint base URL
	UserAgent string // optional additional User-Agent fragment

	Accounts *AccountsService
}

func (s *Service) userAgent() string {
	if s.UserAgent == "" {
		return googleapi.UserAgent
	}
	return googleapi.UserAgent + " " + s.UserAgent
}

func NewAccountsService(s *Service) *AccountsService {
	rs := &AccountsService{s: s}
	rs.LfpInventories = NewAccountsLfpInventoriesService(s)
	rs.LfpMerchantStates = NewAccountsLfpMerchantStatesService(s)
	rs.LfpSales = NewAccountsLfpSalesService(s)
	rs.LfpStores = NewAccountsLfpStoresService(s)
	return rs
}

type AccountsService struct {
	s *Service

	LfpInventories *AccountsLfpInventoriesService

	LfpMerchantStates *AccountsLfpMerchantStatesService

	LfpSales *AccountsLfpSalesService

	LfpStores *AccountsLfpStoresService
}

func NewAccountsLfpInventoriesService(s *Service) *AccountsLfpInventoriesService {
	rs := &AccountsLfpInventoriesService{s: s}
	return rs
}

type AccountsLfpInventoriesService struct {
	s *Service
}

func NewAccountsLfpMerchantStatesService(s *Service) *AccountsLfpMerchantStatesService {
	rs := &AccountsLfpMerchantStatesService{s: s}
	return rs
}

type AccountsLfpMerchantStatesService struct {
	s *Service
}

func NewAccountsLfpSalesService(s *Service) *AccountsLfpSalesService {
	rs := &AccountsLfpSalesService{s: s}
	return rs
}

type AccountsLfpSalesService struct {
	s *Service
}

func NewAccountsLfpStoresService(s *Service) *AccountsLfpStoresService {
	rs := &AccountsLfpStoresService{s: s}
	return rs
}

type AccountsLfpStoresService struct {
	s *Service
}

// CountrySettings: Country-specific settings for the merchant.
type CountrySettings struct {
	// FreeLocalListingsEnabled: True if this merchant has enabled free local
	// listings in MC.
	FreeLocalListingsEnabled bool `json:"freeLocalListingsEnabled,omitempty"`
	// InstockServingVerificationState: Output only. The verification state of this
	// merchant's instock serving feature.
	//
	// Possible values:
	//   "VERIFICATION_STATE_UNSPECIFIED" - Verification state unspecified.
	//   "VERIFICATION_STATE_NOT_APPROVED" - Verification state not approved.
	//   "VERIFICATION_STATE_IN_PROGRESS" - Verification state in progress.
	//   "VERIFICATION_STATE_APPROVED" - Verification state approved.
	InstockServingVerificationState string `json:"instockServingVerificationState,omitempty"`
	// InventoryVerificationState: Output only. The verification state of this
	// merchant's inventory check.
	//
	// Possible values:
	//   "VERIFICATION_STATE_UNSPECIFIED" - Verification state unspecified.
	//   "VERIFICATION_STATE_NOT_APPROVED" - Verification state not approved.
	//   "VERIFICATION_STATE_IN_PROGRESS" - Verification state in progress.
	//   "VERIFICATION_STATE_APPROVED" - Verification state approved.
	InventoryVerificationState string `json:"inventoryVerificationState,omitempty"`
	// LocalInventoryAdsEnabled: True if this merchant has enabled local inventory
	// ads in MC.
	LocalInventoryAdsEnabled bool `json:"localInventoryAdsEnabled,omitempty"`
	// PickupServingVerificationState: Output only. The verification state of this
	// merchant's pickup serving feature.
	//
	// Possible values:
	//   "VERIFICATION_STATE_UNSPECIFIED" - Verification state unspecified.
	//   "VERIFICATION_STATE_NOT_APPROVED" - Verification state not approved.
	//   "VERIFICATION_STATE_IN_PROGRESS" - Verification state in progress.
	//   "VERIFICATION_STATE_APPROVED" - Verification state approved.
	PickupServingVerificationState string `json:"pickupServingVerificationState,omitempty"`
	// ProductPageType: Output only. The product page type selected by this
	// merchant.
	//
	// Possible values:
	//   "PRODUCT_PAGE_TYPE_UNSPECIFIED" - Product page type unspecified.
	//   "GOOGLE_HOSTED" - Google hosted product page.
	//   "MERCHANT_HOSTED" - Merchant hosted product page.
	//   "MERCHANT_HOSTED_STORE_SPECIFIC" - Merchant hosted store specific product
	// page.
	ProductPageType string `json:"productPageType,omitempty"`
	// RegionCode: Required. The CLDR territory code
	// (https://github.com/unicode-org/cldr/blob/latest/common/main/en.xml) for the
	// country for which these settings are defined.
	RegionCode string `json:"regionCode,omitempty"`
	// ForceSendFields is a list of field names (e.g. "FreeLocalListingsEnabled")
	// to unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "FreeLocalListingsEnabled") to
	// include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CountrySettings) MarshalJSON() ([]byte, error) {
	type NoMethod CountrySettings
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Empty: A generic empty message that you can re-use to avoid defining
// duplicated empty messages in your APIs. A typical example is to use it as
// the request or the response type of an API method. For instance: service Foo
// { rpc Bar(google.protobuf.Empty) returns (google.protobuf.Empty); }
type Empty struct {
	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
}

// InventoryStats: The inventory statistics for a merchant.
type InventoryStats struct {
	// SubmittedEntries: Number of entries (understanding entry as a pair of
	// product and store) that were built based on provided inventories/sales and
	// submitted to Google.
	SubmittedEntries int64 `json:"submittedEntries,omitempty,string"`
	// SubmittedInStockEntries: Number of submitted in stock entries.
	SubmittedInStockEntries int64 `json:"submittedInStockEntries,omitempty,string"`
	// SubmittedProducts: Number of products from provided inventories/sales that
	// were created from matches to existing online products provided by the
	// merchant or to the Google catalog.
	SubmittedProducts int64 `json:"submittedProducts,omitempty,string"`
	// UnsubmittedEntries: Number of entries that were built based on provided
	// inventories/sales and couldn't be submitted to Google due to errors like
	// missing product.
	UnsubmittedEntries int64 `json:"unsubmittedEntries,omitempty,string"`
	// ForceSendFields is a list of field names (e.g. "SubmittedEntries") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "SubmittedEntries") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s InventoryStats) MarshalJSON() ([]byte, error) {
	type NoMethod InventoryStats
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// LfpInventory: Local Inventory for the merchant.
type LfpInventory struct {
	// Availability: Required. Availability of the product at this store. For
	// accepted attribute values, see the local product inventory data
	// specification (https://support.google.com/merchants/answer/3061342)
	Availability string `json:"availability,omitempty"`
	// CollectionTime: Optional. The time when the inventory is collected. If not
	// set, it will be set to the time when the inventory is submitted.
	CollectionTime string `json:"collectionTime,omitempty"`
	// ContentLanguage: Required. The two-letter ISO 639-1 language code for the
	// item.
	ContentLanguage string `json:"contentLanguage,omitempty"`
	// FeedLabel: Optional. The feed label
	// (https://developers.google.com/shopping-content/guides/products/feed-labels)
	// for the product. If this is not set, it will default to `regionCode`.
	FeedLabel string `json:"feedLabel,omitempty"`
	// Gtin: Optional. The Global Trade Item Number of the product.
	Gtin string `json:"gtin,omitempty"`
	// Name: Output only. Identifier. The name for the `LfpInventory` resource.
	// Format:
	// `accounts/{account}/lfpInventories/{target_merchant}~{store_code}~{offer}`
	Name string `json:"name,omitempty"`
	// OfferId: Required. Immutable. A unique identifier for the product. If both
	// inventories and sales are submitted for a merchant, this id should match for
	// the same product. **Note**: if the merchant sells the same product new and
	// used, they should have different IDs.
	OfferId string `json:"offerId,omitempty"`
	// PickupMethod: Optional. Supported pickup method for this offer. Unless the
	// value is "not supported", this field must be submitted together with
	// `pickupSla`. For accepted attribute values, see the local product inventory
	// data specification (https://support.google.com/merchants/answer/3061342).
	PickupMethod string `json:"pickupMethod,omitempty"`
	// PickupSla: Optional. Expected date that an order will be ready for pickup
	// relative to the order date. Must be submitted together with `pickupMethod`.
	// For accepted attribute values, see the local product inventory data
	// specification (https://support.google.com/merchants/answer/3061342).
	PickupSla string `json:"pickupSla,omitempty"`
	// Price: Optional. The current price of the product.
	Price *Price `json:"price,omitempty"`
	// Quantity: Optional. Quantity of the product available at this store. Must be
	// greater than or equal to zero.
	Quantity int64 `json:"quantity,omitempty,string"`
	// RegionCode: Required. The CLDR territory code
	// (https://github.com/unicode-org/cldr/blob/latest/common/main/en.xml) for the
	// country where the product is sold.
	RegionCode string `json:"regionCode,omitempty"`
	// StoreCode: Required. The identifier of the merchant's store. Either the
	// store code inserted through `InsertLfpStore` or the store code in the
	// Business Profile.
	StoreCode string `json:"storeCode,omitempty"`
	// TargetAccount: Required. The Merchant Center ID of the merchant to submit
	// the inventory for.
	TargetAccount int64 `json:"targetAccount,omitempty,string"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Availability") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Availability") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s LfpInventory) MarshalJSON() ([]byte, error) {
	type NoMethod LfpInventory
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// LfpMerchantState: The LFP state of a merchant.
type LfpMerchantState struct {
	// CountrySettings: Country-specific settings for the merchant.
	CountrySettings []*CountrySettings `json:"countrySettings,omitempty"`
	// InventoryStats: The inventory statistics for the merchant. The field will be
	// absent if the merchant has no inventory submitted through LFP.
	InventoryStats *InventoryStats `json:"inventoryStats,omitempty"`
	// LinkedGbps: Number of GBPs (https://www.google.com/business/) this merchant
	// has access to.
	LinkedGbps int64 `json:"linkedGbps,omitempty,string"`
	// Name: Identifier. The name of the `LfpMerchantState` resource. Format:
	// `accounts/{account}/lfpMerchantStates/{target_merchant}`. For example,
	// `accounts/123456/lfpMerchantStates/567890`.
	Name string `json:"name,omitempty"`
	// StoreStates: Output only. The state per store from the specified merchant.
	// The field will be absent if the merchant has no stores submitted through
	// LFP.
	StoreStates []*LfpStoreState `json:"storeStates,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "CountrySettings") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CountrySettings") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s LfpMerchantState) MarshalJSON() ([]byte, error) {
	type NoMethod LfpMerchantState
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// LfpSale: A sale for the merchant.
type LfpSale struct {
	// ContentLanguage: Required. The two-letter ISO 639-1 language code for the
	// item.
	ContentLanguage string `json:"contentLanguage,omitempty"`
	// FeedLabel: Optional. The feed label
	// (https://developers.google.com/shopping-content/guides/products/feed-labels)
	// for the product. If this is not set, it will default to `regionCode`.
	FeedLabel string `json:"feedLabel,omitempty"`
	// Gtin: Required. The Global Trade Item Number of the sold product.
	Gtin string `json:"gtin,omitempty"`
	// Name: Output only. Identifier. The name of the `LfpSale` resource. Format:
	// `accounts/{account}/lfpSales/{sale}`
	Name string `json:"name,omitempty"`
	// OfferId: Required. A unique identifier for the product. If both inventories
	// and sales are submitted for a merchant, this id should match for the same
	// product. **Note**: if the merchant sells the same product new and used, they
	// should have different IDs.
	OfferId string `json:"offerId,omitempty"`
	// Price: Required. The unit price of the product.
	Price *Price `json:"price,omitempty"`
	// Quantity: Required. The relative change of the available quantity. Negative
	// for items returned.
	Quantity int64 `json:"quantity,omitempty,string"`
	// RegionCode: Required. The CLDR territory code
	// (https://github.com/unicode-org/cldr/blob/latest/common/main/en.xml) for the
	// country where the product is sold.
	RegionCode string `json:"regionCode,omitempty"`
	// SaleTime: Required. The timestamp for the sale.
	SaleTime string `json:"saleTime,omitempty"`
	// StoreCode: Required. The identifier of the merchant's store. Either a
	// `storeCode` inserted through the API or the code of the store in the
	// Business Profile.
	StoreCode string `json:"storeCode,omitempty"`
	// TargetAccount: Required. The Merchant Center ID of the merchant to submit
	// the sale for.
	TargetAccount int64 `json:"targetAccount,omitempty,string"`
	// Uid: Output only. System generated globally unique ID for the `LfpSale`.
	Uid string `json:"uid,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "ContentLanguage") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ContentLanguage") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s LfpSale) MarshalJSON() ([]byte, error) {
	type NoMethod LfpSale
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// LfpStore: A store for the merchant. This will be used to match to a store
// under the Google Business Profile of the target merchant. If a matching
// store can't be found, the inventories or sales submitted with the store code
// will not be used.
type LfpStore struct {
	// GcidCategory: Optional. Google My Business category id
	// (https://gcid-explorer.corp.google.com/static/gcid.html).
	GcidCategory []string `json:"gcidCategory,omitempty"`
	// MatchingState: Optional. Output only. The state of matching to a Google
	// Business Profile. See matchingStateHint for further details if no match is
	// found.
	//
	// Possible values:
	//   "STORE_MATCHING_STATE_UNSPECIFIED" - Store matching state unspecified.
	//   "STORE_MATCHING_STATE_MATCHED" - The `LfpStore` is successfully matched
	// with a Google Business Profile store.
	//   "STORE_MATCHING_STATE_FAILED" - The `LfpStore` is not matched with a
	// Google Business Profile store.
	MatchingState string `json:"matchingState,omitempty"`
	// MatchingStateHint: Optional. Output only. The hint of why the matching has
	// failed. This is only set when matchingState=`STORE_MATCHING_STATE_FAILED`.
	// Possible values are: - "linked-store-not-found": There aren't any Google
	// Business Profile stores available for matching. - "store-match-not-found":
	// The provided `LfpStore` couldn't be matched to any of the connected Google
	// Business Profile stores. Merchant Center account is connected correctly and
	// stores are available on Google Business Profile, but the `LfpStore` location
	// address does not match with Google Business Profile stores' addresses.
	// Update the `LfpStore` address or Google Business Profile store address to
	// match correctly. - "store-match-unverified": The provided `LfpStore`
	// couldn't be matched to any of the connected Google Business Profile stores,
	// as the matched Google Business Profile store is unverified. Go through the
	// Google Business Profile verification process to match correctly.
	MatchingStateHint string `json:"matchingStateHint,omitempty"`
	// Name: Output only. Identifier. The name of the `LfpStore` resource. Format:
	// `accounts/{account}/lfpStores/{target_merchant}~{store_code}`
	Name string `json:"name,omitempty"`
	// PhoneNumber: Optional. The store phone number in E.164
	// (https://en.wikipedia.org/wiki/E.164) format. Example: `+15556767888`
	PhoneNumber string `json:"phoneNumber,omitempty"`
	// PlaceId: Optional. The Google Place Id
	// (https://developers.google.com/maps/documentation/places/web-service/place-id#id-overview)
	// of the store location.
	PlaceId string `json:"placeId,omitempty"`
	// StoreAddress: Required. The street address of the store. Example: 1600
	// Amphitheatre Pkwy, Mountain View, CA 94043, USA.
	StoreAddress string `json:"storeAddress,omitempty"`
	// StoreCode: Required. Immutable. A store identifier that is unique for the
	// target merchant.
	StoreCode string `json:"storeCode,omitempty"`
	// StoreName: Optional. The merchant or store name.
	StoreName string `json:"storeName,omitempty"`
	// TargetAccount: Required. The Merchant Center id of the merchant to submit
	// the store for.
	TargetAccount int64 `json:"targetAccount,omitempty,string"`
	// WebsiteUri: Optional. The website URL for the store or merchant.
	WebsiteUri string `json:"websiteUri,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "GcidCategory") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "GcidCategory") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s LfpStore) MarshalJSON() ([]byte, error) {
	type NoMethod LfpStore
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// LfpStoreState: The state of a specific merchant's store.
type LfpStoreState struct {
	// MatchingState: Output only. The store matching state.
	//
	// Possible values:
	//   "STORE_MATCHING_STATE_UNSPECIFIED" - Store matching state unspecified.
	//   "STORE_MATCHING_STATE_MATCHED" - The `LfpStore` is successfully matched
	// with a Google Business Profile store.
	//   "STORE_MATCHING_STATE_FAILED" - The `LfpStore` is not matched with a
	// Google Business Profile store.
	MatchingState string `json:"matchingState,omitempty"`
	// MatchingStateHint: The hint of why the matching has failed (only set if
	// matching_state is FAILED).
	MatchingStateHint string `json:"matchingStateHint,omitempty"`
	// StoreCode: Required. Immutable. The identifier of this store.
	StoreCode string `json:"storeCode,omitempty"`
	// ForceSendFields is a list of field names (e.g. "MatchingState") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "MatchingState") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s LfpStoreState) MarshalJSON() ([]byte, error) {
	type NoMethod LfpStoreState
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListLfpStoresResponse: Response message for the ListLfpStores method.
type ListLfpStoresResponse struct {
	// LfpStores: The stores from the specified merchant.
	LfpStores []*LfpStore `json:"lfpStores,omitempty"`
	// NextPageToken: A token, which can be sent as `pageToken` to retrieve the
	// next page. If this field is omitted, there are no subsequent pages.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "LfpStores") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "LfpStores") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListLfpStoresResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListLfpStoresResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Price: The price represented as a number and currency.
type Price struct {
	// AmountMicros: The price represented as a number in micros (1 million micros
	// is an equivalent to one's currency standard unit, for example, 1 USD =
	// 1000000 micros).
	AmountMicros int64 `json:"amountMicros,omitempty,string"`
	// CurrencyCode: The currency of the price using three-letter acronyms
	// according to ISO 4217 (http://en.wikipedia.org/wiki/ISO_4217).
	CurrencyCode string `json:"currencyCode,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AmountMicros") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AmountMicros") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Price) MarshalJSON() ([]byte, error) {
	type NoMethod Price
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ProductChange: The change that happened to the product including old value,
// new value, country code as the region code and reporting context.
type ProductChange struct {
	// NewValue: The new value of the changed resource or attribute. If empty, it
	// means that the product was deleted. Will have one of these values :
	// (`approved`, `pending`, `disapproved`, ``)
	NewValue string `json:"newValue,omitempty"`
	// OldValue: The old value of the changed resource or attribute. If empty, it
	// means that the product was created. Will have one of these values :
	// (`approved`, `pending`, `disapproved`, ``)
	OldValue string `json:"oldValue,omitempty"`
	// RegionCode: Countries that have the change (if applicable). Represented in
	// the ISO 3166 format.
	RegionCode string `json:"regionCode,omitempty"`
	// ReportingContext: Reporting contexts that have the change (if applicable).
	// Currently this field supports only (`SHOPPING_ADS`, `LOCAL_INVENTORY_ADS`,
	// `YOUTUBE_SHOPPING`, `YOUTUBE_CHECKOUT`, `YOUTUBE_AFFILIATE`) from the enum
	// value ReportingContextEnum
	// (/merchant/api/reference/rest/Shared.Types/ReportingContextEnum)
	//
	// Possible values:
	//   "REPORTING_CONTEXT_ENUM_UNSPECIFIED" - Not specified.
	//   "SHOPPING_ADS" - [Shopping
	// ads](https://support.google.com/merchants/answer/6149970).
	//   "DISCOVERY_ADS" - Deprecated: Use `DEMAND_GEN_ADS` instead. [Discovery and
	// Demand Gen ads](https://support.google.com/merchants/answer/13389785).
	//   "DEMAND_GEN_ADS" - [Demand Gen
	// ads](https://support.google.com/merchants/answer/13389785).
	//   "DEMAND_GEN_ADS_DISCOVER_SURFACE" - [Demand Gen ads on Discover
	// surface](https://support.google.com/merchants/answer/13389785).
	//   "VIDEO_ADS" - [Video
	// ads](https://support.google.com/google-ads/answer/6340491).
	//   "DISPLAY_ADS" - [Display
	// ads](https://support.google.com/merchants/answer/6069387).
	//   "LOCAL_INVENTORY_ADS" - [Local inventory
	// ads](https://support.google.com/merchants/answer/3271956).
	//   "VEHICLE_INVENTORY_ADS" - [Vehicle inventory
	// ads](https://support.google.com/merchants/answer/11544533).
	//   "FREE_LISTINGS" - [Free product
	// listings](https://support.google.com/merchants/answer/9199328).
	//   "FREE_LOCAL_LISTINGS" - [Free local product
	// listings](https://support.google.com/merchants/answer/9825611).
	//   "FREE_LOCAL_VEHICLE_LISTINGS" - [Free local vehicle
	// listings](https://support.google.com/merchants/answer/11544533).
	//   "YOUTUBE_AFFILIATE" - [Youtube
	// Affiliate](https://support.google.com/youtube/answer/13376398).
	//   "YOUTUBE_SHOPPING" - [YouTube
	// Shopping](https://support.google.com/merchants/answer/13478370).
	//   "CLOUD_RETAIL" - [Cloud
	// retail](https://cloud.google.com/solutions/retail).
	//   "LOCAL_CLOUD_RETAIL" - [Local cloud
	// retail](https://cloud.google.com/solutions/retail).
	//   "PRODUCT_REVIEWS" - [Product
	// Reviews](https://support.google.com/merchants/answer/14620732).
	//   "MERCHANT_REVIEWS" - [Merchant
	// Reviews](https://developers.google.com/merchant-review-feeds).
	//   "YOUTUBE_CHECKOUT" - YouTube Checkout .
	ReportingContext string `json:"reportingContext,omitempty"`
	// ForceSendFields is a list of field names (e.g. "NewValue") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "NewValue") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ProductChange) MarshalJSON() ([]byte, error) {
	type NoMethod ProductChange
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ProductStatusChangeMessage: The message that the merchant will receive to
// notify about product status change event
type ProductStatusChangeMessage struct {
	// Account: The target account that owns the entity that changed. Format :
	// `accounts/{merchant_id}`
	Account string `json:"account,omitempty"`
	// Attribute: The attribute in the resource that changed, in this case it will
	// be always `Status`.
	//
	// Possible values:
	//   "ATTRIBUTE_UNSPECIFIED" - Unspecified attribute
	//   "STATUS" - Status of the changed entity
	Attribute string `json:"attribute,omitempty"`
	// Changes: A message to describe the change that happened to the product
	Changes []*ProductChange `json:"changes,omitempty"`
	// EventTime: The time at which the event was generated. If you want to order
	// the notification messages you receive you should rely on this field not on
	// the order of receiving the notifications.
	EventTime string `json:"eventTime,omitempty"`
	// ExpirationTime: Optional. The product expiration time. This field will not
	// be set if the notification is sent for a product deletion event.
	ExpirationTime string `json:"expirationTime,omitempty"`
	// ManagingAccount: The account that manages the merchant's account. can be the
	// same as merchant id if it is standalone account. Format :
	// `accounts/{service_provider_id}`
	ManagingAccount string `json:"managingAccount,omitempty"`
	// Resource: The product name. Format: `accounts/{account}/products/{product}`
	Resource string `json:"resource,omitempty"`
	// ResourceId: The product id.
	ResourceId string `json:"resourceId,omitempty"`
	// ResourceType: The resource that changed, in this case it will always be
	// `Product`.
	//
	// Possible values:
	//   "RESOURCE_UNSPECIFIED" - Unspecified resource
	//   "PRODUCT" - Resource type : product
	ResourceType string `json:"resourceType,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Account") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Account") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ProductStatusChangeMessage) MarshalJSON() ([]byte, error) {
	type NoMethod ProductStatusChangeMessage
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type AccountsLfpInventoriesInsertCall struct {
	s            *Service
	parent       string
	lfpinventory *LfpInventory
	urlParams_   gensupport.URLParams
	ctx_         context.Context
	header_      http.Header
}

// Insert: Inserts a `LfpInventory` resource for the given target merchant
// account. If the resource already exists, it will be replaced. The inventory
// automatically expires after 30 days.
//
// - parent: The LFP provider account. Format: `accounts/{account}`.
func (r *AccountsLfpInventoriesService) Insert(parent string, lfpinventory *LfpInventory) *AccountsLfpInventoriesInsertCall {
	c := &AccountsLfpInventoriesInsertCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	c.lfpinventory = lfpinventory
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsLfpInventoriesInsertCall) Fields(s ...googleapi.Field) *AccountsLfpInventoriesInsertCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsLfpInventoriesInsertCall) Context(ctx context.Context) *AccountsLfpInventoriesInsertCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsLfpInventoriesInsertCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsLfpInventoriesInsertCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.lfpinventory)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "lfp/v1beta/{+parent}/lfpInventories:insert")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.lfpInventories.insert", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.lfpInventories.insert" call.
// Any non-2xx status code is an error. Response headers are in either
// *LfpInventory.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsLfpInventoriesInsertCall) Do(opts ...googleapi.CallOption) (*LfpInventory, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &LfpInventory{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.lfpInventories.insert", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsLfpMerchantStatesGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets the LFP state of a merchant
//
//   - name: The name of the state to retrieve. Format:
//     `accounts/{account}/lfpMerchantStates/{target_merchant}`. For example,
//     `accounts/123456/lfpMerchantStates/567890`.
func (r *AccountsLfpMerchantStatesService) Get(name string) *AccountsLfpMerchantStatesGetCall {
	c := &AccountsLfpMerchantStatesGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsLfpMerchantStatesGetCall) Fields(s ...googleapi.Field) *AccountsLfpMerchantStatesGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsLfpMerchantStatesGetCall) IfNoneMatch(entityTag string) *AccountsLfpMerchantStatesGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsLfpMerchantStatesGetCall) Context(ctx context.Context) *AccountsLfpMerchantStatesGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsLfpMerchantStatesGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsLfpMerchantStatesGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "lfp/v1beta/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.lfpMerchantStates.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.lfpMerchantStates.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *LfpMerchantState.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *AccountsLfpMerchantStatesGetCall) Do(opts ...googleapi.CallOption) (*LfpMerchantState, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &LfpMerchantState{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.lfpMerchantStates.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsLfpSalesInsertCall struct {
	s          *Service
	parent     string
	lfpsale    *LfpSale
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Insert: Inserts a `LfpSale` for the given merchant.
//
// - parent: The LFP provider account. Format: `accounts/{lfp_partner}`.
func (r *AccountsLfpSalesService) Insert(parent string, lfpsale *LfpSale) *AccountsLfpSalesInsertCall {
	c := &AccountsLfpSalesInsertCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	c.lfpsale = lfpsale
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsLfpSalesInsertCall) Fields(s ...googleapi.Field) *AccountsLfpSalesInsertCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsLfpSalesInsertCall) Context(ctx context.Context) *AccountsLfpSalesInsertCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsLfpSalesInsertCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsLfpSalesInsertCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.lfpsale)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "lfp/v1beta/{+parent}/lfpSales:insert")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.lfpSales.insert", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.lfpSales.insert" call.
// Any non-2xx status code is an error. Response headers are in either
// *LfpSale.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsLfpSalesInsertCall) Do(opts ...googleapi.CallOption) (*LfpSale, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &LfpSale{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.lfpSales.insert", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsLfpStoresDeleteCall struct {
	s          *Service
	name       string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Deletes a store for a target merchant.
//
//   - name: The name of the store to delete for the target merchant account.
//     Format: `accounts/{account}/lfpStores/{target_merchant}~{store_code}`.
func (r *AccountsLfpStoresService) Delete(name string) *AccountsLfpStoresDeleteCall {
	c := &AccountsLfpStoresDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsLfpStoresDeleteCall) Fields(s ...googleapi.Field) *AccountsLfpStoresDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsLfpStoresDeleteCall) Context(ctx context.Context) *AccountsLfpStoresDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsLfpStoresDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsLfpStoresDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "lfp/v1beta/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.lfpStores.delete", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.lfpStores.delete" call.
// Any non-2xx status code is an error. Response headers are in either
// *Empty.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsLfpStoresDeleteCall) Do(opts ...googleapi.CallOption) (*Empty, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Empty{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.lfpStores.delete", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsLfpStoresGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Retrieves information about a store.
//
//   - name: The name of the store to retrieve. Format:
//     `accounts/{account}/lfpStores/{target_merchant}~{store_code}`.
func (r *AccountsLfpStoresService) Get(name string) *AccountsLfpStoresGetCall {
	c := &AccountsLfpStoresGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsLfpStoresGetCall) Fields(s ...googleapi.Field) *AccountsLfpStoresGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsLfpStoresGetCall) IfNoneMatch(entityTag string) *AccountsLfpStoresGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsLfpStoresGetCall) Context(ctx context.Context) *AccountsLfpStoresGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsLfpStoresGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsLfpStoresGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "lfp/v1beta/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.lfpStores.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.lfpStores.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *LfpStore.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsLfpStoresGetCall) Do(opts ...googleapi.CallOption) (*LfpStore, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &LfpStore{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.lfpStores.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsLfpStoresInsertCall struct {
	s          *Service
	parent     string
	lfpstore   *LfpStore
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Insert: Inserts a store for the target merchant. If the store with the same
// store code already exists, it will be replaced.
//
// - parent: The LFP provider account Format: `accounts/{account}`.
func (r *AccountsLfpStoresService) Insert(parent string, lfpstore *LfpStore) *AccountsLfpStoresInsertCall {
	c := &AccountsLfpStoresInsertCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	c.lfpstore = lfpstore
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsLfpStoresInsertCall) Fields(s ...googleapi.Field) *AccountsLfpStoresInsertCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsLfpStoresInsertCall) Context(ctx context.Context) *AccountsLfpStoresInsertCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsLfpStoresInsertCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsLfpStoresInsertCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.lfpstore)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "lfp/v1beta/{+parent}/lfpStores:insert")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.lfpStores.insert", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.lfpStores.insert" call.
// Any non-2xx status code is an error. Response headers are in either
// *LfpStore.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsLfpStoresInsertCall) Do(opts ...googleapi.CallOption) (*LfpStore, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &LfpStore{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.lfpStores.insert", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsLfpStoresListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists the stores of the target merchant, specified by the filter in
// `ListLfpStoresRequest`.
//
// - parent: The LFP partner. Format: `accounts/{account}`.
func (r *AccountsLfpStoresService) List(parent string) *AccountsLfpStoresListCall {
	c := &AccountsLfpStoresListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of
// `LfpStore` resources for the given account to return. The service returns
// fewer than this value if the number of stores for the given account is less
// than the `pageSize`. The default value is 250. The maximum value is 1000; If
// a value higher than the maximum is specified, then the `pageSize` will
// default to the maximum.
func (c *AccountsLfpStoresListCall) PageSize(pageSize int64) *AccountsLfpStoresListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A page token, received
// from a previous `ListLfpStoresRequest` call. Provide the page token to
// retrieve the subsequent page. When paginating, all other parameters provided
// to `ListLfpStoresRequest` must match the call that provided the page token.
// The token returned as nextPageToken in the response to the previous request.
func (c *AccountsLfpStoresListCall) PageToken(pageToken string) *AccountsLfpStoresListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// TargetAccount sets the optional parameter "targetAccount": Required. The
// Merchant Center id of the merchant to list stores for.
func (c *AccountsLfpStoresListCall) TargetAccount(targetAccount int64) *AccountsLfpStoresListCall {
	c.urlParams_.Set("targetAccount", fmt.Sprint(targetAccount))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsLfpStoresListCall) Fields(s ...googleapi.Field) *AccountsLfpStoresListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsLfpStoresListCall) IfNoneMatch(entityTag string) *AccountsLfpStoresListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsLfpStoresListCall) Context(ctx context.Context) *AccountsLfpStoresListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsLfpStoresListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsLfpStoresListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "lfp/v1beta/{+parent}/lfpStores")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.lfpStores.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.lfpStores.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListLfpStoresResponse.ServerResponse.Header or (if a response was returned
// at all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *AccountsLfpStoresListCall) Do(opts ...googleapi.CallOption) (*ListLfpStoresResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListLfpStoresResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.lfpStores.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *AccountsLfpStoresListCall) Pages(ctx context.Context, f func(*ListLfpStoresResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}
