//---------------------------------------------------------------------------
// Copyright (C) 2000 Dallas Semiconductor Corporation, All Rights Reserved.
// 
// Permission is hereby granted, free of charge, to any person obtaining a 
// copy of this software and associated documentation files (the "Software"), 
// to deal in the Software without restriction, including without limitation 
// the rights to use, copy, modify, merge, publish, distribute, sublicense, 
// and/or sell copies of the Software, and to permit persons to whom the 
// Software is furnished to do so, subject to the following conditions:
// 
// The above copyright notice and this permission notice shall be included 
// in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS 
// OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
// MERCHANTABILITY,  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. 
// IN NO EVENT SHALL DALLAS SEMICONDUCTOR BE LIABLE FOR ANY CLAIM, DAMAGES 
// OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, 
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR 
// OTHER DEALINGS IN THE SOFTWARE.
// 
// Except as contained in this notice, the name of Dallas Semiconductor 
// shall not be used except as stated in the Dallas Semiconductor 
// Branding Policy. 
//---------------------------------------------------------------------------
//
//  ibsha18o.C - SHA-iButton I/O utility functions
//
//  Version:   2.00
//  History:   
//
//  Debug code designated with      //\\//\\//\\//\\//
//
//

#include "ownet.h"   
#include "ibsha18.h"

//\\//\\//\\//\\//\\//\\//\\//\\//\\//
extern void output_status(int, char *);
//\\//\\//\\//\\//\\//\\//\\//\\//\\//

// extern functions
extern int owBlock(int,int,uchar *,int);
extern void setcrc16(int,ushort);
extern ushort docrc16(int,ushort);
extern int  owVerify(int,int);
extern int owTouchReset(int);
extern int owSpeed(int,int);
extern int owWriteByte(int,int);
extern void owSerialNum(int,uchar *, int);


// Local Function Prototypes 
int EraseScratchSHA(int);
int ReadScratchSHA(int,ushort *,uchar *,uchar *);
int WriteScratchSHA(int,ushort,uchar *,int);
int CopyScratchSHA(int,ushort,int);
int ReadAuthPageSHA(int,ushort,uchar *);
int ReadUDP_SHA(int,ushort,uchar *,int *);
int WriteUDP_SHA(int,ushort,uchar *,int);
int ComputeSHA(int,ushort, uchar);
int MatchScratchSHA(int,uchar *);
int FormatUDP(int,ushort,uchar *,int *);
int Select(int);

//\\//\\//\\//\\//\\//\\//\\//\\//\\//
#ifdef DEBUG_DUMP 
   void debugout(uchar *buf, int len);    
#endif
//\\//\\//\\//\\//\\//\\//\\//\\//\\//

//\\//\\//\\//\\//\\//\\//\\//\\//\\//
int TestModeMatch(int);
//\\//\\//\\//\\//\\//\\//\\//\\//\\//

// globals
int in_overdrive[MAX_PORTNUM];


//--------------------------------------------------------------------------
// Attempt to erase the scratchpad of the current SHA iButton  
//
// 'portnum'     - number 0 to MAX_PORTNUM-1.  This number is provided to
//                 indicate the symbolic port number.
//
// Returns: TRUE, success, scratchpad erased
//          FALSE, failure to erase scratchpad
//
int EraseScratchSHA(int portnum)
{
   uchar send_block[50];
   int i;
   int num_verf;
   short send_cnt=0;

   // access the device 
   if (Select(portnum) == 1)
   {
      // change number of verification bytes if in overdrive 
      num_verf = (in_overdrive[portnum]) ? 6 : 2;

      // create a block to send 
      // erase scratchpad command
      send_block[send_cnt++] = 0xC3;

      // now add the address and verification bytes
      for (i = 0; i < 2 + num_verf; i++)
         send_block[send_cnt++] = 0xFF;

      // now send the block
      if (owBlock(portnum,FALSE,send_block,send_cnt))
      {
         //\\//\\//\\//\\//\\//\\//\\//\\//\\//
         #ifdef DEBUG_DUMP 
            debugout(send_block,send_cnt);
         #endif
         //\\//\\//\\//\\//\\//\\//\\//\\//\\//

         // check verification
         if (((send_block[send_cnt-1] & 0xF0) == 0x50) || 
              ((send_block[send_cnt-1] & 0xF0) == 0xA0))
            return TRUE;
      }
   }

   return FALSE;
}

//----------------------------------------------------------------------
// Read the scratchpad with CRC16 verification
//
// 'portnum'     - number 0 to MAX_PORTNUM-1.  This number is provided to
//                 indicate the symbolic port number.
// 'address'     - pointer to address that is read from scratchpad
// 'es'          - pointer to offset byte read from scratchpad
// 'data'        - pointer data buffer read from scratchpad
//
// Return: TRUE - scratch read, address, es, and data returned
//         FALSE - error reading scratch, device not present
//
//                                                           
int ReadScratchSHA(int portnum, ushort *address, uchar *es, uchar *data)
{
   short send_cnt=0;
   uchar send_block[50];
   int i;
   ushort lastcrc16;

   // access the device 
   if (Select(portnum) == 1)
   {
      // read scratchpad command
      send_block[send_cnt++] = 0xAA;
      // now add the read bytes for data bytes and crc16
      for (i = 0; i < 37; i++)
         send_block[send_cnt++] = 0xFF;

      // now send the block
      if (owBlock(portnum,FALSE,send_block,send_cnt))
      {
         //\\//\\//\\//\\//\\//\\//\\//\\//\\//
         #ifdef DEBUG_DUMP 
            debugout(send_block,send_cnt);
         #endif
         //\\//\\//\\//\\//\\//\\//\\//\\//\\//

         // copy data to return buffers
         *address = (send_block[2] << 8) | send_block[1];
         *es = send_block[3];

         // calculate CRC16 of result
         setcrc16(portnum,0);
         for (i = 0; i < send_cnt ; i++)
            lastcrc16 = docrc16(portnum,send_block[i]);

         // verify CRC16 is correct
         if (lastcrc16 == 0xB001)
         {
            for (i = 0; i < 32; i++)
               data[i] = send_block[4 + i];  
            // success
            return TRUE;
         }
      }
   }

   return FALSE;
}

//----------------------------------------------------------------------
// Write the scratchpad with CRC16 verification.  The data is padded 
// until the offset is 0x1F so that the CRC16 is retrieved.
//
// 'portnum'   - number 0 to MAX_PORTNUM-1.  This number is provided to
//               indicate the symbolic port number.
// 'address'   - address to write data to 
// 'data'      - data to write
// 'data_len'  - number of bytes of data to write 
//
// Return: TRUE - write to scratch verified
//         FALSE - error writing scratch, device not present, or HIDE
//                 flag is in incorrect state for address being written.
//                                                           
int WriteScratchSHA(int portnum, ushort address, uchar *data, int data_len)
{
   uchar send_block[50];
   short send_cnt=0,i;
   ushort lastcrc16;

   // access the device 
   if (Select(portnum) == 1)
   {
      setcrc16(portnum,0);
      // write scratchpad command
      send_block[send_cnt] = 0x0F;
      docrc16(portnum,send_block[send_cnt++]);
      // address 1
      send_block[send_cnt] = (uchar)(address & 0xFF);
      docrc16(portnum,send_block[send_cnt++]);
      // address 2
      send_block[send_cnt] = (uchar)((address >> 8) & 0xFF);
      docrc16(portnum,send_block[send_cnt++]);
      // data
      for (i = 0; i < data_len; i++)
      {
         send_block[send_cnt] = data[i];
         docrc16(portnum,send_block[send_cnt++]);
      }
      // pad if needed
      for (i = 0; i < (0x1F - ((address + data_len - 1) & 0x1F)); i++)
      {
         send_block[send_cnt] = 0xFF;
         docrc16(portnum,send_block[send_cnt++]);
      }
      // CRC16
      send_block[send_cnt++] = 0xFF;
      send_block[send_cnt++] = 0xFF;

      // now send the block
      if (owBlock(portnum,FALSE,send_block,send_cnt))
      {
         //\\//\\//\\//\\//\\//\\//\\//\\//\\//
         #ifdef DEBUG_DUMP 
            debugout(send_block,send_cnt);
         #endif
         //\\//\\//\\//\\//\\//\\//\\//\\//\\//

         // perform CRC16 of last 2 byte in packet
         for (i = send_cnt - 2; i < send_cnt; i++)
            lastcrc16 = docrc16(portnum,send_block[i]);
            
         // verify CRC16 is correct
         if (lastcrc16 == 0xB001)
            // success
            return TRUE;
      }
   }

   return FALSE;
}


//----------------------------------------------------------------------
// Copy the scratchpad with verification.  Assume that the data was
// padded to get the CRC16 verification on write scratchpad.  This 
// will result in the 'es' byte to be 0x1F.
//
// 'portnum'     - number 0 to MAX_PORTNUM-1.  This number is provided to
//                 indicate the symbolic port number.
// 'address'     - address of destination
// 'len'         - length of data
//
// Return: TRUE - copy scratch verified
//         FALSE - error during copy scratch, device not present, or HIDE
//                 flag is in incorrect state for address being written.
//                                                           
int CopyScratchSHA(int portnum, ushort address, int len)
{
   short send_cnt=0;
   uchar send_block[80];
   int i,num_verf;

   // access the device 
   if (Select(portnum) == 1)
   {
      // change number of verification bytes if in overdrive 
      num_verf = (in_overdrive[portnum]) ? 4 : 2;

      // copy scratchpad command
      send_block[send_cnt++] = 0x55;
      // address 1
      send_block[send_cnt++] = (uchar)(address & 0xFF);
      // address 2
      send_block[send_cnt++] = (uchar)((address >> 8) & 0xFF);
      // es
      send_block[send_cnt++] = (address + len - 1) & 0x1F;

      // verification bytes
      for (i = 0; i < num_verf; i++)
         send_block[send_cnt++] = 0xFF;

      // now send the block
      if (owBlock(portnum,FALSE,send_block,send_cnt))
      {
         //\\//\\//\\//\\//\\//\\//\\//\\//\\//
         #ifdef DEBUG_DUMP 
            debugout(send_block,send_cnt);
         #endif
         //\\//\\//\\//\\//\\//\\//\\//\\//\\//

         // check verification
         if (((send_block[send_cnt-1] & 0xF0) == 0x50) || 
              ((send_block[send_cnt-1] & 0xF0) == 0xA0))
            return TRUE;
      }
   }

   return FALSE;
}

//----------------------------------------------------------------------
// Read Authenticated Page.  
//
// 'portnum'     - number 0 to MAX_PORTNUM-1.  This number is provided to
//                 indicate the symbolic port number.
// 'address'     - address of page to do a read authenticate
// 'pate_data'   - buffer to how data read from page
//
// Return: TRUE - read sucessfull
//         FALSE - CRC error, device not present
//                                                           
int ReadAuthPageSHA(int portnum, ushort address, uchar *page_data)
{
   short send_cnt=0;
   uchar send_block[80];
   int i,cnt,num_verf;
   ushort lastcrc16;

   // access the device 
   if (Select(portnum) == 1)
   {
      setcrc16(portnum,0);
      // change number of verification bytes if in overdrive 
      num_verf = (in_overdrive[portnum]) ? 10 : 2;

      // copy scratchpad command
      send_block[send_cnt] = 0xA5;
      docrc16(portnum,send_block[send_cnt++]);
      // address 1
      send_block[send_cnt] = (uchar)(address & 0xFF);
      docrc16(portnum,send_block[send_cnt++]);
      // address 2
      send_block[send_cnt] = (uchar)((address >> 8) & 0xFF);
      docrc16(portnum,send_block[send_cnt++]);
      // now add the read bytes for data bytes, counter, and crc16, verification
      for (i = 0; i < (42 + num_verf); i++)
         send_block[send_cnt++] = 0xFF;

      // now send the block
      if (owBlock(portnum,FALSE,send_block,send_cnt))
      {
         //\\//\\//\\//\\//\\//\\//\\//\\//\\//
         #ifdef DEBUG_DUMP 
            debugout(send_block,send_cnt);
         #endif
         //\\//\\//\\//\\//\\//\\//\\//\\//\\//

         // check the CRC
         for (i = 3; i < (send_cnt - num_verf); i++)
            lastcrc16 = docrc16(portnum,send_block[i]);
            
         // verify CRC16 is correct
         if (lastcrc16 != 0xB001)
            return FALSE;

         // check verification
         if (((send_block[send_cnt-1] & 0xF0) != 0x50) && 
              ((send_block[send_cnt-1] & 0xF0) != 0xA0))
            return FALSE;

         // transfer results 
         cnt = 0;
         for (i = send_cnt - 42 - num_verf; i < (send_cnt - 2 - num_verf); i++)
            page_data[cnt++] = send_block[i];
         return TRUE;
      }
   }

   return FALSE;
}

//----------------------------------------------------------------------
// Read a limited (29 byets) Universal Data Packet. Put the data in 
// 'page_data' and return the length in 'length'.  Note, 'page_data' 
// buffer must be at least 29 bytes long.
//
// 'portnum'     - number 0 to MAX_PORTNUM-1.  This number is provided to
//                 indicate the symbolic port number.
// 'address'     - address of page to read UDP 
// 'page_data'   - data read (minus UDP overhead)
// 'length'      - pointer to length of data found
//
// Return: TRUE - valid UDP found on specified page and data returned
//         FALSE - error, device not present or invalid UDP
//
//                                                           
int ReadUDP_SHA(int portnum, ushort address, uchar *page_data, int *length)
{
   uchar send_block[50];
   int i;
   short send_cnt=0,head_len;
   ushort lastcrc16;

   // access the device
   if (Select(portnum) == 1)
   {
      // create a block to send
      // read memory
      send_block[send_cnt++] = 0xF0; 
      // address
      send_block[send_cnt++] = (uchar)(address & 0xFF); 
      send_block[send_cnt++] = (uchar)((address >> 8) & 0xFF); 
      // data to read
      head_len = send_cnt;
      for (i = 0; i < 32; i++)
         send_block[send_cnt++] = 0xFF;

      // now send the block
      if (owBlock(portnum,FALSE,send_block,send_cnt))
      {
         //\\//\\//\\//\\//\\//\\//\\//\\//\\//
         #ifdef DEBUG_DUMP 
            debugout(send_block,send_cnt);
         #endif
         //\\//\\//\\//\\//\\//\\//\\//\\//\\//

         // seed crc with page number
         setcrc16(portnum,(ushort)(address >> 5));               

         // attempt to read UDP from TranBuf
         *length = send_block[head_len];            
         docrc16(portnum,(ushort)(*length));

         // verify length is not too large
         if (*length <= 29)                
         {
            // loop to read packet including CRC
            for (i = 0; i < *length; i++)     
            {
                page_data[i] = send_block[i + 1 + head_len];
                docrc16(portnum,page_data[i]);           
            }
            
            // read and compute the CRC16 
            docrc16(portnum,send_block[i + 1 + head_len]);
            lastcrc16 = docrc16(portnum,send_block[i + 2 + head_len]);
         
            // verify the CRC16 is correct           
            if (lastcrc16 == 0xB001) 
              return TRUE;        // return number of byte in record
         }  
      }
   }

   return FALSE;  
}


//----------------------------------------------------------------------
// Take the data in the 'write_buf' and format to meet UDP standard. 
// Return the new length in 'write_len'. 'write_buf' must be at least
// 32 bytes in size.
//
// 'portnum'      - number 0 to MAX_PORTNUM-1.  This number is provided to
//                  indicate the symbolic port number.
// 'address'      - destination address (used in CRC16)
// 'write_buffer' - buffer that contains the data to format. Also used
//                  as the return buffer.
// 'write_len'    - input length to function and output length after 
//                  formating. 
//
// Return: TRUE - valid UDP for specified page create
//         FALSE - error, device not present or could not write
//
//                                                           
int FormatUDP(int portnum, ushort address, uchar *write_buf, int *write_len)
{
   uchar send_block[32];
   int i,send_cnt=0;
   ushort lastcrc16;

   // check to see if data too long to fit on device
   if (*write_len > 29)
     return FALSE;
              
   // seed crc with page number           
   setcrc16(portnum,(ushort)(address >> 5));
   
   // set length byte
   send_block[send_cnt++] = (uchar)(*write_len);
   docrc16(portnum,(ushort)(*write_len));
      
   // fill in the data to write
   for (i = 0; i < *write_len; i++)
   {
     lastcrc16 = docrc16(portnum,write_buf[i]);
     send_block[send_cnt++] = write_buf[i];
   }  
      
   // add the crc
   send_block[send_cnt++] = (uchar)(~(lastcrc16 & 0xFF));
   send_block[send_cnt++] = (uchar)(~((lastcrc16 & 0xFF00) >> 8));
   
   // copy send block back to the write_buf
   for (i = 0; i < send_cnt; i++)
      write_buf[i] = send_block[i];
   
   // set the new length
   *write_len = send_cnt;

   //  success
   return TRUE;
}


//----------------------------------------------------------------------
// Write a limited (29 byte) Universal Data Packet. Get the data from 
// 'write_buf' with length 'write_len'.  
//
// 'portnum'     - number 0 to MAX_PORTNUM-1.  This number is provided to
//                 indicate the symbolic port number.
// 'address'     - address of page to write UDP 
// 'write_buf'   - buffer of data to write
// 'write_len'   - length of data in bytes
//
// Return: TRUE - valid UDP written to specified page 
//         FALSE - error, device not present or could not write
//
//                                                           
int WriteUDP_SHA(int portnum, ushort address, uchar *write_buf, int write_len)
{
   uchar send_block[32];
   int i,send_cnt= write_len;

   // copy data to temp buffer
   for (i = 0; i < write_len; i++)
      send_block[i] = write_buf[i];

   // format the data into a UDP packet
   if (!FormatUDP(portnum,address,send_block,&send_cnt))
      return FALSE;

   // write the scratchpad
   if (!WriteScratchSHA(portnum,address,send_block,send_cnt))
      return FALSE;

   // copy the scratchpad            
   if (!CopyScratchSHA(portnum,address,send_cnt))
      return FALSE;
  
   // copy scratch pad was good then success
   return TRUE;
}


//----------------------------------------------------------------------
// Computer sha command based on control_byte and page address.  
//
// 'portnum'      - number 0 to MAX_PORTNUM-1.  This number is provided to
//                  indicate the symbolic port number.
// 'address'      - address used in compute sha operation
// 'control_byte' - control byte used in sha operation
//
// Return: TRUE - compute sha finished
//         FALSE - CRC error, device not present
//                                                           
int ComputeSHA(int portnum, ushort address, uchar control_byte)
{
   short send_cnt=0;
   uchar send_block[50];
   int i,num_verf;
   ushort lastcrc16;

   // access the device 
   if (Select(portnum) == 1)
   {
      setcrc16(portnum,0);
      // change number of verification bytes if in overdrive 
      num_verf = (in_overdrive[portnum]) ? 10 : 2;

      // copy scratchpad command
      send_block[send_cnt] = 0x33;
      docrc16(portnum,send_block[send_cnt++]);
      // address 1
      send_block[send_cnt] = (uchar)(address & 0xFF);
      docrc16(portnum,send_block[send_cnt++]);
      // address 2
      send_block[send_cnt] = (uchar)((address >> 8) & 0xFF);
      docrc16(portnum,send_block[send_cnt++]);
      // control byte 
      send_block[send_cnt] = control_byte;
      docrc16(portnum,send_block[send_cnt++]);

      // now read bytes crc16, and verification 
      for (i = 0; i < 2 + num_verf; i++)
         send_block[send_cnt++] = 0xFF;

      // now send the block
      if (owBlock(portnum,FALSE,send_block,send_cnt))
      {
         //\\//\\//\\//\\//\\//\\//\\//\\//\\//
         #ifdef DEBUG_DUMP 
            debugout(send_block,send_cnt);
         #endif
         //\\//\\//\\//\\//\\//\\//\\//\\//\\//

         // check the CRC
         for (i = 4; i < (send_cnt - num_verf); i++)
            lastcrc16 = docrc16(portnum,send_block[i]);
            
         // verify CRC16 is correct
         if (lastcrc16 != 0xB001)
            return FALSE;

         // check verification
         if (((send_block[send_cnt-1] & 0xF0) != 0x50) && 
              ((send_block[send_cnt-1] & 0xF0) != 0xA0))
            return FALSE;

         return TRUE;
      }
   }

   return FALSE;
}


//----------------------------------------------------------------------
// Perform a scratchpad match using provided data. Fixed data length 
// of 20 bytes. 
//
// 'portnum'     - number 0 to MAX_PORTNUM-1.  This number is provided to
//                 indicate the symbolic port number.
// 'data'        - data to use in match scratch operation
//
// Return: TRUE - valid match
//         FALSE - no match or device not present
//                               
int MatchScratchSHA(int portnum, uchar *data)
{
   short send_cnt=0;
   uchar send_block[50];
   int i;
   ushort lastcrc16;

   // access the device 
   if (Select(portnum) == 1)
   {
      setcrc16(portnum,0);
      // copy scratchpad command
      send_block[send_cnt] = CMD_MATCH_SCRATCH;
      docrc16(portnum,send_block[send_cnt++]);

      // send 20 data bytes
      for (i = 0; i < 20; i++)
      {
         send_block[send_cnt] = data[i]; 
         docrc16(portnum,send_block[send_cnt++]);
      }
      // send two crc bytes and verification byte
      for (i = 0; i < 3; i++)
         send_block[send_cnt++] = 0xFF;

      // now send the block
      if (owBlock(portnum,FALSE,send_block,send_cnt))
      {
         //\\//\\//\\//\\//\\//\\//\\//\\//\\//
         #ifdef DEBUG_DUMP 
            debugout(send_block,send_cnt);
         #endif
         //\\//\\//\\//\\//\\//\\//\\//\\//\\//

         // check the CRC
         for (i = (send_cnt - 3); i < (send_cnt - 1); i++)
            lastcrc16 = docrc16(portnum,send_block[i]);
            
         // verify CRC16 is correct
         if (lastcrc16 != 0xB001)
            return FALSE;

         // check verification
         if (((send_block[send_cnt-1] & 0xF0) != 0x50) && 
              ((send_block[send_cnt-1] & 0xF0) != 0xA0))
            return FALSE;

         return TRUE;
      }
   }

   return FALSE;
}

//--------------------------------------------------------------------------
// Select the current device and attempt overdrive if possible.
//
// 'portnum'     - number 0 to MAX_PORTNUM-1.  This number is provided to
//                 indicate the symbolic port number.
//
// Return: TRUE - device selected
//         FALSE - device not select
//
int Select(int portnum)
{
   int rt,cnt=0;

   //\\//\\//\\//\\//\\//\\//\\//\\//\\//
   #ifdef DEBUG_DUMP 
      uchar ROM[8];
      int i, mcnt;
      char msg[255];

      owSerialNum(portnum,ROM, TRUE);   
      mcnt = sprintf(msg,"\n  Device select ");
      for (i = 0; i < 8; i++)
         mcnt += sprintf(msg + mcnt, "%02X",ROM[i]);
      mcnt += sprintf(msg + mcnt,"\n");
      output_status(LV_VERBOSE,msg);
   #endif
   //\\//\\//\\//\\//\\//\\//\\//\\//\\//

   // loop to access the device and optionally do overdrive
   do
   {
      rt = owVerify(portnum,FALSE);

      // check not present
      if (rt != 1)
      {
         // if in overdrive, drop back
         if (in_overdrive[portnum])
         {
            // set to normal speed
            owSpeed(portnum,MODE_NORMAL);
            in_overdrive[portnum] = FALSE;
         }
      }
      // present but not in overdrive
      else if (!in_overdrive[portnum])
      {
         // put all devices in overdrive
         if (owTouchReset(portnum))
         {
            if (owWriteByte(portnum,0x3C))
            {
               // set to overdrive speed
               owSpeed(portnum,MODE_OVERDRIVE);
               in_overdrive[portnum] = TRUE;
            }
         }
         rt = 0;
      }
      else
         break;
   }
   while ((rt != 1) && (cnt++ < 3));

   return rt;
}


//\\//\\//\\//\\//\\//\\//\\//\\//\\//
#ifdef DEBUG_DUMP
//--------------------------------------------------------------------------
// function used in debug dump
//
void debugout(uchar *buf, int len)
{
   char pbuf[1024];
   int i,cnt;

   cnt = sprintf(pbuf,"  (%d) ",len);
   for (i = 0; i < len; i++)
   {
      cnt += sprintf(&pbuf[cnt],"%02X ",buf[i]);
      if ((((i + 1) % 16) == 0) && (i != 0))
         cnt += sprintf(&pbuf[cnt],"\n       ");
   }
   cnt += sprintf(&pbuf[cnt],"\n   ");
   output_status(LV_VERBOSE,pbuf);
}
#endif
//\\//\\//\\//\\//\\//\\//\\//\\//\\//


//\\//\\//\\//\\//\\//\\//\\//\\//\\//
int TestModeMatch(int portnum)
{
   short send_cnt=0;
   uchar send_block[50];
   uchar ROM[8];
   int i;
   char buf[200];
   static int cnt=0;

   sprintf(buf,"**************** attempt test mode ************(%d)\n",cnt++); 
   output_status(LV_VERBOSE,buf);

   owSerialNum(portnum,ROM, TRUE);   

   // enter test mode
   // test mode match
   send_block[send_cnt++] = 0xAA;
   // now add the read bytes for data bytes and crc16
   for (i = 0; i < 8; i++)
      send_block[send_cnt++] = (uchar)ROM[i];
   // enter test mode command
   send_block[send_cnt++] = 0xC3;

   // now send the block
   if (owBlock(portnum,TRUE,send_block,send_cnt))
   {
      //\\//\\//\\//\\//\\//\\//\\//\\//\\//
      #ifdef DEBUG_DUMP 
         debugout(send_block,send_cnt);
      #endif
      //\\//\\//\\//\\//\\//\\//\\//\\//\\//
   }
   else
      return FALSE;

   // exit test mode
   send_cnt=0;
   // test mode match
   send_block[send_cnt++] = 0xAA;
   // now add the read bytes for data bytes and crc16
   for (i = 0; i < 8; i++)
      send_block[send_cnt++] = (uchar)ROM[i];
   // exit test mode command
   send_block[send_cnt++] = 0x3C;

   // now send the block
   if (owBlock(portnum,TRUE,send_block,send_cnt))
   {
      //\\//\\//\\//\\//\\//\\//\\//\\//\\//
      #ifdef DEBUG_DUMP 
         debugout(send_block,send_cnt);
      #endif
      //\\//\\//\\//\\//\\//\\//\\//\\//\\//

      return TRUE;
   }
   else
      return FALSE;
}
//\\//\\//\\//\\//\\//\\//\\//\\//\\//

